/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The data type name for including resource data sync state. There are four sync states:
 * </p>
 * <p>
 * <code>OrganizationNotExists</code> (Your organization doesn't exist)
 * </p>
 * <p>
 * <code>NoPermissions</code> (The system can't locate the service-linked role. This role is automatically created when
 * a user creates a resource data sync in Amazon Web Services Systems Manager Explorer.)
 * </p>
 * <p>
 * <code>InvalidOrganizationalUnit</code> (You specified or selected an invalid unit in the resource data sync
 * configuration.)
 * </p>
 * <p>
 * <code>TrustedAccessDisabled</code> (You disabled Systems Manager access in the organization in Organizations.)
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ResourceDataSyncSourceWithState implements SdkPojo, Serializable,
        ToCopyableBuilder<ResourceDataSyncSourceWithState.Builder, ResourceDataSyncSourceWithState> {
    private static final SdkField<String> SOURCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SourceType").getter(getter(ResourceDataSyncSourceWithState::sourceType))
            .setter(setter(Builder::sourceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceType").build()).build();

    private static final SdkField<ResourceDataSyncAwsOrganizationsSource> AWS_ORGANIZATIONS_SOURCE_FIELD = SdkField
            .<ResourceDataSyncAwsOrganizationsSource> builder(MarshallingType.SDK_POJO).memberName("AwsOrganizationsSource")
            .getter(getter(ResourceDataSyncSourceWithState::awsOrganizationsSource))
            .setter(setter(Builder::awsOrganizationsSource)).constructor(ResourceDataSyncAwsOrganizationsSource::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AwsOrganizationsSource").build())
            .build();

    private static final SdkField<List<String>> SOURCE_REGIONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("SourceRegions")
            .getter(getter(ResourceDataSyncSourceWithState::sourceRegions))
            .setter(setter(Builder::sourceRegions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceRegions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> INCLUDE_FUTURE_REGIONS_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("IncludeFutureRegions").getter(getter(ResourceDataSyncSourceWithState::includeFutureRegions))
            .setter(setter(Builder::includeFutureRegions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IncludeFutureRegions").build())
            .build();

    private static final SdkField<String> STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("State")
            .getter(getter(ResourceDataSyncSourceWithState::state)).setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("State").build()).build();

    private static final SdkField<Boolean> ENABLE_ALL_OPS_DATA_SOURCES_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN).memberName("EnableAllOpsDataSources")
            .getter(getter(ResourceDataSyncSourceWithState::enableAllOpsDataSources))
            .setter(setter(Builder::enableAllOpsDataSources))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EnableAllOpsDataSources").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SOURCE_TYPE_FIELD,
            AWS_ORGANIZATIONS_SOURCE_FIELD, SOURCE_REGIONS_FIELD, INCLUDE_FUTURE_REGIONS_FIELD, STATE_FIELD,
            ENABLE_ALL_OPS_DATA_SOURCES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("SourceType", SOURCE_TYPE_FIELD);
                    put("AwsOrganizationsSource", AWS_ORGANIZATIONS_SOURCE_FIELD);
                    put("SourceRegions", SOURCE_REGIONS_FIELD);
                    put("IncludeFutureRegions", INCLUDE_FUTURE_REGIONS_FIELD);
                    put("State", STATE_FIELD);
                    put("EnableAllOpsDataSources", ENABLE_ALL_OPS_DATA_SOURCES_FIELD);
                }
            });

    private static final long serialVersionUID = 1L;

    private final String sourceType;

    private final ResourceDataSyncAwsOrganizationsSource awsOrganizationsSource;

    private final List<String> sourceRegions;

    private final Boolean includeFutureRegions;

    private final String state;

    private final Boolean enableAllOpsDataSources;

    private ResourceDataSyncSourceWithState(BuilderImpl builder) {
        this.sourceType = builder.sourceType;
        this.awsOrganizationsSource = builder.awsOrganizationsSource;
        this.sourceRegions = builder.sourceRegions;
        this.includeFutureRegions = builder.includeFutureRegions;
        this.state = builder.state;
        this.enableAllOpsDataSources = builder.enableAllOpsDataSources;
    }

    /**
     * <p>
     * The type of data source for the resource data sync. <code>SourceType</code> is either
     * <code>AwsOrganizations</code> (if an organization is present in Organizations) or
     * <code>singleAccountMultiRegions</code>.
     * </p>
     * 
     * @return The type of data source for the resource data sync. <code>SourceType</code> is either
     *         <code>AwsOrganizations</code> (if an organization is present in Organizations) or
     *         <code>singleAccountMultiRegions</code>.
     */
    public final String sourceType() {
        return sourceType;
    }

    /**
     * <p>
     * The field name in <code>SyncSource</code> for the <code>ResourceDataSyncAwsOrganizationsSource</code> type.
     * </p>
     * 
     * @return The field name in <code>SyncSource</code> for the <code>ResourceDataSyncAwsOrganizationsSource</code>
     *         type.
     */
    public final ResourceDataSyncAwsOrganizationsSource awsOrganizationsSource() {
        return awsOrganizationsSource;
    }

    /**
     * For responses, this returns true if the service returned a value for the SourceRegions property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasSourceRegions() {
        return sourceRegions != null && !(sourceRegions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The <code>SyncSource</code> Amazon Web Services Regions included in the resource data sync.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSourceRegions} method.
     * </p>
     * 
     * @return The <code>SyncSource</code> Amazon Web Services Regions included in the resource data sync.
     */
    public final List<String> sourceRegions() {
        return sourceRegions;
    }

    /**
     * <p>
     * Whether to automatically synchronize and aggregate data from new Amazon Web Services Regions when those Regions
     * come online.
     * </p>
     * 
     * @return Whether to automatically synchronize and aggregate data from new Amazon Web Services Regions when those
     *         Regions come online.
     */
    public final Boolean includeFutureRegions() {
        return includeFutureRegions;
    }

    /**
     * <p>
     * The data type name for including resource data sync state. There are four sync states:
     * </p>
     * <p>
     * <code>OrganizationNotExists</code>: Your organization doesn't exist.
     * </p>
     * <p>
     * <code>NoPermissions</code>: The system can't locate the service-linked role. This role is automatically created
     * when a user creates a resource data sync in Explorer.
     * </p>
     * <p>
     * <code>InvalidOrganizationalUnit</code>: You specified or selected an invalid unit in the resource data sync
     * configuration.
     * </p>
     * <p>
     * <code>TrustedAccessDisabled</code>: You disabled Systems Manager access in the organization in Organizations.
     * </p>
     * 
     * @return The data type name for including resource data sync state. There are four sync states:</p>
     *         <p>
     *         <code>OrganizationNotExists</code>: Your organization doesn't exist.
     *         </p>
     *         <p>
     *         <code>NoPermissions</code>: The system can't locate the service-linked role. This role is automatically
     *         created when a user creates a resource data sync in Explorer.
     *         </p>
     *         <p>
     *         <code>InvalidOrganizationalUnit</code>: You specified or selected an invalid unit in the resource data
     *         sync configuration.
     *         </p>
     *         <p>
     *         <code>TrustedAccessDisabled</code>: You disabled Systems Manager access in the organization in
     *         Organizations.
     */
    public final String state() {
        return state;
    }

    /**
     * <p>
     * When you create a resource data sync, if you choose one of the Organizations options, then Systems Manager
     * automatically enables all OpsData sources in the selected Amazon Web Services Regions for all Amazon Web Services
     * accounts in your organization (or in the selected organization units). For more information, see <a
     * href="https://docs.aws.amazon.com/systems-manager/latest/userguide/Explorer-resource-data-sync.html">Setting up
     * Systems Manager Explorer to display data from multiple accounts and Regions</a> in the <i>Amazon Web Services
     * Systems Manager User Guide</i>.
     * </p>
     * 
     * @return When you create a resource data sync, if you choose one of the Organizations options, then Systems
     *         Manager automatically enables all OpsData sources in the selected Amazon Web Services Regions for all
     *         Amazon Web Services accounts in your organization (or in the selected organization units). For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/systems-manager/latest/userguide/Explorer-resource-data-sync.html"
     *         >Setting up Systems Manager Explorer to display data from multiple accounts and Regions</a> in the
     *         <i>Amazon Web Services Systems Manager User Guide</i>.
     */
    public final Boolean enableAllOpsDataSources() {
        return enableAllOpsDataSources;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(sourceType());
        hashCode = 31 * hashCode + Objects.hashCode(awsOrganizationsSource());
        hashCode = 31 * hashCode + Objects.hashCode(hasSourceRegions() ? sourceRegions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(includeFutureRegions());
        hashCode = 31 * hashCode + Objects.hashCode(state());
        hashCode = 31 * hashCode + Objects.hashCode(enableAllOpsDataSources());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ResourceDataSyncSourceWithState)) {
            return false;
        }
        ResourceDataSyncSourceWithState other = (ResourceDataSyncSourceWithState) obj;
        return Objects.equals(sourceType(), other.sourceType())
                && Objects.equals(awsOrganizationsSource(), other.awsOrganizationsSource())
                && hasSourceRegions() == other.hasSourceRegions() && Objects.equals(sourceRegions(), other.sourceRegions())
                && Objects.equals(includeFutureRegions(), other.includeFutureRegions()) && Objects.equals(state(), other.state())
                && Objects.equals(enableAllOpsDataSources(), other.enableAllOpsDataSources());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ResourceDataSyncSourceWithState").add("SourceType", sourceType())
                .add("AwsOrganizationsSource", awsOrganizationsSource())
                .add("SourceRegions", hasSourceRegions() ? sourceRegions() : null)
                .add("IncludeFutureRegions", includeFutureRegions()).add("State", state())
                .add("EnableAllOpsDataSources", enableAllOpsDataSources()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "SourceType":
            return Optional.ofNullable(clazz.cast(sourceType()));
        case "AwsOrganizationsSource":
            return Optional.ofNullable(clazz.cast(awsOrganizationsSource()));
        case "SourceRegions":
            return Optional.ofNullable(clazz.cast(sourceRegions()));
        case "IncludeFutureRegions":
            return Optional.ofNullable(clazz.cast(includeFutureRegions()));
        case "State":
            return Optional.ofNullable(clazz.cast(state()));
        case "EnableAllOpsDataSources":
            return Optional.ofNullable(clazz.cast(enableAllOpsDataSources()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<ResourceDataSyncSourceWithState, T> g) {
        return obj -> g.apply((ResourceDataSyncSourceWithState) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ResourceDataSyncSourceWithState> {
        /**
         * <p>
         * The type of data source for the resource data sync. <code>SourceType</code> is either
         * <code>AwsOrganizations</code> (if an organization is present in Organizations) or
         * <code>singleAccountMultiRegions</code>.
         * </p>
         * 
         * @param sourceType
         *        The type of data source for the resource data sync. <code>SourceType</code> is either
         *        <code>AwsOrganizations</code> (if an organization is present in Organizations) or
         *        <code>singleAccountMultiRegions</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceType(String sourceType);

        /**
         * <p>
         * The field name in <code>SyncSource</code> for the <code>ResourceDataSyncAwsOrganizationsSource</code> type.
         * </p>
         * 
         * @param awsOrganizationsSource
         *        The field name in <code>SyncSource</code> for the <code>ResourceDataSyncAwsOrganizationsSource</code>
         *        type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder awsOrganizationsSource(ResourceDataSyncAwsOrganizationsSource awsOrganizationsSource);

        /**
         * <p>
         * The field name in <code>SyncSource</code> for the <code>ResourceDataSyncAwsOrganizationsSource</code> type.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link ResourceDataSyncAwsOrganizationsSource.Builder} avoiding the need to create one manually via
         * {@link ResourceDataSyncAwsOrganizationsSource#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ResourceDataSyncAwsOrganizationsSource.Builder#build()} is called
         * immediately and its result is passed to
         * {@link #awsOrganizationsSource(ResourceDataSyncAwsOrganizationsSource)}.
         * 
         * @param awsOrganizationsSource
         *        a consumer that will call methods on {@link ResourceDataSyncAwsOrganizationsSource.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #awsOrganizationsSource(ResourceDataSyncAwsOrganizationsSource)
         */
        default Builder awsOrganizationsSource(Consumer<ResourceDataSyncAwsOrganizationsSource.Builder> awsOrganizationsSource) {
            return awsOrganizationsSource(ResourceDataSyncAwsOrganizationsSource.builder().applyMutation(awsOrganizationsSource)
                    .build());
        }

        /**
         * <p>
         * The <code>SyncSource</code> Amazon Web Services Regions included in the resource data sync.
         * </p>
         * 
         * @param sourceRegions
         *        The <code>SyncSource</code> Amazon Web Services Regions included in the resource data sync.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceRegions(Collection<String> sourceRegions);

        /**
         * <p>
         * The <code>SyncSource</code> Amazon Web Services Regions included in the resource data sync.
         * </p>
         * 
         * @param sourceRegions
         *        The <code>SyncSource</code> Amazon Web Services Regions included in the resource data sync.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceRegions(String... sourceRegions);

        /**
         * <p>
         * Whether to automatically synchronize and aggregate data from new Amazon Web Services Regions when those
         * Regions come online.
         * </p>
         * 
         * @param includeFutureRegions
         *        Whether to automatically synchronize and aggregate data from new Amazon Web Services Regions when
         *        those Regions come online.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder includeFutureRegions(Boolean includeFutureRegions);

        /**
         * <p>
         * The data type name for including resource data sync state. There are four sync states:
         * </p>
         * <p>
         * <code>OrganizationNotExists</code>: Your organization doesn't exist.
         * </p>
         * <p>
         * <code>NoPermissions</code>: The system can't locate the service-linked role. This role is automatically
         * created when a user creates a resource data sync in Explorer.
         * </p>
         * <p>
         * <code>InvalidOrganizationalUnit</code>: You specified or selected an invalid unit in the resource data sync
         * configuration.
         * </p>
         * <p>
         * <code>TrustedAccessDisabled</code>: You disabled Systems Manager access in the organization in Organizations.
         * </p>
         * 
         * @param state
         *        The data type name for including resource data sync state. There are four sync states:</p>
         *        <p>
         *        <code>OrganizationNotExists</code>: Your organization doesn't exist.
         *        </p>
         *        <p>
         *        <code>NoPermissions</code>: The system can't locate the service-linked role. This role is
         *        automatically created when a user creates a resource data sync in Explorer.
         *        </p>
         *        <p>
         *        <code>InvalidOrganizationalUnit</code>: You specified or selected an invalid unit in the resource data
         *        sync configuration.
         *        </p>
         *        <p>
         *        <code>TrustedAccessDisabled</code>: You disabled Systems Manager access in the organization in
         *        Organizations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder state(String state);

        /**
         * <p>
         * When you create a resource data sync, if you choose one of the Organizations options, then Systems Manager
         * automatically enables all OpsData sources in the selected Amazon Web Services Regions for all Amazon Web
         * Services accounts in your organization (or in the selected organization units). For more information, see <a
         * href="https://docs.aws.amazon.com/systems-manager/latest/userguide/Explorer-resource-data-sync.html">Setting
         * up Systems Manager Explorer to display data from multiple accounts and Regions</a> in the <i>Amazon Web
         * Services Systems Manager User Guide</i>.
         * </p>
         * 
         * @param enableAllOpsDataSources
         *        When you create a resource data sync, if you choose one of the Organizations options, then Systems
         *        Manager automatically enables all OpsData sources in the selected Amazon Web Services Regions for all
         *        Amazon Web Services accounts in your organization (or in the selected organization units). For more
         *        information, see <a
         *        href="https://docs.aws.amazon.com/systems-manager/latest/userguide/Explorer-resource-data-sync.html"
         *        >Setting up Systems Manager Explorer to display data from multiple accounts and Regions</a> in the
         *        <i>Amazon Web Services Systems Manager User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enableAllOpsDataSources(Boolean enableAllOpsDataSources);
    }

    static final class BuilderImpl implements Builder {
        private String sourceType;

        private ResourceDataSyncAwsOrganizationsSource awsOrganizationsSource;

        private List<String> sourceRegions = DefaultSdkAutoConstructList.getInstance();

        private Boolean includeFutureRegions;

        private String state;

        private Boolean enableAllOpsDataSources;

        private BuilderImpl() {
        }

        private BuilderImpl(ResourceDataSyncSourceWithState model) {
            sourceType(model.sourceType);
            awsOrganizationsSource(model.awsOrganizationsSource);
            sourceRegions(model.sourceRegions);
            includeFutureRegions(model.includeFutureRegions);
            state(model.state);
            enableAllOpsDataSources(model.enableAllOpsDataSources);
        }

        public final String getSourceType() {
            return sourceType;
        }

        public final void setSourceType(String sourceType) {
            this.sourceType = sourceType;
        }

        @Override
        public final Builder sourceType(String sourceType) {
            this.sourceType = sourceType;
            return this;
        }

        public final ResourceDataSyncAwsOrganizationsSource.Builder getAwsOrganizationsSource() {
            return awsOrganizationsSource != null ? awsOrganizationsSource.toBuilder() : null;
        }

        public final void setAwsOrganizationsSource(ResourceDataSyncAwsOrganizationsSource.BuilderImpl awsOrganizationsSource) {
            this.awsOrganizationsSource = awsOrganizationsSource != null ? awsOrganizationsSource.build() : null;
        }

        @Override
        public final Builder awsOrganizationsSource(ResourceDataSyncAwsOrganizationsSource awsOrganizationsSource) {
            this.awsOrganizationsSource = awsOrganizationsSource;
            return this;
        }

        public final Collection<String> getSourceRegions() {
            if (sourceRegions instanceof SdkAutoConstructList) {
                return null;
            }
            return sourceRegions;
        }

        public final void setSourceRegions(Collection<String> sourceRegions) {
            this.sourceRegions = ResourceDataSyncSourceRegionListCopier.copy(sourceRegions);
        }

        @Override
        public final Builder sourceRegions(Collection<String> sourceRegions) {
            this.sourceRegions = ResourceDataSyncSourceRegionListCopier.copy(sourceRegions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder sourceRegions(String... sourceRegions) {
            sourceRegions(Arrays.asList(sourceRegions));
            return this;
        }

        public final Boolean getIncludeFutureRegions() {
            return includeFutureRegions;
        }

        public final void setIncludeFutureRegions(Boolean includeFutureRegions) {
            this.includeFutureRegions = includeFutureRegions;
        }

        @Override
        public final Builder includeFutureRegions(Boolean includeFutureRegions) {
            this.includeFutureRegions = includeFutureRegions;
            return this;
        }

        public final String getState() {
            return state;
        }

        public final void setState(String state) {
            this.state = state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        public final Boolean getEnableAllOpsDataSources() {
            return enableAllOpsDataSources;
        }

        public final void setEnableAllOpsDataSources(Boolean enableAllOpsDataSources) {
            this.enableAllOpsDataSources = enableAllOpsDataSources;
        }

        @Override
        public final Builder enableAllOpsDataSources(Boolean enableAllOpsDataSources) {
            this.enableAllOpsDataSources = enableAllOpsDataSources;
            return this;
        }

        @Override
        public ResourceDataSyncSourceWithState build() {
            return new ResourceDataSyncSourceWithState(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
