/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes an association of a Amazon Web Services Systems Manager document (SSM document) and a managed node.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Association implements SdkPojo, Serializable, ToCopyableBuilder<Association.Builder, Association> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(Association::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> INSTANCE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("InstanceId").getter(getter(Association::instanceId)).setter(setter(Builder::instanceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceId").build()).build();

    private static final SdkField<String> ASSOCIATION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AssociationId").getter(getter(Association::associationId)).setter(setter(Builder::associationId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AssociationId").build()).build();

    private static final SdkField<String> ASSOCIATION_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AssociationVersion").getter(getter(Association::associationVersion))
            .setter(setter(Builder::associationVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AssociationVersion").build())
            .build();

    private static final SdkField<String> DOCUMENT_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DocumentVersion").getter(getter(Association::documentVersion)).setter(setter(Builder::documentVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DocumentVersion").build()).build();

    private static final SdkField<List<Target>> TARGETS_FIELD = SdkField
            .<List<Target>> builder(MarshallingType.LIST)
            .memberName("Targets")
            .getter(getter(Association::targets))
            .setter(setter(Builder::targets))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Targets").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Target> builder(MarshallingType.SDK_POJO)
                                            .constructor(Target::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Instant> LAST_EXECUTION_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastExecutionDate").getter(getter(Association::lastExecutionDate))
            .setter(setter(Builder::lastExecutionDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastExecutionDate").build()).build();

    private static final SdkField<AssociationOverview> OVERVIEW_FIELD = SdkField
            .<AssociationOverview> builder(MarshallingType.SDK_POJO).memberName("Overview").getter(getter(Association::overview))
            .setter(setter(Builder::overview)).constructor(AssociationOverview::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Overview").build()).build();

    private static final SdkField<String> SCHEDULE_EXPRESSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ScheduleExpression").getter(getter(Association::scheduleExpression))
            .setter(setter(Builder::scheduleExpression))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ScheduleExpression").build())
            .build();

    private static final SdkField<String> ASSOCIATION_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AssociationName").getter(getter(Association::associationName)).setter(setter(Builder::associationName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AssociationName").build()).build();

    private static final SdkField<Integer> SCHEDULE_OFFSET_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("ScheduleOffset").getter(getter(Association::scheduleOffset)).setter(setter(Builder::scheduleOffset))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ScheduleOffset").build()).build();

    private static final SdkField<Integer> DURATION_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("Duration").getter(getter(Association::duration)).setter(setter(Builder::duration))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Duration").build()).build();

    private static final SdkField<List<Map<String, List<String>>>> TARGET_MAPS_FIELD = SdkField
            .<List<Map<String, List<String>>>> builder(MarshallingType.LIST)
            .memberName("TargetMaps")
            .getter(getter(Association::targetMaps))
            .setter(setter(Builder::targetMaps))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TargetMaps").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Map<String, List<String>>> builder(MarshallingType.MAP)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build(),
                                                    MapTrait.builder()
                                                            .keyLocationName("key")
                                                            .valueLocationName("value")
                                                            .valueFieldInfo(
                                                                    SdkField.<List<String>> builder(MarshallingType.LIST)
                                                                            .traits(LocationTrait.builder()
                                                                                    .location(MarshallLocation.PAYLOAD)
                                                                                    .locationName("value").build(),
                                                                                    ListTrait
                                                                                            .builder()
                                                                                            .memberLocationName(null)
                                                                                            .memberFieldInfo(
                                                                                                    SdkField.<String> builder(
                                                                                                            MarshallingType.STRING)
                                                                                                            .traits(LocationTrait
                                                                                                                    .builder()
                                                                                                                    .location(
                                                                                                                            MarshallLocation.PAYLOAD)
                                                                                                                    .locationName(
                                                                                                                            "member")
                                                                                                                    .build())
                                                                                                            .build()).build())
                                                                            .build()).build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, INSTANCE_ID_FIELD,
            ASSOCIATION_ID_FIELD, ASSOCIATION_VERSION_FIELD, DOCUMENT_VERSION_FIELD, TARGETS_FIELD, LAST_EXECUTION_DATE_FIELD,
            OVERVIEW_FIELD, SCHEDULE_EXPRESSION_FIELD, ASSOCIATION_NAME_FIELD, SCHEDULE_OFFSET_FIELD, DURATION_FIELD,
            TARGET_MAPS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("Name", NAME_FIELD);
                    put("InstanceId", INSTANCE_ID_FIELD);
                    put("AssociationId", ASSOCIATION_ID_FIELD);
                    put("AssociationVersion", ASSOCIATION_VERSION_FIELD);
                    put("DocumentVersion", DOCUMENT_VERSION_FIELD);
                    put("Targets", TARGETS_FIELD);
                    put("LastExecutionDate", LAST_EXECUTION_DATE_FIELD);
                    put("Overview", OVERVIEW_FIELD);
                    put("ScheduleExpression", SCHEDULE_EXPRESSION_FIELD);
                    put("AssociationName", ASSOCIATION_NAME_FIELD);
                    put("ScheduleOffset", SCHEDULE_OFFSET_FIELD);
                    put("Duration", DURATION_FIELD);
                    put("TargetMaps", TARGET_MAPS_FIELD);
                }
            });

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String instanceId;

    private final String associationId;

    private final String associationVersion;

    private final String documentVersion;

    private final List<Target> targets;

    private final Instant lastExecutionDate;

    private final AssociationOverview overview;

    private final String scheduleExpression;

    private final String associationName;

    private final Integer scheduleOffset;

    private final Integer duration;

    private final List<Map<String, List<String>>> targetMaps;

    private Association(BuilderImpl builder) {
        this.name = builder.name;
        this.instanceId = builder.instanceId;
        this.associationId = builder.associationId;
        this.associationVersion = builder.associationVersion;
        this.documentVersion = builder.documentVersion;
        this.targets = builder.targets;
        this.lastExecutionDate = builder.lastExecutionDate;
        this.overview = builder.overview;
        this.scheduleExpression = builder.scheduleExpression;
        this.associationName = builder.associationName;
        this.scheduleOffset = builder.scheduleOffset;
        this.duration = builder.duration;
        this.targetMaps = builder.targetMaps;
    }

    /**
     * <p>
     * The name of the SSM document.
     * </p>
     * 
     * @return The name of the SSM document.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The managed node ID.
     * </p>
     * 
     * @return The managed node ID.
     */
    public final String instanceId() {
        return instanceId;
    }

    /**
     * <p>
     * The ID created by the system when you create an association. An association is a binding between a document and a
     * set of targets with a schedule.
     * </p>
     * 
     * @return The ID created by the system when you create an association. An association is a binding between a
     *         document and a set of targets with a schedule.
     */
    public final String associationId() {
        return associationId;
    }

    /**
     * <p>
     * The association version.
     * </p>
     * 
     * @return The association version.
     */
    public final String associationVersion() {
        return associationVersion;
    }

    /**
     * <p>
     * The version of the document used in the association. If you change a document version for a State Manager
     * association, Systems Manager immediately runs the association unless you previously specifed the
     * <code>apply-only-at-cron-interval</code> parameter.
     * </p>
     * <important>
     * <p>
     * State Manager doesn't support running associations that use a new version of a document if that document is
     * shared from another account. State Manager always runs the <code>default</code> version of a document if shared
     * from another account, even though the Systems Manager console shows that a new version was processed. If you want
     * to run an association using a new version of a document shared form another account, you must set the document
     * version to <code>default</code>.
     * </p>
     * </important>
     * 
     * @return The version of the document used in the association. If you change a document version for a State Manager
     *         association, Systems Manager immediately runs the association unless you previously specifed the
     *         <code>apply-only-at-cron-interval</code> parameter.</p> <important>
     *         <p>
     *         State Manager doesn't support running associations that use a new version of a document if that document
     *         is shared from another account. State Manager always runs the <code>default</code> version of a document
     *         if shared from another account, even though the Systems Manager console shows that a new version was
     *         processed. If you want to run an association using a new version of a document shared form another
     *         account, you must set the document version to <code>default</code>.
     *         </p>
     */
    public final String documentVersion() {
        return documentVersion;
    }

    /**
     * For responses, this returns true if the service returned a value for the Targets property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasTargets() {
        return targets != null && !(targets instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The managed nodes targeted by the request to create an association. You can target all managed nodes in an Amazon
     * Web Services account by specifying the <code>InstanceIds</code> key with a value of <code>*</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTargets} method.
     * </p>
     * 
     * @return The managed nodes targeted by the request to create an association. You can target all managed nodes in
     *         an Amazon Web Services account by specifying the <code>InstanceIds</code> key with a value of
     *         <code>*</code>.
     */
    public final List<Target> targets() {
        return targets;
    }

    /**
     * <p>
     * The date on which the association was last run.
     * </p>
     * 
     * @return The date on which the association was last run.
     */
    public final Instant lastExecutionDate() {
        return lastExecutionDate;
    }

    /**
     * <p>
     * Information about the association.
     * </p>
     * 
     * @return Information about the association.
     */
    public final AssociationOverview overview() {
        return overview;
    }

    /**
     * <p>
     * A cron expression that specifies a schedule when the association runs. The schedule runs in Coordinated Universal
     * Time (UTC).
     * </p>
     * 
     * @return A cron expression that specifies a schedule when the association runs. The schedule runs in Coordinated
     *         Universal Time (UTC).
     */
    public final String scheduleExpression() {
        return scheduleExpression;
    }

    /**
     * <p>
     * The association name.
     * </p>
     * 
     * @return The association name.
     */
    public final String associationName() {
        return associationName;
    }

    /**
     * <p>
     * Number of days to wait after the scheduled day to run an association.
     * </p>
     * 
     * @return Number of days to wait after the scheduled day to run an association.
     */
    public final Integer scheduleOffset() {
        return scheduleOffset;
    }

    /**
     * <p>
     * The number of hours that an association can run on specified targets. After the resulting cutoff time passes,
     * associations that are currently running are cancelled, and no pending executions are started on remaining
     * targets.
     * </p>
     * 
     * @return The number of hours that an association can run on specified targets. After the resulting cutoff time
     *         passes, associations that are currently running are cancelled, and no pending executions are started on
     *         remaining targets.
     */
    public final Integer duration() {
        return duration;
    }

    /**
     * For responses, this returns true if the service returned a value for the TargetMaps property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasTargetMaps() {
        return targetMaps != null && !(targetMaps instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A key-value mapping of document parameters to target resources. Both Targets and TargetMaps can't be specified
     * together.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTargetMaps} method.
     * </p>
     * 
     * @return A key-value mapping of document parameters to target resources. Both Targets and TargetMaps can't be
     *         specified together.
     */
    public final List<Map<String, List<String>>> targetMaps() {
        return targetMaps;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(instanceId());
        hashCode = 31 * hashCode + Objects.hashCode(associationId());
        hashCode = 31 * hashCode + Objects.hashCode(associationVersion());
        hashCode = 31 * hashCode + Objects.hashCode(documentVersion());
        hashCode = 31 * hashCode + Objects.hashCode(hasTargets() ? targets() : null);
        hashCode = 31 * hashCode + Objects.hashCode(lastExecutionDate());
        hashCode = 31 * hashCode + Objects.hashCode(overview());
        hashCode = 31 * hashCode + Objects.hashCode(scheduleExpression());
        hashCode = 31 * hashCode + Objects.hashCode(associationName());
        hashCode = 31 * hashCode + Objects.hashCode(scheduleOffset());
        hashCode = 31 * hashCode + Objects.hashCode(duration());
        hashCode = 31 * hashCode + Objects.hashCode(hasTargetMaps() ? targetMaps() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Association)) {
            return false;
        }
        Association other = (Association) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(instanceId(), other.instanceId())
                && Objects.equals(associationId(), other.associationId())
                && Objects.equals(associationVersion(), other.associationVersion())
                && Objects.equals(documentVersion(), other.documentVersion()) && hasTargets() == other.hasTargets()
                && Objects.equals(targets(), other.targets()) && Objects.equals(lastExecutionDate(), other.lastExecutionDate())
                && Objects.equals(overview(), other.overview())
                && Objects.equals(scheduleExpression(), other.scheduleExpression())
                && Objects.equals(associationName(), other.associationName())
                && Objects.equals(scheduleOffset(), other.scheduleOffset()) && Objects.equals(duration(), other.duration())
                && hasTargetMaps() == other.hasTargetMaps() && Objects.equals(targetMaps(), other.targetMaps());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Association").add("Name", name()).add("InstanceId", instanceId())
                .add("AssociationId", associationId()).add("AssociationVersion", associationVersion())
                .add("DocumentVersion", documentVersion()).add("Targets", hasTargets() ? targets() : null)
                .add("LastExecutionDate", lastExecutionDate()).add("Overview", overview())
                .add("ScheduleExpression", scheduleExpression()).add("AssociationName", associationName())
                .add("ScheduleOffset", scheduleOffset()).add("Duration", duration())
                .add("TargetMaps", hasTargetMaps() ? targetMaps() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "InstanceId":
            return Optional.ofNullable(clazz.cast(instanceId()));
        case "AssociationId":
            return Optional.ofNullable(clazz.cast(associationId()));
        case "AssociationVersion":
            return Optional.ofNullable(clazz.cast(associationVersion()));
        case "DocumentVersion":
            return Optional.ofNullable(clazz.cast(documentVersion()));
        case "Targets":
            return Optional.ofNullable(clazz.cast(targets()));
        case "LastExecutionDate":
            return Optional.ofNullable(clazz.cast(lastExecutionDate()));
        case "Overview":
            return Optional.ofNullable(clazz.cast(overview()));
        case "ScheduleExpression":
            return Optional.ofNullable(clazz.cast(scheduleExpression()));
        case "AssociationName":
            return Optional.ofNullable(clazz.cast(associationName()));
        case "ScheduleOffset":
            return Optional.ofNullable(clazz.cast(scheduleOffset()));
        case "Duration":
            return Optional.ofNullable(clazz.cast(duration()));
        case "TargetMaps":
            return Optional.ofNullable(clazz.cast(targetMaps()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<Association, T> g) {
        return obj -> g.apply((Association) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Association> {
        /**
         * <p>
         * The name of the SSM document.
         * </p>
         * 
         * @param name
         *        The name of the SSM document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The managed node ID.
         * </p>
         * 
         * @param instanceId
         *        The managed node ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceId(String instanceId);

        /**
         * <p>
         * The ID created by the system when you create an association. An association is a binding between a document
         * and a set of targets with a schedule.
         * </p>
         * 
         * @param associationId
         *        The ID created by the system when you create an association. An association is a binding between a
         *        document and a set of targets with a schedule.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder associationId(String associationId);

        /**
         * <p>
         * The association version.
         * </p>
         * 
         * @param associationVersion
         *        The association version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder associationVersion(String associationVersion);

        /**
         * <p>
         * The version of the document used in the association. If you change a document version for a State Manager
         * association, Systems Manager immediately runs the association unless you previously specifed the
         * <code>apply-only-at-cron-interval</code> parameter.
         * </p>
         * <important>
         * <p>
         * State Manager doesn't support running associations that use a new version of a document if that document is
         * shared from another account. State Manager always runs the <code>default</code> version of a document if
         * shared from another account, even though the Systems Manager console shows that a new version was processed.
         * If you want to run an association using a new version of a document shared form another account, you must set
         * the document version to <code>default</code>.
         * </p>
         * </important>
         * 
         * @param documentVersion
         *        The version of the document used in the association. If you change a document version for a State
         *        Manager association, Systems Manager immediately runs the association unless you previously specifed
         *        the <code>apply-only-at-cron-interval</code> parameter.</p> <important>
         *        <p>
         *        State Manager doesn't support running associations that use a new version of a document if that
         *        document is shared from another account. State Manager always runs the <code>default</code> version of
         *        a document if shared from another account, even though the Systems Manager console shows that a new
         *        version was processed. If you want to run an association using a new version of a document shared form
         *        another account, you must set the document version to <code>default</code>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder documentVersion(String documentVersion);

        /**
         * <p>
         * The managed nodes targeted by the request to create an association. You can target all managed nodes in an
         * Amazon Web Services account by specifying the <code>InstanceIds</code> key with a value of <code>*</code>.
         * </p>
         * 
         * @param targets
         *        The managed nodes targeted by the request to create an association. You can target all managed nodes
         *        in an Amazon Web Services account by specifying the <code>InstanceIds</code> key with a value of
         *        <code>*</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targets(Collection<Target> targets);

        /**
         * <p>
         * The managed nodes targeted by the request to create an association. You can target all managed nodes in an
         * Amazon Web Services account by specifying the <code>InstanceIds</code> key with a value of <code>*</code>.
         * </p>
         * 
         * @param targets
         *        The managed nodes targeted by the request to create an association. You can target all managed nodes
         *        in an Amazon Web Services account by specifying the <code>InstanceIds</code> key with a value of
         *        <code>*</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targets(Target... targets);

        /**
         * <p>
         * The managed nodes targeted by the request to create an association. You can target all managed nodes in an
         * Amazon Web Services account by specifying the <code>InstanceIds</code> key with a value of <code>*</code>.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ssm.model.Target.Builder} avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.ssm.model.Target#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link software.amazon.awssdk.services.ssm.model.Target.Builder#build()}
         * is called immediately and its result is passed to {@link #targets(List<Target>)}.
         * 
         * @param targets
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.ssm.model.Target.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #targets(java.util.Collection<Target>)
         */
        Builder targets(Consumer<Target.Builder>... targets);

        /**
         * <p>
         * The date on which the association was last run.
         * </p>
         * 
         * @param lastExecutionDate
         *        The date on which the association was last run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastExecutionDate(Instant lastExecutionDate);

        /**
         * <p>
         * Information about the association.
         * </p>
         * 
         * @param overview
         *        Information about the association.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder overview(AssociationOverview overview);

        /**
         * <p>
         * Information about the association.
         * </p>
         * This is a convenience method that creates an instance of the {@link AssociationOverview.Builder} avoiding the
         * need to create one manually via {@link AssociationOverview#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AssociationOverview.Builder#build()} is called immediately and
         * its result is passed to {@link #overview(AssociationOverview)}.
         * 
         * @param overview
         *        a consumer that will call methods on {@link AssociationOverview.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #overview(AssociationOverview)
         */
        default Builder overview(Consumer<AssociationOverview.Builder> overview) {
            return overview(AssociationOverview.builder().applyMutation(overview).build());
        }

        /**
         * <p>
         * A cron expression that specifies a schedule when the association runs. The schedule runs in Coordinated
         * Universal Time (UTC).
         * </p>
         * 
         * @param scheduleExpression
         *        A cron expression that specifies a schedule when the association runs. The schedule runs in
         *        Coordinated Universal Time (UTC).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scheduleExpression(String scheduleExpression);

        /**
         * <p>
         * The association name.
         * </p>
         * 
         * @param associationName
         *        The association name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder associationName(String associationName);

        /**
         * <p>
         * Number of days to wait after the scheduled day to run an association.
         * </p>
         * 
         * @param scheduleOffset
         *        Number of days to wait after the scheduled day to run an association.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scheduleOffset(Integer scheduleOffset);

        /**
         * <p>
         * The number of hours that an association can run on specified targets. After the resulting cutoff time passes,
         * associations that are currently running are cancelled, and no pending executions are started on remaining
         * targets.
         * </p>
         * 
         * @param duration
         *        The number of hours that an association can run on specified targets. After the resulting cutoff time
         *        passes, associations that are currently running are cancelled, and no pending executions are started
         *        on remaining targets.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder duration(Integer duration);

        /**
         * <p>
         * A key-value mapping of document parameters to target resources. Both Targets and TargetMaps can't be
         * specified together.
         * </p>
         * 
         * @param targetMaps
         *        A key-value mapping of document parameters to target resources. Both Targets and TargetMaps can't be
         *        specified together.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetMaps(Collection<? extends Map<String, ? extends Collection<String>>> targetMaps);

        /**
         * <p>
         * A key-value mapping of document parameters to target resources. Both Targets and TargetMaps can't be
         * specified together.
         * </p>
         * 
         * @param targetMaps
         *        A key-value mapping of document parameters to target resources. Both Targets and TargetMaps can't be
         *        specified together.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetMaps(Map<String, ? extends Collection<String>>... targetMaps);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String instanceId;

        private String associationId;

        private String associationVersion;

        private String documentVersion;

        private List<Target> targets = DefaultSdkAutoConstructList.getInstance();

        private Instant lastExecutionDate;

        private AssociationOverview overview;

        private String scheduleExpression;

        private String associationName;

        private Integer scheduleOffset;

        private Integer duration;

        private List<Map<String, List<String>>> targetMaps = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Association model) {
            name(model.name);
            instanceId(model.instanceId);
            associationId(model.associationId);
            associationVersion(model.associationVersion);
            documentVersion(model.documentVersion);
            targets(model.targets);
            lastExecutionDate(model.lastExecutionDate);
            overview(model.overview);
            scheduleExpression(model.scheduleExpression);
            associationName(model.associationName);
            scheduleOffset(model.scheduleOffset);
            duration(model.duration);
            targetMaps(model.targetMaps);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getInstanceId() {
            return instanceId;
        }

        public final void setInstanceId(String instanceId) {
            this.instanceId = instanceId;
        }

        @Override
        public final Builder instanceId(String instanceId) {
            this.instanceId = instanceId;
            return this;
        }

        public final String getAssociationId() {
            return associationId;
        }

        public final void setAssociationId(String associationId) {
            this.associationId = associationId;
        }

        @Override
        public final Builder associationId(String associationId) {
            this.associationId = associationId;
            return this;
        }

        public final String getAssociationVersion() {
            return associationVersion;
        }

        public final void setAssociationVersion(String associationVersion) {
            this.associationVersion = associationVersion;
        }

        @Override
        public final Builder associationVersion(String associationVersion) {
            this.associationVersion = associationVersion;
            return this;
        }

        public final String getDocumentVersion() {
            return documentVersion;
        }

        public final void setDocumentVersion(String documentVersion) {
            this.documentVersion = documentVersion;
        }

        @Override
        public final Builder documentVersion(String documentVersion) {
            this.documentVersion = documentVersion;
            return this;
        }

        public final List<Target.Builder> getTargets() {
            List<Target.Builder> result = TargetsCopier.copyToBuilder(this.targets);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTargets(Collection<Target.BuilderImpl> targets) {
            this.targets = TargetsCopier.copyFromBuilder(targets);
        }

        @Override
        public final Builder targets(Collection<Target> targets) {
            this.targets = TargetsCopier.copy(targets);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder targets(Target... targets) {
            targets(Arrays.asList(targets));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder targets(Consumer<Target.Builder>... targets) {
            targets(Stream.of(targets).map(c -> Target.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final Instant getLastExecutionDate() {
            return lastExecutionDate;
        }

        public final void setLastExecutionDate(Instant lastExecutionDate) {
            this.lastExecutionDate = lastExecutionDate;
        }

        @Override
        public final Builder lastExecutionDate(Instant lastExecutionDate) {
            this.lastExecutionDate = lastExecutionDate;
            return this;
        }

        public final AssociationOverview.Builder getOverview() {
            return overview != null ? overview.toBuilder() : null;
        }

        public final void setOverview(AssociationOverview.BuilderImpl overview) {
            this.overview = overview != null ? overview.build() : null;
        }

        @Override
        public final Builder overview(AssociationOverview overview) {
            this.overview = overview;
            return this;
        }

        public final String getScheduleExpression() {
            return scheduleExpression;
        }

        public final void setScheduleExpression(String scheduleExpression) {
            this.scheduleExpression = scheduleExpression;
        }

        @Override
        public final Builder scheduleExpression(String scheduleExpression) {
            this.scheduleExpression = scheduleExpression;
            return this;
        }

        public final String getAssociationName() {
            return associationName;
        }

        public final void setAssociationName(String associationName) {
            this.associationName = associationName;
        }

        @Override
        public final Builder associationName(String associationName) {
            this.associationName = associationName;
            return this;
        }

        public final Integer getScheduleOffset() {
            return scheduleOffset;
        }

        public final void setScheduleOffset(Integer scheduleOffset) {
            this.scheduleOffset = scheduleOffset;
        }

        @Override
        public final Builder scheduleOffset(Integer scheduleOffset) {
            this.scheduleOffset = scheduleOffset;
            return this;
        }

        public final Integer getDuration() {
            return duration;
        }

        public final void setDuration(Integer duration) {
            this.duration = duration;
        }

        @Override
        public final Builder duration(Integer duration) {
            this.duration = duration;
            return this;
        }

        public final Collection<? extends Map<String, ? extends Collection<String>>> getTargetMaps() {
            if (targetMaps instanceof SdkAutoConstructList) {
                return null;
            }
            return targetMaps;
        }

        public final void setTargetMaps(Collection<? extends Map<String, ? extends Collection<String>>> targetMaps) {
            this.targetMaps = TargetMapsCopier.copy(targetMaps);
        }

        @Override
        public final Builder targetMaps(Collection<? extends Map<String, ? extends Collection<String>>> targetMaps) {
            this.targetMaps = TargetMapsCopier.copy(targetMaps);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder targetMaps(Map<String, ? extends Collection<String>>... targetMaps) {
            targetMaps(Arrays.asList(targetMaps));
            return this;
        }

        @Override
        public Association build() {
            return new Association(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
