/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about the patches to use to update the managed nodes, including target operating systems and source
 * repository. Applies to Linux managed nodes only.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class PatchSource implements SdkPojo, Serializable, ToCopyableBuilder<PatchSource.Builder, PatchSource> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(PatchSource::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<List<String>> PRODUCTS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Products")
            .getter(getter(PatchSource::products))
            .setter(setter(Builder::products))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Products").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> CONFIGURATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Configuration").getter(getter(PatchSource::configuration)).setter(setter(Builder::configuration))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Configuration").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, PRODUCTS_FIELD,
            CONFIGURATION_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String name;

    private final List<String> products;

    private final String configuration;

    private PatchSource(BuilderImpl builder) {
        this.name = builder.name;
        this.products = builder.products;
        this.configuration = builder.configuration;
    }

    /**
     * <p>
     * The name specified to identify the patch source.
     * </p>
     * 
     * @return The name specified to identify the patch source.
     */
    public final String name() {
        return name;
    }

    /**
     * For responses, this returns true if the service returned a value for the Products property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasProducts() {
        return products != null && !(products instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The specific operating system versions a patch repository applies to, such as "Ubuntu16.04",
     * "AmazonLinux2016.09", "RedhatEnterpriseLinux7.2" or "Suse12.7". For lists of supported product values, see
     * <a>PatchFilter</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasProducts} method.
     * </p>
     * 
     * @return The specific operating system versions a patch repository applies to, such as "Ubuntu16.04",
     *         "AmazonLinux2016.09", "RedhatEnterpriseLinux7.2" or "Suse12.7". For lists of supported product values,
     *         see <a>PatchFilter</a>.
     */
    public final List<String> products() {
        return products;
    }

    /**
     * <p>
     * The value of the repo configuration.
     * </p>
     * <p>
     * <b>Example for yum repositories</b>
     * </p>
     * <p>
     * <code>[main]</code>
     * </p>
     * <p>
     * <code>name=MyCustomRepository</code>
     * </p>
     * <p>
     * <code>baseurl=https://my-custom-repository</code>
     * </p>
     * <p>
     * <code>enabled=1</code>
     * </p>
     * <p>
     * For information about other options available for your yum repository configuration, see <a
     * href="https://man7.org/linux/man-pages/man5/dnf.conf.5.html">dnf.conf(5)</a> on the <i>man7.org</i> website.
     * </p>
     * <p>
     * <b>Examples for Ubuntu Server and Debian Server</b>
     * </p>
     * <p>
     * <code>deb http://security.ubuntu.com/ubuntu jammy main</code>
     * </p>
     * <p>
     * <code>deb https://site.example.com/debian distribution component1 component2 component3</code>
     * </p>
     * <p>
     * Repo information for Ubuntu Server repositories must be specifed in a single line. For more examples and
     * information, see <a href="https://manpages.ubuntu.com/manpages/jammy/man5/sources.list.5.html">jammy (5)
     * sources.list.5.gz</a> on the <i>Ubuntu Server Manuals</i> website and <a
     * href="https://wiki.debian.org/SourcesList#sources.list_format">sources.list format</a> on the <i>Debian Wiki</i>.
     * </p>
     * 
     * @return The value of the repo configuration.</p>
     *         <p>
     *         <b>Example for yum repositories</b>
     *         </p>
     *         <p>
     *         <code>[main]</code>
     *         </p>
     *         <p>
     *         <code>name=MyCustomRepository</code>
     *         </p>
     *         <p>
     *         <code>baseurl=https://my-custom-repository</code>
     *         </p>
     *         <p>
     *         <code>enabled=1</code>
     *         </p>
     *         <p>
     *         For information about other options available for your yum repository configuration, see <a
     *         href="https://man7.org/linux/man-pages/man5/dnf.conf.5.html">dnf.conf(5)</a> on the <i>man7.org</i>
     *         website.
     *         </p>
     *         <p>
     *         <b>Examples for Ubuntu Server and Debian Server</b>
     *         </p>
     *         <p>
     *         <code>deb http://security.ubuntu.com/ubuntu jammy main</code>
     *         </p>
     *         <p>
     *         <code>deb https://site.example.com/debian distribution component1 component2 component3</code>
     *         </p>
     *         <p>
     *         Repo information for Ubuntu Server repositories must be specifed in a single line. For more examples and
     *         information, see <a href="https://manpages.ubuntu.com/manpages/jammy/man5/sources.list.5.html">jammy (5)
     *         sources.list.5.gz</a> on the <i>Ubuntu Server Manuals</i> website and <a
     *         href="https://wiki.debian.org/SourcesList#sources.list_format">sources.list format</a> on the <i>Debian
     *         Wiki</i>.
     */
    public final String configuration() {
        return configuration;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(hasProducts() ? products() : null);
        hashCode = 31 * hashCode + Objects.hashCode(configuration());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PatchSource)) {
            return false;
        }
        PatchSource other = (PatchSource) obj;
        return Objects.equals(name(), other.name()) && hasProducts() == other.hasProducts()
                && Objects.equals(products(), other.products()) && Objects.equals(configuration(), other.configuration());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("PatchSource").add("Name", name()).add("Products", hasProducts() ? products() : null)
                .add("Configuration", configuration() == null ? null : "*** Sensitive Data Redacted ***").build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Products":
            return Optional.ofNullable(clazz.cast(products()));
        case "Configuration":
            return Optional.ofNullable(clazz.cast(configuration()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Name", NAME_FIELD);
        map.put("Products", PRODUCTS_FIELD);
        map.put("Configuration", CONFIGURATION_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<PatchSource, T> g) {
        return obj -> g.apply((PatchSource) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, PatchSource> {
        /**
         * <p>
         * The name specified to identify the patch source.
         * </p>
         * 
         * @param name
         *        The name specified to identify the patch source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The specific operating system versions a patch repository applies to, such as "Ubuntu16.04",
         * "AmazonLinux2016.09", "RedhatEnterpriseLinux7.2" or "Suse12.7". For lists of supported product values, see
         * <a>PatchFilter</a>.
         * </p>
         * 
         * @param products
         *        The specific operating system versions a patch repository applies to, such as "Ubuntu16.04",
         *        "AmazonLinux2016.09", "RedhatEnterpriseLinux7.2" or "Suse12.7". For lists of supported product values,
         *        see <a>PatchFilter</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder products(Collection<String> products);

        /**
         * <p>
         * The specific operating system versions a patch repository applies to, such as "Ubuntu16.04",
         * "AmazonLinux2016.09", "RedhatEnterpriseLinux7.2" or "Suse12.7". For lists of supported product values, see
         * <a>PatchFilter</a>.
         * </p>
         * 
         * @param products
         *        The specific operating system versions a patch repository applies to, such as "Ubuntu16.04",
         *        "AmazonLinux2016.09", "RedhatEnterpriseLinux7.2" or "Suse12.7". For lists of supported product values,
         *        see <a>PatchFilter</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder products(String... products);

        /**
         * <p>
         * The value of the repo configuration.
         * </p>
         * <p>
         * <b>Example for yum repositories</b>
         * </p>
         * <p>
         * <code>[main]</code>
         * </p>
         * <p>
         * <code>name=MyCustomRepository</code>
         * </p>
         * <p>
         * <code>baseurl=https://my-custom-repository</code>
         * </p>
         * <p>
         * <code>enabled=1</code>
         * </p>
         * <p>
         * For information about other options available for your yum repository configuration, see <a
         * href="https://man7.org/linux/man-pages/man5/dnf.conf.5.html">dnf.conf(5)</a> on the <i>man7.org</i> website.
         * </p>
         * <p>
         * <b>Examples for Ubuntu Server and Debian Server</b>
         * </p>
         * <p>
         * <code>deb http://security.ubuntu.com/ubuntu jammy main</code>
         * </p>
         * <p>
         * <code>deb https://site.example.com/debian distribution component1 component2 component3</code>
         * </p>
         * <p>
         * Repo information for Ubuntu Server repositories must be specifed in a single line. For more examples and
         * information, see <a href="https://manpages.ubuntu.com/manpages/jammy/man5/sources.list.5.html">jammy (5)
         * sources.list.5.gz</a> on the <i>Ubuntu Server Manuals</i> website and <a
         * href="https://wiki.debian.org/SourcesList#sources.list_format">sources.list format</a> on the <i>Debian
         * Wiki</i>.
         * </p>
         * 
         * @param configuration
         *        The value of the repo configuration.</p>
         *        <p>
         *        <b>Example for yum repositories</b>
         *        </p>
         *        <p>
         *        <code>[main]</code>
         *        </p>
         *        <p>
         *        <code>name=MyCustomRepository</code>
         *        </p>
         *        <p>
         *        <code>baseurl=https://my-custom-repository</code>
         *        </p>
         *        <p>
         *        <code>enabled=1</code>
         *        </p>
         *        <p>
         *        For information about other options available for your yum repository configuration, see <a
         *        href="https://man7.org/linux/man-pages/man5/dnf.conf.5.html">dnf.conf(5)</a> on the <i>man7.org</i>
         *        website.
         *        </p>
         *        <p>
         *        <b>Examples for Ubuntu Server and Debian Server</b>
         *        </p>
         *        <p>
         *        <code>deb http://security.ubuntu.com/ubuntu jammy main</code>
         *        </p>
         *        <p>
         *        <code>deb https://site.example.com/debian distribution component1 component2 component3</code>
         *        </p>
         *        <p>
         *        Repo information for Ubuntu Server repositories must be specifed in a single line. For more examples
         *        and information, see <a
         *        href="https://manpages.ubuntu.com/manpages/jammy/man5/sources.list.5.html">jammy (5)
         *        sources.list.5.gz</a> on the <i>Ubuntu Server Manuals</i> website and <a
         *        href="https://wiki.debian.org/SourcesList#sources.list_format">sources.list format</a> on the
         *        <i>Debian Wiki</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configuration(String configuration);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private List<String> products = DefaultSdkAutoConstructList.getInstance();

        private String configuration;

        private BuilderImpl() {
        }

        private BuilderImpl(PatchSource model) {
            name(model.name);
            products(model.products);
            configuration(model.configuration);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final Collection<String> getProducts() {
            if (products instanceof SdkAutoConstructList) {
                return null;
            }
            return products;
        }

        public final void setProducts(Collection<String> products) {
            this.products = PatchSourceProductListCopier.copy(products);
        }

        @Override
        public final Builder products(Collection<String> products) {
            this.products = PatchSourceProductListCopier.copy(products);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder products(String... products) {
            products(Arrays.asList(products));
            return this;
        }

        public final String getConfiguration() {
            return configuration;
        }

        public final void setConfiguration(String configuration) {
            this.configuration = configuration;
        }

        @Override
        public final Builder configuration(String configuration) {
            this.configuration = configuration;
            return this;
        }

        @Override
        public PatchSource build() {
            return new PatchSource(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
