/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The combination of Amazon Web Services Regions and Amazon Web Services accounts targeted by the current Automation
 * execution.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class TargetLocation implements SdkPojo, Serializable, ToCopyableBuilder<TargetLocation.Builder, TargetLocation> {
    private static final SdkField<List<String>> ACCOUNTS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Accounts")
            .getter(getter(TargetLocation::accounts))
            .setter(setter(Builder::accounts))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Accounts").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> REGIONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Regions")
            .getter(getter(TargetLocation::regions))
            .setter(setter(Builder::regions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Regions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> TARGET_LOCATION_MAX_CONCURRENCY_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("TargetLocationMaxConcurrency")
            .getter(getter(TargetLocation::targetLocationMaxConcurrency))
            .setter(setter(Builder::targetLocationMaxConcurrency))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TargetLocationMaxConcurrency")
                    .build()).build();

    private static final SdkField<String> TARGET_LOCATION_MAX_ERRORS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TargetLocationMaxErrors").getter(getter(TargetLocation::targetLocationMaxErrors))
            .setter(setter(Builder::targetLocationMaxErrors))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TargetLocationMaxErrors").build())
            .build();

    private static final SdkField<String> EXECUTION_ROLE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ExecutionRoleName").getter(getter(TargetLocation::executionRoleName))
            .setter(setter(Builder::executionRoleName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExecutionRoleName").build()).build();

    private static final SdkField<AlarmConfiguration> TARGET_LOCATION_ALARM_CONFIGURATION_FIELD = SdkField
            .<AlarmConfiguration> builder(MarshallingType.SDK_POJO)
            .memberName("TargetLocationAlarmConfiguration")
            .getter(getter(TargetLocation::targetLocationAlarmConfiguration))
            .setter(setter(Builder::targetLocationAlarmConfiguration))
            .constructor(AlarmConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TargetLocationAlarmConfiguration")
                    .build()).build();

    private static final SdkField<Boolean> INCLUDE_CHILD_ORGANIZATION_UNITS_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("IncludeChildOrganizationUnits")
            .getter(getter(TargetLocation::includeChildOrganizationUnits))
            .setter(setter(Builder::includeChildOrganizationUnits))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IncludeChildOrganizationUnits")
                    .build()).build();

    private static final SdkField<List<String>> EXCLUDE_ACCOUNTS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ExcludeAccounts")
            .getter(getter(TargetLocation::excludeAccounts))
            .setter(setter(Builder::excludeAccounts))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExcludeAccounts").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<Target>> TARGETS_FIELD = SdkField
            .<List<Target>> builder(MarshallingType.LIST)
            .memberName("Targets")
            .getter(getter(TargetLocation::targets))
            .setter(setter(Builder::targets))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Targets").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Target> builder(MarshallingType.SDK_POJO)
                                            .constructor(Target::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> TARGETS_MAX_CONCURRENCY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TargetsMaxConcurrency").getter(getter(TargetLocation::targetsMaxConcurrency))
            .setter(setter(Builder::targetsMaxConcurrency))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TargetsMaxConcurrency").build())
            .build();

    private static final SdkField<String> TARGETS_MAX_ERRORS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TargetsMaxErrors").getter(getter(TargetLocation::targetsMaxErrors))
            .setter(setter(Builder::targetsMaxErrors))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TargetsMaxErrors").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ACCOUNTS_FIELD, REGIONS_FIELD,
            TARGET_LOCATION_MAX_CONCURRENCY_FIELD, TARGET_LOCATION_MAX_ERRORS_FIELD, EXECUTION_ROLE_NAME_FIELD,
            TARGET_LOCATION_ALARM_CONFIGURATION_FIELD, INCLUDE_CHILD_ORGANIZATION_UNITS_FIELD, EXCLUDE_ACCOUNTS_FIELD,
            TARGETS_FIELD, TARGETS_MAX_CONCURRENCY_FIELD, TARGETS_MAX_ERRORS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final List<String> accounts;

    private final List<String> regions;

    private final String targetLocationMaxConcurrency;

    private final String targetLocationMaxErrors;

    private final String executionRoleName;

    private final AlarmConfiguration targetLocationAlarmConfiguration;

    private final Boolean includeChildOrganizationUnits;

    private final List<String> excludeAccounts;

    private final List<Target> targets;

    private final String targetsMaxConcurrency;

    private final String targetsMaxErrors;

    private TargetLocation(BuilderImpl builder) {
        this.accounts = builder.accounts;
        this.regions = builder.regions;
        this.targetLocationMaxConcurrency = builder.targetLocationMaxConcurrency;
        this.targetLocationMaxErrors = builder.targetLocationMaxErrors;
        this.executionRoleName = builder.executionRoleName;
        this.targetLocationAlarmConfiguration = builder.targetLocationAlarmConfiguration;
        this.includeChildOrganizationUnits = builder.includeChildOrganizationUnits;
        this.excludeAccounts = builder.excludeAccounts;
        this.targets = builder.targets;
        this.targetsMaxConcurrency = builder.targetsMaxConcurrency;
        this.targetsMaxErrors = builder.targetsMaxErrors;
    }

    /**
     * For responses, this returns true if the service returned a value for the Accounts property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasAccounts() {
        return accounts != null && !(accounts instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The Amazon Web Services accounts targeted by the current Automation execution.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAccounts} method.
     * </p>
     * 
     * @return The Amazon Web Services accounts targeted by the current Automation execution.
     */
    public final List<String> accounts() {
        return accounts;
    }

    /**
     * For responses, this returns true if the service returned a value for the Regions property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasRegions() {
        return regions != null && !(regions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The Amazon Web Services Regions targeted by the current Automation execution.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRegions} method.
     * </p>
     * 
     * @return The Amazon Web Services Regions targeted by the current Automation execution.
     */
    public final List<String> regions() {
        return regions;
    }

    /**
     * <p>
     * The maximum number of Amazon Web Services Regions and Amazon Web Services accounts allowed to run the Automation
     * concurrently. <code>TargetLocationMaxConcurrency</code> has a default value of 1.
     * </p>
     * 
     * @return The maximum number of Amazon Web Services Regions and Amazon Web Services accounts allowed to run the
     *         Automation concurrently. <code>TargetLocationMaxConcurrency</code> has a default value of 1.
     */
    public final String targetLocationMaxConcurrency() {
        return targetLocationMaxConcurrency;
    }

    /**
     * <p>
     * The maximum number of errors allowed before the system stops queueing additional Automation executions for the
     * currently running Automation. <code>TargetLocationMaxErrors</code> has a default value of 0.
     * </p>
     * 
     * @return The maximum number of errors allowed before the system stops queueing additional Automation executions
     *         for the currently running Automation. <code>TargetLocationMaxErrors</code> has a default value of 0.
     */
    public final String targetLocationMaxErrors() {
        return targetLocationMaxErrors;
    }

    /**
     * <p>
     * The Automation execution role used by the currently running Automation. If not specified, the default value is
     * <code>AWS-SystemsManager-AutomationExecutionRole</code>.
     * </p>
     * 
     * @return The Automation execution role used by the currently running Automation. If not specified, the default
     *         value is <code>AWS-SystemsManager-AutomationExecutionRole</code>.
     */
    public final String executionRoleName() {
        return executionRoleName;
    }

    /**
     * Returns the value of the TargetLocationAlarmConfiguration property for this object.
     * 
     * @return The value of the TargetLocationAlarmConfiguration property for this object.
     */
    public final AlarmConfiguration targetLocationAlarmConfiguration() {
        return targetLocationAlarmConfiguration;
    }

    /**
     * <p>
     * Indicates whether to include child organizational units (OUs) that are children of the targeted OUs. The default
     * is <code>false</code>.
     * </p>
     * <note>
     * <p>
     * This parameter is not supported by State Manager.
     * </p>
     * </note>
     * 
     * @return Indicates whether to include child organizational units (OUs) that are children of the targeted OUs. The
     *         default is <code>false</code>.</p> <note>
     *         <p>
     *         This parameter is not supported by State Manager.
     *         </p>
     */
    public final Boolean includeChildOrganizationUnits() {
        return includeChildOrganizationUnits;
    }

    /**
     * For responses, this returns true if the service returned a value for the ExcludeAccounts property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasExcludeAccounts() {
        return excludeAccounts != null && !(excludeAccounts instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Amazon Web Services accounts or organizational units to exclude as expanded targets.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasExcludeAccounts} method.
     * </p>
     * 
     * @return Amazon Web Services accounts or organizational units to exclude as expanded targets.
     */
    public final List<String> excludeAccounts() {
        return excludeAccounts;
    }

    /**
     * For responses, this returns true if the service returned a value for the Targets property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasTargets() {
        return targets != null && !(targets instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of key-value mappings to target resources. If you specify values for this data type, you must also specify
     * a value for <code>TargetParameterName</code>.
     * </p>
     * <p>
     * This <code>Targets</code> parameter takes precedence over the <code>StartAutomationExecution:Targets</code>
     * parameter if both are supplied.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTargets} method.
     * </p>
     * 
     * @return A list of key-value mappings to target resources. If you specify values for this data type, you must also
     *         specify a value for <code>TargetParameterName</code>.</p>
     *         <p>
     *         This <code>Targets</code> parameter takes precedence over the
     *         <code>StartAutomationExecution:Targets</code> parameter if both are supplied.
     */
    public final List<Target> targets() {
        return targets;
    }

    /**
     * <p>
     * The maximum number of targets allowed to run this task in parallel. This <code>TargetsMaxConcurrency</code> takes
     * precedence over the <code>StartAutomationExecution:MaxConcurrency</code> parameter if both are supplied.
     * </p>
     * 
     * @return The maximum number of targets allowed to run this task in parallel. This
     *         <code>TargetsMaxConcurrency</code> takes precedence over the
     *         <code>StartAutomationExecution:MaxConcurrency</code> parameter if both are supplied.
     */
    public final String targetsMaxConcurrency() {
        return targetsMaxConcurrency;
    }

    /**
     * <p>
     * The maximum number of errors that are allowed before the system stops running the automation on additional
     * targets. This <code>TargetsMaxErrors</code> parameter takes precedence over the
     * <code>StartAutomationExecution:MaxErrors</code> parameter if both are supplied.
     * </p>
     * 
     * @return The maximum number of errors that are allowed before the system stops running the automation on
     *         additional targets. This <code>TargetsMaxErrors</code> parameter takes precedence over the
     *         <code>StartAutomationExecution:MaxErrors</code> parameter if both are supplied.
     */
    public final String targetsMaxErrors() {
        return targetsMaxErrors;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasAccounts() ? accounts() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasRegions() ? regions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(targetLocationMaxConcurrency());
        hashCode = 31 * hashCode + Objects.hashCode(targetLocationMaxErrors());
        hashCode = 31 * hashCode + Objects.hashCode(executionRoleName());
        hashCode = 31 * hashCode + Objects.hashCode(targetLocationAlarmConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(includeChildOrganizationUnits());
        hashCode = 31 * hashCode + Objects.hashCode(hasExcludeAccounts() ? excludeAccounts() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasTargets() ? targets() : null);
        hashCode = 31 * hashCode + Objects.hashCode(targetsMaxConcurrency());
        hashCode = 31 * hashCode + Objects.hashCode(targetsMaxErrors());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof TargetLocation)) {
            return false;
        }
        TargetLocation other = (TargetLocation) obj;
        return hasAccounts() == other.hasAccounts() && Objects.equals(accounts(), other.accounts())
                && hasRegions() == other.hasRegions() && Objects.equals(regions(), other.regions())
                && Objects.equals(targetLocationMaxConcurrency(), other.targetLocationMaxConcurrency())
                && Objects.equals(targetLocationMaxErrors(), other.targetLocationMaxErrors())
                && Objects.equals(executionRoleName(), other.executionRoleName())
                && Objects.equals(targetLocationAlarmConfiguration(), other.targetLocationAlarmConfiguration())
                && Objects.equals(includeChildOrganizationUnits(), other.includeChildOrganizationUnits())
                && hasExcludeAccounts() == other.hasExcludeAccounts()
                && Objects.equals(excludeAccounts(), other.excludeAccounts()) && hasTargets() == other.hasTargets()
                && Objects.equals(targets(), other.targets())
                && Objects.equals(targetsMaxConcurrency(), other.targetsMaxConcurrency())
                && Objects.equals(targetsMaxErrors(), other.targetsMaxErrors());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("TargetLocation").add("Accounts", hasAccounts() ? accounts() : null)
                .add("Regions", hasRegions() ? regions() : null)
                .add("TargetLocationMaxConcurrency", targetLocationMaxConcurrency())
                .add("TargetLocationMaxErrors", targetLocationMaxErrors()).add("ExecutionRoleName", executionRoleName())
                .add("TargetLocationAlarmConfiguration", targetLocationAlarmConfiguration())
                .add("IncludeChildOrganizationUnits", includeChildOrganizationUnits())
                .add("ExcludeAccounts", hasExcludeAccounts() ? excludeAccounts() : null)
                .add("Targets", hasTargets() ? targets() : null).add("TargetsMaxConcurrency", targetsMaxConcurrency())
                .add("TargetsMaxErrors", targetsMaxErrors()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Accounts":
            return Optional.ofNullable(clazz.cast(accounts()));
        case "Regions":
            return Optional.ofNullable(clazz.cast(regions()));
        case "TargetLocationMaxConcurrency":
            return Optional.ofNullable(clazz.cast(targetLocationMaxConcurrency()));
        case "TargetLocationMaxErrors":
            return Optional.ofNullable(clazz.cast(targetLocationMaxErrors()));
        case "ExecutionRoleName":
            return Optional.ofNullable(clazz.cast(executionRoleName()));
        case "TargetLocationAlarmConfiguration":
            return Optional.ofNullable(clazz.cast(targetLocationAlarmConfiguration()));
        case "IncludeChildOrganizationUnits":
            return Optional.ofNullable(clazz.cast(includeChildOrganizationUnits()));
        case "ExcludeAccounts":
            return Optional.ofNullable(clazz.cast(excludeAccounts()));
        case "Targets":
            return Optional.ofNullable(clazz.cast(targets()));
        case "TargetsMaxConcurrency":
            return Optional.ofNullable(clazz.cast(targetsMaxConcurrency()));
        case "TargetsMaxErrors":
            return Optional.ofNullable(clazz.cast(targetsMaxErrors()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Accounts", ACCOUNTS_FIELD);
        map.put("Regions", REGIONS_FIELD);
        map.put("TargetLocationMaxConcurrency", TARGET_LOCATION_MAX_CONCURRENCY_FIELD);
        map.put("TargetLocationMaxErrors", TARGET_LOCATION_MAX_ERRORS_FIELD);
        map.put("ExecutionRoleName", EXECUTION_ROLE_NAME_FIELD);
        map.put("TargetLocationAlarmConfiguration", TARGET_LOCATION_ALARM_CONFIGURATION_FIELD);
        map.put("IncludeChildOrganizationUnits", INCLUDE_CHILD_ORGANIZATION_UNITS_FIELD);
        map.put("ExcludeAccounts", EXCLUDE_ACCOUNTS_FIELD);
        map.put("Targets", TARGETS_FIELD);
        map.put("TargetsMaxConcurrency", TARGETS_MAX_CONCURRENCY_FIELD);
        map.put("TargetsMaxErrors", TARGETS_MAX_ERRORS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<TargetLocation, T> g) {
        return obj -> g.apply((TargetLocation) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, TargetLocation> {
        /**
         * <p>
         * The Amazon Web Services accounts targeted by the current Automation execution.
         * </p>
         * 
         * @param accounts
         *        The Amazon Web Services accounts targeted by the current Automation execution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accounts(Collection<String> accounts);

        /**
         * <p>
         * The Amazon Web Services accounts targeted by the current Automation execution.
         * </p>
         * 
         * @param accounts
         *        The Amazon Web Services accounts targeted by the current Automation execution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accounts(String... accounts);

        /**
         * <p>
         * The Amazon Web Services Regions targeted by the current Automation execution.
         * </p>
         * 
         * @param regions
         *        The Amazon Web Services Regions targeted by the current Automation execution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder regions(Collection<String> regions);

        /**
         * <p>
         * The Amazon Web Services Regions targeted by the current Automation execution.
         * </p>
         * 
         * @param regions
         *        The Amazon Web Services Regions targeted by the current Automation execution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder regions(String... regions);

        /**
         * <p>
         * The maximum number of Amazon Web Services Regions and Amazon Web Services accounts allowed to run the
         * Automation concurrently. <code>TargetLocationMaxConcurrency</code> has a default value of 1.
         * </p>
         * 
         * @param targetLocationMaxConcurrency
         *        The maximum number of Amazon Web Services Regions and Amazon Web Services accounts allowed to run the
         *        Automation concurrently. <code>TargetLocationMaxConcurrency</code> has a default value of 1.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetLocationMaxConcurrency(String targetLocationMaxConcurrency);

        /**
         * <p>
         * The maximum number of errors allowed before the system stops queueing additional Automation executions for
         * the currently running Automation. <code>TargetLocationMaxErrors</code> has a default value of 0.
         * </p>
         * 
         * @param targetLocationMaxErrors
         *        The maximum number of errors allowed before the system stops queueing additional Automation executions
         *        for the currently running Automation. <code>TargetLocationMaxErrors</code> has a default value of 0.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetLocationMaxErrors(String targetLocationMaxErrors);

        /**
         * <p>
         * The Automation execution role used by the currently running Automation. If not specified, the default value
         * is <code>AWS-SystemsManager-AutomationExecutionRole</code>.
         * </p>
         * 
         * @param executionRoleName
         *        The Automation execution role used by the currently running Automation. If not specified, the default
         *        value is <code>AWS-SystemsManager-AutomationExecutionRole</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder executionRoleName(String executionRoleName);

        /**
         * Sets the value of the TargetLocationAlarmConfiguration property for this object.
         *
         * @param targetLocationAlarmConfiguration
         *        The new value for the TargetLocationAlarmConfiguration property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetLocationAlarmConfiguration(AlarmConfiguration targetLocationAlarmConfiguration);

        /**
         * Sets the value of the TargetLocationAlarmConfiguration property for this object.
         *
         * This is a convenience method that creates an instance of the {@link AlarmConfiguration.Builder} avoiding the
         * need to create one manually via {@link AlarmConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AlarmConfiguration.Builder#build()} is called immediately and its
         * result is passed to {@link #targetLocationAlarmConfiguration(AlarmConfiguration)}.
         * 
         * @param targetLocationAlarmConfiguration
         *        a consumer that will call methods on {@link AlarmConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #targetLocationAlarmConfiguration(AlarmConfiguration)
         */
        default Builder targetLocationAlarmConfiguration(Consumer<AlarmConfiguration.Builder> targetLocationAlarmConfiguration) {
            return targetLocationAlarmConfiguration(AlarmConfiguration.builder().applyMutation(targetLocationAlarmConfiguration)
                    .build());
        }

        /**
         * <p>
         * Indicates whether to include child organizational units (OUs) that are children of the targeted OUs. The
         * default is <code>false</code>.
         * </p>
         * <note>
         * <p>
         * This parameter is not supported by State Manager.
         * </p>
         * </note>
         * 
         * @param includeChildOrganizationUnits
         *        Indicates whether to include child organizational units (OUs) that are children of the targeted OUs.
         *        The default is <code>false</code>.</p> <note>
         *        <p>
         *        This parameter is not supported by State Manager.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder includeChildOrganizationUnits(Boolean includeChildOrganizationUnits);

        /**
         * <p>
         * Amazon Web Services accounts or organizational units to exclude as expanded targets.
         * </p>
         * 
         * @param excludeAccounts
         *        Amazon Web Services accounts or organizational units to exclude as expanded targets.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder excludeAccounts(Collection<String> excludeAccounts);

        /**
         * <p>
         * Amazon Web Services accounts or organizational units to exclude as expanded targets.
         * </p>
         * 
         * @param excludeAccounts
         *        Amazon Web Services accounts or organizational units to exclude as expanded targets.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder excludeAccounts(String... excludeAccounts);

        /**
         * <p>
         * A list of key-value mappings to target resources. If you specify values for this data type, you must also
         * specify a value for <code>TargetParameterName</code>.
         * </p>
         * <p>
         * This <code>Targets</code> parameter takes precedence over the <code>StartAutomationExecution:Targets</code>
         * parameter if both are supplied.
         * </p>
         * 
         * @param targets
         *        A list of key-value mappings to target resources. If you specify values for this data type, you must
         *        also specify a value for <code>TargetParameterName</code>.</p>
         *        <p>
         *        This <code>Targets</code> parameter takes precedence over the
         *        <code>StartAutomationExecution:Targets</code> parameter if both are supplied.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targets(Collection<Target> targets);

        /**
         * <p>
         * A list of key-value mappings to target resources. If you specify values for this data type, you must also
         * specify a value for <code>TargetParameterName</code>.
         * </p>
         * <p>
         * This <code>Targets</code> parameter takes precedence over the <code>StartAutomationExecution:Targets</code>
         * parameter if both are supplied.
         * </p>
         * 
         * @param targets
         *        A list of key-value mappings to target resources. If you specify values for this data type, you must
         *        also specify a value for <code>TargetParameterName</code>.</p>
         *        <p>
         *        This <code>Targets</code> parameter takes precedence over the
         *        <code>StartAutomationExecution:Targets</code> parameter if both are supplied.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targets(Target... targets);

        /**
         * <p>
         * A list of key-value mappings to target resources. If you specify values for this data type, you must also
         * specify a value for <code>TargetParameterName</code>.
         * </p>
         * <p>
         * This <code>Targets</code> parameter takes precedence over the <code>StartAutomationExecution:Targets</code>
         * parameter if both are supplied.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ssm.model.Target.Builder} avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.ssm.model.Target#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link software.amazon.awssdk.services.ssm.model.Target.Builder#build()}
         * is called immediately and its result is passed to {@link #targets(List<Target>)}.
         * 
         * @param targets
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.ssm.model.Target.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #targets(java.util.Collection<Target>)
         */
        Builder targets(Consumer<Target.Builder>... targets);

        /**
         * <p>
         * The maximum number of targets allowed to run this task in parallel. This <code>TargetsMaxConcurrency</code>
         * takes precedence over the <code>StartAutomationExecution:MaxConcurrency</code> parameter if both are
         * supplied.
         * </p>
         * 
         * @param targetsMaxConcurrency
         *        The maximum number of targets allowed to run this task in parallel. This
         *        <code>TargetsMaxConcurrency</code> takes precedence over the
         *        <code>StartAutomationExecution:MaxConcurrency</code> parameter if both are supplied.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetsMaxConcurrency(String targetsMaxConcurrency);

        /**
         * <p>
         * The maximum number of errors that are allowed before the system stops running the automation on additional
         * targets. This <code>TargetsMaxErrors</code> parameter takes precedence over the
         * <code>StartAutomationExecution:MaxErrors</code> parameter if both are supplied.
         * </p>
         * 
         * @param targetsMaxErrors
         *        The maximum number of errors that are allowed before the system stops running the automation on
         *        additional targets. This <code>TargetsMaxErrors</code> parameter takes precedence over the
         *        <code>StartAutomationExecution:MaxErrors</code> parameter if both are supplied.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetsMaxErrors(String targetsMaxErrors);
    }

    static final class BuilderImpl implements Builder {
        private List<String> accounts = DefaultSdkAutoConstructList.getInstance();

        private List<String> regions = DefaultSdkAutoConstructList.getInstance();

        private String targetLocationMaxConcurrency;

        private String targetLocationMaxErrors;

        private String executionRoleName;

        private AlarmConfiguration targetLocationAlarmConfiguration;

        private Boolean includeChildOrganizationUnits;

        private List<String> excludeAccounts = DefaultSdkAutoConstructList.getInstance();

        private List<Target> targets = DefaultSdkAutoConstructList.getInstance();

        private String targetsMaxConcurrency;

        private String targetsMaxErrors;

        private BuilderImpl() {
        }

        private BuilderImpl(TargetLocation model) {
            accounts(model.accounts);
            regions(model.regions);
            targetLocationMaxConcurrency(model.targetLocationMaxConcurrency);
            targetLocationMaxErrors(model.targetLocationMaxErrors);
            executionRoleName(model.executionRoleName);
            targetLocationAlarmConfiguration(model.targetLocationAlarmConfiguration);
            includeChildOrganizationUnits(model.includeChildOrganizationUnits);
            excludeAccounts(model.excludeAccounts);
            targets(model.targets);
            targetsMaxConcurrency(model.targetsMaxConcurrency);
            targetsMaxErrors(model.targetsMaxErrors);
        }

        public final Collection<String> getAccounts() {
            if (accounts instanceof SdkAutoConstructList) {
                return null;
            }
            return accounts;
        }

        public final void setAccounts(Collection<String> accounts) {
            this.accounts = AccountsCopier.copy(accounts);
        }

        @Override
        public final Builder accounts(Collection<String> accounts) {
            this.accounts = AccountsCopier.copy(accounts);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder accounts(String... accounts) {
            accounts(Arrays.asList(accounts));
            return this;
        }

        public final Collection<String> getRegions() {
            if (regions instanceof SdkAutoConstructList) {
                return null;
            }
            return regions;
        }

        public final void setRegions(Collection<String> regions) {
            this.regions = RegionsCopier.copy(regions);
        }

        @Override
        public final Builder regions(Collection<String> regions) {
            this.regions = RegionsCopier.copy(regions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder regions(String... regions) {
            regions(Arrays.asList(regions));
            return this;
        }

        public final String getTargetLocationMaxConcurrency() {
            return targetLocationMaxConcurrency;
        }

        public final void setTargetLocationMaxConcurrency(String targetLocationMaxConcurrency) {
            this.targetLocationMaxConcurrency = targetLocationMaxConcurrency;
        }

        @Override
        public final Builder targetLocationMaxConcurrency(String targetLocationMaxConcurrency) {
            this.targetLocationMaxConcurrency = targetLocationMaxConcurrency;
            return this;
        }

        public final String getTargetLocationMaxErrors() {
            return targetLocationMaxErrors;
        }

        public final void setTargetLocationMaxErrors(String targetLocationMaxErrors) {
            this.targetLocationMaxErrors = targetLocationMaxErrors;
        }

        @Override
        public final Builder targetLocationMaxErrors(String targetLocationMaxErrors) {
            this.targetLocationMaxErrors = targetLocationMaxErrors;
            return this;
        }

        public final String getExecutionRoleName() {
            return executionRoleName;
        }

        public final void setExecutionRoleName(String executionRoleName) {
            this.executionRoleName = executionRoleName;
        }

        @Override
        public final Builder executionRoleName(String executionRoleName) {
            this.executionRoleName = executionRoleName;
            return this;
        }

        public final AlarmConfiguration.Builder getTargetLocationAlarmConfiguration() {
            return targetLocationAlarmConfiguration != null ? targetLocationAlarmConfiguration.toBuilder() : null;
        }

        public final void setTargetLocationAlarmConfiguration(AlarmConfiguration.BuilderImpl targetLocationAlarmConfiguration) {
            this.targetLocationAlarmConfiguration = targetLocationAlarmConfiguration != null ? targetLocationAlarmConfiguration
                    .build() : null;
        }

        @Override
        public final Builder targetLocationAlarmConfiguration(AlarmConfiguration targetLocationAlarmConfiguration) {
            this.targetLocationAlarmConfiguration = targetLocationAlarmConfiguration;
            return this;
        }

        public final Boolean getIncludeChildOrganizationUnits() {
            return includeChildOrganizationUnits;
        }

        public final void setIncludeChildOrganizationUnits(Boolean includeChildOrganizationUnits) {
            this.includeChildOrganizationUnits = includeChildOrganizationUnits;
        }

        @Override
        public final Builder includeChildOrganizationUnits(Boolean includeChildOrganizationUnits) {
            this.includeChildOrganizationUnits = includeChildOrganizationUnits;
            return this;
        }

        public final Collection<String> getExcludeAccounts() {
            if (excludeAccounts instanceof SdkAutoConstructList) {
                return null;
            }
            return excludeAccounts;
        }

        public final void setExcludeAccounts(Collection<String> excludeAccounts) {
            this.excludeAccounts = ExcludeAccountsCopier.copy(excludeAccounts);
        }

        @Override
        public final Builder excludeAccounts(Collection<String> excludeAccounts) {
            this.excludeAccounts = ExcludeAccountsCopier.copy(excludeAccounts);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder excludeAccounts(String... excludeAccounts) {
            excludeAccounts(Arrays.asList(excludeAccounts));
            return this;
        }

        public final List<Target.Builder> getTargets() {
            List<Target.Builder> result = TargetsCopier.copyToBuilder(this.targets);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTargets(Collection<Target.BuilderImpl> targets) {
            this.targets = TargetsCopier.copyFromBuilder(targets);
        }

        @Override
        public final Builder targets(Collection<Target> targets) {
            this.targets = TargetsCopier.copy(targets);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder targets(Target... targets) {
            targets(Arrays.asList(targets));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder targets(Consumer<Target.Builder>... targets) {
            targets(Stream.of(targets).map(c -> Target.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getTargetsMaxConcurrency() {
            return targetsMaxConcurrency;
        }

        public final void setTargetsMaxConcurrency(String targetsMaxConcurrency) {
            this.targetsMaxConcurrency = targetsMaxConcurrency;
        }

        @Override
        public final Builder targetsMaxConcurrency(String targetsMaxConcurrency) {
            this.targetsMaxConcurrency = targetsMaxConcurrency;
            return this;
        }

        public final String getTargetsMaxErrors() {
            return targetsMaxErrors;
        }

        public final void setTargetsMaxErrors(String targetsMaxErrors) {
            this.targetsMaxErrors = targetsMaxErrors;
        }

        @Override
        public final Builder targetsMaxErrors(String targetsMaxErrors) {
            this.targetsMaxErrors = targetsMaxErrors;
            return this;
        }

        @Override
        public TargetLocation build() {
            return new TargetLocation(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
