/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The target registered with the Maintenance Window.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class MaintenanceWindowTarget implements SdkPojo, Serializable,
        ToCopyableBuilder<MaintenanceWindowTarget.Builder, MaintenanceWindowTarget> {
    private static final SdkField<String> WINDOW_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(MaintenanceWindowTarget::windowId)).setter(setter(Builder::windowId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WindowId").build()).build();

    private static final SdkField<String> WINDOW_TARGET_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(MaintenanceWindowTarget::windowTargetId)).setter(setter(Builder::windowTargetId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WindowTargetId").build()).build();

    private static final SdkField<String> RESOURCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(MaintenanceWindowTarget::resourceTypeAsString)).setter(setter(Builder::resourceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceType").build()).build();

    private static final SdkField<List<Target>> TARGETS_FIELD = SdkField
            .<List<Target>> builder(MarshallingType.LIST)
            .getter(getter(MaintenanceWindowTarget::targets))
            .setter(setter(Builder::targets))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Targets").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Target> builder(MarshallingType.SDK_POJO)
                                            .constructor(Target::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> OWNER_INFORMATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(MaintenanceWindowTarget::ownerInformation)).setter(setter(Builder::ownerInformation))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OwnerInformation").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(MaintenanceWindowTarget::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(MaintenanceWindowTarget::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(WINDOW_ID_FIELD,
            WINDOW_TARGET_ID_FIELD, RESOURCE_TYPE_FIELD, TARGETS_FIELD, OWNER_INFORMATION_FIELD, NAME_FIELD, DESCRIPTION_FIELD));

    private static final long serialVersionUID = 1L;

    private final String windowId;

    private final String windowTargetId;

    private final String resourceType;

    private final List<Target> targets;

    private final String ownerInformation;

    private final String name;

    private final String description;

    private MaintenanceWindowTarget(BuilderImpl builder) {
        this.windowId = builder.windowId;
        this.windowTargetId = builder.windowTargetId;
        this.resourceType = builder.resourceType;
        this.targets = builder.targets;
        this.ownerInformation = builder.ownerInformation;
        this.name = builder.name;
        this.description = builder.description;
    }

    /**
     * <p>
     * The ID of the Maintenance Window to register the target with.
     * </p>
     * 
     * @return The ID of the Maintenance Window to register the target with.
     */
    public String windowId() {
        return windowId;
    }

    /**
     * <p>
     * The ID of the target.
     * </p>
     * 
     * @return The ID of the target.
     */
    public String windowTargetId() {
        return windowTargetId;
    }

    /**
     * <p>
     * The type of target that is being registered with the Maintenance Window.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #resourceType} will
     * return {@link MaintenanceWindowResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #resourceTypeAsString}.
     * </p>
     * 
     * @return The type of target that is being registered with the Maintenance Window.
     * @see MaintenanceWindowResourceType
     */
    public MaintenanceWindowResourceType resourceType() {
        return MaintenanceWindowResourceType.fromValue(resourceType);
    }

    /**
     * <p>
     * The type of target that is being registered with the Maintenance Window.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #resourceType} will
     * return {@link MaintenanceWindowResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #resourceTypeAsString}.
     * </p>
     * 
     * @return The type of target that is being registered with the Maintenance Window.
     * @see MaintenanceWindowResourceType
     */
    public String resourceTypeAsString() {
        return resourceType;
    }

    /**
     * <p>
     * The targets, either instances or tags.
     * </p>
     * <p>
     * Specify instances using the following format:
     * </p>
     * <p>
     * <code>Key=instanceids,Values=&lt;instanceid1&gt;,&lt;instanceid2&gt;</code>
     * </p>
     * <p>
     * Tags are specified using the following format:
     * </p>
     * <p>
     * <code>Key=&lt;tag name&gt;,Values=&lt;tag value&gt;</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The targets, either instances or tags.</p>
     *         <p>
     *         Specify instances using the following format:
     *         </p>
     *         <p>
     *         <code>Key=instanceids,Values=&lt;instanceid1&gt;,&lt;instanceid2&gt;</code>
     *         </p>
     *         <p>
     *         Tags are specified using the following format:
     *         </p>
     *         <p>
     *         <code>Key=&lt;tag name&gt;,Values=&lt;tag value&gt;</code>.
     */
    public List<Target> targets() {
        return targets;
    }

    /**
     * <p>
     * A user-provided value that will be included in any CloudWatch events that are raised while running tasks for
     * these targets in this Maintenance Window.
     * </p>
     * 
     * @return A user-provided value that will be included in any CloudWatch events that are raised while running tasks
     *         for these targets in this Maintenance Window.
     */
    public String ownerInformation() {
        return ownerInformation;
    }

    /**
     * <p>
     * The target name.
     * </p>
     * 
     * @return The target name.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * A description for the target.
     * </p>
     * 
     * @return A description for the target.
     */
    public String description() {
        return description;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(windowId());
        hashCode = 31 * hashCode + Objects.hashCode(windowTargetId());
        hashCode = 31 * hashCode + Objects.hashCode(resourceTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(targets());
        hashCode = 31 * hashCode + Objects.hashCode(ownerInformation());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof MaintenanceWindowTarget)) {
            return false;
        }
        MaintenanceWindowTarget other = (MaintenanceWindowTarget) obj;
        return Objects.equals(windowId(), other.windowId()) && Objects.equals(windowTargetId(), other.windowTargetId())
                && Objects.equals(resourceTypeAsString(), other.resourceTypeAsString())
                && Objects.equals(targets(), other.targets()) && Objects.equals(ownerInformation(), other.ownerInformation())
                && Objects.equals(name(), other.name()) && Objects.equals(description(), other.description());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("MaintenanceWindowTarget").add("WindowId", windowId()).add("WindowTargetId", windowTargetId())
                .add("ResourceType", resourceTypeAsString()).add("Targets", targets())
                .add("OwnerInformation", ownerInformation() == null ? null : "*** Sensitive Data Redacted ***")
                .add("Name", name()).add("Description", description() == null ? null : "*** Sensitive Data Redacted ***").build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "WindowId":
            return Optional.ofNullable(clazz.cast(windowId()));
        case "WindowTargetId":
            return Optional.ofNullable(clazz.cast(windowTargetId()));
        case "ResourceType":
            return Optional.ofNullable(clazz.cast(resourceTypeAsString()));
        case "Targets":
            return Optional.ofNullable(clazz.cast(targets()));
        case "OwnerInformation":
            return Optional.ofNullable(clazz.cast(ownerInformation()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<MaintenanceWindowTarget, T> g) {
        return obj -> g.apply((MaintenanceWindowTarget) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, MaintenanceWindowTarget> {
        /**
         * <p>
         * The ID of the Maintenance Window to register the target with.
         * </p>
         * 
         * @param windowId
         *        The ID of the Maintenance Window to register the target with.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder windowId(String windowId);

        /**
         * <p>
         * The ID of the target.
         * </p>
         * 
         * @param windowTargetId
         *        The ID of the target.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder windowTargetId(String windowTargetId);

        /**
         * <p>
         * The type of target that is being registered with the Maintenance Window.
         * </p>
         * 
         * @param resourceType
         *        The type of target that is being registered with the Maintenance Window.
         * @see MaintenanceWindowResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MaintenanceWindowResourceType
         */
        Builder resourceType(String resourceType);

        /**
         * <p>
         * The type of target that is being registered with the Maintenance Window.
         * </p>
         * 
         * @param resourceType
         *        The type of target that is being registered with the Maintenance Window.
         * @see MaintenanceWindowResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MaintenanceWindowResourceType
         */
        Builder resourceType(MaintenanceWindowResourceType resourceType);

        /**
         * <p>
         * The targets, either instances or tags.
         * </p>
         * <p>
         * Specify instances using the following format:
         * </p>
         * <p>
         * <code>Key=instanceids,Values=&lt;instanceid1&gt;,&lt;instanceid2&gt;</code>
         * </p>
         * <p>
         * Tags are specified using the following format:
         * </p>
         * <p>
         * <code>Key=&lt;tag name&gt;,Values=&lt;tag value&gt;</code>.
         * </p>
         * 
         * @param targets
         *        The targets, either instances or tags.</p>
         *        <p>
         *        Specify instances using the following format:
         *        </p>
         *        <p>
         *        <code>Key=instanceids,Values=&lt;instanceid1&gt;,&lt;instanceid2&gt;</code>
         *        </p>
         *        <p>
         *        Tags are specified using the following format:
         *        </p>
         *        <p>
         *        <code>Key=&lt;tag name&gt;,Values=&lt;tag value&gt;</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targets(Collection<Target> targets);

        /**
         * <p>
         * The targets, either instances or tags.
         * </p>
         * <p>
         * Specify instances using the following format:
         * </p>
         * <p>
         * <code>Key=instanceids,Values=&lt;instanceid1&gt;,&lt;instanceid2&gt;</code>
         * </p>
         * <p>
         * Tags are specified using the following format:
         * </p>
         * <p>
         * <code>Key=&lt;tag name&gt;,Values=&lt;tag value&gt;</code>.
         * </p>
         * 
         * @param targets
         *        The targets, either instances or tags.</p>
         *        <p>
         *        Specify instances using the following format:
         *        </p>
         *        <p>
         *        <code>Key=instanceids,Values=&lt;instanceid1&gt;,&lt;instanceid2&gt;</code>
         *        </p>
         *        <p>
         *        Tags are specified using the following format:
         *        </p>
         *        <p>
         *        <code>Key=&lt;tag name&gt;,Values=&lt;tag value&gt;</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targets(Target... targets);

        /**
         * <p>
         * The targets, either instances or tags.
         * </p>
         * <p>
         * Specify instances using the following format:
         * </p>
         * <p>
         * <code>Key=instanceids,Values=&lt;instanceid1&gt;,&lt;instanceid2&gt;</code>
         * </p>
         * <p>
         * Tags are specified using the following format:
         * </p>
         * <p>
         * <code>Key=&lt;tag name&gt;,Values=&lt;tag value&gt;</code>.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Target>.Builder} avoiding the need to
         * create one manually via {@link List<Target>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Target>.Builder#build()} is called immediately and its
         * result is passed to {@link #targets(List<Target>)}.
         * 
         * @param targets
         *        a consumer that will call methods on {@link List<Target>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #targets(List<Target>)
         */
        Builder targets(Consumer<Target.Builder>... targets);

        /**
         * <p>
         * A user-provided value that will be included in any CloudWatch events that are raised while running tasks for
         * these targets in this Maintenance Window.
         * </p>
         * 
         * @param ownerInformation
         *        A user-provided value that will be included in any CloudWatch events that are raised while running
         *        tasks for these targets in this Maintenance Window.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ownerInformation(String ownerInformation);

        /**
         * <p>
         * The target name.
         * </p>
         * 
         * @param name
         *        The target name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * A description for the target.
         * </p>
         * 
         * @param description
         *        A description for the target.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);
    }

    static final class BuilderImpl implements Builder {
        private String windowId;

        private String windowTargetId;

        private String resourceType;

        private List<Target> targets = DefaultSdkAutoConstructList.getInstance();

        private String ownerInformation;

        private String name;

        private String description;

        private BuilderImpl() {
        }

        private BuilderImpl(MaintenanceWindowTarget model) {
            windowId(model.windowId);
            windowTargetId(model.windowTargetId);
            resourceType(model.resourceType);
            targets(model.targets);
            ownerInformation(model.ownerInformation);
            name(model.name);
            description(model.description);
        }

        public final String getWindowId() {
            return windowId;
        }

        @Override
        public final Builder windowId(String windowId) {
            this.windowId = windowId;
            return this;
        }

        public final void setWindowId(String windowId) {
            this.windowId = windowId;
        }

        public final String getWindowTargetId() {
            return windowTargetId;
        }

        @Override
        public final Builder windowTargetId(String windowTargetId) {
            this.windowTargetId = windowTargetId;
            return this;
        }

        public final void setWindowTargetId(String windowTargetId) {
            this.windowTargetId = windowTargetId;
        }

        public final String getResourceTypeAsString() {
            return resourceType;
        }

        @Override
        public final Builder resourceType(String resourceType) {
            this.resourceType = resourceType;
            return this;
        }

        @Override
        public final Builder resourceType(MaintenanceWindowResourceType resourceType) {
            this.resourceType(resourceType.toString());
            return this;
        }

        public final void setResourceType(String resourceType) {
            this.resourceType = resourceType;
        }

        public final Collection<Target.Builder> getTargets() {
            return targets != null ? targets.stream().map(Target::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder targets(Collection<Target> targets) {
            this.targets = TargetsCopier.copy(targets);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder targets(Target... targets) {
            targets(Arrays.asList(targets));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder targets(Consumer<Target.Builder>... targets) {
            targets(Stream.of(targets).map(c -> Target.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTargets(Collection<Target.BuilderImpl> targets) {
            this.targets = TargetsCopier.copyFromBuilder(targets);
        }

        public final String getOwnerInformation() {
            return ownerInformation;
        }

        @Override
        public final Builder ownerInformation(String ownerInformation) {
            this.ownerInformation = ownerInformation;
            return this;
        }

        public final void setOwnerInformation(String ownerInformation) {
            this.ownerInformation = ownerInformation;
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public MaintenanceWindowTarget build() {
            return new MaintenanceWindowTarget(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
