/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Details about a specific Automation execution.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AutomationExecutionMetadata implements SdkPojo, Serializable,
        ToCopyableBuilder<AutomationExecutionMetadata.Builder, AutomationExecutionMetadata> {
    private static final SdkField<String> AUTOMATION_EXECUTION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(AutomationExecutionMetadata::automationExecutionId)).setter(setter(Builder::automationExecutionId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AutomationExecutionId").build())
            .build();

    private static final SdkField<String> DOCUMENT_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(AutomationExecutionMetadata::documentName)).setter(setter(Builder::documentName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DocumentName").build()).build();

    private static final SdkField<String> DOCUMENT_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(AutomationExecutionMetadata::documentVersion)).setter(setter(Builder::documentVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DocumentVersion").build()).build();

    private static final SdkField<String> AUTOMATION_EXECUTION_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(AutomationExecutionMetadata::automationExecutionStatusAsString))
            .setter(setter(Builder::automationExecutionStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AutomationExecutionStatus").build())
            .build();

    private static final SdkField<Instant> EXECUTION_START_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(AutomationExecutionMetadata::executionStartTime)).setter(setter(Builder::executionStartTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExecutionStartTime").build())
            .build();

    private static final SdkField<Instant> EXECUTION_END_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(AutomationExecutionMetadata::executionEndTime)).setter(setter(Builder::executionEndTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExecutionEndTime").build()).build();

    private static final SdkField<String> EXECUTED_BY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(AutomationExecutionMetadata::executedBy)).setter(setter(Builder::executedBy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExecutedBy").build()).build();

    private static final SdkField<String> LOG_FILE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(AutomationExecutionMetadata::logFile)).setter(setter(Builder::logFile))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LogFile").build()).build();

    private static final SdkField<Map<String, List<String>>> OUTPUTS_FIELD = SdkField
            .<Map<String, List<String>>> builder(MarshallingType.MAP)
            .getter(getter(AutomationExecutionMetadata::outputs))
            .setter(setter(Builder::outputs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Outputs").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<List<String>> builder(MarshallingType.LIST)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build(),
                                                    ListTrait
                                                            .builder()
                                                            .memberLocationName(null)
                                                            .memberFieldInfo(
                                                                    SdkField.<String> builder(MarshallingType.STRING)
                                                                            .traits(LocationTrait.builder()
                                                                                    .location(MarshallLocation.PAYLOAD)
                                                                                    .locationName("member").build()).build())
                                                            .build()).build()).build()).build();

    private static final SdkField<String> MODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(AutomationExecutionMetadata::modeAsString)).setter(setter(Builder::mode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Mode").build()).build();

    private static final SdkField<String> PARENT_AUTOMATION_EXECUTION_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(AutomationExecutionMetadata::parentAutomationExecutionId))
            .setter(setter(Builder::parentAutomationExecutionId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ParentAutomationExecutionId")
                    .build()).build();

    private static final SdkField<String> CURRENT_STEP_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(AutomationExecutionMetadata::currentStepName)).setter(setter(Builder::currentStepName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CurrentStepName").build()).build();

    private static final SdkField<String> CURRENT_ACTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(AutomationExecutionMetadata::currentAction)).setter(setter(Builder::currentAction))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CurrentAction").build()).build();

    private static final SdkField<String> FAILURE_MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(AutomationExecutionMetadata::failureMessage)).setter(setter(Builder::failureMessage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FailureMessage").build()).build();

    private static final SdkField<String> TARGET_PARAMETER_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(AutomationExecutionMetadata::targetParameterName)).setter(setter(Builder::targetParameterName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TargetParameterName").build())
            .build();

    private static final SdkField<List<Target>> TARGETS_FIELD = SdkField
            .<List<Target>> builder(MarshallingType.LIST)
            .getter(getter(AutomationExecutionMetadata::targets))
            .setter(setter(Builder::targets))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Targets").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Target> builder(MarshallingType.SDK_POJO)
                                            .constructor(Target::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<Map<String, List<String>>>> TARGET_MAPS_FIELD = SdkField
            .<List<Map<String, List<String>>>> builder(MarshallingType.LIST)
            .getter(getter(AutomationExecutionMetadata::targetMaps))
            .setter(setter(Builder::targetMaps))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TargetMaps").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Map<String, List<String>>> builder(MarshallingType.MAP)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build(),
                                                    MapTrait.builder()
                                                            .keyLocationName("key")
                                                            .valueLocationName("value")
                                                            .valueFieldInfo(
                                                                    SdkField.<List<String>> builder(MarshallingType.LIST)
                                                                            .traits(LocationTrait.builder()
                                                                                    .location(MarshallLocation.PAYLOAD)
                                                                                    .locationName("value").build(),
                                                                                    ListTrait
                                                                                            .builder()
                                                                                            .memberLocationName(null)
                                                                                            .memberFieldInfo(
                                                                                                    SdkField.<String> builder(
                                                                                                            MarshallingType.STRING)
                                                                                                            .traits(LocationTrait
                                                                                                                    .builder()
                                                                                                                    .location(
                                                                                                                            MarshallLocation.PAYLOAD)
                                                                                                                    .locationName(
                                                                                                                            "member")
                                                                                                                    .build())
                                                                                                            .build()).build())
                                                                            .build()).build()).build()).build()).build();

    private static final SdkField<ResolvedTargets> RESOLVED_TARGETS_FIELD = SdkField
            .<ResolvedTargets> builder(MarshallingType.SDK_POJO).getter(getter(AutomationExecutionMetadata::resolvedTargets))
            .setter(setter(Builder::resolvedTargets)).constructor(ResolvedTargets::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResolvedTargets").build()).build();

    private static final SdkField<String> MAX_CONCURRENCY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(AutomationExecutionMetadata::maxConcurrency)).setter(setter(Builder::maxConcurrency))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxConcurrency").build()).build();

    private static final SdkField<String> MAX_ERRORS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(AutomationExecutionMetadata::maxErrors)).setter(setter(Builder::maxErrors))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxErrors").build()).build();

    private static final SdkField<String> TARGET_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(AutomationExecutionMetadata::target)).setter(setter(Builder::target))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Target").build()).build();

    private static final SdkField<String> AUTOMATION_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(AutomationExecutionMetadata::automationTypeAsString)).setter(setter(Builder::automationType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AutomationType").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(AUTOMATION_EXECUTION_ID_FIELD,
            DOCUMENT_NAME_FIELD, DOCUMENT_VERSION_FIELD, AUTOMATION_EXECUTION_STATUS_FIELD, EXECUTION_START_TIME_FIELD,
            EXECUTION_END_TIME_FIELD, EXECUTED_BY_FIELD, LOG_FILE_FIELD, OUTPUTS_FIELD, MODE_FIELD,
            PARENT_AUTOMATION_EXECUTION_ID_FIELD, CURRENT_STEP_NAME_FIELD, CURRENT_ACTION_FIELD, FAILURE_MESSAGE_FIELD,
            TARGET_PARAMETER_NAME_FIELD, TARGETS_FIELD, TARGET_MAPS_FIELD, RESOLVED_TARGETS_FIELD, MAX_CONCURRENCY_FIELD,
            MAX_ERRORS_FIELD, TARGET_FIELD, AUTOMATION_TYPE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String automationExecutionId;

    private final String documentName;

    private final String documentVersion;

    private final String automationExecutionStatus;

    private final Instant executionStartTime;

    private final Instant executionEndTime;

    private final String executedBy;

    private final String logFile;

    private final Map<String, List<String>> outputs;

    private final String mode;

    private final String parentAutomationExecutionId;

    private final String currentStepName;

    private final String currentAction;

    private final String failureMessage;

    private final String targetParameterName;

    private final List<Target> targets;

    private final List<Map<String, List<String>>> targetMaps;

    private final ResolvedTargets resolvedTargets;

    private final String maxConcurrency;

    private final String maxErrors;

    private final String target;

    private final String automationType;

    private AutomationExecutionMetadata(BuilderImpl builder) {
        this.automationExecutionId = builder.automationExecutionId;
        this.documentName = builder.documentName;
        this.documentVersion = builder.documentVersion;
        this.automationExecutionStatus = builder.automationExecutionStatus;
        this.executionStartTime = builder.executionStartTime;
        this.executionEndTime = builder.executionEndTime;
        this.executedBy = builder.executedBy;
        this.logFile = builder.logFile;
        this.outputs = builder.outputs;
        this.mode = builder.mode;
        this.parentAutomationExecutionId = builder.parentAutomationExecutionId;
        this.currentStepName = builder.currentStepName;
        this.currentAction = builder.currentAction;
        this.failureMessage = builder.failureMessage;
        this.targetParameterName = builder.targetParameterName;
        this.targets = builder.targets;
        this.targetMaps = builder.targetMaps;
        this.resolvedTargets = builder.resolvedTargets;
        this.maxConcurrency = builder.maxConcurrency;
        this.maxErrors = builder.maxErrors;
        this.target = builder.target;
        this.automationType = builder.automationType;
    }

    /**
     * <p>
     * The execution ID.
     * </p>
     * 
     * @return The execution ID.
     */
    public String automationExecutionId() {
        return automationExecutionId;
    }

    /**
     * <p>
     * The name of the Automation document used during execution.
     * </p>
     * 
     * @return The name of the Automation document used during execution.
     */
    public String documentName() {
        return documentName;
    }

    /**
     * <p>
     * The document version used during the execution.
     * </p>
     * 
     * @return The document version used during the execution.
     */
    public String documentVersion() {
        return documentVersion;
    }

    /**
     * <p>
     * The status of the execution. Valid values include: Running, Succeeded, Failed, Timed out, or Cancelled.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #automationExecutionStatus} will return {@link AutomationExecutionStatus#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #automationExecutionStatusAsString}.
     * </p>
     * 
     * @return The status of the execution. Valid values include: Running, Succeeded, Failed, Timed out, or Cancelled.
     * @see AutomationExecutionStatus
     */
    public AutomationExecutionStatus automationExecutionStatus() {
        return AutomationExecutionStatus.fromValue(automationExecutionStatus);
    }

    /**
     * <p>
     * The status of the execution. Valid values include: Running, Succeeded, Failed, Timed out, or Cancelled.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #automationExecutionStatus} will return {@link AutomationExecutionStatus#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #automationExecutionStatusAsString}.
     * </p>
     * 
     * @return The status of the execution. Valid values include: Running, Succeeded, Failed, Timed out, or Cancelled.
     * @see AutomationExecutionStatus
     */
    public String automationExecutionStatusAsString() {
        return automationExecutionStatus;
    }

    /**
     * <p>
     * The time the execution started.&gt;
     * </p>
     * 
     * @return The time the execution started.&gt;
     */
    public Instant executionStartTime() {
        return executionStartTime;
    }

    /**
     * <p>
     * The time the execution finished. This is not populated if the execution is still in progress.
     * </p>
     * 
     * @return The time the execution finished. This is not populated if the execution is still in progress.
     */
    public Instant executionEndTime() {
        return executionEndTime;
    }

    /**
     * <p>
     * The IAM role ARN of the user who ran the Automation.
     * </p>
     * 
     * @return The IAM role ARN of the user who ran the Automation.
     */
    public String executedBy() {
        return executedBy;
    }

    /**
     * <p>
     * An Amazon S3 bucket where execution information is stored.
     * </p>
     * 
     * @return An Amazon S3 bucket where execution information is stored.
     */
    public String logFile() {
        return logFile;
    }

    /**
     * <p>
     * The list of execution outputs as defined in the Automation document.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The list of execution outputs as defined in the Automation document.
     */
    public Map<String, List<String>> outputs() {
        return outputs;
    }

    /**
     * <p>
     * The Automation execution mode.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #mode} will return
     * {@link ExecutionMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #modeAsString}.
     * </p>
     * 
     * @return The Automation execution mode.
     * @see ExecutionMode
     */
    public ExecutionMode mode() {
        return ExecutionMode.fromValue(mode);
    }

    /**
     * <p>
     * The Automation execution mode.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #mode} will return
     * {@link ExecutionMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #modeAsString}.
     * </p>
     * 
     * @return The Automation execution mode.
     * @see ExecutionMode
     */
    public String modeAsString() {
        return mode;
    }

    /**
     * <p>
     * The ExecutionId of the parent Automation.
     * </p>
     * 
     * @return The ExecutionId of the parent Automation.
     */
    public String parentAutomationExecutionId() {
        return parentAutomationExecutionId;
    }

    /**
     * <p>
     * The name of the step that is currently running.
     * </p>
     * 
     * @return The name of the step that is currently running.
     */
    public String currentStepName() {
        return currentStepName;
    }

    /**
     * <p>
     * The action of the step that is currently running.
     * </p>
     * 
     * @return The action of the step that is currently running.
     */
    public String currentAction() {
        return currentAction;
    }

    /**
     * <p>
     * The list of execution outputs as defined in the Automation document.
     * </p>
     * 
     * @return The list of execution outputs as defined in the Automation document.
     */
    public String failureMessage() {
        return failureMessage;
    }

    /**
     * <p>
     * The list of execution outputs as defined in the Automation document.
     * </p>
     * 
     * @return The list of execution outputs as defined in the Automation document.
     */
    public String targetParameterName() {
        return targetParameterName;
    }

    /**
     * <p>
     * The targets defined by the user when starting the Automation.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The targets defined by the user when starting the Automation.
     */
    public List<Target> targets() {
        return targets;
    }

    /**
     * <p>
     * The specified key-value mapping of document parameters to target resources.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The specified key-value mapping of document parameters to target resources.
     */
    public List<Map<String, List<String>>> targetMaps() {
        return targetMaps;
    }

    /**
     * <p>
     * A list of targets that resolved during the execution.
     * </p>
     * 
     * @return A list of targets that resolved during the execution.
     */
    public ResolvedTargets resolvedTargets() {
        return resolvedTargets;
    }

    /**
     * <p>
     * The MaxConcurrency value specified by the user when starting the Automation.
     * </p>
     * 
     * @return The MaxConcurrency value specified by the user when starting the Automation.
     */
    public String maxConcurrency() {
        return maxConcurrency;
    }

    /**
     * <p>
     * The MaxErrors value specified by the user when starting the Automation.
     * </p>
     * 
     * @return The MaxErrors value specified by the user when starting the Automation.
     */
    public String maxErrors() {
        return maxErrors;
    }

    /**
     * <p>
     * The list of execution outputs as defined in the Automation document.
     * </p>
     * 
     * @return The list of execution outputs as defined in the Automation document.
     */
    public String target() {
        return target;
    }

    /**
     * <p>
     * Use this filter with <a>DescribeAutomationExecutions</a>. Specify either Local or CrossAccount. CrossAccount is
     * an Automation that runs in multiple AWS Regions and accounts. For more information, see <a href=
     * "http://docs.aws.amazon.com/systems-manager/latest/userguide/systems-manager-automation-multiple-accounts-and-regions.html"
     * >Executing Automations in Multiple AWS Regions and Accounts</a> in the <i>AWS Systems Manager User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #automationType}
     * will return {@link AutomationType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #automationTypeAsString}.
     * </p>
     * 
     * @return Use this filter with <a>DescribeAutomationExecutions</a>. Specify either Local or CrossAccount.
     *         CrossAccount is an Automation that runs in multiple AWS Regions and accounts. For more information, see
     *         <a href=
     *         "http://docs.aws.amazon.com/systems-manager/latest/userguide/systems-manager-automation-multiple-accounts-and-regions.html"
     *         >Executing Automations in Multiple AWS Regions and Accounts</a> in the <i>AWS Systems Manager User
     *         Guide</i>.
     * @see AutomationType
     */
    public AutomationType automationType() {
        return AutomationType.fromValue(automationType);
    }

    /**
     * <p>
     * Use this filter with <a>DescribeAutomationExecutions</a>. Specify either Local or CrossAccount. CrossAccount is
     * an Automation that runs in multiple AWS Regions and accounts. For more information, see <a href=
     * "http://docs.aws.amazon.com/systems-manager/latest/userguide/systems-manager-automation-multiple-accounts-and-regions.html"
     * >Executing Automations in Multiple AWS Regions and Accounts</a> in the <i>AWS Systems Manager User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #automationType}
     * will return {@link AutomationType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #automationTypeAsString}.
     * </p>
     * 
     * @return Use this filter with <a>DescribeAutomationExecutions</a>. Specify either Local or CrossAccount.
     *         CrossAccount is an Automation that runs in multiple AWS Regions and accounts. For more information, see
     *         <a href=
     *         "http://docs.aws.amazon.com/systems-manager/latest/userguide/systems-manager-automation-multiple-accounts-and-regions.html"
     *         >Executing Automations in Multiple AWS Regions and Accounts</a> in the <i>AWS Systems Manager User
     *         Guide</i>.
     * @see AutomationType
     */
    public String automationTypeAsString() {
        return automationType;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(automationExecutionId());
        hashCode = 31 * hashCode + Objects.hashCode(documentName());
        hashCode = 31 * hashCode + Objects.hashCode(documentVersion());
        hashCode = 31 * hashCode + Objects.hashCode(automationExecutionStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(executionStartTime());
        hashCode = 31 * hashCode + Objects.hashCode(executionEndTime());
        hashCode = 31 * hashCode + Objects.hashCode(executedBy());
        hashCode = 31 * hashCode + Objects.hashCode(logFile());
        hashCode = 31 * hashCode + Objects.hashCode(outputs());
        hashCode = 31 * hashCode + Objects.hashCode(modeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(parentAutomationExecutionId());
        hashCode = 31 * hashCode + Objects.hashCode(currentStepName());
        hashCode = 31 * hashCode + Objects.hashCode(currentAction());
        hashCode = 31 * hashCode + Objects.hashCode(failureMessage());
        hashCode = 31 * hashCode + Objects.hashCode(targetParameterName());
        hashCode = 31 * hashCode + Objects.hashCode(targets());
        hashCode = 31 * hashCode + Objects.hashCode(targetMaps());
        hashCode = 31 * hashCode + Objects.hashCode(resolvedTargets());
        hashCode = 31 * hashCode + Objects.hashCode(maxConcurrency());
        hashCode = 31 * hashCode + Objects.hashCode(maxErrors());
        hashCode = 31 * hashCode + Objects.hashCode(target());
        hashCode = 31 * hashCode + Objects.hashCode(automationTypeAsString());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AutomationExecutionMetadata)) {
            return false;
        }
        AutomationExecutionMetadata other = (AutomationExecutionMetadata) obj;
        return Objects.equals(automationExecutionId(), other.automationExecutionId())
                && Objects.equals(documentName(), other.documentName())
                && Objects.equals(documentVersion(), other.documentVersion())
                && Objects.equals(automationExecutionStatusAsString(), other.automationExecutionStatusAsString())
                && Objects.equals(executionStartTime(), other.executionStartTime())
                && Objects.equals(executionEndTime(), other.executionEndTime())
                && Objects.equals(executedBy(), other.executedBy()) && Objects.equals(logFile(), other.logFile())
                && Objects.equals(outputs(), other.outputs()) && Objects.equals(modeAsString(), other.modeAsString())
                && Objects.equals(parentAutomationExecutionId(), other.parentAutomationExecutionId())
                && Objects.equals(currentStepName(), other.currentStepName())
                && Objects.equals(currentAction(), other.currentAction())
                && Objects.equals(failureMessage(), other.failureMessage())
                && Objects.equals(targetParameterName(), other.targetParameterName())
                && Objects.equals(targets(), other.targets()) && Objects.equals(targetMaps(), other.targetMaps())
                && Objects.equals(resolvedTargets(), other.resolvedTargets())
                && Objects.equals(maxConcurrency(), other.maxConcurrency()) && Objects.equals(maxErrors(), other.maxErrors())
                && Objects.equals(target(), other.target())
                && Objects.equals(automationTypeAsString(), other.automationTypeAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("AutomationExecutionMetadata").add("AutomationExecutionId", automationExecutionId())
                .add("DocumentName", documentName()).add("DocumentVersion", documentVersion())
                .add("AutomationExecutionStatus", automationExecutionStatusAsString())
                .add("ExecutionStartTime", executionStartTime()).add("ExecutionEndTime", executionEndTime())
                .add("ExecutedBy", executedBy()).add("LogFile", logFile()).add("Outputs", outputs()).add("Mode", modeAsString())
                .add("ParentAutomationExecutionId", parentAutomationExecutionId()).add("CurrentStepName", currentStepName())
                .add("CurrentAction", currentAction()).add("FailureMessage", failureMessage())
                .add("TargetParameterName", targetParameterName()).add("Targets", targets()).add("TargetMaps", targetMaps())
                .add("ResolvedTargets", resolvedTargets()).add("MaxConcurrency", maxConcurrency()).add("MaxErrors", maxErrors())
                .add("Target", target()).add("AutomationType", automationTypeAsString()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AutomationExecutionId":
            return Optional.ofNullable(clazz.cast(automationExecutionId()));
        case "DocumentName":
            return Optional.ofNullable(clazz.cast(documentName()));
        case "DocumentVersion":
            return Optional.ofNullable(clazz.cast(documentVersion()));
        case "AutomationExecutionStatus":
            return Optional.ofNullable(clazz.cast(automationExecutionStatusAsString()));
        case "ExecutionStartTime":
            return Optional.ofNullable(clazz.cast(executionStartTime()));
        case "ExecutionEndTime":
            return Optional.ofNullable(clazz.cast(executionEndTime()));
        case "ExecutedBy":
            return Optional.ofNullable(clazz.cast(executedBy()));
        case "LogFile":
            return Optional.ofNullable(clazz.cast(logFile()));
        case "Outputs":
            return Optional.ofNullable(clazz.cast(outputs()));
        case "Mode":
            return Optional.ofNullable(clazz.cast(modeAsString()));
        case "ParentAutomationExecutionId":
            return Optional.ofNullable(clazz.cast(parentAutomationExecutionId()));
        case "CurrentStepName":
            return Optional.ofNullable(clazz.cast(currentStepName()));
        case "CurrentAction":
            return Optional.ofNullable(clazz.cast(currentAction()));
        case "FailureMessage":
            return Optional.ofNullable(clazz.cast(failureMessage()));
        case "TargetParameterName":
            return Optional.ofNullable(clazz.cast(targetParameterName()));
        case "Targets":
            return Optional.ofNullable(clazz.cast(targets()));
        case "TargetMaps":
            return Optional.ofNullable(clazz.cast(targetMaps()));
        case "ResolvedTargets":
            return Optional.ofNullable(clazz.cast(resolvedTargets()));
        case "MaxConcurrency":
            return Optional.ofNullable(clazz.cast(maxConcurrency()));
        case "MaxErrors":
            return Optional.ofNullable(clazz.cast(maxErrors()));
        case "Target":
            return Optional.ofNullable(clazz.cast(target()));
        case "AutomationType":
            return Optional.ofNullable(clazz.cast(automationTypeAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<AutomationExecutionMetadata, T> g) {
        return obj -> g.apply((AutomationExecutionMetadata) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AutomationExecutionMetadata> {
        /**
         * <p>
         * The execution ID.
         * </p>
         * 
         * @param automationExecutionId
         *        The execution ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder automationExecutionId(String automationExecutionId);

        /**
         * <p>
         * The name of the Automation document used during execution.
         * </p>
         * 
         * @param documentName
         *        The name of the Automation document used during execution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder documentName(String documentName);

        /**
         * <p>
         * The document version used during the execution.
         * </p>
         * 
         * @param documentVersion
         *        The document version used during the execution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder documentVersion(String documentVersion);

        /**
         * <p>
         * The status of the execution. Valid values include: Running, Succeeded, Failed, Timed out, or Cancelled.
         * </p>
         * 
         * @param automationExecutionStatus
         *        The status of the execution. Valid values include: Running, Succeeded, Failed, Timed out, or
         *        Cancelled.
         * @see AutomationExecutionStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AutomationExecutionStatus
         */
        Builder automationExecutionStatus(String automationExecutionStatus);

        /**
         * <p>
         * The status of the execution. Valid values include: Running, Succeeded, Failed, Timed out, or Cancelled.
         * </p>
         * 
         * @param automationExecutionStatus
         *        The status of the execution. Valid values include: Running, Succeeded, Failed, Timed out, or
         *        Cancelled.
         * @see AutomationExecutionStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AutomationExecutionStatus
         */
        Builder automationExecutionStatus(AutomationExecutionStatus automationExecutionStatus);

        /**
         * <p>
         * The time the execution started.&gt;
         * </p>
         * 
         * @param executionStartTime
         *        The time the execution started.&gt;
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder executionStartTime(Instant executionStartTime);

        /**
         * <p>
         * The time the execution finished. This is not populated if the execution is still in progress.
         * </p>
         * 
         * @param executionEndTime
         *        The time the execution finished. This is not populated if the execution is still in progress.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder executionEndTime(Instant executionEndTime);

        /**
         * <p>
         * The IAM role ARN of the user who ran the Automation.
         * </p>
         * 
         * @param executedBy
         *        The IAM role ARN of the user who ran the Automation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder executedBy(String executedBy);

        /**
         * <p>
         * An Amazon S3 bucket where execution information is stored.
         * </p>
         * 
         * @param logFile
         *        An Amazon S3 bucket where execution information is stored.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logFile(String logFile);

        /**
         * <p>
         * The list of execution outputs as defined in the Automation document.
         * </p>
         * 
         * @param outputs
         *        The list of execution outputs as defined in the Automation document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputs(Map<String, ? extends Collection<String>> outputs);

        /**
         * <p>
         * The Automation execution mode.
         * </p>
         * 
         * @param mode
         *        The Automation execution mode.
         * @see ExecutionMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ExecutionMode
         */
        Builder mode(String mode);

        /**
         * <p>
         * The Automation execution mode.
         * </p>
         * 
         * @param mode
         *        The Automation execution mode.
         * @see ExecutionMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ExecutionMode
         */
        Builder mode(ExecutionMode mode);

        /**
         * <p>
         * The ExecutionId of the parent Automation.
         * </p>
         * 
         * @param parentAutomationExecutionId
         *        The ExecutionId of the parent Automation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parentAutomationExecutionId(String parentAutomationExecutionId);

        /**
         * <p>
         * The name of the step that is currently running.
         * </p>
         * 
         * @param currentStepName
         *        The name of the step that is currently running.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder currentStepName(String currentStepName);

        /**
         * <p>
         * The action of the step that is currently running.
         * </p>
         * 
         * @param currentAction
         *        The action of the step that is currently running.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder currentAction(String currentAction);

        /**
         * <p>
         * The list of execution outputs as defined in the Automation document.
         * </p>
         * 
         * @param failureMessage
         *        The list of execution outputs as defined in the Automation document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failureMessage(String failureMessage);

        /**
         * <p>
         * The list of execution outputs as defined in the Automation document.
         * </p>
         * 
         * @param targetParameterName
         *        The list of execution outputs as defined in the Automation document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetParameterName(String targetParameterName);

        /**
         * <p>
         * The targets defined by the user when starting the Automation.
         * </p>
         * 
         * @param targets
         *        The targets defined by the user when starting the Automation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targets(Collection<Target> targets);

        /**
         * <p>
         * The targets defined by the user when starting the Automation.
         * </p>
         * 
         * @param targets
         *        The targets defined by the user when starting the Automation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targets(Target... targets);

        /**
         * <p>
         * The targets defined by the user when starting the Automation.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Target>.Builder} avoiding the need to
         * create one manually via {@link List<Target>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Target>.Builder#build()} is called immediately and its
         * result is passed to {@link #targets(List<Target>)}.
         * 
         * @param targets
         *        a consumer that will call methods on {@link List<Target>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #targets(List<Target>)
         */
        Builder targets(Consumer<Target.Builder>... targets);

        /**
         * <p>
         * The specified key-value mapping of document parameters to target resources.
         * </p>
         * 
         * @param targetMaps
         *        The specified key-value mapping of document parameters to target resources.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetMaps(Collection<? extends Map<String, ? extends Collection<String>>> targetMaps);

        /**
         * <p>
         * The specified key-value mapping of document parameters to target resources.
         * </p>
         * 
         * @param targetMaps
         *        The specified key-value mapping of document parameters to target resources.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetMaps(Map<String, ? extends Collection<String>>... targetMaps);

        /**
         * <p>
         * A list of targets that resolved during the execution.
         * </p>
         * 
         * @param resolvedTargets
         *        A list of targets that resolved during the execution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resolvedTargets(ResolvedTargets resolvedTargets);

        /**
         * <p>
         * A list of targets that resolved during the execution.
         * </p>
         * This is a convenience that creates an instance of the {@link ResolvedTargets.Builder} avoiding the need to
         * create one manually via {@link ResolvedTargets#builder()}.
         *
         * When the {@link Consumer} completes, {@link ResolvedTargets.Builder#build()} is called immediately and its
         * result is passed to {@link #resolvedTargets(ResolvedTargets)}.
         * 
         * @param resolvedTargets
         *        a consumer that will call methods on {@link ResolvedTargets.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #resolvedTargets(ResolvedTargets)
         */
        default Builder resolvedTargets(Consumer<ResolvedTargets.Builder> resolvedTargets) {
            return resolvedTargets(ResolvedTargets.builder().applyMutation(resolvedTargets).build());
        }

        /**
         * <p>
         * The MaxConcurrency value specified by the user when starting the Automation.
         * </p>
         * 
         * @param maxConcurrency
         *        The MaxConcurrency value specified by the user when starting the Automation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxConcurrency(String maxConcurrency);

        /**
         * <p>
         * The MaxErrors value specified by the user when starting the Automation.
         * </p>
         * 
         * @param maxErrors
         *        The MaxErrors value specified by the user when starting the Automation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxErrors(String maxErrors);

        /**
         * <p>
         * The list of execution outputs as defined in the Automation document.
         * </p>
         * 
         * @param target
         *        The list of execution outputs as defined in the Automation document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder target(String target);

        /**
         * <p>
         * Use this filter with <a>DescribeAutomationExecutions</a>. Specify either Local or CrossAccount. CrossAccount
         * is an Automation that runs in multiple AWS Regions and accounts. For more information, see <a href=
         * "http://docs.aws.amazon.com/systems-manager/latest/userguide/systems-manager-automation-multiple-accounts-and-regions.html"
         * >Executing Automations in Multiple AWS Regions and Accounts</a> in the <i>AWS Systems Manager User Guide</i>.
         * </p>
         * 
         * @param automationType
         *        Use this filter with <a>DescribeAutomationExecutions</a>. Specify either Local or CrossAccount.
         *        CrossAccount is an Automation that runs in multiple AWS Regions and accounts. For more information,
         *        see <a href=
         *        "http://docs.aws.amazon.com/systems-manager/latest/userguide/systems-manager-automation-multiple-accounts-and-regions.html"
         *        >Executing Automations in Multiple AWS Regions and Accounts</a> in the <i>AWS Systems Manager User
         *        Guide</i>.
         * @see AutomationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AutomationType
         */
        Builder automationType(String automationType);

        /**
         * <p>
         * Use this filter with <a>DescribeAutomationExecutions</a>. Specify either Local or CrossAccount. CrossAccount
         * is an Automation that runs in multiple AWS Regions and accounts. For more information, see <a href=
         * "http://docs.aws.amazon.com/systems-manager/latest/userguide/systems-manager-automation-multiple-accounts-and-regions.html"
         * >Executing Automations in Multiple AWS Regions and Accounts</a> in the <i>AWS Systems Manager User Guide</i>.
         * </p>
         * 
         * @param automationType
         *        Use this filter with <a>DescribeAutomationExecutions</a>. Specify either Local or CrossAccount.
         *        CrossAccount is an Automation that runs in multiple AWS Regions and accounts. For more information,
         *        see <a href=
         *        "http://docs.aws.amazon.com/systems-manager/latest/userguide/systems-manager-automation-multiple-accounts-and-regions.html"
         *        >Executing Automations in Multiple AWS Regions and Accounts</a> in the <i>AWS Systems Manager User
         *        Guide</i>.
         * @see AutomationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AutomationType
         */
        Builder automationType(AutomationType automationType);
    }

    static final class BuilderImpl implements Builder {
        private String automationExecutionId;

        private String documentName;

        private String documentVersion;

        private String automationExecutionStatus;

        private Instant executionStartTime;

        private Instant executionEndTime;

        private String executedBy;

        private String logFile;

        private Map<String, List<String>> outputs = DefaultSdkAutoConstructMap.getInstance();

        private String mode;

        private String parentAutomationExecutionId;

        private String currentStepName;

        private String currentAction;

        private String failureMessage;

        private String targetParameterName;

        private List<Target> targets = DefaultSdkAutoConstructList.getInstance();

        private List<Map<String, List<String>>> targetMaps = DefaultSdkAutoConstructList.getInstance();

        private ResolvedTargets resolvedTargets;

        private String maxConcurrency;

        private String maxErrors;

        private String target;

        private String automationType;

        private BuilderImpl() {
        }

        private BuilderImpl(AutomationExecutionMetadata model) {
            automationExecutionId(model.automationExecutionId);
            documentName(model.documentName);
            documentVersion(model.documentVersion);
            automationExecutionStatus(model.automationExecutionStatus);
            executionStartTime(model.executionStartTime);
            executionEndTime(model.executionEndTime);
            executedBy(model.executedBy);
            logFile(model.logFile);
            outputs(model.outputs);
            mode(model.mode);
            parentAutomationExecutionId(model.parentAutomationExecutionId);
            currentStepName(model.currentStepName);
            currentAction(model.currentAction);
            failureMessage(model.failureMessage);
            targetParameterName(model.targetParameterName);
            targets(model.targets);
            targetMaps(model.targetMaps);
            resolvedTargets(model.resolvedTargets);
            maxConcurrency(model.maxConcurrency);
            maxErrors(model.maxErrors);
            target(model.target);
            automationType(model.automationType);
        }

        public final String getAutomationExecutionId() {
            return automationExecutionId;
        }

        @Override
        public final Builder automationExecutionId(String automationExecutionId) {
            this.automationExecutionId = automationExecutionId;
            return this;
        }

        public final void setAutomationExecutionId(String automationExecutionId) {
            this.automationExecutionId = automationExecutionId;
        }

        public final String getDocumentName() {
            return documentName;
        }

        @Override
        public final Builder documentName(String documentName) {
            this.documentName = documentName;
            return this;
        }

        public final void setDocumentName(String documentName) {
            this.documentName = documentName;
        }

        public final String getDocumentVersion() {
            return documentVersion;
        }

        @Override
        public final Builder documentVersion(String documentVersion) {
            this.documentVersion = documentVersion;
            return this;
        }

        public final void setDocumentVersion(String documentVersion) {
            this.documentVersion = documentVersion;
        }

        public final String getAutomationExecutionStatusAsString() {
            return automationExecutionStatus;
        }

        @Override
        public final Builder automationExecutionStatus(String automationExecutionStatus) {
            this.automationExecutionStatus = automationExecutionStatus;
            return this;
        }

        @Override
        public final Builder automationExecutionStatus(AutomationExecutionStatus automationExecutionStatus) {
            this.automationExecutionStatus(automationExecutionStatus.toString());
            return this;
        }

        public final void setAutomationExecutionStatus(String automationExecutionStatus) {
            this.automationExecutionStatus = automationExecutionStatus;
        }

        public final Instant getExecutionStartTime() {
            return executionStartTime;
        }

        @Override
        public final Builder executionStartTime(Instant executionStartTime) {
            this.executionStartTime = executionStartTime;
            return this;
        }

        public final void setExecutionStartTime(Instant executionStartTime) {
            this.executionStartTime = executionStartTime;
        }

        public final Instant getExecutionEndTime() {
            return executionEndTime;
        }

        @Override
        public final Builder executionEndTime(Instant executionEndTime) {
            this.executionEndTime = executionEndTime;
            return this;
        }

        public final void setExecutionEndTime(Instant executionEndTime) {
            this.executionEndTime = executionEndTime;
        }

        public final String getExecutedBy() {
            return executedBy;
        }

        @Override
        public final Builder executedBy(String executedBy) {
            this.executedBy = executedBy;
            return this;
        }

        public final void setExecutedBy(String executedBy) {
            this.executedBy = executedBy;
        }

        public final String getLogFile() {
            return logFile;
        }

        @Override
        public final Builder logFile(String logFile) {
            this.logFile = logFile;
            return this;
        }

        public final void setLogFile(String logFile) {
            this.logFile = logFile;
        }

        public final Map<String, ? extends Collection<String>> getOutputs() {
            return outputs;
        }

        @Override
        public final Builder outputs(Map<String, ? extends Collection<String>> outputs) {
            this.outputs = AutomationParameterMapCopier.copy(outputs);
            return this;
        }

        public final void setOutputs(Map<String, ? extends Collection<String>> outputs) {
            this.outputs = AutomationParameterMapCopier.copy(outputs);
        }

        public final String getModeAsString() {
            return mode;
        }

        @Override
        public final Builder mode(String mode) {
            this.mode = mode;
            return this;
        }

        @Override
        public final Builder mode(ExecutionMode mode) {
            this.mode(mode.toString());
            return this;
        }

        public final void setMode(String mode) {
            this.mode = mode;
        }

        public final String getParentAutomationExecutionId() {
            return parentAutomationExecutionId;
        }

        @Override
        public final Builder parentAutomationExecutionId(String parentAutomationExecutionId) {
            this.parentAutomationExecutionId = parentAutomationExecutionId;
            return this;
        }

        public final void setParentAutomationExecutionId(String parentAutomationExecutionId) {
            this.parentAutomationExecutionId = parentAutomationExecutionId;
        }

        public final String getCurrentStepName() {
            return currentStepName;
        }

        @Override
        public final Builder currentStepName(String currentStepName) {
            this.currentStepName = currentStepName;
            return this;
        }

        public final void setCurrentStepName(String currentStepName) {
            this.currentStepName = currentStepName;
        }

        public final String getCurrentAction() {
            return currentAction;
        }

        @Override
        public final Builder currentAction(String currentAction) {
            this.currentAction = currentAction;
            return this;
        }

        public final void setCurrentAction(String currentAction) {
            this.currentAction = currentAction;
        }

        public final String getFailureMessage() {
            return failureMessage;
        }

        @Override
        public final Builder failureMessage(String failureMessage) {
            this.failureMessage = failureMessage;
            return this;
        }

        public final void setFailureMessage(String failureMessage) {
            this.failureMessage = failureMessage;
        }

        public final String getTargetParameterName() {
            return targetParameterName;
        }

        @Override
        public final Builder targetParameterName(String targetParameterName) {
            this.targetParameterName = targetParameterName;
            return this;
        }

        public final void setTargetParameterName(String targetParameterName) {
            this.targetParameterName = targetParameterName;
        }

        public final Collection<Target.Builder> getTargets() {
            return targets != null ? targets.stream().map(Target::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder targets(Collection<Target> targets) {
            this.targets = TargetsCopier.copy(targets);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder targets(Target... targets) {
            targets(Arrays.asList(targets));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder targets(Consumer<Target.Builder>... targets) {
            targets(Stream.of(targets).map(c -> Target.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTargets(Collection<Target.BuilderImpl> targets) {
            this.targets = TargetsCopier.copyFromBuilder(targets);
        }

        public final Collection<? extends Map<String, ? extends Collection<String>>> getTargetMaps() {
            return targetMaps;
        }

        @Override
        public final Builder targetMaps(Collection<? extends Map<String, ? extends Collection<String>>> targetMaps) {
            this.targetMaps = TargetMapsCopier.copy(targetMaps);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder targetMaps(Map<String, ? extends Collection<String>>... targetMaps) {
            targetMaps(Arrays.asList(targetMaps));
            return this;
        }

        public final void setTargetMaps(Collection<? extends Map<String, ? extends Collection<String>>> targetMaps) {
            this.targetMaps = TargetMapsCopier.copy(targetMaps);
        }

        public final ResolvedTargets.Builder getResolvedTargets() {
            return resolvedTargets != null ? resolvedTargets.toBuilder() : null;
        }

        @Override
        public final Builder resolvedTargets(ResolvedTargets resolvedTargets) {
            this.resolvedTargets = resolvedTargets;
            return this;
        }

        public final void setResolvedTargets(ResolvedTargets.BuilderImpl resolvedTargets) {
            this.resolvedTargets = resolvedTargets != null ? resolvedTargets.build() : null;
        }

        public final String getMaxConcurrency() {
            return maxConcurrency;
        }

        @Override
        public final Builder maxConcurrency(String maxConcurrency) {
            this.maxConcurrency = maxConcurrency;
            return this;
        }

        public final void setMaxConcurrency(String maxConcurrency) {
            this.maxConcurrency = maxConcurrency;
        }

        public final String getMaxErrors() {
            return maxErrors;
        }

        @Override
        public final Builder maxErrors(String maxErrors) {
            this.maxErrors = maxErrors;
            return this;
        }

        public final void setMaxErrors(String maxErrors) {
            this.maxErrors = maxErrors;
        }

        public final String getTarget() {
            return target;
        }

        @Override
        public final Builder target(String target) {
            this.target = target;
            return this;
        }

        public final void setTarget(String target) {
            this.target = target;
        }

        public final String getAutomationTypeAsString() {
            return automationType;
        }

        @Override
        public final Builder automationType(String automationType) {
            this.automationType = automationType;
            return this;
        }

        @Override
        public final Builder automationType(AutomationType automationType) {
            this.automationType(automationType.toString());
            return this;
        }

        public final void setAutomationType(String automationType) {
            this.automationType = automationType;
        }

        @Override
        public AutomationExecutionMetadata build() {
            return new AutomationExecutionMetadata(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
