/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Defines which patches should be included in a patch baseline.
 * </p>
 * <p>
 * A patch filter consists of a key and a set of values. The filter key is a patch property. For example, the available
 * filter keys for WINDOWS are PATCH_SET, PRODUCT, PRODUCT_FAMILY, CLASSIFICATION, and MSRC_SEVERITY. The filter values
 * define a matching criterion for the patch property indicated by the key. For example, if the filter key is PRODUCT
 * and the filter values are ["Office 2013", "Office 2016"], then the filter accepts all patches where product name is
 * either "Office 2013" or "Office 2016". The filter values can be exact values for the patch property given as a key,
 * or a wildcard (*), which matches all values.
 * </p>
 * <p>
 * You can view lists of valid values for the patch properties by running the <code>DescribePatchProperties</code>
 * command. For information about which patch properties can be used with each major operating system, see
 * <a>DescribePatchProperties</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class PatchFilter implements SdkPojo, Serializable, ToCopyableBuilder<PatchFilter.Builder, PatchFilter> {
    private static final SdkField<String> KEY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(PatchFilter::keyAsString)).setter(setter(Builder::key))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Key").build()).build();

    private static final SdkField<List<String>> VALUES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(PatchFilter::values))
            .setter(setter(Builder::values))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Values").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(KEY_FIELD, VALUES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String key;

    private final List<String> values;

    private PatchFilter(BuilderImpl builder) {
        this.key = builder.key;
        this.values = builder.values;
    }

    /**
     * <p>
     * The key for the filter.
     * </p>
     * <p>
     * Run the <a>DescribePatchProperties</a> command to view lists of valid keys for each operating system type.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #key} will return
     * {@link PatchFilterKey#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #keyAsString}.
     * </p>
     * 
     * @return The key for the filter.</p>
     *         <p>
     *         Run the <a>DescribePatchProperties</a> command to view lists of valid keys for each operating system
     *         type.
     * @see PatchFilterKey
     */
    public PatchFilterKey key() {
        return PatchFilterKey.fromValue(key);
    }

    /**
     * <p>
     * The key for the filter.
     * </p>
     * <p>
     * Run the <a>DescribePatchProperties</a> command to view lists of valid keys for each operating system type.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #key} will return
     * {@link PatchFilterKey#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #keyAsString}.
     * </p>
     * 
     * @return The key for the filter.</p>
     *         <p>
     *         Run the <a>DescribePatchProperties</a> command to view lists of valid keys for each operating system
     *         type.
     * @see PatchFilterKey
     */
    public String keyAsString() {
        return key;
    }

    /**
     * <p>
     * The value for the filter key.
     * </p>
     * <p>
     * Run the <a>DescribePatchProperties</a> command to view lists of valid values for each key based on operating
     * system type.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The value for the filter key.</p>
     *         <p>
     *         Run the <a>DescribePatchProperties</a> command to view lists of valid values for each key based on
     *         operating system type.
     */
    public List<String> values() {
        return values;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(keyAsString());
        hashCode = 31 * hashCode + Objects.hashCode(values());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PatchFilter)) {
            return false;
        }
        PatchFilter other = (PatchFilter) obj;
        return Objects.equals(keyAsString(), other.keyAsString()) && Objects.equals(values(), other.values());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("PatchFilter").add("Key", keyAsString()).add("Values", values()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Key":
            return Optional.ofNullable(clazz.cast(keyAsString()));
        case "Values":
            return Optional.ofNullable(clazz.cast(values()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<PatchFilter, T> g) {
        return obj -> g.apply((PatchFilter) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, PatchFilter> {
        /**
         * <p>
         * The key for the filter.
         * </p>
         * <p>
         * Run the <a>DescribePatchProperties</a> command to view lists of valid keys for each operating system type.
         * </p>
         * 
         * @param key
         *        The key for the filter.</p>
         *        <p>
         *        Run the <a>DescribePatchProperties</a> command to view lists of valid keys for each operating system
         *        type.
         * @see PatchFilterKey
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PatchFilterKey
         */
        Builder key(String key);

        /**
         * <p>
         * The key for the filter.
         * </p>
         * <p>
         * Run the <a>DescribePatchProperties</a> command to view lists of valid keys for each operating system type.
         * </p>
         * 
         * @param key
         *        The key for the filter.</p>
         *        <p>
         *        Run the <a>DescribePatchProperties</a> command to view lists of valid keys for each operating system
         *        type.
         * @see PatchFilterKey
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PatchFilterKey
         */
        Builder key(PatchFilterKey key);

        /**
         * <p>
         * The value for the filter key.
         * </p>
         * <p>
         * Run the <a>DescribePatchProperties</a> command to view lists of valid values for each key based on operating
         * system type.
         * </p>
         * 
         * @param values
         *        The value for the filter key.</p>
         *        <p>
         *        Run the <a>DescribePatchProperties</a> command to view lists of valid values for each key based on
         *        operating system type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder values(Collection<String> values);

        /**
         * <p>
         * The value for the filter key.
         * </p>
         * <p>
         * Run the <a>DescribePatchProperties</a> command to view lists of valid values for each key based on operating
         * system type.
         * </p>
         * 
         * @param values
         *        The value for the filter key.</p>
         *        <p>
         *        Run the <a>DescribePatchProperties</a> command to view lists of valid values for each key based on
         *        operating system type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder values(String... values);
    }

    static final class BuilderImpl implements Builder {
        private String key;

        private List<String> values = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(PatchFilter model) {
            key(model.key);
            values(model.values);
        }

        public final String getKeyAsString() {
            return key;
        }

        @Override
        public final Builder key(String key) {
            this.key = key;
            return this;
        }

        @Override
        public final Builder key(PatchFilterKey key) {
            this.key(key.toString());
            return this;
        }

        public final void setKey(String key) {
            this.key = key;
        }

        public final Collection<String> getValues() {
            return values;
        }

        @Override
        public final Builder values(Collection<String> values) {
            this.values = PatchFilterValueListCopier.copy(values);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder values(String... values) {
            values(Arrays.asList(values));
            return this;
        }

        public final void setValues(Collection<String> values) {
            this.values = PatchFilterValueListCopier.copy(values);
        }

        @Override
        public PatchFilter build() {
            return new PatchFilter(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
