/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about the target Amazon S3 bucket for the Resource Data Sync.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ResourceDataSyncS3Destination implements SdkPojo, Serializable,
        ToCopyableBuilder<ResourceDataSyncS3Destination.Builder, ResourceDataSyncS3Destination> {
    private static final SdkField<String> BUCKET_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ResourceDataSyncS3Destination::bucketName)).setter(setter(Builder::bucketName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BucketName").build()).build();

    private static final SdkField<String> PREFIX_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ResourceDataSyncS3Destination::prefix)).setter(setter(Builder::prefix))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Prefix").build()).build();

    private static final SdkField<String> SYNC_FORMAT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ResourceDataSyncS3Destination::syncFormatAsString)).setter(setter(Builder::syncFormat))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SyncFormat").build()).build();

    private static final SdkField<String> REGION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ResourceDataSyncS3Destination::region)).setter(setter(Builder::region))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Region").build()).build();

    private static final SdkField<String> AWSKMS_KEY_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ResourceDataSyncS3Destination::awskmsKeyARN)).setter(setter(Builder::awskmsKeyARN))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AWSKMSKeyARN").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(BUCKET_NAME_FIELD,
            PREFIX_FIELD, SYNC_FORMAT_FIELD, REGION_FIELD, AWSKMS_KEY_ARN_FIELD));

    private static final long serialVersionUID = 1L;

    private final String bucketName;

    private final String prefix;

    private final String syncFormat;

    private final String region;

    private final String awskmsKeyARN;

    private ResourceDataSyncS3Destination(BuilderImpl builder) {
        this.bucketName = builder.bucketName;
        this.prefix = builder.prefix;
        this.syncFormat = builder.syncFormat;
        this.region = builder.region;
        this.awskmsKeyARN = builder.awskmsKeyARN;
    }

    /**
     * <p>
     * The name of the Amazon S3 bucket where the aggregated data is stored.
     * </p>
     * 
     * @return The name of the Amazon S3 bucket where the aggregated data is stored.
     */
    public String bucketName() {
        return bucketName;
    }

    /**
     * <p>
     * An Amazon S3 prefix for the bucket.
     * </p>
     * 
     * @return An Amazon S3 prefix for the bucket.
     */
    public String prefix() {
        return prefix;
    }

    /**
     * <p>
     * A supported sync format. The following format is currently supported: JsonSerDe
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #syncFormat} will
     * return {@link ResourceDataSyncS3Format#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #syncFormatAsString}.
     * </p>
     * 
     * @return A supported sync format. The following format is currently supported: JsonSerDe
     * @see ResourceDataSyncS3Format
     */
    public ResourceDataSyncS3Format syncFormat() {
        return ResourceDataSyncS3Format.fromValue(syncFormat);
    }

    /**
     * <p>
     * A supported sync format. The following format is currently supported: JsonSerDe
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #syncFormat} will
     * return {@link ResourceDataSyncS3Format#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #syncFormatAsString}.
     * </p>
     * 
     * @return A supported sync format. The following format is currently supported: JsonSerDe
     * @see ResourceDataSyncS3Format
     */
    public String syncFormatAsString() {
        return syncFormat;
    }

    /**
     * <p>
     * The AWS Region with the Amazon S3 bucket targeted by the Resource Data Sync.
     * </p>
     * 
     * @return The AWS Region with the Amazon S3 bucket targeted by the Resource Data Sync.
     */
    public String region() {
        return region;
    }

    /**
     * <p>
     * The ARN of an encryption key for a destination in Amazon S3. Must belong to the same Region as the destination
     * Amazon S3 bucket.
     * </p>
     * 
     * @return The ARN of an encryption key for a destination in Amazon S3. Must belong to the same Region as the
     *         destination Amazon S3 bucket.
     */
    public String awskmsKeyARN() {
        return awskmsKeyARN;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(bucketName());
        hashCode = 31 * hashCode + Objects.hashCode(prefix());
        hashCode = 31 * hashCode + Objects.hashCode(syncFormatAsString());
        hashCode = 31 * hashCode + Objects.hashCode(region());
        hashCode = 31 * hashCode + Objects.hashCode(awskmsKeyARN());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ResourceDataSyncS3Destination)) {
            return false;
        }
        ResourceDataSyncS3Destination other = (ResourceDataSyncS3Destination) obj;
        return Objects.equals(bucketName(), other.bucketName()) && Objects.equals(prefix(), other.prefix())
                && Objects.equals(syncFormatAsString(), other.syncFormatAsString()) && Objects.equals(region(), other.region())
                && Objects.equals(awskmsKeyARN(), other.awskmsKeyARN());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("ResourceDataSyncS3Destination").add("BucketName", bucketName()).add("Prefix", prefix())
                .add("SyncFormat", syncFormatAsString()).add("Region", region()).add("AWSKMSKeyARN", awskmsKeyARN()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "BucketName":
            return Optional.ofNullable(clazz.cast(bucketName()));
        case "Prefix":
            return Optional.ofNullable(clazz.cast(prefix()));
        case "SyncFormat":
            return Optional.ofNullable(clazz.cast(syncFormatAsString()));
        case "Region":
            return Optional.ofNullable(clazz.cast(region()));
        case "AWSKMSKeyARN":
            return Optional.ofNullable(clazz.cast(awskmsKeyARN()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ResourceDataSyncS3Destination, T> g) {
        return obj -> g.apply((ResourceDataSyncS3Destination) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ResourceDataSyncS3Destination> {
        /**
         * <p>
         * The name of the Amazon S3 bucket where the aggregated data is stored.
         * </p>
         * 
         * @param bucketName
         *        The name of the Amazon S3 bucket where the aggregated data is stored.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bucketName(String bucketName);

        /**
         * <p>
         * An Amazon S3 prefix for the bucket.
         * </p>
         * 
         * @param prefix
         *        An Amazon S3 prefix for the bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder prefix(String prefix);

        /**
         * <p>
         * A supported sync format. The following format is currently supported: JsonSerDe
         * </p>
         * 
         * @param syncFormat
         *        A supported sync format. The following format is currently supported: JsonSerDe
         * @see ResourceDataSyncS3Format
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResourceDataSyncS3Format
         */
        Builder syncFormat(String syncFormat);

        /**
         * <p>
         * A supported sync format. The following format is currently supported: JsonSerDe
         * </p>
         * 
         * @param syncFormat
         *        A supported sync format. The following format is currently supported: JsonSerDe
         * @see ResourceDataSyncS3Format
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResourceDataSyncS3Format
         */
        Builder syncFormat(ResourceDataSyncS3Format syncFormat);

        /**
         * <p>
         * The AWS Region with the Amazon S3 bucket targeted by the Resource Data Sync.
         * </p>
         * 
         * @param region
         *        The AWS Region with the Amazon S3 bucket targeted by the Resource Data Sync.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder region(String region);

        /**
         * <p>
         * The ARN of an encryption key for a destination in Amazon S3. Must belong to the same Region as the
         * destination Amazon S3 bucket.
         * </p>
         * 
         * @param awskmsKeyARN
         *        The ARN of an encryption key for a destination in Amazon S3. Must belong to the same Region as the
         *        destination Amazon S3 bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder awskmsKeyARN(String awskmsKeyARN);
    }

    static final class BuilderImpl implements Builder {
        private String bucketName;

        private String prefix;

        private String syncFormat;

        private String region;

        private String awskmsKeyARN;

        private BuilderImpl() {
        }

        private BuilderImpl(ResourceDataSyncS3Destination model) {
            bucketName(model.bucketName);
            prefix(model.prefix);
            syncFormat(model.syncFormat);
            region(model.region);
            awskmsKeyARN(model.awskmsKeyARN);
        }

        public final String getBucketName() {
            return bucketName;
        }

        @Override
        public final Builder bucketName(String bucketName) {
            this.bucketName = bucketName;
            return this;
        }

        public final void setBucketName(String bucketName) {
            this.bucketName = bucketName;
        }

        public final String getPrefix() {
            return prefix;
        }

        @Override
        public final Builder prefix(String prefix) {
            this.prefix = prefix;
            return this;
        }

        public final void setPrefix(String prefix) {
            this.prefix = prefix;
        }

        public final String getSyncFormatAsString() {
            return syncFormat;
        }

        @Override
        public final Builder syncFormat(String syncFormat) {
            this.syncFormat = syncFormat;
            return this;
        }

        @Override
        public final Builder syncFormat(ResourceDataSyncS3Format syncFormat) {
            this.syncFormat(syncFormat.toString());
            return this;
        }

        public final void setSyncFormat(String syncFormat) {
            this.syncFormat = syncFormat;
        }

        public final String getRegion() {
            return region;
        }

        @Override
        public final Builder region(String region) {
            this.region = region;
            return this;
        }

        public final void setRegion(String region) {
            this.region = region;
        }

        public final String getAwskmsKeyARN() {
            return awskmsKeyARN;
        }

        @Override
        public final Builder awskmsKeyARN(String awskmsKeyARN) {
            this.awskmsKeyARN = awskmsKeyARN;
            return this;
        }

        public final void setAwskmsKeyARN(String awskmsKeyARN) {
            this.awskmsKeyARN = awskmsKeyARN;
        }

        @Override
        public ResourceDataSyncS3Destination build() {
            return new ResourceDataSyncS3Destination(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
