/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.utils.CollectionUtils;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about a task defined for a maintenance window.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class MaintenanceWindowTask implements SdkPojo, Serializable,
        ToCopyableBuilder<MaintenanceWindowTask.Builder, MaintenanceWindowTask> {
    private static final SdkField<String> WINDOW_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(MaintenanceWindowTask::windowId)).setter(setter(Builder::windowId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WindowId").build()).build();

    private static final SdkField<String> WINDOW_TASK_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(MaintenanceWindowTask::windowTaskId)).setter(setter(Builder::windowTaskId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WindowTaskId").build()).build();

    private static final SdkField<String> TASK_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(MaintenanceWindowTask::taskArn)).setter(setter(Builder::taskArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TaskArn").build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(MaintenanceWindowTask::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Type").build()).build();

    private static final SdkField<List<Target>> TARGETS_FIELD = SdkField
            .<List<Target>> builder(MarshallingType.LIST)
            .getter(getter(MaintenanceWindowTask::targets))
            .setter(setter(Builder::targets))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Targets").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Target> builder(MarshallingType.SDK_POJO)
                                            .constructor(Target::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Map<String, MaintenanceWindowTaskParameterValueExpression>> TASK_PARAMETERS_FIELD = SdkField
            .<Map<String, MaintenanceWindowTaskParameterValueExpression>> builder(MarshallingType.MAP)
            .getter(getter(MaintenanceWindowTask::taskParameters))
            .setter(setter(Builder::taskParameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TaskParameters").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<MaintenanceWindowTaskParameterValueExpression> builder(MarshallingType.SDK_POJO)
                                            .constructor(MaintenanceWindowTaskParameterValueExpression::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Integer> PRIORITY_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(MaintenanceWindowTask::priority)).setter(setter(Builder::priority))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Priority").build()).build();

    private static final SdkField<LoggingInfo> LOGGING_INFO_FIELD = SdkField.<LoggingInfo> builder(MarshallingType.SDK_POJO)
            .getter(getter(MaintenanceWindowTask::loggingInfo)).setter(setter(Builder::loggingInfo))
            .constructor(LoggingInfo::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LoggingInfo").build()).build();

    private static final SdkField<String> SERVICE_ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(MaintenanceWindowTask::serviceRoleArn)).setter(setter(Builder::serviceRoleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServiceRoleArn").build()).build();

    private static final SdkField<String> MAX_CONCURRENCY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(MaintenanceWindowTask::maxConcurrency)).setter(setter(Builder::maxConcurrency))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxConcurrency").build()).build();

    private static final SdkField<String> MAX_ERRORS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(MaintenanceWindowTask::maxErrors)).setter(setter(Builder::maxErrors))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxErrors").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(MaintenanceWindowTask::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(MaintenanceWindowTask::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(WINDOW_ID_FIELD,
            WINDOW_TASK_ID_FIELD, TASK_ARN_FIELD, TYPE_FIELD, TARGETS_FIELD, TASK_PARAMETERS_FIELD, PRIORITY_FIELD,
            LOGGING_INFO_FIELD, SERVICE_ROLE_ARN_FIELD, MAX_CONCURRENCY_FIELD, MAX_ERRORS_FIELD, NAME_FIELD, DESCRIPTION_FIELD));

    private static final long serialVersionUID = 1L;

    private final String windowId;

    private final String windowTaskId;

    private final String taskArn;

    private final String type;

    private final List<Target> targets;

    private final Map<String, MaintenanceWindowTaskParameterValueExpression> taskParameters;

    private final Integer priority;

    private final LoggingInfo loggingInfo;

    private final String serviceRoleArn;

    private final String maxConcurrency;

    private final String maxErrors;

    private final String name;

    private final String description;

    private MaintenanceWindowTask(BuilderImpl builder) {
        this.windowId = builder.windowId;
        this.windowTaskId = builder.windowTaskId;
        this.taskArn = builder.taskArn;
        this.type = builder.type;
        this.targets = builder.targets;
        this.taskParameters = builder.taskParameters;
        this.priority = builder.priority;
        this.loggingInfo = builder.loggingInfo;
        this.serviceRoleArn = builder.serviceRoleArn;
        this.maxConcurrency = builder.maxConcurrency;
        this.maxErrors = builder.maxErrors;
        this.name = builder.name;
        this.description = builder.description;
    }

    /**
     * <p>
     * The ID of the maintenance window where the task is registered.
     * </p>
     * 
     * @return The ID of the maintenance window where the task is registered.
     */
    public String windowId() {
        return windowId;
    }

    /**
     * <p>
     * The task ID.
     * </p>
     * 
     * @return The task ID.
     */
    public String windowTaskId() {
        return windowTaskId;
    }

    /**
     * <p>
     * The resource that the task uses during execution. For RUN_COMMAND and AUTOMATION task types, <code>TaskArn</code>
     * is the Systems Manager document name or ARN. For LAMBDA tasks, it's the function name or ARN. For STEP_FUNCTION
     * tasks, it's the state machine ARN.
     * </p>
     * 
     * @return The resource that the task uses during execution. For RUN_COMMAND and AUTOMATION task types,
     *         <code>TaskArn</code> is the Systems Manager document name or ARN. For LAMBDA tasks, it's the function
     *         name or ARN. For STEP_FUNCTION tasks, it's the state machine ARN.
     */
    public String taskArn() {
        return taskArn;
    }

    /**
     * <p>
     * The type of task. The type can be one of the following: RUN_COMMAND, AUTOMATION, LAMBDA, or STEP_FUNCTION.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link MaintenanceWindowTaskType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of task. The type can be one of the following: RUN_COMMAND, AUTOMATION, LAMBDA, or
     *         STEP_FUNCTION.
     * @see MaintenanceWindowTaskType
     */
    public MaintenanceWindowTaskType type() {
        return MaintenanceWindowTaskType.fromValue(type);
    }

    /**
     * <p>
     * The type of task. The type can be one of the following: RUN_COMMAND, AUTOMATION, LAMBDA, or STEP_FUNCTION.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link MaintenanceWindowTaskType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of task. The type can be one of the following: RUN_COMMAND, AUTOMATION, LAMBDA, or
     *         STEP_FUNCTION.
     * @see MaintenanceWindowTaskType
     */
    public String typeAsString() {
        return type;
    }

    /**
     * <p>
     * The targets (either instances or tags). Instances are specified using
     * Key=instanceids,Values=&lt;instanceid1&gt;,&lt;instanceid2&gt;. Tags are specified using Key=&lt;tag
     * name&gt;,Values=&lt;tag value&gt;.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The targets (either instances or tags). Instances are specified using
     *         Key=instanceids,Values=&lt;instanceid1&gt;,&lt;instanceid2&gt;. Tags are specified using Key=&lt;tag
     *         name&gt;,Values=&lt;tag value&gt;.
     */
    public List<Target> targets() {
        return targets;
    }

    /**
     * <p>
     * The parameters that should be passed to the task when it is run.
     * </p>
     * <note>
     * <p>
     * <code>TaskParameters</code> has been deprecated. To specify parameters to pass to a task when it runs, instead
     * use the <code>Parameters</code> option in the <code>TaskInvocationParameters</code> structure. For information
     * about how Systems Manager handles these options for the supported maintenance window task types, see
     * <a>MaintenanceWindowTaskInvocationParameters</a>.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The parameters that should be passed to the task when it is run.</p> <note>
     *         <p>
     *         <code>TaskParameters</code> has been deprecated. To specify parameters to pass to a task when it runs,
     *         instead use the <code>Parameters</code> option in the <code>TaskInvocationParameters</code> structure.
     *         For information about how Systems Manager handles these options for the supported maintenance window task
     *         types, see <a>MaintenanceWindowTaskInvocationParameters</a>.
     *         </p>
     */
    public Map<String, MaintenanceWindowTaskParameterValueExpression> taskParameters() {
        return taskParameters;
    }

    /**
     * <p>
     * The priority of the task in the maintenance window. The lower the number, the higher the priority. Tasks that
     * have the same priority are scheduled in parallel.
     * </p>
     * 
     * @return The priority of the task in the maintenance window. The lower the number, the higher the priority. Tasks
     *         that have the same priority are scheduled in parallel.
     */
    public Integer priority() {
        return priority;
    }

    /**
     * <p>
     * Information about an Amazon S3 bucket to write task-level logs to.
     * </p>
     * <note>
     * <p>
     * <code>LoggingInfo</code> has been deprecated. To specify an S3 bucket to contain logs, instead use the
     * <code>OutputS3BucketName</code> and <code>OutputS3KeyPrefix</code> options in the
     * <code>TaskInvocationParameters</code> structure. For information about how Systems Manager handles these options
     * for the supported maintenance window task types, see <a>MaintenanceWindowTaskInvocationParameters</a>.
     * </p>
     * </note>
     * 
     * @return Information about an Amazon S3 bucket to write task-level logs to.</p> <note>
     *         <p>
     *         <code>LoggingInfo</code> has been deprecated. To specify an S3 bucket to contain logs, instead use the
     *         <code>OutputS3BucketName</code> and <code>OutputS3KeyPrefix</code> options in the
     *         <code>TaskInvocationParameters</code> structure. For information about how Systems Manager handles these
     *         options for the supported maintenance window task types, see
     *         <a>MaintenanceWindowTaskInvocationParameters</a>.
     *         </p>
     */
    public LoggingInfo loggingInfo() {
        return loggingInfo;
    }

    /**
     * <p>
     * The ARN of the IAM service role to use to publish Amazon Simple Notification Service (Amazon SNS) notifications
     * for maintenance window Run Command tasks.
     * </p>
     * 
     * @return The ARN of the IAM service role to use to publish Amazon Simple Notification Service (Amazon SNS)
     *         notifications for maintenance window Run Command tasks.
     */
    public String serviceRoleArn() {
        return serviceRoleArn;
    }

    /**
     * <p>
     * The maximum number of targets this task can be run for, in parallel.
     * </p>
     * 
     * @return The maximum number of targets this task can be run for, in parallel.
     */
    public String maxConcurrency() {
        return maxConcurrency;
    }

    /**
     * <p>
     * The maximum number of errors allowed before this task stops being scheduled.
     * </p>
     * 
     * @return The maximum number of errors allowed before this task stops being scheduled.
     */
    public String maxErrors() {
        return maxErrors;
    }

    /**
     * <p>
     * The task name.
     * </p>
     * 
     * @return The task name.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * A description of the task.
     * </p>
     * 
     * @return A description of the task.
     */
    public String description() {
        return description;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(windowId());
        hashCode = 31 * hashCode + Objects.hashCode(windowTaskId());
        hashCode = 31 * hashCode + Objects.hashCode(taskArn());
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(targets());
        hashCode = 31 * hashCode + Objects.hashCode(taskParameters());
        hashCode = 31 * hashCode + Objects.hashCode(priority());
        hashCode = 31 * hashCode + Objects.hashCode(loggingInfo());
        hashCode = 31 * hashCode + Objects.hashCode(serviceRoleArn());
        hashCode = 31 * hashCode + Objects.hashCode(maxConcurrency());
        hashCode = 31 * hashCode + Objects.hashCode(maxErrors());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof MaintenanceWindowTask)) {
            return false;
        }
        MaintenanceWindowTask other = (MaintenanceWindowTask) obj;
        return Objects.equals(windowId(), other.windowId()) && Objects.equals(windowTaskId(), other.windowTaskId())
                && Objects.equals(taskArn(), other.taskArn()) && Objects.equals(typeAsString(), other.typeAsString())
                && Objects.equals(targets(), other.targets()) && Objects.equals(taskParameters(), other.taskParameters())
                && Objects.equals(priority(), other.priority()) && Objects.equals(loggingInfo(), other.loggingInfo())
                && Objects.equals(serviceRoleArn(), other.serviceRoleArn())
                && Objects.equals(maxConcurrency(), other.maxConcurrency()) && Objects.equals(maxErrors(), other.maxErrors())
                && Objects.equals(name(), other.name()) && Objects.equals(description(), other.description());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("MaintenanceWindowTask").add("WindowId", windowId()).add("WindowTaskId", windowTaskId())
                .add("TaskArn", taskArn()).add("Type", typeAsString()).add("Targets", targets())
                .add("TaskParameters", taskParameters() == null ? null : "*** Sensitive Data Redacted ***")
                .add("Priority", priority()).add("LoggingInfo", loggingInfo()).add("ServiceRoleArn", serviceRoleArn())
                .add("MaxConcurrency", maxConcurrency()).add("MaxErrors", maxErrors()).add("Name", name())
                .add("Description", description() == null ? null : "*** Sensitive Data Redacted ***").build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "WindowId":
            return Optional.ofNullable(clazz.cast(windowId()));
        case "WindowTaskId":
            return Optional.ofNullable(clazz.cast(windowTaskId()));
        case "TaskArn":
            return Optional.ofNullable(clazz.cast(taskArn()));
        case "Type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "Targets":
            return Optional.ofNullable(clazz.cast(targets()));
        case "TaskParameters":
            return Optional.ofNullable(clazz.cast(taskParameters()));
        case "Priority":
            return Optional.ofNullable(clazz.cast(priority()));
        case "LoggingInfo":
            return Optional.ofNullable(clazz.cast(loggingInfo()));
        case "ServiceRoleArn":
            return Optional.ofNullable(clazz.cast(serviceRoleArn()));
        case "MaxConcurrency":
            return Optional.ofNullable(clazz.cast(maxConcurrency()));
        case "MaxErrors":
            return Optional.ofNullable(clazz.cast(maxErrors()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<MaintenanceWindowTask, T> g) {
        return obj -> g.apply((MaintenanceWindowTask) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, MaintenanceWindowTask> {
        /**
         * <p>
         * The ID of the maintenance window where the task is registered.
         * </p>
         * 
         * @param windowId
         *        The ID of the maintenance window where the task is registered.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder windowId(String windowId);

        /**
         * <p>
         * The task ID.
         * </p>
         * 
         * @param windowTaskId
         *        The task ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder windowTaskId(String windowTaskId);

        /**
         * <p>
         * The resource that the task uses during execution. For RUN_COMMAND and AUTOMATION task types,
         * <code>TaskArn</code> is the Systems Manager document name or ARN. For LAMBDA tasks, it's the function name or
         * ARN. For STEP_FUNCTION tasks, it's the state machine ARN.
         * </p>
         * 
         * @param taskArn
         *        The resource that the task uses during execution. For RUN_COMMAND and AUTOMATION task types,
         *        <code>TaskArn</code> is the Systems Manager document name or ARN. For LAMBDA tasks, it's the function
         *        name or ARN. For STEP_FUNCTION tasks, it's the state machine ARN.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder taskArn(String taskArn);

        /**
         * <p>
         * The type of task. The type can be one of the following: RUN_COMMAND, AUTOMATION, LAMBDA, or STEP_FUNCTION.
         * </p>
         * 
         * @param type
         *        The type of task. The type can be one of the following: RUN_COMMAND, AUTOMATION, LAMBDA, or
         *        STEP_FUNCTION.
         * @see MaintenanceWindowTaskType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MaintenanceWindowTaskType
         */
        Builder type(String type);

        /**
         * <p>
         * The type of task. The type can be one of the following: RUN_COMMAND, AUTOMATION, LAMBDA, or STEP_FUNCTION.
         * </p>
         * 
         * @param type
         *        The type of task. The type can be one of the following: RUN_COMMAND, AUTOMATION, LAMBDA, or
         *        STEP_FUNCTION.
         * @see MaintenanceWindowTaskType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MaintenanceWindowTaskType
         */
        Builder type(MaintenanceWindowTaskType type);

        /**
         * <p>
         * The targets (either instances or tags). Instances are specified using
         * Key=instanceids,Values=&lt;instanceid1&gt;,&lt;instanceid2&gt;. Tags are specified using Key=&lt;tag
         * name&gt;,Values=&lt;tag value&gt;.
         * </p>
         * 
         * @param targets
         *        The targets (either instances or tags). Instances are specified using
         *        Key=instanceids,Values=&lt;instanceid1&gt;,&lt;instanceid2&gt;. Tags are specified using Key=&lt;tag
         *        name&gt;,Values=&lt;tag value&gt;.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targets(Collection<Target> targets);

        /**
         * <p>
         * The targets (either instances or tags). Instances are specified using
         * Key=instanceids,Values=&lt;instanceid1&gt;,&lt;instanceid2&gt;. Tags are specified using Key=&lt;tag
         * name&gt;,Values=&lt;tag value&gt;.
         * </p>
         * 
         * @param targets
         *        The targets (either instances or tags). Instances are specified using
         *        Key=instanceids,Values=&lt;instanceid1&gt;,&lt;instanceid2&gt;. Tags are specified using Key=&lt;tag
         *        name&gt;,Values=&lt;tag value&gt;.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targets(Target... targets);

        /**
         * <p>
         * The targets (either instances or tags). Instances are specified using
         * Key=instanceids,Values=&lt;instanceid1&gt;,&lt;instanceid2&gt;. Tags are specified using Key=&lt;tag
         * name&gt;,Values=&lt;tag value&gt;.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Target>.Builder} avoiding the need to
         * create one manually via {@link List<Target>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Target>.Builder#build()} is called immediately and its
         * result is passed to {@link #targets(List<Target>)}.
         * 
         * @param targets
         *        a consumer that will call methods on {@link List<Target>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #targets(List<Target>)
         */
        Builder targets(Consumer<Target.Builder>... targets);

        /**
         * <p>
         * The parameters that should be passed to the task when it is run.
         * </p>
         * <note>
         * <p>
         * <code>TaskParameters</code> has been deprecated. To specify parameters to pass to a task when it runs,
         * instead use the <code>Parameters</code> option in the <code>TaskInvocationParameters</code> structure. For
         * information about how Systems Manager handles these options for the supported maintenance window task types,
         * see <a>MaintenanceWindowTaskInvocationParameters</a>.
         * </p>
         * </note>
         * 
         * @param taskParameters
         *        The parameters that should be passed to the task when it is run.</p> <note>
         *        <p>
         *        <code>TaskParameters</code> has been deprecated. To specify parameters to pass to a task when it runs,
         *        instead use the <code>Parameters</code> option in the <code>TaskInvocationParameters</code> structure.
         *        For information about how Systems Manager handles these options for the supported maintenance window
         *        task types, see <a>MaintenanceWindowTaskInvocationParameters</a>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder taskParameters(Map<String, MaintenanceWindowTaskParameterValueExpression> taskParameters);

        /**
         * <p>
         * The priority of the task in the maintenance window. The lower the number, the higher the priority. Tasks that
         * have the same priority are scheduled in parallel.
         * </p>
         * 
         * @param priority
         *        The priority of the task in the maintenance window. The lower the number, the higher the priority.
         *        Tasks that have the same priority are scheduled in parallel.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder priority(Integer priority);

        /**
         * <p>
         * Information about an Amazon S3 bucket to write task-level logs to.
         * </p>
         * <note>
         * <p>
         * <code>LoggingInfo</code> has been deprecated. To specify an S3 bucket to contain logs, instead use the
         * <code>OutputS3BucketName</code> and <code>OutputS3KeyPrefix</code> options in the
         * <code>TaskInvocationParameters</code> structure. For information about how Systems Manager handles these
         * options for the supported maintenance window task types, see
         * <a>MaintenanceWindowTaskInvocationParameters</a>.
         * </p>
         * </note>
         * 
         * @param loggingInfo
         *        Information about an Amazon S3 bucket to write task-level logs to.</p> <note>
         *        <p>
         *        <code>LoggingInfo</code> has been deprecated. To specify an S3 bucket to contain logs, instead use the
         *        <code>OutputS3BucketName</code> and <code>OutputS3KeyPrefix</code> options in the
         *        <code>TaskInvocationParameters</code> structure. For information about how Systems Manager handles
         *        these options for the supported maintenance window task types, see
         *        <a>MaintenanceWindowTaskInvocationParameters</a>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder loggingInfo(LoggingInfo loggingInfo);

        /**
         * <p>
         * Information about an Amazon S3 bucket to write task-level logs to.
         * </p>
         * <note>
         * <p>
         * <code>LoggingInfo</code> has been deprecated. To specify an S3 bucket to contain logs, instead use the
         * <code>OutputS3BucketName</code> and <code>OutputS3KeyPrefix</code> options in the
         * <code>TaskInvocationParameters</code> structure. For information about how Systems Manager handles these
         * options for the supported maintenance window task types, see
         * <a>MaintenanceWindowTaskInvocationParameters</a>.
         * </p>
         * </note> This is a convenience that creates an instance of the {@link LoggingInfo.Builder} avoiding the need
         * to create one manually via {@link LoggingInfo#builder()}.
         *
         * When the {@link Consumer} completes, {@link LoggingInfo.Builder#build()} is called immediately and its result
         * is passed to {@link #loggingInfo(LoggingInfo)}.
         * 
         * @param loggingInfo
         *        a consumer that will call methods on {@link LoggingInfo.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #loggingInfo(LoggingInfo)
         */
        default Builder loggingInfo(Consumer<LoggingInfo.Builder> loggingInfo) {
            return loggingInfo(LoggingInfo.builder().applyMutation(loggingInfo).build());
        }

        /**
         * <p>
         * The ARN of the IAM service role to use to publish Amazon Simple Notification Service (Amazon SNS)
         * notifications for maintenance window Run Command tasks.
         * </p>
         * 
         * @param serviceRoleArn
         *        The ARN of the IAM service role to use to publish Amazon Simple Notification Service (Amazon SNS)
         *        notifications for maintenance window Run Command tasks.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serviceRoleArn(String serviceRoleArn);

        /**
         * <p>
         * The maximum number of targets this task can be run for, in parallel.
         * </p>
         * 
         * @param maxConcurrency
         *        The maximum number of targets this task can be run for, in parallel.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxConcurrency(String maxConcurrency);

        /**
         * <p>
         * The maximum number of errors allowed before this task stops being scheduled.
         * </p>
         * 
         * @param maxErrors
         *        The maximum number of errors allowed before this task stops being scheduled.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxErrors(String maxErrors);

        /**
         * <p>
         * The task name.
         * </p>
         * 
         * @param name
         *        The task name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * A description of the task.
         * </p>
         * 
         * @param description
         *        A description of the task.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);
    }

    static final class BuilderImpl implements Builder {
        private String windowId;

        private String windowTaskId;

        private String taskArn;

        private String type;

        private List<Target> targets = DefaultSdkAutoConstructList.getInstance();

        private Map<String, MaintenanceWindowTaskParameterValueExpression> taskParameters = DefaultSdkAutoConstructMap
                .getInstance();

        private Integer priority;

        private LoggingInfo loggingInfo;

        private String serviceRoleArn;

        private String maxConcurrency;

        private String maxErrors;

        private String name;

        private String description;

        private BuilderImpl() {
        }

        private BuilderImpl(MaintenanceWindowTask model) {
            windowId(model.windowId);
            windowTaskId(model.windowTaskId);
            taskArn(model.taskArn);
            type(model.type);
            targets(model.targets);
            taskParameters(model.taskParameters);
            priority(model.priority);
            loggingInfo(model.loggingInfo);
            serviceRoleArn(model.serviceRoleArn);
            maxConcurrency(model.maxConcurrency);
            maxErrors(model.maxErrors);
            name(model.name);
            description(model.description);
        }

        public final String getWindowId() {
            return windowId;
        }

        @Override
        public final Builder windowId(String windowId) {
            this.windowId = windowId;
            return this;
        }

        public final void setWindowId(String windowId) {
            this.windowId = windowId;
        }

        public final String getWindowTaskId() {
            return windowTaskId;
        }

        @Override
        public final Builder windowTaskId(String windowTaskId) {
            this.windowTaskId = windowTaskId;
            return this;
        }

        public final void setWindowTaskId(String windowTaskId) {
            this.windowTaskId = windowTaskId;
        }

        public final String getTaskArn() {
            return taskArn;
        }

        @Override
        public final Builder taskArn(String taskArn) {
            this.taskArn = taskArn;
            return this;
        }

        public final void setTaskArn(String taskArn) {
            this.taskArn = taskArn;
        }

        public final String getTypeAsString() {
            return type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(MaintenanceWindowTaskType type) {
            this.type(type.toString());
            return this;
        }

        public final void setType(String type) {
            this.type = type;
        }

        public final Collection<Target.Builder> getTargets() {
            return targets != null ? targets.stream().map(Target::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder targets(Collection<Target> targets) {
            this.targets = TargetsCopier.copy(targets);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder targets(Target... targets) {
            targets(Arrays.asList(targets));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder targets(Consumer<Target.Builder>... targets) {
            targets(Stream.of(targets).map(c -> Target.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTargets(Collection<Target.BuilderImpl> targets) {
            this.targets = TargetsCopier.copyFromBuilder(targets);
        }

        public final Map<String, MaintenanceWindowTaskParameterValueExpression.Builder> getTaskParameters() {
            return taskParameters != null ? CollectionUtils.mapValues(taskParameters,
                    MaintenanceWindowTaskParameterValueExpression::toBuilder) : null;
        }

        @Override
        public final Builder taskParameters(Map<String, MaintenanceWindowTaskParameterValueExpression> taskParameters) {
            this.taskParameters = MaintenanceWindowTaskParametersCopier.copy(taskParameters);
            return this;
        }

        public final void setTaskParameters(Map<String, MaintenanceWindowTaskParameterValueExpression.BuilderImpl> taskParameters) {
            this.taskParameters = MaintenanceWindowTaskParametersCopier.copyFromBuilder(taskParameters);
        }

        public final Integer getPriority() {
            return priority;
        }

        @Override
        public final Builder priority(Integer priority) {
            this.priority = priority;
            return this;
        }

        public final void setPriority(Integer priority) {
            this.priority = priority;
        }

        public final LoggingInfo.Builder getLoggingInfo() {
            return loggingInfo != null ? loggingInfo.toBuilder() : null;
        }

        @Override
        public final Builder loggingInfo(LoggingInfo loggingInfo) {
            this.loggingInfo = loggingInfo;
            return this;
        }

        public final void setLoggingInfo(LoggingInfo.BuilderImpl loggingInfo) {
            this.loggingInfo = loggingInfo != null ? loggingInfo.build() : null;
        }

        public final String getServiceRoleArn() {
            return serviceRoleArn;
        }

        @Override
        public final Builder serviceRoleArn(String serviceRoleArn) {
            this.serviceRoleArn = serviceRoleArn;
            return this;
        }

        public final void setServiceRoleArn(String serviceRoleArn) {
            this.serviceRoleArn = serviceRoleArn;
        }

        public final String getMaxConcurrency() {
            return maxConcurrency;
        }

        @Override
        public final Builder maxConcurrency(String maxConcurrency) {
            this.maxConcurrency = maxConcurrency;
            return this;
        }

        public final void setMaxConcurrency(String maxConcurrency) {
            this.maxConcurrency = maxConcurrency;
        }

        public final String getMaxErrors() {
            return maxErrors;
        }

        @Override
        public final Builder maxErrors(String maxErrors) {
            this.maxErrors = maxErrors;
            return this;
        }

        public final void setMaxErrors(String maxErrors) {
            this.maxErrors = maxErrors;
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public MaintenanceWindowTask build() {
            return new MaintenanceWindowTask(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
