/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.utils.CollectionUtils;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Operations engineers and IT professionals use OpsCenter to view, investigate, and remediate operational issues
 * impacting the performance and health of their AWS resources. For more information, see <a
 * href="http://docs.aws.amazon.com/systems-manager/latest/userguide/OpsCenter.html">AWS Systems Manager OpsCenter</a>
 * in the <i>AWS Systems Manager User Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class OpsItem implements SdkPojo, Serializable, ToCopyableBuilder<OpsItem.Builder, OpsItem> {
    private static final SdkField<String> CREATED_BY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(OpsItem::createdBy)).setter(setter(Builder::createdBy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedBy").build()).build();

    private static final SdkField<Instant> CREATED_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(OpsItem::createdTime)).setter(setter(Builder::createdTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedTime").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(OpsItem::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<String> LAST_MODIFIED_BY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(OpsItem::lastModifiedBy)).setter(setter(Builder::lastModifiedBy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastModifiedBy").build()).build();

    private static final SdkField<Instant> LAST_MODIFIED_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(OpsItem::lastModifiedTime)).setter(setter(Builder::lastModifiedTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastModifiedTime").build()).build();

    private static final SdkField<List<OpsItemNotification>> NOTIFICATIONS_FIELD = SdkField
            .<List<OpsItemNotification>> builder(MarshallingType.LIST)
            .getter(getter(OpsItem::notifications))
            .setter(setter(Builder::notifications))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Notifications").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<OpsItemNotification> builder(MarshallingType.SDK_POJO)
                                            .constructor(OpsItemNotification::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Integer> PRIORITY_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(OpsItem::priority)).setter(setter(Builder::priority))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Priority").build()).build();

    private static final SdkField<List<RelatedOpsItem>> RELATED_OPS_ITEMS_FIELD = SdkField
            .<List<RelatedOpsItem>> builder(MarshallingType.LIST)
            .getter(getter(OpsItem::relatedOpsItems))
            .setter(setter(Builder::relatedOpsItems))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RelatedOpsItems").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<RelatedOpsItem> builder(MarshallingType.SDK_POJO)
                                            .constructor(RelatedOpsItem::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(OpsItem::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<String> OPS_ITEM_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(OpsItem::opsItemId)).setter(setter(Builder::opsItemId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OpsItemId").build()).build();

    private static final SdkField<String> VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(OpsItem::version)).setter(setter(Builder::version))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Version").build()).build();

    private static final SdkField<String> TITLE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(OpsItem::title)).setter(setter(Builder::title))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Title").build()).build();

    private static final SdkField<String> SOURCE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(OpsItem::source)).setter(setter(Builder::source))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Source").build()).build();

    private static final SdkField<Map<String, OpsItemDataValue>> OPERATIONAL_DATA_FIELD = SdkField
            .<Map<String, OpsItemDataValue>> builder(MarshallingType.MAP)
            .getter(getter(OpsItem::operationalData))
            .setter(setter(Builder::operationalData))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OperationalData").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<OpsItemDataValue> builder(MarshallingType.SDK_POJO)
                                            .constructor(OpsItemDataValue::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CREATED_BY_FIELD,
            CREATED_TIME_FIELD, DESCRIPTION_FIELD, LAST_MODIFIED_BY_FIELD, LAST_MODIFIED_TIME_FIELD, NOTIFICATIONS_FIELD,
            PRIORITY_FIELD, RELATED_OPS_ITEMS_FIELD, STATUS_FIELD, OPS_ITEM_ID_FIELD, VERSION_FIELD, TITLE_FIELD, SOURCE_FIELD,
            OPERATIONAL_DATA_FIELD));

    private static final long serialVersionUID = 1L;

    private final String createdBy;

    private final Instant createdTime;

    private final String description;

    private final String lastModifiedBy;

    private final Instant lastModifiedTime;

    private final List<OpsItemNotification> notifications;

    private final Integer priority;

    private final List<RelatedOpsItem> relatedOpsItems;

    private final String status;

    private final String opsItemId;

    private final String version;

    private final String title;

    private final String source;

    private final Map<String, OpsItemDataValue> operationalData;

    private OpsItem(BuilderImpl builder) {
        this.createdBy = builder.createdBy;
        this.createdTime = builder.createdTime;
        this.description = builder.description;
        this.lastModifiedBy = builder.lastModifiedBy;
        this.lastModifiedTime = builder.lastModifiedTime;
        this.notifications = builder.notifications;
        this.priority = builder.priority;
        this.relatedOpsItems = builder.relatedOpsItems;
        this.status = builder.status;
        this.opsItemId = builder.opsItemId;
        this.version = builder.version;
        this.title = builder.title;
        this.source = builder.source;
        this.operationalData = builder.operationalData;
    }

    /**
     * <p>
     * The ARN of the AWS account that created the OpsItem.
     * </p>
     * 
     * @return The ARN of the AWS account that created the OpsItem.
     */
    public String createdBy() {
        return createdBy;
    }

    /**
     * <p>
     * The date and time the OpsItem was created.
     * </p>
     * 
     * @return The date and time the OpsItem was created.
     */
    public Instant createdTime() {
        return createdTime;
    }

    /**
     * <p>
     * The OpsItem description.
     * </p>
     * 
     * @return The OpsItem description.
     */
    public String description() {
        return description;
    }

    /**
     * <p>
     * The ARN of the AWS account that last updated the OpsItem.
     * </p>
     * 
     * @return The ARN of the AWS account that last updated the OpsItem.
     */
    public String lastModifiedBy() {
        return lastModifiedBy;
    }

    /**
     * <p>
     * The date and time the OpsItem was last updated.
     * </p>
     * 
     * @return The date and time the OpsItem was last updated.
     */
    public Instant lastModifiedTime() {
        return lastModifiedTime;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of an SNS topic where notifications are sent when this OpsItem is edited or
     * changed.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of an SNS topic where notifications are sent when this OpsItem is edited
     *         or changed.
     */
    public List<OpsItemNotification> notifications() {
        return notifications;
    }

    /**
     * <p>
     * The importance of this OpsItem in relation to other OpsItems in the system.
     * </p>
     * 
     * @return The importance of this OpsItem in relation to other OpsItems in the system.
     */
    public Integer priority() {
        return priority;
    }

    /**
     * <p>
     * One or more OpsItems that share something in common with the current OpsItem. For example, related OpsItems can
     * include OpsItems with similar error messages, impacted resources, or statuses for the impacted resource.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return One or more OpsItems that share something in common with the current OpsItem. For example, related
     *         OpsItems can include OpsItems with similar error messages, impacted resources, or statuses for the
     *         impacted resource.
     */
    public List<RelatedOpsItem> relatedOpsItems() {
        return relatedOpsItems;
    }

    /**
     * <p>
     * The OpsItem status. Status can be <code>Open</code>, <code>In Progress</code>, or <code>Resolved</code>. For more
     * information, see <a href=
     * "http://docs.aws.amazon.com/systems-manager/latest/userguide/OpsCenter-working-with-OpsItems-editing-details.html"
     * >Editing OpsItem Details</a> in the <i>AWS Systems Manager User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link OpsItemStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The OpsItem status. Status can be <code>Open</code>, <code>In Progress</code>, or <code>Resolved</code>.
     *         For more information, see <a href=
     *         "http://docs.aws.amazon.com/systems-manager/latest/userguide/OpsCenter-working-with-OpsItems-editing-details.html"
     *         >Editing OpsItem Details</a> in the <i>AWS Systems Manager User Guide</i>.
     * @see OpsItemStatus
     */
    public OpsItemStatus status() {
        return OpsItemStatus.fromValue(status);
    }

    /**
     * <p>
     * The OpsItem status. Status can be <code>Open</code>, <code>In Progress</code>, or <code>Resolved</code>. For more
     * information, see <a href=
     * "http://docs.aws.amazon.com/systems-manager/latest/userguide/OpsCenter-working-with-OpsItems-editing-details.html"
     * >Editing OpsItem Details</a> in the <i>AWS Systems Manager User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link OpsItemStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The OpsItem status. Status can be <code>Open</code>, <code>In Progress</code>, or <code>Resolved</code>.
     *         For more information, see <a href=
     *         "http://docs.aws.amazon.com/systems-manager/latest/userguide/OpsCenter-working-with-OpsItems-editing-details.html"
     *         >Editing OpsItem Details</a> in the <i>AWS Systems Manager User Guide</i>.
     * @see OpsItemStatus
     */
    public String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The ID of the OpsItem.
     * </p>
     * 
     * @return The ID of the OpsItem.
     */
    public String opsItemId() {
        return opsItemId;
    }

    /**
     * <p>
     * The version of this OpsItem. Each time the OpsItem is edited the version number increments by one.
     * </p>
     * 
     * @return The version of this OpsItem. Each time the OpsItem is edited the version number increments by one.
     */
    public String version() {
        return version;
    }

    /**
     * <p>
     * A short heading that describes the nature of the OpsItem and the impacted resource.
     * </p>
     * 
     * @return A short heading that describes the nature of the OpsItem and the impacted resource.
     */
    public String title() {
        return title;
    }

    /**
     * <p>
     * The origin of the OpsItem, such as Amazon EC2 or AWS Systems Manager. The impacted resource is a subset of
     * source.
     * </p>
     * 
     * @return The origin of the OpsItem, such as Amazon EC2 or AWS Systems Manager. The impacted resource is a subset
     *         of source.
     */
    public String source() {
        return source;
    }

    /**
     * <p>
     * Operational data is custom data that provides useful reference details about the OpsItem. For example, you can
     * specify log files, error strings, license keys, troubleshooting tips, or other relevant data. You enter
     * operational data as key-value pairs. The key has a maximum length of 128 characters. The value has a maximum size
     * of 20 KB.
     * </p>
     * <important>
     * <p>
     * Operational data keys <i>can't</i> begin with the following: amazon, aws, amzn, ssm, /amazon, /aws, /amzn, /ssm.
     * </p>
     * </important>
     * <p>
     * You can choose to make the data searchable by other users in the account or you can restrict search access.
     * Searchable data means that all users with access to the OpsItem Overview page (as provided by the
     * <a>DescribeOpsItems</a> API action) can view and search on the specified data. Operational data that is not
     * searchable is only viewable by users who have access to the OpsItem (as provided by the <a>GetOpsItem</a> API
     * action).
     * </p>
     * <p>
     * Use the <code>/aws/resources</code> key in OperationalData to specify a related resource in the request. Use the
     * <code>/aws/automations</code> key in OperationalData to associate an Automation runbook with the OpsItem. To view
     * AWS CLI example commands that use these keys, see <a href=
     * "http://docs.aws.amazon.com/systems-manager/latest/userguide/OpsCenter-creating-OpsItems.html#OpsCenter-manually-create-OpsItems"
     * >Creating OpsItems Manually</a> in the <i>AWS Systems Manager User Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Operational data is custom data that provides useful reference details about the OpsItem. For example,
     *         you can specify log files, error strings, license keys, troubleshooting tips, or other relevant data. You
     *         enter operational data as key-value pairs. The key has a maximum length of 128 characters. The value has
     *         a maximum size of 20 KB.</p> <important>
     *         <p>
     *         Operational data keys <i>can't</i> begin with the following: amazon, aws, amzn, ssm, /amazon, /aws,
     *         /amzn, /ssm.
     *         </p>
     *         </important>
     *         <p>
     *         You can choose to make the data searchable by other users in the account or you can restrict search
     *         access. Searchable data means that all users with access to the OpsItem Overview page (as provided by the
     *         <a>DescribeOpsItems</a> API action) can view and search on the specified data. Operational data that is
     *         not searchable is only viewable by users who have access to the OpsItem (as provided by the
     *         <a>GetOpsItem</a> API action).
     *         </p>
     *         <p>
     *         Use the <code>/aws/resources</code> key in OperationalData to specify a related resource in the request.
     *         Use the <code>/aws/automations</code> key in OperationalData to associate an Automation runbook with the
     *         OpsItem. To view AWS CLI example commands that use these keys, see <a href=
     *         "http://docs.aws.amazon.com/systems-manager/latest/userguide/OpsCenter-creating-OpsItems.html#OpsCenter-manually-create-OpsItems"
     *         >Creating OpsItems Manually</a> in the <i>AWS Systems Manager User Guide</i>.
     */
    public Map<String, OpsItemDataValue> operationalData() {
        return operationalData;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(createdBy());
        hashCode = 31 * hashCode + Objects.hashCode(createdTime());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(lastModifiedBy());
        hashCode = 31 * hashCode + Objects.hashCode(lastModifiedTime());
        hashCode = 31 * hashCode + Objects.hashCode(notifications());
        hashCode = 31 * hashCode + Objects.hashCode(priority());
        hashCode = 31 * hashCode + Objects.hashCode(relatedOpsItems());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(opsItemId());
        hashCode = 31 * hashCode + Objects.hashCode(version());
        hashCode = 31 * hashCode + Objects.hashCode(title());
        hashCode = 31 * hashCode + Objects.hashCode(source());
        hashCode = 31 * hashCode + Objects.hashCode(operationalData());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof OpsItem)) {
            return false;
        }
        OpsItem other = (OpsItem) obj;
        return Objects.equals(createdBy(), other.createdBy()) && Objects.equals(createdTime(), other.createdTime())
                && Objects.equals(description(), other.description()) && Objects.equals(lastModifiedBy(), other.lastModifiedBy())
                && Objects.equals(lastModifiedTime(), other.lastModifiedTime())
                && Objects.equals(notifications(), other.notifications()) && Objects.equals(priority(), other.priority())
                && Objects.equals(relatedOpsItems(), other.relatedOpsItems())
                && Objects.equals(statusAsString(), other.statusAsString()) && Objects.equals(opsItemId(), other.opsItemId())
                && Objects.equals(version(), other.version()) && Objects.equals(title(), other.title())
                && Objects.equals(source(), other.source()) && Objects.equals(operationalData(), other.operationalData());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("OpsItem").add("CreatedBy", createdBy()).add("CreatedTime", createdTime())
                .add("Description", description()).add("LastModifiedBy", lastModifiedBy())
                .add("LastModifiedTime", lastModifiedTime()).add("Notifications", notifications()).add("Priority", priority())
                .add("RelatedOpsItems", relatedOpsItems()).add("Status", statusAsString()).add("OpsItemId", opsItemId())
                .add("Version", version()).add("Title", title()).add("Source", source())
                .add("OperationalData", operationalData()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "CreatedBy":
            return Optional.ofNullable(clazz.cast(createdBy()));
        case "CreatedTime":
            return Optional.ofNullable(clazz.cast(createdTime()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "LastModifiedBy":
            return Optional.ofNullable(clazz.cast(lastModifiedBy()));
        case "LastModifiedTime":
            return Optional.ofNullable(clazz.cast(lastModifiedTime()));
        case "Notifications":
            return Optional.ofNullable(clazz.cast(notifications()));
        case "Priority":
            return Optional.ofNullable(clazz.cast(priority()));
        case "RelatedOpsItems":
            return Optional.ofNullable(clazz.cast(relatedOpsItems()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "OpsItemId":
            return Optional.ofNullable(clazz.cast(opsItemId()));
        case "Version":
            return Optional.ofNullable(clazz.cast(version()));
        case "Title":
            return Optional.ofNullable(clazz.cast(title()));
        case "Source":
            return Optional.ofNullable(clazz.cast(source()));
        case "OperationalData":
            return Optional.ofNullable(clazz.cast(operationalData()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<OpsItem, T> g) {
        return obj -> g.apply((OpsItem) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, OpsItem> {
        /**
         * <p>
         * The ARN of the AWS account that created the OpsItem.
         * </p>
         * 
         * @param createdBy
         *        The ARN of the AWS account that created the OpsItem.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdBy(String createdBy);

        /**
         * <p>
         * The date and time the OpsItem was created.
         * </p>
         * 
         * @param createdTime
         *        The date and time the OpsItem was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdTime(Instant createdTime);

        /**
         * <p>
         * The OpsItem description.
         * </p>
         * 
         * @param description
         *        The OpsItem description.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The ARN of the AWS account that last updated the OpsItem.
         * </p>
         * 
         * @param lastModifiedBy
         *        The ARN of the AWS account that last updated the OpsItem.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastModifiedBy(String lastModifiedBy);

        /**
         * <p>
         * The date and time the OpsItem was last updated.
         * </p>
         * 
         * @param lastModifiedTime
         *        The date and time the OpsItem was last updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastModifiedTime(Instant lastModifiedTime);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of an SNS topic where notifications are sent when this OpsItem is edited or
         * changed.
         * </p>
         * 
         * @param notifications
         *        The Amazon Resource Name (ARN) of an SNS topic where notifications are sent when this OpsItem is
         *        edited or changed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notifications(Collection<OpsItemNotification> notifications);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of an SNS topic where notifications are sent when this OpsItem is edited or
         * changed.
         * </p>
         * 
         * @param notifications
         *        The Amazon Resource Name (ARN) of an SNS topic where notifications are sent when this OpsItem is
         *        edited or changed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notifications(OpsItemNotification... notifications);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of an SNS topic where notifications are sent when this OpsItem is edited or
         * changed.
         * </p>
         * This is a convenience that creates an instance of the {@link List<OpsItemNotification>.Builder} avoiding the
         * need to create one manually via {@link List<OpsItemNotification>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<OpsItemNotification>.Builder#build()} is called immediately
         * and its result is passed to {@link #notifications(List<OpsItemNotification>)}.
         * 
         * @param notifications
         *        a consumer that will call methods on {@link List<OpsItemNotification>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #notifications(List<OpsItemNotification>)
         */
        Builder notifications(Consumer<OpsItemNotification.Builder>... notifications);

        /**
         * <p>
         * The importance of this OpsItem in relation to other OpsItems in the system.
         * </p>
         * 
         * @param priority
         *        The importance of this OpsItem in relation to other OpsItems in the system.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder priority(Integer priority);

        /**
         * <p>
         * One or more OpsItems that share something in common with the current OpsItem. For example, related OpsItems
         * can include OpsItems with similar error messages, impacted resources, or statuses for the impacted resource.
         * </p>
         * 
         * @param relatedOpsItems
         *        One or more OpsItems that share something in common with the current OpsItem. For example, related
         *        OpsItems can include OpsItems with similar error messages, impacted resources, or statuses for the
         *        impacted resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder relatedOpsItems(Collection<RelatedOpsItem> relatedOpsItems);

        /**
         * <p>
         * One or more OpsItems that share something in common with the current OpsItem. For example, related OpsItems
         * can include OpsItems with similar error messages, impacted resources, or statuses for the impacted resource.
         * </p>
         * 
         * @param relatedOpsItems
         *        One or more OpsItems that share something in common with the current OpsItem. For example, related
         *        OpsItems can include OpsItems with similar error messages, impacted resources, or statuses for the
         *        impacted resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder relatedOpsItems(RelatedOpsItem... relatedOpsItems);

        /**
         * <p>
         * One or more OpsItems that share something in common with the current OpsItem. For example, related OpsItems
         * can include OpsItems with similar error messages, impacted resources, or statuses for the impacted resource.
         * </p>
         * This is a convenience that creates an instance of the {@link List<RelatedOpsItem>.Builder} avoiding the need
         * to create one manually via {@link List<RelatedOpsItem>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<RelatedOpsItem>.Builder#build()} is called immediately and
         * its result is passed to {@link #relatedOpsItems(List<RelatedOpsItem>)}.
         * 
         * @param relatedOpsItems
         *        a consumer that will call methods on {@link List<RelatedOpsItem>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #relatedOpsItems(List<RelatedOpsItem>)
         */
        Builder relatedOpsItems(Consumer<RelatedOpsItem.Builder>... relatedOpsItems);

        /**
         * <p>
         * The OpsItem status. Status can be <code>Open</code>, <code>In Progress</code>, or <code>Resolved</code>. For
         * more information, see <a href=
         * "http://docs.aws.amazon.com/systems-manager/latest/userguide/OpsCenter-working-with-OpsItems-editing-details.html"
         * >Editing OpsItem Details</a> in the <i>AWS Systems Manager User Guide</i>.
         * </p>
         * 
         * @param status
         *        The OpsItem status. Status can be <code>Open</code>, <code>In Progress</code>, or
         *        <code>Resolved</code>. For more information, see <a href=
         *        "http://docs.aws.amazon.com/systems-manager/latest/userguide/OpsCenter-working-with-OpsItems-editing-details.html"
         *        >Editing OpsItem Details</a> in the <i>AWS Systems Manager User Guide</i>.
         * @see OpsItemStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OpsItemStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The OpsItem status. Status can be <code>Open</code>, <code>In Progress</code>, or <code>Resolved</code>. For
         * more information, see <a href=
         * "http://docs.aws.amazon.com/systems-manager/latest/userguide/OpsCenter-working-with-OpsItems-editing-details.html"
         * >Editing OpsItem Details</a> in the <i>AWS Systems Manager User Guide</i>.
         * </p>
         * 
         * @param status
         *        The OpsItem status. Status can be <code>Open</code>, <code>In Progress</code>, or
         *        <code>Resolved</code>. For more information, see <a href=
         *        "http://docs.aws.amazon.com/systems-manager/latest/userguide/OpsCenter-working-with-OpsItems-editing-details.html"
         *        >Editing OpsItem Details</a> in the <i>AWS Systems Manager User Guide</i>.
         * @see OpsItemStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OpsItemStatus
         */
        Builder status(OpsItemStatus status);

        /**
         * <p>
         * The ID of the OpsItem.
         * </p>
         * 
         * @param opsItemId
         *        The ID of the OpsItem.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder opsItemId(String opsItemId);

        /**
         * <p>
         * The version of this OpsItem. Each time the OpsItem is edited the version number increments by one.
         * </p>
         * 
         * @param version
         *        The version of this OpsItem. Each time the OpsItem is edited the version number increments by one.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder version(String version);

        /**
         * <p>
         * A short heading that describes the nature of the OpsItem and the impacted resource.
         * </p>
         * 
         * @param title
         *        A short heading that describes the nature of the OpsItem and the impacted resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder title(String title);

        /**
         * <p>
         * The origin of the OpsItem, such as Amazon EC2 or AWS Systems Manager. The impacted resource is a subset of
         * source.
         * </p>
         * 
         * @param source
         *        The origin of the OpsItem, such as Amazon EC2 or AWS Systems Manager. The impacted resource is a
         *        subset of source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder source(String source);

        /**
         * <p>
         * Operational data is custom data that provides useful reference details about the OpsItem. For example, you
         * can specify log files, error strings, license keys, troubleshooting tips, or other relevant data. You enter
         * operational data as key-value pairs. The key has a maximum length of 128 characters. The value has a maximum
         * size of 20 KB.
         * </p>
         * <important>
         * <p>
         * Operational data keys <i>can't</i> begin with the following: amazon, aws, amzn, ssm, /amazon, /aws, /amzn,
         * /ssm.
         * </p>
         * </important>
         * <p>
         * You can choose to make the data searchable by other users in the account or you can restrict search access.
         * Searchable data means that all users with access to the OpsItem Overview page (as provided by the
         * <a>DescribeOpsItems</a> API action) can view and search on the specified data. Operational data that is not
         * searchable is only viewable by users who have access to the OpsItem (as provided by the <a>GetOpsItem</a> API
         * action).
         * </p>
         * <p>
         * Use the <code>/aws/resources</code> key in OperationalData to specify a related resource in the request. Use
         * the <code>/aws/automations</code> key in OperationalData to associate an Automation runbook with the OpsItem.
         * To view AWS CLI example commands that use these keys, see <a href=
         * "http://docs.aws.amazon.com/systems-manager/latest/userguide/OpsCenter-creating-OpsItems.html#OpsCenter-manually-create-OpsItems"
         * >Creating OpsItems Manually</a> in the <i>AWS Systems Manager User Guide</i>.
         * </p>
         * 
         * @param operationalData
         *        Operational data is custom data that provides useful reference details about the OpsItem. For example,
         *        you can specify log files, error strings, license keys, troubleshooting tips, or other relevant data.
         *        You enter operational data as key-value pairs. The key has a maximum length of 128 characters. The
         *        value has a maximum size of 20 KB.</p> <important>
         *        <p>
         *        Operational data keys <i>can't</i> begin with the following: amazon, aws, amzn, ssm, /amazon, /aws,
         *        /amzn, /ssm.
         *        </p>
         *        </important>
         *        <p>
         *        You can choose to make the data searchable by other users in the account or you can restrict search
         *        access. Searchable data means that all users with access to the OpsItem Overview page (as provided by
         *        the <a>DescribeOpsItems</a> API action) can view and search on the specified data. Operational data
         *        that is not searchable is only viewable by users who have access to the OpsItem (as provided by the
         *        <a>GetOpsItem</a> API action).
         *        </p>
         *        <p>
         *        Use the <code>/aws/resources</code> key in OperationalData to specify a related resource in the
         *        request. Use the <code>/aws/automations</code> key in OperationalData to associate an Automation
         *        runbook with the OpsItem. To view AWS CLI example commands that use these keys, see <a href=
         *        "http://docs.aws.amazon.com/systems-manager/latest/userguide/OpsCenter-creating-OpsItems.html#OpsCenter-manually-create-OpsItems"
         *        >Creating OpsItems Manually</a> in the <i>AWS Systems Manager User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder operationalData(Map<String, OpsItemDataValue> operationalData);
    }

    static final class BuilderImpl implements Builder {
        private String createdBy;

        private Instant createdTime;

        private String description;

        private String lastModifiedBy;

        private Instant lastModifiedTime;

        private List<OpsItemNotification> notifications = DefaultSdkAutoConstructList.getInstance();

        private Integer priority;

        private List<RelatedOpsItem> relatedOpsItems = DefaultSdkAutoConstructList.getInstance();

        private String status;

        private String opsItemId;

        private String version;

        private String title;

        private String source;

        private Map<String, OpsItemDataValue> operationalData = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(OpsItem model) {
            createdBy(model.createdBy);
            createdTime(model.createdTime);
            description(model.description);
            lastModifiedBy(model.lastModifiedBy);
            lastModifiedTime(model.lastModifiedTime);
            notifications(model.notifications);
            priority(model.priority);
            relatedOpsItems(model.relatedOpsItems);
            status(model.status);
            opsItemId(model.opsItemId);
            version(model.version);
            title(model.title);
            source(model.source);
            operationalData(model.operationalData);
        }

        public final String getCreatedBy() {
            return createdBy;
        }

        @Override
        public final Builder createdBy(String createdBy) {
            this.createdBy = createdBy;
            return this;
        }

        public final void setCreatedBy(String createdBy) {
            this.createdBy = createdBy;
        }

        public final Instant getCreatedTime() {
            return createdTime;
        }

        @Override
        public final Builder createdTime(Instant createdTime) {
            this.createdTime = createdTime;
            return this;
        }

        public final void setCreatedTime(Instant createdTime) {
            this.createdTime = createdTime;
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final String getLastModifiedBy() {
            return lastModifiedBy;
        }

        @Override
        public final Builder lastModifiedBy(String lastModifiedBy) {
            this.lastModifiedBy = lastModifiedBy;
            return this;
        }

        public final void setLastModifiedBy(String lastModifiedBy) {
            this.lastModifiedBy = lastModifiedBy;
        }

        public final Instant getLastModifiedTime() {
            return lastModifiedTime;
        }

        @Override
        public final Builder lastModifiedTime(Instant lastModifiedTime) {
            this.lastModifiedTime = lastModifiedTime;
            return this;
        }

        public final void setLastModifiedTime(Instant lastModifiedTime) {
            this.lastModifiedTime = lastModifiedTime;
        }

        public final Collection<OpsItemNotification.Builder> getNotifications() {
            return notifications != null ? notifications.stream().map(OpsItemNotification::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder notifications(Collection<OpsItemNotification> notifications) {
            this.notifications = OpsItemNotificationsCopier.copy(notifications);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder notifications(OpsItemNotification... notifications) {
            notifications(Arrays.asList(notifications));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder notifications(Consumer<OpsItemNotification.Builder>... notifications) {
            notifications(Stream.of(notifications).map(c -> OpsItemNotification.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setNotifications(Collection<OpsItemNotification.BuilderImpl> notifications) {
            this.notifications = OpsItemNotificationsCopier.copyFromBuilder(notifications);
        }

        public final Integer getPriority() {
            return priority;
        }

        @Override
        public final Builder priority(Integer priority) {
            this.priority = priority;
            return this;
        }

        public final void setPriority(Integer priority) {
            this.priority = priority;
        }

        public final Collection<RelatedOpsItem.Builder> getRelatedOpsItems() {
            return relatedOpsItems != null ? relatedOpsItems.stream().map(RelatedOpsItem::toBuilder).collect(Collectors.toList())
                    : null;
        }

        @Override
        public final Builder relatedOpsItems(Collection<RelatedOpsItem> relatedOpsItems) {
            this.relatedOpsItems = RelatedOpsItemsCopier.copy(relatedOpsItems);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder relatedOpsItems(RelatedOpsItem... relatedOpsItems) {
            relatedOpsItems(Arrays.asList(relatedOpsItems));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder relatedOpsItems(Consumer<RelatedOpsItem.Builder>... relatedOpsItems) {
            relatedOpsItems(Stream.of(relatedOpsItems).map(c -> RelatedOpsItem.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setRelatedOpsItems(Collection<RelatedOpsItem.BuilderImpl> relatedOpsItems) {
            this.relatedOpsItems = RelatedOpsItemsCopier.copyFromBuilder(relatedOpsItems);
        }

        public final String getStatusAsString() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(OpsItemStatus status) {
            this.status(status.toString());
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        public final String getOpsItemId() {
            return opsItemId;
        }

        @Override
        public final Builder opsItemId(String opsItemId) {
            this.opsItemId = opsItemId;
            return this;
        }

        public final void setOpsItemId(String opsItemId) {
            this.opsItemId = opsItemId;
        }

        public final String getVersion() {
            return version;
        }

        @Override
        public final Builder version(String version) {
            this.version = version;
            return this;
        }

        public final void setVersion(String version) {
            this.version = version;
        }

        public final String getTitle() {
            return title;
        }

        @Override
        public final Builder title(String title) {
            this.title = title;
            return this;
        }

        public final void setTitle(String title) {
            this.title = title;
        }

        public final String getSource() {
            return source;
        }

        @Override
        public final Builder source(String source) {
            this.source = source;
            return this;
        }

        public final void setSource(String source) {
            this.source = source;
        }

        public final Map<String, OpsItemDataValue.Builder> getOperationalData() {
            return operationalData != null ? CollectionUtils.mapValues(operationalData, OpsItemDataValue::toBuilder) : null;
        }

        @Override
        public final Builder operationalData(Map<String, OpsItemDataValue> operationalData) {
            this.operationalData = OpsItemOperationalDataCopier.copy(operationalData);
            return this;
        }

        public final void setOperationalData(Map<String, OpsItemDataValue.BuilderImpl> operationalData) {
            this.operationalData = OpsItemOperationalDataCopier.copyFromBuilder(operationalData);
        }

        @Override
        public OpsItem build() {
            return new OpsItem(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
