/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.utils.CollectionUtils;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateOpsItemRequest extends SsmRequest implements
        ToCopyableBuilder<CreateOpsItemRequest.Builder, CreateOpsItemRequest> {
    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateOpsItemRequest::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<Map<String, OpsItemDataValue>> OPERATIONAL_DATA_FIELD = SdkField
            .<Map<String, OpsItemDataValue>> builder(MarshallingType.MAP)
            .getter(getter(CreateOpsItemRequest::operationalData))
            .setter(setter(Builder::operationalData))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OperationalData").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<OpsItemDataValue> builder(MarshallingType.SDK_POJO)
                                            .constructor(OpsItemDataValue::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<List<OpsItemNotification>> NOTIFICATIONS_FIELD = SdkField
            .<List<OpsItemNotification>> builder(MarshallingType.LIST)
            .getter(getter(CreateOpsItemRequest::notifications))
            .setter(setter(Builder::notifications))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Notifications").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<OpsItemNotification> builder(MarshallingType.SDK_POJO)
                                            .constructor(OpsItemNotification::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Integer> PRIORITY_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(CreateOpsItemRequest::priority)).setter(setter(Builder::priority))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Priority").build()).build();

    private static final SdkField<List<RelatedOpsItem>> RELATED_OPS_ITEMS_FIELD = SdkField
            .<List<RelatedOpsItem>> builder(MarshallingType.LIST)
            .getter(getter(CreateOpsItemRequest::relatedOpsItems))
            .setter(setter(Builder::relatedOpsItems))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RelatedOpsItems").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<RelatedOpsItem> builder(MarshallingType.SDK_POJO)
                                            .constructor(RelatedOpsItem::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> SOURCE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateOpsItemRequest::source)).setter(setter(Builder::source))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Source").build()).build();

    private static final SdkField<String> TITLE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateOpsItemRequest::title)).setter(setter(Builder::title))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Title").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .getter(getter(CreateOpsItemRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DESCRIPTION_FIELD,
            OPERATIONAL_DATA_FIELD, NOTIFICATIONS_FIELD, PRIORITY_FIELD, RELATED_OPS_ITEMS_FIELD, SOURCE_FIELD, TITLE_FIELD,
            TAGS_FIELD));

    private final String description;

    private final Map<String, OpsItemDataValue> operationalData;

    private final List<OpsItemNotification> notifications;

    private final Integer priority;

    private final List<RelatedOpsItem> relatedOpsItems;

    private final String source;

    private final String title;

    private final List<Tag> tags;

    private CreateOpsItemRequest(BuilderImpl builder) {
        super(builder);
        this.description = builder.description;
        this.operationalData = builder.operationalData;
        this.notifications = builder.notifications;
        this.priority = builder.priority;
        this.relatedOpsItems = builder.relatedOpsItems;
        this.source = builder.source;
        this.title = builder.title;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * Information about the OpsItem.
     * </p>
     * 
     * @return Information about the OpsItem.
     */
    public String description() {
        return description;
    }

    /**
     * <p>
     * Operational data is custom data that provides useful reference details about the OpsItem. For example, you can
     * specify log files, error strings, license keys, troubleshooting tips, or other relevant data. You enter
     * operational data as key-value pairs. The key has a maximum length of 128 characters. The value has a maximum size
     * of 20 KB.
     * </p>
     * <important>
     * <p>
     * Operational data keys <i>can't</i> begin with the following: amazon, aws, amzn, ssm, /amazon, /aws, /amzn, /ssm.
     * </p>
     * </important>
     * <p>
     * You can choose to make the data searchable by other users in the account or you can restrict search access.
     * Searchable data means that all users with access to the OpsItem Overview page (as provided by the
     * <a>DescribeOpsItems</a> API action) can view and search on the specified data. Operational data that is not
     * searchable is only viewable by users who have access to the OpsItem (as provided by the <a>GetOpsItem</a> API
     * action).
     * </p>
     * <p>
     * Use the <code>/aws/resources</code> key in OperationalData to specify a related resource in the request. Use the
     * <code>/aws/automations</code> key in OperationalData to associate an Automation runbook with the OpsItem. To view
     * AWS CLI example commands that use these keys, see <a href=
     * "http://docs.aws.amazon.com/systems-manager/latest/userguide/OpsCenter-creating-OpsItems.html#OpsCenter-manually-create-OpsItems"
     * >Creating OpsItems Manually</a> in the <i>AWS Systems Manager User Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Operational data is custom data that provides useful reference details about the OpsItem. For example,
     *         you can specify log files, error strings, license keys, troubleshooting tips, or other relevant data. You
     *         enter operational data as key-value pairs. The key has a maximum length of 128 characters. The value has
     *         a maximum size of 20 KB.</p> <important>
     *         <p>
     *         Operational data keys <i>can't</i> begin with the following: amazon, aws, amzn, ssm, /amazon, /aws,
     *         /amzn, /ssm.
     *         </p>
     *         </important>
     *         <p>
     *         You can choose to make the data searchable by other users in the account or you can restrict search
     *         access. Searchable data means that all users with access to the OpsItem Overview page (as provided by the
     *         <a>DescribeOpsItems</a> API action) can view and search on the specified data. Operational data that is
     *         not searchable is only viewable by users who have access to the OpsItem (as provided by the
     *         <a>GetOpsItem</a> API action).
     *         </p>
     *         <p>
     *         Use the <code>/aws/resources</code> key in OperationalData to specify a related resource in the request.
     *         Use the <code>/aws/automations</code> key in OperationalData to associate an Automation runbook with the
     *         OpsItem. To view AWS CLI example commands that use these keys, see <a href=
     *         "http://docs.aws.amazon.com/systems-manager/latest/userguide/OpsCenter-creating-OpsItems.html#OpsCenter-manually-create-OpsItems"
     *         >Creating OpsItems Manually</a> in the <i>AWS Systems Manager User Guide</i>.
     */
    public Map<String, OpsItemDataValue> operationalData() {
        return operationalData;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of an SNS topic where notifications are sent when this OpsItem is edited or
     * changed.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of an SNS topic where notifications are sent when this OpsItem is edited
     *         or changed.
     */
    public List<OpsItemNotification> notifications() {
        return notifications;
    }

    /**
     * <p>
     * The importance of this OpsItem in relation to other OpsItems in the system.
     * </p>
     * 
     * @return The importance of this OpsItem in relation to other OpsItems in the system.
     */
    public Integer priority() {
        return priority;
    }

    /**
     * <p>
     * One or more OpsItems that share something in common with the current OpsItems. For example, related OpsItems can
     * include OpsItems with similar error messages, impacted resources, or statuses for the impacted resource.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return One or more OpsItems that share something in common with the current OpsItems. For example, related
     *         OpsItems can include OpsItems with similar error messages, impacted resources, or statuses for the
     *         impacted resource.
     */
    public List<RelatedOpsItem> relatedOpsItems() {
        return relatedOpsItems;
    }

    /**
     * <p>
     * The origin of the OpsItem, such as Amazon EC2 or AWS Systems Manager.
     * </p>
     * 
     * @return The origin of the OpsItem, such as Amazon EC2 or AWS Systems Manager.
     */
    public String source() {
        return source;
    }

    /**
     * <p>
     * A short heading that describes the nature of the OpsItem and the impacted resource.
     * </p>
     * 
     * @return A short heading that describes the nature of the OpsItem and the impacted resource.
     */
    public String title() {
        return title;
    }

    /**
     * <p>
     * Optional metadata that you assign to a resource. You can restrict access to OpsItems by using an inline IAM
     * policy that specifies tags. For more information, see <a href=
     * "http://docs.aws.amazon.com/systems-manager/latest/userguide/OpsCenter-getting-started.html#OpsCenter-getting-started-user-permissions"
     * >Getting Started with OpsCenter</a> in the <i>AWS Systems Manager User Guide</i>.
     * </p>
     * <p>
     * Tags use a key-value pair. For example:
     * </p>
     * <p>
     * <code>Key=Department,Value=Finance</code>
     * </p>
     * <note>
     * <p>
     * To add tags to an existing OpsItem, use the <a>AddTagsToResource</a> action.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Optional metadata that you assign to a resource. You can restrict access to OpsItems by using an inline
     *         IAM policy that specifies tags. For more information, see <a href=
     *         "http://docs.aws.amazon.com/systems-manager/latest/userguide/OpsCenter-getting-started.html#OpsCenter-getting-started-user-permissions"
     *         >Getting Started with OpsCenter</a> in the <i>AWS Systems Manager User Guide</i>.</p>
     *         <p>
     *         Tags use a key-value pair. For example:
     *         </p>
     *         <p>
     *         <code>Key=Department,Value=Finance</code>
     *         </p>
     *         <note>
     *         <p>
     *         To add tags to an existing OpsItem, use the <a>AddTagsToResource</a> action.
     *         </p>
     */
    public List<Tag> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(operationalData());
        hashCode = 31 * hashCode + Objects.hashCode(notifications());
        hashCode = 31 * hashCode + Objects.hashCode(priority());
        hashCode = 31 * hashCode + Objects.hashCode(relatedOpsItems());
        hashCode = 31 * hashCode + Objects.hashCode(source());
        hashCode = 31 * hashCode + Objects.hashCode(title());
        hashCode = 31 * hashCode + Objects.hashCode(tags());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateOpsItemRequest)) {
            return false;
        }
        CreateOpsItemRequest other = (CreateOpsItemRequest) obj;
        return Objects.equals(description(), other.description()) && Objects.equals(operationalData(), other.operationalData())
                && Objects.equals(notifications(), other.notifications()) && Objects.equals(priority(), other.priority())
                && Objects.equals(relatedOpsItems(), other.relatedOpsItems()) && Objects.equals(source(), other.source())
                && Objects.equals(title(), other.title()) && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("CreateOpsItemRequest").add("Description", description())
                .add("OperationalData", operationalData()).add("Notifications", notifications()).add("Priority", priority())
                .add("RelatedOpsItems", relatedOpsItems()).add("Source", source()).add("Title", title()).add("Tags", tags())
                .build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "OperationalData":
            return Optional.ofNullable(clazz.cast(operationalData()));
        case "Notifications":
            return Optional.ofNullable(clazz.cast(notifications()));
        case "Priority":
            return Optional.ofNullable(clazz.cast(priority()));
        case "RelatedOpsItems":
            return Optional.ofNullable(clazz.cast(relatedOpsItems()));
        case "Source":
            return Optional.ofNullable(clazz.cast(source()));
        case "Title":
            return Optional.ofNullable(clazz.cast(title()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateOpsItemRequest, T> g) {
        return obj -> g.apply((CreateOpsItemRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SsmRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreateOpsItemRequest> {
        /**
         * <p>
         * Information about the OpsItem.
         * </p>
         * 
         * @param description
         *        Information about the OpsItem.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * Operational data is custom data that provides useful reference details about the OpsItem. For example, you
         * can specify log files, error strings, license keys, troubleshooting tips, or other relevant data. You enter
         * operational data as key-value pairs. The key has a maximum length of 128 characters. The value has a maximum
         * size of 20 KB.
         * </p>
         * <important>
         * <p>
         * Operational data keys <i>can't</i> begin with the following: amazon, aws, amzn, ssm, /amazon, /aws, /amzn,
         * /ssm.
         * </p>
         * </important>
         * <p>
         * You can choose to make the data searchable by other users in the account or you can restrict search access.
         * Searchable data means that all users with access to the OpsItem Overview page (as provided by the
         * <a>DescribeOpsItems</a> API action) can view and search on the specified data. Operational data that is not
         * searchable is only viewable by users who have access to the OpsItem (as provided by the <a>GetOpsItem</a> API
         * action).
         * </p>
         * <p>
         * Use the <code>/aws/resources</code> key in OperationalData to specify a related resource in the request. Use
         * the <code>/aws/automations</code> key in OperationalData to associate an Automation runbook with the OpsItem.
         * To view AWS CLI example commands that use these keys, see <a href=
         * "http://docs.aws.amazon.com/systems-manager/latest/userguide/OpsCenter-creating-OpsItems.html#OpsCenter-manually-create-OpsItems"
         * >Creating OpsItems Manually</a> in the <i>AWS Systems Manager User Guide</i>.
         * </p>
         * 
         * @param operationalData
         *        Operational data is custom data that provides useful reference details about the OpsItem. For example,
         *        you can specify log files, error strings, license keys, troubleshooting tips, or other relevant data.
         *        You enter operational data as key-value pairs. The key has a maximum length of 128 characters. The
         *        value has a maximum size of 20 KB.</p> <important>
         *        <p>
         *        Operational data keys <i>can't</i> begin with the following: amazon, aws, amzn, ssm, /amazon, /aws,
         *        /amzn, /ssm.
         *        </p>
         *        </important>
         *        <p>
         *        You can choose to make the data searchable by other users in the account or you can restrict search
         *        access. Searchable data means that all users with access to the OpsItem Overview page (as provided by
         *        the <a>DescribeOpsItems</a> API action) can view and search on the specified data. Operational data
         *        that is not searchable is only viewable by users who have access to the OpsItem (as provided by the
         *        <a>GetOpsItem</a> API action).
         *        </p>
         *        <p>
         *        Use the <code>/aws/resources</code> key in OperationalData to specify a related resource in the
         *        request. Use the <code>/aws/automations</code> key in OperationalData to associate an Automation
         *        runbook with the OpsItem. To view AWS CLI example commands that use these keys, see <a href=
         *        "http://docs.aws.amazon.com/systems-manager/latest/userguide/OpsCenter-creating-OpsItems.html#OpsCenter-manually-create-OpsItems"
         *        >Creating OpsItems Manually</a> in the <i>AWS Systems Manager User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder operationalData(Map<String, OpsItemDataValue> operationalData);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of an SNS topic where notifications are sent when this OpsItem is edited or
         * changed.
         * </p>
         * 
         * @param notifications
         *        The Amazon Resource Name (ARN) of an SNS topic where notifications are sent when this OpsItem is
         *        edited or changed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notifications(Collection<OpsItemNotification> notifications);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of an SNS topic where notifications are sent when this OpsItem is edited or
         * changed.
         * </p>
         * 
         * @param notifications
         *        The Amazon Resource Name (ARN) of an SNS topic where notifications are sent when this OpsItem is
         *        edited or changed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notifications(OpsItemNotification... notifications);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of an SNS topic where notifications are sent when this OpsItem is edited or
         * changed.
         * </p>
         * This is a convenience that creates an instance of the {@link List<OpsItemNotification>.Builder} avoiding the
         * need to create one manually via {@link List<OpsItemNotification>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<OpsItemNotification>.Builder#build()} is called immediately
         * and its result is passed to {@link #notifications(List<OpsItemNotification>)}.
         * 
         * @param notifications
         *        a consumer that will call methods on {@link List<OpsItemNotification>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #notifications(List<OpsItemNotification>)
         */
        Builder notifications(Consumer<OpsItemNotification.Builder>... notifications);

        /**
         * <p>
         * The importance of this OpsItem in relation to other OpsItems in the system.
         * </p>
         * 
         * @param priority
         *        The importance of this OpsItem in relation to other OpsItems in the system.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder priority(Integer priority);

        /**
         * <p>
         * One or more OpsItems that share something in common with the current OpsItems. For example, related OpsItems
         * can include OpsItems with similar error messages, impacted resources, or statuses for the impacted resource.
         * </p>
         * 
         * @param relatedOpsItems
         *        One or more OpsItems that share something in common with the current OpsItems. For example, related
         *        OpsItems can include OpsItems with similar error messages, impacted resources, or statuses for the
         *        impacted resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder relatedOpsItems(Collection<RelatedOpsItem> relatedOpsItems);

        /**
         * <p>
         * One or more OpsItems that share something in common with the current OpsItems. For example, related OpsItems
         * can include OpsItems with similar error messages, impacted resources, or statuses for the impacted resource.
         * </p>
         * 
         * @param relatedOpsItems
         *        One or more OpsItems that share something in common with the current OpsItems. For example, related
         *        OpsItems can include OpsItems with similar error messages, impacted resources, or statuses for the
         *        impacted resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder relatedOpsItems(RelatedOpsItem... relatedOpsItems);

        /**
         * <p>
         * One or more OpsItems that share something in common with the current OpsItems. For example, related OpsItems
         * can include OpsItems with similar error messages, impacted resources, or statuses for the impacted resource.
         * </p>
         * This is a convenience that creates an instance of the {@link List<RelatedOpsItem>.Builder} avoiding the need
         * to create one manually via {@link List<RelatedOpsItem>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<RelatedOpsItem>.Builder#build()} is called immediately and
         * its result is passed to {@link #relatedOpsItems(List<RelatedOpsItem>)}.
         * 
         * @param relatedOpsItems
         *        a consumer that will call methods on {@link List<RelatedOpsItem>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #relatedOpsItems(List<RelatedOpsItem>)
         */
        Builder relatedOpsItems(Consumer<RelatedOpsItem.Builder>... relatedOpsItems);

        /**
         * <p>
         * The origin of the OpsItem, such as Amazon EC2 or AWS Systems Manager.
         * </p>
         * 
         * @param source
         *        The origin of the OpsItem, such as Amazon EC2 or AWS Systems Manager.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder source(String source);

        /**
         * <p>
         * A short heading that describes the nature of the OpsItem and the impacted resource.
         * </p>
         * 
         * @param title
         *        A short heading that describes the nature of the OpsItem and the impacted resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder title(String title);

        /**
         * <p>
         * Optional metadata that you assign to a resource. You can restrict access to OpsItems by using an inline IAM
         * policy that specifies tags. For more information, see <a href=
         * "http://docs.aws.amazon.com/systems-manager/latest/userguide/OpsCenter-getting-started.html#OpsCenter-getting-started-user-permissions"
         * >Getting Started with OpsCenter</a> in the <i>AWS Systems Manager User Guide</i>.
         * </p>
         * <p>
         * Tags use a key-value pair. For example:
         * </p>
         * <p>
         * <code>Key=Department,Value=Finance</code>
         * </p>
         * <note>
         * <p>
         * To add tags to an existing OpsItem, use the <a>AddTagsToResource</a> action.
         * </p>
         * </note>
         * 
         * @param tags
         *        Optional metadata that you assign to a resource. You can restrict access to OpsItems by using an
         *        inline IAM policy that specifies tags. For more information, see <a href=
         *        "http://docs.aws.amazon.com/systems-manager/latest/userguide/OpsCenter-getting-started.html#OpsCenter-getting-started-user-permissions"
         *        >Getting Started with OpsCenter</a> in the <i>AWS Systems Manager User Guide</i>.</p>
         *        <p>
         *        Tags use a key-value pair. For example:
         *        </p>
         *        <p>
         *        <code>Key=Department,Value=Finance</code>
         *        </p>
         *        <note>
         *        <p>
         *        To add tags to an existing OpsItem, use the <a>AddTagsToResource</a> action.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * Optional metadata that you assign to a resource. You can restrict access to OpsItems by using an inline IAM
         * policy that specifies tags. For more information, see <a href=
         * "http://docs.aws.amazon.com/systems-manager/latest/userguide/OpsCenter-getting-started.html#OpsCenter-getting-started-user-permissions"
         * >Getting Started with OpsCenter</a> in the <i>AWS Systems Manager User Guide</i>.
         * </p>
         * <p>
         * Tags use a key-value pair. For example:
         * </p>
         * <p>
         * <code>Key=Department,Value=Finance</code>
         * </p>
         * <note>
         * <p>
         * To add tags to an existing OpsItem, use the <a>AddTagsToResource</a> action.
         * </p>
         * </note>
         * 
         * @param tags
         *        Optional metadata that you assign to a resource. You can restrict access to OpsItems by using an
         *        inline IAM policy that specifies tags. For more information, see <a href=
         *        "http://docs.aws.amazon.com/systems-manager/latest/userguide/OpsCenter-getting-started.html#OpsCenter-getting-started-user-permissions"
         *        >Getting Started with OpsCenter</a> in the <i>AWS Systems Manager User Guide</i>.</p>
         *        <p>
         *        Tags use a key-value pair. For example:
         *        </p>
         *        <p>
         *        <code>Key=Department,Value=Finance</code>
         *        </p>
         *        <note>
         *        <p>
         *        To add tags to an existing OpsItem, use the <a>AddTagsToResource</a> action.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * Optional metadata that you assign to a resource. You can restrict access to OpsItems by using an inline IAM
         * policy that specifies tags. For more information, see <a href=
         * "http://docs.aws.amazon.com/systems-manager/latest/userguide/OpsCenter-getting-started.html#OpsCenter-getting-started-user-permissions"
         * >Getting Started with OpsCenter</a> in the <i>AWS Systems Manager User Guide</i>.
         * </p>
         * <p>
         * Tags use a key-value pair. For example:
         * </p>
         * <p>
         * <code>Key=Department,Value=Finance</code>
         * </p>
         * <note>
         * <p>
         * To add tags to an existing OpsItem, use the <a>AddTagsToResource</a> action.
         * </p>
         * </note> This is a convenience that creates an instance of the {@link List<Tag>.Builder} avoiding the need to
         * create one manually via {@link List<Tag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Tag>.Builder#build()} is called immediately and its result
         * is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<Tag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends SsmRequest.BuilderImpl implements Builder {
        private String description;

        private Map<String, OpsItemDataValue> operationalData = DefaultSdkAutoConstructMap.getInstance();

        private List<OpsItemNotification> notifications = DefaultSdkAutoConstructList.getInstance();

        private Integer priority;

        private List<RelatedOpsItem> relatedOpsItems = DefaultSdkAutoConstructList.getInstance();

        private String source;

        private String title;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CreateOpsItemRequest model) {
            super(model);
            description(model.description);
            operationalData(model.operationalData);
            notifications(model.notifications);
            priority(model.priority);
            relatedOpsItems(model.relatedOpsItems);
            source(model.source);
            title(model.title);
            tags(model.tags);
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final Map<String, OpsItemDataValue.Builder> getOperationalData() {
            return operationalData != null ? CollectionUtils.mapValues(operationalData, OpsItemDataValue::toBuilder) : null;
        }

        @Override
        public final Builder operationalData(Map<String, OpsItemDataValue> operationalData) {
            this.operationalData = OpsItemOperationalDataCopier.copy(operationalData);
            return this;
        }

        public final void setOperationalData(Map<String, OpsItemDataValue.BuilderImpl> operationalData) {
            this.operationalData = OpsItemOperationalDataCopier.copyFromBuilder(operationalData);
        }

        public final Collection<OpsItemNotification.Builder> getNotifications() {
            return notifications != null ? notifications.stream().map(OpsItemNotification::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder notifications(Collection<OpsItemNotification> notifications) {
            this.notifications = OpsItemNotificationsCopier.copy(notifications);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder notifications(OpsItemNotification... notifications) {
            notifications(Arrays.asList(notifications));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder notifications(Consumer<OpsItemNotification.Builder>... notifications) {
            notifications(Stream.of(notifications).map(c -> OpsItemNotification.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setNotifications(Collection<OpsItemNotification.BuilderImpl> notifications) {
            this.notifications = OpsItemNotificationsCopier.copyFromBuilder(notifications);
        }

        public final Integer getPriority() {
            return priority;
        }

        @Override
        public final Builder priority(Integer priority) {
            this.priority = priority;
            return this;
        }

        public final void setPriority(Integer priority) {
            this.priority = priority;
        }

        public final Collection<RelatedOpsItem.Builder> getRelatedOpsItems() {
            return relatedOpsItems != null ? relatedOpsItems.stream().map(RelatedOpsItem::toBuilder).collect(Collectors.toList())
                    : null;
        }

        @Override
        public final Builder relatedOpsItems(Collection<RelatedOpsItem> relatedOpsItems) {
            this.relatedOpsItems = RelatedOpsItemsCopier.copy(relatedOpsItems);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder relatedOpsItems(RelatedOpsItem... relatedOpsItems) {
            relatedOpsItems(Arrays.asList(relatedOpsItems));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder relatedOpsItems(Consumer<RelatedOpsItem.Builder>... relatedOpsItems) {
            relatedOpsItems(Stream.of(relatedOpsItems).map(c -> RelatedOpsItem.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setRelatedOpsItems(Collection<RelatedOpsItem.BuilderImpl> relatedOpsItems) {
            this.relatedOpsItems = RelatedOpsItemsCopier.copyFromBuilder(relatedOpsItems);
        }

        public final String getSource() {
            return source;
        }

        @Override
        public final Builder source(String source) {
            this.source = source;
            return this;
        }

        public final void setSource(String source) {
            this.source = source;
        }

        public final String getTitle() {
            return title;
        }

        @Override
        public final Builder title(String title) {
            this.title = title;
            return this;
        }

        public final void setTitle(String title) {
            this.title = title;
        }

        public final Collection<Tag.Builder> getTags() {
            return tags != null ? tags.stream().map(Tag::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateOpsItemRequest build() {
            return new CreateOpsItemRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
