/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An invocation is copy of a command sent to a specific instance. A command can apply to one or more instances. A
 * command invocation applies to one instance. For example, if a user runs SendCommand against three instances, then a
 * command invocation is created for each requested instance ID. A command invocation returns status and detail
 * information about a command you ran.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CommandInvocation implements SdkPojo, Serializable,
        ToCopyableBuilder<CommandInvocation.Builder, CommandInvocation> {
    private static final SdkField<String> COMMAND_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CommandInvocation::commandId)).setter(setter(Builder::commandId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CommandId").build()).build();

    private static final SdkField<String> INSTANCE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CommandInvocation::instanceId)).setter(setter(Builder::instanceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceId").build()).build();

    private static final SdkField<String> INSTANCE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CommandInvocation::instanceName)).setter(setter(Builder::instanceName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceName").build()).build();

    private static final SdkField<String> COMMENT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CommandInvocation::comment)).setter(setter(Builder::comment))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Comment").build()).build();

    private static final SdkField<String> DOCUMENT_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CommandInvocation::documentName)).setter(setter(Builder::documentName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DocumentName").build()).build();

    private static final SdkField<String> DOCUMENT_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CommandInvocation::documentVersion)).setter(setter(Builder::documentVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DocumentVersion").build()).build();

    private static final SdkField<Instant> REQUESTED_DATE_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(CommandInvocation::requestedDateTime)).setter(setter(Builder::requestedDateTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RequestedDateTime").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CommandInvocation::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<String> STATUS_DETAILS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CommandInvocation::statusDetails)).setter(setter(Builder::statusDetails))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StatusDetails").build()).build();

    private static final SdkField<String> TRACE_OUTPUT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CommandInvocation::traceOutput)).setter(setter(Builder::traceOutput))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TraceOutput").build()).build();

    private static final SdkField<String> STANDARD_OUTPUT_URL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CommandInvocation::standardOutputUrl)).setter(setter(Builder::standardOutputUrl))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StandardOutputUrl").build()).build();

    private static final SdkField<String> STANDARD_ERROR_URL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CommandInvocation::standardErrorUrl)).setter(setter(Builder::standardErrorUrl))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StandardErrorUrl").build()).build();

    private static final SdkField<List<CommandPlugin>> COMMAND_PLUGINS_FIELD = SdkField
            .<List<CommandPlugin>> builder(MarshallingType.LIST)
            .getter(getter(CommandInvocation::commandPlugins))
            .setter(setter(Builder::commandPlugins))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CommandPlugins").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<CommandPlugin> builder(MarshallingType.SDK_POJO)
                                            .constructor(CommandPlugin::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> SERVICE_ROLE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CommandInvocation::serviceRole)).setter(setter(Builder::serviceRole))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServiceRole").build()).build();

    private static final SdkField<NotificationConfig> NOTIFICATION_CONFIG_FIELD = SdkField
            .<NotificationConfig> builder(MarshallingType.SDK_POJO).getter(getter(CommandInvocation::notificationConfig))
            .setter(setter(Builder::notificationConfig)).constructor(NotificationConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NotificationConfig").build())
            .build();

    private static final SdkField<CloudWatchOutputConfig> CLOUD_WATCH_OUTPUT_CONFIG_FIELD = SdkField
            .<CloudWatchOutputConfig> builder(MarshallingType.SDK_POJO).getter(getter(CommandInvocation::cloudWatchOutputConfig))
            .setter(setter(Builder::cloudWatchOutputConfig)).constructor(CloudWatchOutputConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CloudWatchOutputConfig").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(COMMAND_ID_FIELD,
            INSTANCE_ID_FIELD, INSTANCE_NAME_FIELD, COMMENT_FIELD, DOCUMENT_NAME_FIELD, DOCUMENT_VERSION_FIELD,
            REQUESTED_DATE_TIME_FIELD, STATUS_FIELD, STATUS_DETAILS_FIELD, TRACE_OUTPUT_FIELD, STANDARD_OUTPUT_URL_FIELD,
            STANDARD_ERROR_URL_FIELD, COMMAND_PLUGINS_FIELD, SERVICE_ROLE_FIELD, NOTIFICATION_CONFIG_FIELD,
            CLOUD_WATCH_OUTPUT_CONFIG_FIELD));

    private static final long serialVersionUID = 1L;

    private final String commandId;

    private final String instanceId;

    private final String instanceName;

    private final String comment;

    private final String documentName;

    private final String documentVersion;

    private final Instant requestedDateTime;

    private final String status;

    private final String statusDetails;

    private final String traceOutput;

    private final String standardOutputUrl;

    private final String standardErrorUrl;

    private final List<CommandPlugin> commandPlugins;

    private final String serviceRole;

    private final NotificationConfig notificationConfig;

    private final CloudWatchOutputConfig cloudWatchOutputConfig;

    private CommandInvocation(BuilderImpl builder) {
        this.commandId = builder.commandId;
        this.instanceId = builder.instanceId;
        this.instanceName = builder.instanceName;
        this.comment = builder.comment;
        this.documentName = builder.documentName;
        this.documentVersion = builder.documentVersion;
        this.requestedDateTime = builder.requestedDateTime;
        this.status = builder.status;
        this.statusDetails = builder.statusDetails;
        this.traceOutput = builder.traceOutput;
        this.standardOutputUrl = builder.standardOutputUrl;
        this.standardErrorUrl = builder.standardErrorUrl;
        this.commandPlugins = builder.commandPlugins;
        this.serviceRole = builder.serviceRole;
        this.notificationConfig = builder.notificationConfig;
        this.cloudWatchOutputConfig = builder.cloudWatchOutputConfig;
    }

    /**
     * <p>
     * The command against which this invocation was requested.
     * </p>
     * 
     * @return The command against which this invocation was requested.
     */
    public String commandId() {
        return commandId;
    }

    /**
     * <p>
     * The instance ID in which this invocation was requested.
     * </p>
     * 
     * @return The instance ID in which this invocation was requested.
     */
    public String instanceId() {
        return instanceId;
    }

    /**
     * <p>
     * The name of the invocation target. For Amazon EC2 instances this is the value for the aws:Name tag. For
     * on-premises instances, this is the name of the instance.
     * </p>
     * 
     * @return The name of the invocation target. For Amazon EC2 instances this is the value for the aws:Name tag. For
     *         on-premises instances, this is the name of the instance.
     */
    public String instanceName() {
        return instanceName;
    }

    /**
     * <p>
     * User-specified information about the command, such as a brief description of what the command should do.
     * </p>
     * 
     * @return User-specified information about the command, such as a brief description of what the command should do.
     */
    public String comment() {
        return comment;
    }

    /**
     * <p>
     * The document name that was requested for execution.
     * </p>
     * 
     * @return The document name that was requested for execution.
     */
    public String documentName() {
        return documentName;
    }

    /**
     * <p>
     * The SSM document version.
     * </p>
     * 
     * @return The SSM document version.
     */
    public String documentVersion() {
        return documentVersion;
    }

    /**
     * <p>
     * The time and date the request was sent to this instance.
     * </p>
     * 
     * @return The time and date the request was sent to this instance.
     */
    public Instant requestedDateTime() {
        return requestedDateTime;
    }

    /**
     * <p>
     * Whether or not the invocation succeeded, failed, or is pending.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link CommandInvocationStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return Whether or not the invocation succeeded, failed, or is pending.
     * @see CommandInvocationStatus
     */
    public CommandInvocationStatus status() {
        return CommandInvocationStatus.fromValue(status);
    }

    /**
     * <p>
     * Whether or not the invocation succeeded, failed, or is pending.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link CommandInvocationStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return Whether or not the invocation succeeded, failed, or is pending.
     * @see CommandInvocationStatus
     */
    public String statusAsString() {
        return status;
    }

    /**
     * <p>
     * A detailed status of the command execution for each invocation (each instance targeted by the command).
     * StatusDetails includes more information than Status because it includes states resulting from error and
     * concurrency control parameters. StatusDetails can show different results than Status. For more information about
     * these statuses, see <a
     * href="http://docs.aws.amazon.com/systems-manager/latest/userguide/monitor-commands.html">Understanding Command
     * Statuses</a> in the <i>AWS Systems Manager User Guide</i>. StatusDetails can be one of the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Pending: The command has not been sent to the instance.
     * </p>
     * </li>
     * <li>
     * <p>
     * In Progress: The command has been sent to the instance but has not reached a terminal state.
     * </p>
     * </li>
     * <li>
     * <p>
     * Success: The execution of the command or plugin was successfully completed. This is a terminal state.
     * </p>
     * </li>
     * <li>
     * <p>
     * Delivery Timed Out: The command was not delivered to the instance before the delivery timeout expired. Delivery
     * timeouts do not count against the parent command's MaxErrors limit, but they do contribute to whether the parent
     * command status is Success or Incomplete. This is a terminal state.
     * </p>
     * </li>
     * <li>
     * <p>
     * Execution Timed Out: Command execution started on the instance, but the execution was not complete before the
     * execution timeout expired. Execution timeouts count against the MaxErrors limit of the parent command. This is a
     * terminal state.
     * </p>
     * </li>
     * <li>
     * <p>
     * Failed: The command was not successful on the instance. For a plugin, this indicates that the result code was not
     * zero. For a command invocation, this indicates that the result code for one or more plugins was not zero.
     * Invocation failures count against the MaxErrors limit of the parent command. This is a terminal state.
     * </p>
     * </li>
     * <li>
     * <p>
     * Canceled: The command was terminated before it was completed. This is a terminal state.
     * </p>
     * </li>
     * <li>
     * <p>
     * Undeliverable: The command can't be delivered to the instance. The instance might not exist or might not be
     * responding. Undeliverable invocations don't count against the parent command's MaxErrors limit and don't
     * contribute to whether the parent command status is Success or Incomplete. This is a terminal state.
     * </p>
     * </li>
     * <li>
     * <p>
     * Terminated: The parent command exceeded its MaxErrors limit and subsequent command invocations were canceled by
     * the system. This is a terminal state.
     * </p>
     * </li>
     * </ul>
     * 
     * @return A detailed status of the command execution for each invocation (each instance targeted by the command).
     *         StatusDetails includes more information than Status because it includes states resulting from error and
     *         concurrency control parameters. StatusDetails can show different results than Status. For more
     *         information about these statuses, see <a
     *         href="http://docs.aws.amazon.com/systems-manager/latest/userguide/monitor-commands.html">Understanding
     *         Command Statuses</a> in the <i>AWS Systems Manager User Guide</i>. StatusDetails can be one of the
     *         following values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Pending: The command has not been sent to the instance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         In Progress: The command has been sent to the instance but has not reached a terminal state.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Success: The execution of the command or plugin was successfully completed. This is a terminal state.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Delivery Timed Out: The command was not delivered to the instance before the delivery timeout expired.
     *         Delivery timeouts do not count against the parent command's MaxErrors limit, but they do contribute to
     *         whether the parent command status is Success or Incomplete. This is a terminal state.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Execution Timed Out: Command execution started on the instance, but the execution was not complete before
     *         the execution timeout expired. Execution timeouts count against the MaxErrors limit of the parent
     *         command. This is a terminal state.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Failed: The command was not successful on the instance. For a plugin, this indicates that the result code
     *         was not zero. For a command invocation, this indicates that the result code for one or more plugins was
     *         not zero. Invocation failures count against the MaxErrors limit of the parent command. This is a terminal
     *         state.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Canceled: The command was terminated before it was completed. This is a terminal state.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Undeliverable: The command can't be delivered to the instance. The instance might not exist or might not
     *         be responding. Undeliverable invocations don't count against the parent command's MaxErrors limit and
     *         don't contribute to whether the parent command status is Success or Incomplete. This is a terminal state.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Terminated: The parent command exceeded its MaxErrors limit and subsequent command invocations were
     *         canceled by the system. This is a terminal state.
     *         </p>
     *         </li>
     */
    public String statusDetails() {
        return statusDetails;
    }

    /**
     * <p>
     * Gets the trace output sent by the agent.
     * </p>
     * 
     * @return Gets the trace output sent by the agent.
     */
    public String traceOutput() {
        return traceOutput;
    }

    /**
     * <p>
     * The URL to the plugin's StdOut file in Amazon S3, if the Amazon S3 bucket was defined for the parent command. For
     * an invocation, StandardOutputUrl is populated if there is just one plugin defined for the command, and the Amazon
     * S3 bucket was defined for the command.
     * </p>
     * 
     * @return The URL to the plugin's StdOut file in Amazon S3, if the Amazon S3 bucket was defined for the parent
     *         command. For an invocation, StandardOutputUrl is populated if there is just one plugin defined for the
     *         command, and the Amazon S3 bucket was defined for the command.
     */
    public String standardOutputUrl() {
        return standardOutputUrl;
    }

    /**
     * <p>
     * The URL to the plugin's StdErr file in Amazon S3, if the Amazon S3 bucket was defined for the parent command. For
     * an invocation, StandardErrorUrl is populated if there is just one plugin defined for the command, and the Amazon
     * S3 bucket was defined for the command.
     * </p>
     * 
     * @return The URL to the plugin's StdErr file in Amazon S3, if the Amazon S3 bucket was defined for the parent
     *         command. For an invocation, StandardErrorUrl is populated if there is just one plugin defined for the
     *         command, and the Amazon S3 bucket was defined for the command.
     */
    public String standardErrorUrl() {
        return standardErrorUrl;
    }

    /**
     * Returns the value of the CommandPlugins property for this object.
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The value of the CommandPlugins property for this object.
     */
    public List<CommandPlugin> commandPlugins() {
        return commandPlugins;
    }

    /**
     * <p>
     * The IAM service role that Run Command uses to act on your behalf when sending notifications about command status
     * changes on a per instance basis.
     * </p>
     * 
     * @return The IAM service role that Run Command uses to act on your behalf when sending notifications about command
     *         status changes on a per instance basis.
     */
    public String serviceRole() {
        return serviceRole;
    }

    /**
     * <p>
     * Configurations for sending notifications about command status changes on a per instance basis.
     * </p>
     * 
     * @return Configurations for sending notifications about command status changes on a per instance basis.
     */
    public NotificationConfig notificationConfig() {
        return notificationConfig;
    }

    /**
     * <p>
     * CloudWatch Logs information where you want Systems Manager to send the command output.
     * </p>
     * 
     * @return CloudWatch Logs information where you want Systems Manager to send the command output.
     */
    public CloudWatchOutputConfig cloudWatchOutputConfig() {
        return cloudWatchOutputConfig;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(commandId());
        hashCode = 31 * hashCode + Objects.hashCode(instanceId());
        hashCode = 31 * hashCode + Objects.hashCode(instanceName());
        hashCode = 31 * hashCode + Objects.hashCode(comment());
        hashCode = 31 * hashCode + Objects.hashCode(documentName());
        hashCode = 31 * hashCode + Objects.hashCode(documentVersion());
        hashCode = 31 * hashCode + Objects.hashCode(requestedDateTime());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusDetails());
        hashCode = 31 * hashCode + Objects.hashCode(traceOutput());
        hashCode = 31 * hashCode + Objects.hashCode(standardOutputUrl());
        hashCode = 31 * hashCode + Objects.hashCode(standardErrorUrl());
        hashCode = 31 * hashCode + Objects.hashCode(commandPlugins());
        hashCode = 31 * hashCode + Objects.hashCode(serviceRole());
        hashCode = 31 * hashCode + Objects.hashCode(notificationConfig());
        hashCode = 31 * hashCode + Objects.hashCode(cloudWatchOutputConfig());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CommandInvocation)) {
            return false;
        }
        CommandInvocation other = (CommandInvocation) obj;
        return Objects.equals(commandId(), other.commandId()) && Objects.equals(instanceId(), other.instanceId())
                && Objects.equals(instanceName(), other.instanceName()) && Objects.equals(comment(), other.comment())
                && Objects.equals(documentName(), other.documentName())
                && Objects.equals(documentVersion(), other.documentVersion())
                && Objects.equals(requestedDateTime(), other.requestedDateTime())
                && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(statusDetails(), other.statusDetails()) && Objects.equals(traceOutput(), other.traceOutput())
                && Objects.equals(standardOutputUrl(), other.standardOutputUrl())
                && Objects.equals(standardErrorUrl(), other.standardErrorUrl())
                && Objects.equals(commandPlugins(), other.commandPlugins()) && Objects.equals(serviceRole(), other.serviceRole())
                && Objects.equals(notificationConfig(), other.notificationConfig())
                && Objects.equals(cloudWatchOutputConfig(), other.cloudWatchOutputConfig());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("CommandInvocation").add("CommandId", commandId()).add("InstanceId", instanceId())
                .add("InstanceName", instanceName()).add("Comment", comment()).add("DocumentName", documentName())
                .add("DocumentVersion", documentVersion()).add("RequestedDateTime", requestedDateTime())
                .add("Status", statusAsString()).add("StatusDetails", statusDetails()).add("TraceOutput", traceOutput())
                .add("StandardOutputUrl", standardOutputUrl()).add("StandardErrorUrl", standardErrorUrl())
                .add("CommandPlugins", commandPlugins()).add("ServiceRole", serviceRole())
                .add("NotificationConfig", notificationConfig()).add("CloudWatchOutputConfig", cloudWatchOutputConfig()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "CommandId":
            return Optional.ofNullable(clazz.cast(commandId()));
        case "InstanceId":
            return Optional.ofNullable(clazz.cast(instanceId()));
        case "InstanceName":
            return Optional.ofNullable(clazz.cast(instanceName()));
        case "Comment":
            return Optional.ofNullable(clazz.cast(comment()));
        case "DocumentName":
            return Optional.ofNullable(clazz.cast(documentName()));
        case "DocumentVersion":
            return Optional.ofNullable(clazz.cast(documentVersion()));
        case "RequestedDateTime":
            return Optional.ofNullable(clazz.cast(requestedDateTime()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "StatusDetails":
            return Optional.ofNullable(clazz.cast(statusDetails()));
        case "TraceOutput":
            return Optional.ofNullable(clazz.cast(traceOutput()));
        case "StandardOutputUrl":
            return Optional.ofNullable(clazz.cast(standardOutputUrl()));
        case "StandardErrorUrl":
            return Optional.ofNullable(clazz.cast(standardErrorUrl()));
        case "CommandPlugins":
            return Optional.ofNullable(clazz.cast(commandPlugins()));
        case "ServiceRole":
            return Optional.ofNullable(clazz.cast(serviceRole()));
        case "NotificationConfig":
            return Optional.ofNullable(clazz.cast(notificationConfig()));
        case "CloudWatchOutputConfig":
            return Optional.ofNullable(clazz.cast(cloudWatchOutputConfig()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CommandInvocation, T> g) {
        return obj -> g.apply((CommandInvocation) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CommandInvocation> {
        /**
         * <p>
         * The command against which this invocation was requested.
         * </p>
         * 
         * @param commandId
         *        The command against which this invocation was requested.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder commandId(String commandId);

        /**
         * <p>
         * The instance ID in which this invocation was requested.
         * </p>
         * 
         * @param instanceId
         *        The instance ID in which this invocation was requested.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceId(String instanceId);

        /**
         * <p>
         * The name of the invocation target. For Amazon EC2 instances this is the value for the aws:Name tag. For
         * on-premises instances, this is the name of the instance.
         * </p>
         * 
         * @param instanceName
         *        The name of the invocation target. For Amazon EC2 instances this is the value for the aws:Name tag.
         *        For on-premises instances, this is the name of the instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceName(String instanceName);

        /**
         * <p>
         * User-specified information about the command, such as a brief description of what the command should do.
         * </p>
         * 
         * @param comment
         *        User-specified information about the command, such as a brief description of what the command should
         *        do.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder comment(String comment);

        /**
         * <p>
         * The document name that was requested for execution.
         * </p>
         * 
         * @param documentName
         *        The document name that was requested for execution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder documentName(String documentName);

        /**
         * <p>
         * The SSM document version.
         * </p>
         * 
         * @param documentVersion
         *        The SSM document version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder documentVersion(String documentVersion);

        /**
         * <p>
         * The time and date the request was sent to this instance.
         * </p>
         * 
         * @param requestedDateTime
         *        The time and date the request was sent to this instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requestedDateTime(Instant requestedDateTime);

        /**
         * <p>
         * Whether or not the invocation succeeded, failed, or is pending.
         * </p>
         * 
         * @param status
         *        Whether or not the invocation succeeded, failed, or is pending.
         * @see CommandInvocationStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CommandInvocationStatus
         */
        Builder status(String status);

        /**
         * <p>
         * Whether or not the invocation succeeded, failed, or is pending.
         * </p>
         * 
         * @param status
         *        Whether or not the invocation succeeded, failed, or is pending.
         * @see CommandInvocationStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CommandInvocationStatus
         */
        Builder status(CommandInvocationStatus status);

        /**
         * <p>
         * A detailed status of the command execution for each invocation (each instance targeted by the command).
         * StatusDetails includes more information than Status because it includes states resulting from error and
         * concurrency control parameters. StatusDetails can show different results than Status. For more information
         * about these statuses, see <a
         * href="http://docs.aws.amazon.com/systems-manager/latest/userguide/monitor-commands.html">Understanding
         * Command Statuses</a> in the <i>AWS Systems Manager User Guide</i>. StatusDetails can be one of the following
         * values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Pending: The command has not been sent to the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * In Progress: The command has been sent to the instance but has not reached a terminal state.
         * </p>
         * </li>
         * <li>
         * <p>
         * Success: The execution of the command or plugin was successfully completed. This is a terminal state.
         * </p>
         * </li>
         * <li>
         * <p>
         * Delivery Timed Out: The command was not delivered to the instance before the delivery timeout expired.
         * Delivery timeouts do not count against the parent command's MaxErrors limit, but they do contribute to
         * whether the parent command status is Success or Incomplete. This is a terminal state.
         * </p>
         * </li>
         * <li>
         * <p>
         * Execution Timed Out: Command execution started on the instance, but the execution was not complete before the
         * execution timeout expired. Execution timeouts count against the MaxErrors limit of the parent command. This
         * is a terminal state.
         * </p>
         * </li>
         * <li>
         * <p>
         * Failed: The command was not successful on the instance. For a plugin, this indicates that the result code was
         * not zero. For a command invocation, this indicates that the result code for one or more plugins was not zero.
         * Invocation failures count against the MaxErrors limit of the parent command. This is a terminal state.
         * </p>
         * </li>
         * <li>
         * <p>
         * Canceled: The command was terminated before it was completed. This is a terminal state.
         * </p>
         * </li>
         * <li>
         * <p>
         * Undeliverable: The command can't be delivered to the instance. The instance might not exist or might not be
         * responding. Undeliverable invocations don't count against the parent command's MaxErrors limit and don't
         * contribute to whether the parent command status is Success or Incomplete. This is a terminal state.
         * </p>
         * </li>
         * <li>
         * <p>
         * Terminated: The parent command exceeded its MaxErrors limit and subsequent command invocations were canceled
         * by the system. This is a terminal state.
         * </p>
         * </li>
         * </ul>
         * 
         * @param statusDetails
         *        A detailed status of the command execution for each invocation (each instance targeted by the
         *        command). StatusDetails includes more information than Status because it includes states resulting
         *        from error and concurrency control parameters. StatusDetails can show different results than Status.
         *        For more information about these statuses, see <a
         *        href="http://docs.aws.amazon.com/systems-manager/latest/userguide/monitor-commands.html">Understanding
         *        Command Statuses</a> in the <i>AWS Systems Manager User Guide</i>. StatusDetails can be one of the
         *        following values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        Pending: The command has not been sent to the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        In Progress: The command has been sent to the instance but has not reached a terminal state.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Success: The execution of the command or plugin was successfully completed. This is a terminal state.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Delivery Timed Out: The command was not delivered to the instance before the delivery timeout expired.
         *        Delivery timeouts do not count against the parent command's MaxErrors limit, but they do contribute to
         *        whether the parent command status is Success or Incomplete. This is a terminal state.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Execution Timed Out: Command execution started on the instance, but the execution was not complete
         *        before the execution timeout expired. Execution timeouts count against the MaxErrors limit of the
         *        parent command. This is a terminal state.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Failed: The command was not successful on the instance. For a plugin, this indicates that the result
         *        code was not zero. For a command invocation, this indicates that the result code for one or more
         *        plugins was not zero. Invocation failures count against the MaxErrors limit of the parent command.
         *        This is a terminal state.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Canceled: The command was terminated before it was completed. This is a terminal state.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Undeliverable: The command can't be delivered to the instance. The instance might not exist or might
         *        not be responding. Undeliverable invocations don't count against the parent command's MaxErrors limit
         *        and don't contribute to whether the parent command status is Success or Incomplete. This is a terminal
         *        state.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Terminated: The parent command exceeded its MaxErrors limit and subsequent command invocations were
         *        canceled by the system. This is a terminal state.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusDetails(String statusDetails);

        /**
         * <p>
         * Gets the trace output sent by the agent.
         * </p>
         * 
         * @param traceOutput
         *        Gets the trace output sent by the agent.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder traceOutput(String traceOutput);

        /**
         * <p>
         * The URL to the plugin's StdOut file in Amazon S3, if the Amazon S3 bucket was defined for the parent command.
         * For an invocation, StandardOutputUrl is populated if there is just one plugin defined for the command, and
         * the Amazon S3 bucket was defined for the command.
         * </p>
         * 
         * @param standardOutputUrl
         *        The URL to the plugin's StdOut file in Amazon S3, if the Amazon S3 bucket was defined for the parent
         *        command. For an invocation, StandardOutputUrl is populated if there is just one plugin defined for the
         *        command, and the Amazon S3 bucket was defined for the command.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder standardOutputUrl(String standardOutputUrl);

        /**
         * <p>
         * The URL to the plugin's StdErr file in Amazon S3, if the Amazon S3 bucket was defined for the parent command.
         * For an invocation, StandardErrorUrl is populated if there is just one plugin defined for the command, and the
         * Amazon S3 bucket was defined for the command.
         * </p>
         * 
         * @param standardErrorUrl
         *        The URL to the plugin's StdErr file in Amazon S3, if the Amazon S3 bucket was defined for the parent
         *        command. For an invocation, StandardErrorUrl is populated if there is just one plugin defined for the
         *        command, and the Amazon S3 bucket was defined for the command.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder standardErrorUrl(String standardErrorUrl);

        /**
         * Sets the value of the CommandPlugins property for this object.
         *
         * @param commandPlugins
         *        The new value for the CommandPlugins property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder commandPlugins(Collection<CommandPlugin> commandPlugins);

        /**
         * Sets the value of the CommandPlugins property for this object.
         *
         * @param commandPlugins
         *        The new value for the CommandPlugins property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder commandPlugins(CommandPlugin... commandPlugins);

        /**
         * Sets the value of the CommandPlugins property for this object.
         *
         * This is a convenience that creates an instance of the {@link List<CommandPlugin>.Builder} avoiding the need
         * to create one manually via {@link List<CommandPlugin>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<CommandPlugin>.Builder#build()} is called immediately and
         * its result is passed to {@link #commandPlugins(List<CommandPlugin>)}.
         * 
         * @param commandPlugins
         *        a consumer that will call methods on {@link List<CommandPlugin>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #commandPlugins(List<CommandPlugin>)
         */
        Builder commandPlugins(Consumer<CommandPlugin.Builder>... commandPlugins);

        /**
         * <p>
         * The IAM service role that Run Command uses to act on your behalf when sending notifications about command
         * status changes on a per instance basis.
         * </p>
         * 
         * @param serviceRole
         *        The IAM service role that Run Command uses to act on your behalf when sending notifications about
         *        command status changes on a per instance basis.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serviceRole(String serviceRole);

        /**
         * <p>
         * Configurations for sending notifications about command status changes on a per instance basis.
         * </p>
         * 
         * @param notificationConfig
         *        Configurations for sending notifications about command status changes on a per instance basis.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notificationConfig(NotificationConfig notificationConfig);

        /**
         * <p>
         * Configurations for sending notifications about command status changes on a per instance basis.
         * </p>
         * This is a convenience that creates an instance of the {@link NotificationConfig.Builder} avoiding the need to
         * create one manually via {@link NotificationConfig#builder()}.
         *
         * When the {@link Consumer} completes, {@link NotificationConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #notificationConfig(NotificationConfig)}.
         * 
         * @param notificationConfig
         *        a consumer that will call methods on {@link NotificationConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #notificationConfig(NotificationConfig)
         */
        default Builder notificationConfig(Consumer<NotificationConfig.Builder> notificationConfig) {
            return notificationConfig(NotificationConfig.builder().applyMutation(notificationConfig).build());
        }

        /**
         * <p>
         * CloudWatch Logs information where you want Systems Manager to send the command output.
         * </p>
         * 
         * @param cloudWatchOutputConfig
         *        CloudWatch Logs information where you want Systems Manager to send the command output.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cloudWatchOutputConfig(CloudWatchOutputConfig cloudWatchOutputConfig);

        /**
         * <p>
         * CloudWatch Logs information where you want Systems Manager to send the command output.
         * </p>
         * This is a convenience that creates an instance of the {@link CloudWatchOutputConfig.Builder} avoiding the
         * need to create one manually via {@link CloudWatchOutputConfig#builder()}.
         *
         * When the {@link Consumer} completes, {@link CloudWatchOutputConfig.Builder#build()} is called immediately and
         * its result is passed to {@link #cloudWatchOutputConfig(CloudWatchOutputConfig)}.
         * 
         * @param cloudWatchOutputConfig
         *        a consumer that will call methods on {@link CloudWatchOutputConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #cloudWatchOutputConfig(CloudWatchOutputConfig)
         */
        default Builder cloudWatchOutputConfig(Consumer<CloudWatchOutputConfig.Builder> cloudWatchOutputConfig) {
            return cloudWatchOutputConfig(CloudWatchOutputConfig.builder().applyMutation(cloudWatchOutputConfig).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String commandId;

        private String instanceId;

        private String instanceName;

        private String comment;

        private String documentName;

        private String documentVersion;

        private Instant requestedDateTime;

        private String status;

        private String statusDetails;

        private String traceOutput;

        private String standardOutputUrl;

        private String standardErrorUrl;

        private List<CommandPlugin> commandPlugins = DefaultSdkAutoConstructList.getInstance();

        private String serviceRole;

        private NotificationConfig notificationConfig;

        private CloudWatchOutputConfig cloudWatchOutputConfig;

        private BuilderImpl() {
        }

        private BuilderImpl(CommandInvocation model) {
            commandId(model.commandId);
            instanceId(model.instanceId);
            instanceName(model.instanceName);
            comment(model.comment);
            documentName(model.documentName);
            documentVersion(model.documentVersion);
            requestedDateTime(model.requestedDateTime);
            status(model.status);
            statusDetails(model.statusDetails);
            traceOutput(model.traceOutput);
            standardOutputUrl(model.standardOutputUrl);
            standardErrorUrl(model.standardErrorUrl);
            commandPlugins(model.commandPlugins);
            serviceRole(model.serviceRole);
            notificationConfig(model.notificationConfig);
            cloudWatchOutputConfig(model.cloudWatchOutputConfig);
        }

        public final String getCommandId() {
            return commandId;
        }

        @Override
        public final Builder commandId(String commandId) {
            this.commandId = commandId;
            return this;
        }

        public final void setCommandId(String commandId) {
            this.commandId = commandId;
        }

        public final String getInstanceId() {
            return instanceId;
        }

        @Override
        public final Builder instanceId(String instanceId) {
            this.instanceId = instanceId;
            return this;
        }

        public final void setInstanceId(String instanceId) {
            this.instanceId = instanceId;
        }

        public final String getInstanceName() {
            return instanceName;
        }

        @Override
        public final Builder instanceName(String instanceName) {
            this.instanceName = instanceName;
            return this;
        }

        public final void setInstanceName(String instanceName) {
            this.instanceName = instanceName;
        }

        public final String getComment() {
            return comment;
        }

        @Override
        public final Builder comment(String comment) {
            this.comment = comment;
            return this;
        }

        public final void setComment(String comment) {
            this.comment = comment;
        }

        public final String getDocumentName() {
            return documentName;
        }

        @Override
        public final Builder documentName(String documentName) {
            this.documentName = documentName;
            return this;
        }

        public final void setDocumentName(String documentName) {
            this.documentName = documentName;
        }

        public final String getDocumentVersion() {
            return documentVersion;
        }

        @Override
        public final Builder documentVersion(String documentVersion) {
            this.documentVersion = documentVersion;
            return this;
        }

        public final void setDocumentVersion(String documentVersion) {
            this.documentVersion = documentVersion;
        }

        public final Instant getRequestedDateTime() {
            return requestedDateTime;
        }

        @Override
        public final Builder requestedDateTime(Instant requestedDateTime) {
            this.requestedDateTime = requestedDateTime;
            return this;
        }

        public final void setRequestedDateTime(Instant requestedDateTime) {
            this.requestedDateTime = requestedDateTime;
        }

        public final String getStatusAsString() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(CommandInvocationStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        public final String getStatusDetails() {
            return statusDetails;
        }

        @Override
        public final Builder statusDetails(String statusDetails) {
            this.statusDetails = statusDetails;
            return this;
        }

        public final void setStatusDetails(String statusDetails) {
            this.statusDetails = statusDetails;
        }

        public final String getTraceOutput() {
            return traceOutput;
        }

        @Override
        public final Builder traceOutput(String traceOutput) {
            this.traceOutput = traceOutput;
            return this;
        }

        public final void setTraceOutput(String traceOutput) {
            this.traceOutput = traceOutput;
        }

        public final String getStandardOutputUrl() {
            return standardOutputUrl;
        }

        @Override
        public final Builder standardOutputUrl(String standardOutputUrl) {
            this.standardOutputUrl = standardOutputUrl;
            return this;
        }

        public final void setStandardOutputUrl(String standardOutputUrl) {
            this.standardOutputUrl = standardOutputUrl;
        }

        public final String getStandardErrorUrl() {
            return standardErrorUrl;
        }

        @Override
        public final Builder standardErrorUrl(String standardErrorUrl) {
            this.standardErrorUrl = standardErrorUrl;
            return this;
        }

        public final void setStandardErrorUrl(String standardErrorUrl) {
            this.standardErrorUrl = standardErrorUrl;
        }

        public final Collection<CommandPlugin.Builder> getCommandPlugins() {
            return commandPlugins != null ? commandPlugins.stream().map(CommandPlugin::toBuilder).collect(Collectors.toList())
                    : null;
        }

        @Override
        public final Builder commandPlugins(Collection<CommandPlugin> commandPlugins) {
            this.commandPlugins = CommandPluginListCopier.copy(commandPlugins);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder commandPlugins(CommandPlugin... commandPlugins) {
            commandPlugins(Arrays.asList(commandPlugins));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder commandPlugins(Consumer<CommandPlugin.Builder>... commandPlugins) {
            commandPlugins(Stream.of(commandPlugins).map(c -> CommandPlugin.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setCommandPlugins(Collection<CommandPlugin.BuilderImpl> commandPlugins) {
            this.commandPlugins = CommandPluginListCopier.copyFromBuilder(commandPlugins);
        }

        public final String getServiceRole() {
            return serviceRole;
        }

        @Override
        public final Builder serviceRole(String serviceRole) {
            this.serviceRole = serviceRole;
            return this;
        }

        public final void setServiceRole(String serviceRole) {
            this.serviceRole = serviceRole;
        }

        public final NotificationConfig.Builder getNotificationConfig() {
            return notificationConfig != null ? notificationConfig.toBuilder() : null;
        }

        @Override
        public final Builder notificationConfig(NotificationConfig notificationConfig) {
            this.notificationConfig = notificationConfig;
            return this;
        }

        public final void setNotificationConfig(NotificationConfig.BuilderImpl notificationConfig) {
            this.notificationConfig = notificationConfig != null ? notificationConfig.build() : null;
        }

        public final CloudWatchOutputConfig.Builder getCloudWatchOutputConfig() {
            return cloudWatchOutputConfig != null ? cloudWatchOutputConfig.toBuilder() : null;
        }

        @Override
        public final Builder cloudWatchOutputConfig(CloudWatchOutputConfig cloudWatchOutputConfig) {
            this.cloudWatchOutputConfig = cloudWatchOutputConfig;
            return this;
        }

        public final void setCloudWatchOutputConfig(CloudWatchOutputConfig.BuilderImpl cloudWatchOutputConfig) {
            this.cloudWatchOutputConfig = cloudWatchOutputConfig != null ? cloudWatchOutputConfig.build() : null;
        }

        @Override
        public CommandInvocation build() {
            return new CommandInvocation(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
