/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssmcontacts;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.ssmcontacts.model.AcceptPageRequest;
import software.amazon.awssdk.services.ssmcontacts.model.AcceptPageResponse;
import software.amazon.awssdk.services.ssmcontacts.model.AccessDeniedException;
import software.amazon.awssdk.services.ssmcontacts.model.ActivateContactChannelRequest;
import software.amazon.awssdk.services.ssmcontacts.model.ActivateContactChannelResponse;
import software.amazon.awssdk.services.ssmcontacts.model.ConflictException;
import software.amazon.awssdk.services.ssmcontacts.model.CreateContactChannelRequest;
import software.amazon.awssdk.services.ssmcontacts.model.CreateContactChannelResponse;
import software.amazon.awssdk.services.ssmcontacts.model.CreateContactRequest;
import software.amazon.awssdk.services.ssmcontacts.model.CreateContactResponse;
import software.amazon.awssdk.services.ssmcontacts.model.DataEncryptionException;
import software.amazon.awssdk.services.ssmcontacts.model.DeactivateContactChannelRequest;
import software.amazon.awssdk.services.ssmcontacts.model.DeactivateContactChannelResponse;
import software.amazon.awssdk.services.ssmcontacts.model.DeleteContactChannelRequest;
import software.amazon.awssdk.services.ssmcontacts.model.DeleteContactChannelResponse;
import software.amazon.awssdk.services.ssmcontacts.model.DeleteContactRequest;
import software.amazon.awssdk.services.ssmcontacts.model.DeleteContactResponse;
import software.amazon.awssdk.services.ssmcontacts.model.DescribeEngagementRequest;
import software.amazon.awssdk.services.ssmcontacts.model.DescribeEngagementResponse;
import software.amazon.awssdk.services.ssmcontacts.model.DescribePageRequest;
import software.amazon.awssdk.services.ssmcontacts.model.DescribePageResponse;
import software.amazon.awssdk.services.ssmcontacts.model.GetContactChannelRequest;
import software.amazon.awssdk.services.ssmcontacts.model.GetContactChannelResponse;
import software.amazon.awssdk.services.ssmcontacts.model.GetContactPolicyRequest;
import software.amazon.awssdk.services.ssmcontacts.model.GetContactPolicyResponse;
import software.amazon.awssdk.services.ssmcontacts.model.GetContactRequest;
import software.amazon.awssdk.services.ssmcontacts.model.GetContactResponse;
import software.amazon.awssdk.services.ssmcontacts.model.InternalServerException;
import software.amazon.awssdk.services.ssmcontacts.model.ListContactChannelsRequest;
import software.amazon.awssdk.services.ssmcontacts.model.ListContactChannelsResponse;
import software.amazon.awssdk.services.ssmcontacts.model.ListContactsRequest;
import software.amazon.awssdk.services.ssmcontacts.model.ListContactsResponse;
import software.amazon.awssdk.services.ssmcontacts.model.ListEngagementsRequest;
import software.amazon.awssdk.services.ssmcontacts.model.ListEngagementsResponse;
import software.amazon.awssdk.services.ssmcontacts.model.ListPageReceiptsRequest;
import software.amazon.awssdk.services.ssmcontacts.model.ListPageReceiptsResponse;
import software.amazon.awssdk.services.ssmcontacts.model.ListPagesByContactRequest;
import software.amazon.awssdk.services.ssmcontacts.model.ListPagesByContactResponse;
import software.amazon.awssdk.services.ssmcontacts.model.ListPagesByEngagementRequest;
import software.amazon.awssdk.services.ssmcontacts.model.ListPagesByEngagementResponse;
import software.amazon.awssdk.services.ssmcontacts.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.ssmcontacts.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.ssmcontacts.model.PutContactPolicyRequest;
import software.amazon.awssdk.services.ssmcontacts.model.PutContactPolicyResponse;
import software.amazon.awssdk.services.ssmcontacts.model.ResourceNotFoundException;
import software.amazon.awssdk.services.ssmcontacts.model.SendActivationCodeRequest;
import software.amazon.awssdk.services.ssmcontacts.model.SendActivationCodeResponse;
import software.amazon.awssdk.services.ssmcontacts.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.ssmcontacts.model.SsmContactsException;
import software.amazon.awssdk.services.ssmcontacts.model.SsmContactsRequest;
import software.amazon.awssdk.services.ssmcontacts.model.StartEngagementRequest;
import software.amazon.awssdk.services.ssmcontacts.model.StartEngagementResponse;
import software.amazon.awssdk.services.ssmcontacts.model.StopEngagementRequest;
import software.amazon.awssdk.services.ssmcontacts.model.StopEngagementResponse;
import software.amazon.awssdk.services.ssmcontacts.model.TagResourceRequest;
import software.amazon.awssdk.services.ssmcontacts.model.TagResourceResponse;
import software.amazon.awssdk.services.ssmcontacts.model.ThrottlingException;
import software.amazon.awssdk.services.ssmcontacts.model.UntagResourceRequest;
import software.amazon.awssdk.services.ssmcontacts.model.UntagResourceResponse;
import software.amazon.awssdk.services.ssmcontacts.model.UpdateContactChannelRequest;
import software.amazon.awssdk.services.ssmcontacts.model.UpdateContactChannelResponse;
import software.amazon.awssdk.services.ssmcontacts.model.UpdateContactRequest;
import software.amazon.awssdk.services.ssmcontacts.model.UpdateContactResponse;
import software.amazon.awssdk.services.ssmcontacts.model.ValidationException;
import software.amazon.awssdk.services.ssmcontacts.paginators.ListContactChannelsPublisher;
import software.amazon.awssdk.services.ssmcontacts.paginators.ListContactsPublisher;
import software.amazon.awssdk.services.ssmcontacts.paginators.ListEngagementsPublisher;
import software.amazon.awssdk.services.ssmcontacts.paginators.ListPageReceiptsPublisher;
import software.amazon.awssdk.services.ssmcontacts.paginators.ListPagesByContactPublisher;
import software.amazon.awssdk.services.ssmcontacts.paginators.ListPagesByEngagementPublisher;
import software.amazon.awssdk.services.ssmcontacts.transform.AcceptPageRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.ActivateContactChannelRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.CreateContactChannelRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.CreateContactRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.DeactivateContactChannelRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.DeleteContactChannelRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.DeleteContactRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.DescribeEngagementRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.DescribePageRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.GetContactChannelRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.GetContactPolicyRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.GetContactRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.ListContactChannelsRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.ListContactsRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.ListEngagementsRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.ListPageReceiptsRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.ListPagesByContactRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.ListPagesByEngagementRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.PutContactPolicyRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.SendActivationCodeRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.StartEngagementRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.StopEngagementRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.UpdateContactChannelRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.UpdateContactRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link SsmContactsAsyncClient}.
 *
 * @see SsmContactsAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultSsmContactsAsyncClient implements SsmContactsAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultSsmContactsAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultSsmContactsAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Used to acknowledge an engagement to a contact channel during an incident.
     * </p>
     *
     * @param acceptPageRequest
     * @return A Java Future containing the result of the AcceptPage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.AcceptPage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/AcceptPage" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<AcceptPageResponse> acceptPage(AcceptPageRequest acceptPageRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, acceptPageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AcceptPage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AcceptPageResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    AcceptPageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AcceptPageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AcceptPageRequest, AcceptPageResponse>().withOperationName("AcceptPage")
                            .withMarshaller(new AcceptPageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(acceptPageRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = acceptPageRequest.overrideConfiguration().orElse(null);
            CompletableFuture<AcceptPageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Activates a contact's contact channel. Incident Manager can't engage a contact until the contact channel has been
     * activated.
     * </p>
     *
     * @param activateContactChannelRequest
     * @return A Java Future containing the result of the ActivateContactChannel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.ActivateContactChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/ActivateContactChannel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ActivateContactChannelResponse> activateContactChannel(
            ActivateContactChannelRequest activateContactChannelRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, activateContactChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ActivateContactChannel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ActivateContactChannelResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ActivateContactChannelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ActivateContactChannelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ActivateContactChannelRequest, ActivateContactChannelResponse>()
                            .withOperationName("ActivateContactChannel")
                            .withMarshaller(new ActivateContactChannelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(activateContactChannelRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = activateContactChannelRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<ActivateContactChannelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Contacts are either the contacts that Incident Manager engages during an incident or the escalation plans that
     * Incident Manager uses to engage contacts in phases during an incident.
     * </p>
     *
     * @param createContactRequest
     * @return A Java Future containing the result of the CreateContact operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ConflictException Updating or deleting a resource causes an inconsistent state.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>ServiceQuotaExceededException Request would cause a service quota to be exceeded.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>DataEncryptionException The action failed to due an encryption key error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.CreateContact
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/CreateContact" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateContactResponse> createContact(CreateContactRequest createContactRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createContactRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateContact");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateContactResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateContactResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateContactResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateContactRequest, CreateContactResponse>()
                            .withOperationName("CreateContact")
                            .withMarshaller(new CreateContactRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createContactRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createContactRequest.overrideConfiguration().orElse(null);
            CompletableFuture<CreateContactResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * A contact channel is the method that Incident Manager uses to engage your contact.
     * </p>
     *
     * @param createContactChannelRequest
     * @return A Java Future containing the result of the CreateContactChannel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ConflictException Updating or deleting a resource causes an inconsistent state.</li>
     *         <li>DataEncryptionException The action failed to due an encryption key error.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.CreateContactChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/CreateContactChannel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateContactChannelResponse> createContactChannel(
            CreateContactChannelRequest createContactChannelRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createContactChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateContactChannel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateContactChannelResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateContactChannelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateContactChannelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateContactChannelRequest, CreateContactChannelResponse>()
                            .withOperationName("CreateContactChannel")
                            .withMarshaller(new CreateContactChannelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createContactChannelRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createContactChannelRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<CreateContactChannelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * To no longer receive Incident Manager engagements to a contact channel, you can deactivate the channel.
     * </p>
     *
     * @param deactivateContactChannelRequest
     * @return A Java Future containing the result of the DeactivateContactChannel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.DeactivateContactChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/DeactivateContactChannel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeactivateContactChannelResponse> deactivateContactChannel(
            DeactivateContactChannelRequest deactivateContactChannelRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deactivateContactChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeactivateContactChannel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeactivateContactChannelResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeactivateContactChannelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeactivateContactChannelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeactivateContactChannelRequest, DeactivateContactChannelResponse>()
                            .withOperationName("DeactivateContactChannel")
                            .withMarshaller(new DeactivateContactChannelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deactivateContactChannelRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deactivateContactChannelRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DeactivateContactChannelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * To remove a contact from Incident Manager, you can delete the contact. Deleting a contact removes them from all
     * escalation plans and related response plans. Deleting an escalation plan removes it from all related response
     * plans. You will have to recreate the contact and its contact channels before you can use it again.
     * </p>
     *
     * @param deleteContactRequest
     * @return A Java Future containing the result of the DeleteContact operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.DeleteContact
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/DeleteContact" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteContactResponse> deleteContact(DeleteContactRequest deleteContactRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteContactRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteContact");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteContactResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteContactResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteContactResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteContactRequest, DeleteContactResponse>()
                            .withOperationName("DeleteContact")
                            .withMarshaller(new DeleteContactRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteContactRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteContactRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DeleteContactResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * To no longer receive engagements on a contact channel, you can delete the channel from a contact. Deleting the
     * contact channel removes it from the contact's engagement plan. If you delete the only contact channel for a
     * contact, you won't be able to engage that contact during an incident.
     * </p>
     *
     * @param deleteContactChannelRequest
     * @return A Java Future containing the result of the DeleteContactChannel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.DeleteContactChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/DeleteContactChannel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteContactChannelResponse> deleteContactChannel(
            DeleteContactChannelRequest deleteContactChannelRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteContactChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteContactChannel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteContactChannelResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteContactChannelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteContactChannelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteContactChannelRequest, DeleteContactChannelResponse>()
                            .withOperationName("DeleteContactChannel")
                            .withMarshaller(new DeleteContactChannelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteContactChannelRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteContactChannelRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<DeleteContactChannelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Incident Manager uses engagements to engage contacts and escalation plans during an incident. Use this command to
     * describe the engagement that occurred during an incident.
     * </p>
     *
     * @param describeEngagementRequest
     * @return A Java Future containing the result of the DescribeEngagement operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>DataEncryptionException The action failed to due an encryption key error.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.DescribeEngagement
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/DescribeEngagement"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeEngagementResponse> describeEngagement(DescribeEngagementRequest describeEngagementRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeEngagementRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeEngagement");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeEngagementResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeEngagementResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeEngagementResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeEngagementRequest, DescribeEngagementResponse>()
                            .withOperationName("DescribeEngagement")
                            .withMarshaller(new DescribeEngagementRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeEngagementRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeEngagementRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DescribeEngagementResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists details of the engagement to a contact channel.
     * </p>
     *
     * @param describePageRequest
     * @return A Java Future containing the result of the DescribePage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>DataEncryptionException The action failed to due an encryption key error.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.DescribePage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/DescribePage" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribePageResponse> describePage(DescribePageRequest describePageRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describePageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribePage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribePageResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DescribePageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribePageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribePageRequest, DescribePageResponse>()
                            .withOperationName("DescribePage").withMarshaller(new DescribePageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describePageRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describePageRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DescribePageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves information about the specified contact or escalation plan.
     * </p>
     *
     * @param getContactRequest
     * @return A Java Future containing the result of the GetContact operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>DataEncryptionException The action failed to due an encryption key error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.GetContact
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/GetContact" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetContactResponse> getContact(GetContactRequest getContactRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getContactRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetContact");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetContactResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetContactResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetContactResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetContactRequest, GetContactResponse>().withOperationName("GetContact")
                            .withMarshaller(new GetContactRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getContactRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getContactRequest.overrideConfiguration().orElse(null);
            CompletableFuture<GetContactResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List details about a specific contact channel.
     * </p>
     *
     * @param getContactChannelRequest
     * @return A Java Future containing the result of the GetContactChannel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>DataEncryptionException The action failed to due an encryption key error.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.GetContactChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/GetContactChannel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetContactChannelResponse> getContactChannel(GetContactChannelRequest getContactChannelRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getContactChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetContactChannel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetContactChannelResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetContactChannelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetContactChannelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetContactChannelRequest, GetContactChannelResponse>()
                            .withOperationName("GetContactChannel")
                            .withMarshaller(new GetContactChannelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getContactChannelRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getContactChannelRequest.overrideConfiguration().orElse(null);
            CompletableFuture<GetContactChannelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the resource policies attached to the specified contact or escalation plan.
     * </p>
     *
     * @param getContactPolicyRequest
     * @return A Java Future containing the result of the GetContactPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.GetContactPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/GetContactPolicy" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetContactPolicyResponse> getContactPolicy(GetContactPolicyRequest getContactPolicyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getContactPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetContactPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetContactPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetContactPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetContactPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetContactPolicyRequest, GetContactPolicyResponse>()
                            .withOperationName("GetContactPolicy")
                            .withMarshaller(new GetContactPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getContactPolicyRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getContactPolicyRequest.overrideConfiguration().orElse(null);
            CompletableFuture<GetContactPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all contact channels for the specified contact.
     * </p>
     *
     * @param listContactChannelsRequest
     * @return A Java Future containing the result of the ListContactChannels operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>DataEncryptionException The action failed to due an encryption key error.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.ListContactChannels
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/ListContactChannels"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListContactChannelsResponse> listContactChannels(
            ListContactChannelsRequest listContactChannelsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listContactChannelsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListContactChannels");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListContactChannelsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListContactChannelsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListContactChannelsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListContactChannelsRequest, ListContactChannelsResponse>()
                            .withOperationName("ListContactChannels")
                            .withMarshaller(new ListContactChannelsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listContactChannelsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listContactChannelsRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<ListContactChannelsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all contact channels for the specified contact.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listContactChannels(software.amazon.awssdk.services.ssmcontacts.model.ListContactChannelsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ssmcontacts.paginators.ListContactChannelsPublisher publisher = client.listContactChannelsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ssmcontacts.paginators.ListContactChannelsPublisher publisher = client.listContactChannelsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.ssmcontacts.model.ListContactChannelsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.ssmcontacts.model.ListContactChannelsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listContactChannels(software.amazon.awssdk.services.ssmcontacts.model.ListContactChannelsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listContactChannelsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>DataEncryptionException The action failed to due an encryption key error.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.ListContactChannels
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/ListContactChannels"
     *      target="_top">AWS API Documentation</a>
     */
    public ListContactChannelsPublisher listContactChannelsPaginator(ListContactChannelsRequest listContactChannelsRequest) {
        return new ListContactChannelsPublisher(this, applyPaginatorUserAgent(listContactChannelsRequest));
    }

    /**
     * <p>
     * Lists all contacts and escalation plans in Incident Manager.
     * </p>
     *
     * @param listContactsRequest
     * @return A Java Future containing the result of the ListContacts operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.ListContacts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/ListContacts" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListContactsResponse> listContacts(ListContactsRequest listContactsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listContactsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListContacts");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListContactsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListContactsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListContactsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListContactsRequest, ListContactsResponse>()
                            .withOperationName("ListContacts").withMarshaller(new ListContactsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listContactsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listContactsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListContactsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all contacts and escalation plans in Incident Manager.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listContacts(software.amazon.awssdk.services.ssmcontacts.model.ListContactsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ssmcontacts.paginators.ListContactsPublisher publisher = client.listContactsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ssmcontacts.paginators.ListContactsPublisher publisher = client.listContactsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.ssmcontacts.model.ListContactsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.ssmcontacts.model.ListContactsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listContacts(software.amazon.awssdk.services.ssmcontacts.model.ListContactsRequest)} operation.</b>
     * </p>
     *
     * @param listContactsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.ListContacts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/ListContacts" target="_top">AWS API
     *      Documentation</a>
     */
    public ListContactsPublisher listContactsPaginator(ListContactsRequest listContactsRequest) {
        return new ListContactsPublisher(this, applyPaginatorUserAgent(listContactsRequest));
    }

    /**
     * <p>
     * Lists all engagements that have happened in an incident.
     * </p>
     *
     * @param listEngagementsRequest
     * @return A Java Future containing the result of the ListEngagements operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.ListEngagements
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/ListEngagements" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListEngagementsResponse> listEngagements(ListEngagementsRequest listEngagementsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listEngagementsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListEngagements");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListEngagementsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListEngagementsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListEngagementsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListEngagementsRequest, ListEngagementsResponse>()
                            .withOperationName("ListEngagements")
                            .withMarshaller(new ListEngagementsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listEngagementsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listEngagementsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListEngagementsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all engagements that have happened in an incident.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listEngagements(software.amazon.awssdk.services.ssmcontacts.model.ListEngagementsRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ssmcontacts.paginators.ListEngagementsPublisher publisher = client.listEngagementsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ssmcontacts.paginators.ListEngagementsPublisher publisher = client.listEngagementsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.ssmcontacts.model.ListEngagementsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.ssmcontacts.model.ListEngagementsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listEngagements(software.amazon.awssdk.services.ssmcontacts.model.ListEngagementsRequest)} operation.</b>
     * </p>
     *
     * @param listEngagementsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.ListEngagements
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/ListEngagements" target="_top">AWS
     *      API Documentation</a>
     */
    public ListEngagementsPublisher listEngagementsPaginator(ListEngagementsRequest listEngagementsRequest) {
        return new ListEngagementsPublisher(this, applyPaginatorUserAgent(listEngagementsRequest));
    }

    /**
     * <p>
     * Lists all of the engagements to contact channels that have been acknowledged.
     * </p>
     *
     * @param listPageReceiptsRequest
     * @return A Java Future containing the result of the ListPageReceipts operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.ListPageReceipts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/ListPageReceipts" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListPageReceiptsResponse> listPageReceipts(ListPageReceiptsRequest listPageReceiptsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPageReceiptsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPageReceipts");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPageReceiptsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListPageReceiptsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListPageReceiptsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPageReceiptsRequest, ListPageReceiptsResponse>()
                            .withOperationName("ListPageReceipts")
                            .withMarshaller(new ListPageReceiptsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listPageReceiptsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listPageReceiptsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListPageReceiptsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all of the engagements to contact channels that have been acknowledged.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listPageReceipts(software.amazon.awssdk.services.ssmcontacts.model.ListPageReceiptsRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ssmcontacts.paginators.ListPageReceiptsPublisher publisher = client.listPageReceiptsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ssmcontacts.paginators.ListPageReceiptsPublisher publisher = client.listPageReceiptsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.ssmcontacts.model.ListPageReceiptsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.ssmcontacts.model.ListPageReceiptsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listPageReceipts(software.amazon.awssdk.services.ssmcontacts.model.ListPageReceiptsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listPageReceiptsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.ListPageReceipts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/ListPageReceipts" target="_top">AWS
     *      API Documentation</a>
     */
    public ListPageReceiptsPublisher listPageReceiptsPaginator(ListPageReceiptsRequest listPageReceiptsRequest) {
        return new ListPageReceiptsPublisher(this, applyPaginatorUserAgent(listPageReceiptsRequest));
    }

    /**
     * <p>
     * Lists the engagements to a contact's contact channels.
     * </p>
     *
     * @param listPagesByContactRequest
     * @return A Java Future containing the result of the ListPagesByContact operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.ListPagesByContact
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/ListPagesByContact"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListPagesByContactResponse> listPagesByContact(ListPagesByContactRequest listPagesByContactRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPagesByContactRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPagesByContact");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPagesByContactResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListPagesByContactResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListPagesByContactResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPagesByContactRequest, ListPagesByContactResponse>()
                            .withOperationName("ListPagesByContact")
                            .withMarshaller(new ListPagesByContactRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listPagesByContactRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listPagesByContactRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<ListPagesByContactResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the engagements to a contact's contact channels.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listPagesByContact(software.amazon.awssdk.services.ssmcontacts.model.ListPagesByContactRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ssmcontacts.paginators.ListPagesByContactPublisher publisher = client.listPagesByContactPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ssmcontacts.paginators.ListPagesByContactPublisher publisher = client.listPagesByContactPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.ssmcontacts.model.ListPagesByContactResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.ssmcontacts.model.ListPagesByContactResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listPagesByContact(software.amazon.awssdk.services.ssmcontacts.model.ListPagesByContactRequest)}
     * operation.</b>
     * </p>
     *
     * @param listPagesByContactRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.ListPagesByContact
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/ListPagesByContact"
     *      target="_top">AWS API Documentation</a>
     */
    public ListPagesByContactPublisher listPagesByContactPaginator(ListPagesByContactRequest listPagesByContactRequest) {
        return new ListPagesByContactPublisher(this, applyPaginatorUserAgent(listPagesByContactRequest));
    }

    /**
     * <p>
     * Lists the engagements to contact channels that occurred by engaging a contact.
     * </p>
     *
     * @param listPagesByEngagementRequest
     * @return A Java Future containing the result of the ListPagesByEngagement operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.ListPagesByEngagement
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/ListPagesByEngagement"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListPagesByEngagementResponse> listPagesByEngagement(
            ListPagesByEngagementRequest listPagesByEngagementRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPagesByEngagementRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPagesByEngagement");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPagesByEngagementResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListPagesByEngagementResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListPagesByEngagementResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPagesByEngagementRequest, ListPagesByEngagementResponse>()
                            .withOperationName("ListPagesByEngagement")
                            .withMarshaller(new ListPagesByEngagementRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listPagesByEngagementRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listPagesByEngagementRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<ListPagesByEngagementResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the engagements to contact channels that occurred by engaging a contact.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listPagesByEngagement(software.amazon.awssdk.services.ssmcontacts.model.ListPagesByEngagementRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ssmcontacts.paginators.ListPagesByEngagementPublisher publisher = client.listPagesByEngagementPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ssmcontacts.paginators.ListPagesByEngagementPublisher publisher = client.listPagesByEngagementPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.ssmcontacts.model.ListPagesByEngagementResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.ssmcontacts.model.ListPagesByEngagementResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listPagesByEngagement(software.amazon.awssdk.services.ssmcontacts.model.ListPagesByEngagementRequest)}
     * operation.</b>
     * </p>
     *
     * @param listPagesByEngagementRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.ListPagesByEngagement
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/ListPagesByEngagement"
     *      target="_top">AWS API Documentation</a>
     */
    public ListPagesByEngagementPublisher listPagesByEngagementPaginator(ListPagesByEngagementRequest listPagesByEngagementRequest) {
        return new ListPagesByEngagementPublisher(this, applyPaginatorUserAgent(listPagesByEngagementRequest));
    }

    /**
     * <p>
     * Lists the tags of an escalation plan or contact.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTagsForResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listTagsForResourceRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds a resource to the specified contact or escalation plan.
     * </p>
     *
     * @param putContactPolicyRequest
     * @return A Java Future containing the result of the PutContactPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ConflictException Updating or deleting a resource causes an inconsistent state.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.PutContactPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/PutContactPolicy" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<PutContactPolicyResponse> putContactPolicy(PutContactPolicyRequest putContactPolicyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putContactPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutContactPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutContactPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutContactPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutContactPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutContactPolicyRequest, PutContactPolicyResponse>()
                            .withOperationName("PutContactPolicy")
                            .withMarshaller(new PutContactPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(putContactPolicyRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = putContactPolicyRequest.overrideConfiguration().orElse(null);
            CompletableFuture<PutContactPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Sends an activation code to a contact channel. The contact can use this code to activate the contact channel in
     * the console or with the <code>ActivateChannel</code> action. Incident Manager can't engage a contact channel
     * until it has been activated.
     * </p>
     *
     * @param sendActivationCodeRequest
     * @return A Java Future containing the result of the SendActivationCode operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>DataEncryptionException The action failed to due an encryption key error.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>ServiceQuotaExceededException Request would cause a service quota to be exceeded.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.SendActivationCode
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/SendActivationCode"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<SendActivationCodeResponse> sendActivationCode(SendActivationCodeRequest sendActivationCodeRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, sendActivationCodeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SendActivationCode");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SendActivationCodeResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, SendActivationCodeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<SendActivationCodeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SendActivationCodeRequest, SendActivationCodeResponse>()
                            .withOperationName("SendActivationCode")
                            .withMarshaller(new SendActivationCodeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(sendActivationCodeRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = sendActivationCodeRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<SendActivationCodeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts an engagement to a contact or escalation plan. The engagement engages each contact specified in the
     * incident.
     * </p>
     *
     * @param startEngagementRequest
     * @return A Java Future containing the result of the StartEngagement operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>DataEncryptionException The action failed to due an encryption key error.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.StartEngagement
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/StartEngagement" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<StartEngagementResponse> startEngagement(StartEngagementRequest startEngagementRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startEngagementRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartEngagement");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartEngagementResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartEngagementResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartEngagementResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartEngagementRequest, StartEngagementResponse>()
                            .withOperationName("StartEngagement")
                            .withMarshaller(new StartEngagementRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(startEngagementRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = startEngagementRequest.overrideConfiguration().orElse(null);
            CompletableFuture<StartEngagementResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Stops an engagement before it finishes the final stage of the escalation plan or engagement plan. Further
     * contacts aren't engaged.
     * </p>
     *
     * @param stopEngagementRequest
     * @return A Java Future containing the result of the StopEngagement operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.StopEngagement
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/StopEngagement" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<StopEngagementResponse> stopEngagement(StopEngagementRequest stopEngagementRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopEngagementRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopEngagement");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StopEngagementResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StopEngagementResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StopEngagementResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StopEngagementRequest, StopEngagementResponse>()
                            .withOperationName("StopEngagement")
                            .withMarshaller(new StopEngagementRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(stopEngagementRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = stopEngagementRequest.overrideConfiguration().orElse(null);
            CompletableFuture<StopEngagementResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Tags a contact or escalation plan. You can tag only contacts and escalation plans in the first region of your
     * replication set.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>ServiceQuotaExceededException Request would cause a service quota to be exceeded.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(tagResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = tagResourceRequest.overrideConfiguration().orElse(null);
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes tags from the specified resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(untagResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = untagResourceRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the contact or escalation plan specified.
     * </p>
     *
     * @param updateContactRequest
     * @return A Java Future containing the result of the UpdateContact operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>ServiceQuotaExceededException Request would cause a service quota to be exceeded.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>DataEncryptionException The action failed to due an encryption key error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.UpdateContact
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/UpdateContact" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateContactResponse> updateContact(UpdateContactRequest updateContactRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateContactRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateContact");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateContactResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateContactResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateContactResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateContactRequest, UpdateContactResponse>()
                            .withOperationName("UpdateContact")
                            .withMarshaller(new UpdateContactRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateContactRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateContactRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UpdateContactResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a contact's contact channel.
     * </p>
     *
     * @param updateContactChannelRequest
     * @return A Java Future containing the result of the UpdateContactChannel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ConflictException Updating or deleting a resource causes an inconsistent state.</li>
     *         <li>DataEncryptionException The action failed to due an encryption key error.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.UpdateContactChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/UpdateContactChannel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateContactChannelResponse> updateContactChannel(
            UpdateContactChannelRequest updateContactChannelRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateContactChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateContactChannel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateContactChannelResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateContactChannelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateContactChannelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateContactChannelRequest, UpdateContactChannelResponse>()
                            .withOperationName("UpdateContactChannel")
                            .withMarshaller(new UpdateContactChannelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateContactChannelRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateContactChannelRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<UpdateContactChannelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(SsmContactsException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DataEncryptionException")
                                .exceptionBuilderSupplier(DataEncryptionException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends SsmContactsRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
