/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssmcontacts.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about a resource that another resource is related to or depends on.
 * </p>
 * <p>
 * For example, if a contact is a member of a rotation, the rotation is a dependent entity of the contact.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DependentEntity implements SdkPojo, Serializable, ToCopyableBuilder<DependentEntity.Builder, DependentEntity> {
    private static final SdkField<String> RELATION_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RelationType").getter(getter(DependentEntity::relationType)).setter(setter(Builder::relationType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RelationType").build()).build();

    private static final SdkField<List<String>> DEPENDENT_RESOURCE_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("DependentResourceIds")
            .getter(getter(DependentEntity::dependentResourceIds))
            .setter(setter(Builder::dependentResourceIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DependentResourceIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(RELATION_TYPE_FIELD,
            DEPENDENT_RESOURCE_IDS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String relationType;

    private final List<String> dependentResourceIds;

    private DependentEntity(BuilderImpl builder) {
        this.relationType = builder.relationType;
        this.dependentResourceIds = builder.dependentResourceIds;
    }

    /**
     * <p>
     * The type of relationship between one resource and the other resource that it is related to or depends on.
     * </p>
     * 
     * @return The type of relationship between one resource and the other resource that it is related to or depends on.
     */
    public final String relationType() {
        return relationType;
    }

    /**
     * For responses, this returns true if the service returned a value for the DependentResourceIds property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasDependentResourceIds() {
        return dependentResourceIds != null && !(dependentResourceIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The Amazon Resource Names (ARNs) of the dependent resources.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDependentResourceIds} method.
     * </p>
     * 
     * @return The Amazon Resource Names (ARNs) of the dependent resources.
     */
    public final List<String> dependentResourceIds() {
        return dependentResourceIds;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(relationType());
        hashCode = 31 * hashCode + Objects.hashCode(hasDependentResourceIds() ? dependentResourceIds() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DependentEntity)) {
            return false;
        }
        DependentEntity other = (DependentEntity) obj;
        return Objects.equals(relationType(), other.relationType())
                && hasDependentResourceIds() == other.hasDependentResourceIds()
                && Objects.equals(dependentResourceIds(), other.dependentResourceIds());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DependentEntity").add("RelationType", relationType())
                .add("DependentResourceIds", hasDependentResourceIds() ? dependentResourceIds() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "RelationType":
            return Optional.ofNullable(clazz.cast(relationType()));
        case "DependentResourceIds":
            return Optional.ofNullable(clazz.cast(dependentResourceIds()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DependentEntity, T> g) {
        return obj -> g.apply((DependentEntity) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DependentEntity> {
        /**
         * <p>
         * The type of relationship between one resource and the other resource that it is related to or depends on.
         * </p>
         * 
         * @param relationType
         *        The type of relationship between one resource and the other resource that it is related to or depends
         *        on.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder relationType(String relationType);

        /**
         * <p>
         * The Amazon Resource Names (ARNs) of the dependent resources.
         * </p>
         * 
         * @param dependentResourceIds
         *        The Amazon Resource Names (ARNs) of the dependent resources.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dependentResourceIds(Collection<String> dependentResourceIds);

        /**
         * <p>
         * The Amazon Resource Names (ARNs) of the dependent resources.
         * </p>
         * 
         * @param dependentResourceIds
         *        The Amazon Resource Names (ARNs) of the dependent resources.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dependentResourceIds(String... dependentResourceIds);
    }

    static final class BuilderImpl implements Builder {
        private String relationType;

        private List<String> dependentResourceIds = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(DependentEntity model) {
            relationType(model.relationType);
            dependentResourceIds(model.dependentResourceIds);
        }

        public final String getRelationType() {
            return relationType;
        }

        public final void setRelationType(String relationType) {
            this.relationType = relationType;
        }

        @Override
        public final Builder relationType(String relationType) {
            this.relationType = relationType;
            return this;
        }

        public final Collection<String> getDependentResourceIds() {
            if (dependentResourceIds instanceof SdkAutoConstructList) {
                return null;
            }
            return dependentResourceIds;
        }

        public final void setDependentResourceIds(Collection<String> dependentResourceIds) {
            this.dependentResourceIds = SsmContactsArnListCopier.copy(dependentResourceIds);
        }

        @Override
        public final Builder dependentResourceIds(Collection<String> dependentResourceIds) {
            this.dependentResourceIds = SsmContactsArnListCopier.copy(dependentResourceIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dependentResourceIds(String... dependentResourceIds) {
            dependentResourceIds(Arrays.asList(dependentResourceIds));
            return this;
        }

        @Override
        public DependentEntity build() {
            return new DependentEntity(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
