/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssmcontacts;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.ssmcontacts.model.AcceptPageRequest;
import software.amazon.awssdk.services.ssmcontacts.model.AcceptPageResponse;
import software.amazon.awssdk.services.ssmcontacts.model.AccessDeniedException;
import software.amazon.awssdk.services.ssmcontacts.model.ActivateContactChannelRequest;
import software.amazon.awssdk.services.ssmcontacts.model.ActivateContactChannelResponse;
import software.amazon.awssdk.services.ssmcontacts.model.ConflictException;
import software.amazon.awssdk.services.ssmcontacts.model.CreateContactChannelRequest;
import software.amazon.awssdk.services.ssmcontacts.model.CreateContactChannelResponse;
import software.amazon.awssdk.services.ssmcontacts.model.CreateContactRequest;
import software.amazon.awssdk.services.ssmcontacts.model.CreateContactResponse;
import software.amazon.awssdk.services.ssmcontacts.model.DataEncryptionException;
import software.amazon.awssdk.services.ssmcontacts.model.DeactivateContactChannelRequest;
import software.amazon.awssdk.services.ssmcontacts.model.DeactivateContactChannelResponse;
import software.amazon.awssdk.services.ssmcontacts.model.DeleteContactChannelRequest;
import software.amazon.awssdk.services.ssmcontacts.model.DeleteContactChannelResponse;
import software.amazon.awssdk.services.ssmcontacts.model.DeleteContactRequest;
import software.amazon.awssdk.services.ssmcontacts.model.DeleteContactResponse;
import software.amazon.awssdk.services.ssmcontacts.model.DescribeEngagementRequest;
import software.amazon.awssdk.services.ssmcontacts.model.DescribeEngagementResponse;
import software.amazon.awssdk.services.ssmcontacts.model.DescribePageRequest;
import software.amazon.awssdk.services.ssmcontacts.model.DescribePageResponse;
import software.amazon.awssdk.services.ssmcontacts.model.GetContactChannelRequest;
import software.amazon.awssdk.services.ssmcontacts.model.GetContactChannelResponse;
import software.amazon.awssdk.services.ssmcontacts.model.GetContactPolicyRequest;
import software.amazon.awssdk.services.ssmcontacts.model.GetContactPolicyResponse;
import software.amazon.awssdk.services.ssmcontacts.model.GetContactRequest;
import software.amazon.awssdk.services.ssmcontacts.model.GetContactResponse;
import software.amazon.awssdk.services.ssmcontacts.model.InternalServerException;
import software.amazon.awssdk.services.ssmcontacts.model.ListContactChannelsRequest;
import software.amazon.awssdk.services.ssmcontacts.model.ListContactChannelsResponse;
import software.amazon.awssdk.services.ssmcontacts.model.ListContactsRequest;
import software.amazon.awssdk.services.ssmcontacts.model.ListContactsResponse;
import software.amazon.awssdk.services.ssmcontacts.model.ListEngagementsRequest;
import software.amazon.awssdk.services.ssmcontacts.model.ListEngagementsResponse;
import software.amazon.awssdk.services.ssmcontacts.model.ListPageReceiptsRequest;
import software.amazon.awssdk.services.ssmcontacts.model.ListPageReceiptsResponse;
import software.amazon.awssdk.services.ssmcontacts.model.ListPagesByContactRequest;
import software.amazon.awssdk.services.ssmcontacts.model.ListPagesByContactResponse;
import software.amazon.awssdk.services.ssmcontacts.model.ListPagesByEngagementRequest;
import software.amazon.awssdk.services.ssmcontacts.model.ListPagesByEngagementResponse;
import software.amazon.awssdk.services.ssmcontacts.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.ssmcontacts.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.ssmcontacts.model.PutContactPolicyRequest;
import software.amazon.awssdk.services.ssmcontacts.model.PutContactPolicyResponse;
import software.amazon.awssdk.services.ssmcontacts.model.ResourceNotFoundException;
import software.amazon.awssdk.services.ssmcontacts.model.SendActivationCodeRequest;
import software.amazon.awssdk.services.ssmcontacts.model.SendActivationCodeResponse;
import software.amazon.awssdk.services.ssmcontacts.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.ssmcontacts.model.SsmContactsException;
import software.amazon.awssdk.services.ssmcontacts.model.SsmContactsRequest;
import software.amazon.awssdk.services.ssmcontacts.model.StartEngagementRequest;
import software.amazon.awssdk.services.ssmcontacts.model.StartEngagementResponse;
import software.amazon.awssdk.services.ssmcontacts.model.StopEngagementRequest;
import software.amazon.awssdk.services.ssmcontacts.model.StopEngagementResponse;
import software.amazon.awssdk.services.ssmcontacts.model.TagResourceRequest;
import software.amazon.awssdk.services.ssmcontacts.model.TagResourceResponse;
import software.amazon.awssdk.services.ssmcontacts.model.ThrottlingException;
import software.amazon.awssdk.services.ssmcontacts.model.UntagResourceRequest;
import software.amazon.awssdk.services.ssmcontacts.model.UntagResourceResponse;
import software.amazon.awssdk.services.ssmcontacts.model.UpdateContactChannelRequest;
import software.amazon.awssdk.services.ssmcontacts.model.UpdateContactChannelResponse;
import software.amazon.awssdk.services.ssmcontacts.model.UpdateContactRequest;
import software.amazon.awssdk.services.ssmcontacts.model.UpdateContactResponse;
import software.amazon.awssdk.services.ssmcontacts.model.ValidationException;
import software.amazon.awssdk.services.ssmcontacts.paginators.ListContactChannelsIterable;
import software.amazon.awssdk.services.ssmcontacts.paginators.ListContactsIterable;
import software.amazon.awssdk.services.ssmcontacts.paginators.ListEngagementsIterable;
import software.amazon.awssdk.services.ssmcontacts.paginators.ListPageReceiptsIterable;
import software.amazon.awssdk.services.ssmcontacts.paginators.ListPagesByContactIterable;
import software.amazon.awssdk.services.ssmcontacts.paginators.ListPagesByEngagementIterable;
import software.amazon.awssdk.services.ssmcontacts.transform.AcceptPageRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.ActivateContactChannelRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.CreateContactChannelRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.CreateContactRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.DeactivateContactChannelRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.DeleteContactChannelRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.DeleteContactRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.DescribeEngagementRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.DescribePageRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.GetContactChannelRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.GetContactPolicyRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.GetContactRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.ListContactChannelsRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.ListContactsRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.ListEngagementsRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.ListPageReceiptsRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.ListPagesByContactRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.ListPagesByEngagementRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.PutContactPolicyRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.SendActivationCodeRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.StartEngagementRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.StopEngagementRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.UpdateContactChannelRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.UpdateContactRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link SsmContactsClient}.
 *
 * @see SsmContactsClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultSsmContactsClient implements SsmContactsClient {
    private static final Logger log = Logger.loggerFor(DefaultSsmContactsClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultSsmContactsClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Used to acknowledge an engagement to a contact channel during an incident.
     * </p>
     *
     * @param acceptPageRequest
     * @return Result of the AcceptPage operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.AcceptPage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/AcceptPage" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public AcceptPageResponse acceptPage(AcceptPageRequest acceptPageRequest) throws AccessDeniedException,
            InternalServerException, ResourceNotFoundException, ThrottlingException, ValidationException, AwsServiceException,
            SdkClientException, SsmContactsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AcceptPageResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                AcceptPageResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, acceptPageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AcceptPage");

            return clientHandler
                    .execute(new ClientExecutionParams<AcceptPageRequest, AcceptPageResponse>().withOperationName("AcceptPage")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(acceptPageRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new AcceptPageRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Activates a contact's contact channel. Incident Manager can't engage a contact until the contact channel has been
     * activated.
     * </p>
     *
     * @param activateContactChannelRequest
     * @return Result of the ActivateContactChannel operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.ActivateContactChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/ActivateContactChannel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ActivateContactChannelResponse activateContactChannel(ActivateContactChannelRequest activateContactChannelRequest)
            throws AccessDeniedException, InternalServerException, ResourceNotFoundException, ThrottlingException,
            ValidationException, AwsServiceException, SdkClientException, SsmContactsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ActivateContactChannelResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ActivateContactChannelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, activateContactChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ActivateContactChannel");

            return clientHandler
                    .execute(new ClientExecutionParams<ActivateContactChannelRequest, ActivateContactChannelResponse>()
                            .withOperationName("ActivateContactChannel").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(activateContactChannelRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ActivateContactChannelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Contacts are either the contacts that Incident Manager engages during an incident or the escalation plans that
     * Incident Manager uses to engage contacts in phases during an incident.
     * </p>
     *
     * @param createContactRequest
     * @return Result of the CreateContact operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws ConflictException
     *         Updating or deleting a resource causes an inconsistent state.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws ServiceQuotaExceededException
     *         Request would cause a service quota to be exceeded.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws DataEncryptionException
     *         The operation failed to due an encryption key error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.CreateContact
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/CreateContact" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateContactResponse createContact(CreateContactRequest createContactRequest) throws AccessDeniedException,
            ConflictException, InternalServerException, ServiceQuotaExceededException, ThrottlingException, ValidationException,
            DataEncryptionException, AwsServiceException, SdkClientException, SsmContactsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateContactResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateContactResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createContactRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateContact");

            return clientHandler.execute(new ClientExecutionParams<CreateContactRequest, CreateContactResponse>()
                    .withOperationName("CreateContact").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createContactRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateContactRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * A contact channel is the method that Incident Manager uses to engage your contact.
     * </p>
     *
     * @param createContactChannelRequest
     * @return Result of the CreateContactChannel operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws ConflictException
     *         Updating or deleting a resource causes an inconsistent state.
     * @throws DataEncryptionException
     *         The operation failed to due an encryption key error.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.CreateContactChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/CreateContactChannel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateContactChannelResponse createContactChannel(CreateContactChannelRequest createContactChannelRequest)
            throws AccessDeniedException, ConflictException, DataEncryptionException, InternalServerException,
            ThrottlingException, ValidationException, AwsServiceException, SdkClientException, SsmContactsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateContactChannelResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateContactChannelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createContactChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateContactChannel");

            return clientHandler.execute(new ClientExecutionParams<CreateContactChannelRequest, CreateContactChannelResponse>()
                    .withOperationName("CreateContactChannel").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createContactChannelRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateContactChannelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * To no longer receive Incident Manager engagements to a contact channel, you can deactivate the channel.
     * </p>
     *
     * @param deactivateContactChannelRequest
     * @return Result of the DeactivateContactChannel operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.DeactivateContactChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/DeactivateContactChannel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeactivateContactChannelResponse deactivateContactChannel(
            DeactivateContactChannelRequest deactivateContactChannelRequest) throws AccessDeniedException,
            InternalServerException, ResourceNotFoundException, ThrottlingException, ValidationException, AwsServiceException,
            SdkClientException, SsmContactsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeactivateContactChannelResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeactivateContactChannelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deactivateContactChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeactivateContactChannel");

            return clientHandler
                    .execute(new ClientExecutionParams<DeactivateContactChannelRequest, DeactivateContactChannelResponse>()
                            .withOperationName("DeactivateContactChannel").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(deactivateContactChannelRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeactivateContactChannelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * To remove a contact from Incident Manager, you can delete the contact. Deleting a contact removes them from all
     * escalation plans and related response plans. Deleting an escalation plan removes it from all related response
     * plans. You will have to recreate the contact and its contact channels before you can use it again.
     * </p>
     *
     * @param deleteContactRequest
     * @return Result of the DeleteContact operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.DeleteContact
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/DeleteContact" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteContactResponse deleteContact(DeleteContactRequest deleteContactRequest) throws AccessDeniedException,
            InternalServerException, ResourceNotFoundException, ThrottlingException, ValidationException, AwsServiceException,
            SdkClientException, SsmContactsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteContactResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteContactResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteContactRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteContact");

            return clientHandler.execute(new ClientExecutionParams<DeleteContactRequest, DeleteContactResponse>()
                    .withOperationName("DeleteContact").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteContactRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteContactRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * To no longer receive engagements on a contact channel, you can delete the channel from a contact. Deleting the
     * contact channel removes it from the contact's engagement plan. If you delete the only contact channel for a
     * contact, you won't be able to engage that contact during an incident.
     * </p>
     *
     * @param deleteContactChannelRequest
     * @return Result of the DeleteContactChannel operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.DeleteContactChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/DeleteContactChannel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteContactChannelResponse deleteContactChannel(DeleteContactChannelRequest deleteContactChannelRequest)
            throws AccessDeniedException, InternalServerException, ResourceNotFoundException, ThrottlingException,
            ValidationException, AwsServiceException, SdkClientException, SsmContactsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteContactChannelResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteContactChannelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteContactChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteContactChannel");

            return clientHandler.execute(new ClientExecutionParams<DeleteContactChannelRequest, DeleteContactChannelResponse>()
                    .withOperationName("DeleteContactChannel").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteContactChannelRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteContactChannelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Incident Manager uses engagements to engage contacts and escalation plans during an incident. Use this command to
     * describe the engagement that occurred during an incident.
     * </p>
     *
     * @param describeEngagementRequest
     * @return Result of the DescribeEngagement operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws DataEncryptionException
     *         The operation failed to due an encryption key error.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.DescribeEngagement
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/DescribeEngagement"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeEngagementResponse describeEngagement(DescribeEngagementRequest describeEngagementRequest)
            throws AccessDeniedException, DataEncryptionException, InternalServerException, ResourceNotFoundException,
            ThrottlingException, ValidationException, AwsServiceException, SdkClientException, SsmContactsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeEngagementResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeEngagementResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeEngagementRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeEngagement");

            return clientHandler.execute(new ClientExecutionParams<DescribeEngagementRequest, DescribeEngagementResponse>()
                    .withOperationName("DescribeEngagement").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeEngagementRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeEngagementRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists details of the engagement to a contact channel.
     * </p>
     *
     * @param describePageRequest
     * @return Result of the DescribePage operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws DataEncryptionException
     *         The operation failed to due an encryption key error.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.DescribePage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/DescribePage" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribePageResponse describePage(DescribePageRequest describePageRequest) throws AccessDeniedException,
            DataEncryptionException, InternalServerException, ResourceNotFoundException, ThrottlingException,
            ValidationException, AwsServiceException, SdkClientException, SsmContactsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribePageResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribePageResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describePageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribePage");

            return clientHandler.execute(new ClientExecutionParams<DescribePageRequest, DescribePageResponse>()
                    .withOperationName("DescribePage").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describePageRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribePageRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves information about the specified contact or escalation plan.
     * </p>
     *
     * @param getContactRequest
     * @return Result of the GetContact operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws DataEncryptionException
     *         The operation failed to due an encryption key error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.GetContact
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/GetContact" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetContactResponse getContact(GetContactRequest getContactRequest) throws AccessDeniedException,
            InternalServerException, ResourceNotFoundException, ThrottlingException, ValidationException,
            DataEncryptionException, AwsServiceException, SdkClientException, SsmContactsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetContactResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetContactResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getContactRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetContact");

            return clientHandler
                    .execute(new ClientExecutionParams<GetContactRequest, GetContactResponse>().withOperationName("GetContact")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getContactRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetContactRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * List details about a specific contact channel.
     * </p>
     *
     * @param getContactChannelRequest
     * @return Result of the GetContactChannel operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws DataEncryptionException
     *         The operation failed to due an encryption key error.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.GetContactChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/GetContactChannel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetContactChannelResponse getContactChannel(GetContactChannelRequest getContactChannelRequest)
            throws AccessDeniedException, DataEncryptionException, InternalServerException, ResourceNotFoundException,
            ThrottlingException, ValidationException, AwsServiceException, SdkClientException, SsmContactsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetContactChannelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetContactChannelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getContactChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetContactChannel");

            return clientHandler.execute(new ClientExecutionParams<GetContactChannelRequest, GetContactChannelResponse>()
                    .withOperationName("GetContactChannel").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getContactChannelRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetContactChannelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the resource policies attached to the specified contact or escalation plan.
     * </p>
     *
     * @param getContactPolicyRequest
     * @return Result of the GetContactPolicy operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.GetContactPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/GetContactPolicy" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetContactPolicyResponse getContactPolicy(GetContactPolicyRequest getContactPolicyRequest)
            throws AccessDeniedException, InternalServerException, ResourceNotFoundException, ThrottlingException,
            ValidationException, AwsServiceException, SdkClientException, SsmContactsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetContactPolicyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetContactPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getContactPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetContactPolicy");

            return clientHandler.execute(new ClientExecutionParams<GetContactPolicyRequest, GetContactPolicyResponse>()
                    .withOperationName("GetContactPolicy").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getContactPolicyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetContactPolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all contact channels for the specified contact.
     * </p>
     *
     * @param listContactChannelsRequest
     * @return Result of the ListContactChannels operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws DataEncryptionException
     *         The operation failed to due an encryption key error.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.ListContactChannels
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/ListContactChannels"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListContactChannelsResponse listContactChannels(ListContactChannelsRequest listContactChannelsRequest)
            throws AccessDeniedException, DataEncryptionException, InternalServerException, ResourceNotFoundException,
            ThrottlingException, ValidationException, AwsServiceException, SdkClientException, SsmContactsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListContactChannelsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListContactChannelsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listContactChannelsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListContactChannels");

            return clientHandler.execute(new ClientExecutionParams<ListContactChannelsRequest, ListContactChannelsResponse>()
                    .withOperationName("ListContactChannels").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listContactChannelsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListContactChannelsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all contact channels for the specified contact.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listContactChannels(software.amazon.awssdk.services.ssmcontacts.model.ListContactChannelsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ssmcontacts.paginators.ListContactChannelsIterable responses = client.listContactChannelsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.ssmcontacts.paginators.ListContactChannelsIterable responses = client
     *             .listContactChannelsPaginator(request);
     *     for (software.amazon.awssdk.services.ssmcontacts.model.ListContactChannelsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ssmcontacts.paginators.ListContactChannelsIterable responses = client.listContactChannelsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listContactChannels(software.amazon.awssdk.services.ssmcontacts.model.ListContactChannelsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listContactChannelsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws DataEncryptionException
     *         The operation failed to due an encryption key error.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.ListContactChannels
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/ListContactChannels"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListContactChannelsIterable listContactChannelsPaginator(ListContactChannelsRequest listContactChannelsRequest)
            throws AccessDeniedException, DataEncryptionException, InternalServerException, ResourceNotFoundException,
            ThrottlingException, ValidationException, AwsServiceException, SdkClientException, SsmContactsException {
        return new ListContactChannelsIterable(this, applyPaginatorUserAgent(listContactChannelsRequest));
    }

    /**
     * <p>
     * Lists all contacts and escalation plans in Incident Manager.
     * </p>
     *
     * @param listContactsRequest
     * @return Result of the ListContacts operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.ListContacts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/ListContacts" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListContactsResponse listContacts(ListContactsRequest listContactsRequest) throws AccessDeniedException,
            InternalServerException, ThrottlingException, ValidationException, AwsServiceException, SdkClientException,
            SsmContactsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListContactsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListContactsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listContactsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListContacts");

            return clientHandler.execute(new ClientExecutionParams<ListContactsRequest, ListContactsResponse>()
                    .withOperationName("ListContacts").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listContactsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListContactsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all contacts and escalation plans in Incident Manager.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listContacts(software.amazon.awssdk.services.ssmcontacts.model.ListContactsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ssmcontacts.paginators.ListContactsIterable responses = client.listContactsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.ssmcontacts.paginators.ListContactsIterable responses = client.listContactsPaginator(request);
     *     for (software.amazon.awssdk.services.ssmcontacts.model.ListContactsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ssmcontacts.paginators.ListContactsIterable responses = client.listContactsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listContacts(software.amazon.awssdk.services.ssmcontacts.model.ListContactsRequest)} operation.</b>
     * </p>
     *
     * @param listContactsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.ListContacts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/ListContacts" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListContactsIterable listContactsPaginator(ListContactsRequest listContactsRequest) throws AccessDeniedException,
            InternalServerException, ThrottlingException, ValidationException, AwsServiceException, SdkClientException,
            SsmContactsException {
        return new ListContactsIterable(this, applyPaginatorUserAgent(listContactsRequest));
    }

    /**
     * <p>
     * Lists all engagements that have happened in an incident.
     * </p>
     *
     * @param listEngagementsRequest
     * @return Result of the ListEngagements operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.ListEngagements
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/ListEngagements" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListEngagementsResponse listEngagements(ListEngagementsRequest listEngagementsRequest) throws AccessDeniedException,
            InternalServerException, ThrottlingException, ValidationException, AwsServiceException, SdkClientException,
            SsmContactsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListEngagementsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListEngagementsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listEngagementsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListEngagements");

            return clientHandler.execute(new ClientExecutionParams<ListEngagementsRequest, ListEngagementsResponse>()
                    .withOperationName("ListEngagements").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listEngagementsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListEngagementsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all engagements that have happened in an incident.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listEngagements(software.amazon.awssdk.services.ssmcontacts.model.ListEngagementsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ssmcontacts.paginators.ListEngagementsIterable responses = client.listEngagementsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.ssmcontacts.paginators.ListEngagementsIterable responses = client
     *             .listEngagementsPaginator(request);
     *     for (software.amazon.awssdk.services.ssmcontacts.model.ListEngagementsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ssmcontacts.paginators.ListEngagementsIterable responses = client.listEngagementsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listEngagements(software.amazon.awssdk.services.ssmcontacts.model.ListEngagementsRequest)} operation.</b>
     * </p>
     *
     * @param listEngagementsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.ListEngagements
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/ListEngagements" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListEngagementsIterable listEngagementsPaginator(ListEngagementsRequest listEngagementsRequest)
            throws AccessDeniedException, InternalServerException, ThrottlingException, ValidationException, AwsServiceException,
            SdkClientException, SsmContactsException {
        return new ListEngagementsIterable(this, applyPaginatorUserAgent(listEngagementsRequest));
    }

    /**
     * <p>
     * Lists all of the engagements to contact channels that have been acknowledged.
     * </p>
     *
     * @param listPageReceiptsRequest
     * @return Result of the ListPageReceipts operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.ListPageReceipts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/ListPageReceipts" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListPageReceiptsResponse listPageReceipts(ListPageReceiptsRequest listPageReceiptsRequest)
            throws AccessDeniedException, InternalServerException, ResourceNotFoundException, ThrottlingException,
            ValidationException, AwsServiceException, SdkClientException, SsmContactsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListPageReceiptsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListPageReceiptsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPageReceiptsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPageReceipts");

            return clientHandler.execute(new ClientExecutionParams<ListPageReceiptsRequest, ListPageReceiptsResponse>()
                    .withOperationName("ListPageReceipts").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listPageReceiptsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListPageReceiptsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all of the engagements to contact channels that have been acknowledged.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listPageReceipts(software.amazon.awssdk.services.ssmcontacts.model.ListPageReceiptsRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ssmcontacts.paginators.ListPageReceiptsIterable responses = client.listPageReceiptsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.ssmcontacts.paginators.ListPageReceiptsIterable responses = client
     *             .listPageReceiptsPaginator(request);
     *     for (software.amazon.awssdk.services.ssmcontacts.model.ListPageReceiptsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ssmcontacts.paginators.ListPageReceiptsIterable responses = client.listPageReceiptsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listPageReceipts(software.amazon.awssdk.services.ssmcontacts.model.ListPageReceiptsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listPageReceiptsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.ListPageReceipts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/ListPageReceipts" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListPageReceiptsIterable listPageReceiptsPaginator(ListPageReceiptsRequest listPageReceiptsRequest)
            throws AccessDeniedException, InternalServerException, ResourceNotFoundException, ThrottlingException,
            ValidationException, AwsServiceException, SdkClientException, SsmContactsException {
        return new ListPageReceiptsIterable(this, applyPaginatorUserAgent(listPageReceiptsRequest));
    }

    /**
     * <p>
     * Lists the engagements to a contact's contact channels.
     * </p>
     *
     * @param listPagesByContactRequest
     * @return Result of the ListPagesByContact operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.ListPagesByContact
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/ListPagesByContact"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListPagesByContactResponse listPagesByContact(ListPagesByContactRequest listPagesByContactRequest)
            throws AccessDeniedException, InternalServerException, ResourceNotFoundException, ThrottlingException,
            ValidationException, AwsServiceException, SdkClientException, SsmContactsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListPagesByContactResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListPagesByContactResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPagesByContactRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPagesByContact");

            return clientHandler.execute(new ClientExecutionParams<ListPagesByContactRequest, ListPagesByContactResponse>()
                    .withOperationName("ListPagesByContact").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listPagesByContactRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListPagesByContactRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the engagements to a contact's contact channels.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listPagesByContact(software.amazon.awssdk.services.ssmcontacts.model.ListPagesByContactRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ssmcontacts.paginators.ListPagesByContactIterable responses = client.listPagesByContactPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.ssmcontacts.paginators.ListPagesByContactIterable responses = client
     *             .listPagesByContactPaginator(request);
     *     for (software.amazon.awssdk.services.ssmcontacts.model.ListPagesByContactResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ssmcontacts.paginators.ListPagesByContactIterable responses = client.listPagesByContactPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listPagesByContact(software.amazon.awssdk.services.ssmcontacts.model.ListPagesByContactRequest)}
     * operation.</b>
     * </p>
     *
     * @param listPagesByContactRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.ListPagesByContact
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/ListPagesByContact"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListPagesByContactIterable listPagesByContactPaginator(ListPagesByContactRequest listPagesByContactRequest)
            throws AccessDeniedException, InternalServerException, ResourceNotFoundException, ThrottlingException,
            ValidationException, AwsServiceException, SdkClientException, SsmContactsException {
        return new ListPagesByContactIterable(this, applyPaginatorUserAgent(listPagesByContactRequest));
    }

    /**
     * <p>
     * Lists the engagements to contact channels that occurred by engaging a contact.
     * </p>
     *
     * @param listPagesByEngagementRequest
     * @return Result of the ListPagesByEngagement operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.ListPagesByEngagement
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/ListPagesByEngagement"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListPagesByEngagementResponse listPagesByEngagement(ListPagesByEngagementRequest listPagesByEngagementRequest)
            throws AccessDeniedException, InternalServerException, ResourceNotFoundException, ThrottlingException,
            ValidationException, AwsServiceException, SdkClientException, SsmContactsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListPagesByEngagementResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListPagesByEngagementResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPagesByEngagementRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPagesByEngagement");

            return clientHandler.execute(new ClientExecutionParams<ListPagesByEngagementRequest, ListPagesByEngagementResponse>()
                    .withOperationName("ListPagesByEngagement").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listPagesByEngagementRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListPagesByEngagementRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the engagements to contact channels that occurred by engaging a contact.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listPagesByEngagement(software.amazon.awssdk.services.ssmcontacts.model.ListPagesByEngagementRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ssmcontacts.paginators.ListPagesByEngagementIterable responses = client.listPagesByEngagementPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.ssmcontacts.paginators.ListPagesByEngagementIterable responses = client
     *             .listPagesByEngagementPaginator(request);
     *     for (software.amazon.awssdk.services.ssmcontacts.model.ListPagesByEngagementResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ssmcontacts.paginators.ListPagesByEngagementIterable responses = client.listPagesByEngagementPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listPagesByEngagement(software.amazon.awssdk.services.ssmcontacts.model.ListPagesByEngagementRequest)}
     * operation.</b>
     * </p>
     *
     * @param listPagesByEngagementRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.ListPagesByEngagement
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/ListPagesByEngagement"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListPagesByEngagementIterable listPagesByEngagementPaginator(ListPagesByEngagementRequest listPagesByEngagementRequest)
            throws AccessDeniedException, InternalServerException, ResourceNotFoundException, ThrottlingException,
            ValidationException, AwsServiceException, SdkClientException, SsmContactsException {
        return new ListPagesByEngagementIterable(this, applyPaginatorUserAgent(listPagesByEngagementRequest));
    }

    /**
     * <p>
     * Lists the tags of an escalation plan or contact.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws AccessDeniedException, ThrottlingException, ResourceNotFoundException, ValidationException,
            InternalServerException, AwsServiceException, SdkClientException, SsmContactsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds a resource policy to the specified contact or escalation plan. The resource policy is used to share the
     * contact or escalation plan using Resource Access Manager (RAM). For more information about cross-account sharing,
     * see <a href="https://docs.aws.amazon.com/incident-manager/latest/userguide/xa.html">Setting up cross-account
     * functionality</a>.
     * </p>
     *
     * @param putContactPolicyRequest
     * @return Result of the PutContactPolicy operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws ConflictException
     *         Updating or deleting a resource causes an inconsistent state.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.PutContactPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/PutContactPolicy" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public PutContactPolicyResponse putContactPolicy(PutContactPolicyRequest putContactPolicyRequest)
            throws AccessDeniedException, ConflictException, ValidationException, ResourceNotFoundException, ThrottlingException,
            InternalServerException, AwsServiceException, SdkClientException, SsmContactsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutContactPolicyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PutContactPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putContactPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutContactPolicy");

            return clientHandler.execute(new ClientExecutionParams<PutContactPolicyRequest, PutContactPolicyResponse>()
                    .withOperationName("PutContactPolicy").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(putContactPolicyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutContactPolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Sends an activation code to a contact channel. The contact can use this code to activate the contact channel in
     * the console or with the <code>ActivateChannel</code> operation. Incident Manager can't engage a contact channel
     * until it has been activated.
     * </p>
     *
     * @param sendActivationCodeRequest
     * @return Result of the SendActivationCode operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws DataEncryptionException
     *         The operation failed to due an encryption key error.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws ServiceQuotaExceededException
     *         Request would cause a service quota to be exceeded.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.SendActivationCode
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/SendActivationCode"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public SendActivationCodeResponse sendActivationCode(SendActivationCodeRequest sendActivationCodeRequest)
            throws AccessDeniedException, DataEncryptionException, InternalServerException, ResourceNotFoundException,
            ServiceQuotaExceededException, ThrottlingException, ValidationException, AwsServiceException, SdkClientException,
            SsmContactsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<SendActivationCodeResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, SendActivationCodeResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, sendActivationCodeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SendActivationCode");

            return clientHandler.execute(new ClientExecutionParams<SendActivationCodeRequest, SendActivationCodeResponse>()
                    .withOperationName("SendActivationCode").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(sendActivationCodeRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new SendActivationCodeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Starts an engagement to a contact or escalation plan. The engagement engages each contact specified in the
     * incident.
     * </p>
     *
     * @param startEngagementRequest
     * @return Result of the StartEngagement operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws DataEncryptionException
     *         The operation failed to due an encryption key error.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.StartEngagement
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/StartEngagement" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public StartEngagementResponse startEngagement(StartEngagementRequest startEngagementRequest) throws AccessDeniedException,
            DataEncryptionException, InternalServerException, ResourceNotFoundException, ThrottlingException,
            ValidationException, AwsServiceException, SdkClientException, SsmContactsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartEngagementResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StartEngagementResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startEngagementRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartEngagement");

            return clientHandler.execute(new ClientExecutionParams<StartEngagementRequest, StartEngagementResponse>()
                    .withOperationName("StartEngagement").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(startEngagementRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartEngagementRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Stops an engagement before it finishes the final stage of the escalation plan or engagement plan. Further
     * contacts aren't engaged.
     * </p>
     *
     * @param stopEngagementRequest
     * @return Result of the StopEngagement operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.StopEngagement
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/StopEngagement" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public StopEngagementResponse stopEngagement(StopEngagementRequest stopEngagementRequest) throws AccessDeniedException,
            ValidationException, ResourceNotFoundException, ThrottlingException, InternalServerException, AwsServiceException,
            SdkClientException, SsmContactsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StopEngagementResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StopEngagementResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopEngagementRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopEngagement");

            return clientHandler.execute(new ClientExecutionParams<StopEngagementRequest, StopEngagementResponse>()
                    .withOperationName("StopEngagement").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(stopEngagementRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StopEngagementRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Tags a contact or escalation plan. You can tag only contacts and escalation plans in the first region of your
     * replication set.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws ServiceQuotaExceededException
     *         Request would cause a service quota to be exceeded.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws AccessDeniedException,
            ThrottlingException, ResourceNotFoundException, InternalServerException, ServiceQuotaExceededException,
            ValidationException, AwsServiceException, SdkClientException, SsmContactsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes tags from the specified resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws AccessDeniedException,
            ThrottlingException, ResourceNotFoundException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, SsmContactsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the contact or escalation plan specified.
     * </p>
     *
     * @param updateContactRequest
     * @return Result of the UpdateContact operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws ServiceQuotaExceededException
     *         Request would cause a service quota to be exceeded.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws DataEncryptionException
     *         The operation failed to due an encryption key error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.UpdateContact
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/UpdateContact" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateContactResponse updateContact(UpdateContactRequest updateContactRequest) throws AccessDeniedException,
            InternalServerException, ResourceNotFoundException, ServiceQuotaExceededException, ThrottlingException,
            ValidationException, DataEncryptionException, AwsServiceException, SdkClientException, SsmContactsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateContactResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateContactResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateContactRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateContact");

            return clientHandler.execute(new ClientExecutionParams<UpdateContactRequest, UpdateContactResponse>()
                    .withOperationName("UpdateContact").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateContactRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateContactRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates a contact's contact channel.
     * </p>
     *
     * @param updateContactChannelRequest
     * @return Result of the UpdateContactChannel operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws ConflictException
     *         Updating or deleting a resource causes an inconsistent state.
     * @throws DataEncryptionException
     *         The operation failed to due an encryption key error.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.UpdateContactChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/UpdateContactChannel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateContactChannelResponse updateContactChannel(UpdateContactChannelRequest updateContactChannelRequest)
            throws AccessDeniedException, ConflictException, DataEncryptionException, InternalServerException,
            ResourceNotFoundException, ThrottlingException, ValidationException, AwsServiceException, SdkClientException,
            SsmContactsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateContactChannelResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateContactChannelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateContactChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateContactChannel");

            return clientHandler.execute(new ClientExecutionParams<UpdateContactChannelRequest, UpdateContactChannelResponse>()
                    .withOperationName("UpdateContactChannel").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateContactChannelRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateContactChannelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private <T extends SsmContactsRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(SsmContactsException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DataEncryptionException")
                                .exceptionBuilderSupplier(DataEncryptionException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
