/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssmquicksetup;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.ssmquicksetup.internal.SsmQuickSetupServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.ssmquicksetup.model.AccessDeniedException;
import software.amazon.awssdk.services.ssmquicksetup.model.ConflictException;
import software.amazon.awssdk.services.ssmquicksetup.model.CreateConfigurationManagerRequest;
import software.amazon.awssdk.services.ssmquicksetup.model.CreateConfigurationManagerResponse;
import software.amazon.awssdk.services.ssmquicksetup.model.DeleteConfigurationManagerRequest;
import software.amazon.awssdk.services.ssmquicksetup.model.DeleteConfigurationManagerResponse;
import software.amazon.awssdk.services.ssmquicksetup.model.GetConfigurationManagerRequest;
import software.amazon.awssdk.services.ssmquicksetup.model.GetConfigurationManagerResponse;
import software.amazon.awssdk.services.ssmquicksetup.model.GetServiceSettingsRequest;
import software.amazon.awssdk.services.ssmquicksetup.model.GetServiceSettingsResponse;
import software.amazon.awssdk.services.ssmquicksetup.model.InternalServerException;
import software.amazon.awssdk.services.ssmquicksetup.model.ListConfigurationManagersRequest;
import software.amazon.awssdk.services.ssmquicksetup.model.ListConfigurationManagersResponse;
import software.amazon.awssdk.services.ssmquicksetup.model.ListQuickSetupTypesRequest;
import software.amazon.awssdk.services.ssmquicksetup.model.ListQuickSetupTypesResponse;
import software.amazon.awssdk.services.ssmquicksetup.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.ssmquicksetup.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.ssmquicksetup.model.ResourceNotFoundException;
import software.amazon.awssdk.services.ssmquicksetup.model.SsmQuickSetupException;
import software.amazon.awssdk.services.ssmquicksetup.model.TagResourceRequest;
import software.amazon.awssdk.services.ssmquicksetup.model.TagResourceResponse;
import software.amazon.awssdk.services.ssmquicksetup.model.ThrottlingException;
import software.amazon.awssdk.services.ssmquicksetup.model.UntagResourceRequest;
import software.amazon.awssdk.services.ssmquicksetup.model.UntagResourceResponse;
import software.amazon.awssdk.services.ssmquicksetup.model.UpdateConfigurationDefinitionRequest;
import software.amazon.awssdk.services.ssmquicksetup.model.UpdateConfigurationDefinitionResponse;
import software.amazon.awssdk.services.ssmquicksetup.model.UpdateConfigurationManagerRequest;
import software.amazon.awssdk.services.ssmquicksetup.model.UpdateConfigurationManagerResponse;
import software.amazon.awssdk.services.ssmquicksetup.model.UpdateServiceSettingsRequest;
import software.amazon.awssdk.services.ssmquicksetup.model.UpdateServiceSettingsResponse;
import software.amazon.awssdk.services.ssmquicksetup.model.ValidationException;
import software.amazon.awssdk.services.ssmquicksetup.transform.CreateConfigurationManagerRequestMarshaller;
import software.amazon.awssdk.services.ssmquicksetup.transform.DeleteConfigurationManagerRequestMarshaller;
import software.amazon.awssdk.services.ssmquicksetup.transform.GetConfigurationManagerRequestMarshaller;
import software.amazon.awssdk.services.ssmquicksetup.transform.GetServiceSettingsRequestMarshaller;
import software.amazon.awssdk.services.ssmquicksetup.transform.ListConfigurationManagersRequestMarshaller;
import software.amazon.awssdk.services.ssmquicksetup.transform.ListQuickSetupTypesRequestMarshaller;
import software.amazon.awssdk.services.ssmquicksetup.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.ssmquicksetup.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.ssmquicksetup.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.ssmquicksetup.transform.UpdateConfigurationDefinitionRequestMarshaller;
import software.amazon.awssdk.services.ssmquicksetup.transform.UpdateConfigurationManagerRequestMarshaller;
import software.amazon.awssdk.services.ssmquicksetup.transform.UpdateServiceSettingsRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link SsmQuickSetupAsyncClient}.
 *
 * @see SsmQuickSetupAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultSsmQuickSetupAsyncClient implements SsmQuickSetupAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultSsmQuickSetupAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultSsmQuickSetupAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a Quick Setup configuration manager resource. This object is a collection of desired state configurations
     * for multiple configuration definitions and summaries describing the deployments of those definitions.
     * </p>
     *
     * @param createConfigurationManagerRequest
     * @return A Java Future containing the result of the CreateConfigurationManager operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The request is invalid. Verify the values provided for the request parameters are
     *         accurate.</li>
     *         <li>ConflictException Another request is being processed. Wait a few minutes and try again.</li>
     *         <li>InternalServerException An error occurred on the server side.</li>
     *         <li>AccessDeniedException The requester has insufficient permissions to perform the operation.</li>
     *         <li>ThrottlingException The request or operation exceeds the maximum allowed request rate per Amazon Web
     *         Services account and Amazon Web Services Region.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmQuickSetupException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmQuickSetupAsyncClient.CreateConfigurationManager
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-quicksetup-2018-05-10/CreateConfigurationManager"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateConfigurationManagerResponse> createConfigurationManager(
            CreateConfigurationManagerRequest createConfigurationManagerRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createConfigurationManagerRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createConfigurationManagerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM QuickSetup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateConfigurationManager");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateConfigurationManagerResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateConfigurationManagerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateConfigurationManagerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateConfigurationManagerRequest, CreateConfigurationManagerResponse>()
                            .withOperationName("CreateConfigurationManager").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateConfigurationManagerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createConfigurationManagerRequest));
            CompletableFuture<CreateConfigurationManagerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a configuration manager.
     * </p>
     *
     * @param deleteConfigurationManagerRequest
     * @return A Java Future containing the result of the DeleteConfigurationManager operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The request is invalid. Verify the values provided for the request parameters are
     *         accurate.</li>
     *         <li>ConflictException Another request is being processed. Wait a few minutes and try again.</li>
     *         <li>InternalServerException An error occurred on the server side.</li>
     *         <li>AccessDeniedException The requester has insufficient permissions to perform the operation.</li>
     *         <li>ThrottlingException The request or operation exceeds the maximum allowed request rate per Amazon Web
     *         Services account and Amazon Web Services Region.</li>
     *         <li>ResourceNotFoundException The resource couldn't be found. Check the ID or name and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmQuickSetupException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmQuickSetupAsyncClient.DeleteConfigurationManager
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-quicksetup-2018-05-10/DeleteConfigurationManager"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteConfigurationManagerResponse> deleteConfigurationManager(
            DeleteConfigurationManagerRequest deleteConfigurationManagerRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteConfigurationManagerRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteConfigurationManagerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM QuickSetup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteConfigurationManager");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteConfigurationManagerResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteConfigurationManagerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteConfigurationManagerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteConfigurationManagerRequest, DeleteConfigurationManagerResponse>()
                            .withOperationName("DeleteConfigurationManager").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteConfigurationManagerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteConfigurationManagerRequest));
            CompletableFuture<DeleteConfigurationManagerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a configuration manager.
     * </p>
     *
     * @param getConfigurationManagerRequest
     * @return A Java Future containing the result of the GetConfigurationManager operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The request is invalid. Verify the values provided for the request parameters are
     *         accurate.</li>
     *         <li>ConflictException Another request is being processed. Wait a few minutes and try again.</li>
     *         <li>InternalServerException An error occurred on the server side.</li>
     *         <li>AccessDeniedException The requester has insufficient permissions to perform the operation.</li>
     *         <li>ThrottlingException The request or operation exceeds the maximum allowed request rate per Amazon Web
     *         Services account and Amazon Web Services Region.</li>
     *         <li>ResourceNotFoundException The resource couldn't be found. Check the ID or name and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmQuickSetupException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmQuickSetupAsyncClient.GetConfigurationManager
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-quicksetup-2018-05-10/GetConfigurationManager"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetConfigurationManagerResponse> getConfigurationManager(
            GetConfigurationManagerRequest getConfigurationManagerRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getConfigurationManagerRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getConfigurationManagerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM QuickSetup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetConfigurationManager");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetConfigurationManagerResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetConfigurationManagerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetConfigurationManagerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetConfigurationManagerRequest, GetConfigurationManagerResponse>()
                            .withOperationName("GetConfigurationManager").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetConfigurationManagerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getConfigurationManagerRequest));
            CompletableFuture<GetConfigurationManagerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns settings configured for Quick Setup in the requesting Amazon Web Services account and Amazon Web Services
     * Region.
     * </p>
     *
     * @param getServiceSettingsRequest
     * @return A Java Future containing the result of the GetServiceSettings operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ConflictException Another request is being processed. Wait a few minutes and try again.</li>
     *         <li>InternalServerException An error occurred on the server side.</li>
     *         <li>AccessDeniedException The requester has insufficient permissions to perform the operation.</li>
     *         <li>ThrottlingException The request or operation exceeds the maximum allowed request rate per Amazon Web
     *         Services account and Amazon Web Services Region.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmQuickSetupException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmQuickSetupAsyncClient.GetServiceSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-quicksetup-2018-05-10/GetServiceSettings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetServiceSettingsResponse> getServiceSettings(GetServiceSettingsRequest getServiceSettingsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getServiceSettingsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getServiceSettingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM QuickSetup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetServiceSettings");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetServiceSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetServiceSettingsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetServiceSettingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetServiceSettingsRequest, GetServiceSettingsResponse>()
                            .withOperationName("GetServiceSettings").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetServiceSettingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getServiceSettingsRequest));
            CompletableFuture<GetServiceSettingsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns Quick Setup configuration managers.
     * </p>
     *
     * @param listConfigurationManagersRequest
     * @return A Java Future containing the result of the ListConfigurationManagers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The request is invalid. Verify the values provided for the request parameters are
     *         accurate.</li>
     *         <li>ConflictException Another request is being processed. Wait a few minutes and try again.</li>
     *         <li>InternalServerException An error occurred on the server side.</li>
     *         <li>AccessDeniedException The requester has insufficient permissions to perform the operation.</li>
     *         <li>ThrottlingException The request or operation exceeds the maximum allowed request rate per Amazon Web
     *         Services account and Amazon Web Services Region.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmQuickSetupException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmQuickSetupAsyncClient.ListConfigurationManagers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-quicksetup-2018-05-10/ListConfigurationManagers"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListConfigurationManagersResponse> listConfigurationManagers(
            ListConfigurationManagersRequest listConfigurationManagersRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listConfigurationManagersRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listConfigurationManagersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM QuickSetup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListConfigurationManagers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListConfigurationManagersResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListConfigurationManagersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListConfigurationManagersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListConfigurationManagersRequest, ListConfigurationManagersResponse>()
                            .withOperationName("ListConfigurationManagers").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListConfigurationManagersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listConfigurationManagersRequest));
            CompletableFuture<ListConfigurationManagersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the available Quick Setup types.
     * </p>
     *
     * @param listQuickSetupTypesRequest
     * @return A Java Future containing the result of the ListQuickSetupTypes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ConflictException Another request is being processed. Wait a few minutes and try again.</li>
     *         <li>InternalServerException An error occurred on the server side.</li>
     *         <li>AccessDeniedException The requester has insufficient permissions to perform the operation.</li>
     *         <li>ThrottlingException The request or operation exceeds the maximum allowed request rate per Amazon Web
     *         Services account and Amazon Web Services Region.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmQuickSetupException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmQuickSetupAsyncClient.ListQuickSetupTypes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-quicksetup-2018-05-10/ListQuickSetupTypes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListQuickSetupTypesResponse> listQuickSetupTypes(
            ListQuickSetupTypesRequest listQuickSetupTypesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listQuickSetupTypesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listQuickSetupTypesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM QuickSetup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListQuickSetupTypes");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListQuickSetupTypesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListQuickSetupTypesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListQuickSetupTypesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListQuickSetupTypesRequest, ListQuickSetupTypesResponse>()
                            .withOperationName("ListQuickSetupTypes").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListQuickSetupTypesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listQuickSetupTypesRequest));
            CompletableFuture<ListQuickSetupTypesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns tags assigned to the resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The request is invalid. Verify the values provided for the request parameters are
     *         accurate.</li>
     *         <li>ConflictException Another request is being processed. Wait a few minutes and try again.</li>
     *         <li>InternalServerException An error occurred on the server side.</li>
     *         <li>AccessDeniedException The requester has insufficient permissions to perform the operation.</li>
     *         <li>ThrottlingException The request or operation exceeds the maximum allowed request rate per Amazon Web
     *         Services account and Amazon Web Services Region.</li>
     *         <li>ResourceNotFoundException The resource couldn't be found. Check the ID or name and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmQuickSetupException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmQuickSetupAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-quicksetup-2018-05-10/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM QuickSetup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Assigns key-value pairs of metadata to Amazon Web Services resources.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The request is invalid. Verify the values provided for the request parameters are
     *         accurate.</li>
     *         <li>ConflictException Another request is being processed. Wait a few minutes and try again.</li>
     *         <li>InternalServerException An error occurred on the server side.</li>
     *         <li>AccessDeniedException The requester has insufficient permissions to perform the operation.</li>
     *         <li>ThrottlingException The request or operation exceeds the maximum allowed request rate per Amazon Web
     *         Services account and Amazon Web Services Region.</li>
     *         <li>ResourceNotFoundException The resource couldn't be found. Check the ID or name and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmQuickSetupException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmQuickSetupAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-quicksetup-2018-05-10/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM QuickSetup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes tags from the specified resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The request is invalid. Verify the values provided for the request parameters are
     *         accurate.</li>
     *         <li>ConflictException Another request is being processed. Wait a few minutes and try again.</li>
     *         <li>InternalServerException An error occurred on the server side.</li>
     *         <li>AccessDeniedException The requester has insufficient permissions to perform the operation.</li>
     *         <li>ThrottlingException The request or operation exceeds the maximum allowed request rate per Amazon Web
     *         Services account and Amazon Web Services Region.</li>
     *         <li>ResourceNotFoundException The resource couldn't be found. Check the ID or name and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmQuickSetupException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmQuickSetupAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-quicksetup-2018-05-10/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM QuickSetup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a Quick Setup configuration definition.
     * </p>
     *
     * @param updateConfigurationDefinitionRequest
     * @return A Java Future containing the result of the UpdateConfigurationDefinition operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The request is invalid. Verify the values provided for the request parameters are
     *         accurate.</li>
     *         <li>ConflictException Another request is being processed. Wait a few minutes and try again.</li>
     *         <li>InternalServerException An error occurred on the server side.</li>
     *         <li>AccessDeniedException The requester has insufficient permissions to perform the operation.</li>
     *         <li>ThrottlingException The request or operation exceeds the maximum allowed request rate per Amazon Web
     *         Services account and Amazon Web Services Region.</li>
     *         <li>ResourceNotFoundException The resource couldn't be found. Check the ID or name and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmQuickSetupException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmQuickSetupAsyncClient.UpdateConfigurationDefinition
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-quicksetup-2018-05-10/UpdateConfigurationDefinition"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateConfigurationDefinitionResponse> updateConfigurationDefinition(
            UpdateConfigurationDefinitionRequest updateConfigurationDefinitionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateConfigurationDefinitionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateConfigurationDefinitionRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM QuickSetup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateConfigurationDefinition");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateConfigurationDefinitionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateConfigurationDefinitionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateConfigurationDefinitionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateConfigurationDefinitionRequest, UpdateConfigurationDefinitionResponse>()
                            .withOperationName("UpdateConfigurationDefinition").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateConfigurationDefinitionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateConfigurationDefinitionRequest));
            CompletableFuture<UpdateConfigurationDefinitionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a Quick Setup configuration manager.
     * </p>
     *
     * @param updateConfigurationManagerRequest
     * @return A Java Future containing the result of the UpdateConfigurationManager operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The request is invalid. Verify the values provided for the request parameters are
     *         accurate.</li>
     *         <li>ConflictException Another request is being processed. Wait a few minutes and try again.</li>
     *         <li>InternalServerException An error occurred on the server side.</li>
     *         <li>AccessDeniedException The requester has insufficient permissions to perform the operation.</li>
     *         <li>ThrottlingException The request or operation exceeds the maximum allowed request rate per Amazon Web
     *         Services account and Amazon Web Services Region.</li>
     *         <li>ResourceNotFoundException The resource couldn't be found. Check the ID or name and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmQuickSetupException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmQuickSetupAsyncClient.UpdateConfigurationManager
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-quicksetup-2018-05-10/UpdateConfigurationManager"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateConfigurationManagerResponse> updateConfigurationManager(
            UpdateConfigurationManagerRequest updateConfigurationManagerRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateConfigurationManagerRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateConfigurationManagerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM QuickSetup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateConfigurationManager");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateConfigurationManagerResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateConfigurationManagerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateConfigurationManagerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateConfigurationManagerRequest, UpdateConfigurationManagerResponse>()
                            .withOperationName("UpdateConfigurationManager").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateConfigurationManagerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateConfigurationManagerRequest));
            CompletableFuture<UpdateConfigurationManagerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates settings configured for Quick Setup.
     * </p>
     *
     * @param updateServiceSettingsRequest
     * @return A Java Future containing the result of the UpdateServiceSettings operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The request is invalid. Verify the values provided for the request parameters are
     *         accurate.</li>
     *         <li>ConflictException Another request is being processed. Wait a few minutes and try again.</li>
     *         <li>InternalServerException An error occurred on the server side.</li>
     *         <li>AccessDeniedException The requester has insufficient permissions to perform the operation.</li>
     *         <li>ThrottlingException The request or operation exceeds the maximum allowed request rate per Amazon Web
     *         Services account and Amazon Web Services Region.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmQuickSetupException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmQuickSetupAsyncClient.UpdateServiceSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-quicksetup-2018-05-10/UpdateServiceSettings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateServiceSettingsResponse> updateServiceSettings(
            UpdateServiceSettingsRequest updateServiceSettingsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateServiceSettingsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateServiceSettingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM QuickSetup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateServiceSettings");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateServiceSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateServiceSettingsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateServiceSettingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateServiceSettingsRequest, UpdateServiceSettingsResponse>()
                            .withOperationName("UpdateServiceSettings").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateServiceSettingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateServiceSettingsRequest));
            CompletableFuture<UpdateServiceSettingsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final SsmQuickSetupServiceClientConfiguration serviceClientConfiguration() {
        return new SsmQuickSetupServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(SsmQuickSetupException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        SsmQuickSetupServiceClientConfigurationBuilder serviceConfigBuilder = new SsmQuickSetupServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
