/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssmsap.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The SAP component of your application.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Component implements SdkPojo, Serializable, ToCopyableBuilder<Component.Builder, Component> {
    private static final SdkField<String> COMPONENT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ComponentId").getter(getter(Component::componentId)).setter(setter(Builder::componentId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ComponentId").build()).build();

    private static final SdkField<String> APPLICATION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ApplicationId").getter(getter(Component::applicationId)).setter(setter(Builder::applicationId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ApplicationId").build()).build();

    private static final SdkField<String> COMPONENT_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ComponentType").getter(getter(Component::componentTypeAsString)).setter(setter(Builder::componentType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ComponentType").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(Component::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<List<String>> DATABASES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Databases")
            .getter(getter(Component::databases))
            .setter(setter(Builder::databases))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Databases").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<Host>> HOSTS_FIELD = SdkField
            .<List<Host>> builder(MarshallingType.LIST)
            .memberName("Hosts")
            .getter(getter(Component::hosts))
            .setter(setter(Builder::hosts))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Hosts").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Host> builder(MarshallingType.SDK_POJO)
                                            .constructor(Host::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> PRIMARY_HOST_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PrimaryHost").getter(getter(Component::primaryHost)).setter(setter(Builder::primaryHost))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PrimaryHost").build()).build();

    private static final SdkField<Instant> LAST_UPDATED_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastUpdated").getter(getter(Component::lastUpdated)).setter(setter(Builder::lastUpdated))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastUpdated").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(COMPONENT_ID_FIELD,
            APPLICATION_ID_FIELD, COMPONENT_TYPE_FIELD, STATUS_FIELD, DATABASES_FIELD, HOSTS_FIELD, PRIMARY_HOST_FIELD,
            LAST_UPDATED_FIELD));

    private static final long serialVersionUID = 1L;

    private final String componentId;

    private final String applicationId;

    private final String componentType;

    private final String status;

    private final List<String> databases;

    private final List<Host> hosts;

    private final String primaryHost;

    private final Instant lastUpdated;

    private Component(BuilderImpl builder) {
        this.componentId = builder.componentId;
        this.applicationId = builder.applicationId;
        this.componentType = builder.componentType;
        this.status = builder.status;
        this.databases = builder.databases;
        this.hosts = builder.hosts;
        this.primaryHost = builder.primaryHost;
        this.lastUpdated = builder.lastUpdated;
    }

    /**
     * <p>
     * The ID of the component.
     * </p>
     * 
     * @return The ID of the component.
     */
    public final String componentId() {
        return componentId;
    }

    /**
     * <p>
     * The ID of the application.
     * </p>
     * 
     * @return The ID of the application.
     */
    public final String applicationId() {
        return applicationId;
    }

    /**
     * <p>
     * The type of the component.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #componentType}
     * will return {@link ComponentType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #componentTypeAsString}.
     * </p>
     * 
     * @return The type of the component.
     * @see ComponentType
     */
    public final ComponentType componentType() {
        return ComponentType.fromValue(componentType);
    }

    /**
     * <p>
     * The type of the component.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #componentType}
     * will return {@link ComponentType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #componentTypeAsString}.
     * </p>
     * 
     * @return The type of the component.
     * @see ComponentType
     */
    public final String componentTypeAsString() {
        return componentType;
    }

    /**
     * <p>
     * The status of the component.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ComponentStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the component.
     * @see ComponentStatus
     */
    public final ComponentStatus status() {
        return ComponentStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of the component.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ComponentStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the component.
     * @see ComponentStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * For responses, this returns true if the service returned a value for the Databases property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasDatabases() {
        return databases != null && !(databases instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The SAP HANA databases of the component.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDatabases} method.
     * </p>
     * 
     * @return The SAP HANA databases of the component.
     */
    public final List<String> databases() {
        return databases;
    }

    /**
     * For responses, this returns true if the service returned a value for the Hosts property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasHosts() {
        return hosts != null && !(hosts instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The hosts of the component.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasHosts} method.
     * </p>
     * 
     * @return The hosts of the component.
     */
    public final List<Host> hosts() {
        return hosts;
    }

    /**
     * <p>
     * The primary host of the component.
     * </p>
     * 
     * @return The primary host of the component.
     */
    public final String primaryHost() {
        return primaryHost;
    }

    /**
     * <p>
     * The time at which the component was last updated.
     * </p>
     * 
     * @return The time at which the component was last updated.
     */
    public final Instant lastUpdated() {
        return lastUpdated;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(componentId());
        hashCode = 31 * hashCode + Objects.hashCode(applicationId());
        hashCode = 31 * hashCode + Objects.hashCode(componentTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasDatabases() ? databases() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasHosts() ? hosts() : null);
        hashCode = 31 * hashCode + Objects.hashCode(primaryHost());
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdated());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Component)) {
            return false;
        }
        Component other = (Component) obj;
        return Objects.equals(componentId(), other.componentId()) && Objects.equals(applicationId(), other.applicationId())
                && Objects.equals(componentTypeAsString(), other.componentTypeAsString())
                && Objects.equals(statusAsString(), other.statusAsString()) && hasDatabases() == other.hasDatabases()
                && Objects.equals(databases(), other.databases()) && hasHosts() == other.hasHosts()
                && Objects.equals(hosts(), other.hosts()) && Objects.equals(primaryHost(), other.primaryHost())
                && Objects.equals(lastUpdated(), other.lastUpdated());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Component").add("ComponentId", componentId()).add("ApplicationId", applicationId())
                .add("ComponentType", componentTypeAsString()).add("Status", statusAsString())
                .add("Databases", hasDatabases() ? databases() : null).add("Hosts", hasHosts() ? hosts() : null)
                .add("PrimaryHost", primaryHost()).add("LastUpdated", lastUpdated()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ComponentId":
            return Optional.ofNullable(clazz.cast(componentId()));
        case "ApplicationId":
            return Optional.ofNullable(clazz.cast(applicationId()));
        case "ComponentType":
            return Optional.ofNullable(clazz.cast(componentTypeAsString()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "Databases":
            return Optional.ofNullable(clazz.cast(databases()));
        case "Hosts":
            return Optional.ofNullable(clazz.cast(hosts()));
        case "PrimaryHost":
            return Optional.ofNullable(clazz.cast(primaryHost()));
        case "LastUpdated":
            return Optional.ofNullable(clazz.cast(lastUpdated()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Component, T> g) {
        return obj -> g.apply((Component) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Component> {
        /**
         * <p>
         * The ID of the component.
         * </p>
         * 
         * @param componentId
         *        The ID of the component.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder componentId(String componentId);

        /**
         * <p>
         * The ID of the application.
         * </p>
         * 
         * @param applicationId
         *        The ID of the application.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder applicationId(String applicationId);

        /**
         * <p>
         * The type of the component.
         * </p>
         * 
         * @param componentType
         *        The type of the component.
         * @see ComponentType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ComponentType
         */
        Builder componentType(String componentType);

        /**
         * <p>
         * The type of the component.
         * </p>
         * 
         * @param componentType
         *        The type of the component.
         * @see ComponentType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ComponentType
         */
        Builder componentType(ComponentType componentType);

        /**
         * <p>
         * The status of the component.
         * </p>
         * 
         * @param status
         *        The status of the component.
         * @see ComponentStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ComponentStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the component.
         * </p>
         * 
         * @param status
         *        The status of the component.
         * @see ComponentStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ComponentStatus
         */
        Builder status(ComponentStatus status);

        /**
         * <p>
         * The SAP HANA databases of the component.
         * </p>
         * 
         * @param databases
         *        The SAP HANA databases of the component.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder databases(Collection<String> databases);

        /**
         * <p>
         * The SAP HANA databases of the component.
         * </p>
         * 
         * @param databases
         *        The SAP HANA databases of the component.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder databases(String... databases);

        /**
         * <p>
         * The hosts of the component.
         * </p>
         * 
         * @param hosts
         *        The hosts of the component.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hosts(Collection<Host> hosts);

        /**
         * <p>
         * The hosts of the component.
         * </p>
         * 
         * @param hosts
         *        The hosts of the component.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hosts(Host... hosts);

        /**
         * <p>
         * The hosts of the component.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ssmsap.model.Host.Builder} avoiding the need to create one manually
         * via {@link software.amazon.awssdk.services.ssmsap.model.Host#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ssmsap.model.Host.Builder#build()} is called immediately and its
         * result is passed to {@link #hosts(List<Host>)}.
         * 
         * @param hosts
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.ssmsap.model.Host.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #hosts(java.util.Collection<Host>)
         */
        Builder hosts(Consumer<Host.Builder>... hosts);

        /**
         * <p>
         * The primary host of the component.
         * </p>
         * 
         * @param primaryHost
         *        The primary host of the component.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder primaryHost(String primaryHost);

        /**
         * <p>
         * The time at which the component was last updated.
         * </p>
         * 
         * @param lastUpdated
         *        The time at which the component was last updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdated(Instant lastUpdated);
    }

    static final class BuilderImpl implements Builder {
        private String componentId;

        private String applicationId;

        private String componentType;

        private String status;

        private List<String> databases = DefaultSdkAutoConstructList.getInstance();

        private List<Host> hosts = DefaultSdkAutoConstructList.getInstance();

        private String primaryHost;

        private Instant lastUpdated;

        private BuilderImpl() {
        }

        private BuilderImpl(Component model) {
            componentId(model.componentId);
            applicationId(model.applicationId);
            componentType(model.componentType);
            status(model.status);
            databases(model.databases);
            hosts(model.hosts);
            primaryHost(model.primaryHost);
            lastUpdated(model.lastUpdated);
        }

        public final String getComponentId() {
            return componentId;
        }

        public final void setComponentId(String componentId) {
            this.componentId = componentId;
        }

        @Override
        public final Builder componentId(String componentId) {
            this.componentId = componentId;
            return this;
        }

        public final String getApplicationId() {
            return applicationId;
        }

        public final void setApplicationId(String applicationId) {
            this.applicationId = applicationId;
        }

        @Override
        public final Builder applicationId(String applicationId) {
            this.applicationId = applicationId;
            return this;
        }

        public final String getComponentType() {
            return componentType;
        }

        public final void setComponentType(String componentType) {
            this.componentType = componentType;
        }

        @Override
        public final Builder componentType(String componentType) {
            this.componentType = componentType;
            return this;
        }

        @Override
        public final Builder componentType(ComponentType componentType) {
            this.componentType(componentType == null ? null : componentType.toString());
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(ComponentStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final Collection<String> getDatabases() {
            if (databases instanceof SdkAutoConstructList) {
                return null;
            }
            return databases;
        }

        public final void setDatabases(Collection<String> databases) {
            this.databases = DatabaseIdListCopier.copy(databases);
        }

        @Override
        public final Builder databases(Collection<String> databases) {
            this.databases = DatabaseIdListCopier.copy(databases);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder databases(String... databases) {
            databases(Arrays.asList(databases));
            return this;
        }

        public final List<Host.Builder> getHosts() {
            List<Host.Builder> result = HostListCopier.copyToBuilder(this.hosts);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setHosts(Collection<Host.BuilderImpl> hosts) {
            this.hosts = HostListCopier.copyFromBuilder(hosts);
        }

        @Override
        public final Builder hosts(Collection<Host> hosts) {
            this.hosts = HostListCopier.copy(hosts);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder hosts(Host... hosts) {
            hosts(Arrays.asList(hosts));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder hosts(Consumer<Host.Builder>... hosts) {
            hosts(Stream.of(hosts).map(c -> Host.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getPrimaryHost() {
            return primaryHost;
        }

        public final void setPrimaryHost(String primaryHost) {
            this.primaryHost = primaryHost;
        }

        @Override
        public final Builder primaryHost(String primaryHost) {
            this.primaryHost = primaryHost;
            return this;
        }

        public final Instant getLastUpdated() {
            return lastUpdated;
        }

        public final void setLastUpdated(Instant lastUpdated) {
            this.lastUpdated = lastUpdated;
        }

        @Override
        public final Builder lastUpdated(Instant lastUpdated) {
            this.lastUpdated = lastUpdated;
            return this;
        }

        @Override
        public Component build() {
            return new Component(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
