/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssmsap.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The SAP component of your application.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Component implements SdkPojo, Serializable, ToCopyableBuilder<Component.Builder, Component> {
    private static final SdkField<String> COMPONENT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ComponentId").getter(getter(Component::componentId)).setter(setter(Builder::componentId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ComponentId").build()).build();

    private static final SdkField<String> SID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Sid")
            .getter(getter(Component::sid)).setter(setter(Builder::sid))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Sid").build()).build();

    private static final SdkField<String> SYSTEM_NUMBER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SystemNumber").getter(getter(Component::systemNumber)).setter(setter(Builder::systemNumber))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SystemNumber").build()).build();

    private static final SdkField<String> PARENT_COMPONENT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ParentComponent").getter(getter(Component::parentComponent)).setter(setter(Builder::parentComponent))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ParentComponent").build()).build();

    private static final SdkField<List<String>> CHILD_COMPONENTS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ChildComponents")
            .getter(getter(Component::childComponents))
            .setter(setter(Builder::childComponents))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ChildComponents").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> APPLICATION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ApplicationId").getter(getter(Component::applicationId)).setter(setter(Builder::applicationId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ApplicationId").build()).build();

    private static final SdkField<String> COMPONENT_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ComponentType").getter(getter(Component::componentTypeAsString)).setter(setter(Builder::componentType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ComponentType").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(Component::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<String> SAP_HOSTNAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SapHostname").getter(getter(Component::sapHostname)).setter(setter(Builder::sapHostname))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SapHostname").build()).build();

    private static final SdkField<String> SAP_FEATURE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SapFeature").getter(getter(Component::sapFeature)).setter(setter(Builder::sapFeature))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SapFeature").build()).build();

    private static final SdkField<String> SAP_KERNEL_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SapKernelVersion").getter(getter(Component::sapKernelVersion)).setter(setter(Builder::sapKernelVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SapKernelVersion").build()).build();

    private static final SdkField<String> HDB_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("HdbVersion").getter(getter(Component::hdbVersion)).setter(setter(Builder::hdbVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HdbVersion").build()).build();

    private static final SdkField<Resilience> RESILIENCE_FIELD = SdkField.<Resilience> builder(MarshallingType.SDK_POJO)
            .memberName("Resilience").getter(getter(Component::resilience)).setter(setter(Builder::resilience))
            .constructor(Resilience::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Resilience").build()).build();

    private static final SdkField<AssociatedHost> ASSOCIATED_HOST_FIELD = SdkField
            .<AssociatedHost> builder(MarshallingType.SDK_POJO).memberName("AssociatedHost")
            .getter(getter(Component::associatedHost)).setter(setter(Builder::associatedHost))
            .constructor(AssociatedHost::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AssociatedHost").build()).build();

    private static final SdkField<List<String>> DATABASES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Databases")
            .getter(getter(Component::databases))
            .setter(setter(Builder::databases))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Databases").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<Host>> HOSTS_FIELD = SdkField
            .<List<Host>> builder(MarshallingType.LIST)
            .memberName("Hosts")
            .getter(getter(Component::hosts))
            .setter(setter(Builder::hosts))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Hosts").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Host> builder(MarshallingType.SDK_POJO)
                                            .constructor(Host::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> PRIMARY_HOST_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PrimaryHost").getter(getter(Component::primaryHost)).setter(setter(Builder::primaryHost))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PrimaryHost").build()).build();

    private static final SdkField<DatabaseConnection> DATABASE_CONNECTION_FIELD = SdkField
            .<DatabaseConnection> builder(MarshallingType.SDK_POJO).memberName("DatabaseConnection")
            .getter(getter(Component::databaseConnection)).setter(setter(Builder::databaseConnection))
            .constructor(DatabaseConnection::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DatabaseConnection").build())
            .build();

    private static final SdkField<Instant> LAST_UPDATED_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastUpdated").getter(getter(Component::lastUpdated)).setter(setter(Builder::lastUpdated))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastUpdated").build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Arn")
            .getter(getter(Component::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Arn").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(COMPONENT_ID_FIELD, SID_FIELD,
            SYSTEM_NUMBER_FIELD, PARENT_COMPONENT_FIELD, CHILD_COMPONENTS_FIELD, APPLICATION_ID_FIELD, COMPONENT_TYPE_FIELD,
            STATUS_FIELD, SAP_HOSTNAME_FIELD, SAP_FEATURE_FIELD, SAP_KERNEL_VERSION_FIELD, HDB_VERSION_FIELD, RESILIENCE_FIELD,
            ASSOCIATED_HOST_FIELD, DATABASES_FIELD, HOSTS_FIELD, PRIMARY_HOST_FIELD, DATABASE_CONNECTION_FIELD,
            LAST_UPDATED_FIELD, ARN_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String componentId;

    private final String sid;

    private final String systemNumber;

    private final String parentComponent;

    private final List<String> childComponents;

    private final String applicationId;

    private final String componentType;

    private final String status;

    private final String sapHostname;

    private final String sapFeature;

    private final String sapKernelVersion;

    private final String hdbVersion;

    private final Resilience resilience;

    private final AssociatedHost associatedHost;

    private final List<String> databases;

    private final List<Host> hosts;

    private final String primaryHost;

    private final DatabaseConnection databaseConnection;

    private final Instant lastUpdated;

    private final String arn;

    private Component(BuilderImpl builder) {
        this.componentId = builder.componentId;
        this.sid = builder.sid;
        this.systemNumber = builder.systemNumber;
        this.parentComponent = builder.parentComponent;
        this.childComponents = builder.childComponents;
        this.applicationId = builder.applicationId;
        this.componentType = builder.componentType;
        this.status = builder.status;
        this.sapHostname = builder.sapHostname;
        this.sapFeature = builder.sapFeature;
        this.sapKernelVersion = builder.sapKernelVersion;
        this.hdbVersion = builder.hdbVersion;
        this.resilience = builder.resilience;
        this.associatedHost = builder.associatedHost;
        this.databases = builder.databases;
        this.hosts = builder.hosts;
        this.primaryHost = builder.primaryHost;
        this.databaseConnection = builder.databaseConnection;
        this.lastUpdated = builder.lastUpdated;
        this.arn = builder.arn;
    }

    /**
     * <p>
     * The ID of the component.
     * </p>
     * 
     * @return The ID of the component.
     */
    public final String componentId() {
        return componentId;
    }

    /**
     * <p>
     * The SAP System Identifier of the application component.
     * </p>
     * 
     * @return The SAP System Identifier of the application component.
     */
    public final String sid() {
        return sid;
    }

    /**
     * <p>
     * The SAP system number of the application component.
     * </p>
     * 
     * @return The SAP system number of the application component.
     */
    public final String systemNumber() {
        return systemNumber;
    }

    /**
     * <p>
     * The parent component of a highly available environment. For example, in a highly available SAP on AWS workload,
     * the parent component consists of the entire setup, including the child components.
     * </p>
     * 
     * @return The parent component of a highly available environment. For example, in a highly available SAP on AWS
     *         workload, the parent component consists of the entire setup, including the child components.
     */
    public final String parentComponent() {
        return parentComponent;
    }

    /**
     * For responses, this returns true if the service returned a value for the ChildComponents property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasChildComponents() {
        return childComponents != null && !(childComponents instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The child components of a highly available environment. For example, in a highly available SAP on AWS workload,
     * the child component consists of the primary and secondar instances.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasChildComponents} method.
     * </p>
     * 
     * @return The child components of a highly available environment. For example, in a highly available SAP on AWS
     *         workload, the child component consists of the primary and secondar instances.
     */
    public final List<String> childComponents() {
        return childComponents;
    }

    /**
     * <p>
     * The ID of the application.
     * </p>
     * 
     * @return The ID of the application.
     */
    public final String applicationId() {
        return applicationId;
    }

    /**
     * <p>
     * The type of the component.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #componentType}
     * will return {@link ComponentType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #componentTypeAsString}.
     * </p>
     * 
     * @return The type of the component.
     * @see ComponentType
     */
    public final ComponentType componentType() {
        return ComponentType.fromValue(componentType);
    }

    /**
     * <p>
     * The type of the component.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #componentType}
     * will return {@link ComponentType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #componentTypeAsString}.
     * </p>
     * 
     * @return The type of the component.
     * @see ComponentType
     */
    public final String componentTypeAsString() {
        return componentType;
    }

    /**
     * <p>
     * The status of the component.
     * </p>
     * <ul>
     * <li>
     * <p>
     * ACTIVATED - this status has been deprecated.
     * </p>
     * </li>
     * <li>
     * <p>
     * STARTING - the component is in the process of being started.
     * </p>
     * </li>
     * <li>
     * <p>
     * STOPPED - the component is not running.
     * </p>
     * </li>
     * <li>
     * <p>
     * STOPPING - the component is in the process of being stopped.
     * </p>
     * </li>
     * <li>
     * <p>
     * RUNNING - the component is running.
     * </p>
     * </li>
     * <li>
     * <p>
     * RUNNING_WITH_ERROR - one or more child component(s) of the parent component is not running. Call <a
     * href="https://docs.aws.amazon.com/ssmsap/latest/APIReference/API_GetComponent.html"> <code>GetComponent</code>
     * </a> to review the status of each child component.
     * </p>
     * </li>
     * <li>
     * <p>
     * UNDEFINED - AWS Systems Manager for SAP cannot provide the component status based on the discovered information.
     * Verify your SAP application.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ComponentStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the component.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         ACTIVATED - this status has been deprecated.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         STARTING - the component is in the process of being started.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         STOPPED - the component is not running.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         STOPPING - the component is in the process of being stopped.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         RUNNING - the component is running.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         RUNNING_WITH_ERROR - one or more child component(s) of the parent component is not running. Call <a
     *         href="https://docs.aws.amazon.com/ssmsap/latest/APIReference/API_GetComponent.html">
     *         <code>GetComponent</code> </a> to review the status of each child component.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         UNDEFINED - AWS Systems Manager for SAP cannot provide the component status based on the discovered
     *         information. Verify your SAP application.
     *         </p>
     *         </li>
     * @see ComponentStatus
     */
    public final ComponentStatus status() {
        return ComponentStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of the component.
     * </p>
     * <ul>
     * <li>
     * <p>
     * ACTIVATED - this status has been deprecated.
     * </p>
     * </li>
     * <li>
     * <p>
     * STARTING - the component is in the process of being started.
     * </p>
     * </li>
     * <li>
     * <p>
     * STOPPED - the component is not running.
     * </p>
     * </li>
     * <li>
     * <p>
     * STOPPING - the component is in the process of being stopped.
     * </p>
     * </li>
     * <li>
     * <p>
     * RUNNING - the component is running.
     * </p>
     * </li>
     * <li>
     * <p>
     * RUNNING_WITH_ERROR - one or more child component(s) of the parent component is not running. Call <a
     * href="https://docs.aws.amazon.com/ssmsap/latest/APIReference/API_GetComponent.html"> <code>GetComponent</code>
     * </a> to review the status of each child component.
     * </p>
     * </li>
     * <li>
     * <p>
     * UNDEFINED - AWS Systems Manager for SAP cannot provide the component status based on the discovered information.
     * Verify your SAP application.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ComponentStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the component.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         ACTIVATED - this status has been deprecated.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         STARTING - the component is in the process of being started.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         STOPPED - the component is not running.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         STOPPING - the component is in the process of being stopped.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         RUNNING - the component is running.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         RUNNING_WITH_ERROR - one or more child component(s) of the parent component is not running. Call <a
     *         href="https://docs.aws.amazon.com/ssmsap/latest/APIReference/API_GetComponent.html">
     *         <code>GetComponent</code> </a> to review the status of each child component.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         UNDEFINED - AWS Systems Manager for SAP cannot provide the component status based on the discovered
     *         information. Verify your SAP application.
     *         </p>
     *         </li>
     * @see ComponentStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The hostname of the component.
     * </p>
     * 
     * @return The hostname of the component.
     */
    public final String sapHostname() {
        return sapHostname;
    }

    /**
     * <p>
     * The SAP feature of the component.
     * </p>
     * 
     * @return The SAP feature of the component.
     */
    public final String sapFeature() {
        return sapFeature;
    }

    /**
     * <p>
     * The kernel version of the component.
     * </p>
     * 
     * @return The kernel version of the component.
     */
    public final String sapKernelVersion() {
        return sapKernelVersion;
    }

    /**
     * <p>
     * The SAP HANA version of the component.
     * </p>
     * 
     * @return The SAP HANA version of the component.
     */
    public final String hdbVersion() {
        return hdbVersion;
    }

    /**
     * <p>
     * Details of the SAP HANA system replication for the component.
     * </p>
     * 
     * @return Details of the SAP HANA system replication for the component.
     */
    public final Resilience resilience() {
        return resilience;
    }

    /**
     * <p>
     * The associated host of the component.
     * </p>
     * 
     * @return The associated host of the component.
     */
    public final AssociatedHost associatedHost() {
        return associatedHost;
    }

    /**
     * For responses, this returns true if the service returned a value for the Databases property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasDatabases() {
        return databases != null && !(databases instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The SAP HANA databases of the component.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDatabases} method.
     * </p>
     * 
     * @return The SAP HANA databases of the component.
     */
    public final List<String> databases() {
        return databases;
    }

    /**
     * For responses, this returns true if the service returned a value for the Hosts property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     *
     * @deprecated This shape is no longer used. Please use AssociatedHost.
     */
    @Deprecated
    public final boolean hasHosts() {
        return hosts != null && !(hosts instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The hosts of the component.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasHosts} method.
     * </p>
     * 
     * @return The hosts of the component.
     * @deprecated This shape is no longer used. Please use AssociatedHost.
     */
    @Deprecated
    public final List<Host> hosts() {
        return hosts;
    }

    /**
     * <p>
     * The primary host of the component.
     * </p>
     * 
     * @return The primary host of the component.
     * @deprecated This shape is no longer used. Please use AssociatedHost.
     */
    @Deprecated
    public final String primaryHost() {
        return primaryHost;
    }

    /**
     * <p>
     * The connection specifications for the database of the component.
     * </p>
     * 
     * @return The connection specifications for the database of the component.
     */
    public final DatabaseConnection databaseConnection() {
        return databaseConnection;
    }

    /**
     * <p>
     * The time at which the component was last updated.
     * </p>
     * 
     * @return The time at which the component was last updated.
     */
    public final Instant lastUpdated() {
        return lastUpdated;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the component.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the component.
     */
    public final String arn() {
        return arn;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(componentId());
        hashCode = 31 * hashCode + Objects.hashCode(sid());
        hashCode = 31 * hashCode + Objects.hashCode(systemNumber());
        hashCode = 31 * hashCode + Objects.hashCode(parentComponent());
        hashCode = 31 * hashCode + Objects.hashCode(hasChildComponents() ? childComponents() : null);
        hashCode = 31 * hashCode + Objects.hashCode(applicationId());
        hashCode = 31 * hashCode + Objects.hashCode(componentTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(sapHostname());
        hashCode = 31 * hashCode + Objects.hashCode(sapFeature());
        hashCode = 31 * hashCode + Objects.hashCode(sapKernelVersion());
        hashCode = 31 * hashCode + Objects.hashCode(hdbVersion());
        hashCode = 31 * hashCode + Objects.hashCode(resilience());
        hashCode = 31 * hashCode + Objects.hashCode(associatedHost());
        hashCode = 31 * hashCode + Objects.hashCode(hasDatabases() ? databases() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasHosts() ? hosts() : null);
        hashCode = 31 * hashCode + Objects.hashCode(primaryHost());
        hashCode = 31 * hashCode + Objects.hashCode(databaseConnection());
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdated());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Component)) {
            return false;
        }
        Component other = (Component) obj;
        return Objects.equals(componentId(), other.componentId()) && Objects.equals(sid(), other.sid())
                && Objects.equals(systemNumber(), other.systemNumber())
                && Objects.equals(parentComponent(), other.parentComponent())
                && hasChildComponents() == other.hasChildComponents()
                && Objects.equals(childComponents(), other.childComponents())
                && Objects.equals(applicationId(), other.applicationId())
                && Objects.equals(componentTypeAsString(), other.componentTypeAsString())
                && Objects.equals(statusAsString(), other.statusAsString()) && Objects.equals(sapHostname(), other.sapHostname())
                && Objects.equals(sapFeature(), other.sapFeature())
                && Objects.equals(sapKernelVersion(), other.sapKernelVersion())
                && Objects.equals(hdbVersion(), other.hdbVersion()) && Objects.equals(resilience(), other.resilience())
                && Objects.equals(associatedHost(), other.associatedHost()) && hasDatabases() == other.hasDatabases()
                && Objects.equals(databases(), other.databases()) && hasHosts() == other.hasHosts()
                && Objects.equals(hosts(), other.hosts()) && Objects.equals(primaryHost(), other.primaryHost())
                && Objects.equals(databaseConnection(), other.databaseConnection())
                && Objects.equals(lastUpdated(), other.lastUpdated()) && Objects.equals(arn(), other.arn());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Component").add("ComponentId", componentId()).add("Sid", sid())
                .add("SystemNumber", systemNumber()).add("ParentComponent", parentComponent())
                .add("ChildComponents", hasChildComponents() ? childComponents() : null).add("ApplicationId", applicationId())
                .add("ComponentType", componentTypeAsString()).add("Status", statusAsString()).add("SapHostname", sapHostname())
                .add("SapFeature", sapFeature()).add("SapKernelVersion", sapKernelVersion()).add("HdbVersion", hdbVersion())
                .add("Resilience", resilience()).add("AssociatedHost", associatedHost())
                .add("Databases", hasDatabases() ? databases() : null).add("Hosts", hasHosts() ? hosts() : null)
                .add("PrimaryHost", primaryHost()).add("DatabaseConnection", databaseConnection())
                .add("LastUpdated", lastUpdated()).add("Arn", arn()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ComponentId":
            return Optional.ofNullable(clazz.cast(componentId()));
        case "Sid":
            return Optional.ofNullable(clazz.cast(sid()));
        case "SystemNumber":
            return Optional.ofNullable(clazz.cast(systemNumber()));
        case "ParentComponent":
            return Optional.ofNullable(clazz.cast(parentComponent()));
        case "ChildComponents":
            return Optional.ofNullable(clazz.cast(childComponents()));
        case "ApplicationId":
            return Optional.ofNullable(clazz.cast(applicationId()));
        case "ComponentType":
            return Optional.ofNullable(clazz.cast(componentTypeAsString()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "SapHostname":
            return Optional.ofNullable(clazz.cast(sapHostname()));
        case "SapFeature":
            return Optional.ofNullable(clazz.cast(sapFeature()));
        case "SapKernelVersion":
            return Optional.ofNullable(clazz.cast(sapKernelVersion()));
        case "HdbVersion":
            return Optional.ofNullable(clazz.cast(hdbVersion()));
        case "Resilience":
            return Optional.ofNullable(clazz.cast(resilience()));
        case "AssociatedHost":
            return Optional.ofNullable(clazz.cast(associatedHost()));
        case "Databases":
            return Optional.ofNullable(clazz.cast(databases()));
        case "Hosts":
            return Optional.ofNullable(clazz.cast(hosts()));
        case "PrimaryHost":
            return Optional.ofNullable(clazz.cast(primaryHost()));
        case "DatabaseConnection":
            return Optional.ofNullable(clazz.cast(databaseConnection()));
        case "LastUpdated":
            return Optional.ofNullable(clazz.cast(lastUpdated()));
        case "Arn":
            return Optional.ofNullable(clazz.cast(arn()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("ComponentId", COMPONENT_ID_FIELD);
        map.put("Sid", SID_FIELD);
        map.put("SystemNumber", SYSTEM_NUMBER_FIELD);
        map.put("ParentComponent", PARENT_COMPONENT_FIELD);
        map.put("ChildComponents", CHILD_COMPONENTS_FIELD);
        map.put("ApplicationId", APPLICATION_ID_FIELD);
        map.put("ComponentType", COMPONENT_TYPE_FIELD);
        map.put("Status", STATUS_FIELD);
        map.put("SapHostname", SAP_HOSTNAME_FIELD);
        map.put("SapFeature", SAP_FEATURE_FIELD);
        map.put("SapKernelVersion", SAP_KERNEL_VERSION_FIELD);
        map.put("HdbVersion", HDB_VERSION_FIELD);
        map.put("Resilience", RESILIENCE_FIELD);
        map.put("AssociatedHost", ASSOCIATED_HOST_FIELD);
        map.put("Databases", DATABASES_FIELD);
        map.put("Hosts", HOSTS_FIELD);
        map.put("PrimaryHost", PRIMARY_HOST_FIELD);
        map.put("DatabaseConnection", DATABASE_CONNECTION_FIELD);
        map.put("LastUpdated", LAST_UPDATED_FIELD);
        map.put("Arn", ARN_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Component, T> g) {
        return obj -> g.apply((Component) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Component> {
        /**
         * <p>
         * The ID of the component.
         * </p>
         * 
         * @param componentId
         *        The ID of the component.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder componentId(String componentId);

        /**
         * <p>
         * The SAP System Identifier of the application component.
         * </p>
         * 
         * @param sid
         *        The SAP System Identifier of the application component.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sid(String sid);

        /**
         * <p>
         * The SAP system number of the application component.
         * </p>
         * 
         * @param systemNumber
         *        The SAP system number of the application component.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder systemNumber(String systemNumber);

        /**
         * <p>
         * The parent component of a highly available environment. For example, in a highly available SAP on AWS
         * workload, the parent component consists of the entire setup, including the child components.
         * </p>
         * 
         * @param parentComponent
         *        The parent component of a highly available environment. For example, in a highly available SAP on AWS
         *        workload, the parent component consists of the entire setup, including the child components.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parentComponent(String parentComponent);

        /**
         * <p>
         * The child components of a highly available environment. For example, in a highly available SAP on AWS
         * workload, the child component consists of the primary and secondar instances.
         * </p>
         * 
         * @param childComponents
         *        The child components of a highly available environment. For example, in a highly available SAP on AWS
         *        workload, the child component consists of the primary and secondar instances.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder childComponents(Collection<String> childComponents);

        /**
         * <p>
         * The child components of a highly available environment. For example, in a highly available SAP on AWS
         * workload, the child component consists of the primary and secondar instances.
         * </p>
         * 
         * @param childComponents
         *        The child components of a highly available environment. For example, in a highly available SAP on AWS
         *        workload, the child component consists of the primary and secondar instances.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder childComponents(String... childComponents);

        /**
         * <p>
         * The ID of the application.
         * </p>
         * 
         * @param applicationId
         *        The ID of the application.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder applicationId(String applicationId);

        /**
         * <p>
         * The type of the component.
         * </p>
         * 
         * @param componentType
         *        The type of the component.
         * @see ComponentType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ComponentType
         */
        Builder componentType(String componentType);

        /**
         * <p>
         * The type of the component.
         * </p>
         * 
         * @param componentType
         *        The type of the component.
         * @see ComponentType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ComponentType
         */
        Builder componentType(ComponentType componentType);

        /**
         * <p>
         * The status of the component.
         * </p>
         * <ul>
         * <li>
         * <p>
         * ACTIVATED - this status has been deprecated.
         * </p>
         * </li>
         * <li>
         * <p>
         * STARTING - the component is in the process of being started.
         * </p>
         * </li>
         * <li>
         * <p>
         * STOPPED - the component is not running.
         * </p>
         * </li>
         * <li>
         * <p>
         * STOPPING - the component is in the process of being stopped.
         * </p>
         * </li>
         * <li>
         * <p>
         * RUNNING - the component is running.
         * </p>
         * </li>
         * <li>
         * <p>
         * RUNNING_WITH_ERROR - one or more child component(s) of the parent component is not running. Call <a
         * href="https://docs.aws.amazon.com/ssmsap/latest/APIReference/API_GetComponent.html">
         * <code>GetComponent</code> </a> to review the status of each child component.
         * </p>
         * </li>
         * <li>
         * <p>
         * UNDEFINED - AWS Systems Manager for SAP cannot provide the component status based on the discovered
         * information. Verify your SAP application.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        The status of the component.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        ACTIVATED - this status has been deprecated.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        STARTING - the component is in the process of being started.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        STOPPED - the component is not running.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        STOPPING - the component is in the process of being stopped.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        RUNNING - the component is running.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        RUNNING_WITH_ERROR - one or more child component(s) of the parent component is not running. Call <a
         *        href="https://docs.aws.amazon.com/ssmsap/latest/APIReference/API_GetComponent.html">
         *        <code>GetComponent</code> </a> to review the status of each child component.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        UNDEFINED - AWS Systems Manager for SAP cannot provide the component status based on the discovered
         *        information. Verify your SAP application.
         *        </p>
         *        </li>
         * @see ComponentStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ComponentStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the component.
         * </p>
         * <ul>
         * <li>
         * <p>
         * ACTIVATED - this status has been deprecated.
         * </p>
         * </li>
         * <li>
         * <p>
         * STARTING - the component is in the process of being started.
         * </p>
         * </li>
         * <li>
         * <p>
         * STOPPED - the component is not running.
         * </p>
         * </li>
         * <li>
         * <p>
         * STOPPING - the component is in the process of being stopped.
         * </p>
         * </li>
         * <li>
         * <p>
         * RUNNING - the component is running.
         * </p>
         * </li>
         * <li>
         * <p>
         * RUNNING_WITH_ERROR - one or more child component(s) of the parent component is not running. Call <a
         * href="https://docs.aws.amazon.com/ssmsap/latest/APIReference/API_GetComponent.html">
         * <code>GetComponent</code> </a> to review the status of each child component.
         * </p>
         * </li>
         * <li>
         * <p>
         * UNDEFINED - AWS Systems Manager for SAP cannot provide the component status based on the discovered
         * information. Verify your SAP application.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        The status of the component.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        ACTIVATED - this status has been deprecated.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        STARTING - the component is in the process of being started.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        STOPPED - the component is not running.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        STOPPING - the component is in the process of being stopped.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        RUNNING - the component is running.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        RUNNING_WITH_ERROR - one or more child component(s) of the parent component is not running. Call <a
         *        href="https://docs.aws.amazon.com/ssmsap/latest/APIReference/API_GetComponent.html">
         *        <code>GetComponent</code> </a> to review the status of each child component.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        UNDEFINED - AWS Systems Manager for SAP cannot provide the component status based on the discovered
         *        information. Verify your SAP application.
         *        </p>
         *        </li>
         * @see ComponentStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ComponentStatus
         */
        Builder status(ComponentStatus status);

        /**
         * <p>
         * The hostname of the component.
         * </p>
         * 
         * @param sapHostname
         *        The hostname of the component.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sapHostname(String sapHostname);

        /**
         * <p>
         * The SAP feature of the component.
         * </p>
         * 
         * @param sapFeature
         *        The SAP feature of the component.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sapFeature(String sapFeature);

        /**
         * <p>
         * The kernel version of the component.
         * </p>
         * 
         * @param sapKernelVersion
         *        The kernel version of the component.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sapKernelVersion(String sapKernelVersion);

        /**
         * <p>
         * The SAP HANA version of the component.
         * </p>
         * 
         * @param hdbVersion
         *        The SAP HANA version of the component.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hdbVersion(String hdbVersion);

        /**
         * <p>
         * Details of the SAP HANA system replication for the component.
         * </p>
         * 
         * @param resilience
         *        Details of the SAP HANA system replication for the component.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resilience(Resilience resilience);

        /**
         * <p>
         * Details of the SAP HANA system replication for the component.
         * </p>
         * This is a convenience method that creates an instance of the {@link Resilience.Builder} avoiding the need to
         * create one manually via {@link Resilience#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Resilience.Builder#build()} is called immediately and its result
         * is passed to {@link #resilience(Resilience)}.
         * 
         * @param resilience
         *        a consumer that will call methods on {@link Resilience.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #resilience(Resilience)
         */
        default Builder resilience(Consumer<Resilience.Builder> resilience) {
            return resilience(Resilience.builder().applyMutation(resilience).build());
        }

        /**
         * <p>
         * The associated host of the component.
         * </p>
         * 
         * @param associatedHost
         *        The associated host of the component.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder associatedHost(AssociatedHost associatedHost);

        /**
         * <p>
         * The associated host of the component.
         * </p>
         * This is a convenience method that creates an instance of the {@link AssociatedHost.Builder} avoiding the need
         * to create one manually via {@link AssociatedHost#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AssociatedHost.Builder#build()} is called immediately and its
         * result is passed to {@link #associatedHost(AssociatedHost)}.
         * 
         * @param associatedHost
         *        a consumer that will call methods on {@link AssociatedHost.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #associatedHost(AssociatedHost)
         */
        default Builder associatedHost(Consumer<AssociatedHost.Builder> associatedHost) {
            return associatedHost(AssociatedHost.builder().applyMutation(associatedHost).build());
        }

        /**
         * <p>
         * The SAP HANA databases of the component.
         * </p>
         * 
         * @param databases
         *        The SAP HANA databases of the component.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder databases(Collection<String> databases);

        /**
         * <p>
         * The SAP HANA databases of the component.
         * </p>
         * 
         * @param databases
         *        The SAP HANA databases of the component.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder databases(String... databases);

        /**
         * <p>
         * The hosts of the component.
         * </p>
         * 
         * @param hosts
         *        The hosts of the component.
         * @return Returns a reference to this object so that method calls can be chained together.
         * @deprecated This shape is no longer used. Please use AssociatedHost.
         */
        @Deprecated
        Builder hosts(Collection<Host> hosts);

        /**
         * <p>
         * The hosts of the component.
         * </p>
         * 
         * @param hosts
         *        The hosts of the component.
         * @return Returns a reference to this object so that method calls can be chained together.
         * @deprecated This shape is no longer used. Please use AssociatedHost.
         */
        @Deprecated
        Builder hosts(Host... hosts);

        /**
         * <p>
         * The hosts of the component.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ssmsap.model.Host.Builder} avoiding the need to create one manually
         * via {@link software.amazon.awssdk.services.ssmsap.model.Host#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ssmsap.model.Host.Builder#build()} is called immediately and its
         * result is passed to {@link #hosts(List<Host>)}.
         * 
         * @param hosts
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.ssmsap.model.Host.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #hosts(java.util.Collection<Host>)
         * @deprecated This shape is no longer used. Please use AssociatedHost.
         */
        @Deprecated
        Builder hosts(Consumer<Host.Builder>... hosts);

        /**
         * <p>
         * The primary host of the component.
         * </p>
         * 
         * @param primaryHost
         *        The primary host of the component.
         * @return Returns a reference to this object so that method calls can be chained together.
         * @deprecated This shape is no longer used. Please use AssociatedHost.
         */
        @Deprecated
        Builder primaryHost(String primaryHost);

        /**
         * <p>
         * The connection specifications for the database of the component.
         * </p>
         * 
         * @param databaseConnection
         *        The connection specifications for the database of the component.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder databaseConnection(DatabaseConnection databaseConnection);

        /**
         * <p>
         * The connection specifications for the database of the component.
         * </p>
         * This is a convenience method that creates an instance of the {@link DatabaseConnection.Builder} avoiding the
         * need to create one manually via {@link DatabaseConnection#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link DatabaseConnection.Builder#build()} is called immediately and its
         * result is passed to {@link #databaseConnection(DatabaseConnection)}.
         * 
         * @param databaseConnection
         *        a consumer that will call methods on {@link DatabaseConnection.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #databaseConnection(DatabaseConnection)
         */
        default Builder databaseConnection(Consumer<DatabaseConnection.Builder> databaseConnection) {
            return databaseConnection(DatabaseConnection.builder().applyMutation(databaseConnection).build());
        }

        /**
         * <p>
         * The time at which the component was last updated.
         * </p>
         * 
         * @param lastUpdated
         *        The time at which the component was last updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdated(Instant lastUpdated);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the component.
         * </p>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) of the component.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);
    }

    static final class BuilderImpl implements Builder {
        private String componentId;

        private String sid;

        private String systemNumber;

        private String parentComponent;

        private List<String> childComponents = DefaultSdkAutoConstructList.getInstance();

        private String applicationId;

        private String componentType;

        private String status;

        private String sapHostname;

        private String sapFeature;

        private String sapKernelVersion;

        private String hdbVersion;

        private Resilience resilience;

        private AssociatedHost associatedHost;

        private List<String> databases = DefaultSdkAutoConstructList.getInstance();

        private List<Host> hosts = DefaultSdkAutoConstructList.getInstance();

        private String primaryHost;

        private DatabaseConnection databaseConnection;

        private Instant lastUpdated;

        private String arn;

        private BuilderImpl() {
        }

        private BuilderImpl(Component model) {
            componentId(model.componentId);
            sid(model.sid);
            systemNumber(model.systemNumber);
            parentComponent(model.parentComponent);
            childComponents(model.childComponents);
            applicationId(model.applicationId);
            componentType(model.componentType);
            status(model.status);
            sapHostname(model.sapHostname);
            sapFeature(model.sapFeature);
            sapKernelVersion(model.sapKernelVersion);
            hdbVersion(model.hdbVersion);
            resilience(model.resilience);
            associatedHost(model.associatedHost);
            databases(model.databases);
            hosts(model.hosts);
            primaryHost(model.primaryHost);
            databaseConnection(model.databaseConnection);
            lastUpdated(model.lastUpdated);
            arn(model.arn);
        }

        public final String getComponentId() {
            return componentId;
        }

        public final void setComponentId(String componentId) {
            this.componentId = componentId;
        }

        @Override
        public final Builder componentId(String componentId) {
            this.componentId = componentId;
            return this;
        }

        public final String getSid() {
            return sid;
        }

        public final void setSid(String sid) {
            this.sid = sid;
        }

        @Override
        public final Builder sid(String sid) {
            this.sid = sid;
            return this;
        }

        public final String getSystemNumber() {
            return systemNumber;
        }

        public final void setSystemNumber(String systemNumber) {
            this.systemNumber = systemNumber;
        }

        @Override
        public final Builder systemNumber(String systemNumber) {
            this.systemNumber = systemNumber;
            return this;
        }

        public final String getParentComponent() {
            return parentComponent;
        }

        public final void setParentComponent(String parentComponent) {
            this.parentComponent = parentComponent;
        }

        @Override
        public final Builder parentComponent(String parentComponent) {
            this.parentComponent = parentComponent;
            return this;
        }

        public final Collection<String> getChildComponents() {
            if (childComponents instanceof SdkAutoConstructList) {
                return null;
            }
            return childComponents;
        }

        public final void setChildComponents(Collection<String> childComponents) {
            this.childComponents = ComponentIdListCopier.copy(childComponents);
        }

        @Override
        public final Builder childComponents(Collection<String> childComponents) {
            this.childComponents = ComponentIdListCopier.copy(childComponents);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder childComponents(String... childComponents) {
            childComponents(Arrays.asList(childComponents));
            return this;
        }

        public final String getApplicationId() {
            return applicationId;
        }

        public final void setApplicationId(String applicationId) {
            this.applicationId = applicationId;
        }

        @Override
        public final Builder applicationId(String applicationId) {
            this.applicationId = applicationId;
            return this;
        }

        public final String getComponentType() {
            return componentType;
        }

        public final void setComponentType(String componentType) {
            this.componentType = componentType;
        }

        @Override
        public final Builder componentType(String componentType) {
            this.componentType = componentType;
            return this;
        }

        @Override
        public final Builder componentType(ComponentType componentType) {
            this.componentType(componentType == null ? null : componentType.toString());
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(ComponentStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final String getSapHostname() {
            return sapHostname;
        }

        public final void setSapHostname(String sapHostname) {
            this.sapHostname = sapHostname;
        }

        @Override
        public final Builder sapHostname(String sapHostname) {
            this.sapHostname = sapHostname;
            return this;
        }

        public final String getSapFeature() {
            return sapFeature;
        }

        public final void setSapFeature(String sapFeature) {
            this.sapFeature = sapFeature;
        }

        @Override
        public final Builder sapFeature(String sapFeature) {
            this.sapFeature = sapFeature;
            return this;
        }

        public final String getSapKernelVersion() {
            return sapKernelVersion;
        }

        public final void setSapKernelVersion(String sapKernelVersion) {
            this.sapKernelVersion = sapKernelVersion;
        }

        @Override
        public final Builder sapKernelVersion(String sapKernelVersion) {
            this.sapKernelVersion = sapKernelVersion;
            return this;
        }

        public final String getHdbVersion() {
            return hdbVersion;
        }

        public final void setHdbVersion(String hdbVersion) {
            this.hdbVersion = hdbVersion;
        }

        @Override
        public final Builder hdbVersion(String hdbVersion) {
            this.hdbVersion = hdbVersion;
            return this;
        }

        public final Resilience.Builder getResilience() {
            return resilience != null ? resilience.toBuilder() : null;
        }

        public final void setResilience(Resilience.BuilderImpl resilience) {
            this.resilience = resilience != null ? resilience.build() : null;
        }

        @Override
        public final Builder resilience(Resilience resilience) {
            this.resilience = resilience;
            return this;
        }

        public final AssociatedHost.Builder getAssociatedHost() {
            return associatedHost != null ? associatedHost.toBuilder() : null;
        }

        public final void setAssociatedHost(AssociatedHost.BuilderImpl associatedHost) {
            this.associatedHost = associatedHost != null ? associatedHost.build() : null;
        }

        @Override
        public final Builder associatedHost(AssociatedHost associatedHost) {
            this.associatedHost = associatedHost;
            return this;
        }

        public final Collection<String> getDatabases() {
            if (databases instanceof SdkAutoConstructList) {
                return null;
            }
            return databases;
        }

        public final void setDatabases(Collection<String> databases) {
            this.databases = DatabaseIdListCopier.copy(databases);
        }

        @Override
        public final Builder databases(Collection<String> databases) {
            this.databases = DatabaseIdListCopier.copy(databases);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder databases(String... databases) {
            databases(Arrays.asList(databases));
            return this;
        }

        @Deprecated
        public final List<Host.Builder> getHosts() {
            List<Host.Builder> result = HostListCopier.copyToBuilder(this.hosts);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        @Deprecated
        public final void setHosts(Collection<Host.BuilderImpl> hosts) {
            this.hosts = HostListCopier.copyFromBuilder(hosts);
        }

        @Override
        @Deprecated
        public final Builder hosts(Collection<Host> hosts) {
            this.hosts = HostListCopier.copy(hosts);
            return this;
        }

        @Override
        @SafeVarargs
        @Deprecated
        public final Builder hosts(Host... hosts) {
            hosts(Arrays.asList(hosts));
            return this;
        }

        @Override
        @SafeVarargs
        @Deprecated
        public final Builder hosts(Consumer<Host.Builder>... hosts) {
            hosts(Stream.of(hosts).map(c -> Host.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Deprecated
        public final String getPrimaryHost() {
            return primaryHost;
        }

        @Deprecated
        public final void setPrimaryHost(String primaryHost) {
            this.primaryHost = primaryHost;
        }

        @Override
        @Deprecated
        public final Builder primaryHost(String primaryHost) {
            this.primaryHost = primaryHost;
            return this;
        }

        public final DatabaseConnection.Builder getDatabaseConnection() {
            return databaseConnection != null ? databaseConnection.toBuilder() : null;
        }

        public final void setDatabaseConnection(DatabaseConnection.BuilderImpl databaseConnection) {
            this.databaseConnection = databaseConnection != null ? databaseConnection.build() : null;
        }

        @Override
        public final Builder databaseConnection(DatabaseConnection databaseConnection) {
            this.databaseConnection = databaseConnection;
            return this;
        }

        public final Instant getLastUpdated() {
            return lastUpdated;
        }

        public final void setLastUpdated(Instant lastUpdated) {
            this.lastUpdated = lastUpdated;
        }

        @Override
        public final Builder lastUpdated(Instant lastUpdated) {
            this.lastUpdated = lastUpdated;
            return this;
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        @Override
        public Component build() {
            return new Component(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
