/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssmsap.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An operation event returns details for an operation, including key milestones which can be used to monitor and track
 * operations in progress.
 * </p>
 * <p>
 * Operation events contain:
 * </p>
 * <ul>
 * <li>
 * <p>
 * Description string
 * </p>
 * </li>
 * <li>
 * <p>
 * Resource, including its ARN and type
 * </p>
 * </li>
 * <li>
 * <p>
 * Status
 * </p>
 * </li>
 * <li>
 * <p>
 * StatusMessage string
 * </p>
 * </li>
 * <li>
 * <p>
 * TimeStamp
 * </p>
 * </li>
 * </ul>
 * <p>
 * Operation event examples include StartApplication or StopApplication.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class OperationEvent implements SdkPojo, Serializable, ToCopyableBuilder<OperationEvent.Builder, OperationEvent> {
    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(OperationEvent::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<Resource> RESOURCE_FIELD = SdkField.<Resource> builder(MarshallingType.SDK_POJO)
            .memberName("Resource").getter(getter(OperationEvent::resource)).setter(setter(Builder::resource))
            .constructor(Resource::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Resource").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(OperationEvent::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<String> STATUS_MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StatusMessage").getter(getter(OperationEvent::statusMessage)).setter(setter(Builder::statusMessage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StatusMessage").build()).build();

    private static final SdkField<Instant> TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("Timestamp").getter(getter(OperationEvent::timestamp)).setter(setter(Builder::timestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Timestamp").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DESCRIPTION_FIELD,
            RESOURCE_FIELD, STATUS_FIELD, STATUS_MESSAGE_FIELD, TIMESTAMP_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String description;

    private final Resource resource;

    private final String status;

    private final String statusMessage;

    private final Instant timestamp;

    private OperationEvent(BuilderImpl builder) {
        this.description = builder.description;
        this.resource = builder.resource;
        this.status = builder.status;
        this.statusMessage = builder.statusMessage;
        this.timestamp = builder.timestamp;
    }

    /**
     * <p>
     * A description of the operation event. For example, "Stop the EC2 instance i-abcdefgh987654321".
     * </p>
     * 
     * @return A description of the operation event. For example, "Stop the EC2 instance i-abcdefgh987654321".
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * The resource involved in the operations event.
     * </p>
     * <p>
     * Contains <code>ResourceArn</code> ARN and <code>ResourceType</code>.
     * </p>
     * 
     * @return The resource involved in the operations event.</p>
     *         <p>
     *         Contains <code>ResourceArn</code> ARN and <code>ResourceType</code>.
     */
    public final Resource resource() {
        return resource;
    }

    /**
     * <p>
     * The status of the operation event. The possible statuses are: <code>IN_PROGRESS</code>, <code>COMPLETED</code>,
     * and <code>FAILED</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link OperationEventStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the operation event. The possible statuses are: <code>IN_PROGRESS</code>,
     *         <code>COMPLETED</code>, and <code>FAILED</code>.
     * @see OperationEventStatus
     */
    public final OperationEventStatus status() {
        return OperationEventStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of the operation event. The possible statuses are: <code>IN_PROGRESS</code>, <code>COMPLETED</code>,
     * and <code>FAILED</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link OperationEventStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the operation event. The possible statuses are: <code>IN_PROGRESS</code>,
     *         <code>COMPLETED</code>, and <code>FAILED</code>.
     * @see OperationEventStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The status message relating to a specific operation event.
     * </p>
     * 
     * @return The status message relating to a specific operation event.
     */
    public final String statusMessage() {
        return statusMessage;
    }

    /**
     * <p>
     * The timestamp of the specified operation event.
     * </p>
     * 
     * @return The timestamp of the specified operation event.
     */
    public final Instant timestamp() {
        return timestamp;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(resource());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusMessage());
        hashCode = 31 * hashCode + Objects.hashCode(timestamp());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof OperationEvent)) {
            return false;
        }
        OperationEvent other = (OperationEvent) obj;
        return Objects.equals(description(), other.description()) && Objects.equals(resource(), other.resource())
                && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(statusMessage(), other.statusMessage()) && Objects.equals(timestamp(), other.timestamp());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("OperationEvent").add("Description", description()).add("Resource", resource())
                .add("Status", statusAsString()).add("StatusMessage", statusMessage()).add("Timestamp", timestamp()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "Resource":
            return Optional.ofNullable(clazz.cast(resource()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "StatusMessage":
            return Optional.ofNullable(clazz.cast(statusMessage()));
        case "Timestamp":
            return Optional.ofNullable(clazz.cast(timestamp()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Description", DESCRIPTION_FIELD);
        map.put("Resource", RESOURCE_FIELD);
        map.put("Status", STATUS_FIELD);
        map.put("StatusMessage", STATUS_MESSAGE_FIELD);
        map.put("Timestamp", TIMESTAMP_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<OperationEvent, T> g) {
        return obj -> g.apply((OperationEvent) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, OperationEvent> {
        /**
         * <p>
         * A description of the operation event. For example, "Stop the EC2 instance i-abcdefgh987654321".
         * </p>
         * 
         * @param description
         *        A description of the operation event. For example, "Stop the EC2 instance i-abcdefgh987654321".
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The resource involved in the operations event.
         * </p>
         * <p>
         * Contains <code>ResourceArn</code> ARN and <code>ResourceType</code>.
         * </p>
         * 
         * @param resource
         *        The resource involved in the operations event.</p>
         *        <p>
         *        Contains <code>ResourceArn</code> ARN and <code>ResourceType</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resource(Resource resource);

        /**
         * <p>
         * The resource involved in the operations event.
         * </p>
         * <p>
         * Contains <code>ResourceArn</code> ARN and <code>ResourceType</code>.
         * </p>
         * This is a convenience method that creates an instance of the {@link Resource.Builder} avoiding the need to
         * create one manually via {@link Resource#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Resource.Builder#build()} is called immediately and its result is
         * passed to {@link #resource(Resource)}.
         * 
         * @param resource
         *        a consumer that will call methods on {@link Resource.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #resource(Resource)
         */
        default Builder resource(Consumer<Resource.Builder> resource) {
            return resource(Resource.builder().applyMutation(resource).build());
        }

        /**
         * <p>
         * The status of the operation event. The possible statuses are: <code>IN_PROGRESS</code>,
         * <code>COMPLETED</code>, and <code>FAILED</code>.
         * </p>
         * 
         * @param status
         *        The status of the operation event. The possible statuses are: <code>IN_PROGRESS</code>,
         *        <code>COMPLETED</code>, and <code>FAILED</code>.
         * @see OperationEventStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OperationEventStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the operation event. The possible statuses are: <code>IN_PROGRESS</code>,
         * <code>COMPLETED</code>, and <code>FAILED</code>.
         * </p>
         * 
         * @param status
         *        The status of the operation event. The possible statuses are: <code>IN_PROGRESS</code>,
         *        <code>COMPLETED</code>, and <code>FAILED</code>.
         * @see OperationEventStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OperationEventStatus
         */
        Builder status(OperationEventStatus status);

        /**
         * <p>
         * The status message relating to a specific operation event.
         * </p>
         * 
         * @param statusMessage
         *        The status message relating to a specific operation event.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusMessage(String statusMessage);

        /**
         * <p>
         * The timestamp of the specified operation event.
         * </p>
         * 
         * @param timestamp
         *        The timestamp of the specified operation event.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timestamp(Instant timestamp);
    }

    static final class BuilderImpl implements Builder {
        private String description;

        private Resource resource;

        private String status;

        private String statusMessage;

        private Instant timestamp;

        private BuilderImpl() {
        }

        private BuilderImpl(OperationEvent model) {
            description(model.description);
            resource(model.resource);
            status(model.status);
            statusMessage(model.statusMessage);
            timestamp(model.timestamp);
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final Resource.Builder getResource() {
            return resource != null ? resource.toBuilder() : null;
        }

        public final void setResource(Resource.BuilderImpl resource) {
            this.resource = resource != null ? resource.build() : null;
        }

        @Override
        public final Builder resource(Resource resource) {
            this.resource = resource;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(OperationEventStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final String getStatusMessage() {
            return statusMessage;
        }

        public final void setStatusMessage(String statusMessage) {
            this.statusMessage = statusMessage;
        }

        @Override
        public final Builder statusMessage(String statusMessage) {
            this.statusMessage = statusMessage;
            return this;
        }

        public final Instant getTimestamp() {
            return timestamp;
        }

        public final void setTimestamp(Instant timestamp) {
            this.timestamp = timestamp;
        }

        @Override
        public final Builder timestamp(Instant timestamp) {
            this.timestamp = timestamp;
            return this;
        }

        @Override
        public OperationEvent build() {
            return new OperationEvent(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
