/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sso;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.interceptor.SdkInternalExecutionAttribute;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.sso.model.GetRoleCredentialsRequest;
import software.amazon.awssdk.services.sso.model.GetRoleCredentialsResponse;
import software.amazon.awssdk.services.sso.model.InvalidRequestException;
import software.amazon.awssdk.services.sso.model.ListAccountRolesRequest;
import software.amazon.awssdk.services.sso.model.ListAccountRolesResponse;
import software.amazon.awssdk.services.sso.model.ListAccountsRequest;
import software.amazon.awssdk.services.sso.model.ListAccountsResponse;
import software.amazon.awssdk.services.sso.model.LogoutRequest;
import software.amazon.awssdk.services.sso.model.LogoutResponse;
import software.amazon.awssdk.services.sso.model.ResourceNotFoundException;
import software.amazon.awssdk.services.sso.model.SsoException;
import software.amazon.awssdk.services.sso.model.TooManyRequestsException;
import software.amazon.awssdk.services.sso.model.UnauthorizedException;
import software.amazon.awssdk.services.sso.transform.GetRoleCredentialsRequestMarshaller;
import software.amazon.awssdk.services.sso.transform.ListAccountRolesRequestMarshaller;
import software.amazon.awssdk.services.sso.transform.ListAccountsRequestMarshaller;
import software.amazon.awssdk.services.sso.transform.LogoutRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link SsoClient}.
 *
 * @see SsoClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultSsoClient implements SsoClient {
    private static final Logger log = Logger.loggerFor(DefaultSsoClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final SsoServiceClientConfiguration serviceClientConfiguration;

    protected DefaultSsoClient(SsoServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Returns the STS short-term credentials for a given role name that is assigned to the user.
     * </p>
     *
     * @param getRoleCredentialsRequest
     * @return Result of the GetRoleCredentials operation returned by the service.
     * @throws InvalidRequestException
     *         Indicates that a problem occurred with the input to the request. For example, a required parameter might
     *         be missing or out of range.
     * @throws UnauthorizedException
     *         Indicates that the request is not authorized. This can happen due to an invalid access token in the
     *         request.
     * @throws TooManyRequestsException
     *         Indicates that the request is being made too frequently and is more than what the server can handle.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsoException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsoClient.GetRoleCredentials
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sso-2019-06-10/GetRoleCredentials" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetRoleCredentialsResponse getRoleCredentials(GetRoleCredentialsRequest getRoleCredentialsRequest)
            throws InvalidRequestException, UnauthorizedException, TooManyRequestsException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, SsoException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetRoleCredentialsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetRoleCredentialsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getRoleCredentialsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSO");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRoleCredentials");

            return clientHandler.execute(new ClientExecutionParams<GetRoleCredentialsRequest, GetRoleCredentialsResponse>()
                    .withOperationName("GetRoleCredentials").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getRoleCredentialsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .putExecutionAttribute(SdkInternalExecutionAttribute.IS_NONE_AUTH_TYPE_REQUEST, false)
                    .withMarshaller(new GetRoleCredentialsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all roles that are assigned to the user for a given AWS account.
     * </p>
     *
     * @param listAccountRolesRequest
     * @return Result of the ListAccountRoles operation returned by the service.
     * @throws InvalidRequestException
     *         Indicates that a problem occurred with the input to the request. For example, a required parameter might
     *         be missing or out of range.
     * @throws UnauthorizedException
     *         Indicates that the request is not authorized. This can happen due to an invalid access token in the
     *         request.
     * @throws TooManyRequestsException
     *         Indicates that the request is being made too frequently and is more than what the server can handle.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsoException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsoClient.ListAccountRoles
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sso-2019-06-10/ListAccountRoles" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListAccountRolesResponse listAccountRoles(ListAccountRolesRequest listAccountRolesRequest)
            throws InvalidRequestException, UnauthorizedException, TooManyRequestsException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, SsoException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListAccountRolesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListAccountRolesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAccountRolesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSO");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAccountRoles");

            return clientHandler.execute(new ClientExecutionParams<ListAccountRolesRequest, ListAccountRolesResponse>()
                    .withOperationName("ListAccountRoles").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listAccountRolesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .putExecutionAttribute(SdkInternalExecutionAttribute.IS_NONE_AUTH_TYPE_REQUEST, false)
                    .withMarshaller(new ListAccountRolesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all AWS accounts assigned to the user. These AWS accounts are assigned by the administrator of the account.
     * For more information, see <a
     * href="https://docs.aws.amazon.com/singlesignon/latest/userguide/useraccess.html#assignusers">Assign User
     * Access</a> in the <i>IAM Identity Center User Guide</i>. This operation returns a paginated response.
     * </p>
     *
     * @param listAccountsRequest
     * @return Result of the ListAccounts operation returned by the service.
     * @throws InvalidRequestException
     *         Indicates that a problem occurred with the input to the request. For example, a required parameter might
     *         be missing or out of range.
     * @throws UnauthorizedException
     *         Indicates that the request is not authorized. This can happen due to an invalid access token in the
     *         request.
     * @throws TooManyRequestsException
     *         Indicates that the request is being made too frequently and is more than what the server can handle.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsoException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsoClient.ListAccounts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sso-2019-06-10/ListAccounts" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListAccountsResponse listAccounts(ListAccountsRequest listAccountsRequest) throws InvalidRequestException,
            UnauthorizedException, TooManyRequestsException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            SsoException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListAccountsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListAccountsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAccountsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSO");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAccounts");

            return clientHandler.execute(new ClientExecutionParams<ListAccountsRequest, ListAccountsResponse>()
                    .withOperationName("ListAccounts").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listAccountsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .putExecutionAttribute(SdkInternalExecutionAttribute.IS_NONE_AUTH_TYPE_REQUEST, false)
                    .withMarshaller(new ListAccountsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes the locally stored SSO tokens from the client-side cache and sends an API call to the IAM Identity Center
     * service to invalidate the corresponding server-side IAM Identity Center sign in session.
     * </p>
     * <note>
     * <p>
     * If a user uses IAM Identity Center to access the AWS CLI, the user’s IAM Identity Center sign in session is used
     * to obtain an IAM session, as specified in the corresponding IAM Identity Center permission set. More
     * specifically, IAM Identity Center assumes an IAM role in the target account on behalf of the user, and the
     * corresponding temporary AWS credentials are returned to the client.
     * </p>
     * <p>
     * After user logout, any existing IAM role sessions that were created by using IAM Identity Center permission sets
     * continue based on the duration configured in the permission set. For more information, see <a
     * href="https://docs.aws.amazon.com/singlesignon/latest/userguide/authconcept.html">User authentications</a> in the
     * <i>IAM Identity Center User Guide</i>.
     * </p>
     * </note>
     *
     * @param logoutRequest
     * @return Result of the Logout operation returned by the service.
     * @throws InvalidRequestException
     *         Indicates that a problem occurred with the input to the request. For example, a required parameter might
     *         be missing or out of range.
     * @throws UnauthorizedException
     *         Indicates that the request is not authorized. This can happen due to an invalid access token in the
     *         request.
     * @throws TooManyRequestsException
     *         Indicates that the request is being made too frequently and is more than what the server can handle.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsoException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsoClient.Logout
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sso-2019-06-10/Logout" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public LogoutResponse logout(LogoutRequest logoutRequest) throws InvalidRequestException, UnauthorizedException,
            TooManyRequestsException, AwsServiceException, SdkClientException, SsoException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<LogoutResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                LogoutResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, logoutRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSO");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "Logout");

            return clientHandler.execute(new ClientExecutionParams<LogoutRequest, LogoutResponse>().withOperationName("Logout")
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler).withInput(logoutRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .putExecutionAttribute(SdkInternalExecutionAttribute.IS_NONE_AUTH_TYPE_REQUEST, false)
                    .withMarshaller(new LogoutRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(SsoException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyRequestsException")
                                .exceptionBuilderSupplier(TooManyRequestsException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnauthorizedException")
                                .exceptionBuilderSupplier(UnauthorizedException::builder).httpStatusCode(401).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRequestException")
                                .exceptionBuilderSupplier(InvalidRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build());
    }

    @Override
    public final SsoServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
