/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssooidc;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.interceptor.SdkInternalExecutionAttribute;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.ssooidc.model.AccessDeniedException;
import software.amazon.awssdk.services.ssooidc.model.AuthorizationPendingException;
import software.amazon.awssdk.services.ssooidc.model.CreateTokenRequest;
import software.amazon.awssdk.services.ssooidc.model.CreateTokenResponse;
import software.amazon.awssdk.services.ssooidc.model.ExpiredTokenException;
import software.amazon.awssdk.services.ssooidc.model.InternalServerException;
import software.amazon.awssdk.services.ssooidc.model.InvalidClientException;
import software.amazon.awssdk.services.ssooidc.model.InvalidClientMetadataException;
import software.amazon.awssdk.services.ssooidc.model.InvalidGrantException;
import software.amazon.awssdk.services.ssooidc.model.InvalidRequestException;
import software.amazon.awssdk.services.ssooidc.model.InvalidScopeException;
import software.amazon.awssdk.services.ssooidc.model.RegisterClientRequest;
import software.amazon.awssdk.services.ssooidc.model.RegisterClientResponse;
import software.amazon.awssdk.services.ssooidc.model.SlowDownException;
import software.amazon.awssdk.services.ssooidc.model.SsoOidcException;
import software.amazon.awssdk.services.ssooidc.model.StartDeviceAuthorizationRequest;
import software.amazon.awssdk.services.ssooidc.model.StartDeviceAuthorizationResponse;
import software.amazon.awssdk.services.ssooidc.model.UnauthorizedClientException;
import software.amazon.awssdk.services.ssooidc.model.UnsupportedGrantTypeException;
import software.amazon.awssdk.services.ssooidc.transform.CreateTokenRequestMarshaller;
import software.amazon.awssdk.services.ssooidc.transform.RegisterClientRequestMarshaller;
import software.amazon.awssdk.services.ssooidc.transform.StartDeviceAuthorizationRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link SsoOidcClient}.
 *
 * @see SsoOidcClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultSsoOidcClient implements SsoOidcClient {
    private static final Logger log = Logger.loggerFor(DefaultSsoOidcClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultSsoOidcClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates and returns an access token for the authorized client. The access token issued will be used to fetch
     * short-term credentials for the assigned roles in the AWS account.
     * </p>
     *
     * @param createTokenRequest
     * @return Result of the CreateToken operation returned by the service.
     * @throws InvalidRequestException
     *         Indicates that something is wrong with the input to the request. For example, a required parameter might
     *         be missing or out of range.
     * @throws InvalidClientException
     *         Indicates that the <code>clientId</code> or <code>clientSecret</code> in the request is invalid. For
     *         example, this can occur when a client sends an incorrect <code>clientId</code> or an expired
     *         <code>clientSecret</code>.
     * @throws InvalidGrantException
     *         Indicates that a request contains an invalid grant. This can occur if a client makes a <a>CreateToken</a>
     *         request with an invalid grant type.
     * @throws UnauthorizedClientException
     *         Indicates that the client is not currently authorized to make the request. This can happen when a
     *         <code>clientId</code> is not issued for a public client.
     * @throws UnsupportedGrantTypeException
     *         Indicates that the grant type in the request is not supported by the service.
     * @throws InvalidScopeException
     *         Indicates that the scope provided in the request is invalid.
     * @throws AuthorizationPendingException
     *         Indicates that a request to authorize a client with an access user session token is pending.
     * @throws SlowDownException
     *         Indicates that the client is making the request too frequently and is more than the service can handle.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ExpiredTokenException
     *         Indicates that the token issued by the service is expired and is no longer valid.
     * @throws InternalServerException
     *         Indicates that an error from the service occurred while trying to process a request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsoOidcException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsoOidcClient.CreateToken
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sso-oidc-2019-06-10/CreateToken" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateTokenResponse createToken(CreateTokenRequest createTokenRequest) throws InvalidRequestException,
            InvalidClientException, InvalidGrantException, UnauthorizedClientException, UnsupportedGrantTypeException,
            InvalidScopeException, AuthorizationPendingException, SlowDownException, AccessDeniedException,
            ExpiredTokenException, InternalServerException, AwsServiceException, SdkClientException, SsoOidcException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateTokenResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateTokenResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createTokenRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSO OIDC");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateToken");

            return clientHandler.execute(new ClientExecutionParams<CreateTokenRequest, CreateTokenResponse>()
                    .withOperationName("CreateToken").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createTokenRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .putExecutionAttribute(SdkInternalExecutionAttribute.IS_NONE_AUTH_TYPE_REQUEST, false)
                    .withMarshaller(new CreateTokenRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Registers a client with AWS SSO. This allows clients to initiate device authorization. The output should be
     * persisted for reuse through many authentication requests.
     * </p>
     *
     * @param registerClientRequest
     * @return Result of the RegisterClient operation returned by the service.
     * @throws InvalidRequestException
     *         Indicates that something is wrong with the input to the request. For example, a required parameter might
     *         be missing or out of range.
     * @throws InvalidScopeException
     *         Indicates that the scope provided in the request is invalid.
     * @throws InvalidClientMetadataException
     *         Indicates that the client information sent in the request during registration is invalid.
     * @throws InternalServerException
     *         Indicates that an error from the service occurred while trying to process a request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsoOidcException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsoOidcClient.RegisterClient
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sso-oidc-2019-06-10/RegisterClient" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public RegisterClientResponse registerClient(RegisterClientRequest registerClientRequest) throws InvalidRequestException,
            InvalidScopeException, InvalidClientMetadataException, InternalServerException, AwsServiceException,
            SdkClientException, SsoOidcException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RegisterClientResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                RegisterClientResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, registerClientRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSO OIDC");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RegisterClient");

            return clientHandler.execute(new ClientExecutionParams<RegisterClientRequest, RegisterClientResponse>()
                    .withOperationName("RegisterClient").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(registerClientRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .putExecutionAttribute(SdkInternalExecutionAttribute.IS_NONE_AUTH_TYPE_REQUEST, false)
                    .withMarshaller(new RegisterClientRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Initiates device authorization by requesting a pair of verification codes from the authorization service.
     * </p>
     *
     * @param startDeviceAuthorizationRequest
     * @return Result of the StartDeviceAuthorization operation returned by the service.
     * @throws InvalidRequestException
     *         Indicates that something is wrong with the input to the request. For example, a required parameter might
     *         be missing or out of range.
     * @throws InvalidClientException
     *         Indicates that the <code>clientId</code> or <code>clientSecret</code> in the request is invalid. For
     *         example, this can occur when a client sends an incorrect <code>clientId</code> or an expired
     *         <code>clientSecret</code>.
     * @throws UnauthorizedClientException
     *         Indicates that the client is not currently authorized to make the request. This can happen when a
     *         <code>clientId</code> is not issued for a public client.
     * @throws SlowDownException
     *         Indicates that the client is making the request too frequently and is more than the service can handle.
     * @throws InternalServerException
     *         Indicates that an error from the service occurred while trying to process a request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsoOidcException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsoOidcClient.StartDeviceAuthorization
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sso-oidc-2019-06-10/StartDeviceAuthorization"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StartDeviceAuthorizationResponse startDeviceAuthorization(
            StartDeviceAuthorizationRequest startDeviceAuthorizationRequest) throws InvalidRequestException,
            InvalidClientException, UnauthorizedClientException, SlowDownException, InternalServerException, AwsServiceException,
            SdkClientException, SsoOidcException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartDeviceAuthorizationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StartDeviceAuthorizationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startDeviceAuthorizationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSO OIDC");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartDeviceAuthorization");

            return clientHandler
                    .execute(new ClientExecutionParams<StartDeviceAuthorizationRequest, StartDeviceAuthorizationResponse>()
                            .withOperationName("StartDeviceAuthorization").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(startDeviceAuthorizationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .putExecutionAttribute(SdkInternalExecutionAttribute.IS_NONE_AUTH_TYPE_REQUEST, false)
                            .withMarshaller(new StartDeviceAuthorizationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(SsoOidcException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("SlowDownException")
                                .exceptionBuilderSupplier(SlowDownException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ExpiredTokenException")
                                .exceptionBuilderSupplier(ExpiredTokenException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidClientMetadataException")
                                .exceptionBuilderSupplier(InvalidClientMetadataException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidGrantException")
                                .exceptionBuilderSupplier(InvalidGrantException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnauthorizedClientException")
                                .exceptionBuilderSupplier(UnauthorizedClientException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidClientException")
                                .exceptionBuilderSupplier(InvalidClientException::builder).httpStatusCode(401).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AuthorizationPendingException")
                                .exceptionBuilderSupplier(AuthorizationPendingException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRequestException")
                                .exceptionBuilderSupplier(InvalidRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidScopeException")
                                .exceptionBuilderSupplier(InvalidScopeException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnsupportedGrantTypeException")
                                .exceptionBuilderSupplier(UnsupportedGrantTypeException::builder).httpStatusCode(400).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
