/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssooidc;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.interceptor.SdkInternalExecutionAttribute;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.ssooidc.internal.SsoOidcServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.ssooidc.model.AccessDeniedException;
import software.amazon.awssdk.services.ssooidc.model.AuthorizationPendingException;
import software.amazon.awssdk.services.ssooidc.model.CreateTokenRequest;
import software.amazon.awssdk.services.ssooidc.model.CreateTokenResponse;
import software.amazon.awssdk.services.ssooidc.model.CreateTokenWithIamRequest;
import software.amazon.awssdk.services.ssooidc.model.CreateTokenWithIamResponse;
import software.amazon.awssdk.services.ssooidc.model.ExpiredTokenException;
import software.amazon.awssdk.services.ssooidc.model.InternalServerException;
import software.amazon.awssdk.services.ssooidc.model.InvalidClientException;
import software.amazon.awssdk.services.ssooidc.model.InvalidClientMetadataException;
import software.amazon.awssdk.services.ssooidc.model.InvalidGrantException;
import software.amazon.awssdk.services.ssooidc.model.InvalidRequestException;
import software.amazon.awssdk.services.ssooidc.model.InvalidRequestRegionException;
import software.amazon.awssdk.services.ssooidc.model.InvalidScopeException;
import software.amazon.awssdk.services.ssooidc.model.RegisterClientRequest;
import software.amazon.awssdk.services.ssooidc.model.RegisterClientResponse;
import software.amazon.awssdk.services.ssooidc.model.SlowDownException;
import software.amazon.awssdk.services.ssooidc.model.SsoOidcException;
import software.amazon.awssdk.services.ssooidc.model.StartDeviceAuthorizationRequest;
import software.amazon.awssdk.services.ssooidc.model.StartDeviceAuthorizationResponse;
import software.amazon.awssdk.services.ssooidc.model.UnauthorizedClientException;
import software.amazon.awssdk.services.ssooidc.model.UnsupportedGrantTypeException;
import software.amazon.awssdk.services.ssooidc.transform.CreateTokenRequestMarshaller;
import software.amazon.awssdk.services.ssooidc.transform.CreateTokenWithIamRequestMarshaller;
import software.amazon.awssdk.services.ssooidc.transform.RegisterClientRequestMarshaller;
import software.amazon.awssdk.services.ssooidc.transform.StartDeviceAuthorizationRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link SsoOidcAsyncClient}.
 *
 * @see SsoOidcAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultSsoOidcAsyncClient implements SsoOidcAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultSsoOidcAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultSsoOidcAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates and returns access and refresh tokens for clients that are authenticated using client secrets. The access
     * token can be used to fetch short-term credentials for the assigned AWS accounts or to access application APIs
     * using <code>bearer</code> authentication.
     * </p>
     *
     * @param createTokenRequest
     * @return A Java Future containing the result of the CreateToken operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter might be missing or out of range.</li>
     *         <li>InvalidClientException Indicates that the <code>clientId</code> or <code>clientSecret</code> in the
     *         request is invalid. For example, this can occur when a client sends an incorrect <code>clientId</code> or
     *         an expired <code>clientSecret</code>.</li>
     *         <li>InvalidGrantException Indicates that a request contains an invalid grant. This can occur if a client
     *         makes a <a>CreateToken</a> request with an invalid grant type.</li>
     *         <li>UnauthorizedClientException Indicates that the client is not currently authorized to make the
     *         request. This can happen when a <code>clientId</code> is not issued for a public client.</li>
     *         <li>UnsupportedGrantTypeException Indicates that the grant type in the request is not supported by the
     *         service.</li>
     *         <li>InvalidScopeException Indicates that the scope provided in the request is invalid.</li>
     *         <li>AuthorizationPendingException Indicates that a request to authorize a client with an access user
     *         session token is pending.</li>
     *         <li>SlowDownException Indicates that the client is making the request too frequently and is more than the
     *         service can handle.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ExpiredTokenException Indicates that the token issued by the service is expired and is no longer
     *         valid.</li>
     *         <li>InternalServerException Indicates that an error from the service occurred while trying to process a
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsoOidcException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsoOidcAsyncClient.CreateToken
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sso-oidc-2019-06-10/CreateToken" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateTokenResponse> createToken(CreateTokenRequest createTokenRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createTokenRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createTokenRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSO OIDC");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateToken");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateTokenResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateTokenResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateTokenResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateTokenRequest, CreateTokenResponse>()
                            .withOperationName("CreateToken").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateTokenRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .putExecutionAttribute(SdkInternalExecutionAttribute.IS_NONE_AUTH_TYPE_REQUEST, false)
                            .withInput(createTokenRequest));
            CompletableFuture<CreateTokenResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates and returns access and refresh tokens for clients and applications that are authenticated using IAM
     * entities. The access token can be used to fetch short-term credentials for the assigned AWS accounts or to access
     * application APIs using <code>bearer</code> authentication.
     * </p>
     *
     * @param createTokenWithIamRequest
     * @return A Java Future containing the result of the CreateTokenWithIAM operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter might be missing or out of range.</li>
     *         <li>InvalidClientException Indicates that the <code>clientId</code> or <code>clientSecret</code> in the
     *         request is invalid. For example, this can occur when a client sends an incorrect <code>clientId</code> or
     *         an expired <code>clientSecret</code>.</li>
     *         <li>InvalidGrantException Indicates that a request contains an invalid grant. This can occur if a client
     *         makes a <a>CreateToken</a> request with an invalid grant type.</li>
     *         <li>UnauthorizedClientException Indicates that the client is not currently authorized to make the
     *         request. This can happen when a <code>clientId</code> is not issued for a public client.</li>
     *         <li>UnsupportedGrantTypeException Indicates that the grant type in the request is not supported by the
     *         service.</li>
     *         <li>InvalidScopeException Indicates that the scope provided in the request is invalid.</li>
     *         <li>AuthorizationPendingException Indicates that a request to authorize a client with an access user
     *         session token is pending.</li>
     *         <li>SlowDownException Indicates that the client is making the request too frequently and is more than the
     *         service can handle.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ExpiredTokenException Indicates that the token issued by the service is expired and is no longer
     *         valid.</li>
     *         <li>InternalServerException Indicates that an error from the service occurred while trying to process a
     *         request.</li>
     *         <li>InvalidRequestRegionException Indicates that a token provided as input to the request was issued by
     *         and is only usable by calling IAM Identity Center endpoints in another region.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsoOidcException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsoOidcAsyncClient.CreateTokenWithIAM
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sso-oidc-2019-06-10/CreateTokenWithIAM" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateTokenWithIamResponse> createTokenWithIAM(CreateTokenWithIamRequest createTokenWithIamRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createTokenWithIamRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createTokenWithIamRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSO OIDC");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateTokenWithIAM");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateTokenWithIamResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateTokenWithIamResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateTokenWithIamResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateTokenWithIamRequest, CreateTokenWithIamResponse>()
                            .withOperationName("CreateTokenWithIAM").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateTokenWithIamRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createTokenWithIamRequest));
            CompletableFuture<CreateTokenWithIamResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Registers a client with IAM Identity Center. This allows clients to initiate device authorization. The output
     * should be persisted for reuse through many authentication requests.
     * </p>
     *
     * @param registerClientRequest
     * @return A Java Future containing the result of the RegisterClient operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter might be missing or out of range.</li>
     *         <li>InvalidScopeException Indicates that the scope provided in the request is invalid.</li>
     *         <li>InvalidClientMetadataException Indicates that the client information sent in the request during
     *         registration is invalid.</li>
     *         <li>InternalServerException Indicates that an error from the service occurred while trying to process a
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsoOidcException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsoOidcAsyncClient.RegisterClient
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sso-oidc-2019-06-10/RegisterClient" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<RegisterClientResponse> registerClient(RegisterClientRequest registerClientRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(registerClientRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, registerClientRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSO OIDC");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RegisterClient");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RegisterClientResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RegisterClientResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RegisterClientResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RegisterClientRequest, RegisterClientResponse>()
                            .withOperationName("RegisterClient").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new RegisterClientRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .putExecutionAttribute(SdkInternalExecutionAttribute.IS_NONE_AUTH_TYPE_REQUEST, false)
                            .withInput(registerClientRequest));
            CompletableFuture<RegisterClientResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Initiates device authorization by requesting a pair of verification codes from the authorization service.
     * </p>
     *
     * @param startDeviceAuthorizationRequest
     * @return A Java Future containing the result of the StartDeviceAuthorization operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter might be missing or out of range.</li>
     *         <li>InvalidClientException Indicates that the <code>clientId</code> or <code>clientSecret</code> in the
     *         request is invalid. For example, this can occur when a client sends an incorrect <code>clientId</code> or
     *         an expired <code>clientSecret</code>.</li>
     *         <li>UnauthorizedClientException Indicates that the client is not currently authorized to make the
     *         request. This can happen when a <code>clientId</code> is not issued for a public client.</li>
     *         <li>SlowDownException Indicates that the client is making the request too frequently and is more than the
     *         service can handle.</li>
     *         <li>InternalServerException Indicates that an error from the service occurred while trying to process a
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsoOidcException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsoOidcAsyncClient.StartDeviceAuthorization
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sso-oidc-2019-06-10/StartDeviceAuthorization"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartDeviceAuthorizationResponse> startDeviceAuthorization(
            StartDeviceAuthorizationRequest startDeviceAuthorizationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startDeviceAuthorizationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startDeviceAuthorizationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSO OIDC");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartDeviceAuthorization");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartDeviceAuthorizationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartDeviceAuthorizationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartDeviceAuthorizationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartDeviceAuthorizationRequest, StartDeviceAuthorizationResponse>()
                            .withOperationName("StartDeviceAuthorization").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StartDeviceAuthorizationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .putExecutionAttribute(SdkInternalExecutionAttribute.IS_NONE_AUTH_TYPE_REQUEST, false)
                            .withInput(startDeviceAuthorizationRequest));
            CompletableFuture<StartDeviceAuthorizationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final SsoOidcServiceClientConfiguration serviceClientConfiguration() {
        return new SsoOidcServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(SsoOidcException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("SlowDownException")
                                .exceptionBuilderSupplier(SlowDownException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ExpiredTokenException")
                                .exceptionBuilderSupplier(ExpiredTokenException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidClientMetadataException")
                                .exceptionBuilderSupplier(InvalidClientMetadataException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidGrantException")
                                .exceptionBuilderSupplier(InvalidGrantException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnauthorizedClientException")
                                .exceptionBuilderSupplier(UnauthorizedClientException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidClientException")
                                .exceptionBuilderSupplier(InvalidClientException::builder).httpStatusCode(401).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AuthorizationPendingException")
                                .exceptionBuilderSupplier(AuthorizationPendingException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRequestException")
                                .exceptionBuilderSupplier(InvalidRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidScopeException")
                                .exceptionBuilderSupplier(InvalidScopeException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnsupportedGrantTypeException")
                                .exceptionBuilderSupplier(UnsupportedGrantTypeException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRequestRegionException")
                                .exceptionBuilderSupplier(InvalidRequestRegionException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        SsoOidcServiceClientConfigurationBuilder serviceConfigBuilder = new SsoOidcServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
