/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.supplychain;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.supplychain.internal.SupplyChainServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.supplychain.model.AccessDeniedException;
import software.amazon.awssdk.services.supplychain.model.ConflictException;
import software.amazon.awssdk.services.supplychain.model.CreateBillOfMaterialsImportJobRequest;
import software.amazon.awssdk.services.supplychain.model.CreateBillOfMaterialsImportJobResponse;
import software.amazon.awssdk.services.supplychain.model.GetBillOfMaterialsImportJobRequest;
import software.amazon.awssdk.services.supplychain.model.GetBillOfMaterialsImportJobResponse;
import software.amazon.awssdk.services.supplychain.model.InternalServerException;
import software.amazon.awssdk.services.supplychain.model.ResourceNotFoundException;
import software.amazon.awssdk.services.supplychain.model.SendDataIntegrationEventRequest;
import software.amazon.awssdk.services.supplychain.model.SendDataIntegrationEventResponse;
import software.amazon.awssdk.services.supplychain.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.supplychain.model.SupplyChainException;
import software.amazon.awssdk.services.supplychain.model.ThrottlingException;
import software.amazon.awssdk.services.supplychain.model.ValidationException;
import software.amazon.awssdk.services.supplychain.transform.CreateBillOfMaterialsImportJobRequestMarshaller;
import software.amazon.awssdk.services.supplychain.transform.GetBillOfMaterialsImportJobRequestMarshaller;
import software.amazon.awssdk.services.supplychain.transform.SendDataIntegrationEventRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link SupplyChainAsyncClient}.
 *
 * @see SupplyChainAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultSupplyChainAsyncClient implements SupplyChainAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultSupplyChainAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultSupplyChainAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * CreateBillOfMaterialsImportJob creates an import job for the Product Bill Of Materials (BOM) entity. For
     * information on the product_bom entity, see the AWS Supply Chain User Guide.
     * </p>
     * <p>
     * The CSV file must be located in an Amazon S3 location accessible to AWS Supply Chain. It is recommended to use
     * the same Amazon S3 bucket created during your AWS Supply Chain instance creation.
     * </p>
     *
     * @param createBillOfMaterialsImportJobRequest
     *        The request parameters for CreateBillOfMaterialsImportJob.
     * @return A Java Future containing the result of the CreateBillOfMaterialsImportJob operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException Request would cause a service quota to be exceeded.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>AccessDeniedException You do not have the required privileges to perform this action.</li>
     *         <li>ValidationException The input does not satisfy the constraints specified by an AWS service.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SupplyChainException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SupplyChainAsyncClient.CreateBillOfMaterialsImportJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/supplychain-2024-01-01/CreateBillOfMaterialsImportJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateBillOfMaterialsImportJobResponse> createBillOfMaterialsImportJob(
            CreateBillOfMaterialsImportJobRequest createBillOfMaterialsImportJobRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createBillOfMaterialsImportJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                createBillOfMaterialsImportJobRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SupplyChain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateBillOfMaterialsImportJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateBillOfMaterialsImportJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateBillOfMaterialsImportJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateBillOfMaterialsImportJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateBillOfMaterialsImportJobRequest, CreateBillOfMaterialsImportJobResponse>()
                            .withOperationName("CreateBillOfMaterialsImportJob").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateBillOfMaterialsImportJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createBillOfMaterialsImportJobRequest));
            CompletableFuture<CreateBillOfMaterialsImportJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Get status and details of a BillOfMaterialsImportJob.
     * </p>
     *
     * @param getBillOfMaterialsImportJobRequest
     *        The request parameters for GetBillOfMaterialsImportJob.
     * @return A Java Future containing the result of the GetBillOfMaterialsImportJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException Request would cause a service quota to be exceeded.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have the required privileges to perform this action.</li>
     *         <li>ValidationException The input does not satisfy the constraints specified by an AWS service.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SupplyChainException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SupplyChainAsyncClient.GetBillOfMaterialsImportJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/supplychain-2024-01-01/GetBillOfMaterialsImportJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetBillOfMaterialsImportJobResponse> getBillOfMaterialsImportJob(
            GetBillOfMaterialsImportJobRequest getBillOfMaterialsImportJobRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getBillOfMaterialsImportJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getBillOfMaterialsImportJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SupplyChain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetBillOfMaterialsImportJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetBillOfMaterialsImportJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetBillOfMaterialsImportJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetBillOfMaterialsImportJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetBillOfMaterialsImportJobRequest, GetBillOfMaterialsImportJobResponse>()
                            .withOperationName("GetBillOfMaterialsImportJob").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetBillOfMaterialsImportJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getBillOfMaterialsImportJobRequest));
            CompletableFuture<GetBillOfMaterialsImportJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Send transactional data events with real-time data for analysis or monitoring.
     * </p>
     *
     * @param sendDataIntegrationEventRequest
     *        The request parameters for SendDataIntegrationEvent.
     * @return A Java Future containing the result of the SendDataIntegrationEvent operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException Request would cause a service quota to be exceeded.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have the required privileges to perform this action.</li>
     *         <li>ValidationException The input does not satisfy the constraints specified by an AWS service.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SupplyChainException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SupplyChainAsyncClient.SendDataIntegrationEvent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/supplychain-2024-01-01/SendDataIntegrationEvent"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<SendDataIntegrationEventResponse> sendDataIntegrationEvent(
            SendDataIntegrationEventRequest sendDataIntegrationEventRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(sendDataIntegrationEventRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, sendDataIntegrationEventRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SupplyChain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SendDataIntegrationEvent");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SendDataIntegrationEventResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, SendDataIntegrationEventResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<SendDataIntegrationEventResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SendDataIntegrationEventRequest, SendDataIntegrationEventResponse>()
                            .withOperationName("SendDataIntegrationEvent").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new SendDataIntegrationEventRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(sendDataIntegrationEventRequest));
            CompletableFuture<SendDataIntegrationEventResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final SupplyChainServiceClientConfiguration serviceClientConfiguration() {
        return new SupplyChainServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(SupplyChainException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        SupplyChainServiceClientConfigurationBuilder serviceConfigBuilder = new SupplyChainServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
