/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.supplychain;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.supplychain.internal.SupplyChainServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.supplychain.model.AccessDeniedException;
import software.amazon.awssdk.services.supplychain.model.ConflictException;
import software.amazon.awssdk.services.supplychain.model.CreateBillOfMaterialsImportJobRequest;
import software.amazon.awssdk.services.supplychain.model.CreateBillOfMaterialsImportJobResponse;
import software.amazon.awssdk.services.supplychain.model.GetBillOfMaterialsImportJobRequest;
import software.amazon.awssdk.services.supplychain.model.GetBillOfMaterialsImportJobResponse;
import software.amazon.awssdk.services.supplychain.model.InternalServerException;
import software.amazon.awssdk.services.supplychain.model.ResourceNotFoundException;
import software.amazon.awssdk.services.supplychain.model.SendDataIntegrationEventRequest;
import software.amazon.awssdk.services.supplychain.model.SendDataIntegrationEventResponse;
import software.amazon.awssdk.services.supplychain.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.supplychain.model.SupplyChainException;
import software.amazon.awssdk.services.supplychain.model.ThrottlingException;
import software.amazon.awssdk.services.supplychain.model.ValidationException;
import software.amazon.awssdk.services.supplychain.transform.CreateBillOfMaterialsImportJobRequestMarshaller;
import software.amazon.awssdk.services.supplychain.transform.GetBillOfMaterialsImportJobRequestMarshaller;
import software.amazon.awssdk.services.supplychain.transform.SendDataIntegrationEventRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link SupplyChainClient}.
 *
 * @see SupplyChainClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultSupplyChainClient implements SupplyChainClient {
    private static final Logger log = Logger.loggerFor(DefaultSupplyChainClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultSupplyChainClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * CreateBillOfMaterialsImportJob creates an import job for the Product Bill Of Materials (BOM) entity. For
     * information on the product_bom entity, see the AWS Supply Chain User Guide.
     * </p>
     * <p>
     * The CSV file must be located in an Amazon S3 location accessible to AWS Supply Chain. It is recommended to use
     * the same Amazon S3 bucket created during your AWS Supply Chain instance creation.
     * </p>
     *
     * @param createBillOfMaterialsImportJobRequest
     *        The request parameters for CreateBillOfMaterialsImportJob.
     * @return Result of the CreateBillOfMaterialsImportJob operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         Request would cause a service quota to be exceeded.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         Request references a resource which does not exist.
     * @throws AccessDeniedException
     *         You do not have the required privileges to perform this action.
     * @throws ValidationException
     *         The input does not satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         Unexpected error during processing of request.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SupplyChainException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SupplyChainClient.CreateBillOfMaterialsImportJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/supplychain-2024-01-01/CreateBillOfMaterialsImportJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateBillOfMaterialsImportJobResponse createBillOfMaterialsImportJob(
            CreateBillOfMaterialsImportJobRequest createBillOfMaterialsImportJobRequest) throws ServiceQuotaExceededException,
            ThrottlingException, ResourceNotFoundException, AccessDeniedException, ValidationException, InternalServerException,
            ConflictException, AwsServiceException, SdkClientException, SupplyChainException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateBillOfMaterialsImportJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateBillOfMaterialsImportJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createBillOfMaterialsImportJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                createBillOfMaterialsImportJobRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SupplyChain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateBillOfMaterialsImportJob");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateBillOfMaterialsImportJobRequest, CreateBillOfMaterialsImportJobResponse>()
                            .withOperationName("CreateBillOfMaterialsImportJob").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(createBillOfMaterialsImportJobRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateBillOfMaterialsImportJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Get status and details of a BillOfMaterialsImportJob.
     * </p>
     *
     * @param getBillOfMaterialsImportJobRequest
     *        The request parameters for GetBillOfMaterialsImportJob.
     * @return Result of the GetBillOfMaterialsImportJob operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         Request would cause a service quota to be exceeded.
     * @throws ResourceNotFoundException
     *         Request references a resource which does not exist.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have the required privileges to perform this action.
     * @throws ValidationException
     *         The input does not satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         Unexpected error during processing of request.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SupplyChainException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SupplyChainClient.GetBillOfMaterialsImportJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/supplychain-2024-01-01/GetBillOfMaterialsImportJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetBillOfMaterialsImportJobResponse getBillOfMaterialsImportJob(
            GetBillOfMaterialsImportJobRequest getBillOfMaterialsImportJobRequest) throws ServiceQuotaExceededException,
            ResourceNotFoundException, ThrottlingException, AccessDeniedException, ValidationException, InternalServerException,
            ConflictException, AwsServiceException, SdkClientException, SupplyChainException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetBillOfMaterialsImportJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetBillOfMaterialsImportJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getBillOfMaterialsImportJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getBillOfMaterialsImportJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SupplyChain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetBillOfMaterialsImportJob");

            return clientHandler
                    .execute(new ClientExecutionParams<GetBillOfMaterialsImportJobRequest, GetBillOfMaterialsImportJobResponse>()
                            .withOperationName("GetBillOfMaterialsImportJob").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(getBillOfMaterialsImportJobRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetBillOfMaterialsImportJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Send the transactional data payload for the event with real-time data for analysis or monitoring. The real-time
     * data events are stored in an Amazon Web Services service before being processed and stored in data lake. New data
     * events are synced with data lake at 5 PM GMT everyday. The updated transactional data is available in data lake
     * after ingestion.
     * </p>
     *
     * @param sendDataIntegrationEventRequest
     *        The request parameters for SendDataIntegrationEvent.
     * @return Result of the SendDataIntegrationEvent operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         Request would cause a service quota to be exceeded.
     * @throws ResourceNotFoundException
     *         Request references a resource which does not exist.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have the required privileges to perform this action.
     * @throws ValidationException
     *         The input does not satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         Unexpected error during processing of request.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SupplyChainException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SupplyChainClient.SendDataIntegrationEvent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/supplychain-2024-01-01/SendDataIntegrationEvent"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public SendDataIntegrationEventResponse sendDataIntegrationEvent(
            SendDataIntegrationEventRequest sendDataIntegrationEventRequest) throws ServiceQuotaExceededException,
            ResourceNotFoundException, ThrottlingException, AccessDeniedException, ValidationException, InternalServerException,
            ConflictException, AwsServiceException, SdkClientException, SupplyChainException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<SendDataIntegrationEventResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, SendDataIntegrationEventResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(sendDataIntegrationEventRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, sendDataIntegrationEventRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SupplyChain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SendDataIntegrationEvent");

            return clientHandler
                    .execute(new ClientExecutionParams<SendDataIntegrationEventRequest, SendDataIntegrationEventResponse>()
                            .withOperationName("SendDataIntegrationEvent").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(sendDataIntegrationEventRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new SendDataIntegrationEventRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        SupplyChainServiceClientConfigurationBuilder serviceConfigBuilder = new SupplyChainServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(SupplyChainException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build());
    }

    @Override
    public final SupplyChainServiceClientConfiguration serviceClientConfiguration() {
        return new SupplyChainServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
