/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.supplychain.model;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.DefaultValueTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The request parameters for CreateInstance.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateInstanceRequest extends SupplyChainRequest implements
        ToCopyableBuilder<CreateInstanceRequest.Builder, CreateInstanceRequest> {
    private static final SdkField<String> INSTANCE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("instanceName").getter(getter(CreateInstanceRequest::instanceName)).setter(setter(Builder::instanceName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("instanceName").build()).build();

    private static final SdkField<String> INSTANCE_DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("instanceDescription").getter(getter(CreateInstanceRequest::instanceDescription))
            .setter(setter(Builder::instanceDescription))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("instanceDescription").build())
            .build();

    private static final SdkField<String> KMS_KEY_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("kmsKeyArn").getter(getter(CreateInstanceRequest::kmsKeyArn)).setter(setter(Builder::kmsKeyArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("kmsKeyArn").build()).build();

    private static final SdkField<Map<String, String>> TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("tags")
            .getter(getter(CreateInstanceRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> CLIENT_TOKEN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("clientToken")
            .getter(getter(CreateInstanceRequest::clientToken))
            .setter(setter(Builder::clientToken))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("clientToken").build(),
                    DefaultValueTrait.idempotencyToken()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(INSTANCE_NAME_FIELD,
            INSTANCE_DESCRIPTION_FIELD, KMS_KEY_ARN_FIELD, TAGS_FIELD, CLIENT_TOKEN_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("instanceName", INSTANCE_NAME_FIELD);
                    put("instanceDescription", INSTANCE_DESCRIPTION_FIELD);
                    put("kmsKeyArn", KMS_KEY_ARN_FIELD);
                    put("tags", TAGS_FIELD);
                    put("clientToken", CLIENT_TOKEN_FIELD);
                }
            });

    private final String instanceName;

    private final String instanceDescription;

    private final String kmsKeyArn;

    private final Map<String, String> tags;

    private final String clientToken;

    private CreateInstanceRequest(BuilderImpl builder) {
        super(builder);
        this.instanceName = builder.instanceName;
        this.instanceDescription = builder.instanceDescription;
        this.kmsKeyArn = builder.kmsKeyArn;
        this.tags = builder.tags;
        this.clientToken = builder.clientToken;
    }

    /**
     * <p>
     * The AWS Supply Chain instance name.
     * </p>
     * 
     * @return The AWS Supply Chain instance name.
     */
    public final String instanceName() {
        return instanceName;
    }

    /**
     * <p>
     * The AWS Supply Chain instance description.
     * </p>
     * 
     * @return The AWS Supply Chain instance description.
     */
    public final String instanceDescription() {
        return instanceDescription;
    }

    /**
     * <p>
     * The ARN (Amazon Resource Name) of the Key Management Service (KMS) key you provide for encryption. This is
     * required if you do not want to use the Amazon Web Services owned KMS key. If you don't provide anything here, AWS
     * Supply Chain uses the Amazon Web Services owned KMS key.
     * </p>
     * 
     * @return The ARN (Amazon Resource Name) of the Key Management Service (KMS) key you provide for encryption. This
     *         is required if you do not want to use the Amazon Web Services owned KMS key. If you don't provide
     *         anything here, AWS Supply Chain uses the Amazon Web Services owned KMS key.
     */
    public final String kmsKeyArn() {
        return kmsKeyArn;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The Amazon Web Services tags of an instance to be created.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The Amazon Web Services tags of an instance to be created.
     */
    public final Map<String, String> tags() {
        return tags;
    }

    /**
     * <p>
     * The client token for idempotency.
     * </p>
     * 
     * @return The client token for idempotency.
     */
    public final String clientToken() {
        return clientToken;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(instanceName());
        hashCode = 31 * hashCode + Objects.hashCode(instanceDescription());
        hashCode = 31 * hashCode + Objects.hashCode(kmsKeyArn());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(clientToken());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateInstanceRequest)) {
            return false;
        }
        CreateInstanceRequest other = (CreateInstanceRequest) obj;
        return Objects.equals(instanceName(), other.instanceName())
                && Objects.equals(instanceDescription(), other.instanceDescription())
                && Objects.equals(kmsKeyArn(), other.kmsKeyArn()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags()) && Objects.equals(clientToken(), other.clientToken());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateInstanceRequest").add("InstanceName", instanceName())
                .add("InstanceDescription", instanceDescription()).add("KmsKeyArn", kmsKeyArn())
                .add("Tags", hasTags() ? tags() : null).add("ClientToken", clientToken()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "instanceName":
            return Optional.ofNullable(clazz.cast(instanceName()));
        case "instanceDescription":
            return Optional.ofNullable(clazz.cast(instanceDescription()));
        case "kmsKeyArn":
            return Optional.ofNullable(clazz.cast(kmsKeyArn()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "clientToken":
            return Optional.ofNullable(clazz.cast(clientToken()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<CreateInstanceRequest, T> g) {
        return obj -> g.apply((CreateInstanceRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SupplyChainRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreateInstanceRequest> {
        /**
         * <p>
         * The AWS Supply Chain instance name.
         * </p>
         * 
         * @param instanceName
         *        The AWS Supply Chain instance name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceName(String instanceName);

        /**
         * <p>
         * The AWS Supply Chain instance description.
         * </p>
         * 
         * @param instanceDescription
         *        The AWS Supply Chain instance description.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceDescription(String instanceDescription);

        /**
         * <p>
         * The ARN (Amazon Resource Name) of the Key Management Service (KMS) key you provide for encryption. This is
         * required if you do not want to use the Amazon Web Services owned KMS key. If you don't provide anything here,
         * AWS Supply Chain uses the Amazon Web Services owned KMS key.
         * </p>
         * 
         * @param kmsKeyArn
         *        The ARN (Amazon Resource Name) of the Key Management Service (KMS) key you provide for encryption.
         *        This is required if you do not want to use the Amazon Web Services owned KMS key. If you don't provide
         *        anything here, AWS Supply Chain uses the Amazon Web Services owned KMS key.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsKeyArn(String kmsKeyArn);

        /**
         * <p>
         * The Amazon Web Services tags of an instance to be created.
         * </p>
         * 
         * @param tags
         *        The Amazon Web Services tags of an instance to be created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);

        /**
         * <p>
         * The client token for idempotency.
         * </p>
         * 
         * @param clientToken
         *        The client token for idempotency.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientToken(String clientToken);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends SupplyChainRequest.BuilderImpl implements Builder {
        private String instanceName;

        private String instanceDescription;

        private String kmsKeyArn;

        private Map<String, String> tags = DefaultSdkAutoConstructMap.getInstance();

        private String clientToken;

        private BuilderImpl() {
        }

        private BuilderImpl(CreateInstanceRequest model) {
            super(model);
            instanceName(model.instanceName);
            instanceDescription(model.instanceDescription);
            kmsKeyArn(model.kmsKeyArn);
            tags(model.tags);
            clientToken(model.clientToken);
        }

        public final String getInstanceName() {
            return instanceName;
        }

        public final void setInstanceName(String instanceName) {
            this.instanceName = instanceName;
        }

        @Override
        public final Builder instanceName(String instanceName) {
            this.instanceName = instanceName;
            return this;
        }

        public final String getInstanceDescription() {
            return instanceDescription;
        }

        public final void setInstanceDescription(String instanceDescription) {
            this.instanceDescription = instanceDescription;
        }

        @Override
        public final Builder instanceDescription(String instanceDescription) {
            this.instanceDescription = instanceDescription;
            return this;
        }

        public final String getKmsKeyArn() {
            return kmsKeyArn;
        }

        public final void setKmsKeyArn(String kmsKeyArn) {
            this.kmsKeyArn = kmsKeyArn;
        }

        @Override
        public final Builder kmsKeyArn(String kmsKeyArn) {
            this.kmsKeyArn = kmsKeyArn;
            return this;
        }

        public final Map<String, String> getTags() {
            if (tags instanceof SdkAutoConstructMap) {
                return null;
            }
            return tags;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
        }

        @Override
        public final Builder tags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
            return this;
        }

        public final String getClientToken() {
            return clientToken;
        }

        public final void setClientToken(String clientToken) {
            this.clientToken = clientToken;
        }

        @Override
        public final Builder clientToken(String clientToken) {
            this.clientToken = clientToken;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateInstanceRequest build() {
            return new CreateInstanceRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
