/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.supplychain;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import java.util.function.Function;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.supplychain.internal.SupplyChainServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.supplychain.model.AccessDeniedException;
import software.amazon.awssdk.services.supplychain.model.ConflictException;
import software.amazon.awssdk.services.supplychain.model.CreateBillOfMaterialsImportJobRequest;
import software.amazon.awssdk.services.supplychain.model.CreateBillOfMaterialsImportJobResponse;
import software.amazon.awssdk.services.supplychain.model.CreateDataIntegrationFlowRequest;
import software.amazon.awssdk.services.supplychain.model.CreateDataIntegrationFlowResponse;
import software.amazon.awssdk.services.supplychain.model.CreateDataLakeDatasetRequest;
import software.amazon.awssdk.services.supplychain.model.CreateDataLakeDatasetResponse;
import software.amazon.awssdk.services.supplychain.model.CreateInstanceRequest;
import software.amazon.awssdk.services.supplychain.model.CreateInstanceResponse;
import software.amazon.awssdk.services.supplychain.model.DeleteDataIntegrationFlowRequest;
import software.amazon.awssdk.services.supplychain.model.DeleteDataIntegrationFlowResponse;
import software.amazon.awssdk.services.supplychain.model.DeleteDataLakeDatasetRequest;
import software.amazon.awssdk.services.supplychain.model.DeleteDataLakeDatasetResponse;
import software.amazon.awssdk.services.supplychain.model.DeleteInstanceRequest;
import software.amazon.awssdk.services.supplychain.model.DeleteInstanceResponse;
import software.amazon.awssdk.services.supplychain.model.GetBillOfMaterialsImportJobRequest;
import software.amazon.awssdk.services.supplychain.model.GetBillOfMaterialsImportJobResponse;
import software.amazon.awssdk.services.supplychain.model.GetDataIntegrationFlowRequest;
import software.amazon.awssdk.services.supplychain.model.GetDataIntegrationFlowResponse;
import software.amazon.awssdk.services.supplychain.model.GetDataLakeDatasetRequest;
import software.amazon.awssdk.services.supplychain.model.GetDataLakeDatasetResponse;
import software.amazon.awssdk.services.supplychain.model.GetInstanceRequest;
import software.amazon.awssdk.services.supplychain.model.GetInstanceResponse;
import software.amazon.awssdk.services.supplychain.model.InternalServerException;
import software.amazon.awssdk.services.supplychain.model.ListDataIntegrationFlowsRequest;
import software.amazon.awssdk.services.supplychain.model.ListDataIntegrationFlowsResponse;
import software.amazon.awssdk.services.supplychain.model.ListDataLakeDatasetsRequest;
import software.amazon.awssdk.services.supplychain.model.ListDataLakeDatasetsResponse;
import software.amazon.awssdk.services.supplychain.model.ListInstancesRequest;
import software.amazon.awssdk.services.supplychain.model.ListInstancesResponse;
import software.amazon.awssdk.services.supplychain.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.supplychain.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.supplychain.model.ResourceNotFoundException;
import software.amazon.awssdk.services.supplychain.model.SendDataIntegrationEventRequest;
import software.amazon.awssdk.services.supplychain.model.SendDataIntegrationEventResponse;
import software.amazon.awssdk.services.supplychain.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.supplychain.model.SupplyChainException;
import software.amazon.awssdk.services.supplychain.model.TagResourceRequest;
import software.amazon.awssdk.services.supplychain.model.TagResourceResponse;
import software.amazon.awssdk.services.supplychain.model.ThrottlingException;
import software.amazon.awssdk.services.supplychain.model.UntagResourceRequest;
import software.amazon.awssdk.services.supplychain.model.UntagResourceResponse;
import software.amazon.awssdk.services.supplychain.model.UpdateDataIntegrationFlowRequest;
import software.amazon.awssdk.services.supplychain.model.UpdateDataIntegrationFlowResponse;
import software.amazon.awssdk.services.supplychain.model.UpdateDataLakeDatasetRequest;
import software.amazon.awssdk.services.supplychain.model.UpdateDataLakeDatasetResponse;
import software.amazon.awssdk.services.supplychain.model.UpdateInstanceRequest;
import software.amazon.awssdk.services.supplychain.model.UpdateInstanceResponse;
import software.amazon.awssdk.services.supplychain.model.ValidationException;
import software.amazon.awssdk.services.supplychain.transform.CreateBillOfMaterialsImportJobRequestMarshaller;
import software.amazon.awssdk.services.supplychain.transform.CreateDataIntegrationFlowRequestMarshaller;
import software.amazon.awssdk.services.supplychain.transform.CreateDataLakeDatasetRequestMarshaller;
import software.amazon.awssdk.services.supplychain.transform.CreateInstanceRequestMarshaller;
import software.amazon.awssdk.services.supplychain.transform.DeleteDataIntegrationFlowRequestMarshaller;
import software.amazon.awssdk.services.supplychain.transform.DeleteDataLakeDatasetRequestMarshaller;
import software.amazon.awssdk.services.supplychain.transform.DeleteInstanceRequestMarshaller;
import software.amazon.awssdk.services.supplychain.transform.GetBillOfMaterialsImportJobRequestMarshaller;
import software.amazon.awssdk.services.supplychain.transform.GetDataIntegrationFlowRequestMarshaller;
import software.amazon.awssdk.services.supplychain.transform.GetDataLakeDatasetRequestMarshaller;
import software.amazon.awssdk.services.supplychain.transform.GetInstanceRequestMarshaller;
import software.amazon.awssdk.services.supplychain.transform.ListDataIntegrationFlowsRequestMarshaller;
import software.amazon.awssdk.services.supplychain.transform.ListDataLakeDatasetsRequestMarshaller;
import software.amazon.awssdk.services.supplychain.transform.ListInstancesRequestMarshaller;
import software.amazon.awssdk.services.supplychain.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.supplychain.transform.SendDataIntegrationEventRequestMarshaller;
import software.amazon.awssdk.services.supplychain.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.supplychain.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.supplychain.transform.UpdateDataIntegrationFlowRequestMarshaller;
import software.amazon.awssdk.services.supplychain.transform.UpdateDataLakeDatasetRequestMarshaller;
import software.amazon.awssdk.services.supplychain.transform.UpdateInstanceRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link SupplyChainAsyncClient}.
 *
 * @see SupplyChainAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultSupplyChainAsyncClient implements SupplyChainAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultSupplyChainAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultSupplyChainAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * CreateBillOfMaterialsImportJob creates an import job for the Product Bill Of Materials (BOM) entity. For
     * information on the product_bom entity, see the AWS Supply Chain User Guide.
     * </p>
     * <p>
     * The CSV file must be located in an Amazon S3 location accessible to AWS Supply Chain. It is recommended to use
     * the same Amazon S3 bucket created during your AWS Supply Chain instance creation.
     * </p>
     *
     * @param createBillOfMaterialsImportJobRequest
     *        The request parameters for CreateBillOfMaterialsImportJob.
     * @return A Java Future containing the result of the CreateBillOfMaterialsImportJob operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException Request would cause a service quota to be exceeded.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>AccessDeniedException You do not have the required privileges to perform this action.</li>
     *         <li>ValidationException The input does not satisfy the constraints specified by an AWS service.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SupplyChainException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SupplyChainAsyncClient.CreateBillOfMaterialsImportJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/supplychain-2024-01-01/CreateBillOfMaterialsImportJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateBillOfMaterialsImportJobResponse> createBillOfMaterialsImportJob(
            CreateBillOfMaterialsImportJobRequest createBillOfMaterialsImportJobRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createBillOfMaterialsImportJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                createBillOfMaterialsImportJobRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SupplyChain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateBillOfMaterialsImportJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateBillOfMaterialsImportJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateBillOfMaterialsImportJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateBillOfMaterialsImportJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateBillOfMaterialsImportJobRequest, CreateBillOfMaterialsImportJobResponse>()
                            .withOperationName("CreateBillOfMaterialsImportJob").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateBillOfMaterialsImportJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createBillOfMaterialsImportJobRequest));
            CompletableFuture<CreateBillOfMaterialsImportJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Enables you to programmatically create a data pipeline to ingest data from source systems such as Amazon S3
     * buckets, to a predefined Amazon Web Services Supply Chain dataset (product, inbound_order) or a temporary dataset
     * along with the data transformation query provided with the API.
     * </p>
     *
     * @param createDataIntegrationFlowRequest
     *        The request parameters for CreateDataIntegrationFlow.
     * @return A Java Future containing the result of the CreateDataIntegrationFlow operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException Request would cause a service quota to be exceeded.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>AccessDeniedException You do not have the required privileges to perform this action.</li>
     *         <li>ValidationException The input does not satisfy the constraints specified by an AWS service.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SupplyChainException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SupplyChainAsyncClient.CreateDataIntegrationFlow
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/supplychain-2024-01-01/CreateDataIntegrationFlow"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateDataIntegrationFlowResponse> createDataIntegrationFlow(
            CreateDataIntegrationFlowRequest createDataIntegrationFlowRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createDataIntegrationFlowRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDataIntegrationFlowRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SupplyChain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDataIntegrationFlow");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateDataIntegrationFlowResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateDataIntegrationFlowResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateDataIntegrationFlowResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateDataIntegrationFlowRequest, CreateDataIntegrationFlowResponse>()
                            .withOperationName("CreateDataIntegrationFlow").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateDataIntegrationFlowRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createDataIntegrationFlowRequest));
            CompletableFuture<CreateDataIntegrationFlowResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Enables you to programmatically create an Amazon Web Services Supply Chain data lake dataset. Developers can
     * create the datasets using their pre-defined or custom schema for a given instance ID, namespace, and dataset
     * name.
     * </p>
     *
     * @param createDataLakeDatasetRequest
     *        The request parameters for CreateDataLakeDataset.
     * @return A Java Future containing the result of the CreateDataLakeDataset operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException Request would cause a service quota to be exceeded.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>AccessDeniedException You do not have the required privileges to perform this action.</li>
     *         <li>ValidationException The input does not satisfy the constraints specified by an AWS service.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SupplyChainException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SupplyChainAsyncClient.CreateDataLakeDataset
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/supplychain-2024-01-01/CreateDataLakeDataset"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateDataLakeDatasetResponse> createDataLakeDataset(
            CreateDataLakeDatasetRequest createDataLakeDatasetRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createDataLakeDatasetRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDataLakeDatasetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SupplyChain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDataLakeDataset");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateDataLakeDatasetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateDataLakeDatasetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateDataLakeDatasetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateDataLakeDatasetRequest, CreateDataLakeDatasetResponse>()
                            .withOperationName("CreateDataLakeDataset").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateDataLakeDatasetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createDataLakeDatasetRequest));
            CompletableFuture<CreateDataLakeDatasetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Enables you to programmatically create an Amazon Web Services Supply Chain instance by applying KMS keys and
     * relevant information associated with the API without using the Amazon Web Services console.
     * </p>
     * <p>
     * This is an asynchronous operation. Upon receiving a CreateInstance request, Amazon Web Services Supply Chain
     * immediately returns the instance resource, instance ID, and the initializing state while simultaneously creating
     * all required Amazon Web Services resources for an instance creation. You can use GetInstance to check the status
     * of the instance. If the instance results in an unhealthy state, you need to check the error message, delete the
     * current instance, and recreate a new one based on the mitigation from the error message.
     * </p>
     *
     * @param createInstanceRequest
     *        The request parameters for CreateInstance.
     * @return A Java Future containing the result of the CreateInstance operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException Request would cause a service quota to be exceeded.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>AccessDeniedException You do not have the required privileges to perform this action.</li>
     *         <li>ValidationException The input does not satisfy the constraints specified by an AWS service.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SupplyChainException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SupplyChainAsyncClient.CreateInstance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/supplychain-2024-01-01/CreateInstance" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateInstanceResponse> createInstance(CreateInstanceRequest createInstanceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createInstanceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createInstanceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SupplyChain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateInstance");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateInstanceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateInstanceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateInstanceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateInstanceRequest, CreateInstanceResponse>()
                            .withOperationName("CreateInstance").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateInstanceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createInstanceRequest));
            CompletableFuture<CreateInstanceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Enable you to programmatically delete an existing data pipeline for the provided Amazon Web Services Supply Chain
     * instance and DataIntegrationFlow name.
     * </p>
     *
     * @param deleteDataIntegrationFlowRequest
     *        The request parameters for DeleteDataIntegrationFlow.
     * @return A Java Future containing the result of the DeleteDataIntegrationFlow operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException Request would cause a service quota to be exceeded.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have the required privileges to perform this action.</li>
     *         <li>ValidationException The input does not satisfy the constraints specified by an AWS service.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SupplyChainException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SupplyChainAsyncClient.DeleteDataIntegrationFlow
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/supplychain-2024-01-01/DeleteDataIntegrationFlow"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteDataIntegrationFlowResponse> deleteDataIntegrationFlow(
            DeleteDataIntegrationFlowRequest deleteDataIntegrationFlowRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteDataIntegrationFlowRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDataIntegrationFlowRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SupplyChain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDataIntegrationFlow");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteDataIntegrationFlowResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteDataIntegrationFlowResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteDataIntegrationFlowResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteDataIntegrationFlowRequest, DeleteDataIntegrationFlowResponse>()
                            .withOperationName("DeleteDataIntegrationFlow").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteDataIntegrationFlowRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteDataIntegrationFlowRequest));
            CompletableFuture<DeleteDataIntegrationFlowResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Enables you to programmatically delete an Amazon Web Services Supply Chain data lake dataset. Developers can
     * delete the existing datasets for a given instance ID, namespace, and instance name.
     * </p>
     *
     * @param deleteDataLakeDatasetRequest
     *        The request parameters of DeleteDataLakeDataset.
     * @return A Java Future containing the result of the DeleteDataLakeDataset operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException Request would cause a service quota to be exceeded.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have the required privileges to perform this action.</li>
     *         <li>ValidationException The input does not satisfy the constraints specified by an AWS service.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SupplyChainException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SupplyChainAsyncClient.DeleteDataLakeDataset
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/supplychain-2024-01-01/DeleteDataLakeDataset"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteDataLakeDatasetResponse> deleteDataLakeDataset(
            DeleteDataLakeDatasetRequest deleteDataLakeDatasetRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteDataLakeDatasetRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDataLakeDatasetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SupplyChain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDataLakeDataset");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteDataLakeDatasetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteDataLakeDatasetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteDataLakeDatasetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteDataLakeDatasetRequest, DeleteDataLakeDatasetResponse>()
                            .withOperationName("DeleteDataLakeDataset").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteDataLakeDatasetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteDataLakeDatasetRequest));
            CompletableFuture<DeleteDataLakeDatasetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Enables you to programmatically delete an Amazon Web Services Supply Chain instance by deleting the KMS keys and
     * relevant information associated with the API without using the Amazon Web Services console.
     * </p>
     * <p>
     * This is an asynchronous operation. Upon receiving a DeleteInstance request, Amazon Web Services Supply Chain
     * immediately returns a response with the instance resource, delete state while cleaning up all Amazon Web Services
     * resources created during the instance creation process. You can use the GetInstance action to check the instance
     * status.
     * </p>
     *
     * @param deleteInstanceRequest
     *        The request parameters for DeleteInstance.
     * @return A Java Future containing the result of the DeleteInstance operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException Request would cause a service quota to be exceeded.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have the required privileges to perform this action.</li>
     *         <li>ValidationException The input does not satisfy the constraints specified by an AWS service.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SupplyChainException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SupplyChainAsyncClient.DeleteInstance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/supplychain-2024-01-01/DeleteInstance" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteInstanceResponse> deleteInstance(DeleteInstanceRequest deleteInstanceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteInstanceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteInstanceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SupplyChain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteInstance");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteInstanceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteInstanceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteInstanceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteInstanceRequest, DeleteInstanceResponse>()
                            .withOperationName("DeleteInstance").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteInstanceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteInstanceRequest));
            CompletableFuture<DeleteInstanceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Get status and details of a BillOfMaterialsImportJob.
     * </p>
     *
     * @param getBillOfMaterialsImportJobRequest
     *        The request parameters for GetBillOfMaterialsImportJob.
     * @return A Java Future containing the result of the GetBillOfMaterialsImportJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException Request would cause a service quota to be exceeded.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have the required privileges to perform this action.</li>
     *         <li>ValidationException The input does not satisfy the constraints specified by an AWS service.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SupplyChainException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SupplyChainAsyncClient.GetBillOfMaterialsImportJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/supplychain-2024-01-01/GetBillOfMaterialsImportJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetBillOfMaterialsImportJobResponse> getBillOfMaterialsImportJob(
            GetBillOfMaterialsImportJobRequest getBillOfMaterialsImportJobRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getBillOfMaterialsImportJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getBillOfMaterialsImportJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SupplyChain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetBillOfMaterialsImportJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetBillOfMaterialsImportJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetBillOfMaterialsImportJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetBillOfMaterialsImportJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetBillOfMaterialsImportJobRequest, GetBillOfMaterialsImportJobResponse>()
                            .withOperationName("GetBillOfMaterialsImportJob").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetBillOfMaterialsImportJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getBillOfMaterialsImportJobRequest));
            CompletableFuture<GetBillOfMaterialsImportJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Enables you to programmatically view a specific data pipeline for the provided Amazon Web Services Supply Chain
     * instance and DataIntegrationFlow name.
     * </p>
     *
     * @param getDataIntegrationFlowRequest
     *        The request parameters for GetDataIntegrationFlow.
     * @return A Java Future containing the result of the GetDataIntegrationFlow operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException Request would cause a service quota to be exceeded.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have the required privileges to perform this action.</li>
     *         <li>ValidationException The input does not satisfy the constraints specified by an AWS service.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SupplyChainException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SupplyChainAsyncClient.GetDataIntegrationFlow
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/supplychain-2024-01-01/GetDataIntegrationFlow"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetDataIntegrationFlowResponse> getDataIntegrationFlow(
            GetDataIntegrationFlowRequest getDataIntegrationFlowRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getDataIntegrationFlowRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDataIntegrationFlowRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SupplyChain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDataIntegrationFlow");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDataIntegrationFlowResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetDataIntegrationFlowResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetDataIntegrationFlowResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDataIntegrationFlowRequest, GetDataIntegrationFlowResponse>()
                            .withOperationName("GetDataIntegrationFlow").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetDataIntegrationFlowRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getDataIntegrationFlowRequest));
            CompletableFuture<GetDataIntegrationFlowResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Enables you to programmatically view an Amazon Web Services Supply Chain data lake dataset. Developers can view
     * the data lake dataset information such as namespace, schema, and so on for a given instance ID, namespace, and
     * dataset name.
     * </p>
     *
     * @param getDataLakeDatasetRequest
     *        The request parameters for GetDataLakeDataset.
     * @return A Java Future containing the result of the GetDataLakeDataset operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException Request would cause a service quota to be exceeded.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have the required privileges to perform this action.</li>
     *         <li>ValidationException The input does not satisfy the constraints specified by an AWS service.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SupplyChainException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SupplyChainAsyncClient.GetDataLakeDataset
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/supplychain-2024-01-01/GetDataLakeDataset"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetDataLakeDatasetResponse> getDataLakeDataset(GetDataLakeDatasetRequest getDataLakeDatasetRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getDataLakeDatasetRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDataLakeDatasetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SupplyChain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDataLakeDataset");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDataLakeDatasetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetDataLakeDatasetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetDataLakeDatasetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDataLakeDatasetRequest, GetDataLakeDatasetResponse>()
                            .withOperationName("GetDataLakeDataset").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetDataLakeDatasetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getDataLakeDatasetRequest));
            CompletableFuture<GetDataLakeDatasetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Enables you to programmatically retrieve the information related to an Amazon Web Services Supply Chain instance
     * ID.
     * </p>
     *
     * @param getInstanceRequest
     *        The request parameters for GetInstance.
     * @return A Java Future containing the result of the GetInstance operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException Request would cause a service quota to be exceeded.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have the required privileges to perform this action.</li>
     *         <li>ValidationException The input does not satisfy the constraints specified by an AWS service.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SupplyChainException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SupplyChainAsyncClient.GetInstance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/supplychain-2024-01-01/GetInstance" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetInstanceResponse> getInstance(GetInstanceRequest getInstanceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getInstanceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getInstanceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SupplyChain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetInstance");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetInstanceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetInstanceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetInstanceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetInstanceRequest, GetInstanceResponse>()
                            .withOperationName("GetInstance").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetInstanceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getInstanceRequest));
            CompletableFuture<GetInstanceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Enables you to programmatically list all data pipelines for the provided Amazon Web Services Supply Chain
     * instance.
     * </p>
     *
     * @param listDataIntegrationFlowsRequest
     *        The request parameters for ListDataIntegrationFlows.
     * @return A Java Future containing the result of the ListDataIntegrationFlows operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException Request would cause a service quota to be exceeded.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>AccessDeniedException You do not have the required privileges to perform this action.</li>
     *         <li>ValidationException The input does not satisfy the constraints specified by an AWS service.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SupplyChainException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SupplyChainAsyncClient.ListDataIntegrationFlows
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/supplychain-2024-01-01/ListDataIntegrationFlows"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListDataIntegrationFlowsResponse> listDataIntegrationFlows(
            ListDataIntegrationFlowsRequest listDataIntegrationFlowsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listDataIntegrationFlowsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDataIntegrationFlowsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SupplyChain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDataIntegrationFlows");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDataIntegrationFlowsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListDataIntegrationFlowsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDataIntegrationFlowsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDataIntegrationFlowsRequest, ListDataIntegrationFlowsResponse>()
                            .withOperationName("ListDataIntegrationFlows").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListDataIntegrationFlowsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listDataIntegrationFlowsRequest));
            CompletableFuture<ListDataIntegrationFlowsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Enables you to programmatically view the list of Amazon Web Services Supply Chain data lake datasets. Developers
     * can view the datasets and the corresponding information such as namespace, schema, and so on for a given instance
     * ID and namespace.
     * </p>
     *
     * @param listDataLakeDatasetsRequest
     *        The request parameters of ListDataLakeDatasets.
     * @return A Java Future containing the result of the ListDataLakeDatasets operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException Request would cause a service quota to be exceeded.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have the required privileges to perform this action.</li>
     *         <li>ValidationException The input does not satisfy the constraints specified by an AWS service.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SupplyChainException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SupplyChainAsyncClient.ListDataLakeDatasets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/supplychain-2024-01-01/ListDataLakeDatasets"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListDataLakeDatasetsResponse> listDataLakeDatasets(
            ListDataLakeDatasetsRequest listDataLakeDatasetsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listDataLakeDatasetsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDataLakeDatasetsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SupplyChain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDataLakeDatasets");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDataLakeDatasetsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListDataLakeDatasetsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDataLakeDatasetsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDataLakeDatasetsRequest, ListDataLakeDatasetsResponse>()
                            .withOperationName("ListDataLakeDatasets").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListDataLakeDatasetsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listDataLakeDatasetsRequest));
            CompletableFuture<ListDataLakeDatasetsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List all Amazon Web Services Supply Chain instances for a specific account. Enables you to programmatically list
     * all Amazon Web Services Supply Chain instances based on their account ID, instance name, and state of the
     * instance (active or delete).
     * </p>
     *
     * @param listInstancesRequest
     *        The request parameters for ListInstances.
     * @return A Java Future containing the result of the ListInstances operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException Request would cause a service quota to be exceeded.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>AccessDeniedException You do not have the required privileges to perform this action.</li>
     *         <li>ValidationException The input does not satisfy the constraints specified by an AWS service.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SupplyChainException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SupplyChainAsyncClient.ListInstances
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/supplychain-2024-01-01/ListInstances" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListInstancesResponse> listInstances(ListInstancesRequest listInstancesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listInstancesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listInstancesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SupplyChain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListInstances");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListInstancesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListInstancesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListInstancesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListInstancesRequest, ListInstancesResponse>()
                            .withOperationName("ListInstances").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListInstancesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listInstancesRequest));
            CompletableFuture<ListInstancesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List all the tags for an Amazon Web ServicesSupply Chain resource. You can list all the tags added to a resource.
     * By listing the tags, developers can view the tag level information on a resource and perform actions such as,
     * deleting a resource associated with a particular tag.
     * </p>
     *
     * @param listTagsForResourceRequest
     *        The request parameters of ListTagsForResource.
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException Request would cause a service quota to be exceeded.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have the required privileges to perform this action.</li>
     *         <li>ValidationException The input does not satisfy the constraints specified by an AWS service.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SupplyChainException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SupplyChainAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/supplychain-2024-01-01/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SupplyChain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Send the transactional data payload for the event with real-time data for analysis or monitoring. The real-time
     * data events are stored in an Amazon Web Services service before being processed and stored in data lake. New data
     * events are synced with data lake at 5 PM GMT everyday. The updated transactional data is available in data lake
     * after ingestion.
     * </p>
     *
     * @param sendDataIntegrationEventRequest
     *        The request parameters for SendDataIntegrationEvent.
     * @return A Java Future containing the result of the SendDataIntegrationEvent operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException Request would cause a service quota to be exceeded.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have the required privileges to perform this action.</li>
     *         <li>ValidationException The input does not satisfy the constraints specified by an AWS service.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SupplyChainException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SupplyChainAsyncClient.SendDataIntegrationEvent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/supplychain-2024-01-01/SendDataIntegrationEvent"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<SendDataIntegrationEventResponse> sendDataIntegrationEvent(
            SendDataIntegrationEventRequest sendDataIntegrationEventRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(sendDataIntegrationEventRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, sendDataIntegrationEventRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SupplyChain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SendDataIntegrationEvent");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SendDataIntegrationEventResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, SendDataIntegrationEventResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<SendDataIntegrationEventResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SendDataIntegrationEventRequest, SendDataIntegrationEventResponse>()
                            .withOperationName("SendDataIntegrationEvent").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new SendDataIntegrationEventRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(sendDataIntegrationEventRequest));
            CompletableFuture<SendDataIntegrationEventResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * You can create tags during or after creating a resource such as instance, data flow, or dataset in AWS Supply
     * chain. During the data ingestion process, you can add tags such as dev, test, or prod to data flows created
     * during the data ingestion process in the AWS Supply Chain datasets. You can use these tags to identify a group of
     * resources or a single resource used by the developer.
     * </p>
     *
     * @param tagResourceRequest
     *        The request parameters of TagResource.
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException Request would cause a service quota to be exceeded.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have the required privileges to perform this action.</li>
     *         <li>ValidationException The input does not satisfy the constraints specified by an AWS service.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SupplyChainException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SupplyChainAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/supplychain-2024-01-01/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SupplyChain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * You can delete tags for an Amazon Web Services Supply chain resource such as instance, data flow, or dataset in
     * AWS Supply Chain. During the data ingestion process, you can delete tags such as dev, test, or prod to data flows
     * created during the data ingestion process in the AWS Supply Chain datasets.
     * </p>
     *
     * @param untagResourceRequest
     *        The request parameters of UntagResource.
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException Request would cause a service quota to be exceeded.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have the required privileges to perform this action.</li>
     *         <li>ValidationException The input does not satisfy the constraints specified by an AWS service.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SupplyChainException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SupplyChainAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/supplychain-2024-01-01/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SupplyChain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Enables you to programmatically update an existing data pipeline to ingest data from the source systems such as,
     * Amazon S3 buckets, to a predefined Amazon Web Services Supply Chain dataset (product, inbound_order) or a
     * temporary dataset along with the data transformation query provided with the API.
     * </p>
     *
     * @param updateDataIntegrationFlowRequest
     *        The request parameters for UpdateDataIntegrationFlow.
     * @return A Java Future containing the result of the UpdateDataIntegrationFlow operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException Request would cause a service quota to be exceeded.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have the required privileges to perform this action.</li>
     *         <li>ValidationException The input does not satisfy the constraints specified by an AWS service.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SupplyChainException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SupplyChainAsyncClient.UpdateDataIntegrationFlow
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/supplychain-2024-01-01/UpdateDataIntegrationFlow"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateDataIntegrationFlowResponse> updateDataIntegrationFlow(
            UpdateDataIntegrationFlowRequest updateDataIntegrationFlowRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateDataIntegrationFlowRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateDataIntegrationFlowRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SupplyChain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDataIntegrationFlow");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateDataIntegrationFlowResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateDataIntegrationFlowResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateDataIntegrationFlowResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateDataIntegrationFlowRequest, UpdateDataIntegrationFlowResponse>()
                            .withOperationName("UpdateDataIntegrationFlow").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateDataIntegrationFlowRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateDataIntegrationFlowRequest));
            CompletableFuture<UpdateDataIntegrationFlowResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Enables you to programmatically update an Amazon Web Services Supply Chain data lake dataset. Developers can
     * update the description of a data lake dataset for a given instance ID, namespace, and dataset name.
     * </p>
     *
     * @param updateDataLakeDatasetRequest
     *        The request parameters of UpdateDataLakeDataset.
     * @return A Java Future containing the result of the UpdateDataLakeDataset operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException Request would cause a service quota to be exceeded.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have the required privileges to perform this action.</li>
     *         <li>ValidationException The input does not satisfy the constraints specified by an AWS service.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SupplyChainException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SupplyChainAsyncClient.UpdateDataLakeDataset
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/supplychain-2024-01-01/UpdateDataLakeDataset"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateDataLakeDatasetResponse> updateDataLakeDataset(
            UpdateDataLakeDatasetRequest updateDataLakeDatasetRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateDataLakeDatasetRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateDataLakeDatasetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SupplyChain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDataLakeDataset");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateDataLakeDatasetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateDataLakeDatasetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateDataLakeDatasetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateDataLakeDatasetRequest, UpdateDataLakeDatasetResponse>()
                            .withOperationName("UpdateDataLakeDataset").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateDataLakeDatasetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateDataLakeDatasetRequest));
            CompletableFuture<UpdateDataLakeDatasetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Enables you to programmatically update an Amazon Web Services Supply Chain instance description by providing all
     * the relevant information such as account ID, instance ID and so on without using the AWS console.
     * </p>
     *
     * @param updateInstanceRequest
     *        The request parameters for UpdateInstance.
     * @return A Java Future containing the result of the UpdateInstance operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException Request would cause a service quota to be exceeded.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have the required privileges to perform this action.</li>
     *         <li>ValidationException The input does not satisfy the constraints specified by an AWS service.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SupplyChainException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SupplyChainAsyncClient.UpdateInstance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/supplychain-2024-01-01/UpdateInstance" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateInstanceResponse> updateInstance(UpdateInstanceRequest updateInstanceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateInstanceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateInstanceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SupplyChain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateInstance");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateInstanceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateInstanceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateInstanceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateInstanceRequest, UpdateInstanceResponse>()
                            .withOperationName("UpdateInstance").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateInstanceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateInstanceRequest));
            CompletableFuture<UpdateInstanceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final SupplyChainServiceClientConfiguration serviceClientConfiguration() {
        return new SupplyChainServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(SupplyChainException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        SupplyChainServiceClientConfigurationBuilder serviceConfigBuilder = new SupplyChainServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
