/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.support.paginators;

import java.util.Collections;
import java.util.Iterator;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;
import org.reactivestreams.Subscriber;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.pagination.async.AsyncPageFetcher;
import software.amazon.awssdk.core.pagination.async.EmptySubscription;
import software.amazon.awssdk.core.pagination.async.PaginatedItemsPublisher;
import software.amazon.awssdk.core.pagination.async.ResponsesSubscription;
import software.amazon.awssdk.core.pagination.async.SdkPublisher;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.services.support.SupportAsyncClient;
import software.amazon.awssdk.services.support.model.CaseDetails;
import software.amazon.awssdk.services.support.model.DescribeCasesRequest;
import software.amazon.awssdk.services.support.model.DescribeCasesResponse;

/**
 * <p>
 * Represents the output for the
 * {@link software.amazon.awssdk.services.support.SupportAsyncClient#describeCasesPaginator(software.amazon.awssdk.services.support.model.DescribeCasesRequest)}
 * operation which is a paginated operation. This class is a type of {@link org.reactivestreams.Publisher} which can be
 * used to provide a sequence of {@link software.amazon.awssdk.services.support.model.DescribeCasesResponse} response
 * pages as per demand from the subscriber.
 * </p>
 * <p>
 * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet and
 * so there is no guarantee that the request is valid. If there are errors in your request, you will see the failures
 * only after you start streaming the data. The subscribe method should be called as a request to start streaming data.
 * For more info, see {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the
 * subscribe method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data
 * from the starting request.
 * </p>
 *
 * <p>
 * The following are few ways to use the response class:
 * </p>
 * 1) Using the forEach helper method
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.support.paginators.DescribeCasesPublisher publisher = client.describeCasesPaginator(request);
 * CompletableFuture<Void> future = publisher.forEach(res -> { // Do something with the response });
 * future.get();
 * }
 * </pre>
 *
 * 2) Using a custom subscriber
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.support.paginators.DescribeCasesPublisher publisher = client.describeCasesPaginator(request);
 * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.support.model.DescribeCasesResponse>() {
 * 
 * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
 * 
 * 
 * public void onNext(software.amazon.awssdk.services.support.model.DescribeCasesResponse response) { //... };
 * });}
 * </pre>
 * 
 * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
 * <p>
 * <b>Note: If you prefer to have control on service calls, use the
 * {@link #describeCases(software.amazon.awssdk.services.support.model.DescribeCasesRequest)} operation.</b>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class DescribeCasesPublisher implements SdkPublisher<DescribeCasesResponse> {
    private final SupportAsyncClient client;

    private final DescribeCasesRequest firstRequest;

    private final AsyncPageFetcher nextPageFetcher;

    private boolean isLastPage;

    public DescribeCasesPublisher(SupportAsyncClient client, DescribeCasesRequest firstRequest) {
        this(client, firstRequest, false);
    }

    private DescribeCasesPublisher(SupportAsyncClient client, DescribeCasesRequest firstRequest, boolean isLastPage) {
        this.client = client;
        this.firstRequest = firstRequest;
        this.isLastPage = isLastPage;
        this.nextPageFetcher = new DescribeCasesResponseFetcher();
    }

    @Override
    public void subscribe(Subscriber<? super DescribeCasesResponse> subscriber) {
        subscriber.onSubscribe(ResponsesSubscription.builder().subscriber(subscriber).nextPageFetcher(nextPageFetcher).build());
    }

    /**
     * Returns a publisher that can be used to get a stream of data. You need to subscribe to the publisher to request
     * the stream of data. The publisher has a helper forEach method that takes in a {@link java.util.function.Consumer}
     * and then applies that consumer to each response returned by the service.
     */
    public final SdkPublisher<CaseDetails> cases() {
        Function<DescribeCasesResponse, Iterator<CaseDetails>> getIterator = response -> {
            if (response != null && response.cases() != null) {
                return response.cases().iterator();
            }
            return Collections.emptyIterator();
        };
        return PaginatedItemsPublisher.builder().nextPageFetcher(new DescribeCasesResponseFetcher())
                .iteratorFunction(getIterator).isLastPage(isLastPage).build();
    }

    /**
     * <p>
     * A helper method to resume the pages in case of unexpected failures. The method takes the last successful response
     * page as input and returns an instance of {@link DescribeCasesPublisher} that can be used to retrieve the
     * consecutive pages that follows the input page.
     * </p>
     */
    private final DescribeCasesPublisher resume(DescribeCasesResponse lastSuccessfulPage) {
        if (nextPageFetcher.hasNextPage(lastSuccessfulPage)) {
            return new DescribeCasesPublisher(client, firstRequest.toBuilder().nextToken(lastSuccessfulPage.nextToken()).build());
        }
        return new DescribeCasesPublisher(client, firstRequest, true) {
            @Override
            public void subscribe(Subscriber<? super DescribeCasesResponse> subscriber) {
                subscriber.onSubscribe(new EmptySubscription(subscriber));
            }
        };
    }

    private class DescribeCasesResponseFetcher implements AsyncPageFetcher<DescribeCasesResponse> {
        @Override
        public boolean hasNextPage(final DescribeCasesResponse previousPage) {
            return previousPage.nextToken() != null && !SdkAutoConstructList.class.isInstance(previousPage.nextToken())
                    && !SdkAutoConstructMap.class.isInstance(previousPage.nextToken());
        }

        @Override
        public CompletableFuture<DescribeCasesResponse> nextPage(final DescribeCasesResponse previousPage) {
            if (previousPage == null) {
                return client.describeCases(firstRequest);
            }
            return client.describeCases(firstRequest.toBuilder().nextToken(previousPage.nextToken()).build());
        }
    }
}
