/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.support;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.support.model.AddAttachmentsToSetRequest;
import software.amazon.awssdk.services.support.model.AddAttachmentsToSetResponse;
import software.amazon.awssdk.services.support.model.AddCommunicationToCaseRequest;
import software.amazon.awssdk.services.support.model.AddCommunicationToCaseResponse;
import software.amazon.awssdk.services.support.model.AttachmentIdNotFoundException;
import software.amazon.awssdk.services.support.model.AttachmentLimitExceededException;
import software.amazon.awssdk.services.support.model.AttachmentSetExpiredException;
import software.amazon.awssdk.services.support.model.AttachmentSetIdNotFoundException;
import software.amazon.awssdk.services.support.model.AttachmentSetSizeLimitExceededException;
import software.amazon.awssdk.services.support.model.CaseCreationLimitExceededException;
import software.amazon.awssdk.services.support.model.CaseIdNotFoundException;
import software.amazon.awssdk.services.support.model.CreateCaseRequest;
import software.amazon.awssdk.services.support.model.CreateCaseResponse;
import software.amazon.awssdk.services.support.model.DescribeAttachmentLimitExceededException;
import software.amazon.awssdk.services.support.model.DescribeAttachmentRequest;
import software.amazon.awssdk.services.support.model.DescribeAttachmentResponse;
import software.amazon.awssdk.services.support.model.DescribeCasesRequest;
import software.amazon.awssdk.services.support.model.DescribeCasesResponse;
import software.amazon.awssdk.services.support.model.DescribeCommunicationsRequest;
import software.amazon.awssdk.services.support.model.DescribeCommunicationsResponse;
import software.amazon.awssdk.services.support.model.DescribeServicesRequest;
import software.amazon.awssdk.services.support.model.DescribeServicesResponse;
import software.amazon.awssdk.services.support.model.DescribeSeverityLevelsRequest;
import software.amazon.awssdk.services.support.model.DescribeSeverityLevelsResponse;
import software.amazon.awssdk.services.support.model.DescribeTrustedAdvisorCheckRefreshStatusesRequest;
import software.amazon.awssdk.services.support.model.DescribeTrustedAdvisorCheckRefreshStatusesResponse;
import software.amazon.awssdk.services.support.model.DescribeTrustedAdvisorCheckResultRequest;
import software.amazon.awssdk.services.support.model.DescribeTrustedAdvisorCheckResultResponse;
import software.amazon.awssdk.services.support.model.DescribeTrustedAdvisorCheckSummariesRequest;
import software.amazon.awssdk.services.support.model.DescribeTrustedAdvisorCheckSummariesResponse;
import software.amazon.awssdk.services.support.model.DescribeTrustedAdvisorChecksRequest;
import software.amazon.awssdk.services.support.model.DescribeTrustedAdvisorChecksResponse;
import software.amazon.awssdk.services.support.model.InternalServerErrorException;
import software.amazon.awssdk.services.support.model.RefreshTrustedAdvisorCheckRequest;
import software.amazon.awssdk.services.support.model.RefreshTrustedAdvisorCheckResponse;
import software.amazon.awssdk.services.support.model.ResolveCaseRequest;
import software.amazon.awssdk.services.support.model.ResolveCaseResponse;
import software.amazon.awssdk.services.support.model.SupportException;
import software.amazon.awssdk.services.support.model.SupportRequest;
import software.amazon.awssdk.services.support.paginators.DescribeCasesPublisher;
import software.amazon.awssdk.services.support.paginators.DescribeCommunicationsPublisher;
import software.amazon.awssdk.services.support.transform.AddAttachmentsToSetRequestMarshaller;
import software.amazon.awssdk.services.support.transform.AddCommunicationToCaseRequestMarshaller;
import software.amazon.awssdk.services.support.transform.CreateCaseRequestMarshaller;
import software.amazon.awssdk.services.support.transform.DescribeAttachmentRequestMarshaller;
import software.amazon.awssdk.services.support.transform.DescribeCasesRequestMarshaller;
import software.amazon.awssdk.services.support.transform.DescribeCommunicationsRequestMarshaller;
import software.amazon.awssdk.services.support.transform.DescribeServicesRequestMarshaller;
import software.amazon.awssdk.services.support.transform.DescribeSeverityLevelsRequestMarshaller;
import software.amazon.awssdk.services.support.transform.DescribeTrustedAdvisorCheckRefreshStatusesRequestMarshaller;
import software.amazon.awssdk.services.support.transform.DescribeTrustedAdvisorCheckResultRequestMarshaller;
import software.amazon.awssdk.services.support.transform.DescribeTrustedAdvisorCheckSummariesRequestMarshaller;
import software.amazon.awssdk.services.support.transform.DescribeTrustedAdvisorChecksRequestMarshaller;
import software.amazon.awssdk.services.support.transform.RefreshTrustedAdvisorCheckRequestMarshaller;
import software.amazon.awssdk.services.support.transform.ResolveCaseRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link SupportAsyncClient}.
 *
 * @see SupportAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultSupportAsyncClient implements SupportAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultSupportAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultSupportAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Adds one or more attachments to an attachment set.
     * </p>
     * <p>
     * An attachment set is a temporary container for attachments that you add to a case or case communication. The set
     * is available for 1 hour after it's created. The <code>expiryTime</code> returned in the response is when the set
     * expires.
     * </p>
     * <note>
     * <ul>
     * <li>
     * <p>
     * You must have a Business, Enterprise On-Ramp, or Enterprise Support plan to use the Amazon Web Services Support
     * API.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you call the Amazon Web Services Support API from an account that does not have a Business, Enterprise
     * On-Ramp, or Enterprise Support plan, the <code>SubscriptionRequiredException</code> error message appears. For
     * information about changing your support plan, see <a href="http://aws.amazon.com/premiumsupport/">Amazon Web
     * Services Support</a>.
     * </p>
     * </li>
     * </ul>
     * </note>
     *
     * @param addAttachmentsToSetRequest
     * @return A Java Future containing the result of the AddAttachmentsToSet operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerErrorException An internal server error occurred.</li>
     *         <li>AttachmentSetIdNotFoundException An attachment set with the specified ID could not be found.</li>
     *         <li>AttachmentSetExpiredException The expiration time of the attachment set has passed. The set expires 1
     *         hour after it is created.</li>
     *         <li>AttachmentSetSizeLimitExceededException A limit for the size of an attachment set has been exceeded.
     *         The limits are three attachments and 5 MB per attachment.</li>
     *         <li>AttachmentLimitExceededException The limit for the number of attachment sets created in a short
     *         period of time has been exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SupportException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SupportAsyncClient.AddAttachmentsToSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/support-2013-04-15/AddAttachmentsToSet" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<AddAttachmentsToSetResponse> addAttachmentsToSet(
            AddAttachmentsToSetRequest addAttachmentsToSetRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, addAttachmentsToSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Support");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AddAttachmentsToSet");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AddAttachmentsToSetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AddAttachmentsToSetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AddAttachmentsToSetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AddAttachmentsToSetRequest, AddAttachmentsToSetResponse>()
                            .withOperationName("AddAttachmentsToSet")
                            .withMarshaller(new AddAttachmentsToSetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(addAttachmentsToSetRequest));
            CompletableFuture<AddAttachmentsToSetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds additional customer communication to an Amazon Web Services Support case. Use the <code>caseId</code>
     * parameter to identify the case to which to add communication. You can list a set of email addresses to copy on
     * the communication by using the <code>ccEmailAddresses</code> parameter. The <code>communicationBody</code> value
     * contains the text of the communication.
     * </p>
     * <note>
     * <ul>
     * <li>
     * <p>
     * You must have a Business, Enterprise On-Ramp, or Enterprise Support plan to use the Amazon Web Services Support
     * API.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you call the Amazon Web Services Support API from an account that does not have a Business, Enterprise
     * On-Ramp, or Enterprise Support plan, the <code>SubscriptionRequiredException</code> error message appears. For
     * information about changing your support plan, see <a href="http://aws.amazon.com/premiumsupport/">Amazon Web
     * Services Support</a>.
     * </p>
     * </li>
     * </ul>
     * </note>
     *
     * @param addCommunicationToCaseRequest
     * @return A Java Future containing the result of the AddCommunicationToCase operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerErrorException An internal server error occurred.</li>
     *         <li>CaseIdNotFoundException The requested <code>caseId</code> couldn't be located.</li>
     *         <li>AttachmentSetIdNotFoundException An attachment set with the specified ID could not be found.</li>
     *         <li>AttachmentSetExpiredException The expiration time of the attachment set has passed. The set expires 1
     *         hour after it is created.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SupportException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SupportAsyncClient.AddCommunicationToCase
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/support-2013-04-15/AddCommunicationToCase"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AddCommunicationToCaseResponse> addCommunicationToCase(
            AddCommunicationToCaseRequest addCommunicationToCaseRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, addCommunicationToCaseRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Support");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AddCommunicationToCase");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AddCommunicationToCaseResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AddCommunicationToCaseResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AddCommunicationToCaseResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AddCommunicationToCaseRequest, AddCommunicationToCaseResponse>()
                            .withOperationName("AddCommunicationToCase")
                            .withMarshaller(new AddCommunicationToCaseRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(addCommunicationToCaseRequest));
            CompletableFuture<AddCommunicationToCaseResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a case in the Amazon Web Services Support Center. This operation is similar to how you create a case in
     * the Amazon Web Services Support Center <a href="https://console.aws.amazon.com/support/home#/case/create">Create
     * Case</a> page.
     * </p>
     * <p>
     * The Amazon Web Services Support API doesn't support requesting service limit increases. You can submit a service
     * limit increase in the following ways:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Submit a request from the Amazon Web Services Support Center <a
     * href="https://console.aws.amazon.com/support/home#/case/create">Create Case</a> page.
     * </p>
     * </li>
     * <li>
     * <p>
     * Use the Service Quotas <a
     * href="https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_RequestServiceQuotaIncrease.html"
     * >RequestServiceQuotaIncrease</a> operation.
     * </p>
     * </li>
     * </ul>
     * <p>
     * A successful <code>CreateCase</code> request returns an Amazon Web Services Support case number. You can use the
     * <a>DescribeCases</a> operation and specify the case number to get existing Amazon Web Services Support cases.
     * After you create a case, use the <a>AddCommunicationToCase</a> operation to add additional communication or
     * attachments to an existing case.
     * </p>
     * <p>
     * The <code>caseId</code> is separate from the <code>displayId</code> that appears in the <a
     * href="https://console.aws.amazon.com/support">Amazon Web Services Support Center</a>. Use the
     * <a>DescribeCases</a> operation to get the <code>displayId</code>.
     * </p>
     * <note>
     * <ul>
     * <li>
     * <p>
     * You must have a Business, Enterprise On-Ramp, or Enterprise Support plan to use the Amazon Web Services Support
     * API.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you call the Amazon Web Services Support API from an account that does not have a Business, Enterprise
     * On-Ramp, or Enterprise Support plan, the <code>SubscriptionRequiredException</code> error message appears. For
     * information about changing your support plan, see <a href="http://aws.amazon.com/premiumsupport/">Amazon Web
     * Services Support</a>.
     * </p>
     * </li>
     * </ul>
     * </note>
     *
     * @param createCaseRequest
     * @return A Java Future containing the result of the CreateCase operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerErrorException An internal server error occurred.</li>
     *         <li>CaseCreationLimitExceededException The case creation limit for the account has been exceeded.</li>
     *         <li>AttachmentSetIdNotFoundException An attachment set with the specified ID could not be found.</li>
     *         <li>AttachmentSetExpiredException The expiration time of the attachment set has passed. The set expires 1
     *         hour after it is created.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SupportException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SupportAsyncClient.CreateCase
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/support-2013-04-15/CreateCase" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateCaseResponse> createCase(CreateCaseRequest createCaseRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createCaseRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Support");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateCase");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateCaseResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateCaseResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateCaseResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateCaseRequest, CreateCaseResponse>().withOperationName("CreateCase")
                            .withMarshaller(new CreateCaseRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createCaseRequest));
            CompletableFuture<CreateCaseResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the attachment that has the specified ID. Attachments can include screenshots, error logs, or other files
     * that describe your issue. Attachment IDs are generated by the case management system when you add an attachment
     * to a case or case communication. Attachment IDs are returned in the <a>AttachmentDetails</a> objects that are
     * returned by the <a>DescribeCommunications</a> operation.
     * </p>
     * <note>
     * <ul>
     * <li>
     * <p>
     * You must have a Business, Enterprise On-Ramp, or Enterprise Support plan to use the Amazon Web Services Support
     * API.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you call the Amazon Web Services Support API from an account that does not have a Business, Enterprise
     * On-Ramp, or Enterprise Support plan, the <code>SubscriptionRequiredException</code> error message appears. For
     * information about changing your support plan, see <a href="http://aws.amazon.com/premiumsupport/">Amazon Web
     * Services Support</a>.
     * </p>
     * </li>
     * </ul>
     * </note>
     *
     * @param describeAttachmentRequest
     * @return A Java Future containing the result of the DescribeAttachment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerErrorException An internal server error occurred.</li>
     *         <li>DescribeAttachmentLimitExceededException The limit for the number of <a>DescribeAttachment</a>
     *         requests in a short period of time has been exceeded.</li>
     *         <li>AttachmentIdNotFoundException An attachment with the specified ID could not be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SupportException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SupportAsyncClient.DescribeAttachment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/support-2013-04-15/DescribeAttachment" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeAttachmentResponse> describeAttachment(DescribeAttachmentRequest describeAttachmentRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAttachmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Support");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAttachment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeAttachmentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeAttachmentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeAttachmentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeAttachmentRequest, DescribeAttachmentResponse>()
                            .withOperationName("DescribeAttachment")
                            .withMarshaller(new DescribeAttachmentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeAttachmentRequest));
            CompletableFuture<DescribeAttachmentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of cases that you specify by passing one or more case IDs. You can use the <code>afterTime</code>
     * and <code>beforeTime</code> parameters to filter the cases by date. You can set values for the
     * <code>includeResolvedCases</code> and <code>includeCommunications</code> parameters to specify how much
     * information to return.
     * </p>
     * <p>
     * The response returns the following in JSON format:
     * </p>
     * <ul>
     * <li>
     * <p>
     * One or more <a
     * href="https://docs.aws.amazon.com/awssupport/latest/APIReference/API_CaseDetails.html">CaseDetails</a> data
     * types.
     * </p>
     * </li>
     * <li>
     * <p>
     * One or more <code>nextToken</code> values, which specify where to paginate the returned records represented by
     * the <code>CaseDetails</code> objects.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Case data is available for 12 months after creation. If a case was created more than 12 months ago, a request
     * might return an error.
     * </p>
     * <note>
     * <ul>
     * <li>
     * <p>
     * You must have a Business, Enterprise On-Ramp, or Enterprise Support plan to use the Amazon Web Services Support
     * API.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you call the Amazon Web Services Support API from an account that does not have a Business, Enterprise
     * On-Ramp, or Enterprise Support plan, the <code>SubscriptionRequiredException</code> error message appears. For
     * information about changing your support plan, see <a href="http://aws.amazon.com/premiumsupport/">Amazon Web
     * Services Support</a>.
     * </p>
     * </li>
     * </ul>
     * </note>
     *
     * @param describeCasesRequest
     * @return A Java Future containing the result of the DescribeCases operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerErrorException An internal server error occurred.</li>
     *         <li>CaseIdNotFoundException The requested <code>caseId</code> couldn't be located.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SupportException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SupportAsyncClient.DescribeCases
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/support-2013-04-15/DescribeCases" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeCasesResponse> describeCases(DescribeCasesRequest describeCasesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeCasesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Support");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeCases");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeCasesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DescribeCasesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeCasesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeCasesRequest, DescribeCasesResponse>()
                            .withOperationName("DescribeCases")
                            .withMarshaller(new DescribeCasesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeCasesRequest));
            CompletableFuture<DescribeCasesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of cases that you specify by passing one or more case IDs. You can use the <code>afterTime</code>
     * and <code>beforeTime</code> parameters to filter the cases by date. You can set values for the
     * <code>includeResolvedCases</code> and <code>includeCommunications</code> parameters to specify how much
     * information to return.
     * </p>
     * <p>
     * The response returns the following in JSON format:
     * </p>
     * <ul>
     * <li>
     * <p>
     * One or more <a
     * href="https://docs.aws.amazon.com/awssupport/latest/APIReference/API_CaseDetails.html">CaseDetails</a> data
     * types.
     * </p>
     * </li>
     * <li>
     * <p>
     * One or more <code>nextToken</code> values, which specify where to paginate the returned records represented by
     * the <code>CaseDetails</code> objects.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Case data is available for 12 months after creation. If a case was created more than 12 months ago, a request
     * might return an error.
     * </p>
     * <note>
     * <ul>
     * <li>
     * <p>
     * You must have a Business, Enterprise On-Ramp, or Enterprise Support plan to use the Amazon Web Services Support
     * API.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you call the Amazon Web Services Support API from an account that does not have a Business, Enterprise
     * On-Ramp, or Enterprise Support plan, the <code>SubscriptionRequiredException</code> error message appears. For
     * information about changing your support plan, see <a href="http://aws.amazon.com/premiumsupport/">Amazon Web
     * Services Support</a>.
     * </p>
     * </li>
     * </ul>
     * </note><br/>
     * <p>
     * This is a variant of {@link #describeCases(software.amazon.awssdk.services.support.model.DescribeCasesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.support.paginators.DescribeCasesPublisher publisher = client.describeCasesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.support.paginators.DescribeCasesPublisher publisher = client.describeCasesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.support.model.DescribeCasesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.support.model.DescribeCasesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeCases(software.amazon.awssdk.services.support.model.DescribeCasesRequest)} operation.</b>
     * </p>
     *
     * @param describeCasesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerErrorException An internal server error occurred.</li>
     *         <li>CaseIdNotFoundException The requested <code>caseId</code> couldn't be located.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SupportException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SupportAsyncClient.DescribeCases
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/support-2013-04-15/DescribeCases" target="_top">AWS API
     *      Documentation</a>
     */
    public DescribeCasesPublisher describeCasesPaginator(DescribeCasesRequest describeCasesRequest) {
        return new DescribeCasesPublisher(this, applyPaginatorUserAgent(describeCasesRequest));
    }

    /**
     * <p>
     * Returns communications and attachments for one or more support cases. Use the <code>afterTime</code> and
     * <code>beforeTime</code> parameters to filter by date. You can use the <code>caseId</code> parameter to restrict
     * the results to a specific case.
     * </p>
     * <p>
     * Case data is available for 12 months after creation. If a case was created more than 12 months ago, a request for
     * data might cause an error.
     * </p>
     * <p>
     * You can use the <code>maxResults</code> and <code>nextToken</code> parameters to control the pagination of the
     * results. Set <code>maxResults</code> to the number of cases that you want to display on each page, and use
     * <code>nextToken</code> to specify the resumption of pagination.
     * </p>
     * <note>
     * <ul>
     * <li>
     * <p>
     * You must have a Business, Enterprise On-Ramp, or Enterprise Support plan to use the Amazon Web Services Support
     * API.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you call the Amazon Web Services Support API from an account that does not have a Business, Enterprise
     * On-Ramp, or Enterprise Support plan, the <code>SubscriptionRequiredException</code> error message appears. For
     * information about changing your support plan, see <a href="http://aws.amazon.com/premiumsupport/">Amazon Web
     * Services Support</a>.
     * </p>
     * </li>
     * </ul>
     * </note>
     *
     * @param describeCommunicationsRequest
     * @return A Java Future containing the result of the DescribeCommunications operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerErrorException An internal server error occurred.</li>
     *         <li>CaseIdNotFoundException The requested <code>caseId</code> couldn't be located.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SupportException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SupportAsyncClient.DescribeCommunications
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/support-2013-04-15/DescribeCommunications"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeCommunicationsResponse> describeCommunications(
            DescribeCommunicationsRequest describeCommunicationsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeCommunicationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Support");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeCommunications");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeCommunicationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeCommunicationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeCommunicationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeCommunicationsRequest, DescribeCommunicationsResponse>()
                            .withOperationName("DescribeCommunications")
                            .withMarshaller(new DescribeCommunicationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeCommunicationsRequest));
            CompletableFuture<DescribeCommunicationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns communications and attachments for one or more support cases. Use the <code>afterTime</code> and
     * <code>beforeTime</code> parameters to filter by date. You can use the <code>caseId</code> parameter to restrict
     * the results to a specific case.
     * </p>
     * <p>
     * Case data is available for 12 months after creation. If a case was created more than 12 months ago, a request for
     * data might cause an error.
     * </p>
     * <p>
     * You can use the <code>maxResults</code> and <code>nextToken</code> parameters to control the pagination of the
     * results. Set <code>maxResults</code> to the number of cases that you want to display on each page, and use
     * <code>nextToken</code> to specify the resumption of pagination.
     * </p>
     * <note>
     * <ul>
     * <li>
     * <p>
     * You must have a Business, Enterprise On-Ramp, or Enterprise Support plan to use the Amazon Web Services Support
     * API.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you call the Amazon Web Services Support API from an account that does not have a Business, Enterprise
     * On-Ramp, or Enterprise Support plan, the <code>SubscriptionRequiredException</code> error message appears. For
     * information about changing your support plan, see <a href="http://aws.amazon.com/premiumsupport/">Amazon Web
     * Services Support</a>.
     * </p>
     * </li>
     * </ul>
     * </note><br/>
     * <p>
     * This is a variant of
     * {@link #describeCommunications(software.amazon.awssdk.services.support.model.DescribeCommunicationsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.support.paginators.DescribeCommunicationsPublisher publisher = client.describeCommunicationsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.support.paginators.DescribeCommunicationsPublisher publisher = client.describeCommunicationsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.support.model.DescribeCommunicationsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.support.model.DescribeCommunicationsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeCommunications(software.amazon.awssdk.services.support.model.DescribeCommunicationsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeCommunicationsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerErrorException An internal server error occurred.</li>
     *         <li>CaseIdNotFoundException The requested <code>caseId</code> couldn't be located.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SupportException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SupportAsyncClient.DescribeCommunications
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/support-2013-04-15/DescribeCommunications"
     *      target="_top">AWS API Documentation</a>
     */
    public DescribeCommunicationsPublisher describeCommunicationsPaginator(
            DescribeCommunicationsRequest describeCommunicationsRequest) {
        return new DescribeCommunicationsPublisher(this, applyPaginatorUserAgent(describeCommunicationsRequest));
    }

    /**
     * <p>
     * Returns the current list of Amazon Web Services services and a list of service categories for each service. You
     * then use service names and categories in your <a>CreateCase</a> requests. Each Amazon Web Services service has
     * its own set of categories.
     * </p>
     * <p>
     * The service codes and category codes correspond to the values that appear in the <b>Service</b> and
     * <b>Category</b> lists on the Amazon Web Services Support Center <a
     * href="https://console.aws.amazon.com/support/home#/case/create">Create Case</a> page. The values in those fields
     * don't necessarily match the service codes and categories returned by the <code>DescribeServices</code> operation.
     * Always use the service codes and categories that the <code>DescribeServices</code> operation returns, so that you
     * have the most recent set of service and category codes.
     * </p>
     * <note>
     * <ul>
     * <li>
     * <p>
     * You must have a Business, Enterprise On-Ramp, or Enterprise Support plan to use the Amazon Web Services Support
     * API.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you call the Amazon Web Services Support API from an account that does not have a Business, Enterprise
     * On-Ramp, or Enterprise Support plan, the <code>SubscriptionRequiredException</code> error message appears. For
     * information about changing your support plan, see <a href="http://aws.amazon.com/premiumsupport/">Amazon Web
     * Services Support</a>.
     * </p>
     * </li>
     * </ul>
     * </note>
     *
     * @param describeServicesRequest
     * @return A Java Future containing the result of the DescribeServices operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerErrorException An internal server error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SupportException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SupportAsyncClient.DescribeServices
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/support-2013-04-15/DescribeServices" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeServicesResponse> describeServices(DescribeServicesRequest describeServicesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeServicesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Support");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeServices");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeServicesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeServicesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeServicesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeServicesRequest, DescribeServicesResponse>()
                            .withOperationName("DescribeServices")
                            .withMarshaller(new DescribeServicesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeServicesRequest));
            CompletableFuture<DescribeServicesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the list of severity levels that you can assign to a support case. The severity level for a case is also
     * a field in the <a>CaseDetails</a> data type that you include for a <a>CreateCase</a> request.
     * </p>
     * <note>
     * <ul>
     * <li>
     * <p>
     * You must have a Business, Enterprise On-Ramp, or Enterprise Support plan to use the Amazon Web Services Support
     * API.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you call the Amazon Web Services Support API from an account that does not have a Business, Enterprise
     * On-Ramp, or Enterprise Support plan, the <code>SubscriptionRequiredException</code> error message appears. For
     * information about changing your support plan, see <a href="http://aws.amazon.com/premiumsupport/">Amazon Web
     * Services Support</a>.
     * </p>
     * </li>
     * </ul>
     * </note>
     *
     * @param describeSeverityLevelsRequest
     * @return A Java Future containing the result of the DescribeSeverityLevels operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerErrorException An internal server error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SupportException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SupportAsyncClient.DescribeSeverityLevels
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/support-2013-04-15/DescribeSeverityLevels"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeSeverityLevelsResponse> describeSeverityLevels(
            DescribeSeverityLevelsRequest describeSeverityLevelsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeSeverityLevelsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Support");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeSeverityLevels");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeSeverityLevelsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeSeverityLevelsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeSeverityLevelsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeSeverityLevelsRequest, DescribeSeverityLevelsResponse>()
                            .withOperationName("DescribeSeverityLevels")
                            .withMarshaller(new DescribeSeverityLevelsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeSeverityLevelsRequest));
            CompletableFuture<DescribeSeverityLevelsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the refresh status of the Trusted Advisor checks that have the specified check IDs. You can get the check
     * IDs by calling the <a>DescribeTrustedAdvisorChecks</a> operation.
     * </p>
     * <p>
     * Some checks are refreshed automatically, and you can't return their refresh statuses by using the
     * <code>DescribeTrustedAdvisorCheckRefreshStatuses</code> operation. If you call this operation for these checks,
     * you might see an <code>InvalidParameterValue</code> error.
     * </p>
     * <note>
     * <ul>
     * <li>
     * <p>
     * You must have a Business, Enterprise On-Ramp, or Enterprise Support plan to use the Amazon Web Services Support
     * API.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you call the Amazon Web Services Support API from an account that does not have a Business, Enterprise
     * On-Ramp, or Enterprise Support plan, the <code>SubscriptionRequiredException</code> error message appears. For
     * information about changing your support plan, see <a href="http://aws.amazon.com/premiumsupport/">Amazon Web
     * Services Support</a>.
     * </p>
     * </li>
     * </ul>
     * </note>
     *
     * @param describeTrustedAdvisorCheckRefreshStatusesRequest
     * @return A Java Future containing the result of the DescribeTrustedAdvisorCheckRefreshStatuses operation returned
     *         by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerErrorException An internal server error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SupportException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SupportAsyncClient.DescribeTrustedAdvisorCheckRefreshStatuses
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/support-2013-04-15/DescribeTrustedAdvisorCheckRefreshStatuses"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeTrustedAdvisorCheckRefreshStatusesResponse> describeTrustedAdvisorCheckRefreshStatuses(
            DescribeTrustedAdvisorCheckRefreshStatusesRequest describeTrustedAdvisorCheckRefreshStatusesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeTrustedAdvisorCheckRefreshStatusesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Support");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeTrustedAdvisorCheckRefreshStatuses");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeTrustedAdvisorCheckRefreshStatusesResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DescribeTrustedAdvisorCheckRefreshStatusesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeTrustedAdvisorCheckRefreshStatusesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeTrustedAdvisorCheckRefreshStatusesRequest, DescribeTrustedAdvisorCheckRefreshStatusesResponse>()
                            .withOperationName("DescribeTrustedAdvisorCheckRefreshStatuses")
                            .withMarshaller(new DescribeTrustedAdvisorCheckRefreshStatusesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector)
                            .withInput(describeTrustedAdvisorCheckRefreshStatusesRequest));
            CompletableFuture<DescribeTrustedAdvisorCheckRefreshStatusesResponse> whenCompleted = executeFuture.whenComplete((r,
                    e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the results of the Trusted Advisor check that has the specified check ID. You can get the check IDs by
     * calling the <a>DescribeTrustedAdvisorChecks</a> operation.
     * </p>
     * <p>
     * The response contains a <a>TrustedAdvisorCheckResult</a> object, which contains these three objects:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>TrustedAdvisorCategorySpecificSummary</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>TrustedAdvisorResourceDetail</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>TrustedAdvisorResourcesSummary</a>
     * </p>
     * </li>
     * </ul>
     * <p>
     * In addition, the response contains these fields:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>status</b> - The alert status of the check can be <code>ok</code> (green), <code>warning</code> (yellow),
     * <code>error</code> (red), or <code>not_available</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>timestamp</b> - The time of the last refresh of the check.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>checkId</b> - The unique identifier for the check.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <ul>
     * <li>
     * <p>
     * You must have a Business, Enterprise On-Ramp, or Enterprise Support plan to use the Amazon Web Services Support
     * API.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you call the Amazon Web Services Support API from an account that does not have a Business, Enterprise
     * On-Ramp, or Enterprise Support plan, the <code>SubscriptionRequiredException</code> error message appears. For
     * information about changing your support plan, see <a href="http://aws.amazon.com/premiumsupport/">Amazon Web
     * Services Support</a>.
     * </p>
     * </li>
     * </ul>
     * </note>
     *
     * @param describeTrustedAdvisorCheckResultRequest
     * @return A Java Future containing the result of the DescribeTrustedAdvisorCheckResult operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerErrorException An internal server error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SupportException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SupportAsyncClient.DescribeTrustedAdvisorCheckResult
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/support-2013-04-15/DescribeTrustedAdvisorCheckResult"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeTrustedAdvisorCheckResultResponse> describeTrustedAdvisorCheckResult(
            DescribeTrustedAdvisorCheckResultRequest describeTrustedAdvisorCheckResultRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeTrustedAdvisorCheckResultRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Support");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeTrustedAdvisorCheckResult");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeTrustedAdvisorCheckResultResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DescribeTrustedAdvisorCheckResultResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeTrustedAdvisorCheckResultResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeTrustedAdvisorCheckResultRequest, DescribeTrustedAdvisorCheckResultResponse>()
                            .withOperationName("DescribeTrustedAdvisorCheckResult")
                            .withMarshaller(new DescribeTrustedAdvisorCheckResultRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeTrustedAdvisorCheckResultRequest));
            CompletableFuture<DescribeTrustedAdvisorCheckResultResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the results for the Trusted Advisor check summaries for the check IDs that you specified. You can get the
     * check IDs by calling the <a>DescribeTrustedAdvisorChecks</a> operation.
     * </p>
     * <p>
     * The response contains an array of <a>TrustedAdvisorCheckSummary</a> objects.
     * </p>
     * <note>
     * <ul>
     * <li>
     * <p>
     * You must have a Business, Enterprise On-Ramp, or Enterprise Support plan to use the Amazon Web Services Support
     * API.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you call the Amazon Web Services Support API from an account that does not have a Business, Enterprise
     * On-Ramp, or Enterprise Support plan, the <code>SubscriptionRequiredException</code> error message appears. For
     * information about changing your support plan, see <a href="http://aws.amazon.com/premiumsupport/">Amazon Web
     * Services Support</a>.
     * </p>
     * </li>
     * </ul>
     * </note>
     *
     * @param describeTrustedAdvisorCheckSummariesRequest
     * @return A Java Future containing the result of the DescribeTrustedAdvisorCheckSummaries operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerErrorException An internal server error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SupportException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SupportAsyncClient.DescribeTrustedAdvisorCheckSummaries
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/support-2013-04-15/DescribeTrustedAdvisorCheckSummaries"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeTrustedAdvisorCheckSummariesResponse> describeTrustedAdvisorCheckSummaries(
            DescribeTrustedAdvisorCheckSummariesRequest describeTrustedAdvisorCheckSummariesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeTrustedAdvisorCheckSummariesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Support");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeTrustedAdvisorCheckSummaries");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeTrustedAdvisorCheckSummariesResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DescribeTrustedAdvisorCheckSummariesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeTrustedAdvisorCheckSummariesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeTrustedAdvisorCheckSummariesRequest, DescribeTrustedAdvisorCheckSummariesResponse>()
                            .withOperationName("DescribeTrustedAdvisorCheckSummaries")
                            .withMarshaller(new DescribeTrustedAdvisorCheckSummariesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeTrustedAdvisorCheckSummariesRequest));
            CompletableFuture<DescribeTrustedAdvisorCheckSummariesResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about all available Trusted Advisor checks, including the name, ID, category, description,
     * and metadata. You must specify a language code. The Amazon Web Services Support API currently supports English
     * ("en") and Japanese ("ja"). The response contains a <a>TrustedAdvisorCheckDescription</a> object for each check.
     * You must set the Amazon Web Services Region to us-east-1.
     * </p>
     * <note>
     * <ul>
     * <li>
     * <p>
     * You must have a Business, Enterprise On-Ramp, or Enterprise Support plan to use the Amazon Web Services Support
     * API.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you call the Amazon Web Services Support API from an account that does not have a Business, Enterprise
     * On-Ramp, or Enterprise Support plan, the <code>SubscriptionRequiredException</code> error message appears. For
     * information about changing your support plan, see <a href="http://aws.amazon.com/premiumsupport/">Amazon Web
     * Services Support</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * The names and descriptions for Trusted Advisor checks are subject to change. We recommend that you specify the
     * check ID in your code to uniquely identify a check.
     * </p>
     * </li>
     * </ul>
     * </note>
     *
     * @param describeTrustedAdvisorChecksRequest
     * @return A Java Future containing the result of the DescribeTrustedAdvisorChecks operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerErrorException An internal server error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SupportException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SupportAsyncClient.DescribeTrustedAdvisorChecks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/support-2013-04-15/DescribeTrustedAdvisorChecks"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeTrustedAdvisorChecksResponse> describeTrustedAdvisorChecks(
            DescribeTrustedAdvisorChecksRequest describeTrustedAdvisorChecksRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeTrustedAdvisorChecksRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Support");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeTrustedAdvisorChecks");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeTrustedAdvisorChecksResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeTrustedAdvisorChecksResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeTrustedAdvisorChecksResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeTrustedAdvisorChecksRequest, DescribeTrustedAdvisorChecksResponse>()
                            .withOperationName("DescribeTrustedAdvisorChecks")
                            .withMarshaller(new DescribeTrustedAdvisorChecksRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeTrustedAdvisorChecksRequest));
            CompletableFuture<DescribeTrustedAdvisorChecksResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Refreshes the Trusted Advisor check that you specify using the check ID. You can get the check IDs by calling the
     * <a>DescribeTrustedAdvisorChecks</a> operation.
     * </p>
     * <note>
     * <p>
     * Some checks are refreshed automatically. If you call the <code>RefreshTrustedAdvisorCheck</code> operation to
     * refresh them, you might see the <code>InvalidParameterValue</code> error.
     * </p>
     * </note>
     * <p>
     * The response contains a <a>TrustedAdvisorCheckRefreshStatus</a> object.
     * </p>
     * <note>
     * <ul>
     * <li>
     * <p>
     * You must have a Business, Enterprise On-Ramp, or Enterprise Support plan to use the Amazon Web Services Support
     * API.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you call the Amazon Web Services Support API from an account that does not have a Business, Enterprise
     * On-Ramp, or Enterprise Support plan, the <code>SubscriptionRequiredException</code> error message appears. For
     * information about changing your support plan, see <a href="http://aws.amazon.com/premiumsupport/">Amazon Web
     * Services Support</a>.
     * </p>
     * </li>
     * </ul>
     * </note>
     *
     * @param refreshTrustedAdvisorCheckRequest
     * @return A Java Future containing the result of the RefreshTrustedAdvisorCheck operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerErrorException An internal server error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SupportException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SupportAsyncClient.RefreshTrustedAdvisorCheck
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/support-2013-04-15/RefreshTrustedAdvisorCheck"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<RefreshTrustedAdvisorCheckResponse> refreshTrustedAdvisorCheck(
            RefreshTrustedAdvisorCheckRequest refreshTrustedAdvisorCheckRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, refreshTrustedAdvisorCheckRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Support");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RefreshTrustedAdvisorCheck");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RefreshTrustedAdvisorCheckResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RefreshTrustedAdvisorCheckResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RefreshTrustedAdvisorCheckResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RefreshTrustedAdvisorCheckRequest, RefreshTrustedAdvisorCheckResponse>()
                            .withOperationName("RefreshTrustedAdvisorCheck")
                            .withMarshaller(new RefreshTrustedAdvisorCheckRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(refreshTrustedAdvisorCheckRequest));
            CompletableFuture<RefreshTrustedAdvisorCheckResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Resolves a support case. This operation takes a <code>caseId</code> and returns the initial and final state of
     * the case.
     * </p>
     * <note>
     * <ul>
     * <li>
     * <p>
     * You must have a Business, Enterprise On-Ramp, or Enterprise Support plan to use the Amazon Web Services Support
     * API.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you call the Amazon Web Services Support API from an account that does not have a Business, Enterprise
     * On-Ramp, or Enterprise Support plan, the <code>SubscriptionRequiredException</code> error message appears. For
     * information about changing your support plan, see <a href="http://aws.amazon.com/premiumsupport/">Amazon Web
     * Services Support</a>.
     * </p>
     * </li>
     * </ul>
     * </note>
     *
     * @param resolveCaseRequest
     * @return A Java Future containing the result of the ResolveCase operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerErrorException An internal server error occurred.</li>
     *         <li>CaseIdNotFoundException The requested <code>caseId</code> couldn't be located.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SupportException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SupportAsyncClient.ResolveCase
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/support-2013-04-15/ResolveCase" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ResolveCaseResponse> resolveCase(ResolveCaseRequest resolveCaseRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, resolveCaseRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Support");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ResolveCase");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ResolveCaseResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ResolveCaseResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ResolveCaseResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ResolveCaseRequest, ResolveCaseResponse>()
                            .withOperationName("ResolveCase").withMarshaller(new ResolveCaseRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(resolveCaseRequest));
            CompletableFuture<ResolveCaseResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(SupportException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AttachmentSetIdNotFound")
                                .exceptionBuilderSupplier(AttachmentSetIdNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CaseCreationLimitExceeded")
                                .exceptionBuilderSupplier(CaseCreationLimitExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AttachmentIdNotFound")
                                .exceptionBuilderSupplier(AttachmentIdNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AttachmentSetExpired")
                                .exceptionBuilderSupplier(AttachmentSetExpiredException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AttachmentLimitExceeded")
                                .exceptionBuilderSupplier(AttachmentLimitExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CaseIdNotFound")
                                .exceptionBuilderSupplier(CaseIdNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DescribeAttachmentLimitExceeded")
                                .exceptionBuilderSupplier(DescribeAttachmentLimitExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerError")
                                .exceptionBuilderSupplier(InternalServerErrorException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AttachmentSetSizeLimitExceeded")
                                .exceptionBuilderSupplier(AttachmentSetSizeLimitExceededException::builder).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends SupportRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
