/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.support;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.support.model.AddAttachmentsToSetRequest;
import software.amazon.awssdk.services.support.model.AddAttachmentsToSetResponse;
import software.amazon.awssdk.services.support.model.AddCommunicationToCaseRequest;
import software.amazon.awssdk.services.support.model.AddCommunicationToCaseResponse;
import software.amazon.awssdk.services.support.model.AttachmentIdNotFoundException;
import software.amazon.awssdk.services.support.model.AttachmentLimitExceededException;
import software.amazon.awssdk.services.support.model.AttachmentSetExpiredException;
import software.amazon.awssdk.services.support.model.AttachmentSetIdNotFoundException;
import software.amazon.awssdk.services.support.model.AttachmentSetSizeLimitExceededException;
import software.amazon.awssdk.services.support.model.CaseCreationLimitExceededException;
import software.amazon.awssdk.services.support.model.CaseIdNotFoundException;
import software.amazon.awssdk.services.support.model.CreateCaseRequest;
import software.amazon.awssdk.services.support.model.CreateCaseResponse;
import software.amazon.awssdk.services.support.model.DescribeAttachmentLimitExceededException;
import software.amazon.awssdk.services.support.model.DescribeAttachmentRequest;
import software.amazon.awssdk.services.support.model.DescribeAttachmentResponse;
import software.amazon.awssdk.services.support.model.DescribeCasesRequest;
import software.amazon.awssdk.services.support.model.DescribeCasesResponse;
import software.amazon.awssdk.services.support.model.DescribeCommunicationsRequest;
import software.amazon.awssdk.services.support.model.DescribeCommunicationsResponse;
import software.amazon.awssdk.services.support.model.DescribeServicesRequest;
import software.amazon.awssdk.services.support.model.DescribeServicesResponse;
import software.amazon.awssdk.services.support.model.DescribeSeverityLevelsRequest;
import software.amazon.awssdk.services.support.model.DescribeSeverityLevelsResponse;
import software.amazon.awssdk.services.support.model.DescribeTrustedAdvisorCheckRefreshStatusesRequest;
import software.amazon.awssdk.services.support.model.DescribeTrustedAdvisorCheckRefreshStatusesResponse;
import software.amazon.awssdk.services.support.model.DescribeTrustedAdvisorCheckResultRequest;
import software.amazon.awssdk.services.support.model.DescribeTrustedAdvisorCheckResultResponse;
import software.amazon.awssdk.services.support.model.DescribeTrustedAdvisorCheckSummariesRequest;
import software.amazon.awssdk.services.support.model.DescribeTrustedAdvisorCheckSummariesResponse;
import software.amazon.awssdk.services.support.model.DescribeTrustedAdvisorChecksRequest;
import software.amazon.awssdk.services.support.model.DescribeTrustedAdvisorChecksResponse;
import software.amazon.awssdk.services.support.model.InternalServerErrorException;
import software.amazon.awssdk.services.support.model.RefreshTrustedAdvisorCheckRequest;
import software.amazon.awssdk.services.support.model.RefreshTrustedAdvisorCheckResponse;
import software.amazon.awssdk.services.support.model.ResolveCaseRequest;
import software.amazon.awssdk.services.support.model.ResolveCaseResponse;
import software.amazon.awssdk.services.support.model.SupportException;
import software.amazon.awssdk.services.support.model.SupportRequest;
import software.amazon.awssdk.services.support.paginators.DescribeCasesIterable;
import software.amazon.awssdk.services.support.paginators.DescribeCommunicationsIterable;
import software.amazon.awssdk.services.support.transform.AddAttachmentsToSetRequestMarshaller;
import software.amazon.awssdk.services.support.transform.AddCommunicationToCaseRequestMarshaller;
import software.amazon.awssdk.services.support.transform.CreateCaseRequestMarshaller;
import software.amazon.awssdk.services.support.transform.DescribeAttachmentRequestMarshaller;
import software.amazon.awssdk.services.support.transform.DescribeCasesRequestMarshaller;
import software.amazon.awssdk.services.support.transform.DescribeCommunicationsRequestMarshaller;
import software.amazon.awssdk.services.support.transform.DescribeServicesRequestMarshaller;
import software.amazon.awssdk.services.support.transform.DescribeSeverityLevelsRequestMarshaller;
import software.amazon.awssdk.services.support.transform.DescribeTrustedAdvisorCheckRefreshStatusesRequestMarshaller;
import software.amazon.awssdk.services.support.transform.DescribeTrustedAdvisorCheckResultRequestMarshaller;
import software.amazon.awssdk.services.support.transform.DescribeTrustedAdvisorCheckSummariesRequestMarshaller;
import software.amazon.awssdk.services.support.transform.DescribeTrustedAdvisorChecksRequestMarshaller;
import software.amazon.awssdk.services.support.transform.RefreshTrustedAdvisorCheckRequestMarshaller;
import software.amazon.awssdk.services.support.transform.ResolveCaseRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link SupportClient}.
 *
 * @see SupportClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultSupportClient implements SupportClient {
    private static final Logger log = Logger.loggerFor(DefaultSupportClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultSupportClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Adds one or more attachments to an attachment set.
     * </p>
     * <p>
     * An attachment set is a temporary container for attachments that you add to a case or case communication. The set
     * is available for 1 hour after it's created. The <code>expiryTime</code> returned in the response is when the set
     * expires.
     * </p>
     * <note>
     * <ul>
     * <li>
     * <p>
     * You must have a Business, Enterprise On-Ramp, or Enterprise Support plan to use the Amazon Web Services Support
     * API.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you call the Amazon Web Services Support API from an account that does not have a Business, Enterprise
     * On-Ramp, or Enterprise Support plan, the <code>SubscriptionRequiredException</code> error message appears. For
     * information about changing your support plan, see <a href="http://aws.amazon.com/premiumsupport/">Amazon Web
     * Services Support</a>.
     * </p>
     * </li>
     * </ul>
     * </note>
     *
     * @param addAttachmentsToSetRequest
     * @return Result of the AddAttachmentsToSet operation returned by the service.
     * @throws InternalServerErrorException
     *         An internal server error occurred.
     * @throws AttachmentSetIdNotFoundException
     *         An attachment set with the specified ID could not be found.
     * @throws AttachmentSetExpiredException
     *         The expiration time of the attachment set has passed. The set expires 1 hour after it is created.
     * @throws AttachmentSetSizeLimitExceededException
     *         A limit for the size of an attachment set has been exceeded. The limits are three attachments and 5 MB
     *         per attachment.
     * @throws AttachmentLimitExceededException
     *         The limit for the number of attachment sets created in a short period of time has been exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SupportException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SupportClient.AddAttachmentsToSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/support-2013-04-15/AddAttachmentsToSet" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public AddAttachmentsToSetResponse addAttachmentsToSet(AddAttachmentsToSetRequest addAttachmentsToSetRequest)
            throws InternalServerErrorException, AttachmentSetIdNotFoundException, AttachmentSetExpiredException,
            AttachmentSetSizeLimitExceededException, AttachmentLimitExceededException, AwsServiceException, SdkClientException,
            SupportException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AddAttachmentsToSetResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, AddAttachmentsToSetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, addAttachmentsToSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Support");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AddAttachmentsToSet");

            return clientHandler.execute(new ClientExecutionParams<AddAttachmentsToSetRequest, AddAttachmentsToSetResponse>()
                    .withOperationName("AddAttachmentsToSet").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(addAttachmentsToSetRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new AddAttachmentsToSetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds additional customer communication to an Amazon Web Services Support case. Use the <code>caseId</code>
     * parameter to identify the case to which to add communication. You can list a set of email addresses to copy on
     * the communication by using the <code>ccEmailAddresses</code> parameter. The <code>communicationBody</code> value
     * contains the text of the communication.
     * </p>
     * <note>
     * <ul>
     * <li>
     * <p>
     * You must have a Business, Enterprise On-Ramp, or Enterprise Support plan to use the Amazon Web Services Support
     * API.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you call the Amazon Web Services Support API from an account that does not have a Business, Enterprise
     * On-Ramp, or Enterprise Support plan, the <code>SubscriptionRequiredException</code> error message appears. For
     * information about changing your support plan, see <a href="http://aws.amazon.com/premiumsupport/">Amazon Web
     * Services Support</a>.
     * </p>
     * </li>
     * </ul>
     * </note>
     *
     * @param addCommunicationToCaseRequest
     * @return Result of the AddCommunicationToCase operation returned by the service.
     * @throws InternalServerErrorException
     *         An internal server error occurred.
     * @throws CaseIdNotFoundException
     *         The requested <code>caseId</code> couldn't be located.
     * @throws AttachmentSetIdNotFoundException
     *         An attachment set with the specified ID could not be found.
     * @throws AttachmentSetExpiredException
     *         The expiration time of the attachment set has passed. The set expires 1 hour after it is created.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SupportException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SupportClient.AddCommunicationToCase
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/support-2013-04-15/AddCommunicationToCase"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public AddCommunicationToCaseResponse addCommunicationToCase(AddCommunicationToCaseRequest addCommunicationToCaseRequest)
            throws InternalServerErrorException, CaseIdNotFoundException, AttachmentSetIdNotFoundException,
            AttachmentSetExpiredException, AwsServiceException, SdkClientException, SupportException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AddCommunicationToCaseResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, AddCommunicationToCaseResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, addCommunicationToCaseRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Support");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AddCommunicationToCase");

            return clientHandler
                    .execute(new ClientExecutionParams<AddCommunicationToCaseRequest, AddCommunicationToCaseResponse>()
                            .withOperationName("AddCommunicationToCase").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(addCommunicationToCaseRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new AddCommunicationToCaseRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a case in the Amazon Web Services Support Center. This operation is similar to how you create a case in
     * the Amazon Web Services Support Center <a href="https://console.aws.amazon.com/support/home#/case/create">Create
     * Case</a> page.
     * </p>
     * <p>
     * The Amazon Web Services Support API doesn't support requesting service limit increases. You can submit a service
     * limit increase in the following ways:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Submit a request from the Amazon Web Services Support Center <a
     * href="https://console.aws.amazon.com/support/home#/case/create">Create Case</a> page.
     * </p>
     * </li>
     * <li>
     * <p>
     * Use the Service Quotas <a
     * href="https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_RequestServiceQuotaIncrease.html"
     * >RequestServiceQuotaIncrease</a> operation.
     * </p>
     * </li>
     * </ul>
     * <p>
     * A successful <code>CreateCase</code> request returns an Amazon Web Services Support case number. You can use the
     * <a>DescribeCases</a> operation and specify the case number to get existing Amazon Web Services Support cases.
     * After you create a case, use the <a>AddCommunicationToCase</a> operation to add additional communication or
     * attachments to an existing case.
     * </p>
     * <p>
     * The <code>caseId</code> is separate from the <code>displayId</code> that appears in the <a
     * href="https://console.aws.amazon.com/support">Amazon Web Services Support Center</a>. Use the
     * <a>DescribeCases</a> operation to get the <code>displayId</code>.
     * </p>
     * <note>
     * <ul>
     * <li>
     * <p>
     * You must have a Business, Enterprise On-Ramp, or Enterprise Support plan to use the Amazon Web Services Support
     * API.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you call the Amazon Web Services Support API from an account that does not have a Business, Enterprise
     * On-Ramp, or Enterprise Support plan, the <code>SubscriptionRequiredException</code> error message appears. For
     * information about changing your support plan, see <a href="http://aws.amazon.com/premiumsupport/">Amazon Web
     * Services Support</a>.
     * </p>
     * </li>
     * </ul>
     * </note>
     *
     * @param createCaseRequest
     * @return Result of the CreateCase operation returned by the service.
     * @throws InternalServerErrorException
     *         An internal server error occurred.
     * @throws CaseCreationLimitExceededException
     *         The case creation limit for the account has been exceeded.
     * @throws AttachmentSetIdNotFoundException
     *         An attachment set with the specified ID could not be found.
     * @throws AttachmentSetExpiredException
     *         The expiration time of the attachment set has passed. The set expires 1 hour after it is created.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SupportException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SupportClient.CreateCase
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/support-2013-04-15/CreateCase" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateCaseResponse createCase(CreateCaseRequest createCaseRequest) throws InternalServerErrorException,
            CaseCreationLimitExceededException, AttachmentSetIdNotFoundException, AttachmentSetExpiredException,
            AwsServiceException, SdkClientException, SupportException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateCaseResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateCaseResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createCaseRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Support");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateCase");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateCaseRequest, CreateCaseResponse>().withOperationName("CreateCase")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createCaseRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateCaseRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the attachment that has the specified ID. Attachments can include screenshots, error logs, or other files
     * that describe your issue. Attachment IDs are generated by the case management system when you add an attachment
     * to a case or case communication. Attachment IDs are returned in the <a>AttachmentDetails</a> objects that are
     * returned by the <a>DescribeCommunications</a> operation.
     * </p>
     * <note>
     * <ul>
     * <li>
     * <p>
     * You must have a Business, Enterprise On-Ramp, or Enterprise Support plan to use the Amazon Web Services Support
     * API.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you call the Amazon Web Services Support API from an account that does not have a Business, Enterprise
     * On-Ramp, or Enterprise Support plan, the <code>SubscriptionRequiredException</code> error message appears. For
     * information about changing your support plan, see <a href="http://aws.amazon.com/premiumsupport/">Amazon Web
     * Services Support</a>.
     * </p>
     * </li>
     * </ul>
     * </note>
     *
     * @param describeAttachmentRequest
     * @return Result of the DescribeAttachment operation returned by the service.
     * @throws InternalServerErrorException
     *         An internal server error occurred.
     * @throws DescribeAttachmentLimitExceededException
     *         The limit for the number of <a>DescribeAttachment</a> requests in a short period of time has been
     *         exceeded.
     * @throws AttachmentIdNotFoundException
     *         An attachment with the specified ID could not be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SupportException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SupportClient.DescribeAttachment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/support-2013-04-15/DescribeAttachment" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeAttachmentResponse describeAttachment(DescribeAttachmentRequest describeAttachmentRequest)
            throws InternalServerErrorException, DescribeAttachmentLimitExceededException, AttachmentIdNotFoundException,
            AwsServiceException, SdkClientException, SupportException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeAttachmentResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeAttachmentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAttachmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Support");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAttachment");

            return clientHandler.execute(new ClientExecutionParams<DescribeAttachmentRequest, DescribeAttachmentResponse>()
                    .withOperationName("DescribeAttachment").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeAttachmentRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeAttachmentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of cases that you specify by passing one or more case IDs. You can use the <code>afterTime</code>
     * and <code>beforeTime</code> parameters to filter the cases by date. You can set values for the
     * <code>includeResolvedCases</code> and <code>includeCommunications</code> parameters to specify how much
     * information to return.
     * </p>
     * <p>
     * The response returns the following in JSON format:
     * </p>
     * <ul>
     * <li>
     * <p>
     * One or more <a
     * href="https://docs.aws.amazon.com/awssupport/latest/APIReference/API_CaseDetails.html">CaseDetails</a> data
     * types.
     * </p>
     * </li>
     * <li>
     * <p>
     * One or more <code>nextToken</code> values, which specify where to paginate the returned records represented by
     * the <code>CaseDetails</code> objects.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Case data is available for 12 months after creation. If a case was created more than 12 months ago, a request
     * might return an error.
     * </p>
     * <note>
     * <ul>
     * <li>
     * <p>
     * You must have a Business, Enterprise On-Ramp, or Enterprise Support plan to use the Amazon Web Services Support
     * API.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you call the Amazon Web Services Support API from an account that does not have a Business, Enterprise
     * On-Ramp, or Enterprise Support plan, the <code>SubscriptionRequiredException</code> error message appears. For
     * information about changing your support plan, see <a href="http://aws.amazon.com/premiumsupport/">Amazon Web
     * Services Support</a>.
     * </p>
     * </li>
     * </ul>
     * </note>
     *
     * @param describeCasesRequest
     * @return Result of the DescribeCases operation returned by the service.
     * @throws InternalServerErrorException
     *         An internal server error occurred.
     * @throws CaseIdNotFoundException
     *         The requested <code>caseId</code> couldn't be located.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SupportException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SupportClient.DescribeCases
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/support-2013-04-15/DescribeCases" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeCasesResponse describeCases(DescribeCasesRequest describeCasesRequest) throws InternalServerErrorException,
            CaseIdNotFoundException, AwsServiceException, SdkClientException, SupportException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeCasesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeCasesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeCasesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Support");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeCases");

            return clientHandler.execute(new ClientExecutionParams<DescribeCasesRequest, DescribeCasesResponse>()
                    .withOperationName("DescribeCases").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeCasesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeCasesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of cases that you specify by passing one or more case IDs. You can use the <code>afterTime</code>
     * and <code>beforeTime</code> parameters to filter the cases by date. You can set values for the
     * <code>includeResolvedCases</code> and <code>includeCommunications</code> parameters to specify how much
     * information to return.
     * </p>
     * <p>
     * The response returns the following in JSON format:
     * </p>
     * <ul>
     * <li>
     * <p>
     * One or more <a
     * href="https://docs.aws.amazon.com/awssupport/latest/APIReference/API_CaseDetails.html">CaseDetails</a> data
     * types.
     * </p>
     * </li>
     * <li>
     * <p>
     * One or more <code>nextToken</code> values, which specify where to paginate the returned records represented by
     * the <code>CaseDetails</code> objects.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Case data is available for 12 months after creation. If a case was created more than 12 months ago, a request
     * might return an error.
     * </p>
     * <note>
     * <ul>
     * <li>
     * <p>
     * You must have a Business, Enterprise On-Ramp, or Enterprise Support plan to use the Amazon Web Services Support
     * API.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you call the Amazon Web Services Support API from an account that does not have a Business, Enterprise
     * On-Ramp, or Enterprise Support plan, the <code>SubscriptionRequiredException</code> error message appears. For
     * information about changing your support plan, see <a href="http://aws.amazon.com/premiumsupport/">Amazon Web
     * Services Support</a>.
     * </p>
     * </li>
     * </ul>
     * </note><br/>
     * <p>
     * This is a variant of {@link #describeCases(software.amazon.awssdk.services.support.model.DescribeCasesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.support.paginators.DescribeCasesIterable responses = client.describeCasesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.support.paginators.DescribeCasesIterable responses = client.describeCasesPaginator(request);
     *     for (software.amazon.awssdk.services.support.model.DescribeCasesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.support.paginators.DescribeCasesIterable responses = client.describeCasesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeCases(software.amazon.awssdk.services.support.model.DescribeCasesRequest)} operation.</b>
     * </p>
     *
     * @param describeCasesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerErrorException
     *         An internal server error occurred.
     * @throws CaseIdNotFoundException
     *         The requested <code>caseId</code> couldn't be located.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SupportException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SupportClient.DescribeCases
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/support-2013-04-15/DescribeCases" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeCasesIterable describeCasesPaginator(DescribeCasesRequest describeCasesRequest)
            throws InternalServerErrorException, CaseIdNotFoundException, AwsServiceException, SdkClientException,
            SupportException {
        return new DescribeCasesIterable(this, applyPaginatorUserAgent(describeCasesRequest));
    }

    /**
     * <p>
     * Returns communications and attachments for one or more support cases. Use the <code>afterTime</code> and
     * <code>beforeTime</code> parameters to filter by date. You can use the <code>caseId</code> parameter to restrict
     * the results to a specific case.
     * </p>
     * <p>
     * Case data is available for 12 months after creation. If a case was created more than 12 months ago, a request for
     * data might cause an error.
     * </p>
     * <p>
     * You can use the <code>maxResults</code> and <code>nextToken</code> parameters to control the pagination of the
     * results. Set <code>maxResults</code> to the number of cases that you want to display on each page, and use
     * <code>nextToken</code> to specify the resumption of pagination.
     * </p>
     * <note>
     * <ul>
     * <li>
     * <p>
     * You must have a Business, Enterprise On-Ramp, or Enterprise Support plan to use the Amazon Web Services Support
     * API.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you call the Amazon Web Services Support API from an account that does not have a Business, Enterprise
     * On-Ramp, or Enterprise Support plan, the <code>SubscriptionRequiredException</code> error message appears. For
     * information about changing your support plan, see <a href="http://aws.amazon.com/premiumsupport/">Amazon Web
     * Services Support</a>.
     * </p>
     * </li>
     * </ul>
     * </note>
     *
     * @param describeCommunicationsRequest
     * @return Result of the DescribeCommunications operation returned by the service.
     * @throws InternalServerErrorException
     *         An internal server error occurred.
     * @throws CaseIdNotFoundException
     *         The requested <code>caseId</code> couldn't be located.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SupportException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SupportClient.DescribeCommunications
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/support-2013-04-15/DescribeCommunications"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeCommunicationsResponse describeCommunications(DescribeCommunicationsRequest describeCommunicationsRequest)
            throws InternalServerErrorException, CaseIdNotFoundException, AwsServiceException, SdkClientException,
            SupportException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeCommunicationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeCommunicationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeCommunicationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Support");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeCommunications");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeCommunicationsRequest, DescribeCommunicationsResponse>()
                            .withOperationName("DescribeCommunications").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeCommunicationsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeCommunicationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns communications and attachments for one or more support cases. Use the <code>afterTime</code> and
     * <code>beforeTime</code> parameters to filter by date. You can use the <code>caseId</code> parameter to restrict
     * the results to a specific case.
     * </p>
     * <p>
     * Case data is available for 12 months after creation. If a case was created more than 12 months ago, a request for
     * data might cause an error.
     * </p>
     * <p>
     * You can use the <code>maxResults</code> and <code>nextToken</code> parameters to control the pagination of the
     * results. Set <code>maxResults</code> to the number of cases that you want to display on each page, and use
     * <code>nextToken</code> to specify the resumption of pagination.
     * </p>
     * <note>
     * <ul>
     * <li>
     * <p>
     * You must have a Business, Enterprise On-Ramp, or Enterprise Support plan to use the Amazon Web Services Support
     * API.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you call the Amazon Web Services Support API from an account that does not have a Business, Enterprise
     * On-Ramp, or Enterprise Support plan, the <code>SubscriptionRequiredException</code> error message appears. For
     * information about changing your support plan, see <a href="http://aws.amazon.com/premiumsupport/">Amazon Web
     * Services Support</a>.
     * </p>
     * </li>
     * </ul>
     * </note><br/>
     * <p>
     * This is a variant of
     * {@link #describeCommunications(software.amazon.awssdk.services.support.model.DescribeCommunicationsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.support.paginators.DescribeCommunicationsIterable responses = client.describeCommunicationsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.support.paginators.DescribeCommunicationsIterable responses = client
     *             .describeCommunicationsPaginator(request);
     *     for (software.amazon.awssdk.services.support.model.DescribeCommunicationsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.support.paginators.DescribeCommunicationsIterable responses = client.describeCommunicationsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeCommunications(software.amazon.awssdk.services.support.model.DescribeCommunicationsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeCommunicationsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerErrorException
     *         An internal server error occurred.
     * @throws CaseIdNotFoundException
     *         The requested <code>caseId</code> couldn't be located.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SupportException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SupportClient.DescribeCommunications
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/support-2013-04-15/DescribeCommunications"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeCommunicationsIterable describeCommunicationsPaginator(
            DescribeCommunicationsRequest describeCommunicationsRequest) throws InternalServerErrorException,
            CaseIdNotFoundException, AwsServiceException, SdkClientException, SupportException {
        return new DescribeCommunicationsIterable(this, applyPaginatorUserAgent(describeCommunicationsRequest));
    }

    /**
     * <p>
     * Returns the current list of Amazon Web Services services and a list of service categories for each service. You
     * then use service names and categories in your <a>CreateCase</a> requests. Each Amazon Web Services service has
     * its own set of categories.
     * </p>
     * <p>
     * The service codes and category codes correspond to the values that appear in the <b>Service</b> and
     * <b>Category</b> lists on the Amazon Web Services Support Center <a
     * href="https://console.aws.amazon.com/support/home#/case/create">Create Case</a> page. The values in those fields
     * don't necessarily match the service codes and categories returned by the <code>DescribeServices</code> operation.
     * Always use the service codes and categories that the <code>DescribeServices</code> operation returns, so that you
     * have the most recent set of service and category codes.
     * </p>
     * <note>
     * <ul>
     * <li>
     * <p>
     * You must have a Business, Enterprise On-Ramp, or Enterprise Support plan to use the Amazon Web Services Support
     * API.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you call the Amazon Web Services Support API from an account that does not have a Business, Enterprise
     * On-Ramp, or Enterprise Support plan, the <code>SubscriptionRequiredException</code> error message appears. For
     * information about changing your support plan, see <a href="http://aws.amazon.com/premiumsupport/">Amazon Web
     * Services Support</a>.
     * </p>
     * </li>
     * </ul>
     * </note>
     *
     * @param describeServicesRequest
     * @return Result of the DescribeServices operation returned by the service.
     * @throws InternalServerErrorException
     *         An internal server error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SupportException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SupportClient.DescribeServices
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/support-2013-04-15/DescribeServices" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeServicesResponse describeServices(DescribeServicesRequest describeServicesRequest)
            throws InternalServerErrorException, AwsServiceException, SdkClientException, SupportException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeServicesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeServicesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeServicesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Support");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeServices");

            return clientHandler.execute(new ClientExecutionParams<DescribeServicesRequest, DescribeServicesResponse>()
                    .withOperationName("DescribeServices").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeServicesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeServicesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the list of severity levels that you can assign to a support case. The severity level for a case is also
     * a field in the <a>CaseDetails</a> data type that you include for a <a>CreateCase</a> request.
     * </p>
     * <note>
     * <ul>
     * <li>
     * <p>
     * You must have a Business, Enterprise On-Ramp, or Enterprise Support plan to use the Amazon Web Services Support
     * API.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you call the Amazon Web Services Support API from an account that does not have a Business, Enterprise
     * On-Ramp, or Enterprise Support plan, the <code>SubscriptionRequiredException</code> error message appears. For
     * information about changing your support plan, see <a href="http://aws.amazon.com/premiumsupport/">Amazon Web
     * Services Support</a>.
     * </p>
     * </li>
     * </ul>
     * </note>
     *
     * @param describeSeverityLevelsRequest
     * @return Result of the DescribeSeverityLevels operation returned by the service.
     * @throws InternalServerErrorException
     *         An internal server error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SupportException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SupportClient.DescribeSeverityLevels
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/support-2013-04-15/DescribeSeverityLevels"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeSeverityLevelsResponse describeSeverityLevels(DescribeSeverityLevelsRequest describeSeverityLevelsRequest)
            throws InternalServerErrorException, AwsServiceException, SdkClientException, SupportException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeSeverityLevelsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeSeverityLevelsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeSeverityLevelsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Support");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeSeverityLevels");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeSeverityLevelsRequest, DescribeSeverityLevelsResponse>()
                            .withOperationName("DescribeSeverityLevels").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeSeverityLevelsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeSeverityLevelsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the refresh status of the Trusted Advisor checks that have the specified check IDs. You can get the check
     * IDs by calling the <a>DescribeTrustedAdvisorChecks</a> operation.
     * </p>
     * <p>
     * Some checks are refreshed automatically, and you can't return their refresh statuses by using the
     * <code>DescribeTrustedAdvisorCheckRefreshStatuses</code> operation. If you call this operation for these checks,
     * you might see an <code>InvalidParameterValue</code> error.
     * </p>
     * <note>
     * <ul>
     * <li>
     * <p>
     * You must have a Business, Enterprise On-Ramp, or Enterprise Support plan to use the Amazon Web Services Support
     * API.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you call the Amazon Web Services Support API from an account that does not have a Business, Enterprise
     * On-Ramp, or Enterprise Support plan, the <code>SubscriptionRequiredException</code> error message appears. For
     * information about changing your support plan, see <a href="http://aws.amazon.com/premiumsupport/">Amazon Web
     * Services Support</a>.
     * </p>
     * </li>
     * </ul>
     * </note>
     *
     * @param describeTrustedAdvisorCheckRefreshStatusesRequest
     * @return Result of the DescribeTrustedAdvisorCheckRefreshStatuses operation returned by the service.
     * @throws InternalServerErrorException
     *         An internal server error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SupportException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SupportClient.DescribeTrustedAdvisorCheckRefreshStatuses
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/support-2013-04-15/DescribeTrustedAdvisorCheckRefreshStatuses"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeTrustedAdvisorCheckRefreshStatusesResponse describeTrustedAdvisorCheckRefreshStatuses(
            DescribeTrustedAdvisorCheckRefreshStatusesRequest describeTrustedAdvisorCheckRefreshStatusesRequest)
            throws InternalServerErrorException, AwsServiceException, SdkClientException, SupportException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeTrustedAdvisorCheckRefreshStatusesResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, DescribeTrustedAdvisorCheckRefreshStatusesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeTrustedAdvisorCheckRefreshStatusesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Support");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeTrustedAdvisorCheckRefreshStatuses");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeTrustedAdvisorCheckRefreshStatusesRequest, DescribeTrustedAdvisorCheckRefreshStatusesResponse>()
                            .withOperationName("DescribeTrustedAdvisorCheckRefreshStatuses").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeTrustedAdvisorCheckRefreshStatusesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeTrustedAdvisorCheckRefreshStatusesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the results of the Trusted Advisor check that has the specified check ID. You can get the check IDs by
     * calling the <a>DescribeTrustedAdvisorChecks</a> operation.
     * </p>
     * <p>
     * The response contains a <a>TrustedAdvisorCheckResult</a> object, which contains these three objects:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>TrustedAdvisorCategorySpecificSummary</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>TrustedAdvisorResourceDetail</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>TrustedAdvisorResourcesSummary</a>
     * </p>
     * </li>
     * </ul>
     * <p>
     * In addition, the response contains these fields:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>status</b> - The alert status of the check can be <code>ok</code> (green), <code>warning</code> (yellow),
     * <code>error</code> (red), or <code>not_available</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>timestamp</b> - The time of the last refresh of the check.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>checkId</b> - The unique identifier for the check.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <ul>
     * <li>
     * <p>
     * You must have a Business, Enterprise On-Ramp, or Enterprise Support plan to use the Amazon Web Services Support
     * API.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you call the Amazon Web Services Support API from an account that does not have a Business, Enterprise
     * On-Ramp, or Enterprise Support plan, the <code>SubscriptionRequiredException</code> error message appears. For
     * information about changing your support plan, see <a href="http://aws.amazon.com/premiumsupport/">Amazon Web
     * Services Support</a>.
     * </p>
     * </li>
     * </ul>
     * </note>
     *
     * @param describeTrustedAdvisorCheckResultRequest
     * @return Result of the DescribeTrustedAdvisorCheckResult operation returned by the service.
     * @throws InternalServerErrorException
     *         An internal server error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SupportException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SupportClient.DescribeTrustedAdvisorCheckResult
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/support-2013-04-15/DescribeTrustedAdvisorCheckResult"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeTrustedAdvisorCheckResultResponse describeTrustedAdvisorCheckResult(
            DescribeTrustedAdvisorCheckResultRequest describeTrustedAdvisorCheckResultRequest)
            throws InternalServerErrorException, AwsServiceException, SdkClientException, SupportException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeTrustedAdvisorCheckResultResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeTrustedAdvisorCheckResultResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeTrustedAdvisorCheckResultRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Support");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeTrustedAdvisorCheckResult");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeTrustedAdvisorCheckResultRequest, DescribeTrustedAdvisorCheckResultResponse>()
                            .withOperationName("DescribeTrustedAdvisorCheckResult").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeTrustedAdvisorCheckResultRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeTrustedAdvisorCheckResultRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the results for the Trusted Advisor check summaries for the check IDs that you specified. You can get the
     * check IDs by calling the <a>DescribeTrustedAdvisorChecks</a> operation.
     * </p>
     * <p>
     * The response contains an array of <a>TrustedAdvisorCheckSummary</a> objects.
     * </p>
     * <note>
     * <ul>
     * <li>
     * <p>
     * You must have a Business, Enterprise On-Ramp, or Enterprise Support plan to use the Amazon Web Services Support
     * API.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you call the Amazon Web Services Support API from an account that does not have a Business, Enterprise
     * On-Ramp, or Enterprise Support plan, the <code>SubscriptionRequiredException</code> error message appears. For
     * information about changing your support plan, see <a href="http://aws.amazon.com/premiumsupport/">Amazon Web
     * Services Support</a>.
     * </p>
     * </li>
     * </ul>
     * </note>
     *
     * @param describeTrustedAdvisorCheckSummariesRequest
     * @return Result of the DescribeTrustedAdvisorCheckSummaries operation returned by the service.
     * @throws InternalServerErrorException
     *         An internal server error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SupportException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SupportClient.DescribeTrustedAdvisorCheckSummaries
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/support-2013-04-15/DescribeTrustedAdvisorCheckSummaries"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeTrustedAdvisorCheckSummariesResponse describeTrustedAdvisorCheckSummaries(
            DescribeTrustedAdvisorCheckSummariesRequest describeTrustedAdvisorCheckSummariesRequest)
            throws InternalServerErrorException, AwsServiceException, SdkClientException, SupportException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeTrustedAdvisorCheckSummariesResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, DescribeTrustedAdvisorCheckSummariesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeTrustedAdvisorCheckSummariesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Support");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeTrustedAdvisorCheckSummaries");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeTrustedAdvisorCheckSummariesRequest, DescribeTrustedAdvisorCheckSummariesResponse>()
                            .withOperationName("DescribeTrustedAdvisorCheckSummaries").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeTrustedAdvisorCheckSummariesRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeTrustedAdvisorCheckSummariesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about all available Trusted Advisor checks, including the name, ID, category, description,
     * and metadata. You must specify a language code. The Amazon Web Services Support API currently supports English
     * ("en") and Japanese ("ja"). The response contains a <a>TrustedAdvisorCheckDescription</a> object for each check.
     * You must set the Amazon Web Services Region to us-east-1.
     * </p>
     * <note>
     * <ul>
     * <li>
     * <p>
     * You must have a Business, Enterprise On-Ramp, or Enterprise Support plan to use the Amazon Web Services Support
     * API.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you call the Amazon Web Services Support API from an account that does not have a Business, Enterprise
     * On-Ramp, or Enterprise Support plan, the <code>SubscriptionRequiredException</code> error message appears. For
     * information about changing your support plan, see <a href="http://aws.amazon.com/premiumsupport/">Amazon Web
     * Services Support</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * The names and descriptions for Trusted Advisor checks are subject to change. We recommend that you specify the
     * check ID in your code to uniquely identify a check.
     * </p>
     * </li>
     * </ul>
     * </note>
     *
     * @param describeTrustedAdvisorChecksRequest
     * @return Result of the DescribeTrustedAdvisorChecks operation returned by the service.
     * @throws InternalServerErrorException
     *         An internal server error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SupportException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SupportClient.DescribeTrustedAdvisorChecks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/support-2013-04-15/DescribeTrustedAdvisorChecks"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeTrustedAdvisorChecksResponse describeTrustedAdvisorChecks(
            DescribeTrustedAdvisorChecksRequest describeTrustedAdvisorChecksRequest) throws InternalServerErrorException,
            AwsServiceException, SdkClientException, SupportException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeTrustedAdvisorChecksResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeTrustedAdvisorChecksResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeTrustedAdvisorChecksRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Support");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeTrustedAdvisorChecks");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeTrustedAdvisorChecksRequest, DescribeTrustedAdvisorChecksResponse>()
                            .withOperationName("DescribeTrustedAdvisorChecks").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeTrustedAdvisorChecksRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeTrustedAdvisorChecksRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Refreshes the Trusted Advisor check that you specify using the check ID. You can get the check IDs by calling the
     * <a>DescribeTrustedAdvisorChecks</a> operation.
     * </p>
     * <note>
     * <p>
     * Some checks are refreshed automatically. If you call the <code>RefreshTrustedAdvisorCheck</code> operation to
     * refresh them, you might see the <code>InvalidParameterValue</code> error.
     * </p>
     * </note>
     * <p>
     * The response contains a <a>TrustedAdvisorCheckRefreshStatus</a> object.
     * </p>
     * <note>
     * <ul>
     * <li>
     * <p>
     * You must have a Business, Enterprise On-Ramp, or Enterprise Support plan to use the Amazon Web Services Support
     * API.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you call the Amazon Web Services Support API from an account that does not have a Business, Enterprise
     * On-Ramp, or Enterprise Support plan, the <code>SubscriptionRequiredException</code> error message appears. For
     * information about changing your support plan, see <a href="http://aws.amazon.com/premiumsupport/">Amazon Web
     * Services Support</a>.
     * </p>
     * </li>
     * </ul>
     * </note>
     *
     * @param refreshTrustedAdvisorCheckRequest
     * @return Result of the RefreshTrustedAdvisorCheck operation returned by the service.
     * @throws InternalServerErrorException
     *         An internal server error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SupportException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SupportClient.RefreshTrustedAdvisorCheck
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/support-2013-04-15/RefreshTrustedAdvisorCheck"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public RefreshTrustedAdvisorCheckResponse refreshTrustedAdvisorCheck(
            RefreshTrustedAdvisorCheckRequest refreshTrustedAdvisorCheckRequest) throws InternalServerErrorException,
            AwsServiceException, SdkClientException, SupportException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RefreshTrustedAdvisorCheckResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, RefreshTrustedAdvisorCheckResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, refreshTrustedAdvisorCheckRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Support");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RefreshTrustedAdvisorCheck");

            return clientHandler
                    .execute(new ClientExecutionParams<RefreshTrustedAdvisorCheckRequest, RefreshTrustedAdvisorCheckResponse>()
                            .withOperationName("RefreshTrustedAdvisorCheck").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(refreshTrustedAdvisorCheckRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new RefreshTrustedAdvisorCheckRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Resolves a support case. This operation takes a <code>caseId</code> and returns the initial and final state of
     * the case.
     * </p>
     * <note>
     * <ul>
     * <li>
     * <p>
     * You must have a Business, Enterprise On-Ramp, or Enterprise Support plan to use the Amazon Web Services Support
     * API.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you call the Amazon Web Services Support API from an account that does not have a Business, Enterprise
     * On-Ramp, or Enterprise Support plan, the <code>SubscriptionRequiredException</code> error message appears. For
     * information about changing your support plan, see <a href="http://aws.amazon.com/premiumsupport/">Amazon Web
     * Services Support</a>.
     * </p>
     * </li>
     * </ul>
     * </note>
     *
     * @param resolveCaseRequest
     * @return Result of the ResolveCase operation returned by the service.
     * @throws InternalServerErrorException
     *         An internal server error occurred.
     * @throws CaseIdNotFoundException
     *         The requested <code>caseId</code> couldn't be located.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SupportException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SupportClient.ResolveCase
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/support-2013-04-15/ResolveCase" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ResolveCaseResponse resolveCase(ResolveCaseRequest resolveCaseRequest) throws InternalServerErrorException,
            CaseIdNotFoundException, AwsServiceException, SdkClientException, SupportException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ResolveCaseResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ResolveCaseResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, resolveCaseRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Support");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ResolveCase");

            return clientHandler.execute(new ClientExecutionParams<ResolveCaseRequest, ResolveCaseResponse>()
                    .withOperationName("ResolveCase").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(resolveCaseRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ResolveCaseRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(SupportException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AttachmentSetIdNotFound")
                                .exceptionBuilderSupplier(AttachmentSetIdNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CaseCreationLimitExceeded")
                                .exceptionBuilderSupplier(CaseCreationLimitExceededException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AttachmentIdNotFound")
                                .exceptionBuilderSupplier(AttachmentIdNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AttachmentSetExpired")
                                .exceptionBuilderSupplier(AttachmentSetExpiredException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AttachmentLimitExceeded")
                                .exceptionBuilderSupplier(AttachmentLimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CaseIdNotFound")
                                .exceptionBuilderSupplier(CaseIdNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DescribeAttachmentLimitExceeded")
                                .exceptionBuilderSupplier(DescribeAttachmentLimitExceededException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerError")
                                .exceptionBuilderSupplier(InternalServerErrorException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AttachmentSetSizeLimitExceeded")
                                .exceptionBuilderSupplier(AttachmentSetSizeLimitExceededException::builder).httpStatusCode(400)
                                .build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends SupportRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
