/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.support;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.support.model.AddAttachmentsToSetRequest;
import software.amazon.awssdk.services.support.model.AddAttachmentsToSetResponse;
import software.amazon.awssdk.services.support.model.AddCommunicationToCaseRequest;
import software.amazon.awssdk.services.support.model.AddCommunicationToCaseResponse;
import software.amazon.awssdk.services.support.model.AttachmentIdNotFoundException;
import software.amazon.awssdk.services.support.model.AttachmentLimitExceededException;
import software.amazon.awssdk.services.support.model.AttachmentSetExpiredException;
import software.amazon.awssdk.services.support.model.AttachmentSetIdNotFoundException;
import software.amazon.awssdk.services.support.model.AttachmentSetSizeLimitExceededException;
import software.amazon.awssdk.services.support.model.CaseCreationLimitExceededException;
import software.amazon.awssdk.services.support.model.CaseIdNotFoundException;
import software.amazon.awssdk.services.support.model.CreateCaseRequest;
import software.amazon.awssdk.services.support.model.CreateCaseResponse;
import software.amazon.awssdk.services.support.model.DescribeAttachmentLimitExceededException;
import software.amazon.awssdk.services.support.model.DescribeAttachmentRequest;
import software.amazon.awssdk.services.support.model.DescribeAttachmentResponse;
import software.amazon.awssdk.services.support.model.DescribeCasesRequest;
import software.amazon.awssdk.services.support.model.DescribeCasesResponse;
import software.amazon.awssdk.services.support.model.DescribeCommunicationsRequest;
import software.amazon.awssdk.services.support.model.DescribeCommunicationsResponse;
import software.amazon.awssdk.services.support.model.DescribeServicesRequest;
import software.amazon.awssdk.services.support.model.DescribeServicesResponse;
import software.amazon.awssdk.services.support.model.DescribeSeverityLevelsRequest;
import software.amazon.awssdk.services.support.model.DescribeSeverityLevelsResponse;
import software.amazon.awssdk.services.support.model.DescribeTrustedAdvisorCheckRefreshStatusesRequest;
import software.amazon.awssdk.services.support.model.DescribeTrustedAdvisorCheckRefreshStatusesResponse;
import software.amazon.awssdk.services.support.model.DescribeTrustedAdvisorCheckResultRequest;
import software.amazon.awssdk.services.support.model.DescribeTrustedAdvisorCheckResultResponse;
import software.amazon.awssdk.services.support.model.DescribeTrustedAdvisorCheckSummariesRequest;
import software.amazon.awssdk.services.support.model.DescribeTrustedAdvisorCheckSummariesResponse;
import software.amazon.awssdk.services.support.model.DescribeTrustedAdvisorChecksRequest;
import software.amazon.awssdk.services.support.model.DescribeTrustedAdvisorChecksResponse;
import software.amazon.awssdk.services.support.model.InternalServerErrorException;
import software.amazon.awssdk.services.support.model.RefreshTrustedAdvisorCheckRequest;
import software.amazon.awssdk.services.support.model.RefreshTrustedAdvisorCheckResponse;
import software.amazon.awssdk.services.support.model.ResolveCaseRequest;
import software.amazon.awssdk.services.support.model.ResolveCaseResponse;
import software.amazon.awssdk.services.support.model.SupportException;
import software.amazon.awssdk.services.support.model.SupportRequest;
import software.amazon.awssdk.services.support.paginators.DescribeCasesIterable;
import software.amazon.awssdk.services.support.paginators.DescribeCommunicationsIterable;
import software.amazon.awssdk.services.support.transform.AddAttachmentsToSetRequestMarshaller;
import software.amazon.awssdk.services.support.transform.AddCommunicationToCaseRequestMarshaller;
import software.amazon.awssdk.services.support.transform.CreateCaseRequestMarshaller;
import software.amazon.awssdk.services.support.transform.DescribeAttachmentRequestMarshaller;
import software.amazon.awssdk.services.support.transform.DescribeCasesRequestMarshaller;
import software.amazon.awssdk.services.support.transform.DescribeCommunicationsRequestMarshaller;
import software.amazon.awssdk.services.support.transform.DescribeServicesRequestMarshaller;
import software.amazon.awssdk.services.support.transform.DescribeSeverityLevelsRequestMarshaller;
import software.amazon.awssdk.services.support.transform.DescribeTrustedAdvisorCheckRefreshStatusesRequestMarshaller;
import software.amazon.awssdk.services.support.transform.DescribeTrustedAdvisorCheckResultRequestMarshaller;
import software.amazon.awssdk.services.support.transform.DescribeTrustedAdvisorCheckSummariesRequestMarshaller;
import software.amazon.awssdk.services.support.transform.DescribeTrustedAdvisorChecksRequestMarshaller;
import software.amazon.awssdk.services.support.transform.RefreshTrustedAdvisorCheckRequestMarshaller;
import software.amazon.awssdk.services.support.transform.ResolveCaseRequestMarshaller;

/**
 * Internal implementation of {@link SupportClient}.
 *
 * @see SupportClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultSupportClient implements SupportClient {
    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultSupportClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Adds one or more attachments to an attachment set. If an <code>attachmentSetId</code> is not specified, a new
     * attachment set is created, and the ID of the set is returned in the response. If an <code>attachmentSetId</code>
     * is specified, the attachments are added to the specified set, if it exists.
     * </p>
     * <p>
     * An attachment set is a temporary container for attachments that are to be added to a case or case communication.
     * The set is available for one hour after it is created; the <code>expiryTime</code> returned in the response
     * indicates when the set expires. The maximum number of attachments in a set is 3, and the maximum size of any
     * attachment in the set is 5 MB.
     * </p>
     *
     * @param addAttachmentsToSetRequest
     * @return Result of the AddAttachmentsToSet operation returned by the service.
     * @throws InternalServerErrorException
     *         An internal server error occurred.
     * @throws AttachmentSetIdNotFoundException
     *         An attachment set with the specified ID could not be found.
     * @throws AttachmentSetExpiredException
     *         The expiration time of the attachment set has passed. The set expires 1 hour after it is created.
     * @throws AttachmentSetSizeLimitExceededException
     *         A limit for the size of an attachment set has been exceeded. The limits are 3 attachments and 5 MB per
     *         attachment.
     * @throws AttachmentLimitExceededException
     *         The limit for the number of attachment sets created in a short period of time has been exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SupportException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SupportClient.AddAttachmentsToSet
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/support-2013-04-15/AddAttachmentsToSet" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public AddAttachmentsToSetResponse addAttachmentsToSet(AddAttachmentsToSetRequest addAttachmentsToSetRequest)
            throws InternalServerErrorException, AttachmentSetIdNotFoundException, AttachmentSetExpiredException,
            AttachmentSetSizeLimitExceededException, AttachmentLimitExceededException, AwsServiceException, SdkClientException,
            SupportException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AddAttachmentsToSetResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, AddAttachmentsToSetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<AddAttachmentsToSetRequest, AddAttachmentsToSetResponse>()
                .withOperationName("AddAttachmentsToSet").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(addAttachmentsToSetRequest)
                .withMarshaller(new AddAttachmentsToSetRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Adds additional customer communication to an AWS Support case. You use the <code>caseId</code> value to identify
     * the case to add communication to. You can list a set of email addresses to copy on the communication using the
     * <code>ccEmailAddresses</code> value. The <code>communicationBody</code> value contains the text of the
     * communication.
     * </p>
     * <p>
     * The response indicates the success or failure of the request.
     * </p>
     * <p>
     * This operation implements a subset of the features of the AWS Support Center.
     * </p>
     *
     * @param addCommunicationToCaseRequest
     *        To be written.
     * @return Result of the AddCommunicationToCase operation returned by the service.
     * @throws InternalServerErrorException
     *         An internal server error occurred.
     * @throws CaseIdNotFoundException
     *         The requested <code>caseId</code> could not be located.
     * @throws AttachmentSetIdNotFoundException
     *         An attachment set with the specified ID could not be found.
     * @throws AttachmentSetExpiredException
     *         The expiration time of the attachment set has passed. The set expires 1 hour after it is created.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SupportException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SupportClient.AddCommunicationToCase
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/support-2013-04-15/AddCommunicationToCase" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public AddCommunicationToCaseResponse addCommunicationToCase(AddCommunicationToCaseRequest addCommunicationToCaseRequest)
            throws InternalServerErrorException, CaseIdNotFoundException, AttachmentSetIdNotFoundException,
            AttachmentSetExpiredException, AwsServiceException, SdkClientException, SupportException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AddCommunicationToCaseResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, AddCommunicationToCaseResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<AddCommunicationToCaseRequest, AddCommunicationToCaseResponse>()
                .withOperationName("AddCommunicationToCase").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(addCommunicationToCaseRequest)
                .withMarshaller(new AddCommunicationToCaseRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Creates a new case in the AWS Support Center. This operation is modeled on the behavior of the AWS Support Center
     * <a href="https://console.aws.amazon.com/support/home#/case/create">Create Case</a> page. Its parameters require
     * you to specify the following information:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>issueType.</b> The type of issue for the case. You can specify either "customer-service" or "technical." If
     * you do not indicate a value, the default is "technical."
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>serviceCode.</b> The code for an AWS service. You obtain the <code>serviceCode</code> by calling
     * <a>DescribeServices</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>categoryCode.</b> The category for the service defined for the <code>serviceCode</code> value. You also obtain
     * the category code for a service by calling <a>DescribeServices</a>. Each AWS service defines its own set of
     * category codes.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>severityCode.</b> A value that indicates the urgency of the case, which in turn determines the response time
     * according to your service level agreement with AWS Support. You obtain the SeverityCode by calling
     * <a>DescribeSeverityLevels</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>subject.</b> The <b>Subject</b> field on the AWS Support Center <a
     * href="https://console.aws.amazon.com/support/home#/case/create">Create Case</a> page.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>communicationBody.</b> The <b>Description</b> field on the AWS Support Center <a
     * href="https://console.aws.amazon.com/support/home#/case/create">Create Case</a> page.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>attachmentSetId.</b> The ID of a set of attachments that has been created by using <a>AddAttachmentsToSet</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>language.</b> The human language in which AWS Support handles the case. English and Japanese are currently
     * supported.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>ccEmailAddresses.</b> The AWS Support Center <b>CC</b> field on the <a
     * href="https://console.aws.amazon.com/support/home#/case/create">Create Case</a> page. You can list email
     * addresses to be copied on any correspondence about the case. The account that opens the case is already
     * identified by passing the AWS Credentials in the HTTP POST method or in a method or function call from one of the
     * programming languages supported by an <a href="http://aws.amazon.com/tools/">AWS SDK</a>.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * To add additional communication or attachments to an existing case, use <a>AddCommunicationToCase</a>.
     * </p>
     * </note>
     * <p>
     * A successful <a>CreateCase</a> request returns an AWS Support case number. Case numbers are used by the
     * <a>DescribeCases</a> operation to retrieve existing AWS Support cases.
     * </p>
     *
     * @param createCaseRequest
     * @return Result of the CreateCase operation returned by the service.
     * @throws InternalServerErrorException
     *         An internal server error occurred.
     * @throws CaseCreationLimitExceededException
     *         The case creation limit for the account has been exceeded.
     * @throws AttachmentSetIdNotFoundException
     *         An attachment set with the specified ID could not be found.
     * @throws AttachmentSetExpiredException
     *         The expiration time of the attachment set has passed. The set expires 1 hour after it is created.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SupportException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SupportClient.CreateCase
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/support-2013-04-15/CreateCase" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateCaseResponse createCase(CreateCaseRequest createCaseRequest) throws InternalServerErrorException,
            CaseCreationLimitExceededException, AttachmentSetIdNotFoundException, AttachmentSetExpiredException,
            AwsServiceException, SdkClientException, SupportException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateCaseResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateCaseResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<CreateCaseRequest, CreateCaseResponse>()
                .withOperationName("CreateCase").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(createCaseRequest)
                .withMarshaller(new CreateCaseRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns the attachment that has the specified ID. Attachment IDs are generated by the case management system when
     * you add an attachment to a case or case communication. Attachment IDs are returned in the
     * <a>AttachmentDetails</a> objects that are returned by the <a>DescribeCommunications</a> operation.
     * </p>
     *
     * @param describeAttachmentRequest
     * @return Result of the DescribeAttachment operation returned by the service.
     * @throws InternalServerErrorException
     *         An internal server error occurred.
     * @throws DescribeAttachmentLimitExceededException
     *         The limit for the number of <a>DescribeAttachment</a> requests in a short period of time has been
     *         exceeded.
     * @throws AttachmentIdNotFoundException
     *         An attachment with the specified ID could not be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SupportException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SupportClient.DescribeAttachment
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/support-2013-04-15/DescribeAttachment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeAttachmentResponse describeAttachment(DescribeAttachmentRequest describeAttachmentRequest)
            throws InternalServerErrorException, DescribeAttachmentLimitExceededException, AttachmentIdNotFoundException,
            AwsServiceException, SdkClientException, SupportException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeAttachmentResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeAttachmentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DescribeAttachmentRequest, DescribeAttachmentResponse>()
                .withOperationName("DescribeAttachment").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(describeAttachmentRequest)
                .withMarshaller(new DescribeAttachmentRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns a list of cases that you specify by passing one or more case IDs. In addition, you can filter the cases
     * by date by setting values for the <code>afterTime</code> and <code>beforeTime</code> request parameters. You can
     * set values for the <code>includeResolvedCases</code> and <code>includeCommunications</code> request parameters to
     * control how much information is returned.
     * </p>
     * <p>
     * Case data is available for 12 months after creation. If a case was created more than 12 months ago, a request for
     * data might cause an error.
     * </p>
     * <p>
     * The response returns the following in JSON format:
     * </p>
     * <ul>
     * <li>
     * <p>
     * One or more <a>CaseDetails</a> data types.
     * </p>
     * </li>
     * <li>
     * <p>
     * One or more <code>nextToken</code> values, which specify where to paginate the returned records represented by
     * the <code>CaseDetails</code> objects.
     * </p>
     * </li>
     * </ul>
     *
     * @param describeCasesRequest
     * @return Result of the DescribeCases operation returned by the service.
     * @throws InternalServerErrorException
     *         An internal server error occurred.
     * @throws CaseIdNotFoundException
     *         The requested <code>caseId</code> could not be located.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SupportException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SupportClient.DescribeCases
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/support-2013-04-15/DescribeCases" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeCasesResponse describeCases(DescribeCasesRequest describeCasesRequest) throws InternalServerErrorException,
            CaseIdNotFoundException, AwsServiceException, SdkClientException, SupportException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeCasesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeCasesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DescribeCasesRequest, DescribeCasesResponse>()
                .withOperationName("DescribeCases").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(describeCasesRequest)
                .withMarshaller(new DescribeCasesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns a list of cases that you specify by passing one or more case IDs. In addition, you can filter the cases
     * by date by setting values for the <code>afterTime</code> and <code>beforeTime</code> request parameters. You can
     * set values for the <code>includeResolvedCases</code> and <code>includeCommunications</code> request parameters to
     * control how much information is returned.
     * </p>
     * <p>
     * Case data is available for 12 months after creation. If a case was created more than 12 months ago, a request for
     * data might cause an error.
     * </p>
     * <p>
     * The response returns the following in JSON format:
     * </p>
     * <ul>
     * <li>
     * <p>
     * One or more <a>CaseDetails</a> data types.
     * </p>
     * </li>
     * <li>
     * <p>
     * One or more <code>nextToken</code> values, which specify where to paginate the returned records represented by
     * the <code>CaseDetails</code> objects.
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a variant of {@link #describeCases(software.amazon.awssdk.services.support.model.DescribeCasesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.support.paginators.DescribeCasesIterable responses = client.describeCasesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.support.paginators.DescribeCasesIterable responses = client.describeCasesPaginator(request);
     *     for (software.amazon.awssdk.services.support.model.DescribeCasesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.support.paginators.DescribeCasesIterable responses = client.describeCasesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeCases(software.amazon.awssdk.services.support.model.DescribeCasesRequest)} operation.</b>
     * </p>
     *
     * @param describeCasesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerErrorException
     *         An internal server error occurred.
     * @throws CaseIdNotFoundException
     *         The requested <code>caseId</code> could not be located.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SupportException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SupportClient.DescribeCases
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/support-2013-04-15/DescribeCases" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeCasesIterable describeCasesPaginator(DescribeCasesRequest describeCasesRequest)
            throws InternalServerErrorException, CaseIdNotFoundException, AwsServiceException, SdkClientException,
            SupportException {
        return new DescribeCasesIterable(this, applyPaginatorUserAgent(describeCasesRequest));
    }

    /**
     * <p>
     * Returns communications (and attachments) for one or more support cases. You can use the <code>afterTime</code>
     * and <code>beforeTime</code> parameters to filter by date. You can use the <code>caseId</code> parameter to
     * restrict the results to a particular case.
     * </p>
     * <p>
     * Case data is available for 12 months after creation. If a case was created more than 12 months ago, a request for
     * data might cause an error.
     * </p>
     * <p>
     * You can use the <code>maxResults</code> and <code>nextToken</code> parameters to control the pagination of the
     * result set. Set <code>maxResults</code> to the number of cases you want displayed on each page, and use
     * <code>nextToken</code> to specify the resumption of pagination.
     * </p>
     *
     * @param describeCommunicationsRequest
     * @return Result of the DescribeCommunications operation returned by the service.
     * @throws InternalServerErrorException
     *         An internal server error occurred.
     * @throws CaseIdNotFoundException
     *         The requested <code>caseId</code> could not be located.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SupportException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SupportClient.DescribeCommunications
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/support-2013-04-15/DescribeCommunications" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeCommunicationsResponse describeCommunications(DescribeCommunicationsRequest describeCommunicationsRequest)
            throws InternalServerErrorException, CaseIdNotFoundException, AwsServiceException, SdkClientException,
            SupportException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeCommunicationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeCommunicationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DescribeCommunicationsRequest, DescribeCommunicationsResponse>()
                .withOperationName("DescribeCommunications").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(describeCommunicationsRequest)
                .withMarshaller(new DescribeCommunicationsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns communications (and attachments) for one or more support cases. You can use the <code>afterTime</code>
     * and <code>beforeTime</code> parameters to filter by date. You can use the <code>caseId</code> parameter to
     * restrict the results to a particular case.
     * </p>
     * <p>
     * Case data is available for 12 months after creation. If a case was created more than 12 months ago, a request for
     * data might cause an error.
     * </p>
     * <p>
     * You can use the <code>maxResults</code> and <code>nextToken</code> parameters to control the pagination of the
     * result set. Set <code>maxResults</code> to the number of cases you want displayed on each page, and use
     * <code>nextToken</code> to specify the resumption of pagination.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeCommunications(software.amazon.awssdk.services.support.model.DescribeCommunicationsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.support.paginators.DescribeCommunicationsIterable responses = client.describeCommunicationsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.support.paginators.DescribeCommunicationsIterable responses = client
     *             .describeCommunicationsPaginator(request);
     *     for (software.amazon.awssdk.services.support.model.DescribeCommunicationsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.support.paginators.DescribeCommunicationsIterable responses = client.describeCommunicationsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeCommunications(software.amazon.awssdk.services.support.model.DescribeCommunicationsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeCommunicationsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerErrorException
     *         An internal server error occurred.
     * @throws CaseIdNotFoundException
     *         The requested <code>caseId</code> could not be located.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SupportException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SupportClient.DescribeCommunications
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/support-2013-04-15/DescribeCommunications" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeCommunicationsIterable describeCommunicationsPaginator(
            DescribeCommunicationsRequest describeCommunicationsRequest) throws InternalServerErrorException,
            CaseIdNotFoundException, AwsServiceException, SdkClientException, SupportException {
        return new DescribeCommunicationsIterable(this, applyPaginatorUserAgent(describeCommunicationsRequest));
    }

    /**
     * <p>
     * Returns the current list of AWS services and a list of service categories that applies to each one. You then use
     * service names and categories in your <a>CreateCase</a> requests. Each AWS service has its own set of categories.
     * </p>
     * <p>
     * The service codes and category codes correspond to the values that are displayed in the <b>Service</b> and
     * <b>Category</b> drop-down lists on the AWS Support Center <a
     * href="https://console.aws.amazon.com/support/home#/case/create">Create Case</a> page. The values in those fields,
     * however, do not necessarily match the service codes and categories returned by the <code>DescribeServices</code>
     * request. Always use the service codes and categories obtained programmatically. This practice ensures that you
     * always have the most recent set of service and category codes.
     * </p>
     *
     * @param describeServicesRequest
     * @return Result of the DescribeServices operation returned by the service.
     * @throws InternalServerErrorException
     *         An internal server error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SupportException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SupportClient.DescribeServices
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/support-2013-04-15/DescribeServices" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeServicesResponse describeServices(DescribeServicesRequest describeServicesRequest)
            throws InternalServerErrorException, AwsServiceException, SdkClientException, SupportException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeServicesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeServicesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DescribeServicesRequest, DescribeServicesResponse>()
                .withOperationName("DescribeServices").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(describeServicesRequest)
                .withMarshaller(new DescribeServicesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns the list of severity levels that you can assign to an AWS Support case. The severity level for a case is
     * also a field in the <a>CaseDetails</a> data type included in any <a>CreateCase</a> request.
     * </p>
     *
     * @param describeSeverityLevelsRequest
     * @return Result of the DescribeSeverityLevels operation returned by the service.
     * @throws InternalServerErrorException
     *         An internal server error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SupportException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SupportClient.DescribeSeverityLevels
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/support-2013-04-15/DescribeSeverityLevels" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeSeverityLevelsResponse describeSeverityLevels(DescribeSeverityLevelsRequest describeSeverityLevelsRequest)
            throws InternalServerErrorException, AwsServiceException, SdkClientException, SupportException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeSeverityLevelsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeSeverityLevelsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DescribeSeverityLevelsRequest, DescribeSeverityLevelsResponse>()
                .withOperationName("DescribeSeverityLevels").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(describeSeverityLevelsRequest)
                .withMarshaller(new DescribeSeverityLevelsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns the refresh status of the Trusted Advisor checks that have the specified check IDs. Check IDs can be
     * obtained by calling <a>DescribeTrustedAdvisorChecks</a>.
     * </p>
     * <note>
     * <p>
     * Some checks are refreshed automatically, and their refresh statuses cannot be retrieved by using this operation.
     * Use of the <code>DescribeTrustedAdvisorCheckRefreshStatuses</code> operation for these checks causes an
     * <code>InvalidParameterValue</code> error.
     * </p>
     * </note>
     *
     * @param describeTrustedAdvisorCheckRefreshStatusesRequest
     * @return Result of the DescribeTrustedAdvisorCheckRefreshStatuses operation returned by the service.
     * @throws InternalServerErrorException
     *         An internal server error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SupportException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SupportClient.DescribeTrustedAdvisorCheckRefreshStatuses
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/support-2013-04-15/DescribeTrustedAdvisorCheckRefreshStatuses"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeTrustedAdvisorCheckRefreshStatusesResponse describeTrustedAdvisorCheckRefreshStatuses(
            DescribeTrustedAdvisorCheckRefreshStatusesRequest describeTrustedAdvisorCheckRefreshStatusesRequest)
            throws InternalServerErrorException, AwsServiceException, SdkClientException, SupportException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeTrustedAdvisorCheckRefreshStatusesResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, DescribeTrustedAdvisorCheckRefreshStatusesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<DescribeTrustedAdvisorCheckRefreshStatusesRequest, DescribeTrustedAdvisorCheckRefreshStatusesResponse>()
                        .withOperationName("DescribeTrustedAdvisorCheckRefreshStatuses").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler)
                        .withInput(describeTrustedAdvisorCheckRefreshStatusesRequest)
                        .withMarshaller(new DescribeTrustedAdvisorCheckRefreshStatusesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns the results of the Trusted Advisor check that has the specified check ID. Check IDs can be obtained by
     * calling <a>DescribeTrustedAdvisorChecks</a>.
     * </p>
     * <p>
     * The response contains a <a>TrustedAdvisorCheckResult</a> object, which contains these three objects:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>TrustedAdvisorCategorySpecificSummary</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>TrustedAdvisorResourceDetail</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>TrustedAdvisorResourcesSummary</a>
     * </p>
     * </li>
     * </ul>
     * <p>
     * In addition, the response contains these fields:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>status.</b> The alert status of the check: "ok" (green), "warning" (yellow), "error" (red), or
     * "not_available".
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>timestamp.</b> The time of the last refresh of the check.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>checkId.</b> The unique identifier for the check.
     * </p>
     * </li>
     * </ul>
     *
     * @param describeTrustedAdvisorCheckResultRequest
     * @return Result of the DescribeTrustedAdvisorCheckResult operation returned by the service.
     * @throws InternalServerErrorException
     *         An internal server error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SupportException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SupportClient.DescribeTrustedAdvisorCheckResult
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/support-2013-04-15/DescribeTrustedAdvisorCheckResult"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeTrustedAdvisorCheckResultResponse describeTrustedAdvisorCheckResult(
            DescribeTrustedAdvisorCheckResultRequest describeTrustedAdvisorCheckResultRequest)
            throws InternalServerErrorException, AwsServiceException, SdkClientException, SupportException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeTrustedAdvisorCheckResultResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeTrustedAdvisorCheckResultResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<DescribeTrustedAdvisorCheckResultRequest, DescribeTrustedAdvisorCheckResultResponse>()
                        .withOperationName("DescribeTrustedAdvisorCheckResult").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(describeTrustedAdvisorCheckResultRequest)
                        .withMarshaller(new DescribeTrustedAdvisorCheckResultRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns the summaries of the results of the Trusted Advisor checks that have the specified check IDs. Check IDs
     * can be obtained by calling <a>DescribeTrustedAdvisorChecks</a>.
     * </p>
     * <p>
     * The response contains an array of <a>TrustedAdvisorCheckSummary</a> objects.
     * </p>
     *
     * @param describeTrustedAdvisorCheckSummariesRequest
     * @return Result of the DescribeTrustedAdvisorCheckSummaries operation returned by the service.
     * @throws InternalServerErrorException
     *         An internal server error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SupportException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SupportClient.DescribeTrustedAdvisorCheckSummaries
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/support-2013-04-15/DescribeTrustedAdvisorCheckSummaries"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeTrustedAdvisorCheckSummariesResponse describeTrustedAdvisorCheckSummaries(
            DescribeTrustedAdvisorCheckSummariesRequest describeTrustedAdvisorCheckSummariesRequest)
            throws InternalServerErrorException, AwsServiceException, SdkClientException, SupportException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeTrustedAdvisorCheckSummariesResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, DescribeTrustedAdvisorCheckSummariesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<DescribeTrustedAdvisorCheckSummariesRequest, DescribeTrustedAdvisorCheckSummariesResponse>()
                        .withOperationName("DescribeTrustedAdvisorCheckSummaries").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(describeTrustedAdvisorCheckSummariesRequest)
                        .withMarshaller(new DescribeTrustedAdvisorCheckSummariesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns information about all available Trusted Advisor checks, including name, ID, category, description, and
     * metadata. You must specify a language code; English ("en") and Japanese ("ja") are currently supported. The
     * response contains a <a>TrustedAdvisorCheckDescription</a> for each check.
     * </p>
     *
     * @param describeTrustedAdvisorChecksRequest
     * @return Result of the DescribeTrustedAdvisorChecks operation returned by the service.
     * @throws InternalServerErrorException
     *         An internal server error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SupportException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SupportClient.DescribeTrustedAdvisorChecks
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/support-2013-04-15/DescribeTrustedAdvisorChecks"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeTrustedAdvisorChecksResponse describeTrustedAdvisorChecks(
            DescribeTrustedAdvisorChecksRequest describeTrustedAdvisorChecksRequest) throws InternalServerErrorException,
            AwsServiceException, SdkClientException, SupportException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeTrustedAdvisorChecksResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeTrustedAdvisorChecksResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<DescribeTrustedAdvisorChecksRequest, DescribeTrustedAdvisorChecksResponse>()
                        .withOperationName("DescribeTrustedAdvisorChecks").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(describeTrustedAdvisorChecksRequest)
                        .withMarshaller(new DescribeTrustedAdvisorChecksRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Requests a refresh of the Trusted Advisor check that has the specified check ID. Check IDs can be obtained by
     * calling <a>DescribeTrustedAdvisorChecks</a>.
     * </p>
     * <note>
     * <p>
     * Some checks are refreshed automatically, and they cannot be refreshed by using this operation. Use of the
     * <code>RefreshTrustedAdvisorCheck</code> operation for these checks causes an <code>InvalidParameterValue</code>
     * error.
     * </p>
     * </note>
     * <p>
     * The response contains a <a>TrustedAdvisorCheckRefreshStatus</a> object, which contains these fields:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>status.</b> The refresh status of the check: "none", "enqueued", "processing", "success", or "abandoned".
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>millisUntilNextRefreshable.</b> The amount of time, in milliseconds, until the check is eligible for refresh.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>checkId.</b> The unique identifier for the check.
     * </p>
     * </li>
     * </ul>
     *
     * @param refreshTrustedAdvisorCheckRequest
     * @return Result of the RefreshTrustedAdvisorCheck operation returned by the service.
     * @throws InternalServerErrorException
     *         An internal server error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SupportException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SupportClient.RefreshTrustedAdvisorCheck
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/support-2013-04-15/RefreshTrustedAdvisorCheck"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public RefreshTrustedAdvisorCheckResponse refreshTrustedAdvisorCheck(
            RefreshTrustedAdvisorCheckRequest refreshTrustedAdvisorCheckRequest) throws InternalServerErrorException,
            AwsServiceException, SdkClientException, SupportException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RefreshTrustedAdvisorCheckResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, RefreshTrustedAdvisorCheckResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<RefreshTrustedAdvisorCheckRequest, RefreshTrustedAdvisorCheckResponse>()
                        .withOperationName("RefreshTrustedAdvisorCheck").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(refreshTrustedAdvisorCheckRequest)
                        .withMarshaller(new RefreshTrustedAdvisorCheckRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Takes a <code>caseId</code> and returns the initial state of the case along with the state of the case after the
     * call to <a>ResolveCase</a> completed.
     * </p>
     *
     * @param resolveCaseRequest
     * @return Result of the ResolveCase operation returned by the service.
     * @throws InternalServerErrorException
     *         An internal server error occurred.
     * @throws CaseIdNotFoundException
     *         The requested <code>caseId</code> could not be located.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SupportException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SupportClient.ResolveCase
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/support-2013-04-15/ResolveCase" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ResolveCaseResponse resolveCase(ResolveCaseRequest resolveCaseRequest) throws InternalServerErrorException,
            CaseIdNotFoundException, AwsServiceException, SdkClientException, SupportException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ResolveCaseResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ResolveCaseResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ResolveCaseRequest, ResolveCaseResponse>()
                .withOperationName("ResolveCase").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(resolveCaseRequest)
                .withMarshaller(new ResolveCaseRequestMarshaller(protocolFactory)));
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(SupportException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AttachmentSetIdNotFound")
                                .exceptionBuilderSupplier(AttachmentSetIdNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CaseCreationLimitExceeded")
                                .exceptionBuilderSupplier(CaseCreationLimitExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AttachmentIdNotFound")
                                .exceptionBuilderSupplier(AttachmentIdNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AttachmentSetExpired")
                                .exceptionBuilderSupplier(AttachmentSetExpiredException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AttachmentLimitExceeded")
                                .exceptionBuilderSupplier(AttachmentLimitExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CaseIdNotFound")
                                .exceptionBuilderSupplier(CaseIdNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DescribeAttachmentLimitExceeded")
                                .exceptionBuilderSupplier(DescribeAttachmentLimitExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerError")
                                .exceptionBuilderSupplier(InternalServerErrorException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AttachmentSetSizeLimitExceeded")
                                .exceptionBuilderSupplier(AttachmentSetSizeLimitExceededException::builder).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends SupportRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
