/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.supportapp;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import java.util.function.Function;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.supportapp.internal.ServiceVersionInfo;
import software.amazon.awssdk.services.supportapp.internal.SupportAppServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.supportapp.model.AccessDeniedException;
import software.amazon.awssdk.services.supportapp.model.ConflictException;
import software.amazon.awssdk.services.supportapp.model.CreateSlackChannelConfigurationRequest;
import software.amazon.awssdk.services.supportapp.model.CreateSlackChannelConfigurationResponse;
import software.amazon.awssdk.services.supportapp.model.DeleteAccountAliasRequest;
import software.amazon.awssdk.services.supportapp.model.DeleteAccountAliasResponse;
import software.amazon.awssdk.services.supportapp.model.DeleteSlackChannelConfigurationRequest;
import software.amazon.awssdk.services.supportapp.model.DeleteSlackChannelConfigurationResponse;
import software.amazon.awssdk.services.supportapp.model.DeleteSlackWorkspaceConfigurationRequest;
import software.amazon.awssdk.services.supportapp.model.DeleteSlackWorkspaceConfigurationResponse;
import software.amazon.awssdk.services.supportapp.model.GetAccountAliasRequest;
import software.amazon.awssdk.services.supportapp.model.GetAccountAliasResponse;
import software.amazon.awssdk.services.supportapp.model.InternalServerException;
import software.amazon.awssdk.services.supportapp.model.ListSlackChannelConfigurationsRequest;
import software.amazon.awssdk.services.supportapp.model.ListSlackChannelConfigurationsResponse;
import software.amazon.awssdk.services.supportapp.model.ListSlackWorkspaceConfigurationsRequest;
import software.amazon.awssdk.services.supportapp.model.ListSlackWorkspaceConfigurationsResponse;
import software.amazon.awssdk.services.supportapp.model.PutAccountAliasRequest;
import software.amazon.awssdk.services.supportapp.model.PutAccountAliasResponse;
import software.amazon.awssdk.services.supportapp.model.RegisterSlackWorkspaceForOrganizationRequest;
import software.amazon.awssdk.services.supportapp.model.RegisterSlackWorkspaceForOrganizationResponse;
import software.amazon.awssdk.services.supportapp.model.ResourceNotFoundException;
import software.amazon.awssdk.services.supportapp.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.supportapp.model.SupportAppException;
import software.amazon.awssdk.services.supportapp.model.UpdateSlackChannelConfigurationRequest;
import software.amazon.awssdk.services.supportapp.model.UpdateSlackChannelConfigurationResponse;
import software.amazon.awssdk.services.supportapp.model.ValidationException;
import software.amazon.awssdk.services.supportapp.transform.CreateSlackChannelConfigurationRequestMarshaller;
import software.amazon.awssdk.services.supportapp.transform.DeleteAccountAliasRequestMarshaller;
import software.amazon.awssdk.services.supportapp.transform.DeleteSlackChannelConfigurationRequestMarshaller;
import software.amazon.awssdk.services.supportapp.transform.DeleteSlackWorkspaceConfigurationRequestMarshaller;
import software.amazon.awssdk.services.supportapp.transform.GetAccountAliasRequestMarshaller;
import software.amazon.awssdk.services.supportapp.transform.ListSlackChannelConfigurationsRequestMarshaller;
import software.amazon.awssdk.services.supportapp.transform.ListSlackWorkspaceConfigurationsRequestMarshaller;
import software.amazon.awssdk.services.supportapp.transform.PutAccountAliasRequestMarshaller;
import software.amazon.awssdk.services.supportapp.transform.RegisterSlackWorkspaceForOrganizationRequestMarshaller;
import software.amazon.awssdk.services.supportapp.transform.UpdateSlackChannelConfigurationRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link SupportAppAsyncClient}.
 *
 * @see SupportAppAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultSupportAppAsyncClient implements SupportAppAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultSupportAppAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultSupportAppAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this)
                .option(SdkClientOption.API_METADATA, "Support_App" + "#" + ServiceVersionInfo.VERSION).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a Slack channel configuration for your Amazon Web Services account.
     * </p>
     * <note>
     * <ul>
     * <li>
     * <p>
     * You can add up to 5 Slack workspaces for your account.
     * </p>
     * </li>
     * <li>
     * <p>
     * You can add up to 20 Slack channels for your account.
     * </p>
     * </li>
     * </ul>
     * </note>
     * <p>
     * A Slack channel can have up to 100 Amazon Web Services accounts. This means that only 100 accounts can add the
     * same Slack channel to the Amazon Web Services Support App. We recommend that you only add the accounts that you
     * need to manage support cases for your organization. This can reduce the notifications about case updates that you
     * receive in the Slack channel.
     * </p>
     * <note>
     * <p>
     * We recommend that you choose a private Slack channel so that only members in that channel have read and write
     * access to your support cases. Anyone in your Slack channel can create, update, or resolve support cases for your
     * account. Users require an invitation to join private channels.
     * </p>
     * </note>
     *
     * @param createSlackChannelConfigurationRequest
     * @return A Java Future containing the result of the CreateSlackChannelConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException Your Service Quotas request exceeds the quota for the service. For
     *         example, your Service Quotas request to Amazon Web Services Support App might exceed the maximum number
     *         of workspaces or channels per account, or the maximum number of accounts per Slack channel.</li>
     *         <li>ConflictException Your request has a conflict. For example, you might receive this error if you try
     *         the following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Add, update, or delete a Slack channel configuration before you add a Slack workspace to your Amazon Web
     *         Services account.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Add a Slack channel configuration that already exists in your Amazon Web Services account.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Delete a Slack channel configuration for a live chat channel.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Delete a Slack workspace from your Amazon Web Services account that has an active live chat channel.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Call the <code>RegisterSlackWorkspaceForOrganization</code> API from an Amazon Web Services account that
     *         doesn't belong to an organization.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Call the <code>RegisterSlackWorkspaceForOrganization</code> API from a member account, but the management
     *         account hasn't registered that workspace yet for the organization.
     *         </p>
     *         </li></li>
     *         <li>AccessDeniedException You don't have sufficient permission to perform this action.</li>
     *         <li>InternalServerException We can’t process your request right now because of a server issue. Try again
     *         later.</li>
     *         <li>ValidationException Your request input doesn't meet the constraints that the Amazon Web Services
     *         Support App specifies.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SupportAppException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SupportAppAsyncClient.CreateSlackChannelConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/support-app-2021-08-20/CreateSlackChannelConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateSlackChannelConfigurationResponse> createSlackChannelConfiguration(
            CreateSlackChannelConfigurationRequest createSlackChannelConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createSlackChannelConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                createSlackChannelConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Support App");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateSlackChannelConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateSlackChannelConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateSlackChannelConfigurationResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<CreateSlackChannelConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateSlackChannelConfigurationRequest, CreateSlackChannelConfigurationResponse>()
                            .withOperationName("CreateSlackChannelConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateSlackChannelConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createSlackChannelConfigurationRequest));
            CompletableFuture<CreateSlackChannelConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an alias for an Amazon Web Services account ID. The alias appears in the Amazon Web Services Support App
     * page of the Amazon Web Services Support Center. The alias also appears in Slack messages from the Amazon Web
     * Services Support App.
     * </p>
     *
     * @param deleteAccountAliasRequest
     * @return A Java Future containing the result of the DeleteAccountAlias operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource is missing or doesn't exist, such as an account
     *         alias, Slack channel configuration, or Slack workspace configuration.</li>
     *         <li>AccessDeniedException You don't have sufficient permission to perform this action.</li>
     *         <li>InternalServerException We can’t process your request right now because of a server issue. Try again
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SupportAppException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SupportAppAsyncClient.DeleteAccountAlias
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/support-app-2021-08-20/DeleteAccountAlias"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteAccountAliasResponse> deleteAccountAlias(DeleteAccountAliasRequest deleteAccountAliasRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteAccountAliasRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAccountAliasRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Support App");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAccountAlias");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteAccountAliasResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteAccountAliasResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DeleteAccountAliasResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteAccountAliasRequest, DeleteAccountAliasResponse>()
                            .withOperationName("DeleteAccountAlias").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteAccountAliasRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteAccountAliasRequest));
            CompletableFuture<DeleteAccountAliasResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a Slack channel configuration from your Amazon Web Services account. This operation doesn't delete your
     * Slack channel.
     * </p>
     *
     * @param deleteSlackChannelConfigurationRequest
     * @return A Java Future containing the result of the DeleteSlackChannelConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ConflictException Your request has a conflict. For example, you might receive this error if you try
     *         the following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Add, update, or delete a Slack channel configuration before you add a Slack workspace to your Amazon Web
     *         Services account.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Add a Slack channel configuration that already exists in your Amazon Web Services account.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Delete a Slack channel configuration for a live chat channel.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Delete a Slack workspace from your Amazon Web Services account that has an active live chat channel.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Call the <code>RegisterSlackWorkspaceForOrganization</code> API from an Amazon Web Services account that
     *         doesn't belong to an organization.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Call the <code>RegisterSlackWorkspaceForOrganization</code> API from a member account, but the management
     *         account hasn't registered that workspace yet for the organization.
     *         </p>
     *         </li></li>
     *         <li>ResourceNotFoundException The specified resource is missing or doesn't exist, such as an account
     *         alias, Slack channel configuration, or Slack workspace configuration.</li>
     *         <li>AccessDeniedException You don't have sufficient permission to perform this action.</li>
     *         <li>InternalServerException We can’t process your request right now because of a server issue. Try again
     *         later.</li>
     *         <li>ValidationException Your request input doesn't meet the constraints that the Amazon Web Services
     *         Support App specifies.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SupportAppException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SupportAppAsyncClient.DeleteSlackChannelConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/support-app-2021-08-20/DeleteSlackChannelConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteSlackChannelConfigurationResponse> deleteSlackChannelConfiguration(
            DeleteSlackChannelConfigurationRequest deleteSlackChannelConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteSlackChannelConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteSlackChannelConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Support App");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSlackChannelConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteSlackChannelConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteSlackChannelConfigurationResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DeleteSlackChannelConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteSlackChannelConfigurationRequest, DeleteSlackChannelConfigurationResponse>()
                            .withOperationName("DeleteSlackChannelConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteSlackChannelConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteSlackChannelConfigurationRequest));
            CompletableFuture<DeleteSlackChannelConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a Slack workspace configuration from your Amazon Web Services account. This operation doesn't delete your
     * Slack workspace.
     * </p>
     *
     * @param deleteSlackWorkspaceConfigurationRequest
     * @return A Java Future containing the result of the DeleteSlackWorkspaceConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ConflictException Your request has a conflict. For example, you might receive this error if you try
     *         the following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Add, update, or delete a Slack channel configuration before you add a Slack workspace to your Amazon Web
     *         Services account.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Add a Slack channel configuration that already exists in your Amazon Web Services account.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Delete a Slack channel configuration for a live chat channel.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Delete a Slack workspace from your Amazon Web Services account that has an active live chat channel.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Call the <code>RegisterSlackWorkspaceForOrganization</code> API from an Amazon Web Services account that
     *         doesn't belong to an organization.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Call the <code>RegisterSlackWorkspaceForOrganization</code> API from a member account, but the management
     *         account hasn't registered that workspace yet for the organization.
     *         </p>
     *         </li></li>
     *         <li>ResourceNotFoundException The specified resource is missing or doesn't exist, such as an account
     *         alias, Slack channel configuration, or Slack workspace configuration.</li>
     *         <li>AccessDeniedException You don't have sufficient permission to perform this action.</li>
     *         <li>InternalServerException We can’t process your request right now because of a server issue. Try again
     *         later.</li>
     *         <li>ValidationException Your request input doesn't meet the constraints that the Amazon Web Services
     *         Support App specifies.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SupportAppException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SupportAppAsyncClient.DeleteSlackWorkspaceConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/support-app-2021-08-20/DeleteSlackWorkspaceConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteSlackWorkspaceConfigurationResponse> deleteSlackWorkspaceConfiguration(
            DeleteSlackWorkspaceConfigurationRequest deleteSlackWorkspaceConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteSlackWorkspaceConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteSlackWorkspaceConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Support App");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSlackWorkspaceConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteSlackWorkspaceConfigurationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DeleteSlackWorkspaceConfigurationResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DeleteSlackWorkspaceConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteSlackWorkspaceConfigurationRequest, DeleteSlackWorkspaceConfigurationResponse>()
                            .withOperationName("DeleteSlackWorkspaceConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteSlackWorkspaceConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteSlackWorkspaceConfigurationRequest));
            CompletableFuture<DeleteSlackWorkspaceConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the alias from an Amazon Web Services account ID. The alias appears in the Amazon Web Services Support
     * App page of the Amazon Web Services Support Center. The alias also appears in Slack messages from the Amazon Web
     * Services Support App.
     * </p>
     *
     * @param getAccountAliasRequest
     * @return A Java Future containing the result of the GetAccountAlias operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException We can’t process your request right now because of a server issue. Try again
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SupportAppException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SupportAppAsyncClient.GetAccountAlias
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/support-app-2021-08-20/GetAccountAlias" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetAccountAliasResponse> getAccountAlias(GetAccountAliasRequest getAccountAliasRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getAccountAliasRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAccountAliasRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Support App");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAccountAlias");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetAccountAliasResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetAccountAliasResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetAccountAliasResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAccountAliasRequest, GetAccountAliasResponse>()
                            .withOperationName("GetAccountAlias").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetAccountAliasRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getAccountAliasRequest));
            CompletableFuture<GetAccountAliasResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the Slack channel configurations for an Amazon Web Services account.
     * </p>
     *
     * @param listSlackChannelConfigurationsRequest
     * @return A Java Future containing the result of the ListSlackChannelConfigurations operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient permission to perform this action.</li>
     *         <li>InternalServerException We can’t process your request right now because of a server issue. Try again
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SupportAppException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SupportAppAsyncClient.ListSlackChannelConfigurations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/support-app-2021-08-20/ListSlackChannelConfigurations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListSlackChannelConfigurationsResponse> listSlackChannelConfigurations(
            ListSlackChannelConfigurationsRequest listSlackChannelConfigurationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listSlackChannelConfigurationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listSlackChannelConfigurationsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Support App");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSlackChannelConfigurations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListSlackChannelConfigurationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListSlackChannelConfigurationsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListSlackChannelConfigurationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListSlackChannelConfigurationsRequest, ListSlackChannelConfigurationsResponse>()
                            .withOperationName("ListSlackChannelConfigurations").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListSlackChannelConfigurationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listSlackChannelConfigurationsRequest));
            CompletableFuture<ListSlackChannelConfigurationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the Slack workspace configurations for an Amazon Web Services account.
     * </p>
     *
     * @param listSlackWorkspaceConfigurationsRequest
     * @return A Java Future containing the result of the ListSlackWorkspaceConfigurations operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient permission to perform this action.</li>
     *         <li>InternalServerException We can’t process your request right now because of a server issue. Try again
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SupportAppException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SupportAppAsyncClient.ListSlackWorkspaceConfigurations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/support-app-2021-08-20/ListSlackWorkspaceConfigurations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListSlackWorkspaceConfigurationsResponse> listSlackWorkspaceConfigurations(
            ListSlackWorkspaceConfigurationsRequest listSlackWorkspaceConfigurationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listSlackWorkspaceConfigurationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listSlackWorkspaceConfigurationsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Support App");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSlackWorkspaceConfigurations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListSlackWorkspaceConfigurationsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, ListSlackWorkspaceConfigurationsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListSlackWorkspaceConfigurationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListSlackWorkspaceConfigurationsRequest, ListSlackWorkspaceConfigurationsResponse>()
                            .withOperationName("ListSlackWorkspaceConfigurations").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListSlackWorkspaceConfigurationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listSlackWorkspaceConfigurationsRequest));
            CompletableFuture<ListSlackWorkspaceConfigurationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates or updates an individual alias for each Amazon Web Services account ID. The alias appears in the Amazon
     * Web Services Support App page of the Amazon Web Services Support Center. The alias also appears in Slack messages
     * from the Amazon Web Services Support App.
     * </p>
     *
     * @param putAccountAliasRequest
     * @return A Java Future containing the result of the PutAccountAlias operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient permission to perform this action.</li>
     *         <li>InternalServerException We can’t process your request right now because of a server issue. Try again
     *         later.</li>
     *         <li>ValidationException Your request input doesn't meet the constraints that the Amazon Web Services
     *         Support App specifies.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SupportAppException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SupportAppAsyncClient.PutAccountAlias
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/support-app-2021-08-20/PutAccountAlias" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<PutAccountAliasResponse> putAccountAlias(PutAccountAliasRequest putAccountAliasRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putAccountAliasRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putAccountAliasRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Support App");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutAccountAlias");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutAccountAliasResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutAccountAliasResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<PutAccountAliasResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutAccountAliasRequest, PutAccountAliasResponse>()
                            .withOperationName("PutAccountAlias").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PutAccountAliasRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putAccountAliasRequest));
            CompletableFuture<PutAccountAliasResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Registers a Slack workspace for your Amazon Web Services account. To call this API, your account must be part of
     * an organization in Organizations.
     * </p>
     * <p>
     * If you're the <i>management account</i> and you want to register Slack workspaces for your organization, you must
     * complete the following tasks:
     * </p>
     * <ol>
     * <li>
     * <p>
     * Sign in to the <a href="https://console.aws.amazon.com/support/app">Amazon Web Services Support Center</a> and
     * authorize the Slack workspaces where you want your organization to have access to. See <a
     * href="https://docs.aws.amazon.com/awssupport/latest/user/authorize-slack-workspace.html">Authorize a Slack
     * workspace</a> in the <i>Amazon Web Services Support User Guide</i>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Call the <code>RegisterSlackWorkspaceForOrganization</code> API to authorize each Slack workspace for the
     * organization.
     * </p>
     * </li>
     * </ol>
     * <p>
     * After the management account authorizes the Slack workspace, member accounts can call this API to authorize the
     * same Slack workspace for their individual accounts. Member accounts don't need to authorize the Slack workspace
     * manually through the <a href="https://console.aws.amazon.com/support/app">Amazon Web Services Support Center</a>.
     * </p>
     * <p>
     * To use the Amazon Web Services Support App, each account must then complete the following tasks:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Create an Identity and Access Management (IAM) role with the required permission. For more information, see <a
     * href="https://docs.aws.amazon.com/awssupport/latest/user/support-app-permissions.html">Managing access to the
     * Amazon Web Services Support App</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Configure a Slack channel to use the Amazon Web Services Support App for support cases for that account. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/awssupport/latest/user/add-your-slack-channel.html">Configuring a Slack
     * channel</a>.
     * </p>
     * </li>
     * </ul>
     *
     * @param registerSlackWorkspaceForOrganizationRequest
     * @return A Java Future containing the result of the RegisterSlackWorkspaceForOrganization operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ConflictException Your request has a conflict. For example, you might receive this error if you try
     *         the following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Add, update, or delete a Slack channel configuration before you add a Slack workspace to your Amazon Web
     *         Services account.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Add a Slack channel configuration that already exists in your Amazon Web Services account.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Delete a Slack channel configuration for a live chat channel.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Delete a Slack workspace from your Amazon Web Services account that has an active live chat channel.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Call the <code>RegisterSlackWorkspaceForOrganization</code> API from an Amazon Web Services account that
     *         doesn't belong to an organization.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Call the <code>RegisterSlackWorkspaceForOrganization</code> API from a member account, but the management
     *         account hasn't registered that workspace yet for the organization.
     *         </p>
     *         </li></li>
     *         <li>ResourceNotFoundException The specified resource is missing or doesn't exist, such as an account
     *         alias, Slack channel configuration, or Slack workspace configuration.</li>
     *         <li>AccessDeniedException You don't have sufficient permission to perform this action.</li>
     *         <li>InternalServerException We can’t process your request right now because of a server issue. Try again
     *         later.</li>
     *         <li>ValidationException Your request input doesn't meet the constraints that the Amazon Web Services
     *         Support App specifies.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SupportAppException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SupportAppAsyncClient.RegisterSlackWorkspaceForOrganization
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/support-app-2021-08-20/RegisterSlackWorkspaceForOrganization"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<RegisterSlackWorkspaceForOrganizationResponse> registerSlackWorkspaceForOrganization(
            RegisterSlackWorkspaceForOrganizationRequest registerSlackWorkspaceForOrganizationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(registerSlackWorkspaceForOrganizationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                registerSlackWorkspaceForOrganizationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Support App");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RegisterSlackWorkspaceForOrganization");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RegisterSlackWorkspaceForOrganizationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, RegisterSlackWorkspaceForOrganizationResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<RegisterSlackWorkspaceForOrganizationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RegisterSlackWorkspaceForOrganizationRequest, RegisterSlackWorkspaceForOrganizationResponse>()
                            .withOperationName("RegisterSlackWorkspaceForOrganization").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new RegisterSlackWorkspaceForOrganizationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(registerSlackWorkspaceForOrganizationRequest));
            CompletableFuture<RegisterSlackWorkspaceForOrganizationResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the configuration for a Slack channel, such as case update notifications.
     * </p>
     *
     * @param updateSlackChannelConfigurationRequest
     * @return A Java Future containing the result of the UpdateSlackChannelConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ConflictException Your request has a conflict. For example, you might receive this error if you try
     *         the following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Add, update, or delete a Slack channel configuration before you add a Slack workspace to your Amazon Web
     *         Services account.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Add a Slack channel configuration that already exists in your Amazon Web Services account.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Delete a Slack channel configuration for a live chat channel.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Delete a Slack workspace from your Amazon Web Services account that has an active live chat channel.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Call the <code>RegisterSlackWorkspaceForOrganization</code> API from an Amazon Web Services account that
     *         doesn't belong to an organization.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Call the <code>RegisterSlackWorkspaceForOrganization</code> API from a member account, but the management
     *         account hasn't registered that workspace yet for the organization.
     *         </p>
     *         </li></li>
     *         <li>ResourceNotFoundException The specified resource is missing or doesn't exist, such as an account
     *         alias, Slack channel configuration, or Slack workspace configuration.</li>
     *         <li>AccessDeniedException You don't have sufficient permission to perform this action.</li>
     *         <li>InternalServerException We can’t process your request right now because of a server issue. Try again
     *         later.</li>
     *         <li>ValidationException Your request input doesn't meet the constraints that the Amazon Web Services
     *         Support App specifies.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SupportAppException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SupportAppAsyncClient.UpdateSlackChannelConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/support-app-2021-08-20/UpdateSlackChannelConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateSlackChannelConfigurationResponse> updateSlackChannelConfiguration(
            UpdateSlackChannelConfigurationRequest updateSlackChannelConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateSlackChannelConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateSlackChannelConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Support App");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateSlackChannelConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateSlackChannelConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateSlackChannelConfigurationResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UpdateSlackChannelConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateSlackChannelConfigurationRequest, UpdateSlackChannelConfigurationResponse>()
                            .withOperationName("UpdateSlackChannelConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateSlackChannelConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateSlackChannelConfigurationRequest));
            CompletableFuture<UpdateSlackChannelConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final SupportAppServiceClientConfiguration serviceClientConfiguration() {
        return new SupportAppServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder.clientConfiguration(clientConfiguration).defaultServiceExceptionSupplier(SupportAppException::builder)
                .protocol(AwsJsonProtocol.REST_JSON).protocolVersion("1.1");
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        SupportAppServiceClientConfigurationBuilder serviceConfigBuilder = new SupportAppServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
