/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.swf.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information about a workflow execution.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class WorkflowExecutionInfo implements SdkPojo, Serializable,
        ToCopyableBuilder<WorkflowExecutionInfo.Builder, WorkflowExecutionInfo> {
    private static final SdkField<WorkflowExecution> EXECUTION_FIELD = SdkField
            .<WorkflowExecution> builder(MarshallingType.SDK_POJO).getter(getter(WorkflowExecutionInfo::execution))
            .setter(setter(Builder::execution)).constructor(WorkflowExecution::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("execution").build()).build();

    private static final SdkField<WorkflowType> WORKFLOW_TYPE_FIELD = SdkField.<WorkflowType> builder(MarshallingType.SDK_POJO)
            .getter(getter(WorkflowExecutionInfo::workflowType)).setter(setter(Builder::workflowType))
            .constructor(WorkflowType::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("workflowType").build()).build();

    private static final SdkField<Instant> START_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(WorkflowExecutionInfo::startTimestamp)).setter(setter(Builder::startTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("startTimestamp").build()).build();

    private static final SdkField<Instant> CLOSE_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(WorkflowExecutionInfo::closeTimestamp)).setter(setter(Builder::closeTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("closeTimestamp").build()).build();

    private static final SdkField<String> EXECUTION_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(WorkflowExecutionInfo::executionStatusAsString)).setter(setter(Builder::executionStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("executionStatus").build()).build();

    private static final SdkField<String> CLOSE_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(WorkflowExecutionInfo::closeStatusAsString)).setter(setter(Builder::closeStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("closeStatus").build()).build();

    private static final SdkField<WorkflowExecution> PARENT_FIELD = SdkField
            .<WorkflowExecution> builder(MarshallingType.SDK_POJO).getter(getter(WorkflowExecutionInfo::parent))
            .setter(setter(Builder::parent)).constructor(WorkflowExecution::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("parent").build()).build();

    private static final SdkField<List<String>> TAG_LIST_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(WorkflowExecutionInfo::tagList))
            .setter(setter(Builder::tagList))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tagList").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> CANCEL_REQUESTED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(WorkflowExecutionInfo::cancelRequested)).setter(setter(Builder::cancelRequested))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("cancelRequested").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(EXECUTION_FIELD,
            WORKFLOW_TYPE_FIELD, START_TIMESTAMP_FIELD, CLOSE_TIMESTAMP_FIELD, EXECUTION_STATUS_FIELD, CLOSE_STATUS_FIELD,
            PARENT_FIELD, TAG_LIST_FIELD, CANCEL_REQUESTED_FIELD));

    private static final long serialVersionUID = 1L;

    private final WorkflowExecution execution;

    private final WorkflowType workflowType;

    private final Instant startTimestamp;

    private final Instant closeTimestamp;

    private final String executionStatus;

    private final String closeStatus;

    private final WorkflowExecution parent;

    private final List<String> tagList;

    private final Boolean cancelRequested;

    private WorkflowExecutionInfo(BuilderImpl builder) {
        this.execution = builder.execution;
        this.workflowType = builder.workflowType;
        this.startTimestamp = builder.startTimestamp;
        this.closeTimestamp = builder.closeTimestamp;
        this.executionStatus = builder.executionStatus;
        this.closeStatus = builder.closeStatus;
        this.parent = builder.parent;
        this.tagList = builder.tagList;
        this.cancelRequested = builder.cancelRequested;
    }

    /**
     * <p>
     * The workflow execution this information is about.
     * </p>
     * 
     * @return The workflow execution this information is about.
     */
    public WorkflowExecution execution() {
        return execution;
    }

    /**
     * <p>
     * The type of the workflow execution.
     * </p>
     * 
     * @return The type of the workflow execution.
     */
    public WorkflowType workflowType() {
        return workflowType;
    }

    /**
     * <p>
     * The time when the execution was started.
     * </p>
     * 
     * @return The time when the execution was started.
     */
    public Instant startTimestamp() {
        return startTimestamp;
    }

    /**
     * <p>
     * The time when the workflow execution was closed. Set only if the execution status is CLOSED.
     * </p>
     * 
     * @return The time when the workflow execution was closed. Set only if the execution status is CLOSED.
     */
    public Instant closeTimestamp() {
        return closeTimestamp;
    }

    /**
     * <p>
     * The current status of the execution.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #executionStatus}
     * will return {@link ExecutionStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #executionStatusAsString}.
     * </p>
     * 
     * @return The current status of the execution.
     * @see ExecutionStatus
     */
    public ExecutionStatus executionStatus() {
        return ExecutionStatus.fromValue(executionStatus);
    }

    /**
     * <p>
     * The current status of the execution.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #executionStatus}
     * will return {@link ExecutionStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #executionStatusAsString}.
     * </p>
     * 
     * @return The current status of the execution.
     * @see ExecutionStatus
     */
    public String executionStatusAsString() {
        return executionStatus;
    }

    /**
     * <p>
     * If the execution status is closed then this specifies how the execution was closed:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>COMPLETED</code> – the execution was successfully completed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CANCELED</code> – the execution was canceled.Cancellation allows the implementation to gracefully clean up
     * before the execution is closed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>TERMINATED</code> – the execution was force terminated.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FAILED</code> – the execution failed to complete.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>TIMED_OUT</code> – the execution did not complete in the alloted time and was automatically timed out.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CONTINUED_AS_NEW</code> – the execution is logically continued. This means the current execution was
     * completed and a new execution was started to carry on the workflow.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #closeStatus} will
     * return {@link CloseStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #closeStatusAsString}.
     * </p>
     * 
     * @return If the execution status is closed then this specifies how the execution was closed:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>COMPLETED</code> – the execution was successfully completed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CANCELED</code> – the execution was canceled.Cancellation allows the implementation to gracefully
     *         clean up before the execution is closed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>TERMINATED</code> – the execution was force terminated.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FAILED</code> – the execution failed to complete.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>TIMED_OUT</code> – the execution did not complete in the alloted time and was automatically timed
     *         out.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CONTINUED_AS_NEW</code> – the execution is logically continued. This means the current execution
     *         was completed and a new execution was started to carry on the workflow.
     *         </p>
     *         </li>
     * @see CloseStatus
     */
    public CloseStatus closeStatus() {
        return CloseStatus.fromValue(closeStatus);
    }

    /**
     * <p>
     * If the execution status is closed then this specifies how the execution was closed:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>COMPLETED</code> – the execution was successfully completed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CANCELED</code> – the execution was canceled.Cancellation allows the implementation to gracefully clean up
     * before the execution is closed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>TERMINATED</code> – the execution was force terminated.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FAILED</code> – the execution failed to complete.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>TIMED_OUT</code> – the execution did not complete in the alloted time and was automatically timed out.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CONTINUED_AS_NEW</code> – the execution is logically continued. This means the current execution was
     * completed and a new execution was started to carry on the workflow.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #closeStatus} will
     * return {@link CloseStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #closeStatusAsString}.
     * </p>
     * 
     * @return If the execution status is closed then this specifies how the execution was closed:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>COMPLETED</code> – the execution was successfully completed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CANCELED</code> – the execution was canceled.Cancellation allows the implementation to gracefully
     *         clean up before the execution is closed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>TERMINATED</code> – the execution was force terminated.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FAILED</code> – the execution failed to complete.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>TIMED_OUT</code> – the execution did not complete in the alloted time and was automatically timed
     *         out.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CONTINUED_AS_NEW</code> – the execution is logically continued. This means the current execution
     *         was completed and a new execution was started to carry on the workflow.
     *         </p>
     *         </li>
     * @see CloseStatus
     */
    public String closeStatusAsString() {
        return closeStatus;
    }

    /**
     * <p>
     * If this workflow execution is a child of another execution then contains the workflow execution that started this
     * execution.
     * </p>
     * 
     * @return If this workflow execution is a child of another execution then contains the workflow execution that
     *         started this execution.
     */
    public WorkflowExecution parent() {
        return parent;
    }

    /**
     * <p>
     * The list of tags associated with the workflow execution. Tags can be used to identify and list workflow
     * executions of interest through the visibility APIs. A workflow execution can have a maximum of 5 tags.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The list of tags associated with the workflow execution. Tags can be used to identify and list workflow
     *         executions of interest through the visibility APIs. A workflow execution can have a maximum of 5 tags.
     */
    public List<String> tagList() {
        return tagList;
    }

    /**
     * <p>
     * Set to true if a cancellation is requested for this workflow execution.
     * </p>
     * 
     * @return Set to true if a cancellation is requested for this workflow execution.
     */
    public Boolean cancelRequested() {
        return cancelRequested;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(execution());
        hashCode = 31 * hashCode + Objects.hashCode(workflowType());
        hashCode = 31 * hashCode + Objects.hashCode(startTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(closeTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(executionStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(closeStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(parent());
        hashCode = 31 * hashCode + Objects.hashCode(tagList());
        hashCode = 31 * hashCode + Objects.hashCode(cancelRequested());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof WorkflowExecutionInfo)) {
            return false;
        }
        WorkflowExecutionInfo other = (WorkflowExecutionInfo) obj;
        return Objects.equals(execution(), other.execution()) && Objects.equals(workflowType(), other.workflowType())
                && Objects.equals(startTimestamp(), other.startTimestamp())
                && Objects.equals(closeTimestamp(), other.closeTimestamp())
                && Objects.equals(executionStatusAsString(), other.executionStatusAsString())
                && Objects.equals(closeStatusAsString(), other.closeStatusAsString()) && Objects.equals(parent(), other.parent())
                && Objects.equals(tagList(), other.tagList()) && Objects.equals(cancelRequested(), other.cancelRequested());
    }

    @Override
    public String toString() {
        return ToString.builder("WorkflowExecutionInfo").add("Execution", execution()).add("WorkflowType", workflowType())
                .add("StartTimestamp", startTimestamp()).add("CloseTimestamp", closeTimestamp())
                .add("ExecutionStatus", executionStatusAsString()).add("CloseStatus", closeStatusAsString())
                .add("Parent", parent()).add("TagList", tagList()).add("CancelRequested", cancelRequested()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "execution":
            return Optional.ofNullable(clazz.cast(execution()));
        case "workflowType":
            return Optional.ofNullable(clazz.cast(workflowType()));
        case "startTimestamp":
            return Optional.ofNullable(clazz.cast(startTimestamp()));
        case "closeTimestamp":
            return Optional.ofNullable(clazz.cast(closeTimestamp()));
        case "executionStatus":
            return Optional.ofNullable(clazz.cast(executionStatusAsString()));
        case "closeStatus":
            return Optional.ofNullable(clazz.cast(closeStatusAsString()));
        case "parent":
            return Optional.ofNullable(clazz.cast(parent()));
        case "tagList":
            return Optional.ofNullable(clazz.cast(tagList()));
        case "cancelRequested":
            return Optional.ofNullable(clazz.cast(cancelRequested()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<WorkflowExecutionInfo, T> g) {
        return obj -> g.apply((WorkflowExecutionInfo) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, WorkflowExecutionInfo> {
        /**
         * <p>
         * The workflow execution this information is about.
         * </p>
         * 
         * @param execution
         *        The workflow execution this information is about.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder execution(WorkflowExecution execution);

        /**
         * <p>
         * The workflow execution this information is about.
         * </p>
         * This is a convenience that creates an instance of the {@link WorkflowExecution.Builder} avoiding the need to
         * create one manually via {@link WorkflowExecution#builder()}.
         *
         * When the {@link Consumer} completes, {@link WorkflowExecution.Builder#build()} is called immediately and its
         * result is passed to {@link #execution(WorkflowExecution)}.
         * 
         * @param execution
         *        a consumer that will call methods on {@link WorkflowExecution.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #execution(WorkflowExecution)
         */
        default Builder execution(Consumer<WorkflowExecution.Builder> execution) {
            return execution(WorkflowExecution.builder().applyMutation(execution).build());
        }

        /**
         * <p>
         * The type of the workflow execution.
         * </p>
         * 
         * @param workflowType
         *        The type of the workflow execution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder workflowType(WorkflowType workflowType);

        /**
         * <p>
         * The type of the workflow execution.
         * </p>
         * This is a convenience that creates an instance of the {@link WorkflowType.Builder} avoiding the need to
         * create one manually via {@link WorkflowType#builder()}.
         *
         * When the {@link Consumer} completes, {@link WorkflowType.Builder#build()} is called immediately and its
         * result is passed to {@link #workflowType(WorkflowType)}.
         * 
         * @param workflowType
         *        a consumer that will call methods on {@link WorkflowType.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #workflowType(WorkflowType)
         */
        default Builder workflowType(Consumer<WorkflowType.Builder> workflowType) {
            return workflowType(WorkflowType.builder().applyMutation(workflowType).build());
        }

        /**
         * <p>
         * The time when the execution was started.
         * </p>
         * 
         * @param startTimestamp
         *        The time when the execution was started.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startTimestamp(Instant startTimestamp);

        /**
         * <p>
         * The time when the workflow execution was closed. Set only if the execution status is CLOSED.
         * </p>
         * 
         * @param closeTimestamp
         *        The time when the workflow execution was closed. Set only if the execution status is CLOSED.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder closeTimestamp(Instant closeTimestamp);

        /**
         * <p>
         * The current status of the execution.
         * </p>
         * 
         * @param executionStatus
         *        The current status of the execution.
         * @see ExecutionStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ExecutionStatus
         */
        Builder executionStatus(String executionStatus);

        /**
         * <p>
         * The current status of the execution.
         * </p>
         * 
         * @param executionStatus
         *        The current status of the execution.
         * @see ExecutionStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ExecutionStatus
         */
        Builder executionStatus(ExecutionStatus executionStatus);

        /**
         * <p>
         * If the execution status is closed then this specifies how the execution was closed:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>COMPLETED</code> – the execution was successfully completed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CANCELED</code> – the execution was canceled.Cancellation allows the implementation to gracefully clean
         * up before the execution is closed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>TERMINATED</code> – the execution was force terminated.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FAILED</code> – the execution failed to complete.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>TIMED_OUT</code> – the execution did not complete in the alloted time and was automatically timed out.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CONTINUED_AS_NEW</code> – the execution is logically continued. This means the current execution was
         * completed and a new execution was started to carry on the workflow.
         * </p>
         * </li>
         * </ul>
         * 
         * @param closeStatus
         *        If the execution status is closed then this specifies how the execution was closed:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>COMPLETED</code> – the execution was successfully completed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CANCELED</code> – the execution was canceled.Cancellation allows the implementation to
         *        gracefully clean up before the execution is closed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>TERMINATED</code> – the execution was force terminated.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FAILED</code> – the execution failed to complete.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>TIMED_OUT</code> – the execution did not complete in the alloted time and was automatically
         *        timed out.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CONTINUED_AS_NEW</code> – the execution is logically continued. This means the current execution
         *        was completed and a new execution was started to carry on the workflow.
         *        </p>
         *        </li>
         * @see CloseStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CloseStatus
         */
        Builder closeStatus(String closeStatus);

        /**
         * <p>
         * If the execution status is closed then this specifies how the execution was closed:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>COMPLETED</code> – the execution was successfully completed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CANCELED</code> – the execution was canceled.Cancellation allows the implementation to gracefully clean
         * up before the execution is closed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>TERMINATED</code> – the execution was force terminated.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FAILED</code> – the execution failed to complete.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>TIMED_OUT</code> – the execution did not complete in the alloted time and was automatically timed out.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CONTINUED_AS_NEW</code> – the execution is logically continued. This means the current execution was
         * completed and a new execution was started to carry on the workflow.
         * </p>
         * </li>
         * </ul>
         * 
         * @param closeStatus
         *        If the execution status is closed then this specifies how the execution was closed:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>COMPLETED</code> – the execution was successfully completed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CANCELED</code> – the execution was canceled.Cancellation allows the implementation to
         *        gracefully clean up before the execution is closed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>TERMINATED</code> – the execution was force terminated.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FAILED</code> – the execution failed to complete.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>TIMED_OUT</code> – the execution did not complete in the alloted time and was automatically
         *        timed out.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CONTINUED_AS_NEW</code> – the execution is logically continued. This means the current execution
         *        was completed and a new execution was started to carry on the workflow.
         *        </p>
         *        </li>
         * @see CloseStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CloseStatus
         */
        Builder closeStatus(CloseStatus closeStatus);

        /**
         * <p>
         * If this workflow execution is a child of another execution then contains the workflow execution that started
         * this execution.
         * </p>
         * 
         * @param parent
         *        If this workflow execution is a child of another execution then contains the workflow execution that
         *        started this execution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parent(WorkflowExecution parent);

        /**
         * <p>
         * If this workflow execution is a child of another execution then contains the workflow execution that started
         * this execution.
         * </p>
         * This is a convenience that creates an instance of the {@link WorkflowExecution.Builder} avoiding the need to
         * create one manually via {@link WorkflowExecution#builder()}.
         *
         * When the {@link Consumer} completes, {@link WorkflowExecution.Builder#build()} is called immediately and its
         * result is passed to {@link #parent(WorkflowExecution)}.
         * 
         * @param parent
         *        a consumer that will call methods on {@link WorkflowExecution.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #parent(WorkflowExecution)
         */
        default Builder parent(Consumer<WorkflowExecution.Builder> parent) {
            return parent(WorkflowExecution.builder().applyMutation(parent).build());
        }

        /**
         * <p>
         * The list of tags associated with the workflow execution. Tags can be used to identify and list workflow
         * executions of interest through the visibility APIs. A workflow execution can have a maximum of 5 tags.
         * </p>
         * 
         * @param tagList
         *        The list of tags associated with the workflow execution. Tags can be used to identify and list
         *        workflow executions of interest through the visibility APIs. A workflow execution can have a maximum
         *        of 5 tags.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tagList(Collection<String> tagList);

        /**
         * <p>
         * The list of tags associated with the workflow execution. Tags can be used to identify and list workflow
         * executions of interest through the visibility APIs. A workflow execution can have a maximum of 5 tags.
         * </p>
         * 
         * @param tagList
         *        The list of tags associated with the workflow execution. Tags can be used to identify and list
         *        workflow executions of interest through the visibility APIs. A workflow execution can have a maximum
         *        of 5 tags.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tagList(String... tagList);

        /**
         * <p>
         * Set to true if a cancellation is requested for this workflow execution.
         * </p>
         * 
         * @param cancelRequested
         *        Set to true if a cancellation is requested for this workflow execution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cancelRequested(Boolean cancelRequested);
    }

    static final class BuilderImpl implements Builder {
        private WorkflowExecution execution;

        private WorkflowType workflowType;

        private Instant startTimestamp;

        private Instant closeTimestamp;

        private String executionStatus;

        private String closeStatus;

        private WorkflowExecution parent;

        private List<String> tagList = DefaultSdkAutoConstructList.getInstance();

        private Boolean cancelRequested;

        private BuilderImpl() {
        }

        private BuilderImpl(WorkflowExecutionInfo model) {
            execution(model.execution);
            workflowType(model.workflowType);
            startTimestamp(model.startTimestamp);
            closeTimestamp(model.closeTimestamp);
            executionStatus(model.executionStatus);
            closeStatus(model.closeStatus);
            parent(model.parent);
            tagList(model.tagList);
            cancelRequested(model.cancelRequested);
        }

        public final WorkflowExecution.Builder getExecution() {
            return execution != null ? execution.toBuilder() : null;
        }

        @Override
        public final Builder execution(WorkflowExecution execution) {
            this.execution = execution;
            return this;
        }

        public final void setExecution(WorkflowExecution.BuilderImpl execution) {
            this.execution = execution != null ? execution.build() : null;
        }

        public final WorkflowType.Builder getWorkflowType() {
            return workflowType != null ? workflowType.toBuilder() : null;
        }

        @Override
        public final Builder workflowType(WorkflowType workflowType) {
            this.workflowType = workflowType;
            return this;
        }

        public final void setWorkflowType(WorkflowType.BuilderImpl workflowType) {
            this.workflowType = workflowType != null ? workflowType.build() : null;
        }

        public final Instant getStartTimestamp() {
            return startTimestamp;
        }

        @Override
        public final Builder startTimestamp(Instant startTimestamp) {
            this.startTimestamp = startTimestamp;
            return this;
        }

        public final void setStartTimestamp(Instant startTimestamp) {
            this.startTimestamp = startTimestamp;
        }

        public final Instant getCloseTimestamp() {
            return closeTimestamp;
        }

        @Override
        public final Builder closeTimestamp(Instant closeTimestamp) {
            this.closeTimestamp = closeTimestamp;
            return this;
        }

        public final void setCloseTimestamp(Instant closeTimestamp) {
            this.closeTimestamp = closeTimestamp;
        }

        public final String getExecutionStatus() {
            return executionStatus;
        }

        @Override
        public final Builder executionStatus(String executionStatus) {
            this.executionStatus = executionStatus;
            return this;
        }

        @Override
        public final Builder executionStatus(ExecutionStatus executionStatus) {
            this.executionStatus(executionStatus.toString());
            return this;
        }

        public final void setExecutionStatus(String executionStatus) {
            this.executionStatus = executionStatus;
        }

        public final String getCloseStatus() {
            return closeStatus;
        }

        @Override
        public final Builder closeStatus(String closeStatus) {
            this.closeStatus = closeStatus;
            return this;
        }

        @Override
        public final Builder closeStatus(CloseStatus closeStatus) {
            this.closeStatus(closeStatus.toString());
            return this;
        }

        public final void setCloseStatus(String closeStatus) {
            this.closeStatus = closeStatus;
        }

        public final WorkflowExecution.Builder getParent() {
            return parent != null ? parent.toBuilder() : null;
        }

        @Override
        public final Builder parent(WorkflowExecution parent) {
            this.parent = parent;
            return this;
        }

        public final void setParent(WorkflowExecution.BuilderImpl parent) {
            this.parent = parent != null ? parent.build() : null;
        }

        public final Collection<String> getTagList() {
            return tagList;
        }

        @Override
        public final Builder tagList(Collection<String> tagList) {
            this.tagList = TagListCopier.copy(tagList);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tagList(String... tagList) {
            tagList(Arrays.asList(tagList));
            return this;
        }

        public final void setTagList(Collection<String> tagList) {
            this.tagList = TagListCopier.copy(tagList);
        }

        public final Boolean getCancelRequested() {
            return cancelRequested;
        }

        @Override
        public final Builder cancelRequested(Boolean cancelRequested) {
            this.cancelRequested = cancelRequested;
            return this;
        }

        public final void setCancelRequested(Boolean cancelRequested) {
            this.cancelRequested = cancelRequested;
        }

        @Override
        public WorkflowExecutionInfo build() {
            return new WorkflowExecutionInfo(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
