/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.swf.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides the details of the <code>ContinueAsNewWorkflowExecution</code> decision.
 * </p>
 * <p>
 * <b>Access Control</b>
 * </p>
 * <p>
 * You can use IAM policies to control this decision's access to Amazon SWF resources as follows:
 * </p>
 * <ul>
 * <li>
 * <p>
 * Use a <code>Resource</code> element with the domain name to limit the action to only specified domains.
 * </p>
 * </li>
 * <li>
 * <p>
 * Use an <code>Action</code> element to allow or deny permission to call this action.
 * </p>
 * </li>
 * <li>
 * <p>
 * Constrain the following parameters by using a <code>Condition</code> element with the appropriate keys.
 * </p>
 * <ul>
 * <li>
 * <p>
 * <code>tag</code> – A tag used to identify the workflow execution
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>taskList</code> – String constraint. The key is <code>swf:taskList.name</code>.
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>workflowType.version</code> – String constraint. The key is <code>swf:workflowType.version</code>.
 * </p>
 * </li>
 * </ul>
 * </li>
 * </ul>
 * <p>
 * If the caller doesn't have sufficient permissions to invoke the action, or the parameter values fall outside the
 * specified constraints, the action fails. The associated event attribute's <code>cause</code> parameter is set to
 * <code>OPERATION_NOT_PERMITTED</code>. For details and example IAM policies, see <a
 * href="http://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html">Using IAM to Manage Access to
 * Amazon SWF Workflows</a> in the <i>Amazon SWF Developer Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ContinueAsNewWorkflowExecutionDecisionAttributes
        implements
        SdkPojo,
        Serializable,
        ToCopyableBuilder<ContinueAsNewWorkflowExecutionDecisionAttributes.Builder, ContinueAsNewWorkflowExecutionDecisionAttributes> {
    private static final SdkField<String> INPUT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ContinueAsNewWorkflowExecutionDecisionAttributes::input)).setter(setter(Builder::input))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("input").build()).build();

    private static final SdkField<String> EXECUTION_START_TO_CLOSE_TIMEOUT_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(ContinueAsNewWorkflowExecutionDecisionAttributes::executionStartToCloseTimeout))
            .setter(setter(Builder::executionStartToCloseTimeout))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("executionStartToCloseTimeout")
                    .build()).build();

    private static final SdkField<TaskList> TASK_LIST_FIELD = SdkField.<TaskList> builder(MarshallingType.SDK_POJO)
            .getter(getter(ContinueAsNewWorkflowExecutionDecisionAttributes::taskList)).setter(setter(Builder::taskList))
            .constructor(TaskList::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("taskList").build()).build();

    private static final SdkField<String> TASK_PRIORITY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ContinueAsNewWorkflowExecutionDecisionAttributes::taskPriority)).setter(setter(Builder::taskPriority))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("taskPriority").build()).build();

    private static final SdkField<String> TASK_START_TO_CLOSE_TIMEOUT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ContinueAsNewWorkflowExecutionDecisionAttributes::taskStartToCloseTimeout))
            .setter(setter(Builder::taskStartToCloseTimeout))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("taskStartToCloseTimeout").build())
            .build();

    private static final SdkField<String> CHILD_POLICY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ContinueAsNewWorkflowExecutionDecisionAttributes::childPolicyAsString))
            .setter(setter(Builder::childPolicy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("childPolicy").build()).build();

    private static final SdkField<List<String>> TAG_LIST_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(ContinueAsNewWorkflowExecutionDecisionAttributes::tagList))
            .setter(setter(Builder::tagList))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tagList").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> WORKFLOW_TYPE_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ContinueAsNewWorkflowExecutionDecisionAttributes::workflowTypeVersion))
            .setter(setter(Builder::workflowTypeVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("workflowTypeVersion").build())
            .build();

    private static final SdkField<String> LAMBDA_ROLE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ContinueAsNewWorkflowExecutionDecisionAttributes::lambdaRole)).setter(setter(Builder::lambdaRole))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lambdaRole").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(INPUT_FIELD,
            EXECUTION_START_TO_CLOSE_TIMEOUT_FIELD, TASK_LIST_FIELD, TASK_PRIORITY_FIELD, TASK_START_TO_CLOSE_TIMEOUT_FIELD,
            CHILD_POLICY_FIELD, TAG_LIST_FIELD, WORKFLOW_TYPE_VERSION_FIELD, LAMBDA_ROLE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String input;

    private final String executionStartToCloseTimeout;

    private final TaskList taskList;

    private final String taskPriority;

    private final String taskStartToCloseTimeout;

    private final String childPolicy;

    private final List<String> tagList;

    private final String workflowTypeVersion;

    private final String lambdaRole;

    private ContinueAsNewWorkflowExecutionDecisionAttributes(BuilderImpl builder) {
        this.input = builder.input;
        this.executionStartToCloseTimeout = builder.executionStartToCloseTimeout;
        this.taskList = builder.taskList;
        this.taskPriority = builder.taskPriority;
        this.taskStartToCloseTimeout = builder.taskStartToCloseTimeout;
        this.childPolicy = builder.childPolicy;
        this.tagList = builder.tagList;
        this.workflowTypeVersion = builder.workflowTypeVersion;
        this.lambdaRole = builder.lambdaRole;
    }

    /**
     * <p>
     * The input provided to the new workflow execution.
     * </p>
     * 
     * @return The input provided to the new workflow execution.
     */
    public String input() {
        return input;
    }

    /**
     * <p>
     * If set, specifies the total duration for this workflow execution. This overrides the
     * <code>defaultExecutionStartToCloseTimeout</code> specified when registering the workflow type.
     * </p>
     * <p>
     * The duration is specified in seconds, an integer greater than or equal to <code>0</code>. You can use
     * <code>NONE</code> to specify unlimited duration.
     * </p>
     * <note>
     * <p>
     * An execution start-to-close timeout for this workflow execution must be specified either as a default for the
     * workflow type or through this field. If neither this field is set nor a default execution start-to-close timeout
     * was specified at registration time then a fault is returned.
     * </p>
     * </note>
     * 
     * @return If set, specifies the total duration for this workflow execution. This overrides the
     *         <code>defaultExecutionStartToCloseTimeout</code> specified when registering the workflow type.</p>
     *         <p>
     *         The duration is specified in seconds, an integer greater than or equal to <code>0</code>. You can use
     *         <code>NONE</code> to specify unlimited duration.
     *         </p>
     *         <note>
     *         <p>
     *         An execution start-to-close timeout for this workflow execution must be specified either as a default for
     *         the workflow type or through this field. If neither this field is set nor a default execution
     *         start-to-close timeout was specified at registration time then a fault is returned.
     *         </p>
     */
    public String executionStartToCloseTimeout() {
        return executionStartToCloseTimeout;
    }

    /**
     * <p>
     * The task list to use for the decisions of the new (continued) workflow execution.
     * </p>
     * 
     * @return The task list to use for the decisions of the new (continued) workflow execution.
     */
    public TaskList taskList() {
        return taskList;
    }

    /**
     * <p>
     * The task priority that, if set, specifies the priority for the decision tasks for this workflow execution. This
     * overrides the defaultTaskPriority specified when registering the workflow type. Valid values are integers that
     * range from Java's <code>Integer.MIN_VALUE</code> (-2147483648) to <code>Integer.MAX_VALUE</code> (2147483647).
     * Higher numbers indicate higher priority.
     * </p>
     * <p>
     * For more information about setting task priority, see <a
     * href="http://docs.aws.amazon.com/amazonswf/latest/developerguide/programming-priority.html">Setting Task
     * Priority</a> in the <i>Amazon SWF Developer Guide</i>.
     * </p>
     * 
     * @return The task priority that, if set, specifies the priority for the decision tasks for this workflow
     *         execution. This overrides the defaultTaskPriority specified when registering the workflow type. Valid
     *         values are integers that range from Java's <code>Integer.MIN_VALUE</code> (-2147483648) to
     *         <code>Integer.MAX_VALUE</code> (2147483647). Higher numbers indicate higher priority.</p>
     *         <p>
     *         For more information about setting task priority, see <a
     *         href="http://docs.aws.amazon.com/amazonswf/latest/developerguide/programming-priority.html">Setting Task
     *         Priority</a> in the <i>Amazon SWF Developer Guide</i>.
     */
    public String taskPriority() {
        return taskPriority;
    }

    /**
     * <p>
     * Specifies the maximum duration of decision tasks for the new workflow execution. This parameter overrides the
     * <code>defaultTaskStartToCloseTimout</code> specified when registering the workflow type using
     * <a>RegisterWorkflowType</a>.
     * </p>
     * <p>
     * The duration is specified in seconds, an integer greater than or equal to <code>0</code>. You can use
     * <code>NONE</code> to specify unlimited duration.
     * </p>
     * <note>
     * <p>
     * A task start-to-close timeout for the new workflow execution must be specified either as a default for the
     * workflow type or through this parameter. If neither this parameter is set nor a default task start-to-close
     * timeout was specified at registration time then a fault is returned.
     * </p>
     * </note>
     * 
     * @return Specifies the maximum duration of decision tasks for the new workflow execution. This parameter overrides
     *         the <code>defaultTaskStartToCloseTimout</code> specified when registering the workflow type using
     *         <a>RegisterWorkflowType</a>.</p>
     *         <p>
     *         The duration is specified in seconds, an integer greater than or equal to <code>0</code>. You can use
     *         <code>NONE</code> to specify unlimited duration.
     *         </p>
     *         <note>
     *         <p>
     *         A task start-to-close timeout for the new workflow execution must be specified either as a default for
     *         the workflow type or through this parameter. If neither this parameter is set nor a default task
     *         start-to-close timeout was specified at registration time then a fault is returned.
     *         </p>
     */
    public String taskStartToCloseTimeout() {
        return taskStartToCloseTimeout;
    }

    /**
     * <p>
     * If set, specifies the policy to use for the child workflow executions of the new execution if it is terminated by
     * calling the <a>TerminateWorkflowExecution</a> action explicitly or due to an expired timeout. This policy
     * overrides the default child policy specified when registering the workflow type using
     * <a>RegisterWorkflowType</a>.
     * </p>
     * <p>
     * The supported child policies are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>TERMINATE</code> – The child executions are terminated.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>REQUEST_CANCEL</code> – A request to cancel is attempted for each child execution by recording a
     * <code>WorkflowExecutionCancelRequested</code> event in its history. It is up to the decider to take appropriate
     * actions when it receives an execution history with this event.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ABANDON</code> – No action is taken. The child executions continue to run.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * A child policy for this workflow execution must be specified either as a default for the workflow type or through
     * this parameter. If neither this parameter is set nor a default child policy was specified at registration time
     * then a fault is returned.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #childPolicy} will
     * return {@link ChildPolicy#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #childPolicyAsString}.
     * </p>
     * 
     * @return If set, specifies the policy to use for the child workflow executions of the new execution if it is
     *         terminated by calling the <a>TerminateWorkflowExecution</a> action explicitly or due to an expired
     *         timeout. This policy overrides the default child policy specified when registering the workflow type
     *         using <a>RegisterWorkflowType</a>.</p>
     *         <p>
     *         The supported child policies are:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>TERMINATE</code> – The child executions are terminated.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>REQUEST_CANCEL</code> – A request to cancel is attempted for each child execution by recording a
     *         <code>WorkflowExecutionCancelRequested</code> event in its history. It is up to the decider to take
     *         appropriate actions when it receives an execution history with this event.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ABANDON</code> – No action is taken. The child executions continue to run.
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         A child policy for this workflow execution must be specified either as a default for the workflow type or
     *         through this parameter. If neither this parameter is set nor a default child policy was specified at
     *         registration time then a fault is returned.
     *         </p>
     * @see ChildPolicy
     */
    public ChildPolicy childPolicy() {
        return ChildPolicy.fromValue(childPolicy);
    }

    /**
     * <p>
     * If set, specifies the policy to use for the child workflow executions of the new execution if it is terminated by
     * calling the <a>TerminateWorkflowExecution</a> action explicitly or due to an expired timeout. This policy
     * overrides the default child policy specified when registering the workflow type using
     * <a>RegisterWorkflowType</a>.
     * </p>
     * <p>
     * The supported child policies are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>TERMINATE</code> – The child executions are terminated.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>REQUEST_CANCEL</code> – A request to cancel is attempted for each child execution by recording a
     * <code>WorkflowExecutionCancelRequested</code> event in its history. It is up to the decider to take appropriate
     * actions when it receives an execution history with this event.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ABANDON</code> – No action is taken. The child executions continue to run.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * A child policy for this workflow execution must be specified either as a default for the workflow type or through
     * this parameter. If neither this parameter is set nor a default child policy was specified at registration time
     * then a fault is returned.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #childPolicy} will
     * return {@link ChildPolicy#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #childPolicyAsString}.
     * </p>
     * 
     * @return If set, specifies the policy to use for the child workflow executions of the new execution if it is
     *         terminated by calling the <a>TerminateWorkflowExecution</a> action explicitly or due to an expired
     *         timeout. This policy overrides the default child policy specified when registering the workflow type
     *         using <a>RegisterWorkflowType</a>.</p>
     *         <p>
     *         The supported child policies are:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>TERMINATE</code> – The child executions are terminated.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>REQUEST_CANCEL</code> – A request to cancel is attempted for each child execution by recording a
     *         <code>WorkflowExecutionCancelRequested</code> event in its history. It is up to the decider to take
     *         appropriate actions when it receives an execution history with this event.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ABANDON</code> – No action is taken. The child executions continue to run.
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         A child policy for this workflow execution must be specified either as a default for the workflow type or
     *         through this parameter. If neither this parameter is set nor a default child policy was specified at
     *         registration time then a fault is returned.
     *         </p>
     * @see ChildPolicy
     */
    public String childPolicyAsString() {
        return childPolicy;
    }

    /**
     * <p>
     * The list of tags to associate with the new workflow execution. A maximum of 5 tags can be specified. You can list
     * workflow executions with a specific tag by calling <a>ListOpenWorkflowExecutions</a> or
     * <a>ListClosedWorkflowExecutions</a> and specifying a <a>TagFilter</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The list of tags to associate with the new workflow execution. A maximum of 5 tags can be specified. You
     *         can list workflow executions with a specific tag by calling <a>ListOpenWorkflowExecutions</a> or
     *         <a>ListClosedWorkflowExecutions</a> and specifying a <a>TagFilter</a>.
     */
    public List<String> tagList() {
        return tagList;
    }

    /**
     * <p>
     * The version of the workflow to start.
     * </p>
     * 
     * @return The version of the workflow to start.
     */
    public String workflowTypeVersion() {
        return workflowTypeVersion;
    }

    /**
     * <p>
     * The IAM role to attach to the new (continued) execution.
     * </p>
     * 
     * @return The IAM role to attach to the new (continued) execution.
     */
    public String lambdaRole() {
        return lambdaRole;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(input());
        hashCode = 31 * hashCode + Objects.hashCode(executionStartToCloseTimeout());
        hashCode = 31 * hashCode + Objects.hashCode(taskList());
        hashCode = 31 * hashCode + Objects.hashCode(taskPriority());
        hashCode = 31 * hashCode + Objects.hashCode(taskStartToCloseTimeout());
        hashCode = 31 * hashCode + Objects.hashCode(childPolicyAsString());
        hashCode = 31 * hashCode + Objects.hashCode(tagList());
        hashCode = 31 * hashCode + Objects.hashCode(workflowTypeVersion());
        hashCode = 31 * hashCode + Objects.hashCode(lambdaRole());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ContinueAsNewWorkflowExecutionDecisionAttributes)) {
            return false;
        }
        ContinueAsNewWorkflowExecutionDecisionAttributes other = (ContinueAsNewWorkflowExecutionDecisionAttributes) obj;
        return Objects.equals(input(), other.input())
                && Objects.equals(executionStartToCloseTimeout(), other.executionStartToCloseTimeout())
                && Objects.equals(taskList(), other.taskList()) && Objects.equals(taskPriority(), other.taskPriority())
                && Objects.equals(taskStartToCloseTimeout(), other.taskStartToCloseTimeout())
                && Objects.equals(childPolicyAsString(), other.childPolicyAsString())
                && Objects.equals(tagList(), other.tagList())
                && Objects.equals(workflowTypeVersion(), other.workflowTypeVersion())
                && Objects.equals(lambdaRole(), other.lambdaRole());
    }

    @Override
    public String toString() {
        return ToString.builder("ContinueAsNewWorkflowExecutionDecisionAttributes").add("Input", input())
                .add("ExecutionStartToCloseTimeout", executionStartToCloseTimeout()).add("TaskList", taskList())
                .add("TaskPriority", taskPriority()).add("TaskStartToCloseTimeout", taskStartToCloseTimeout())
                .add("ChildPolicy", childPolicyAsString()).add("TagList", tagList())
                .add("WorkflowTypeVersion", workflowTypeVersion()).add("LambdaRole", lambdaRole()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "input":
            return Optional.ofNullable(clazz.cast(input()));
        case "executionStartToCloseTimeout":
            return Optional.ofNullable(clazz.cast(executionStartToCloseTimeout()));
        case "taskList":
            return Optional.ofNullable(clazz.cast(taskList()));
        case "taskPriority":
            return Optional.ofNullable(clazz.cast(taskPriority()));
        case "taskStartToCloseTimeout":
            return Optional.ofNullable(clazz.cast(taskStartToCloseTimeout()));
        case "childPolicy":
            return Optional.ofNullable(clazz.cast(childPolicyAsString()));
        case "tagList":
            return Optional.ofNullable(clazz.cast(tagList()));
        case "workflowTypeVersion":
            return Optional.ofNullable(clazz.cast(workflowTypeVersion()));
        case "lambdaRole":
            return Optional.ofNullable(clazz.cast(lambdaRole()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ContinueAsNewWorkflowExecutionDecisionAttributes, T> g) {
        return obj -> g.apply((ContinueAsNewWorkflowExecutionDecisionAttributes) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ContinueAsNewWorkflowExecutionDecisionAttributes> {
        /**
         * <p>
         * The input provided to the new workflow execution.
         * </p>
         * 
         * @param input
         *        The input provided to the new workflow execution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder input(String input);

        /**
         * <p>
         * If set, specifies the total duration for this workflow execution. This overrides the
         * <code>defaultExecutionStartToCloseTimeout</code> specified when registering the workflow type.
         * </p>
         * <p>
         * The duration is specified in seconds, an integer greater than or equal to <code>0</code>. You can use
         * <code>NONE</code> to specify unlimited duration.
         * </p>
         * <note>
         * <p>
         * An execution start-to-close timeout for this workflow execution must be specified either as a default for the
         * workflow type or through this field. If neither this field is set nor a default execution start-to-close
         * timeout was specified at registration time then a fault is returned.
         * </p>
         * </note>
         * 
         * @param executionStartToCloseTimeout
         *        If set, specifies the total duration for this workflow execution. This overrides the
         *        <code>defaultExecutionStartToCloseTimeout</code> specified when registering the workflow type.</p>
         *        <p>
         *        The duration is specified in seconds, an integer greater than or equal to <code>0</code>. You can use
         *        <code>NONE</code> to specify unlimited duration.
         *        </p>
         *        <note>
         *        <p>
         *        An execution start-to-close timeout for this workflow execution must be specified either as a default
         *        for the workflow type or through this field. If neither this field is set nor a default execution
         *        start-to-close timeout was specified at registration time then a fault is returned.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder executionStartToCloseTimeout(String executionStartToCloseTimeout);

        /**
         * <p>
         * The task list to use for the decisions of the new (continued) workflow execution.
         * </p>
         * 
         * @param taskList
         *        The task list to use for the decisions of the new (continued) workflow execution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder taskList(TaskList taskList);

        /**
         * <p>
         * The task list to use for the decisions of the new (continued) workflow execution.
         * </p>
         * This is a convenience that creates an instance of the {@link TaskList.Builder} avoiding the need to create
         * one manually via {@link TaskList#builder()}.
         *
         * When the {@link Consumer} completes, {@link TaskList.Builder#build()} is called immediately and its result is
         * passed to {@link #taskList(TaskList)}.
         * 
         * @param taskList
         *        a consumer that will call methods on {@link TaskList.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #taskList(TaskList)
         */
        default Builder taskList(Consumer<TaskList.Builder> taskList) {
            return taskList(TaskList.builder().applyMutation(taskList).build());
        }

        /**
         * <p>
         * The task priority that, if set, specifies the priority for the decision tasks for this workflow execution.
         * This overrides the defaultTaskPriority specified when registering the workflow type. Valid values are
         * integers that range from Java's <code>Integer.MIN_VALUE</code> (-2147483648) to
         * <code>Integer.MAX_VALUE</code> (2147483647). Higher numbers indicate higher priority.
         * </p>
         * <p>
         * For more information about setting task priority, see <a
         * href="http://docs.aws.amazon.com/amazonswf/latest/developerguide/programming-priority.html">Setting Task
         * Priority</a> in the <i>Amazon SWF Developer Guide</i>.
         * </p>
         * 
         * @param taskPriority
         *        The task priority that, if set, specifies the priority for the decision tasks for this workflow
         *        execution. This overrides the defaultTaskPriority specified when registering the workflow type. Valid
         *        values are integers that range from Java's <code>Integer.MIN_VALUE</code> (-2147483648) to
         *        <code>Integer.MAX_VALUE</code> (2147483647). Higher numbers indicate higher priority.</p>
         *        <p>
         *        For more information about setting task priority, see <a
         *        href="http://docs.aws.amazon.com/amazonswf/latest/developerguide/programming-priority.html">Setting
         *        Task Priority</a> in the <i>Amazon SWF Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder taskPriority(String taskPriority);

        /**
         * <p>
         * Specifies the maximum duration of decision tasks for the new workflow execution. This parameter overrides the
         * <code>defaultTaskStartToCloseTimout</code> specified when registering the workflow type using
         * <a>RegisterWorkflowType</a>.
         * </p>
         * <p>
         * The duration is specified in seconds, an integer greater than or equal to <code>0</code>. You can use
         * <code>NONE</code> to specify unlimited duration.
         * </p>
         * <note>
         * <p>
         * A task start-to-close timeout for the new workflow execution must be specified either as a default for the
         * workflow type or through this parameter. If neither this parameter is set nor a default task start-to-close
         * timeout was specified at registration time then a fault is returned.
         * </p>
         * </note>
         * 
         * @param taskStartToCloseTimeout
         *        Specifies the maximum duration of decision tasks for the new workflow execution. This parameter
         *        overrides the <code>defaultTaskStartToCloseTimout</code> specified when registering the workflow type
         *        using <a>RegisterWorkflowType</a>.</p>
         *        <p>
         *        The duration is specified in seconds, an integer greater than or equal to <code>0</code>. You can use
         *        <code>NONE</code> to specify unlimited duration.
         *        </p>
         *        <note>
         *        <p>
         *        A task start-to-close timeout for the new workflow execution must be specified either as a default for
         *        the workflow type or through this parameter. If neither this parameter is set nor a default task
         *        start-to-close timeout was specified at registration time then a fault is returned.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder taskStartToCloseTimeout(String taskStartToCloseTimeout);

        /**
         * <p>
         * If set, specifies the policy to use for the child workflow executions of the new execution if it is
         * terminated by calling the <a>TerminateWorkflowExecution</a> action explicitly or due to an expired timeout.
         * This policy overrides the default child policy specified when registering the workflow type using
         * <a>RegisterWorkflowType</a>.
         * </p>
         * <p>
         * The supported child policies are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>TERMINATE</code> – The child executions are terminated.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>REQUEST_CANCEL</code> – A request to cancel is attempted for each child execution by recording a
         * <code>WorkflowExecutionCancelRequested</code> event in its history. It is up to the decider to take
         * appropriate actions when it receives an execution history with this event.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ABANDON</code> – No action is taken. The child executions continue to run.
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * A child policy for this workflow execution must be specified either as a default for the workflow type or
         * through this parameter. If neither this parameter is set nor a default child policy was specified at
         * registration time then a fault is returned.
         * </p>
         * </note>
         * 
         * @param childPolicy
         *        If set, specifies the policy to use for the child workflow executions of the new execution if it is
         *        terminated by calling the <a>TerminateWorkflowExecution</a> action explicitly or due to an expired
         *        timeout. This policy overrides the default child policy specified when registering the workflow type
         *        using <a>RegisterWorkflowType</a>.</p>
         *        <p>
         *        The supported child policies are:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>TERMINATE</code> – The child executions are terminated.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>REQUEST_CANCEL</code> – A request to cancel is attempted for each child execution by recording a
         *        <code>WorkflowExecutionCancelRequested</code> event in its history. It is up to the decider to take
         *        appropriate actions when it receives an execution history with this event.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ABANDON</code> – No action is taken. The child executions continue to run.
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        A child policy for this workflow execution must be specified either as a default for the workflow type
         *        or through this parameter. If neither this parameter is set nor a default child policy was specified
         *        at registration time then a fault is returned.
         *        </p>
         * @see ChildPolicy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ChildPolicy
         */
        Builder childPolicy(String childPolicy);

        /**
         * <p>
         * If set, specifies the policy to use for the child workflow executions of the new execution if it is
         * terminated by calling the <a>TerminateWorkflowExecution</a> action explicitly or due to an expired timeout.
         * This policy overrides the default child policy specified when registering the workflow type using
         * <a>RegisterWorkflowType</a>.
         * </p>
         * <p>
         * The supported child policies are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>TERMINATE</code> – The child executions are terminated.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>REQUEST_CANCEL</code> – A request to cancel is attempted for each child execution by recording a
         * <code>WorkflowExecutionCancelRequested</code> event in its history. It is up to the decider to take
         * appropriate actions when it receives an execution history with this event.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ABANDON</code> – No action is taken. The child executions continue to run.
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * A child policy for this workflow execution must be specified either as a default for the workflow type or
         * through this parameter. If neither this parameter is set nor a default child policy was specified at
         * registration time then a fault is returned.
         * </p>
         * </note>
         * 
         * @param childPolicy
         *        If set, specifies the policy to use for the child workflow executions of the new execution if it is
         *        terminated by calling the <a>TerminateWorkflowExecution</a> action explicitly or due to an expired
         *        timeout. This policy overrides the default child policy specified when registering the workflow type
         *        using <a>RegisterWorkflowType</a>.</p>
         *        <p>
         *        The supported child policies are:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>TERMINATE</code> – The child executions are terminated.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>REQUEST_CANCEL</code> – A request to cancel is attempted for each child execution by recording a
         *        <code>WorkflowExecutionCancelRequested</code> event in its history. It is up to the decider to take
         *        appropriate actions when it receives an execution history with this event.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ABANDON</code> – No action is taken. The child executions continue to run.
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        A child policy for this workflow execution must be specified either as a default for the workflow type
         *        or through this parameter. If neither this parameter is set nor a default child policy was specified
         *        at registration time then a fault is returned.
         *        </p>
         * @see ChildPolicy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ChildPolicy
         */
        Builder childPolicy(ChildPolicy childPolicy);

        /**
         * <p>
         * The list of tags to associate with the new workflow execution. A maximum of 5 tags can be specified. You can
         * list workflow executions with a specific tag by calling <a>ListOpenWorkflowExecutions</a> or
         * <a>ListClosedWorkflowExecutions</a> and specifying a <a>TagFilter</a>.
         * </p>
         * 
         * @param tagList
         *        The list of tags to associate with the new workflow execution. A maximum of 5 tags can be specified.
         *        You can list workflow executions with a specific tag by calling <a>ListOpenWorkflowExecutions</a> or
         *        <a>ListClosedWorkflowExecutions</a> and specifying a <a>TagFilter</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tagList(Collection<String> tagList);

        /**
         * <p>
         * The list of tags to associate with the new workflow execution. A maximum of 5 tags can be specified. You can
         * list workflow executions with a specific tag by calling <a>ListOpenWorkflowExecutions</a> or
         * <a>ListClosedWorkflowExecutions</a> and specifying a <a>TagFilter</a>.
         * </p>
         * 
         * @param tagList
         *        The list of tags to associate with the new workflow execution. A maximum of 5 tags can be specified.
         *        You can list workflow executions with a specific tag by calling <a>ListOpenWorkflowExecutions</a> or
         *        <a>ListClosedWorkflowExecutions</a> and specifying a <a>TagFilter</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tagList(String... tagList);

        /**
         * <p>
         * The version of the workflow to start.
         * </p>
         * 
         * @param workflowTypeVersion
         *        The version of the workflow to start.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder workflowTypeVersion(String workflowTypeVersion);

        /**
         * <p>
         * The IAM role to attach to the new (continued) execution.
         * </p>
         * 
         * @param lambdaRole
         *        The IAM role to attach to the new (continued) execution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lambdaRole(String lambdaRole);
    }

    static final class BuilderImpl implements Builder {
        private String input;

        private String executionStartToCloseTimeout;

        private TaskList taskList;

        private String taskPriority;

        private String taskStartToCloseTimeout;

        private String childPolicy;

        private List<String> tagList = DefaultSdkAutoConstructList.getInstance();

        private String workflowTypeVersion;

        private String lambdaRole;

        private BuilderImpl() {
        }

        private BuilderImpl(ContinueAsNewWorkflowExecutionDecisionAttributes model) {
            input(model.input);
            executionStartToCloseTimeout(model.executionStartToCloseTimeout);
            taskList(model.taskList);
            taskPriority(model.taskPriority);
            taskStartToCloseTimeout(model.taskStartToCloseTimeout);
            childPolicy(model.childPolicy);
            tagList(model.tagList);
            workflowTypeVersion(model.workflowTypeVersion);
            lambdaRole(model.lambdaRole);
        }

        public final String getInput() {
            return input;
        }

        @Override
        public final Builder input(String input) {
            this.input = input;
            return this;
        }

        public final void setInput(String input) {
            this.input = input;
        }

        public final String getExecutionStartToCloseTimeout() {
            return executionStartToCloseTimeout;
        }

        @Override
        public final Builder executionStartToCloseTimeout(String executionStartToCloseTimeout) {
            this.executionStartToCloseTimeout = executionStartToCloseTimeout;
            return this;
        }

        public final void setExecutionStartToCloseTimeout(String executionStartToCloseTimeout) {
            this.executionStartToCloseTimeout = executionStartToCloseTimeout;
        }

        public final TaskList.Builder getTaskList() {
            return taskList != null ? taskList.toBuilder() : null;
        }

        @Override
        public final Builder taskList(TaskList taskList) {
            this.taskList = taskList;
            return this;
        }

        public final void setTaskList(TaskList.BuilderImpl taskList) {
            this.taskList = taskList != null ? taskList.build() : null;
        }

        public final String getTaskPriority() {
            return taskPriority;
        }

        @Override
        public final Builder taskPriority(String taskPriority) {
            this.taskPriority = taskPriority;
            return this;
        }

        public final void setTaskPriority(String taskPriority) {
            this.taskPriority = taskPriority;
        }

        public final String getTaskStartToCloseTimeout() {
            return taskStartToCloseTimeout;
        }

        @Override
        public final Builder taskStartToCloseTimeout(String taskStartToCloseTimeout) {
            this.taskStartToCloseTimeout = taskStartToCloseTimeout;
            return this;
        }

        public final void setTaskStartToCloseTimeout(String taskStartToCloseTimeout) {
            this.taskStartToCloseTimeout = taskStartToCloseTimeout;
        }

        public final String getChildPolicyAsString() {
            return childPolicy;
        }

        @Override
        public final Builder childPolicy(String childPolicy) {
            this.childPolicy = childPolicy;
            return this;
        }

        @Override
        public final Builder childPolicy(ChildPolicy childPolicy) {
            this.childPolicy(childPolicy.toString());
            return this;
        }

        public final void setChildPolicy(String childPolicy) {
            this.childPolicy = childPolicy;
        }

        public final Collection<String> getTagList() {
            return tagList;
        }

        @Override
        public final Builder tagList(Collection<String> tagList) {
            this.tagList = TagListCopier.copy(tagList);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tagList(String... tagList) {
            tagList(Arrays.asList(tagList));
            return this;
        }

        public final void setTagList(Collection<String> tagList) {
            this.tagList = TagListCopier.copy(tagList);
        }

        public final String getWorkflowTypeVersion() {
            return workflowTypeVersion;
        }

        @Override
        public final Builder workflowTypeVersion(String workflowTypeVersion) {
            this.workflowTypeVersion = workflowTypeVersion;
            return this;
        }

        public final void setWorkflowTypeVersion(String workflowTypeVersion) {
            this.workflowTypeVersion = workflowTypeVersion;
        }

        public final String getLambdaRole() {
            return lambdaRole;
        }

        @Override
        public final Builder lambdaRole(String lambdaRole) {
            this.lambdaRole = lambdaRole;
            return this;
        }

        public final void setLambdaRole(String lambdaRole) {
            this.lambdaRole = lambdaRole;
        }

        @Override
        public ContinueAsNewWorkflowExecutionDecisionAttributes build() {
            return new ContinueAsNewWorkflowExecutionDecisionAttributes(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
