/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.swf.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Specifies a decision made by the decider. A decision can be one of these types:
 * </p>
 * <ul>
 * <li>
 * <p>
 * <code>CancelTimer</code> – Cancels a previously started timer and records a <code>TimerCanceled</code> event in the
 * history.
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>CancelWorkflowExecution</code> – Closes the workflow execution and records a
 * <code>WorkflowExecutionCanceled</code> event in the history.
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>CompleteWorkflowExecution</code> – Closes the workflow execution and records a
 * <code>WorkflowExecutionCompleted</code> event in the history .
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>ContinueAsNewWorkflowExecution</code> – Closes the workflow execution and starts a new workflow execution of
 * the same type using the same workflow ID and a unique run Id. A <code>WorkflowExecutionContinuedAsNew</code> event is
 * recorded in the history.
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>FailWorkflowExecution</code> – Closes the workflow execution and records a <code>WorkflowExecutionFailed</code>
 * event in the history.
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>RecordMarker</code> – Records a <code>MarkerRecorded</code> event in the history. Markers can be used for
 * adding custom information in the history for instance to let deciders know that they don't need to look at the
 * history beyond the marker event.
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>RequestCancelActivityTask</code> – Attempts to cancel a previously scheduled activity task. If the activity
 * task was scheduled but has not been assigned to a worker, then it is canceled. If the activity task was already
 * assigned to a worker, then the worker is informed that cancellation has been requested in the response to
 * <a>RecordActivityTaskHeartbeat</a>.
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>RequestCancelExternalWorkflowExecution</code> – Requests that a request be made to cancel the specified
 * external workflow execution and records a <code>RequestCancelExternalWorkflowExecutionInitiated</code> event in the
 * history.
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>ScheduleActivityTask</code> – Schedules an activity task.
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>SignalExternalWorkflowExecution</code> – Requests a signal to be delivered to the specified external workflow
 * execution and records a <code>SignalExternalWorkflowExecutionInitiated</code> event in the history.
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>StartChildWorkflowExecution</code> – Requests that a child workflow execution be started and records a
 * <code>StartChildWorkflowExecutionInitiated</code> event in the history. The child workflow execution is a separate
 * workflow execution with its own history.
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>StartTimer</code> – Starts a timer for this workflow execution and records a <code>TimerStarted</code> event in
 * the history. This timer fires after the specified delay and record a <code>TimerFired</code> event.
 * </p>
 * </li>
 * </ul>
 * <p>
 * <b>Access Control</b>
 * </p>
 * <p>
 * If you grant permission to use <code>RespondDecisionTaskCompleted</code>, you can use IAM policies to express
 * permissions for the list of decisions returned by this action as if they were members of the API. Treating decisions
 * as a pseudo API maintains a uniform conceptual model and helps keep policies readable. For details and example IAM
 * policies, see <a href="http://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html">Using IAM to
 * Manage Access to Amazon SWF Workflows</a> in the <i>Amazon SWF Developer Guide</i>.
 * </p>
 * <p>
 * <b>Decision Failure</b>
 * </p>
 * <p>
 * Decisions can fail for several reasons
 * </p>
 * <ul>
 * <li>
 * <p>
 * The ordering of decisions should follow a logical flow. Some decisions might not make sense in the current context of
 * the workflow execution and therefore fails.
 * </p>
 * </li>
 * <li>
 * <p>
 * A limit on your account was reached.
 * </p>
 * </li>
 * <li>
 * <p>
 * The decision lacks sufficient permissions.
 * </p>
 * </li>
 * </ul>
 * <p>
 * One of the following events might be added to the history to indicate an error. The event attribute's
 * <code>cause</code> parameter indicates the cause. If <code>cause</code> is set to
 * <code>OPERATION_NOT_PERMITTED</code>, the decision failed because it lacked sufficient permissions. For details and
 * example IAM policies, see <a href="http://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html">Using
 * IAM to Manage Access to Amazon SWF Workflows</a> in the <i>Amazon SWF Developer Guide</i>.
 * </p>
 * <ul>
 * <li>
 * <p>
 * <code>ScheduleActivityTaskFailed</code> – A <code>ScheduleActivityTask</code> decision failed. This could happen if
 * the activity type specified in the decision isn't registered, is in a deprecated state, or the decision isn't
 * properly configured.
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>RequestCancelActivityTaskFailed</code> – A <code>RequestCancelActivityTask</code> decision failed. This could
 * happen if there is no open activity task with the specified activityId.
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>StartTimerFailed</code> – A <code>StartTimer</code> decision failed. This could happen if there is another open
 * timer with the same timerId.
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>CancelTimerFailed</code> – A <code>CancelTimer</code> decision failed. This could happen if there is no open
 * timer with the specified timerId.
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>StartChildWorkflowExecutionFailed</code> – A <code>StartChildWorkflowExecution</code> decision failed. This
 * could happen if the workflow type specified isn't registered, is deprecated, or the decision isn't properly
 * configured.
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>SignalExternalWorkflowExecutionFailed</code> – A <code>SignalExternalWorkflowExecution</code> decision failed.
 * This could happen if the <code>workflowID</code> specified in the decision was incorrect.
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>RequestCancelExternalWorkflowExecutionFailed</code> – A <code>RequestCancelExternalWorkflowExecution</code>
 * decision failed. This could happen if the <code>workflowID</code> specified in the decision was incorrect.
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>CancelWorkflowExecutionFailed</code> – A <code>CancelWorkflowExecution</code> decision failed. This could
 * happen if there is an unhandled decision task pending in the workflow execution.
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>CompleteWorkflowExecutionFailed</code> – A <code>CompleteWorkflowExecution</code> decision failed. This could
 * happen if there is an unhandled decision task pending in the workflow execution.
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>ContinueAsNewWorkflowExecutionFailed</code> – A <code>ContinueAsNewWorkflowExecution</code> decision failed.
 * This could happen if there is an unhandled decision task pending in the workflow execution or the
 * ContinueAsNewWorkflowExecution decision was not configured correctly.
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>FailWorkflowExecutionFailed</code> – A <code>FailWorkflowExecution</code> decision failed. This could happen if
 * there is an unhandled decision task pending in the workflow execution.
 * </p>
 * </li>
 * </ul>
 * <p>
 * The preceding error events might occur due to an error in the decider logic, which might put the workflow execution
 * in an unstable state The cause field in the event structure for the error event indicates the cause of the error.
 * </p>
 * <note>
 * <p>
 * A workflow execution may be closed by the decider by returning one of the following decisions when completing a
 * decision task: <code>CompleteWorkflowExecution</code>, <code>FailWorkflowExecution</code>,
 * <code>CancelWorkflowExecution</code> and <code>ContinueAsNewWorkflowExecution</code>. An
 * <code>UnhandledDecision</code> fault is returned if a workflow closing decision is specified and a signal or activity
 * event had been added to the history while the decision task was being performed by the decider. Unlike the above
 * situations which are logic issues, this fault is always possible because of race conditions in a distributed system.
 * The right action here is to call <a>RespondDecisionTaskCompleted</a> without any decisions. This would result in
 * another decision task with these new events included in the history. The decider should handle the new events and may
 * decide to close the workflow execution.
 * </p>
 * </note>
 * <p>
 * <b>How to Code a Decision</b>
 * </p>
 * <p>
 * You code a decision by first setting the decision type field to one of the above decision values, and then set the
 * corresponding attributes field shown below:
 * </p>
 * <ul>
 * <li>
 * <p>
 * <code> <a>ScheduleActivityTaskDecisionAttributes</a> </code>
 * </p>
 * </li>
 * <li>
 * <p>
 * <code> <a>RequestCancelActivityTaskDecisionAttributes</a> </code>
 * </p>
 * </li>
 * <li>
 * <p>
 * <code> <a>CompleteWorkflowExecutionDecisionAttributes</a> </code>
 * </p>
 * </li>
 * <li>
 * <p>
 * <code> <a>FailWorkflowExecutionDecisionAttributes</a> </code>
 * </p>
 * </li>
 * <li>
 * <p>
 * <code> <a>CancelWorkflowExecutionDecisionAttributes</a> </code>
 * </p>
 * </li>
 * <li>
 * <p>
 * <code> <a>ContinueAsNewWorkflowExecutionDecisionAttributes</a> </code>
 * </p>
 * </li>
 * <li>
 * <p>
 * <code> <a>RecordMarkerDecisionAttributes</a> </code>
 * </p>
 * </li>
 * <li>
 * <p>
 * <code> <a>StartTimerDecisionAttributes</a> </code>
 * </p>
 * </li>
 * <li>
 * <p>
 * <code> <a>CancelTimerDecisionAttributes</a> </code>
 * </p>
 * </li>
 * <li>
 * <p>
 * <code> <a>SignalExternalWorkflowExecutionDecisionAttributes</a> </code>
 * </p>
 * </li>
 * <li>
 * <p>
 * <code> <a>RequestCancelExternalWorkflowExecutionDecisionAttributes</a> </code>
 * </p>
 * </li>
 * <li>
 * <p>
 * <code> <a>StartChildWorkflowExecutionDecisionAttributes</a> </code>
 * </p>
 * </li>
 * </ul>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Decision implements SdkPojo, Serializable, ToCopyableBuilder<Decision.Builder, Decision> {
    private static final SdkField<String> DECISION_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Decision::decisionTypeAsString)).setter(setter(Builder::decisionType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("decisionType").build()).build();

    private static final SdkField<ScheduleActivityTaskDecisionAttributes> SCHEDULE_ACTIVITY_TASK_DECISION_ATTRIBUTES_FIELD = SdkField
            .<ScheduleActivityTaskDecisionAttributes> builder(MarshallingType.SDK_POJO)
            .getter(getter(Decision::scheduleActivityTaskDecisionAttributes))
            .setter(setter(Builder::scheduleActivityTaskDecisionAttributes))
            .constructor(ScheduleActivityTaskDecisionAttributes::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("scheduleActivityTaskDecisionAttributes").build()).build();

    private static final SdkField<RequestCancelActivityTaskDecisionAttributes> REQUEST_CANCEL_ACTIVITY_TASK_DECISION_ATTRIBUTES_FIELD = SdkField
            .<RequestCancelActivityTaskDecisionAttributes> builder(MarshallingType.SDK_POJO)
            .getter(getter(Decision::requestCancelActivityTaskDecisionAttributes))
            .setter(setter(Builder::requestCancelActivityTaskDecisionAttributes))
            .constructor(RequestCancelActivityTaskDecisionAttributes::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("requestCancelActivityTaskDecisionAttributes").build()).build();

    private static final SdkField<CompleteWorkflowExecutionDecisionAttributes> COMPLETE_WORKFLOW_EXECUTION_DECISION_ATTRIBUTES_FIELD = SdkField
            .<CompleteWorkflowExecutionDecisionAttributes> builder(MarshallingType.SDK_POJO)
            .getter(getter(Decision::completeWorkflowExecutionDecisionAttributes))
            .setter(setter(Builder::completeWorkflowExecutionDecisionAttributes))
            .constructor(CompleteWorkflowExecutionDecisionAttributes::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("completeWorkflowExecutionDecisionAttributes").build()).build();

    private static final SdkField<FailWorkflowExecutionDecisionAttributes> FAIL_WORKFLOW_EXECUTION_DECISION_ATTRIBUTES_FIELD = SdkField
            .<FailWorkflowExecutionDecisionAttributes> builder(MarshallingType.SDK_POJO)
            .getter(getter(Decision::failWorkflowExecutionDecisionAttributes))
            .setter(setter(Builder::failWorkflowExecutionDecisionAttributes))
            .constructor(FailWorkflowExecutionDecisionAttributes::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("failWorkflowExecutionDecisionAttributes").build()).build();

    private static final SdkField<CancelWorkflowExecutionDecisionAttributes> CANCEL_WORKFLOW_EXECUTION_DECISION_ATTRIBUTES_FIELD = SdkField
            .<CancelWorkflowExecutionDecisionAttributes> builder(MarshallingType.SDK_POJO)
            .getter(getter(Decision::cancelWorkflowExecutionDecisionAttributes))
            .setter(setter(Builder::cancelWorkflowExecutionDecisionAttributes))
            .constructor(CancelWorkflowExecutionDecisionAttributes::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("cancelWorkflowExecutionDecisionAttributes").build()).build();

    private static final SdkField<ContinueAsNewWorkflowExecutionDecisionAttributes> CONTINUE_AS_NEW_WORKFLOW_EXECUTION_DECISION_ATTRIBUTES_FIELD = SdkField
            .<ContinueAsNewWorkflowExecutionDecisionAttributes> builder(MarshallingType.SDK_POJO)
            .getter(getter(Decision::continueAsNewWorkflowExecutionDecisionAttributes))
            .setter(setter(Builder::continueAsNewWorkflowExecutionDecisionAttributes))
            .constructor(ContinueAsNewWorkflowExecutionDecisionAttributes::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("continueAsNewWorkflowExecutionDecisionAttributes").build()).build();

    private static final SdkField<RecordMarkerDecisionAttributes> RECORD_MARKER_DECISION_ATTRIBUTES_FIELD = SdkField
            .<RecordMarkerDecisionAttributes> builder(MarshallingType.SDK_POJO)
            .getter(getter(Decision::recordMarkerDecisionAttributes))
            .setter(setter(Builder::recordMarkerDecisionAttributes))
            .constructor(RecordMarkerDecisionAttributes::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("recordMarkerDecisionAttributes")
                    .build()).build();

    private static final SdkField<StartTimerDecisionAttributes> START_TIMER_DECISION_ATTRIBUTES_FIELD = SdkField
            .<StartTimerDecisionAttributes> builder(MarshallingType.SDK_POJO)
            .getter(getter(Decision::startTimerDecisionAttributes))
            .setter(setter(Builder::startTimerDecisionAttributes))
            .constructor(StartTimerDecisionAttributes::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("startTimerDecisionAttributes")
                    .build()).build();

    private static final SdkField<CancelTimerDecisionAttributes> CANCEL_TIMER_DECISION_ATTRIBUTES_FIELD = SdkField
            .<CancelTimerDecisionAttributes> builder(MarshallingType.SDK_POJO)
            .getter(getter(Decision::cancelTimerDecisionAttributes))
            .setter(setter(Builder::cancelTimerDecisionAttributes))
            .constructor(CancelTimerDecisionAttributes::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("cancelTimerDecisionAttributes")
                    .build()).build();

    private static final SdkField<SignalExternalWorkflowExecutionDecisionAttributes> SIGNAL_EXTERNAL_WORKFLOW_EXECUTION_DECISION_ATTRIBUTES_FIELD = SdkField
            .<SignalExternalWorkflowExecutionDecisionAttributes> builder(MarshallingType.SDK_POJO)
            .getter(getter(Decision::signalExternalWorkflowExecutionDecisionAttributes))
            .setter(setter(Builder::signalExternalWorkflowExecutionDecisionAttributes))
            .constructor(SignalExternalWorkflowExecutionDecisionAttributes::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("signalExternalWorkflowExecutionDecisionAttributes").build()).build();

    private static final SdkField<RequestCancelExternalWorkflowExecutionDecisionAttributes> REQUEST_CANCEL_EXTERNAL_WORKFLOW_EXECUTION_DECISION_ATTRIBUTES_FIELD = SdkField
            .<RequestCancelExternalWorkflowExecutionDecisionAttributes> builder(MarshallingType.SDK_POJO)
            .getter(getter(Decision::requestCancelExternalWorkflowExecutionDecisionAttributes))
            .setter(setter(Builder::requestCancelExternalWorkflowExecutionDecisionAttributes))
            .constructor(RequestCancelExternalWorkflowExecutionDecisionAttributes::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("requestCancelExternalWorkflowExecutionDecisionAttributes").build()).build();

    private static final SdkField<StartChildWorkflowExecutionDecisionAttributes> START_CHILD_WORKFLOW_EXECUTION_DECISION_ATTRIBUTES_FIELD = SdkField
            .<StartChildWorkflowExecutionDecisionAttributes> builder(MarshallingType.SDK_POJO)
            .getter(getter(Decision::startChildWorkflowExecutionDecisionAttributes))
            .setter(setter(Builder::startChildWorkflowExecutionDecisionAttributes))
            .constructor(StartChildWorkflowExecutionDecisionAttributes::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("startChildWorkflowExecutionDecisionAttributes").build()).build();

    private static final SdkField<ScheduleLambdaFunctionDecisionAttributes> SCHEDULE_LAMBDA_FUNCTION_DECISION_ATTRIBUTES_FIELD = SdkField
            .<ScheduleLambdaFunctionDecisionAttributes> builder(MarshallingType.SDK_POJO)
            .getter(getter(Decision::scheduleLambdaFunctionDecisionAttributes))
            .setter(setter(Builder::scheduleLambdaFunctionDecisionAttributes))
            .constructor(ScheduleLambdaFunctionDecisionAttributes::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("scheduleLambdaFunctionDecisionAttributes").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DECISION_TYPE_FIELD,
            SCHEDULE_ACTIVITY_TASK_DECISION_ATTRIBUTES_FIELD, REQUEST_CANCEL_ACTIVITY_TASK_DECISION_ATTRIBUTES_FIELD,
            COMPLETE_WORKFLOW_EXECUTION_DECISION_ATTRIBUTES_FIELD, FAIL_WORKFLOW_EXECUTION_DECISION_ATTRIBUTES_FIELD,
            CANCEL_WORKFLOW_EXECUTION_DECISION_ATTRIBUTES_FIELD, CONTINUE_AS_NEW_WORKFLOW_EXECUTION_DECISION_ATTRIBUTES_FIELD,
            RECORD_MARKER_DECISION_ATTRIBUTES_FIELD, START_TIMER_DECISION_ATTRIBUTES_FIELD,
            CANCEL_TIMER_DECISION_ATTRIBUTES_FIELD, SIGNAL_EXTERNAL_WORKFLOW_EXECUTION_DECISION_ATTRIBUTES_FIELD,
            REQUEST_CANCEL_EXTERNAL_WORKFLOW_EXECUTION_DECISION_ATTRIBUTES_FIELD,
            START_CHILD_WORKFLOW_EXECUTION_DECISION_ATTRIBUTES_FIELD, SCHEDULE_LAMBDA_FUNCTION_DECISION_ATTRIBUTES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String decisionType;

    private final ScheduleActivityTaskDecisionAttributes scheduleActivityTaskDecisionAttributes;

    private final RequestCancelActivityTaskDecisionAttributes requestCancelActivityTaskDecisionAttributes;

    private final CompleteWorkflowExecutionDecisionAttributes completeWorkflowExecutionDecisionAttributes;

    private final FailWorkflowExecutionDecisionAttributes failWorkflowExecutionDecisionAttributes;

    private final CancelWorkflowExecutionDecisionAttributes cancelWorkflowExecutionDecisionAttributes;

    private final ContinueAsNewWorkflowExecutionDecisionAttributes continueAsNewWorkflowExecutionDecisionAttributes;

    private final RecordMarkerDecisionAttributes recordMarkerDecisionAttributes;

    private final StartTimerDecisionAttributes startTimerDecisionAttributes;

    private final CancelTimerDecisionAttributes cancelTimerDecisionAttributes;

    private final SignalExternalWorkflowExecutionDecisionAttributes signalExternalWorkflowExecutionDecisionAttributes;

    private final RequestCancelExternalWorkflowExecutionDecisionAttributes requestCancelExternalWorkflowExecutionDecisionAttributes;

    private final StartChildWorkflowExecutionDecisionAttributes startChildWorkflowExecutionDecisionAttributes;

    private final ScheduleLambdaFunctionDecisionAttributes scheduleLambdaFunctionDecisionAttributes;

    private Decision(BuilderImpl builder) {
        this.decisionType = builder.decisionType;
        this.scheduleActivityTaskDecisionAttributes = builder.scheduleActivityTaskDecisionAttributes;
        this.requestCancelActivityTaskDecisionAttributes = builder.requestCancelActivityTaskDecisionAttributes;
        this.completeWorkflowExecutionDecisionAttributes = builder.completeWorkflowExecutionDecisionAttributes;
        this.failWorkflowExecutionDecisionAttributes = builder.failWorkflowExecutionDecisionAttributes;
        this.cancelWorkflowExecutionDecisionAttributes = builder.cancelWorkflowExecutionDecisionAttributes;
        this.continueAsNewWorkflowExecutionDecisionAttributes = builder.continueAsNewWorkflowExecutionDecisionAttributes;
        this.recordMarkerDecisionAttributes = builder.recordMarkerDecisionAttributes;
        this.startTimerDecisionAttributes = builder.startTimerDecisionAttributes;
        this.cancelTimerDecisionAttributes = builder.cancelTimerDecisionAttributes;
        this.signalExternalWorkflowExecutionDecisionAttributes = builder.signalExternalWorkflowExecutionDecisionAttributes;
        this.requestCancelExternalWorkflowExecutionDecisionAttributes = builder.requestCancelExternalWorkflowExecutionDecisionAttributes;
        this.startChildWorkflowExecutionDecisionAttributes = builder.startChildWorkflowExecutionDecisionAttributes;
        this.scheduleLambdaFunctionDecisionAttributes = builder.scheduleLambdaFunctionDecisionAttributes;
    }

    /**
     * <p>
     * Specifies the type of the decision.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #decisionType} will
     * return {@link DecisionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #decisionTypeAsString}.
     * </p>
     * 
     * @return Specifies the type of the decision.
     * @see DecisionType
     */
    public DecisionType decisionType() {
        return DecisionType.fromValue(decisionType);
    }

    /**
     * <p>
     * Specifies the type of the decision.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #decisionType} will
     * return {@link DecisionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #decisionTypeAsString}.
     * </p>
     * 
     * @return Specifies the type of the decision.
     * @see DecisionType
     */
    public String decisionTypeAsString() {
        return decisionType;
    }

    /**
     * <p>
     * Provides the details of the <code>ScheduleActivityTask</code> decision. It isn't set for other decision types.
     * </p>
     * 
     * @return Provides the details of the <code>ScheduleActivityTask</code> decision. It isn't set for other decision
     *         types.
     */
    public ScheduleActivityTaskDecisionAttributes scheduleActivityTaskDecisionAttributes() {
        return scheduleActivityTaskDecisionAttributes;
    }

    /**
     * <p>
     * Provides the details of the <code>RequestCancelActivityTask</code> decision. It isn't set for other decision
     * types.
     * </p>
     * 
     * @return Provides the details of the <code>RequestCancelActivityTask</code> decision. It isn't set for other
     *         decision types.
     */
    public RequestCancelActivityTaskDecisionAttributes requestCancelActivityTaskDecisionAttributes() {
        return requestCancelActivityTaskDecisionAttributes;
    }

    /**
     * <p>
     * Provides the details of the <code>CompleteWorkflowExecution</code> decision. It isn't set for other decision
     * types.
     * </p>
     * 
     * @return Provides the details of the <code>CompleteWorkflowExecution</code> decision. It isn't set for other
     *         decision types.
     */
    public CompleteWorkflowExecutionDecisionAttributes completeWorkflowExecutionDecisionAttributes() {
        return completeWorkflowExecutionDecisionAttributes;
    }

    /**
     * <p>
     * Provides the details of the <code>FailWorkflowExecution</code> decision. It isn't set for other decision types.
     * </p>
     * 
     * @return Provides the details of the <code>FailWorkflowExecution</code> decision. It isn't set for other decision
     *         types.
     */
    public FailWorkflowExecutionDecisionAttributes failWorkflowExecutionDecisionAttributes() {
        return failWorkflowExecutionDecisionAttributes;
    }

    /**
     * <p>
     * Provides the details of the <code>CancelWorkflowExecution</code> decision. It isn't set for other decision types.
     * </p>
     * 
     * @return Provides the details of the <code>CancelWorkflowExecution</code> decision. It isn't set for other
     *         decision types.
     */
    public CancelWorkflowExecutionDecisionAttributes cancelWorkflowExecutionDecisionAttributes() {
        return cancelWorkflowExecutionDecisionAttributes;
    }

    /**
     * <p>
     * Provides the details of the <code>ContinueAsNewWorkflowExecution</code> decision. It isn't set for other decision
     * types.
     * </p>
     * 
     * @return Provides the details of the <code>ContinueAsNewWorkflowExecution</code> decision. It isn't set for other
     *         decision types.
     */
    public ContinueAsNewWorkflowExecutionDecisionAttributes continueAsNewWorkflowExecutionDecisionAttributes() {
        return continueAsNewWorkflowExecutionDecisionAttributes;
    }

    /**
     * <p>
     * Provides the details of the <code>RecordMarker</code> decision. It isn't set for other decision types.
     * </p>
     * 
     * @return Provides the details of the <code>RecordMarker</code> decision. It isn't set for other decision types.
     */
    public RecordMarkerDecisionAttributes recordMarkerDecisionAttributes() {
        return recordMarkerDecisionAttributes;
    }

    /**
     * <p>
     * Provides the details of the <code>StartTimer</code> decision. It isn't set for other decision types.
     * </p>
     * 
     * @return Provides the details of the <code>StartTimer</code> decision. It isn't set for other decision types.
     */
    public StartTimerDecisionAttributes startTimerDecisionAttributes() {
        return startTimerDecisionAttributes;
    }

    /**
     * <p>
     * Provides the details of the <code>CancelTimer</code> decision. It isn't set for other decision types.
     * </p>
     * 
     * @return Provides the details of the <code>CancelTimer</code> decision. It isn't set for other decision types.
     */
    public CancelTimerDecisionAttributes cancelTimerDecisionAttributes() {
        return cancelTimerDecisionAttributes;
    }

    /**
     * <p>
     * Provides the details of the <code>SignalExternalWorkflowExecution</code> decision. It isn't set for other
     * decision types.
     * </p>
     * 
     * @return Provides the details of the <code>SignalExternalWorkflowExecution</code> decision. It isn't set for other
     *         decision types.
     */
    public SignalExternalWorkflowExecutionDecisionAttributes signalExternalWorkflowExecutionDecisionAttributes() {
        return signalExternalWorkflowExecutionDecisionAttributes;
    }

    /**
     * <p>
     * Provides the details of the <code>RequestCancelExternalWorkflowExecution</code> decision. It isn't set for other
     * decision types.
     * </p>
     * 
     * @return Provides the details of the <code>RequestCancelExternalWorkflowExecution</code> decision. It isn't set
     *         for other decision types.
     */
    public RequestCancelExternalWorkflowExecutionDecisionAttributes requestCancelExternalWorkflowExecutionDecisionAttributes() {
        return requestCancelExternalWorkflowExecutionDecisionAttributes;
    }

    /**
     * <p>
     * Provides the details of the <code>StartChildWorkflowExecution</code> decision. It isn't set for other decision
     * types.
     * </p>
     * 
     * @return Provides the details of the <code>StartChildWorkflowExecution</code> decision. It isn't set for other
     *         decision types.
     */
    public StartChildWorkflowExecutionDecisionAttributes startChildWorkflowExecutionDecisionAttributes() {
        return startChildWorkflowExecutionDecisionAttributes;
    }

    /**
     * <p>
     * Provides the details of the <code>ScheduleLambdaFunction</code> decision. It isn't set for other decision types.
     * </p>
     * 
     * @return Provides the details of the <code>ScheduleLambdaFunction</code> decision. It isn't set for other decision
     *         types.
     */
    public ScheduleLambdaFunctionDecisionAttributes scheduleLambdaFunctionDecisionAttributes() {
        return scheduleLambdaFunctionDecisionAttributes;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(decisionTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(scheduleActivityTaskDecisionAttributes());
        hashCode = 31 * hashCode + Objects.hashCode(requestCancelActivityTaskDecisionAttributes());
        hashCode = 31 * hashCode + Objects.hashCode(completeWorkflowExecutionDecisionAttributes());
        hashCode = 31 * hashCode + Objects.hashCode(failWorkflowExecutionDecisionAttributes());
        hashCode = 31 * hashCode + Objects.hashCode(cancelWorkflowExecutionDecisionAttributes());
        hashCode = 31 * hashCode + Objects.hashCode(continueAsNewWorkflowExecutionDecisionAttributes());
        hashCode = 31 * hashCode + Objects.hashCode(recordMarkerDecisionAttributes());
        hashCode = 31 * hashCode + Objects.hashCode(startTimerDecisionAttributes());
        hashCode = 31 * hashCode + Objects.hashCode(cancelTimerDecisionAttributes());
        hashCode = 31 * hashCode + Objects.hashCode(signalExternalWorkflowExecutionDecisionAttributes());
        hashCode = 31 * hashCode + Objects.hashCode(requestCancelExternalWorkflowExecutionDecisionAttributes());
        hashCode = 31 * hashCode + Objects.hashCode(startChildWorkflowExecutionDecisionAttributes());
        hashCode = 31 * hashCode + Objects.hashCode(scheduleLambdaFunctionDecisionAttributes());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Decision)) {
            return false;
        }
        Decision other = (Decision) obj;
        return Objects.equals(decisionTypeAsString(), other.decisionTypeAsString())
                && Objects.equals(scheduleActivityTaskDecisionAttributes(), other.scheduleActivityTaskDecisionAttributes())
                && Objects.equals(requestCancelActivityTaskDecisionAttributes(),
                        other.requestCancelActivityTaskDecisionAttributes())
                && Objects.equals(completeWorkflowExecutionDecisionAttributes(),
                        other.completeWorkflowExecutionDecisionAttributes())
                && Objects.equals(failWorkflowExecutionDecisionAttributes(), other.failWorkflowExecutionDecisionAttributes())
                && Objects.equals(cancelWorkflowExecutionDecisionAttributes(), other.cancelWorkflowExecutionDecisionAttributes())
                && Objects.equals(continueAsNewWorkflowExecutionDecisionAttributes(),
                        other.continueAsNewWorkflowExecutionDecisionAttributes())
                && Objects.equals(recordMarkerDecisionAttributes(), other.recordMarkerDecisionAttributes())
                && Objects.equals(startTimerDecisionAttributes(), other.startTimerDecisionAttributes())
                && Objects.equals(cancelTimerDecisionAttributes(), other.cancelTimerDecisionAttributes())
                && Objects.equals(signalExternalWorkflowExecutionDecisionAttributes(),
                        other.signalExternalWorkflowExecutionDecisionAttributes())
                && Objects.equals(requestCancelExternalWorkflowExecutionDecisionAttributes(),
                        other.requestCancelExternalWorkflowExecutionDecisionAttributes())
                && Objects.equals(startChildWorkflowExecutionDecisionAttributes(),
                        other.startChildWorkflowExecutionDecisionAttributes())
                && Objects.equals(scheduleLambdaFunctionDecisionAttributes(), other.scheduleLambdaFunctionDecisionAttributes());
    }

    @Override
    public String toString() {
        return ToString
                .builder("Decision")
                .add("DecisionType", decisionTypeAsString())
                .add("ScheduleActivityTaskDecisionAttributes", scheduleActivityTaskDecisionAttributes())
                .add("RequestCancelActivityTaskDecisionAttributes", requestCancelActivityTaskDecisionAttributes())
                .add("CompleteWorkflowExecutionDecisionAttributes", completeWorkflowExecutionDecisionAttributes())
                .add("FailWorkflowExecutionDecisionAttributes", failWorkflowExecutionDecisionAttributes())
                .add("CancelWorkflowExecutionDecisionAttributes", cancelWorkflowExecutionDecisionAttributes())
                .add("ContinueAsNewWorkflowExecutionDecisionAttributes", continueAsNewWorkflowExecutionDecisionAttributes())
                .add("RecordMarkerDecisionAttributes", recordMarkerDecisionAttributes())
                .add("StartTimerDecisionAttributes", startTimerDecisionAttributes())
                .add("CancelTimerDecisionAttributes", cancelTimerDecisionAttributes())
                .add("SignalExternalWorkflowExecutionDecisionAttributes", signalExternalWorkflowExecutionDecisionAttributes())
                .add("RequestCancelExternalWorkflowExecutionDecisionAttributes",
                        requestCancelExternalWorkflowExecutionDecisionAttributes())
                .add("StartChildWorkflowExecutionDecisionAttributes", startChildWorkflowExecutionDecisionAttributes())
                .add("ScheduleLambdaFunctionDecisionAttributes", scheduleLambdaFunctionDecisionAttributes()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "decisionType":
            return Optional.ofNullable(clazz.cast(decisionTypeAsString()));
        case "scheduleActivityTaskDecisionAttributes":
            return Optional.ofNullable(clazz.cast(scheduleActivityTaskDecisionAttributes()));
        case "requestCancelActivityTaskDecisionAttributes":
            return Optional.ofNullable(clazz.cast(requestCancelActivityTaskDecisionAttributes()));
        case "completeWorkflowExecutionDecisionAttributes":
            return Optional.ofNullable(clazz.cast(completeWorkflowExecutionDecisionAttributes()));
        case "failWorkflowExecutionDecisionAttributes":
            return Optional.ofNullable(clazz.cast(failWorkflowExecutionDecisionAttributes()));
        case "cancelWorkflowExecutionDecisionAttributes":
            return Optional.ofNullable(clazz.cast(cancelWorkflowExecutionDecisionAttributes()));
        case "continueAsNewWorkflowExecutionDecisionAttributes":
            return Optional.ofNullable(clazz.cast(continueAsNewWorkflowExecutionDecisionAttributes()));
        case "recordMarkerDecisionAttributes":
            return Optional.ofNullable(clazz.cast(recordMarkerDecisionAttributes()));
        case "startTimerDecisionAttributes":
            return Optional.ofNullable(clazz.cast(startTimerDecisionAttributes()));
        case "cancelTimerDecisionAttributes":
            return Optional.ofNullable(clazz.cast(cancelTimerDecisionAttributes()));
        case "signalExternalWorkflowExecutionDecisionAttributes":
            return Optional.ofNullable(clazz.cast(signalExternalWorkflowExecutionDecisionAttributes()));
        case "requestCancelExternalWorkflowExecutionDecisionAttributes":
            return Optional.ofNullable(clazz.cast(requestCancelExternalWorkflowExecutionDecisionAttributes()));
        case "startChildWorkflowExecutionDecisionAttributes":
            return Optional.ofNullable(clazz.cast(startChildWorkflowExecutionDecisionAttributes()));
        case "scheduleLambdaFunctionDecisionAttributes":
            return Optional.ofNullable(clazz.cast(scheduleLambdaFunctionDecisionAttributes()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Decision, T> g) {
        return obj -> g.apply((Decision) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Decision> {
        /**
         * <p>
         * Specifies the type of the decision.
         * </p>
         * 
         * @param decisionType
         *        Specifies the type of the decision.
         * @see DecisionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DecisionType
         */
        Builder decisionType(String decisionType);

        /**
         * <p>
         * Specifies the type of the decision.
         * </p>
         * 
         * @param decisionType
         *        Specifies the type of the decision.
         * @see DecisionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DecisionType
         */
        Builder decisionType(DecisionType decisionType);

        /**
         * <p>
         * Provides the details of the <code>ScheduleActivityTask</code> decision. It isn't set for other decision
         * types.
         * </p>
         * 
         * @param scheduleActivityTaskDecisionAttributes
         *        Provides the details of the <code>ScheduleActivityTask</code> decision. It isn't set for other
         *        decision types.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scheduleActivityTaskDecisionAttributes(
                ScheduleActivityTaskDecisionAttributes scheduleActivityTaskDecisionAttributes);

        /**
         * <p>
         * Provides the details of the <code>ScheduleActivityTask</code> decision. It isn't set for other decision
         * types.
         * </p>
         * This is a convenience that creates an instance of the {@link ScheduleActivityTaskDecisionAttributes.Builder}
         * avoiding the need to create one manually via {@link ScheduleActivityTaskDecisionAttributes#builder()}.
         *
         * When the {@link Consumer} completes, {@link ScheduleActivityTaskDecisionAttributes.Builder#build()} is called
         * immediately and its result is passed to
         * {@link #scheduleActivityTaskDecisionAttributes(ScheduleActivityTaskDecisionAttributes)}.
         * 
         * @param scheduleActivityTaskDecisionAttributes
         *        a consumer that will call methods on {@link ScheduleActivityTaskDecisionAttributes.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #scheduleActivityTaskDecisionAttributes(ScheduleActivityTaskDecisionAttributes)
         */
        default Builder scheduleActivityTaskDecisionAttributes(
                Consumer<ScheduleActivityTaskDecisionAttributes.Builder> scheduleActivityTaskDecisionAttributes) {
            return scheduleActivityTaskDecisionAttributes(ScheduleActivityTaskDecisionAttributes.builder()
                    .applyMutation(scheduleActivityTaskDecisionAttributes).build());
        }

        /**
         * <p>
         * Provides the details of the <code>RequestCancelActivityTask</code> decision. It isn't set for other decision
         * types.
         * </p>
         * 
         * @param requestCancelActivityTaskDecisionAttributes
         *        Provides the details of the <code>RequestCancelActivityTask</code> decision. It isn't set for other
         *        decision types.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requestCancelActivityTaskDecisionAttributes(
                RequestCancelActivityTaskDecisionAttributes requestCancelActivityTaskDecisionAttributes);

        /**
         * <p>
         * Provides the details of the <code>RequestCancelActivityTask</code> decision. It isn't set for other decision
         * types.
         * </p>
         * This is a convenience that creates an instance of the
         * {@link RequestCancelActivityTaskDecisionAttributes.Builder} avoiding the need to create one manually via
         * {@link RequestCancelActivityTaskDecisionAttributes#builder()}.
         *
         * When the {@link Consumer} completes, {@link RequestCancelActivityTaskDecisionAttributes.Builder#build()} is
         * called immediately and its result is passed to
         * {@link #requestCancelActivityTaskDecisionAttributes(RequestCancelActivityTaskDecisionAttributes)}.
         * 
         * @param requestCancelActivityTaskDecisionAttributes
         *        a consumer that will call methods on {@link RequestCancelActivityTaskDecisionAttributes.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #requestCancelActivityTaskDecisionAttributes(RequestCancelActivityTaskDecisionAttributes)
         */
        default Builder requestCancelActivityTaskDecisionAttributes(
                Consumer<RequestCancelActivityTaskDecisionAttributes.Builder> requestCancelActivityTaskDecisionAttributes) {
            return requestCancelActivityTaskDecisionAttributes(RequestCancelActivityTaskDecisionAttributes.builder()
                    .applyMutation(requestCancelActivityTaskDecisionAttributes).build());
        }

        /**
         * <p>
         * Provides the details of the <code>CompleteWorkflowExecution</code> decision. It isn't set for other decision
         * types.
         * </p>
         * 
         * @param completeWorkflowExecutionDecisionAttributes
         *        Provides the details of the <code>CompleteWorkflowExecution</code> decision. It isn't set for other
         *        decision types.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder completeWorkflowExecutionDecisionAttributes(
                CompleteWorkflowExecutionDecisionAttributes completeWorkflowExecutionDecisionAttributes);

        /**
         * <p>
         * Provides the details of the <code>CompleteWorkflowExecution</code> decision. It isn't set for other decision
         * types.
         * </p>
         * This is a convenience that creates an instance of the
         * {@link CompleteWorkflowExecutionDecisionAttributes.Builder} avoiding the need to create one manually via
         * {@link CompleteWorkflowExecutionDecisionAttributes#builder()}.
         *
         * When the {@link Consumer} completes, {@link CompleteWorkflowExecutionDecisionAttributes.Builder#build()} is
         * called immediately and its result is passed to
         * {@link #completeWorkflowExecutionDecisionAttributes(CompleteWorkflowExecutionDecisionAttributes)}.
         * 
         * @param completeWorkflowExecutionDecisionAttributes
         *        a consumer that will call methods on {@link CompleteWorkflowExecutionDecisionAttributes.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #completeWorkflowExecutionDecisionAttributes(CompleteWorkflowExecutionDecisionAttributes)
         */
        default Builder completeWorkflowExecutionDecisionAttributes(
                Consumer<CompleteWorkflowExecutionDecisionAttributes.Builder> completeWorkflowExecutionDecisionAttributes) {
            return completeWorkflowExecutionDecisionAttributes(CompleteWorkflowExecutionDecisionAttributes.builder()
                    .applyMutation(completeWorkflowExecutionDecisionAttributes).build());
        }

        /**
         * <p>
         * Provides the details of the <code>FailWorkflowExecution</code> decision. It isn't set for other decision
         * types.
         * </p>
         * 
         * @param failWorkflowExecutionDecisionAttributes
         *        Provides the details of the <code>FailWorkflowExecution</code> decision. It isn't set for other
         *        decision types.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failWorkflowExecutionDecisionAttributes(
                FailWorkflowExecutionDecisionAttributes failWorkflowExecutionDecisionAttributes);

        /**
         * <p>
         * Provides the details of the <code>FailWorkflowExecution</code> decision. It isn't set for other decision
         * types.
         * </p>
         * This is a convenience that creates an instance of the {@link FailWorkflowExecutionDecisionAttributes.Builder}
         * avoiding the need to create one manually via {@link FailWorkflowExecutionDecisionAttributes#builder()}.
         *
         * When the {@link Consumer} completes, {@link FailWorkflowExecutionDecisionAttributes.Builder#build()} is
         * called immediately and its result is passed to
         * {@link #failWorkflowExecutionDecisionAttributes(FailWorkflowExecutionDecisionAttributes)}.
         * 
         * @param failWorkflowExecutionDecisionAttributes
         *        a consumer that will call methods on {@link FailWorkflowExecutionDecisionAttributes.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #failWorkflowExecutionDecisionAttributes(FailWorkflowExecutionDecisionAttributes)
         */
        default Builder failWorkflowExecutionDecisionAttributes(
                Consumer<FailWorkflowExecutionDecisionAttributes.Builder> failWorkflowExecutionDecisionAttributes) {
            return failWorkflowExecutionDecisionAttributes(FailWorkflowExecutionDecisionAttributes.builder()
                    .applyMutation(failWorkflowExecutionDecisionAttributes).build());
        }

        /**
         * <p>
         * Provides the details of the <code>CancelWorkflowExecution</code> decision. It isn't set for other decision
         * types.
         * </p>
         * 
         * @param cancelWorkflowExecutionDecisionAttributes
         *        Provides the details of the <code>CancelWorkflowExecution</code> decision. It isn't set for other
         *        decision types.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cancelWorkflowExecutionDecisionAttributes(
                CancelWorkflowExecutionDecisionAttributes cancelWorkflowExecutionDecisionAttributes);

        /**
         * <p>
         * Provides the details of the <code>CancelWorkflowExecution</code> decision. It isn't set for other decision
         * types.
         * </p>
         * This is a convenience that creates an instance of the
         * {@link CancelWorkflowExecutionDecisionAttributes.Builder} avoiding the need to create one manually via
         * {@link CancelWorkflowExecutionDecisionAttributes#builder()}.
         *
         * When the {@link Consumer} completes, {@link CancelWorkflowExecutionDecisionAttributes.Builder#build()} is
         * called immediately and its result is passed to
         * {@link #cancelWorkflowExecutionDecisionAttributes(CancelWorkflowExecutionDecisionAttributes)}.
         * 
         * @param cancelWorkflowExecutionDecisionAttributes
         *        a consumer that will call methods on {@link CancelWorkflowExecutionDecisionAttributes.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #cancelWorkflowExecutionDecisionAttributes(CancelWorkflowExecutionDecisionAttributes)
         */
        default Builder cancelWorkflowExecutionDecisionAttributes(
                Consumer<CancelWorkflowExecutionDecisionAttributes.Builder> cancelWorkflowExecutionDecisionAttributes) {
            return cancelWorkflowExecutionDecisionAttributes(CancelWorkflowExecutionDecisionAttributes.builder()
                    .applyMutation(cancelWorkflowExecutionDecisionAttributes).build());
        }

        /**
         * <p>
         * Provides the details of the <code>ContinueAsNewWorkflowExecution</code> decision. It isn't set for other
         * decision types.
         * </p>
         * 
         * @param continueAsNewWorkflowExecutionDecisionAttributes
         *        Provides the details of the <code>ContinueAsNewWorkflowExecution</code> decision. It isn't set for
         *        other decision types.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder continueAsNewWorkflowExecutionDecisionAttributes(
                ContinueAsNewWorkflowExecutionDecisionAttributes continueAsNewWorkflowExecutionDecisionAttributes);

        /**
         * <p>
         * Provides the details of the <code>ContinueAsNewWorkflowExecution</code> decision. It isn't set for other
         * decision types.
         * </p>
         * This is a convenience that creates an instance of the
         * {@link ContinueAsNewWorkflowExecutionDecisionAttributes.Builder} avoiding the need to create one manually via
         * {@link ContinueAsNewWorkflowExecutionDecisionAttributes#builder()}.
         *
         * When the {@link Consumer} completes, {@link ContinueAsNewWorkflowExecutionDecisionAttributes.Builder#build()}
         * is called immediately and its result is passed to
         * {@link #continueAsNewWorkflowExecutionDecisionAttributes(ContinueAsNewWorkflowExecutionDecisionAttributes)}.
         * 
         * @param continueAsNewWorkflowExecutionDecisionAttributes
         *        a consumer that will call methods on {@link ContinueAsNewWorkflowExecutionDecisionAttributes.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #continueAsNewWorkflowExecutionDecisionAttributes(ContinueAsNewWorkflowExecutionDecisionAttributes)
         */
        default Builder continueAsNewWorkflowExecutionDecisionAttributes(
                Consumer<ContinueAsNewWorkflowExecutionDecisionAttributes.Builder> continueAsNewWorkflowExecutionDecisionAttributes) {
            return continueAsNewWorkflowExecutionDecisionAttributes(ContinueAsNewWorkflowExecutionDecisionAttributes.builder()
                    .applyMutation(continueAsNewWorkflowExecutionDecisionAttributes).build());
        }

        /**
         * <p>
         * Provides the details of the <code>RecordMarker</code> decision. It isn't set for other decision types.
         * </p>
         * 
         * @param recordMarkerDecisionAttributes
         *        Provides the details of the <code>RecordMarker</code> decision. It isn't set for other decision types.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recordMarkerDecisionAttributes(RecordMarkerDecisionAttributes recordMarkerDecisionAttributes);

        /**
         * <p>
         * Provides the details of the <code>RecordMarker</code> decision. It isn't set for other decision types.
         * </p>
         * This is a convenience that creates an instance of the {@link RecordMarkerDecisionAttributes.Builder} avoiding
         * the need to create one manually via {@link RecordMarkerDecisionAttributes#builder()}.
         *
         * When the {@link Consumer} completes, {@link RecordMarkerDecisionAttributes.Builder#build()} is called
         * immediately and its result is passed to
         * {@link #recordMarkerDecisionAttributes(RecordMarkerDecisionAttributes)}.
         * 
         * @param recordMarkerDecisionAttributes
         *        a consumer that will call methods on {@link RecordMarkerDecisionAttributes.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #recordMarkerDecisionAttributes(RecordMarkerDecisionAttributes)
         */
        default Builder recordMarkerDecisionAttributes(
                Consumer<RecordMarkerDecisionAttributes.Builder> recordMarkerDecisionAttributes) {
            return recordMarkerDecisionAttributes(RecordMarkerDecisionAttributes.builder()
                    .applyMutation(recordMarkerDecisionAttributes).build());
        }

        /**
         * <p>
         * Provides the details of the <code>StartTimer</code> decision. It isn't set for other decision types.
         * </p>
         * 
         * @param startTimerDecisionAttributes
         *        Provides the details of the <code>StartTimer</code> decision. It isn't set for other decision types.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startTimerDecisionAttributes(StartTimerDecisionAttributes startTimerDecisionAttributes);

        /**
         * <p>
         * Provides the details of the <code>StartTimer</code> decision. It isn't set for other decision types.
         * </p>
         * This is a convenience that creates an instance of the {@link StartTimerDecisionAttributes.Builder} avoiding
         * the need to create one manually via {@link StartTimerDecisionAttributes#builder()}.
         *
         * When the {@link Consumer} completes, {@link StartTimerDecisionAttributes.Builder#build()} is called
         * immediately and its result is passed to {@link #startTimerDecisionAttributes(StartTimerDecisionAttributes)}.
         * 
         * @param startTimerDecisionAttributes
         *        a consumer that will call methods on {@link StartTimerDecisionAttributes.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #startTimerDecisionAttributes(StartTimerDecisionAttributes)
         */
        default Builder startTimerDecisionAttributes(Consumer<StartTimerDecisionAttributes.Builder> startTimerDecisionAttributes) {
            return startTimerDecisionAttributes(StartTimerDecisionAttributes.builder()
                    .applyMutation(startTimerDecisionAttributes).build());
        }

        /**
         * <p>
         * Provides the details of the <code>CancelTimer</code> decision. It isn't set for other decision types.
         * </p>
         * 
         * @param cancelTimerDecisionAttributes
         *        Provides the details of the <code>CancelTimer</code> decision. It isn't set for other decision types.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cancelTimerDecisionAttributes(CancelTimerDecisionAttributes cancelTimerDecisionAttributes);

        /**
         * <p>
         * Provides the details of the <code>CancelTimer</code> decision. It isn't set for other decision types.
         * </p>
         * This is a convenience that creates an instance of the {@link CancelTimerDecisionAttributes.Builder} avoiding
         * the need to create one manually via {@link CancelTimerDecisionAttributes#builder()}.
         *
         * When the {@link Consumer} completes, {@link CancelTimerDecisionAttributes.Builder#build()} is called
         * immediately and its result is passed to {@link #cancelTimerDecisionAttributes(CancelTimerDecisionAttributes)}
         * .
         * 
         * @param cancelTimerDecisionAttributes
         *        a consumer that will call methods on {@link CancelTimerDecisionAttributes.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #cancelTimerDecisionAttributes(CancelTimerDecisionAttributes)
         */
        default Builder cancelTimerDecisionAttributes(
                Consumer<CancelTimerDecisionAttributes.Builder> cancelTimerDecisionAttributes) {
            return cancelTimerDecisionAttributes(CancelTimerDecisionAttributes.builder()
                    .applyMutation(cancelTimerDecisionAttributes).build());
        }

        /**
         * <p>
         * Provides the details of the <code>SignalExternalWorkflowExecution</code> decision. It isn't set for other
         * decision types.
         * </p>
         * 
         * @param signalExternalWorkflowExecutionDecisionAttributes
         *        Provides the details of the <code>SignalExternalWorkflowExecution</code> decision. It isn't set for
         *        other decision types.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder signalExternalWorkflowExecutionDecisionAttributes(
                SignalExternalWorkflowExecutionDecisionAttributes signalExternalWorkflowExecutionDecisionAttributes);

        /**
         * <p>
         * Provides the details of the <code>SignalExternalWorkflowExecution</code> decision. It isn't set for other
         * decision types.
         * </p>
         * This is a convenience that creates an instance of the
         * {@link SignalExternalWorkflowExecutionDecisionAttributes.Builder} avoiding the need to create one manually
         * via {@link SignalExternalWorkflowExecutionDecisionAttributes#builder()}.
         *
         * When the {@link Consumer} completes,
         * {@link SignalExternalWorkflowExecutionDecisionAttributes.Builder#build()} is called immediately and its
         * result is passed to
         * {@link #signalExternalWorkflowExecutionDecisionAttributes(SignalExternalWorkflowExecutionDecisionAttributes)}
         * .
         * 
         * @param signalExternalWorkflowExecutionDecisionAttributes
         *        a consumer that will call methods on {@link SignalExternalWorkflowExecutionDecisionAttributes.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #signalExternalWorkflowExecutionDecisionAttributes(SignalExternalWorkflowExecutionDecisionAttributes)
         */
        default Builder signalExternalWorkflowExecutionDecisionAttributes(
                Consumer<SignalExternalWorkflowExecutionDecisionAttributes.Builder> signalExternalWorkflowExecutionDecisionAttributes) {
            return signalExternalWorkflowExecutionDecisionAttributes(SignalExternalWorkflowExecutionDecisionAttributes.builder()
                    .applyMutation(signalExternalWorkflowExecutionDecisionAttributes).build());
        }

        /**
         * <p>
         * Provides the details of the <code>RequestCancelExternalWorkflowExecution</code> decision. It isn't set for
         * other decision types.
         * </p>
         * 
         * @param requestCancelExternalWorkflowExecutionDecisionAttributes
         *        Provides the details of the <code>RequestCancelExternalWorkflowExecution</code> decision. It isn't set
         *        for other decision types.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requestCancelExternalWorkflowExecutionDecisionAttributes(
                RequestCancelExternalWorkflowExecutionDecisionAttributes requestCancelExternalWorkflowExecutionDecisionAttributes);

        /**
         * <p>
         * Provides the details of the <code>RequestCancelExternalWorkflowExecution</code> decision. It isn't set for
         * other decision types.
         * </p>
         * This is a convenience that creates an instance of the
         * {@link RequestCancelExternalWorkflowExecutionDecisionAttributes.Builder} avoiding the need to create one
         * manually via {@link RequestCancelExternalWorkflowExecutionDecisionAttributes#builder()}.
         *
         * When the {@link Consumer} completes,
         * {@link RequestCancelExternalWorkflowExecutionDecisionAttributes.Builder#build()} is called immediately and
         * its result is passed to
         * {@link #requestCancelExternalWorkflowExecutionDecisionAttributes(RequestCancelExternalWorkflowExecutionDecisionAttributes)}
         * .
         * 
         * @param requestCancelExternalWorkflowExecutionDecisionAttributes
         *        a consumer that will call methods on
         *        {@link RequestCancelExternalWorkflowExecutionDecisionAttributes.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #requestCancelExternalWorkflowExecutionDecisionAttributes(RequestCancelExternalWorkflowExecutionDecisionAttributes)
         */
        default Builder requestCancelExternalWorkflowExecutionDecisionAttributes(
                Consumer<RequestCancelExternalWorkflowExecutionDecisionAttributes.Builder> requestCancelExternalWorkflowExecutionDecisionAttributes) {
            return requestCancelExternalWorkflowExecutionDecisionAttributes(RequestCancelExternalWorkflowExecutionDecisionAttributes
                    .builder().applyMutation(requestCancelExternalWorkflowExecutionDecisionAttributes).build());
        }

        /**
         * <p>
         * Provides the details of the <code>StartChildWorkflowExecution</code> decision. It isn't set for other
         * decision types.
         * </p>
         * 
         * @param startChildWorkflowExecutionDecisionAttributes
         *        Provides the details of the <code>StartChildWorkflowExecution</code> decision. It isn't set for other
         *        decision types.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startChildWorkflowExecutionDecisionAttributes(
                StartChildWorkflowExecutionDecisionAttributes startChildWorkflowExecutionDecisionAttributes);

        /**
         * <p>
         * Provides the details of the <code>StartChildWorkflowExecution</code> decision. It isn't set for other
         * decision types.
         * </p>
         * This is a convenience that creates an instance of the
         * {@link StartChildWorkflowExecutionDecisionAttributes.Builder} avoiding the need to create one manually via
         * {@link StartChildWorkflowExecutionDecisionAttributes#builder()}.
         *
         * When the {@link Consumer} completes, {@link StartChildWorkflowExecutionDecisionAttributes.Builder#build()} is
         * called immediately and its result is passed to
         * {@link #startChildWorkflowExecutionDecisionAttributes(StartChildWorkflowExecutionDecisionAttributes)}.
         * 
         * @param startChildWorkflowExecutionDecisionAttributes
         *        a consumer that will call methods on {@link StartChildWorkflowExecutionDecisionAttributes.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #startChildWorkflowExecutionDecisionAttributes(StartChildWorkflowExecutionDecisionAttributes)
         */
        default Builder startChildWorkflowExecutionDecisionAttributes(
                Consumer<StartChildWorkflowExecutionDecisionAttributes.Builder> startChildWorkflowExecutionDecisionAttributes) {
            return startChildWorkflowExecutionDecisionAttributes(StartChildWorkflowExecutionDecisionAttributes.builder()
                    .applyMutation(startChildWorkflowExecutionDecisionAttributes).build());
        }

        /**
         * <p>
         * Provides the details of the <code>ScheduleLambdaFunction</code> decision. It isn't set for other decision
         * types.
         * </p>
         * 
         * @param scheduleLambdaFunctionDecisionAttributes
         *        Provides the details of the <code>ScheduleLambdaFunction</code> decision. It isn't set for other
         *        decision types.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scheduleLambdaFunctionDecisionAttributes(
                ScheduleLambdaFunctionDecisionAttributes scheduleLambdaFunctionDecisionAttributes);

        /**
         * <p>
         * Provides the details of the <code>ScheduleLambdaFunction</code> decision. It isn't set for other decision
         * types.
         * </p>
         * This is a convenience that creates an instance of the
         * {@link ScheduleLambdaFunctionDecisionAttributes.Builder} avoiding the need to create one manually via
         * {@link ScheduleLambdaFunctionDecisionAttributes#builder()}.
         *
         * When the {@link Consumer} completes, {@link ScheduleLambdaFunctionDecisionAttributes.Builder#build()} is
         * called immediately and its result is passed to
         * {@link #scheduleLambdaFunctionDecisionAttributes(ScheduleLambdaFunctionDecisionAttributes)}.
         * 
         * @param scheduleLambdaFunctionDecisionAttributes
         *        a consumer that will call methods on {@link ScheduleLambdaFunctionDecisionAttributes.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #scheduleLambdaFunctionDecisionAttributes(ScheduleLambdaFunctionDecisionAttributes)
         */
        default Builder scheduleLambdaFunctionDecisionAttributes(
                Consumer<ScheduleLambdaFunctionDecisionAttributes.Builder> scheduleLambdaFunctionDecisionAttributes) {
            return scheduleLambdaFunctionDecisionAttributes(ScheduleLambdaFunctionDecisionAttributes.builder()
                    .applyMutation(scheduleLambdaFunctionDecisionAttributes).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String decisionType;

        private ScheduleActivityTaskDecisionAttributes scheduleActivityTaskDecisionAttributes;

        private RequestCancelActivityTaskDecisionAttributes requestCancelActivityTaskDecisionAttributes;

        private CompleteWorkflowExecutionDecisionAttributes completeWorkflowExecutionDecisionAttributes;

        private FailWorkflowExecutionDecisionAttributes failWorkflowExecutionDecisionAttributes;

        private CancelWorkflowExecutionDecisionAttributes cancelWorkflowExecutionDecisionAttributes;

        private ContinueAsNewWorkflowExecutionDecisionAttributes continueAsNewWorkflowExecutionDecisionAttributes;

        private RecordMarkerDecisionAttributes recordMarkerDecisionAttributes;

        private StartTimerDecisionAttributes startTimerDecisionAttributes;

        private CancelTimerDecisionAttributes cancelTimerDecisionAttributes;

        private SignalExternalWorkflowExecutionDecisionAttributes signalExternalWorkflowExecutionDecisionAttributes;

        private RequestCancelExternalWorkflowExecutionDecisionAttributes requestCancelExternalWorkflowExecutionDecisionAttributes;

        private StartChildWorkflowExecutionDecisionAttributes startChildWorkflowExecutionDecisionAttributes;

        private ScheduleLambdaFunctionDecisionAttributes scheduleLambdaFunctionDecisionAttributes;

        private BuilderImpl() {
        }

        private BuilderImpl(Decision model) {
            decisionType(model.decisionType);
            scheduleActivityTaskDecisionAttributes(model.scheduleActivityTaskDecisionAttributes);
            requestCancelActivityTaskDecisionAttributes(model.requestCancelActivityTaskDecisionAttributes);
            completeWorkflowExecutionDecisionAttributes(model.completeWorkflowExecutionDecisionAttributes);
            failWorkflowExecutionDecisionAttributes(model.failWorkflowExecutionDecisionAttributes);
            cancelWorkflowExecutionDecisionAttributes(model.cancelWorkflowExecutionDecisionAttributes);
            continueAsNewWorkflowExecutionDecisionAttributes(model.continueAsNewWorkflowExecutionDecisionAttributes);
            recordMarkerDecisionAttributes(model.recordMarkerDecisionAttributes);
            startTimerDecisionAttributes(model.startTimerDecisionAttributes);
            cancelTimerDecisionAttributes(model.cancelTimerDecisionAttributes);
            signalExternalWorkflowExecutionDecisionAttributes(model.signalExternalWorkflowExecutionDecisionAttributes);
            requestCancelExternalWorkflowExecutionDecisionAttributes(model.requestCancelExternalWorkflowExecutionDecisionAttributes);
            startChildWorkflowExecutionDecisionAttributes(model.startChildWorkflowExecutionDecisionAttributes);
            scheduleLambdaFunctionDecisionAttributes(model.scheduleLambdaFunctionDecisionAttributes);
        }

        public final String getDecisionTypeAsString() {
            return decisionType;
        }

        @Override
        public final Builder decisionType(String decisionType) {
            this.decisionType = decisionType;
            return this;
        }

        @Override
        public final Builder decisionType(DecisionType decisionType) {
            this.decisionType(decisionType.toString());
            return this;
        }

        public final void setDecisionType(String decisionType) {
            this.decisionType = decisionType;
        }

        public final ScheduleActivityTaskDecisionAttributes.Builder getScheduleActivityTaskDecisionAttributes() {
            return scheduleActivityTaskDecisionAttributes != null ? scheduleActivityTaskDecisionAttributes.toBuilder() : null;
        }

        @Override
        public final Builder scheduleActivityTaskDecisionAttributes(
                ScheduleActivityTaskDecisionAttributes scheduleActivityTaskDecisionAttributes) {
            this.scheduleActivityTaskDecisionAttributes = scheduleActivityTaskDecisionAttributes;
            return this;
        }

        public final void setScheduleActivityTaskDecisionAttributes(
                ScheduleActivityTaskDecisionAttributes.BuilderImpl scheduleActivityTaskDecisionAttributes) {
            this.scheduleActivityTaskDecisionAttributes = scheduleActivityTaskDecisionAttributes != null ? scheduleActivityTaskDecisionAttributes
                    .build() : null;
        }

        public final RequestCancelActivityTaskDecisionAttributes.Builder getRequestCancelActivityTaskDecisionAttributes() {
            return requestCancelActivityTaskDecisionAttributes != null ? requestCancelActivityTaskDecisionAttributes.toBuilder()
                    : null;
        }

        @Override
        public final Builder requestCancelActivityTaskDecisionAttributes(
                RequestCancelActivityTaskDecisionAttributes requestCancelActivityTaskDecisionAttributes) {
            this.requestCancelActivityTaskDecisionAttributes = requestCancelActivityTaskDecisionAttributes;
            return this;
        }

        public final void setRequestCancelActivityTaskDecisionAttributes(
                RequestCancelActivityTaskDecisionAttributes.BuilderImpl requestCancelActivityTaskDecisionAttributes) {
            this.requestCancelActivityTaskDecisionAttributes = requestCancelActivityTaskDecisionAttributes != null ? requestCancelActivityTaskDecisionAttributes
                    .build() : null;
        }

        public final CompleteWorkflowExecutionDecisionAttributes.Builder getCompleteWorkflowExecutionDecisionAttributes() {
            return completeWorkflowExecutionDecisionAttributes != null ? completeWorkflowExecutionDecisionAttributes.toBuilder()
                    : null;
        }

        @Override
        public final Builder completeWorkflowExecutionDecisionAttributes(
                CompleteWorkflowExecutionDecisionAttributes completeWorkflowExecutionDecisionAttributes) {
            this.completeWorkflowExecutionDecisionAttributes = completeWorkflowExecutionDecisionAttributes;
            return this;
        }

        public final void setCompleteWorkflowExecutionDecisionAttributes(
                CompleteWorkflowExecutionDecisionAttributes.BuilderImpl completeWorkflowExecutionDecisionAttributes) {
            this.completeWorkflowExecutionDecisionAttributes = completeWorkflowExecutionDecisionAttributes != null ? completeWorkflowExecutionDecisionAttributes
                    .build() : null;
        }

        public final FailWorkflowExecutionDecisionAttributes.Builder getFailWorkflowExecutionDecisionAttributes() {
            return failWorkflowExecutionDecisionAttributes != null ? failWorkflowExecutionDecisionAttributes.toBuilder() : null;
        }

        @Override
        public final Builder failWorkflowExecutionDecisionAttributes(
                FailWorkflowExecutionDecisionAttributes failWorkflowExecutionDecisionAttributes) {
            this.failWorkflowExecutionDecisionAttributes = failWorkflowExecutionDecisionAttributes;
            return this;
        }

        public final void setFailWorkflowExecutionDecisionAttributes(
                FailWorkflowExecutionDecisionAttributes.BuilderImpl failWorkflowExecutionDecisionAttributes) {
            this.failWorkflowExecutionDecisionAttributes = failWorkflowExecutionDecisionAttributes != null ? failWorkflowExecutionDecisionAttributes
                    .build() : null;
        }

        public final CancelWorkflowExecutionDecisionAttributes.Builder getCancelWorkflowExecutionDecisionAttributes() {
            return cancelWorkflowExecutionDecisionAttributes != null ? cancelWorkflowExecutionDecisionAttributes.toBuilder()
                    : null;
        }

        @Override
        public final Builder cancelWorkflowExecutionDecisionAttributes(
                CancelWorkflowExecutionDecisionAttributes cancelWorkflowExecutionDecisionAttributes) {
            this.cancelWorkflowExecutionDecisionAttributes = cancelWorkflowExecutionDecisionAttributes;
            return this;
        }

        public final void setCancelWorkflowExecutionDecisionAttributes(
                CancelWorkflowExecutionDecisionAttributes.BuilderImpl cancelWorkflowExecutionDecisionAttributes) {
            this.cancelWorkflowExecutionDecisionAttributes = cancelWorkflowExecutionDecisionAttributes != null ? cancelWorkflowExecutionDecisionAttributes
                    .build() : null;
        }

        public final ContinueAsNewWorkflowExecutionDecisionAttributes.Builder getContinueAsNewWorkflowExecutionDecisionAttributes() {
            return continueAsNewWorkflowExecutionDecisionAttributes != null ? continueAsNewWorkflowExecutionDecisionAttributes
                    .toBuilder() : null;
        }

        @Override
        public final Builder continueAsNewWorkflowExecutionDecisionAttributes(
                ContinueAsNewWorkflowExecutionDecisionAttributes continueAsNewWorkflowExecutionDecisionAttributes) {
            this.continueAsNewWorkflowExecutionDecisionAttributes = continueAsNewWorkflowExecutionDecisionAttributes;
            return this;
        }

        public final void setContinueAsNewWorkflowExecutionDecisionAttributes(
                ContinueAsNewWorkflowExecutionDecisionAttributes.BuilderImpl continueAsNewWorkflowExecutionDecisionAttributes) {
            this.continueAsNewWorkflowExecutionDecisionAttributes = continueAsNewWorkflowExecutionDecisionAttributes != null ? continueAsNewWorkflowExecutionDecisionAttributes
                    .build() : null;
        }

        public final RecordMarkerDecisionAttributes.Builder getRecordMarkerDecisionAttributes() {
            return recordMarkerDecisionAttributes != null ? recordMarkerDecisionAttributes.toBuilder() : null;
        }

        @Override
        public final Builder recordMarkerDecisionAttributes(RecordMarkerDecisionAttributes recordMarkerDecisionAttributes) {
            this.recordMarkerDecisionAttributes = recordMarkerDecisionAttributes;
            return this;
        }

        public final void setRecordMarkerDecisionAttributes(
                RecordMarkerDecisionAttributes.BuilderImpl recordMarkerDecisionAttributes) {
            this.recordMarkerDecisionAttributes = recordMarkerDecisionAttributes != null ? recordMarkerDecisionAttributes.build()
                    : null;
        }

        public final StartTimerDecisionAttributes.Builder getStartTimerDecisionAttributes() {
            return startTimerDecisionAttributes != null ? startTimerDecisionAttributes.toBuilder() : null;
        }

        @Override
        public final Builder startTimerDecisionAttributes(StartTimerDecisionAttributes startTimerDecisionAttributes) {
            this.startTimerDecisionAttributes = startTimerDecisionAttributes;
            return this;
        }

        public final void setStartTimerDecisionAttributes(StartTimerDecisionAttributes.BuilderImpl startTimerDecisionAttributes) {
            this.startTimerDecisionAttributes = startTimerDecisionAttributes != null ? startTimerDecisionAttributes.build()
                    : null;
        }

        public final CancelTimerDecisionAttributes.Builder getCancelTimerDecisionAttributes() {
            return cancelTimerDecisionAttributes != null ? cancelTimerDecisionAttributes.toBuilder() : null;
        }

        @Override
        public final Builder cancelTimerDecisionAttributes(CancelTimerDecisionAttributes cancelTimerDecisionAttributes) {
            this.cancelTimerDecisionAttributes = cancelTimerDecisionAttributes;
            return this;
        }

        public final void setCancelTimerDecisionAttributes(CancelTimerDecisionAttributes.BuilderImpl cancelTimerDecisionAttributes) {
            this.cancelTimerDecisionAttributes = cancelTimerDecisionAttributes != null ? cancelTimerDecisionAttributes.build()
                    : null;
        }

        public final SignalExternalWorkflowExecutionDecisionAttributes.Builder getSignalExternalWorkflowExecutionDecisionAttributes() {
            return signalExternalWorkflowExecutionDecisionAttributes != null ? signalExternalWorkflowExecutionDecisionAttributes
                    .toBuilder() : null;
        }

        @Override
        public final Builder signalExternalWorkflowExecutionDecisionAttributes(
                SignalExternalWorkflowExecutionDecisionAttributes signalExternalWorkflowExecutionDecisionAttributes) {
            this.signalExternalWorkflowExecutionDecisionAttributes = signalExternalWorkflowExecutionDecisionAttributes;
            return this;
        }

        public final void setSignalExternalWorkflowExecutionDecisionAttributes(
                SignalExternalWorkflowExecutionDecisionAttributes.BuilderImpl signalExternalWorkflowExecutionDecisionAttributes) {
            this.signalExternalWorkflowExecutionDecisionAttributes = signalExternalWorkflowExecutionDecisionAttributes != null ? signalExternalWorkflowExecutionDecisionAttributes
                    .build() : null;
        }

        public final RequestCancelExternalWorkflowExecutionDecisionAttributes.Builder getRequestCancelExternalWorkflowExecutionDecisionAttributes() {
            return requestCancelExternalWorkflowExecutionDecisionAttributes != null ? requestCancelExternalWorkflowExecutionDecisionAttributes
                    .toBuilder() : null;
        }

        @Override
        public final Builder requestCancelExternalWorkflowExecutionDecisionAttributes(
                RequestCancelExternalWorkflowExecutionDecisionAttributes requestCancelExternalWorkflowExecutionDecisionAttributes) {
            this.requestCancelExternalWorkflowExecutionDecisionAttributes = requestCancelExternalWorkflowExecutionDecisionAttributes;
            return this;
        }

        public final void setRequestCancelExternalWorkflowExecutionDecisionAttributes(
                RequestCancelExternalWorkflowExecutionDecisionAttributes.BuilderImpl requestCancelExternalWorkflowExecutionDecisionAttributes) {
            this.requestCancelExternalWorkflowExecutionDecisionAttributes = requestCancelExternalWorkflowExecutionDecisionAttributes != null ? requestCancelExternalWorkflowExecutionDecisionAttributes
                    .build() : null;
        }

        public final StartChildWorkflowExecutionDecisionAttributes.Builder getStartChildWorkflowExecutionDecisionAttributes() {
            return startChildWorkflowExecutionDecisionAttributes != null ? startChildWorkflowExecutionDecisionAttributes
                    .toBuilder() : null;
        }

        @Override
        public final Builder startChildWorkflowExecutionDecisionAttributes(
                StartChildWorkflowExecutionDecisionAttributes startChildWorkflowExecutionDecisionAttributes) {
            this.startChildWorkflowExecutionDecisionAttributes = startChildWorkflowExecutionDecisionAttributes;
            return this;
        }

        public final void setStartChildWorkflowExecutionDecisionAttributes(
                StartChildWorkflowExecutionDecisionAttributes.BuilderImpl startChildWorkflowExecutionDecisionAttributes) {
            this.startChildWorkflowExecutionDecisionAttributes = startChildWorkflowExecutionDecisionAttributes != null ? startChildWorkflowExecutionDecisionAttributes
                    .build() : null;
        }

        public final ScheduleLambdaFunctionDecisionAttributes.Builder getScheduleLambdaFunctionDecisionAttributes() {
            return scheduleLambdaFunctionDecisionAttributes != null ? scheduleLambdaFunctionDecisionAttributes.toBuilder() : null;
        }

        @Override
        public final Builder scheduleLambdaFunctionDecisionAttributes(
                ScheduleLambdaFunctionDecisionAttributes scheduleLambdaFunctionDecisionAttributes) {
            this.scheduleLambdaFunctionDecisionAttributes = scheduleLambdaFunctionDecisionAttributes;
            return this;
        }

        public final void setScheduleLambdaFunctionDecisionAttributes(
                ScheduleLambdaFunctionDecisionAttributes.BuilderImpl scheduleLambdaFunctionDecisionAttributes) {
            this.scheduleLambdaFunctionDecisionAttributes = scheduleLambdaFunctionDecisionAttributes != null ? scheduleLambdaFunctionDecisionAttributes
                    .build() : null;
        }

        @Override
        public Decision build() {
            return new Decision(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
