/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.swf.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class StartWorkflowExecutionRequest extends SwfRequest implements
        ToCopyableBuilder<StartWorkflowExecutionRequest.Builder, StartWorkflowExecutionRequest> {
    private static final SdkField<String> DOMAIN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(StartWorkflowExecutionRequest::domain)).setter(setter(Builder::domain))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("domain").build()).build();

    private static final SdkField<String> WORKFLOW_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(StartWorkflowExecutionRequest::workflowId)).setter(setter(Builder::workflowId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("workflowId").build()).build();

    private static final SdkField<WorkflowType> WORKFLOW_TYPE_FIELD = SdkField.<WorkflowType> builder(MarshallingType.SDK_POJO)
            .getter(getter(StartWorkflowExecutionRequest::workflowType)).setter(setter(Builder::workflowType))
            .constructor(WorkflowType::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("workflowType").build()).build();

    private static final SdkField<TaskList> TASK_LIST_FIELD = SdkField.<TaskList> builder(MarshallingType.SDK_POJO)
            .getter(getter(StartWorkflowExecutionRequest::taskList)).setter(setter(Builder::taskList))
            .constructor(TaskList::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("taskList").build()).build();

    private static final SdkField<String> TASK_PRIORITY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(StartWorkflowExecutionRequest::taskPriority)).setter(setter(Builder::taskPriority))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("taskPriority").build()).build();

    private static final SdkField<String> INPUT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(StartWorkflowExecutionRequest::input)).setter(setter(Builder::input))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("input").build()).build();

    private static final SdkField<String> EXECUTION_START_TO_CLOSE_TIMEOUT_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(StartWorkflowExecutionRequest::executionStartToCloseTimeout))
            .setter(setter(Builder::executionStartToCloseTimeout))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("executionStartToCloseTimeout")
                    .build()).build();

    private static final SdkField<List<String>> TAG_LIST_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(StartWorkflowExecutionRequest::tagList))
            .setter(setter(Builder::tagList))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tagList").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> TASK_START_TO_CLOSE_TIMEOUT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(StartWorkflowExecutionRequest::taskStartToCloseTimeout))
            .setter(setter(Builder::taskStartToCloseTimeout))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("taskStartToCloseTimeout").build())
            .build();

    private static final SdkField<String> CHILD_POLICY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(StartWorkflowExecutionRequest::childPolicyAsString)).setter(setter(Builder::childPolicy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("childPolicy").build()).build();

    private static final SdkField<String> LAMBDA_ROLE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(StartWorkflowExecutionRequest::lambdaRole)).setter(setter(Builder::lambdaRole))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lambdaRole").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DOMAIN_FIELD,
            WORKFLOW_ID_FIELD, WORKFLOW_TYPE_FIELD, TASK_LIST_FIELD, TASK_PRIORITY_FIELD, INPUT_FIELD,
            EXECUTION_START_TO_CLOSE_TIMEOUT_FIELD, TAG_LIST_FIELD, TASK_START_TO_CLOSE_TIMEOUT_FIELD, CHILD_POLICY_FIELD,
            LAMBDA_ROLE_FIELD));

    private final String domain;

    private final String workflowId;

    private final WorkflowType workflowType;

    private final TaskList taskList;

    private final String taskPriority;

    private final String input;

    private final String executionStartToCloseTimeout;

    private final List<String> tagList;

    private final String taskStartToCloseTimeout;

    private final String childPolicy;

    private final String lambdaRole;

    private StartWorkflowExecutionRequest(BuilderImpl builder) {
        super(builder);
        this.domain = builder.domain;
        this.workflowId = builder.workflowId;
        this.workflowType = builder.workflowType;
        this.taskList = builder.taskList;
        this.taskPriority = builder.taskPriority;
        this.input = builder.input;
        this.executionStartToCloseTimeout = builder.executionStartToCloseTimeout;
        this.tagList = builder.tagList;
        this.taskStartToCloseTimeout = builder.taskStartToCloseTimeout;
        this.childPolicy = builder.childPolicy;
        this.lambdaRole = builder.lambdaRole;
    }

    /**
     * <p>
     * The name of the domain in which the workflow execution is created.
     * </p>
     * 
     * @return The name of the domain in which the workflow execution is created.
     */
    public String domain() {
        return domain;
    }

    /**
     * <p>
     * The user defined identifier associated with the workflow execution. You can use this to associate a custom
     * identifier with the workflow execution. You may specify the same identifier if a workflow execution is logically
     * a <i>restart</i> of a previous execution. You cannot have two open workflow executions with the same
     * <code>workflowId</code> at the same time.
     * </p>
     * <p>
     * The specified string must not start or end with whitespace. It must not contain a <code>:</code> (colon),
     * <code>/</code> (slash), <code>|</code> (vertical bar), or any control characters (<code>\u0000-\u001f</code> |
     * <code>\u007f-\u009f</code>). Also, it must not contain the literal string <code>arn</code>.
     * </p>
     * 
     * @return The user defined identifier associated with the workflow execution. You can use this to associate a
     *         custom identifier with the workflow execution. You may specify the same identifier if a workflow
     *         execution is logically a <i>restart</i> of a previous execution. You cannot have two open workflow
     *         executions with the same <code>workflowId</code> at the same time.</p>
     *         <p>
     *         The specified string must not start or end with whitespace. It must not contain a <code>:</code> (colon),
     *         <code>/</code> (slash), <code>|</code> (vertical bar), or any control characters (
     *         <code>\u0000-\u001f</code> | <code>\u007f-\u009f</code>). Also, it must not contain the literal string
     *         <code>arn</code>.
     */
    public String workflowId() {
        return workflowId;
    }

    /**
     * <p>
     * The type of the workflow to start.
     * </p>
     * 
     * @return The type of the workflow to start.
     */
    public WorkflowType workflowType() {
        return workflowType;
    }

    /**
     * <p>
     * The task list to use for the decision tasks generated for this workflow execution. This overrides the
     * <code>defaultTaskList</code> specified when registering the workflow type.
     * </p>
     * <note>
     * <p>
     * A task list for this workflow execution must be specified either as a default for the workflow type or through
     * this parameter. If neither this parameter is set nor a default task list was specified at registration time then
     * a fault is returned.
     * </p>
     * </note>
     * <p>
     * The specified string must not start or end with whitespace. It must not contain a <code>:</code> (colon),
     * <code>/</code> (slash), <code>|</code> (vertical bar), or any control characters (<code>\u0000-\u001f</code> |
     * <code>\u007f-\u009f</code>). Also, it must not contain the literal string <code>arn</code>.
     * </p>
     * 
     * @return The task list to use for the decision tasks generated for this workflow execution. This overrides the
     *         <code>defaultTaskList</code> specified when registering the workflow type.</p> <note>
     *         <p>
     *         A task list for this workflow execution must be specified either as a default for the workflow type or
     *         through this parameter. If neither this parameter is set nor a default task list was specified at
     *         registration time then a fault is returned.
     *         </p>
     *         </note>
     *         <p>
     *         The specified string must not start or end with whitespace. It must not contain a <code>:</code> (colon),
     *         <code>/</code> (slash), <code>|</code> (vertical bar), or any control characters (
     *         <code>\u0000-\u001f</code> | <code>\u007f-\u009f</code>). Also, it must not contain the literal string
     *         <code>arn</code>.
     */
    public TaskList taskList() {
        return taskList;
    }

    /**
     * <p>
     * The task priority to use for this workflow execution. This overrides any default priority that was assigned when
     * the workflow type was registered. If not set, then the default task priority for the workflow type is used. Valid
     * values are integers that range from Java's <code>Integer.MIN_VALUE</code> (-2147483648) to
     * <code>Integer.MAX_VALUE</code> (2147483647). Higher numbers indicate higher priority.
     * </p>
     * <p>
     * For more information about setting task priority, see <a
     * href="http://docs.aws.amazon.com/amazonswf/latest/developerguide/programming-priority.html">Setting Task
     * Priority</a> in the <i>Amazon SWF Developer Guide</i>.
     * </p>
     * 
     * @return The task priority to use for this workflow execution. This overrides any default priority that was
     *         assigned when the workflow type was registered. If not set, then the default task priority for the
     *         workflow type is used. Valid values are integers that range from Java's <code>Integer.MIN_VALUE</code>
     *         (-2147483648) to <code>Integer.MAX_VALUE</code> (2147483647). Higher numbers indicate higher
     *         priority.</p>
     *         <p>
     *         For more information about setting task priority, see <a
     *         href="http://docs.aws.amazon.com/amazonswf/latest/developerguide/programming-priority.html">Setting Task
     *         Priority</a> in the <i>Amazon SWF Developer Guide</i>.
     */
    public String taskPriority() {
        return taskPriority;
    }

    /**
     * <p>
     * The input for the workflow execution. This is a free form string which should be meaningful to the workflow you
     * are starting. This <code>input</code> is made available to the new workflow execution in the
     * <code>WorkflowExecutionStarted</code> history event.
     * </p>
     * 
     * @return The input for the workflow execution. This is a free form string which should be meaningful to the
     *         workflow you are starting. This <code>input</code> is made available to the new workflow execution in the
     *         <code>WorkflowExecutionStarted</code> history event.
     */
    public String input() {
        return input;
    }

    /**
     * <p>
     * The total duration for this workflow execution. This overrides the defaultExecutionStartToCloseTimeout specified
     * when registering the workflow type.
     * </p>
     * <p>
     * The duration is specified in seconds; an integer greater than or equal to <code>0</code>. Exceeding this limit
     * causes the workflow execution to time out. Unlike some of the other timeout parameters in Amazon SWF, you cannot
     * specify a value of "NONE" for this timeout; there is a one-year max limit on the time that a workflow execution
     * can run.
     * </p>
     * <note>
     * <p>
     * An execution start-to-close timeout must be specified either through this parameter or as a default when the
     * workflow type is registered. If neither this parameter nor a default execution start-to-close timeout is
     * specified, a fault is returned.
     * </p>
     * </note>
     * 
     * @return The total duration for this workflow execution. This overrides the defaultExecutionStartToCloseTimeout
     *         specified when registering the workflow type.</p>
     *         <p>
     *         The duration is specified in seconds; an integer greater than or equal to <code>0</code>. Exceeding this
     *         limit causes the workflow execution to time out. Unlike some of the other timeout parameters in Amazon
     *         SWF, you cannot specify a value of "NONE" for this timeout; there is a one-year max limit on the time
     *         that a workflow execution can run.
     *         </p>
     *         <note>
     *         <p>
     *         An execution start-to-close timeout must be specified either through this parameter or as a default when
     *         the workflow type is registered. If neither this parameter nor a default execution start-to-close timeout
     *         is specified, a fault is returned.
     *         </p>
     */
    public String executionStartToCloseTimeout() {
        return executionStartToCloseTimeout;
    }

    /**
     * <p>
     * The list of tags to associate with the workflow execution. You can specify a maximum of 5 tags. You can list
     * workflow executions with a specific tag by calling <a>ListOpenWorkflowExecutions</a> or
     * <a>ListClosedWorkflowExecutions</a> and specifying a <a>TagFilter</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The list of tags to associate with the workflow execution. You can specify a maximum of 5 tags. You can
     *         list workflow executions with a specific tag by calling <a>ListOpenWorkflowExecutions</a> or
     *         <a>ListClosedWorkflowExecutions</a> and specifying a <a>TagFilter</a>.
     */
    public List<String> tagList() {
        return tagList;
    }

    /**
     * <p>
     * Specifies the maximum duration of decision tasks for this workflow execution. This parameter overrides the
     * <code>defaultTaskStartToCloseTimout</code> specified when registering the workflow type using
     * <a>RegisterWorkflowType</a>.
     * </p>
     * <p>
     * The duration is specified in seconds, an integer greater than or equal to <code>0</code>. You can use
     * <code>NONE</code> to specify unlimited duration.
     * </p>
     * <note>
     * <p>
     * A task start-to-close timeout for this workflow execution must be specified either as a default for the workflow
     * type or through this parameter. If neither this parameter is set nor a default task start-to-close timeout was
     * specified at registration time then a fault is returned.
     * </p>
     * </note>
     * 
     * @return Specifies the maximum duration of decision tasks for this workflow execution. This parameter overrides
     *         the <code>defaultTaskStartToCloseTimout</code> specified when registering the workflow type using
     *         <a>RegisterWorkflowType</a>.</p>
     *         <p>
     *         The duration is specified in seconds, an integer greater than or equal to <code>0</code>. You can use
     *         <code>NONE</code> to specify unlimited duration.
     *         </p>
     *         <note>
     *         <p>
     *         A task start-to-close timeout for this workflow execution must be specified either as a default for the
     *         workflow type or through this parameter. If neither this parameter is set nor a default task
     *         start-to-close timeout was specified at registration time then a fault is returned.
     *         </p>
     */
    public String taskStartToCloseTimeout() {
        return taskStartToCloseTimeout;
    }

    /**
     * <p>
     * If set, specifies the policy to use for the child workflow executions of this workflow execution if it is
     * terminated, by calling the <a>TerminateWorkflowExecution</a> action explicitly or due to an expired timeout. This
     * policy overrides the default child policy specified when registering the workflow type using
     * <a>RegisterWorkflowType</a>.
     * </p>
     * <p>
     * The supported child policies are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>TERMINATE</code> – The child executions are terminated.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>REQUEST_CANCEL</code> – A request to cancel is attempted for each child execution by recording a
     * <code>WorkflowExecutionCancelRequested</code> event in its history. It is up to the decider to take appropriate
     * actions when it receives an execution history with this event.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ABANDON</code> – No action is taken. The child executions continue to run.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * A child policy for this workflow execution must be specified either as a default for the workflow type or through
     * this parameter. If neither this parameter is set nor a default child policy was specified at registration time
     * then a fault is returned.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #childPolicy} will
     * return {@link ChildPolicy#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #childPolicyAsString}.
     * </p>
     * 
     * @return If set, specifies the policy to use for the child workflow executions of this workflow execution if it is
     *         terminated, by calling the <a>TerminateWorkflowExecution</a> action explicitly or due to an expired
     *         timeout. This policy overrides the default child policy specified when registering the workflow type
     *         using <a>RegisterWorkflowType</a>.</p>
     *         <p>
     *         The supported child policies are:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>TERMINATE</code> – The child executions are terminated.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>REQUEST_CANCEL</code> – A request to cancel is attempted for each child execution by recording a
     *         <code>WorkflowExecutionCancelRequested</code> event in its history. It is up to the decider to take
     *         appropriate actions when it receives an execution history with this event.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ABANDON</code> – No action is taken. The child executions continue to run.
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         A child policy for this workflow execution must be specified either as a default for the workflow type or
     *         through this parameter. If neither this parameter is set nor a default child policy was specified at
     *         registration time then a fault is returned.
     *         </p>
     * @see ChildPolicy
     */
    public ChildPolicy childPolicy() {
        return ChildPolicy.fromValue(childPolicy);
    }

    /**
     * <p>
     * If set, specifies the policy to use for the child workflow executions of this workflow execution if it is
     * terminated, by calling the <a>TerminateWorkflowExecution</a> action explicitly or due to an expired timeout. This
     * policy overrides the default child policy specified when registering the workflow type using
     * <a>RegisterWorkflowType</a>.
     * </p>
     * <p>
     * The supported child policies are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>TERMINATE</code> – The child executions are terminated.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>REQUEST_CANCEL</code> – A request to cancel is attempted for each child execution by recording a
     * <code>WorkflowExecutionCancelRequested</code> event in its history. It is up to the decider to take appropriate
     * actions when it receives an execution history with this event.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ABANDON</code> – No action is taken. The child executions continue to run.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * A child policy for this workflow execution must be specified either as a default for the workflow type or through
     * this parameter. If neither this parameter is set nor a default child policy was specified at registration time
     * then a fault is returned.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #childPolicy} will
     * return {@link ChildPolicy#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #childPolicyAsString}.
     * </p>
     * 
     * @return If set, specifies the policy to use for the child workflow executions of this workflow execution if it is
     *         terminated, by calling the <a>TerminateWorkflowExecution</a> action explicitly or due to an expired
     *         timeout. This policy overrides the default child policy specified when registering the workflow type
     *         using <a>RegisterWorkflowType</a>.</p>
     *         <p>
     *         The supported child policies are:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>TERMINATE</code> – The child executions are terminated.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>REQUEST_CANCEL</code> – A request to cancel is attempted for each child execution by recording a
     *         <code>WorkflowExecutionCancelRequested</code> event in its history. It is up to the decider to take
     *         appropriate actions when it receives an execution history with this event.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ABANDON</code> – No action is taken. The child executions continue to run.
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         A child policy for this workflow execution must be specified either as a default for the workflow type or
     *         through this parameter. If neither this parameter is set nor a default child policy was specified at
     *         registration time then a fault is returned.
     *         </p>
     * @see ChildPolicy
     */
    public String childPolicyAsString() {
        return childPolicy;
    }

    /**
     * <p>
     * The IAM role to attach to this workflow execution.
     * </p>
     * <note>
     * <p>
     * Executions of this workflow type need IAM roles to invoke Lambda functions. If you don't attach an IAM role, any
     * attempt to schedule a Lambda task fails. This results in a <code>ScheduleLambdaFunctionFailed</code> history
     * event. For more information, see <a
     * href="http://docs.aws.amazon.com/amazonswf/latest/developerguide/lambda-task.html"
     * >http://docs.aws.amazon.com/amazonswf/latest/developerguide/lambda-task.html</a> in the <i>Amazon SWF Developer
     * Guide</i>.
     * </p>
     * </note>
     * 
     * @return The IAM role to attach to this workflow execution.</p> <note>
     *         <p>
     *         Executions of this workflow type need IAM roles to invoke Lambda functions. If you don't attach an IAM
     *         role, any attempt to schedule a Lambda task fails. This results in a
     *         <code>ScheduleLambdaFunctionFailed</code> history event. For more information, see <a
     *         href="http://docs.aws.amazon.com/amazonswf/latest/developerguide/lambda-task.html"
     *         >http://docs.aws.amazon.com/amazonswf/latest/developerguide/lambda-task.html</a> in the <i>Amazon SWF
     *         Developer Guide</i>.
     *         </p>
     */
    public String lambdaRole() {
        return lambdaRole;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(domain());
        hashCode = 31 * hashCode + Objects.hashCode(workflowId());
        hashCode = 31 * hashCode + Objects.hashCode(workflowType());
        hashCode = 31 * hashCode + Objects.hashCode(taskList());
        hashCode = 31 * hashCode + Objects.hashCode(taskPriority());
        hashCode = 31 * hashCode + Objects.hashCode(input());
        hashCode = 31 * hashCode + Objects.hashCode(executionStartToCloseTimeout());
        hashCode = 31 * hashCode + Objects.hashCode(tagList());
        hashCode = 31 * hashCode + Objects.hashCode(taskStartToCloseTimeout());
        hashCode = 31 * hashCode + Objects.hashCode(childPolicyAsString());
        hashCode = 31 * hashCode + Objects.hashCode(lambdaRole());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof StartWorkflowExecutionRequest)) {
            return false;
        }
        StartWorkflowExecutionRequest other = (StartWorkflowExecutionRequest) obj;
        return Objects.equals(domain(), other.domain()) && Objects.equals(workflowId(), other.workflowId())
                && Objects.equals(workflowType(), other.workflowType()) && Objects.equals(taskList(), other.taskList())
                && Objects.equals(taskPriority(), other.taskPriority()) && Objects.equals(input(), other.input())
                && Objects.equals(executionStartToCloseTimeout(), other.executionStartToCloseTimeout())
                && Objects.equals(tagList(), other.tagList())
                && Objects.equals(taskStartToCloseTimeout(), other.taskStartToCloseTimeout())
                && Objects.equals(childPolicyAsString(), other.childPolicyAsString())
                && Objects.equals(lambdaRole(), other.lambdaRole());
    }

    @Override
    public String toString() {
        return ToString.builder("StartWorkflowExecutionRequest").add("Domain", domain()).add("WorkflowId", workflowId())
                .add("WorkflowType", workflowType()).add("TaskList", taskList()).add("TaskPriority", taskPriority())
                .add("Input", input()).add("ExecutionStartToCloseTimeout", executionStartToCloseTimeout())
                .add("TagList", tagList()).add("TaskStartToCloseTimeout", taskStartToCloseTimeout())
                .add("ChildPolicy", childPolicyAsString()).add("LambdaRole", lambdaRole()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "domain":
            return Optional.ofNullable(clazz.cast(domain()));
        case "workflowId":
            return Optional.ofNullable(clazz.cast(workflowId()));
        case "workflowType":
            return Optional.ofNullable(clazz.cast(workflowType()));
        case "taskList":
            return Optional.ofNullable(clazz.cast(taskList()));
        case "taskPriority":
            return Optional.ofNullable(clazz.cast(taskPriority()));
        case "input":
            return Optional.ofNullable(clazz.cast(input()));
        case "executionStartToCloseTimeout":
            return Optional.ofNullable(clazz.cast(executionStartToCloseTimeout()));
        case "tagList":
            return Optional.ofNullable(clazz.cast(tagList()));
        case "taskStartToCloseTimeout":
            return Optional.ofNullable(clazz.cast(taskStartToCloseTimeout()));
        case "childPolicy":
            return Optional.ofNullable(clazz.cast(childPolicyAsString()));
        case "lambdaRole":
            return Optional.ofNullable(clazz.cast(lambdaRole()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<StartWorkflowExecutionRequest, T> g) {
        return obj -> g.apply((StartWorkflowExecutionRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SwfRequest.Builder, SdkPojo, CopyableBuilder<Builder, StartWorkflowExecutionRequest> {
        /**
         * <p>
         * The name of the domain in which the workflow execution is created.
         * </p>
         * 
         * @param domain
         *        The name of the domain in which the workflow execution is created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder domain(String domain);

        /**
         * <p>
         * The user defined identifier associated with the workflow execution. You can use this to associate a custom
         * identifier with the workflow execution. You may specify the same identifier if a workflow execution is
         * logically a <i>restart</i> of a previous execution. You cannot have two open workflow executions with the
         * same <code>workflowId</code> at the same time.
         * </p>
         * <p>
         * The specified string must not start or end with whitespace. It must not contain a <code>:</code> (colon),
         * <code>/</code> (slash), <code>|</code> (vertical bar), or any control characters (<code>\u0000-\u001f</code>
         * | <code>\u007f-\u009f</code>). Also, it must not contain the literal string <code>arn</code>.
         * </p>
         * 
         * @param workflowId
         *        The user defined identifier associated with the workflow execution. You can use this to associate a
         *        custom identifier with the workflow execution. You may specify the same identifier if a workflow
         *        execution is logically a <i>restart</i> of a previous execution. You cannot have two open workflow
         *        executions with the same <code>workflowId</code> at the same time.</p>
         *        <p>
         *        The specified string must not start or end with whitespace. It must not contain a <code>:</code>
         *        (colon), <code>/</code> (slash), <code>|</code> (vertical bar), or any control characters (
         *        <code>\u0000-\u001f</code> | <code>\u007f-\u009f</code>). Also, it must not contain the literal string
         *        <code>arn</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder workflowId(String workflowId);

        /**
         * <p>
         * The type of the workflow to start.
         * </p>
         * 
         * @param workflowType
         *        The type of the workflow to start.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder workflowType(WorkflowType workflowType);

        /**
         * <p>
         * The type of the workflow to start.
         * </p>
         * This is a convenience that creates an instance of the {@link WorkflowType.Builder} avoiding the need to
         * create one manually via {@link WorkflowType#builder()}.
         *
         * When the {@link Consumer} completes, {@link WorkflowType.Builder#build()} is called immediately and its
         * result is passed to {@link #workflowType(WorkflowType)}.
         * 
         * @param workflowType
         *        a consumer that will call methods on {@link WorkflowType.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #workflowType(WorkflowType)
         */
        default Builder workflowType(Consumer<WorkflowType.Builder> workflowType) {
            return workflowType(WorkflowType.builder().applyMutation(workflowType).build());
        }

        /**
         * <p>
         * The task list to use for the decision tasks generated for this workflow execution. This overrides the
         * <code>defaultTaskList</code> specified when registering the workflow type.
         * </p>
         * <note>
         * <p>
         * A task list for this workflow execution must be specified either as a default for the workflow type or
         * through this parameter. If neither this parameter is set nor a default task list was specified at
         * registration time then a fault is returned.
         * </p>
         * </note>
         * <p>
         * The specified string must not start or end with whitespace. It must not contain a <code>:</code> (colon),
         * <code>/</code> (slash), <code>|</code> (vertical bar), or any control characters (<code>\u0000-\u001f</code>
         * | <code>\u007f-\u009f</code>). Also, it must not contain the literal string <code>arn</code>.
         * </p>
         * 
         * @param taskList
         *        The task list to use for the decision tasks generated for this workflow execution. This overrides the
         *        <code>defaultTaskList</code> specified when registering the workflow type.</p> <note>
         *        <p>
         *        A task list for this workflow execution must be specified either as a default for the workflow type or
         *        through this parameter. If neither this parameter is set nor a default task list was specified at
         *        registration time then a fault is returned.
         *        </p>
         *        </note>
         *        <p>
         *        The specified string must not start or end with whitespace. It must not contain a <code>:</code>
         *        (colon), <code>/</code> (slash), <code>|</code> (vertical bar), or any control characters (
         *        <code>\u0000-\u001f</code> | <code>\u007f-\u009f</code>). Also, it must not contain the literal string
         *        <code>arn</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder taskList(TaskList taskList);

        /**
         * <p>
         * The task list to use for the decision tasks generated for this workflow execution. This overrides the
         * <code>defaultTaskList</code> specified when registering the workflow type.
         * </p>
         * <note>
         * <p>
         * A task list for this workflow execution must be specified either as a default for the workflow type or
         * through this parameter. If neither this parameter is set nor a default task list was specified at
         * registration time then a fault is returned.
         * </p>
         * </note>
         * <p>
         * The specified string must not start or end with whitespace. It must not contain a <code>:</code> (colon),
         * <code>/</code> (slash), <code>|</code> (vertical bar), or any control characters (<code>\u0000-\u001f</code>
         * | <code>\u007f-\u009f</code>). Also, it must not contain the literal string <code>arn</code>.
         * </p>
         * This is a convenience that creates an instance of the {@link TaskList.Builder} avoiding the need to create
         * one manually via {@link TaskList#builder()}.
         *
         * When the {@link Consumer} completes, {@link TaskList.Builder#build()} is called immediately and its result is
         * passed to {@link #taskList(TaskList)}.
         * 
         * @param taskList
         *        a consumer that will call methods on {@link TaskList.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #taskList(TaskList)
         */
        default Builder taskList(Consumer<TaskList.Builder> taskList) {
            return taskList(TaskList.builder().applyMutation(taskList).build());
        }

        /**
         * <p>
         * The task priority to use for this workflow execution. This overrides any default priority that was assigned
         * when the workflow type was registered. If not set, then the default task priority for the workflow type is
         * used. Valid values are integers that range from Java's <code>Integer.MIN_VALUE</code> (-2147483648) to
         * <code>Integer.MAX_VALUE</code> (2147483647). Higher numbers indicate higher priority.
         * </p>
         * <p>
         * For more information about setting task priority, see <a
         * href="http://docs.aws.amazon.com/amazonswf/latest/developerguide/programming-priority.html">Setting Task
         * Priority</a> in the <i>Amazon SWF Developer Guide</i>.
         * </p>
         * 
         * @param taskPriority
         *        The task priority to use for this workflow execution. This overrides any default priority that was
         *        assigned when the workflow type was registered. If not set, then the default task priority for the
         *        workflow type is used. Valid values are integers that range from Java's <code>Integer.MIN_VALUE</code>
         *        (-2147483648) to <code>Integer.MAX_VALUE</code> (2147483647). Higher numbers indicate higher
         *        priority.</p>
         *        <p>
         *        For more information about setting task priority, see <a
         *        href="http://docs.aws.amazon.com/amazonswf/latest/developerguide/programming-priority.html">Setting
         *        Task Priority</a> in the <i>Amazon SWF Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder taskPriority(String taskPriority);

        /**
         * <p>
         * The input for the workflow execution. This is a free form string which should be meaningful to the workflow
         * you are starting. This <code>input</code> is made available to the new workflow execution in the
         * <code>WorkflowExecutionStarted</code> history event.
         * </p>
         * 
         * @param input
         *        The input for the workflow execution. This is a free form string which should be meaningful to the
         *        workflow you are starting. This <code>input</code> is made available to the new workflow execution in
         *        the <code>WorkflowExecutionStarted</code> history event.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder input(String input);

        /**
         * <p>
         * The total duration for this workflow execution. This overrides the defaultExecutionStartToCloseTimeout
         * specified when registering the workflow type.
         * </p>
         * <p>
         * The duration is specified in seconds; an integer greater than or equal to <code>0</code>. Exceeding this
         * limit causes the workflow execution to time out. Unlike some of the other timeout parameters in Amazon SWF,
         * you cannot specify a value of "NONE" for this timeout; there is a one-year max limit on the time that a
         * workflow execution can run.
         * </p>
         * <note>
         * <p>
         * An execution start-to-close timeout must be specified either through this parameter or as a default when the
         * workflow type is registered. If neither this parameter nor a default execution start-to-close timeout is
         * specified, a fault is returned.
         * </p>
         * </note>
         * 
         * @param executionStartToCloseTimeout
         *        The total duration for this workflow execution. This overrides the defaultExecutionStartToCloseTimeout
         *        specified when registering the workflow type.</p>
         *        <p>
         *        The duration is specified in seconds; an integer greater than or equal to <code>0</code>. Exceeding
         *        this limit causes the workflow execution to time out. Unlike some of the other timeout parameters in
         *        Amazon SWF, you cannot specify a value of "NONE" for this timeout; there is a one-year max limit on
         *        the time that a workflow execution can run.
         *        </p>
         *        <note>
         *        <p>
         *        An execution start-to-close timeout must be specified either through this parameter or as a default
         *        when the workflow type is registered. If neither this parameter nor a default execution start-to-close
         *        timeout is specified, a fault is returned.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder executionStartToCloseTimeout(String executionStartToCloseTimeout);

        /**
         * <p>
         * The list of tags to associate with the workflow execution. You can specify a maximum of 5 tags. You can list
         * workflow executions with a specific tag by calling <a>ListOpenWorkflowExecutions</a> or
         * <a>ListClosedWorkflowExecutions</a> and specifying a <a>TagFilter</a>.
         * </p>
         * 
         * @param tagList
         *        The list of tags to associate with the workflow execution. You can specify a maximum of 5 tags. You
         *        can list workflow executions with a specific tag by calling <a>ListOpenWorkflowExecutions</a> or
         *        <a>ListClosedWorkflowExecutions</a> and specifying a <a>TagFilter</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tagList(Collection<String> tagList);

        /**
         * <p>
         * The list of tags to associate with the workflow execution. You can specify a maximum of 5 tags. You can list
         * workflow executions with a specific tag by calling <a>ListOpenWorkflowExecutions</a> or
         * <a>ListClosedWorkflowExecutions</a> and specifying a <a>TagFilter</a>.
         * </p>
         * 
         * @param tagList
         *        The list of tags to associate with the workflow execution. You can specify a maximum of 5 tags. You
         *        can list workflow executions with a specific tag by calling <a>ListOpenWorkflowExecutions</a> or
         *        <a>ListClosedWorkflowExecutions</a> and specifying a <a>TagFilter</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tagList(String... tagList);

        /**
         * <p>
         * Specifies the maximum duration of decision tasks for this workflow execution. This parameter overrides the
         * <code>defaultTaskStartToCloseTimout</code> specified when registering the workflow type using
         * <a>RegisterWorkflowType</a>.
         * </p>
         * <p>
         * The duration is specified in seconds, an integer greater than or equal to <code>0</code>. You can use
         * <code>NONE</code> to specify unlimited duration.
         * </p>
         * <note>
         * <p>
         * A task start-to-close timeout for this workflow execution must be specified either as a default for the
         * workflow type or through this parameter. If neither this parameter is set nor a default task start-to-close
         * timeout was specified at registration time then a fault is returned.
         * </p>
         * </note>
         * 
         * @param taskStartToCloseTimeout
         *        Specifies the maximum duration of decision tasks for this workflow execution. This parameter overrides
         *        the <code>defaultTaskStartToCloseTimout</code> specified when registering the workflow type using
         *        <a>RegisterWorkflowType</a>.</p>
         *        <p>
         *        The duration is specified in seconds, an integer greater than or equal to <code>0</code>. You can use
         *        <code>NONE</code> to specify unlimited duration.
         *        </p>
         *        <note>
         *        <p>
         *        A task start-to-close timeout for this workflow execution must be specified either as a default for
         *        the workflow type or through this parameter. If neither this parameter is set nor a default task
         *        start-to-close timeout was specified at registration time then a fault is returned.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder taskStartToCloseTimeout(String taskStartToCloseTimeout);

        /**
         * <p>
         * If set, specifies the policy to use for the child workflow executions of this workflow execution if it is
         * terminated, by calling the <a>TerminateWorkflowExecution</a> action explicitly or due to an expired timeout.
         * This policy overrides the default child policy specified when registering the workflow type using
         * <a>RegisterWorkflowType</a>.
         * </p>
         * <p>
         * The supported child policies are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>TERMINATE</code> – The child executions are terminated.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>REQUEST_CANCEL</code> – A request to cancel is attempted for each child execution by recording a
         * <code>WorkflowExecutionCancelRequested</code> event in its history. It is up to the decider to take
         * appropriate actions when it receives an execution history with this event.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ABANDON</code> – No action is taken. The child executions continue to run.
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * A child policy for this workflow execution must be specified either as a default for the workflow type or
         * through this parameter. If neither this parameter is set nor a default child policy was specified at
         * registration time then a fault is returned.
         * </p>
         * </note>
         * 
         * @param childPolicy
         *        If set, specifies the policy to use for the child workflow executions of this workflow execution if it
         *        is terminated, by calling the <a>TerminateWorkflowExecution</a> action explicitly or due to an expired
         *        timeout. This policy overrides the default child policy specified when registering the workflow type
         *        using <a>RegisterWorkflowType</a>.</p>
         *        <p>
         *        The supported child policies are:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>TERMINATE</code> – The child executions are terminated.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>REQUEST_CANCEL</code> – A request to cancel is attempted for each child execution by recording a
         *        <code>WorkflowExecutionCancelRequested</code> event in its history. It is up to the decider to take
         *        appropriate actions when it receives an execution history with this event.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ABANDON</code> – No action is taken. The child executions continue to run.
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        A child policy for this workflow execution must be specified either as a default for the workflow type
         *        or through this parameter. If neither this parameter is set nor a default child policy was specified
         *        at registration time then a fault is returned.
         *        </p>
         * @see ChildPolicy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ChildPolicy
         */
        Builder childPolicy(String childPolicy);

        /**
         * <p>
         * If set, specifies the policy to use for the child workflow executions of this workflow execution if it is
         * terminated, by calling the <a>TerminateWorkflowExecution</a> action explicitly or due to an expired timeout.
         * This policy overrides the default child policy specified when registering the workflow type using
         * <a>RegisterWorkflowType</a>.
         * </p>
         * <p>
         * The supported child policies are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>TERMINATE</code> – The child executions are terminated.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>REQUEST_CANCEL</code> – A request to cancel is attempted for each child execution by recording a
         * <code>WorkflowExecutionCancelRequested</code> event in its history. It is up to the decider to take
         * appropriate actions when it receives an execution history with this event.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ABANDON</code> – No action is taken. The child executions continue to run.
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * A child policy for this workflow execution must be specified either as a default for the workflow type or
         * through this parameter. If neither this parameter is set nor a default child policy was specified at
         * registration time then a fault is returned.
         * </p>
         * </note>
         * 
         * @param childPolicy
         *        If set, specifies the policy to use for the child workflow executions of this workflow execution if it
         *        is terminated, by calling the <a>TerminateWorkflowExecution</a> action explicitly or due to an expired
         *        timeout. This policy overrides the default child policy specified when registering the workflow type
         *        using <a>RegisterWorkflowType</a>.</p>
         *        <p>
         *        The supported child policies are:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>TERMINATE</code> – The child executions are terminated.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>REQUEST_CANCEL</code> – A request to cancel is attempted for each child execution by recording a
         *        <code>WorkflowExecutionCancelRequested</code> event in its history. It is up to the decider to take
         *        appropriate actions when it receives an execution history with this event.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ABANDON</code> – No action is taken. The child executions continue to run.
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        A child policy for this workflow execution must be specified either as a default for the workflow type
         *        or through this parameter. If neither this parameter is set nor a default child policy was specified
         *        at registration time then a fault is returned.
         *        </p>
         * @see ChildPolicy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ChildPolicy
         */
        Builder childPolicy(ChildPolicy childPolicy);

        /**
         * <p>
         * The IAM role to attach to this workflow execution.
         * </p>
         * <note>
         * <p>
         * Executions of this workflow type need IAM roles to invoke Lambda functions. If you don't attach an IAM role,
         * any attempt to schedule a Lambda task fails. This results in a <code>ScheduleLambdaFunctionFailed</code>
         * history event. For more information, see <a
         * href="http://docs.aws.amazon.com/amazonswf/latest/developerguide/lambda-task.html"
         * >http://docs.aws.amazon.com/amazonswf/latest/developerguide/lambda-task.html</a> in the <i>Amazon SWF
         * Developer Guide</i>.
         * </p>
         * </note>
         * 
         * @param lambdaRole
         *        The IAM role to attach to this workflow execution.</p> <note>
         *        <p>
         *        Executions of this workflow type need IAM roles to invoke Lambda functions. If you don't attach an IAM
         *        role, any attempt to schedule a Lambda task fails. This results in a
         *        <code>ScheduleLambdaFunctionFailed</code> history event. For more information, see <a
         *        href="http://docs.aws.amazon.com/amazonswf/latest/developerguide/lambda-task.html"
         *        >http://docs.aws.amazon.com/amazonswf/latest/developerguide/lambda-task.html</a> in the <i>Amazon SWF
         *        Developer Guide</i>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lambdaRole(String lambdaRole);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends SwfRequest.BuilderImpl implements Builder {
        private String domain;

        private String workflowId;

        private WorkflowType workflowType;

        private TaskList taskList;

        private String taskPriority;

        private String input;

        private String executionStartToCloseTimeout;

        private List<String> tagList = DefaultSdkAutoConstructList.getInstance();

        private String taskStartToCloseTimeout;

        private String childPolicy;

        private String lambdaRole;

        private BuilderImpl() {
        }

        private BuilderImpl(StartWorkflowExecutionRequest model) {
            super(model);
            domain(model.domain);
            workflowId(model.workflowId);
            workflowType(model.workflowType);
            taskList(model.taskList);
            taskPriority(model.taskPriority);
            input(model.input);
            executionStartToCloseTimeout(model.executionStartToCloseTimeout);
            tagList(model.tagList);
            taskStartToCloseTimeout(model.taskStartToCloseTimeout);
            childPolicy(model.childPolicy);
            lambdaRole(model.lambdaRole);
        }

        public final String getDomain() {
            return domain;
        }

        @Override
        public final Builder domain(String domain) {
            this.domain = domain;
            return this;
        }

        public final void setDomain(String domain) {
            this.domain = domain;
        }

        public final String getWorkflowId() {
            return workflowId;
        }

        @Override
        public final Builder workflowId(String workflowId) {
            this.workflowId = workflowId;
            return this;
        }

        public final void setWorkflowId(String workflowId) {
            this.workflowId = workflowId;
        }

        public final WorkflowType.Builder getWorkflowType() {
            return workflowType != null ? workflowType.toBuilder() : null;
        }

        @Override
        public final Builder workflowType(WorkflowType workflowType) {
            this.workflowType = workflowType;
            return this;
        }

        public final void setWorkflowType(WorkflowType.BuilderImpl workflowType) {
            this.workflowType = workflowType != null ? workflowType.build() : null;
        }

        public final TaskList.Builder getTaskList() {
            return taskList != null ? taskList.toBuilder() : null;
        }

        @Override
        public final Builder taskList(TaskList taskList) {
            this.taskList = taskList;
            return this;
        }

        public final void setTaskList(TaskList.BuilderImpl taskList) {
            this.taskList = taskList != null ? taskList.build() : null;
        }

        public final String getTaskPriority() {
            return taskPriority;
        }

        @Override
        public final Builder taskPriority(String taskPriority) {
            this.taskPriority = taskPriority;
            return this;
        }

        public final void setTaskPriority(String taskPriority) {
            this.taskPriority = taskPriority;
        }

        public final String getInput() {
            return input;
        }

        @Override
        public final Builder input(String input) {
            this.input = input;
            return this;
        }

        public final void setInput(String input) {
            this.input = input;
        }

        public final String getExecutionStartToCloseTimeout() {
            return executionStartToCloseTimeout;
        }

        @Override
        public final Builder executionStartToCloseTimeout(String executionStartToCloseTimeout) {
            this.executionStartToCloseTimeout = executionStartToCloseTimeout;
            return this;
        }

        public final void setExecutionStartToCloseTimeout(String executionStartToCloseTimeout) {
            this.executionStartToCloseTimeout = executionStartToCloseTimeout;
        }

        public final Collection<String> getTagList() {
            return tagList;
        }

        @Override
        public final Builder tagList(Collection<String> tagList) {
            this.tagList = TagListCopier.copy(tagList);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tagList(String... tagList) {
            tagList(Arrays.asList(tagList));
            return this;
        }

        public final void setTagList(Collection<String> tagList) {
            this.tagList = TagListCopier.copy(tagList);
        }

        public final String getTaskStartToCloseTimeout() {
            return taskStartToCloseTimeout;
        }

        @Override
        public final Builder taskStartToCloseTimeout(String taskStartToCloseTimeout) {
            this.taskStartToCloseTimeout = taskStartToCloseTimeout;
            return this;
        }

        public final void setTaskStartToCloseTimeout(String taskStartToCloseTimeout) {
            this.taskStartToCloseTimeout = taskStartToCloseTimeout;
        }

        public final String getChildPolicyAsString() {
            return childPolicy;
        }

        @Override
        public final Builder childPolicy(String childPolicy) {
            this.childPolicy = childPolicy;
            return this;
        }

        @Override
        public final Builder childPolicy(ChildPolicy childPolicy) {
            this.childPolicy(childPolicy.toString());
            return this;
        }

        public final void setChildPolicy(String childPolicy) {
            this.childPolicy = childPolicy;
        }

        public final String getLambdaRole() {
            return lambdaRole;
        }

        @Override
        public final Builder lambdaRole(String lambdaRole) {
            this.lambdaRole = lambdaRole;
            return this;
        }

        public final void setLambdaRole(String lambdaRole) {
            this.lambdaRole = lambdaRole;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public StartWorkflowExecutionRequest build() {
            return new StartWorkflowExecutionRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
