/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.synthetics.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An object that specifies what screenshots to use as a baseline for visual monitoring by this canary. It can
 * optionally also specify parts of the screenshots to ignore during the visual monitoring comparison.
 * </p>
 * <p>
 * Visual monitoring is supported only on canaries running the <b>syn-puppeteer-node-3.2</b> runtime or later. For more
 * information, see <a href=
 * "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Library_SyntheticsLogger_VisualTesting.html"
 * > Visual monitoring</a> and <a href=
 * "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries_Blueprints_VisualTesting.html"
 * > Visual monitoring blueprint</a>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class VisualReferenceInput implements SdkPojo, Serializable,
        ToCopyableBuilder<VisualReferenceInput.Builder, VisualReferenceInput> {
    private static final SdkField<List<BaseScreenshot>> BASE_SCREENSHOTS_FIELD = SdkField
            .<List<BaseScreenshot>> builder(MarshallingType.LIST)
            .memberName("BaseScreenshots")
            .getter(getter(VisualReferenceInput::baseScreenshots))
            .setter(setter(Builder::baseScreenshots))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BaseScreenshots").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<BaseScreenshot> builder(MarshallingType.SDK_POJO)
                                            .constructor(BaseScreenshot::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> BASE_CANARY_RUN_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("BaseCanaryRunId").getter(getter(VisualReferenceInput::baseCanaryRunId))
            .setter(setter(Builder::baseCanaryRunId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BaseCanaryRunId").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(BASE_SCREENSHOTS_FIELD,
            BASE_CANARY_RUN_ID_FIELD));

    private static final long serialVersionUID = 1L;

    private final List<BaseScreenshot> baseScreenshots;

    private final String baseCanaryRunId;

    private VisualReferenceInput(BuilderImpl builder) {
        this.baseScreenshots = builder.baseScreenshots;
        this.baseCanaryRunId = builder.baseCanaryRunId;
    }

    /**
     * For responses, this returns true if the service returned a value for the BaseScreenshots property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasBaseScreenshots() {
        return baseScreenshots != null && !(baseScreenshots instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of screenshots that will be used as the baseline for visual monitoring in future runs of this canary. If
     * there is a screenshot that you don't want to be used for visual monitoring, remove it from this array.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasBaseScreenshots} method.
     * </p>
     * 
     * @return An array of screenshots that will be used as the baseline for visual monitoring in future runs of this
     *         canary. If there is a screenshot that you don't want to be used for visual monitoring, remove it from
     *         this array.
     */
    public final List<BaseScreenshot> baseScreenshots() {
        return baseScreenshots;
    }

    /**
     * <p>
     * Specifies which canary run to use the screenshots from as the baseline for future visual monitoring with this
     * canary. Valid values are <code>nextrun</code> to use the screenshots from the next run after this update is made,
     * <code>lastrun</code> to use the screenshots from the most recent run before this update was made, or the value of
     * <code>Id</code> in the <a
     * href="https://docs.aws.amazon.com/AmazonSynthetics/latest/APIReference/API_CanaryRun.html"> CanaryRun</a> from
     * any past run of this canary.
     * </p>
     * 
     * @return Specifies which canary run to use the screenshots from as the baseline for future visual monitoring with
     *         this canary. Valid values are <code>nextrun</code> to use the screenshots from the next run after this
     *         update is made, <code>lastrun</code> to use the screenshots from the most recent run before this update
     *         was made, or the value of <code>Id</code> in the <a
     *         href="https://docs.aws.amazon.com/AmazonSynthetics/latest/APIReference/API_CanaryRun.html"> CanaryRun</a>
     *         from any past run of this canary.
     */
    public final String baseCanaryRunId() {
        return baseCanaryRunId;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasBaseScreenshots() ? baseScreenshots() : null);
        hashCode = 31 * hashCode + Objects.hashCode(baseCanaryRunId());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof VisualReferenceInput)) {
            return false;
        }
        VisualReferenceInput other = (VisualReferenceInput) obj;
        return hasBaseScreenshots() == other.hasBaseScreenshots() && Objects.equals(baseScreenshots(), other.baseScreenshots())
                && Objects.equals(baseCanaryRunId(), other.baseCanaryRunId());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("VisualReferenceInput").add("BaseScreenshots", hasBaseScreenshots() ? baseScreenshots() : null)
                .add("BaseCanaryRunId", baseCanaryRunId()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "BaseScreenshots":
            return Optional.ofNullable(clazz.cast(baseScreenshots()));
        case "BaseCanaryRunId":
            return Optional.ofNullable(clazz.cast(baseCanaryRunId()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<VisualReferenceInput, T> g) {
        return obj -> g.apply((VisualReferenceInput) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, VisualReferenceInput> {
        /**
         * <p>
         * An array of screenshots that will be used as the baseline for visual monitoring in future runs of this
         * canary. If there is a screenshot that you don't want to be used for visual monitoring, remove it from this
         * array.
         * </p>
         * 
         * @param baseScreenshots
         *        An array of screenshots that will be used as the baseline for visual monitoring in future runs of this
         *        canary. If there is a screenshot that you don't want to be used for visual monitoring, remove it from
         *        this array.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder baseScreenshots(Collection<BaseScreenshot> baseScreenshots);

        /**
         * <p>
         * An array of screenshots that will be used as the baseline for visual monitoring in future runs of this
         * canary. If there is a screenshot that you don't want to be used for visual monitoring, remove it from this
         * array.
         * </p>
         * 
         * @param baseScreenshots
         *        An array of screenshots that will be used as the baseline for visual monitoring in future runs of this
         *        canary. If there is a screenshot that you don't want to be used for visual monitoring, remove it from
         *        this array.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder baseScreenshots(BaseScreenshot... baseScreenshots);

        /**
         * <p>
         * An array of screenshots that will be used as the baseline for visual monitoring in future runs of this
         * canary. If there is a screenshot that you don't want to be used for visual monitoring, remove it from this
         * array.
         * </p>
         * This is a convenience method that creates an instance of the {@link List<BaseScreenshot>.Builder} avoiding
         * the need to create one manually via {@link List<BaseScreenshot>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<BaseScreenshot>.Builder#build()} is called immediately and
         * its result is passed to {@link #baseScreenshots(List<BaseScreenshot>)}.
         * 
         * @param baseScreenshots
         *        a consumer that will call methods on {@link List<BaseScreenshot>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #baseScreenshots(List<BaseScreenshot>)
         */
        Builder baseScreenshots(Consumer<BaseScreenshot.Builder>... baseScreenshots);

        /**
         * <p>
         * Specifies which canary run to use the screenshots from as the baseline for future visual monitoring with this
         * canary. Valid values are <code>nextrun</code> to use the screenshots from the next run after this update is
         * made, <code>lastrun</code> to use the screenshots from the most recent run before this update was made, or
         * the value of <code>Id</code> in the <a
         * href="https://docs.aws.amazon.com/AmazonSynthetics/latest/APIReference/API_CanaryRun.html"> CanaryRun</a>
         * from any past run of this canary.
         * </p>
         * 
         * @param baseCanaryRunId
         *        Specifies which canary run to use the screenshots from as the baseline for future visual monitoring
         *        with this canary. Valid values are <code>nextrun</code> to use the screenshots from the next run after
         *        this update is made, <code>lastrun</code> to use the screenshots from the most recent run before this
         *        update was made, or the value of <code>Id</code> in the <a
         *        href="https://docs.aws.amazon.com/AmazonSynthetics/latest/APIReference/API_CanaryRun.html">
         *        CanaryRun</a> from any past run of this canary.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder baseCanaryRunId(String baseCanaryRunId);
    }

    static final class BuilderImpl implements Builder {
        private List<BaseScreenshot> baseScreenshots = DefaultSdkAutoConstructList.getInstance();

        private String baseCanaryRunId;

        private BuilderImpl() {
        }

        private BuilderImpl(VisualReferenceInput model) {
            baseScreenshots(model.baseScreenshots);
            baseCanaryRunId(model.baseCanaryRunId);
        }

        public final List<BaseScreenshot.Builder> getBaseScreenshots() {
            List<BaseScreenshot.Builder> result = BaseScreenshotsCopier.copyToBuilder(this.baseScreenshots);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setBaseScreenshots(Collection<BaseScreenshot.BuilderImpl> baseScreenshots) {
            this.baseScreenshots = BaseScreenshotsCopier.copyFromBuilder(baseScreenshots);
        }

        @Override
        public final Builder baseScreenshots(Collection<BaseScreenshot> baseScreenshots) {
            this.baseScreenshots = BaseScreenshotsCopier.copy(baseScreenshots);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder baseScreenshots(BaseScreenshot... baseScreenshots) {
            baseScreenshots(Arrays.asList(baseScreenshots));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder baseScreenshots(Consumer<BaseScreenshot.Builder>... baseScreenshots) {
            baseScreenshots(Stream.of(baseScreenshots).map(c -> BaseScreenshot.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getBaseCanaryRunId() {
            return baseCanaryRunId;
        }

        public final void setBaseCanaryRunId(String baseCanaryRunId) {
            this.baseCanaryRunId = baseCanaryRunId;
        }

        @Override
        public final Builder baseCanaryRunId(String baseCanaryRunId) {
            this.baseCanaryRunId = baseCanaryRunId;
            return this;
        }

        @Override
        public VisualReferenceInput build() {
            return new VisualReferenceInput(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
