/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.synthetics.model;

import java.io.Serializable;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkBytes;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Use this structure to input your script code for the canary. This structure contains the Lambda handler with the
 * location where the canary should start running the script. If the script is stored in an S3 bucket, the bucket name,
 * key, and version are also included. If the script was passed into the canary directly, the script code is contained
 * in the value of <code>Zipfile</code>.
 * </p>
 * <p>
 * If you are uploading your canary scripts with an Amazon S3 bucket, your zip file should include your script in a
 * certain folder structure.
 * </p>
 * <ul>
 * <li>
 * <p>
 * For Node.js canaries, the folder structure must be <code>nodejs/node_modules/<i>myCanaryFilename.js</i> </code> For
 * more information, see <a href=
 * "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries_WritingCanary_Nodejs.html#CloudWatch_Synthetics_Canaries_package"
 * >Packaging your Node.js canary files</a>
 * </p>
 * </li>
 * <li>
 * <p>
 * For Python canaries, the folder structure must be <code>python/<i>myCanaryFilename.p</i> </code> or
 * <code>python/<i>myFolder/myCanaryFilename.py</i> </code> For more information, see <a href=
 * "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries_WritingCanary_Python.html#CloudWatch_Synthetics_Canaries_WritingCanary_Python_package"
 * >Packaging your Python canary files</a>
 * </p>
 * </li>
 * </ul>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CanaryCodeInput implements SdkPojo, Serializable, ToCopyableBuilder<CanaryCodeInput.Builder, CanaryCodeInput> {
    private static final SdkField<String> S3_BUCKET_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("S3Bucket").getter(getter(CanaryCodeInput::s3Bucket)).setter(setter(Builder::s3Bucket))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("S3Bucket").build()).build();

    private static final SdkField<String> S3_KEY_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("S3Key")
            .getter(getter(CanaryCodeInput::s3Key)).setter(setter(Builder::s3Key))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("S3Key").build()).build();

    private static final SdkField<String> S3_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("S3Version").getter(getter(CanaryCodeInput::s3Version)).setter(setter(Builder::s3Version))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("S3Version").build()).build();

    private static final SdkField<SdkBytes> ZIP_FILE_FIELD = SdkField.<SdkBytes> builder(MarshallingType.SDK_BYTES)
            .memberName("ZipFile").getter(getter(CanaryCodeInput::zipFile)).setter(setter(Builder::zipFile))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ZipFile").build()).build();

    private static final SdkField<String> HANDLER_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Handler")
            .getter(getter(CanaryCodeInput::handler)).setter(setter(Builder::handler))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Handler").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(S3_BUCKET_FIELD, S3_KEY_FIELD,
            S3_VERSION_FIELD, ZIP_FILE_FIELD, HANDLER_FIELD));

    private static final long serialVersionUID = 1L;

    private final String s3Bucket;

    private final String s3Key;

    private final String s3Version;

    private final SdkBytes zipFile;

    private final String handler;

    private CanaryCodeInput(BuilderImpl builder) {
        this.s3Bucket = builder.s3Bucket;
        this.s3Key = builder.s3Key;
        this.s3Version = builder.s3Version;
        this.zipFile = builder.zipFile;
        this.handler = builder.handler;
    }

    /**
     * <p>
     * If your canary script is located in S3, specify the bucket name here. Do not include <code>s3://</code> as the
     * start of the bucket name.
     * </p>
     * 
     * @return If your canary script is located in S3, specify the bucket name here. Do not include <code>s3://</code>
     *         as the start of the bucket name.
     */
    public final String s3Bucket() {
        return s3Bucket;
    }

    /**
     * <p>
     * The S3 key of your script. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingObjects.html">Working with Amazon S3 Objects</a>.
     * </p>
     * 
     * @return The S3 key of your script. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingObjects.html">Working with Amazon S3
     *         Objects</a>.
     */
    public final String s3Key() {
        return s3Key;
    }

    /**
     * <p>
     * The S3 version ID of your script.
     * </p>
     * 
     * @return The S3 version ID of your script.
     */
    public final String s3Version() {
        return s3Version;
    }

    /**
     * <p>
     * If you input your canary script directly into the canary instead of referring to an S3 location, the value of
     * this parameter is the base64-encoded contents of the .zip file that contains the script. It must be smaller than
     * 225 Kb.
     * </p>
     * <p>
     * For large canary scripts, we recommend that you use an S3 location instead of inputting it directly with this
     * parameter.
     * </p>
     * 
     * @return If you input your canary script directly into the canary instead of referring to an S3 location, the
     *         value of this parameter is the base64-encoded contents of the .zip file that contains the script. It must
     *         be smaller than 225 Kb.</p>
     *         <p>
     *         For large canary scripts, we recommend that you use an S3 location instead of inputting it directly with
     *         this parameter.
     */
    public final SdkBytes zipFile() {
        return zipFile;
    }

    /**
     * <p>
     * The entry point to use for the source code when running the canary. For canaries that use the
     * <code>syn-python-selenium-1.0</code> runtime or a <code>syn-nodejs.puppeteer</code> runtime earlier than
     * <code>syn-nodejs.puppeteer-3.4</code>, the handler must be specified as <code> <i>fileName</i>.handler</code>.
     * For <code>syn-python-selenium-1.1</code>, <code>syn-nodejs.puppeteer-3.4</code>, and later runtimes, the handler
     * can be specified as <code> <i>fileName</i>.<i>functionName</i> </code>, or you can specify a folder where canary
     * scripts reside as <code> <i>folder</i>/<i>fileName</i>.<i>functionName</i> </code>.
     * </p>
     * 
     * @return The entry point to use for the source code when running the canary. For canaries that use the
     *         <code>syn-python-selenium-1.0</code> runtime or a <code>syn-nodejs.puppeteer</code> runtime earlier than
     *         <code>syn-nodejs.puppeteer-3.4</code>, the handler must be specified as
     *         <code> <i>fileName</i>.handler</code>. For <code>syn-python-selenium-1.1</code>,
     *         <code>syn-nodejs.puppeteer-3.4</code>, and later runtimes, the handler can be specified as
     *         <code> <i>fileName</i>.<i>functionName</i> </code>, or you can specify a folder where canary scripts
     *         reside as <code> <i>folder</i>/<i>fileName</i>.<i>functionName</i> </code>.
     */
    public final String handler() {
        return handler;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(s3Bucket());
        hashCode = 31 * hashCode + Objects.hashCode(s3Key());
        hashCode = 31 * hashCode + Objects.hashCode(s3Version());
        hashCode = 31 * hashCode + Objects.hashCode(zipFile());
        hashCode = 31 * hashCode + Objects.hashCode(handler());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CanaryCodeInput)) {
            return false;
        }
        CanaryCodeInput other = (CanaryCodeInput) obj;
        return Objects.equals(s3Bucket(), other.s3Bucket()) && Objects.equals(s3Key(), other.s3Key())
                && Objects.equals(s3Version(), other.s3Version()) && Objects.equals(zipFile(), other.zipFile())
                && Objects.equals(handler(), other.handler());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CanaryCodeInput").add("S3Bucket", s3Bucket()).add("S3Key", s3Key())
                .add("S3Version", s3Version()).add("ZipFile", zipFile()).add("Handler", handler()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "S3Bucket":
            return Optional.ofNullable(clazz.cast(s3Bucket()));
        case "S3Key":
            return Optional.ofNullable(clazz.cast(s3Key()));
        case "S3Version":
            return Optional.ofNullable(clazz.cast(s3Version()));
        case "ZipFile":
            return Optional.ofNullable(clazz.cast(zipFile()));
        case "Handler":
            return Optional.ofNullable(clazz.cast(handler()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CanaryCodeInput, T> g) {
        return obj -> g.apply((CanaryCodeInput) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CanaryCodeInput> {
        /**
         * <p>
         * If your canary script is located in S3, specify the bucket name here. Do not include <code>s3://</code> as
         * the start of the bucket name.
         * </p>
         * 
         * @param s3Bucket
         *        If your canary script is located in S3, specify the bucket name here. Do not include
         *        <code>s3://</code> as the start of the bucket name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3Bucket(String s3Bucket);

        /**
         * <p>
         * The S3 key of your script. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingObjects.html">Working with Amazon S3 Objects</a>.
         * </p>
         * 
         * @param s3Key
         *        The S3 key of your script. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingObjects.html">Working with Amazon S3
         *        Objects</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3Key(String s3Key);

        /**
         * <p>
         * The S3 version ID of your script.
         * </p>
         * 
         * @param s3Version
         *        The S3 version ID of your script.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3Version(String s3Version);

        /**
         * <p>
         * If you input your canary script directly into the canary instead of referring to an S3 location, the value of
         * this parameter is the base64-encoded contents of the .zip file that contains the script. It must be smaller
         * than 225 Kb.
         * </p>
         * <p>
         * For large canary scripts, we recommend that you use an S3 location instead of inputting it directly with this
         * parameter.
         * </p>
         * 
         * @param zipFile
         *        If you input your canary script directly into the canary instead of referring to an S3 location, the
         *        value of this parameter is the base64-encoded contents of the .zip file that contains the script. It
         *        must be smaller than 225 Kb.</p>
         *        <p>
         *        For large canary scripts, we recommend that you use an S3 location instead of inputting it directly
         *        with this parameter.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder zipFile(SdkBytes zipFile);

        /**
         * <p>
         * The entry point to use for the source code when running the canary. For canaries that use the
         * <code>syn-python-selenium-1.0</code> runtime or a <code>syn-nodejs.puppeteer</code> runtime earlier than
         * <code>syn-nodejs.puppeteer-3.4</code>, the handler must be specified as <code> <i>fileName</i>.handler</code>
         * . For <code>syn-python-selenium-1.1</code>, <code>syn-nodejs.puppeteer-3.4</code>, and later runtimes, the
         * handler can be specified as <code> <i>fileName</i>.<i>functionName</i> </code>, or you can specify a folder
         * where canary scripts reside as <code> <i>folder</i>/<i>fileName</i>.<i>functionName</i> </code>.
         * </p>
         * 
         * @param handler
         *        The entry point to use for the source code when running the canary. For canaries that use the
         *        <code>syn-python-selenium-1.0</code> runtime or a <code>syn-nodejs.puppeteer</code> runtime earlier
         *        than <code>syn-nodejs.puppeteer-3.4</code>, the handler must be specified as
         *        <code> <i>fileName</i>.handler</code>. For <code>syn-python-selenium-1.1</code>,
         *        <code>syn-nodejs.puppeteer-3.4</code>, and later runtimes, the handler can be specified as
         *        <code> <i>fileName</i>.<i>functionName</i> </code>, or you can specify a folder where canary scripts
         *        reside as <code> <i>folder</i>/<i>fileName</i>.<i>functionName</i> </code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder handler(String handler);
    }

    static final class BuilderImpl implements Builder {
        private String s3Bucket;

        private String s3Key;

        private String s3Version;

        private SdkBytes zipFile;

        private String handler;

        private BuilderImpl() {
        }

        private BuilderImpl(CanaryCodeInput model) {
            s3Bucket(model.s3Bucket);
            s3Key(model.s3Key);
            s3Version(model.s3Version);
            zipFile(model.zipFile);
            handler(model.handler);
        }

        public final String getS3Bucket() {
            return s3Bucket;
        }

        public final void setS3Bucket(String s3Bucket) {
            this.s3Bucket = s3Bucket;
        }

        @Override
        public final Builder s3Bucket(String s3Bucket) {
            this.s3Bucket = s3Bucket;
            return this;
        }

        public final String getS3Key() {
            return s3Key;
        }

        public final void setS3Key(String s3Key) {
            this.s3Key = s3Key;
        }

        @Override
        public final Builder s3Key(String s3Key) {
            this.s3Key = s3Key;
            return this;
        }

        public final String getS3Version() {
            return s3Version;
        }

        public final void setS3Version(String s3Version) {
            this.s3Version = s3Version;
        }

        @Override
        public final Builder s3Version(String s3Version) {
            this.s3Version = s3Version;
            return this;
        }

        public final ByteBuffer getZipFile() {
            return zipFile == null ? null : zipFile.asByteBuffer();
        }

        public final void setZipFile(ByteBuffer zipFile) {
            zipFile(zipFile == null ? null : SdkBytes.fromByteBuffer(zipFile));
        }

        @Override
        public final Builder zipFile(SdkBytes zipFile) {
            this.zipFile = zipFile;
            return this;
        }

        public final String getHandler() {
            return handler;
        }

        public final void setHandler(String handler) {
            this.handler = handler;
        }

        @Override
        public final Builder handler(String handler) {
            this.handler = handler;
            return this;
        }

        @Override
        public CanaryCodeInput build() {
            return new CanaryCodeInput(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
