/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.synthetics.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A structure that contains the configuration of encryption-at-rest settings for canary artifacts that the canary
 * uploads to Amazon S3.
 * </p>
 * <p>
 * For more information, see <a href=
 * "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_artifact_encryption.html"
 * >Encrypting canary artifacts</a>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class S3EncryptionConfig implements SdkPojo, Serializable,
        ToCopyableBuilder<S3EncryptionConfig.Builder, S3EncryptionConfig> {
    private static final SdkField<String> ENCRYPTION_MODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EncryptionMode").getter(getter(S3EncryptionConfig::encryptionModeAsString))
            .setter(setter(Builder::encryptionMode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EncryptionMode").build()).build();

    private static final SdkField<String> KMS_KEY_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("KmsKeyArn").getter(getter(S3EncryptionConfig::kmsKeyArn)).setter(setter(Builder::kmsKeyArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KmsKeyArn").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ENCRYPTION_MODE_FIELD,
            KMS_KEY_ARN_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String encryptionMode;

    private final String kmsKeyArn;

    private S3EncryptionConfig(BuilderImpl builder) {
        this.encryptionMode = builder.encryptionMode;
        this.kmsKeyArn = builder.kmsKeyArn;
    }

    /**
     * <p>
     * The encryption method to use for artifacts created by this canary. Specify <code>SSE_S3</code> to use server-side
     * encryption (SSE) with an Amazon S3-managed key. Specify <code>SSE-KMS</code> to use server-side encryption with a
     * customer-managed KMS key.
     * </p>
     * <p>
     * If you omit this parameter, an Amazon Web Services-managed KMS key is used.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #encryptionMode}
     * will return {@link EncryptionMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #encryptionModeAsString}.
     * </p>
     * 
     * @return The encryption method to use for artifacts created by this canary. Specify <code>SSE_S3</code> to use
     *         server-side encryption (SSE) with an Amazon S3-managed key. Specify <code>SSE-KMS</code> to use
     *         server-side encryption with a customer-managed KMS key.</p>
     *         <p>
     *         If you omit this parameter, an Amazon Web Services-managed KMS key is used.
     * @see EncryptionMode
     */
    public final EncryptionMode encryptionMode() {
        return EncryptionMode.fromValue(encryptionMode);
    }

    /**
     * <p>
     * The encryption method to use for artifacts created by this canary. Specify <code>SSE_S3</code> to use server-side
     * encryption (SSE) with an Amazon S3-managed key. Specify <code>SSE-KMS</code> to use server-side encryption with a
     * customer-managed KMS key.
     * </p>
     * <p>
     * If you omit this parameter, an Amazon Web Services-managed KMS key is used.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #encryptionMode}
     * will return {@link EncryptionMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #encryptionModeAsString}.
     * </p>
     * 
     * @return The encryption method to use for artifacts created by this canary. Specify <code>SSE_S3</code> to use
     *         server-side encryption (SSE) with an Amazon S3-managed key. Specify <code>SSE-KMS</code> to use
     *         server-side encryption with a customer-managed KMS key.</p>
     *         <p>
     *         If you omit this parameter, an Amazon Web Services-managed KMS key is used.
     * @see EncryptionMode
     */
    public final String encryptionModeAsString() {
        return encryptionMode;
    }

    /**
     * <p>
     * The ARN of the customer-managed KMS key to use, if you specify <code>SSE-KMS</code> for
     * <code>EncryptionMode</code>
     * </p>
     * 
     * @return The ARN of the customer-managed KMS key to use, if you specify <code>SSE-KMS</code> for
     *         <code>EncryptionMode</code>
     */
    public final String kmsKeyArn() {
        return kmsKeyArn;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(encryptionModeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(kmsKeyArn());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof S3EncryptionConfig)) {
            return false;
        }
        S3EncryptionConfig other = (S3EncryptionConfig) obj;
        return Objects.equals(encryptionModeAsString(), other.encryptionModeAsString())
                && Objects.equals(kmsKeyArn(), other.kmsKeyArn());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("S3EncryptionConfig").add("EncryptionMode", encryptionModeAsString())
                .add("KmsKeyArn", kmsKeyArn()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "EncryptionMode":
            return Optional.ofNullable(clazz.cast(encryptionModeAsString()));
        case "KmsKeyArn":
            return Optional.ofNullable(clazz.cast(kmsKeyArn()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("EncryptionMode", ENCRYPTION_MODE_FIELD);
        map.put("KmsKeyArn", KMS_KEY_ARN_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<S3EncryptionConfig, T> g) {
        return obj -> g.apply((S3EncryptionConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, S3EncryptionConfig> {
        /**
         * <p>
         * The encryption method to use for artifacts created by this canary. Specify <code>SSE_S3</code> to use
         * server-side encryption (SSE) with an Amazon S3-managed key. Specify <code>SSE-KMS</code> to use server-side
         * encryption with a customer-managed KMS key.
         * </p>
         * <p>
         * If you omit this parameter, an Amazon Web Services-managed KMS key is used.
         * </p>
         * 
         * @param encryptionMode
         *        The encryption method to use for artifacts created by this canary. Specify <code>SSE_S3</code> to use
         *        server-side encryption (SSE) with an Amazon S3-managed key. Specify <code>SSE-KMS</code> to use
         *        server-side encryption with a customer-managed KMS key.</p>
         *        <p>
         *        If you omit this parameter, an Amazon Web Services-managed KMS key is used.
         * @see EncryptionMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EncryptionMode
         */
        Builder encryptionMode(String encryptionMode);

        /**
         * <p>
         * The encryption method to use for artifacts created by this canary. Specify <code>SSE_S3</code> to use
         * server-side encryption (SSE) with an Amazon S3-managed key. Specify <code>SSE-KMS</code> to use server-side
         * encryption with a customer-managed KMS key.
         * </p>
         * <p>
         * If you omit this parameter, an Amazon Web Services-managed KMS key is used.
         * </p>
         * 
         * @param encryptionMode
         *        The encryption method to use for artifacts created by this canary. Specify <code>SSE_S3</code> to use
         *        server-side encryption (SSE) with an Amazon S3-managed key. Specify <code>SSE-KMS</code> to use
         *        server-side encryption with a customer-managed KMS key.</p>
         *        <p>
         *        If you omit this parameter, an Amazon Web Services-managed KMS key is used.
         * @see EncryptionMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EncryptionMode
         */
        Builder encryptionMode(EncryptionMode encryptionMode);

        /**
         * <p>
         * The ARN of the customer-managed KMS key to use, if you specify <code>SSE-KMS</code> for
         * <code>EncryptionMode</code>
         * </p>
         * 
         * @param kmsKeyArn
         *        The ARN of the customer-managed KMS key to use, if you specify <code>SSE-KMS</code> for
         *        <code>EncryptionMode</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsKeyArn(String kmsKeyArn);
    }

    static final class BuilderImpl implements Builder {
        private String encryptionMode;

        private String kmsKeyArn;

        private BuilderImpl() {
        }

        private BuilderImpl(S3EncryptionConfig model) {
            encryptionMode(model.encryptionMode);
            kmsKeyArn(model.kmsKeyArn);
        }

        public final String getEncryptionMode() {
            return encryptionMode;
        }

        public final void setEncryptionMode(String encryptionMode) {
            this.encryptionMode = encryptionMode;
        }

        @Override
        public final Builder encryptionMode(String encryptionMode) {
            this.encryptionMode = encryptionMode;
            return this;
        }

        @Override
        public final Builder encryptionMode(EncryptionMode encryptionMode) {
            this.encryptionMode(encryptionMode == null ? null : encryptionMode.toString());
            return this;
        }

        public final String getKmsKeyArn() {
            return kmsKeyArn;
        }

        public final void setKmsKeyArn(String kmsKeyArn) {
            this.kmsKeyArn = kmsKeyArn;
        }

        @Override
        public final Builder kmsKeyArn(String kmsKeyArn) {
            this.kmsKeyArn = kmsKeyArn;
            return this;
        }

        @Override
        public S3EncryptionConfig build() {
            return new S3EncryptionConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
