/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.taxsettings;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import java.util.function.Function;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.taxsettings.internal.ServiceVersionInfo;
import software.amazon.awssdk.services.taxsettings.internal.TaxSettingsServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.taxsettings.model.AccessDeniedException;
import software.amazon.awssdk.services.taxsettings.model.AttachmentUploadException;
import software.amazon.awssdk.services.taxsettings.model.BatchDeleteTaxRegistrationRequest;
import software.amazon.awssdk.services.taxsettings.model.BatchDeleteTaxRegistrationResponse;
import software.amazon.awssdk.services.taxsettings.model.BatchGetTaxExemptionsRequest;
import software.amazon.awssdk.services.taxsettings.model.BatchGetTaxExemptionsResponse;
import software.amazon.awssdk.services.taxsettings.model.BatchPutTaxRegistrationRequest;
import software.amazon.awssdk.services.taxsettings.model.BatchPutTaxRegistrationResponse;
import software.amazon.awssdk.services.taxsettings.model.CaseCreationLimitExceededException;
import software.amazon.awssdk.services.taxsettings.model.ConflictException;
import software.amazon.awssdk.services.taxsettings.model.DeleteSupplementalTaxRegistrationRequest;
import software.amazon.awssdk.services.taxsettings.model.DeleteSupplementalTaxRegistrationResponse;
import software.amazon.awssdk.services.taxsettings.model.DeleteTaxRegistrationRequest;
import software.amazon.awssdk.services.taxsettings.model.DeleteTaxRegistrationResponse;
import software.amazon.awssdk.services.taxsettings.model.GetTaxExemptionTypesRequest;
import software.amazon.awssdk.services.taxsettings.model.GetTaxExemptionTypesResponse;
import software.amazon.awssdk.services.taxsettings.model.GetTaxInheritanceRequest;
import software.amazon.awssdk.services.taxsettings.model.GetTaxInheritanceResponse;
import software.amazon.awssdk.services.taxsettings.model.GetTaxRegistrationDocumentRequest;
import software.amazon.awssdk.services.taxsettings.model.GetTaxRegistrationDocumentResponse;
import software.amazon.awssdk.services.taxsettings.model.GetTaxRegistrationRequest;
import software.amazon.awssdk.services.taxsettings.model.GetTaxRegistrationResponse;
import software.amazon.awssdk.services.taxsettings.model.InternalServerException;
import software.amazon.awssdk.services.taxsettings.model.ListSupplementalTaxRegistrationsRequest;
import software.amazon.awssdk.services.taxsettings.model.ListSupplementalTaxRegistrationsResponse;
import software.amazon.awssdk.services.taxsettings.model.ListTaxExemptionsRequest;
import software.amazon.awssdk.services.taxsettings.model.ListTaxExemptionsResponse;
import software.amazon.awssdk.services.taxsettings.model.ListTaxRegistrationsRequest;
import software.amazon.awssdk.services.taxsettings.model.ListTaxRegistrationsResponse;
import software.amazon.awssdk.services.taxsettings.model.PutSupplementalTaxRegistrationRequest;
import software.amazon.awssdk.services.taxsettings.model.PutSupplementalTaxRegistrationResponse;
import software.amazon.awssdk.services.taxsettings.model.PutTaxExemptionRequest;
import software.amazon.awssdk.services.taxsettings.model.PutTaxExemptionResponse;
import software.amazon.awssdk.services.taxsettings.model.PutTaxInheritanceRequest;
import software.amazon.awssdk.services.taxsettings.model.PutTaxInheritanceResponse;
import software.amazon.awssdk.services.taxsettings.model.PutTaxRegistrationRequest;
import software.amazon.awssdk.services.taxsettings.model.PutTaxRegistrationResponse;
import software.amazon.awssdk.services.taxsettings.model.ResourceNotFoundException;
import software.amazon.awssdk.services.taxsettings.model.TaxSettingsException;
import software.amazon.awssdk.services.taxsettings.model.ValidationException;
import software.amazon.awssdk.services.taxsettings.transform.BatchDeleteTaxRegistrationRequestMarshaller;
import software.amazon.awssdk.services.taxsettings.transform.BatchGetTaxExemptionsRequestMarshaller;
import software.amazon.awssdk.services.taxsettings.transform.BatchPutTaxRegistrationRequestMarshaller;
import software.amazon.awssdk.services.taxsettings.transform.DeleteSupplementalTaxRegistrationRequestMarshaller;
import software.amazon.awssdk.services.taxsettings.transform.DeleteTaxRegistrationRequestMarshaller;
import software.amazon.awssdk.services.taxsettings.transform.GetTaxExemptionTypesRequestMarshaller;
import software.amazon.awssdk.services.taxsettings.transform.GetTaxInheritanceRequestMarshaller;
import software.amazon.awssdk.services.taxsettings.transform.GetTaxRegistrationDocumentRequestMarshaller;
import software.amazon.awssdk.services.taxsettings.transform.GetTaxRegistrationRequestMarshaller;
import software.amazon.awssdk.services.taxsettings.transform.ListSupplementalTaxRegistrationsRequestMarshaller;
import software.amazon.awssdk.services.taxsettings.transform.ListTaxExemptionsRequestMarshaller;
import software.amazon.awssdk.services.taxsettings.transform.ListTaxRegistrationsRequestMarshaller;
import software.amazon.awssdk.services.taxsettings.transform.PutSupplementalTaxRegistrationRequestMarshaller;
import software.amazon.awssdk.services.taxsettings.transform.PutTaxExemptionRequestMarshaller;
import software.amazon.awssdk.services.taxsettings.transform.PutTaxInheritanceRequestMarshaller;
import software.amazon.awssdk.services.taxsettings.transform.PutTaxRegistrationRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link TaxSettingsAsyncClient}.
 *
 * @see TaxSettingsAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultTaxSettingsAsyncClient implements TaxSettingsAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultTaxSettingsAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultTaxSettingsAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this)
                .option(SdkClientOption.API_METADATA, "TaxSettings" + "#" + ServiceVersionInfo.VERSION).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Deletes tax registration for multiple accounts in batch. This can be used to delete tax registrations for up to
     * five accounts in one batch.
     * </p>
     * <note>
     * <p>
     * This API operation can't be used to delete your tax registration in Brazil. Use the <a
     * href="https://console.aws.amazon.com/billing/home#/paymentpreferences/paymentmethods">Payment preferences</a>
     * page in the Billing and Cost Management console instead.
     * </p>
     * </note>
     *
     * @param batchDeleteTaxRegistrationRequest
     * @return A Java Future containing the result of the BatchDeleteTaxRegistration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The exception when the input doesn't pass validation for at least one of the
     *         input parameters.</li>
     *         <li>ConflictException The exception when the input is creating conflict with the given state.</li>
     *         <li>InternalServerException The exception thrown when an unexpected error occurs when processing a
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TaxSettingsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TaxSettingsAsyncClient.BatchDeleteTaxRegistration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/taxsettings-2018-05-10/BatchDeleteTaxRegistration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchDeleteTaxRegistrationResponse> batchDeleteTaxRegistration(
            BatchDeleteTaxRegistrationRequest batchDeleteTaxRegistrationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchDeleteTaxRegistrationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchDeleteTaxRegistrationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "TaxSettings");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchDeleteTaxRegistration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchDeleteTaxRegistrationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchDeleteTaxRegistrationResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "CaseCreationLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CaseCreationLimitExceededException")
                            .httpStatusCode(413).exceptionBuilderSupplier(CaseCreationLimitExceededException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "AttachmentUploadException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AttachmentUploadException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AttachmentUploadException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<BatchDeleteTaxRegistrationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchDeleteTaxRegistrationRequest, BatchDeleteTaxRegistrationResponse>()
                            .withOperationName("BatchDeleteTaxRegistration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new BatchDeleteTaxRegistrationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(batchDeleteTaxRegistrationRequest));
            CompletableFuture<BatchDeleteTaxRegistrationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Get the active tax exemptions for a given list of accounts. The IAM action is <code>tax:GetExemptions</code>.
     * </p>
     *
     * @param batchGetTaxExemptionsRequest
     * @return A Java Future containing the result of the BatchGetTaxExemptions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The exception when the input doesn't pass validation for at least one of the
     *         input parameters.</li>
     *         <li>ResourceNotFoundException The exception thrown when the input doesn't have a resource associated to
     *         it.</li>
     *         <li>InternalServerException The exception thrown when an unexpected error occurs when processing a
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TaxSettingsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TaxSettingsAsyncClient.BatchGetTaxExemptions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/taxsettings-2018-05-10/BatchGetTaxExemptions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchGetTaxExemptionsResponse> batchGetTaxExemptions(
            BatchGetTaxExemptionsRequest batchGetTaxExemptionsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchGetTaxExemptionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchGetTaxExemptionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "TaxSettings");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchGetTaxExemptions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchGetTaxExemptionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchGetTaxExemptionsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "CaseCreationLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CaseCreationLimitExceededException")
                            .httpStatusCode(413).exceptionBuilderSupplier(CaseCreationLimitExceededException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "AttachmentUploadException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AttachmentUploadException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AttachmentUploadException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<BatchGetTaxExemptionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchGetTaxExemptionsRequest, BatchGetTaxExemptionsResponse>()
                            .withOperationName("BatchGetTaxExemptions").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new BatchGetTaxExemptionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(batchGetTaxExemptionsRequest));
            CompletableFuture<BatchGetTaxExemptionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds or updates tax registration for multiple accounts in batch. This can be used to add or update tax
     * registrations for up to five accounts in one batch. You can't set a TRN if there's a pending TRN. You'll need to
     * delete the pending TRN first.
     * </p>
     * <p>
     * To call this API operation for specific countries, see the following country-specific requirements.
     * </p>
     * <p>
     * <b>Bangladesh</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * You must specify the tax registration certificate document in the <code>taxRegistrationDocuments</code> field of
     * the <code>VerificationDetails</code> object.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>Brazil</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * You must complete the tax registration process in the <a
     * href="https://console.aws.amazon.com/billing/home#/paymentpreferences/paymentmethods">Payment preferences</a>
     * page in the Billing and Cost Management console. After your TRN and billing address are verified, you can call
     * this API operation.
     * </p>
     * </li>
     * <li>
     * <p>
     * For Amazon Web Services accounts created through Organizations, you can call this API operation when you don't
     * have a billing address.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>Georgia</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * The valid <code>personType</code> values are <code>Physical Person</code> and <code>Business</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>Indonesia</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>PutTaxRegistration</code>: The use of this operation to submit tax information is subject to the <a
     * href="http://aws.amazon.com/service-terms/">Amazon Web Services service terms</a>. By submitting, you’re
     * providing consent for Amazon Web Services to validate NIK, NPWP, and NITKU data, provided by you with the
     * Directorate General of Taxes of Indonesia in accordance with the Minister of Finance Regulation (PMK) Number
     * 112/PMK.03/2022.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>BatchPutTaxRegistration</code>: The use of this operation to submit tax information is subject to the <a
     * href="http://aws.amazon.com/service-terms/">Amazon Web Services service terms</a>. By submitting, you’re
     * providing consent for Amazon Web Services to validate NIK, NPWP, and NITKU data, provided by you with the
     * Directorate General of Taxes of Indonesia in accordance with the Minister of Finance Regulation (PMK) Number
     * 112/PMK.03/2022, through our third-party partner PT Achilles Advanced Management (OnlinePajak).
     * </p>
     * </li>
     * <li>
     * <p>
     * You must specify the <code>taxRegistrationNumberType</code> in the <code>indonesiaAdditionalInfo</code> field of
     * the <code>additionalTaxInformation</code> object.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you specify <code>decisionNumber</code>, you must specify the <code>ppnExceptionDesignationCode</code> in the
     * <code>indonesiaAdditionalInfo</code> field of the <code>additionalTaxInformation</code> object. If the
     * <code>taxRegistrationNumberType</code> is set to NPWP or NITKU, valid values for
     * <code>ppnExceptionDesignationCode</code> are either <code>01</code>, <code>02</code>, <code>03</code>,
     * <code>07</code>, or <code>08</code>.
     * </p>
     * <p>
     * For other <code>taxRegistrationNumberType</code> values, <code>ppnExceptionDesignationCode</code> must be either
     * <code>01</code>, <code>07</code>, or <code>08</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * If <code>ppnExceptionDesignationCode</code> is <code>07</code>, you must specify the <code>decisionNumber</code>
     * in the <code>indonesiaAdditionalInfo</code> field of the <code>additionalTaxInformation</code> object.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>Kenya</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * You must specify the <code>personType</code> in the <code>kenyaAdditionalInfo</code> field of the
     * <code>additionalTaxInformation</code> object.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the <code>personType</code> is <code>Physical Person</code>, you must specify the tax registration certificate
     * document in the <code>taxRegistrationDocuments</code> field of the <code>VerificationDetails</code> object.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>Malaysia</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * The sector valid values are <code>Business</code> and <code>Individual</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>RegistrationType</code> valid values are <code>NRIC</code> for individual, and TIN and sales and service
     * tax (SST) for Business.
     * </p>
     * </li>
     * <li>
     * <p>
     * For individual, you can specify the <code>taxInformationNumber</code> in <code>MalaysiaAdditionalInfo</code> with
     * NRIC type, and a valid <code>MyKad</code> or NRIC number.
     * </p>
     * </li>
     * <li>
     * <p>
     * For business, you must specify a <code>businessRegistrationNumber</code> in <code>MalaysiaAdditionalInfo</code>
     * with a TIN type and tax identification number.
     * </p>
     * </li>
     * <li>
     * <p>
     * For business resellers, you must specify a <code>businessRegistrationNumber</code> and
     * <code>taxInformationNumber</code> in <code>MalaysiaAdditionalInfo</code> with a sales and service tax (SST) type
     * and a valid SST number.
     * </p>
     * </li>
     * <li>
     * <p>
     * For business resellers with service codes, you must specify <code>businessRegistrationNumber</code>,
     * <code>taxInformationNumber</code>, and distinct <code>serviceTaxCodes</code> in
     * <code>MalaysiaAdditionalInfo</code> with a SST type and valid sales and service tax (SST) number. By using this
     * API operation, Amazon Web Services registers your self-declaration that you’re an authorized business reseller
     * registered with the Royal Malaysia Customs Department (RMCD), and have a valid SST number.
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon Web Services reserves the right to seek additional information and/or take other actions to support your
     * self-declaration as appropriate.
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon Web Services is currently registered under the following service tax codes. You must include at least one
     * of the service tax codes in the service tax code strings to declare yourself as an authorized registered business
     * reseller.
     * </p>
     * <p>
     * Taxable service and service tax codes:
     * </p>
     * <p>
     * Consultancy - 9907061674
     * </p>
     * <p>
     * Training or coaching service - 9907071685
     * </p>
     * <p>
     * IT service - 9907101676
     * </p>
     * <p>
     * Digital services and electronic medium - 9907121690
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>Nepal</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * The sector valid values are <code>Business</code> and <code>Individual</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>Saudi Arabia</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * For <code>address</code>, you must specify <code>addressLine3</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>South Korea</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * You must specify the <code>certifiedEmailId</code> and <code>legalName</code> in the
     * <code>TaxRegistrationEntry</code> object. Use Korean characters for <code>legalName</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * You must specify the <code>businessRepresentativeName</code>, <code>itemOfBusiness</code>, and
     * <code>lineOfBusiness</code> in the <code>southKoreaAdditionalInfo</code> field of the
     * <code>additionalTaxInformation</code> object. Use Korean characters for these fields.
     * </p>
     * </li>
     * <li>
     * <p>
     * You must specify the tax registration certificate document in the <code>taxRegistrationDocuments</code> field of
     * the <code>VerificationDetails</code> object.
     * </p>
     * </li>
     * <li>
     * <p>
     * For the <code>address</code> object, use Korean characters for <code>addressLine1</code>,
     * <code>addressLine2</code> <code>city</code>, <code>postalCode</code>, and <code>stateOrRegion</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>Spain</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * You must specify the <code>registrationType</code> in the <code>spainAdditionalInfo</code> field of the
     * <code>additionalTaxInformation</code> object.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the <code>registrationType</code> is <code>Local</code>, you must specify the tax registration certificate
     * document in the <code>taxRegistrationDocuments</code> field of the <code>VerificationDetails</code> object.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>Turkey</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * You must specify the <code>sector</code> in the <code>taxRegistrationEntry</code> object.
     * </p>
     * </li>
     * <li>
     * <p>
     * If your <code>sector</code> is <code>Business</code>, <code>Individual</code>, or <code>Government</code>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Specify the <code>taxOffice</code>. If your <code>sector</code> is <code>Individual</code>, don't enter this
     * value.
     * </p>
     * </li>
     * <li>
     * <p>
     * (Optional) Specify the <code>kepEmailId</code>. If your <code>sector</code> is <code>Individual</code>, don't
     * enter this value.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Note:</b> In the <b>Tax Settings</b> page of the Billing console, <code>Government</code> appears as <b>Public
     * institutions</b>
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * If your <code>sector</code> is <code>Business</code> and you're subject to KDV tax, you must specify your
     * industry in the <code>industries</code> field.
     * </p>
     * </li>
     * <li>
     * <p>
     * For <code>address</code>, you must specify <code>districtOrCounty</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>Ukraine</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * The sector valid values are <code>Business</code> and <code>Individual</code>.
     * </p>
     * </li>
     * </ul>
     *
     * @param batchPutTaxRegistrationRequest
     * @return A Java Future containing the result of the BatchPutTaxRegistration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The exception when the input doesn't pass validation for at least one of the
     *         input parameters.</li>
     *         <li>ConflictException The exception when the input is creating conflict with the given state.</li>
     *         <li>InternalServerException The exception thrown when an unexpected error occurs when processing a
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TaxSettingsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TaxSettingsAsyncClient.BatchPutTaxRegistration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/taxsettings-2018-05-10/BatchPutTaxRegistration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchPutTaxRegistrationResponse> batchPutTaxRegistration(
            BatchPutTaxRegistrationRequest batchPutTaxRegistrationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchPutTaxRegistrationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchPutTaxRegistrationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "TaxSettings");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchPutTaxRegistration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchPutTaxRegistrationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchPutTaxRegistrationResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "CaseCreationLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CaseCreationLimitExceededException")
                            .httpStatusCode(413).exceptionBuilderSupplier(CaseCreationLimitExceededException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "AttachmentUploadException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AttachmentUploadException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AttachmentUploadException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<BatchPutTaxRegistrationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchPutTaxRegistrationRequest, BatchPutTaxRegistrationResponse>()
                            .withOperationName("BatchPutTaxRegistration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new BatchPutTaxRegistrationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(batchPutTaxRegistrationRequest));
            CompletableFuture<BatchPutTaxRegistrationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a supplemental tax registration for a single account.
     * </p>
     *
     * @param deleteSupplementalTaxRegistrationRequest
     * @return A Java Future containing the result of the DeleteSupplementalTaxRegistration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The exception when the input doesn't pass validation for at least one of the
     *         input parameters.</li>
     *         <li>ConflictException The exception when the input is creating conflict with the given state.</li>
     *         <li>ResourceNotFoundException The exception thrown when the input doesn't have a resource associated to
     *         it.</li>
     *         <li>InternalServerException The exception thrown when an unexpected error occurs when processing a
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TaxSettingsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TaxSettingsAsyncClient.DeleteSupplementalTaxRegistration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/taxsettings-2018-05-10/DeleteSupplementalTaxRegistration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteSupplementalTaxRegistrationResponse> deleteSupplementalTaxRegistration(
            DeleteSupplementalTaxRegistrationRequest deleteSupplementalTaxRegistrationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteSupplementalTaxRegistrationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteSupplementalTaxRegistrationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "TaxSettings");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSupplementalTaxRegistration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteSupplementalTaxRegistrationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DeleteSupplementalTaxRegistrationResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "CaseCreationLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CaseCreationLimitExceededException")
                            .httpStatusCode(413).exceptionBuilderSupplier(CaseCreationLimitExceededException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "AttachmentUploadException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AttachmentUploadException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AttachmentUploadException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DeleteSupplementalTaxRegistrationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteSupplementalTaxRegistrationRequest, DeleteSupplementalTaxRegistrationResponse>()
                            .withOperationName("DeleteSupplementalTaxRegistration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteSupplementalTaxRegistrationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteSupplementalTaxRegistrationRequest));
            CompletableFuture<DeleteSupplementalTaxRegistrationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes tax registration for a single account.
     * </p>
     * <note>
     * <p>
     * This API operation can't be used to delete your tax registration in Brazil. Use the <a
     * href="https://console.aws.amazon.com/billing/home#/paymentpreferences/paymentmethods">Payment preferences</a>
     * page in the Billing and Cost Management console instead.
     * </p>
     * </note>
     *
     * @param deleteTaxRegistrationRequest
     * @return A Java Future containing the result of the DeleteTaxRegistration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The exception when the input doesn't pass validation for at least one of the
     *         input parameters.</li>
     *         <li>ConflictException The exception when the input is creating conflict with the given state.</li>
     *         <li>ResourceNotFoundException The exception thrown when the input doesn't have a resource associated to
     *         it.</li>
     *         <li>InternalServerException The exception thrown when an unexpected error occurs when processing a
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TaxSettingsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TaxSettingsAsyncClient.DeleteTaxRegistration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/taxsettings-2018-05-10/DeleteTaxRegistration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteTaxRegistrationResponse> deleteTaxRegistration(
            DeleteTaxRegistrationRequest deleteTaxRegistrationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteTaxRegistrationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteTaxRegistrationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "TaxSettings");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteTaxRegistration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteTaxRegistrationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteTaxRegistrationResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "CaseCreationLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CaseCreationLimitExceededException")
                            .httpStatusCode(413).exceptionBuilderSupplier(CaseCreationLimitExceededException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "AttachmentUploadException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AttachmentUploadException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AttachmentUploadException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DeleteTaxRegistrationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteTaxRegistrationRequest, DeleteTaxRegistrationResponse>()
                            .withOperationName("DeleteTaxRegistration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteTaxRegistrationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteTaxRegistrationRequest));
            CompletableFuture<DeleteTaxRegistrationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Get supported tax exemption types. The IAM action is <code>tax:GetExemptions</code>.
     * </p>
     *
     * @param getTaxExemptionTypesRequest
     * @return A Java Future containing the result of the GetTaxExemptionTypes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The exception when the input doesn't pass validation for at least one of the
     *         input parameters.</li>
     *         <li>ResourceNotFoundException The exception thrown when the input doesn't have a resource associated to
     *         it.</li>
     *         <li>InternalServerException The exception thrown when an unexpected error occurs when processing a
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TaxSettingsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TaxSettingsAsyncClient.GetTaxExemptionTypes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/taxsettings-2018-05-10/GetTaxExemptionTypes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetTaxExemptionTypesResponse> getTaxExemptionTypes(
            GetTaxExemptionTypesRequest getTaxExemptionTypesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getTaxExemptionTypesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getTaxExemptionTypesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "TaxSettings");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTaxExemptionTypes");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetTaxExemptionTypesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetTaxExemptionTypesResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "CaseCreationLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CaseCreationLimitExceededException")
                            .httpStatusCode(413).exceptionBuilderSupplier(CaseCreationLimitExceededException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "AttachmentUploadException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AttachmentUploadException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AttachmentUploadException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetTaxExemptionTypesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetTaxExemptionTypesRequest, GetTaxExemptionTypesResponse>()
                            .withOperationName("GetTaxExemptionTypes").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetTaxExemptionTypesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getTaxExemptionTypesRequest));
            CompletableFuture<GetTaxExemptionTypesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The get account tax inheritance status.
     * </p>
     *
     * @param getTaxInheritanceRequest
     * @return A Java Future containing the result of the GetTaxInheritance operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The exception when the input doesn't pass validation for at least one of the
     *         input parameters.</li>
     *         <li>ResourceNotFoundException The exception thrown when the input doesn't have a resource associated to
     *         it.</li>
     *         <li>InternalServerException The exception thrown when an unexpected error occurs when processing a
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TaxSettingsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TaxSettingsAsyncClient.GetTaxInheritance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/taxsettings-2018-05-10/GetTaxInheritance" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetTaxInheritanceResponse> getTaxInheritance(GetTaxInheritanceRequest getTaxInheritanceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getTaxInheritanceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getTaxInheritanceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "TaxSettings");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTaxInheritance");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetTaxInheritanceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetTaxInheritanceResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "CaseCreationLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CaseCreationLimitExceededException")
                            .httpStatusCode(413).exceptionBuilderSupplier(CaseCreationLimitExceededException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "AttachmentUploadException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AttachmentUploadException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AttachmentUploadException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetTaxInheritanceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetTaxInheritanceRequest, GetTaxInheritanceResponse>()
                            .withOperationName("GetTaxInheritance").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetTaxInheritanceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getTaxInheritanceRequest));
            CompletableFuture<GetTaxInheritanceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves tax registration for a single account.
     * </p>
     *
     * @param getTaxRegistrationRequest
     * @return A Java Future containing the result of the GetTaxRegistration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The exception when the input doesn't pass validation for at least one of the
     *         input parameters.</li>
     *         <li>ResourceNotFoundException The exception thrown when the input doesn't have a resource associated to
     *         it.</li>
     *         <li>InternalServerException The exception thrown when an unexpected error occurs when processing a
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TaxSettingsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TaxSettingsAsyncClient.GetTaxRegistration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/taxsettings-2018-05-10/GetTaxRegistration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetTaxRegistrationResponse> getTaxRegistration(GetTaxRegistrationRequest getTaxRegistrationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getTaxRegistrationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getTaxRegistrationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "TaxSettings");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTaxRegistration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetTaxRegistrationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetTaxRegistrationResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "CaseCreationLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CaseCreationLimitExceededException")
                            .httpStatusCode(413).exceptionBuilderSupplier(CaseCreationLimitExceededException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "AttachmentUploadException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AttachmentUploadException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AttachmentUploadException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetTaxRegistrationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetTaxRegistrationRequest, GetTaxRegistrationResponse>()
                            .withOperationName("GetTaxRegistration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetTaxRegistrationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getTaxRegistrationRequest));
            CompletableFuture<GetTaxRegistrationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Downloads your tax documents to the Amazon S3 bucket that you specify in your request.
     * </p>
     *
     * @param getTaxRegistrationDocumentRequest
     * @return A Java Future containing the result of the GetTaxRegistrationDocument operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The exception when the input doesn't pass validation for at least one of the
     *         input parameters.</li>
     *         <li>InternalServerException The exception thrown when an unexpected error occurs when processing a
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TaxSettingsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TaxSettingsAsyncClient.GetTaxRegistrationDocument
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/taxsettings-2018-05-10/GetTaxRegistrationDocument"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetTaxRegistrationDocumentResponse> getTaxRegistrationDocument(
            GetTaxRegistrationDocumentRequest getTaxRegistrationDocumentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getTaxRegistrationDocumentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getTaxRegistrationDocumentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "TaxSettings");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTaxRegistrationDocument");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetTaxRegistrationDocumentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetTaxRegistrationDocumentResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "CaseCreationLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CaseCreationLimitExceededException")
                            .httpStatusCode(413).exceptionBuilderSupplier(CaseCreationLimitExceededException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "AttachmentUploadException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AttachmentUploadException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AttachmentUploadException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetTaxRegistrationDocumentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetTaxRegistrationDocumentRequest, GetTaxRegistrationDocumentResponse>()
                            .withOperationName("GetTaxRegistrationDocument").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetTaxRegistrationDocumentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getTaxRegistrationDocumentRequest));
            CompletableFuture<GetTaxRegistrationDocumentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves supplemental tax registrations for a single account.
     * </p>
     *
     * @param listSupplementalTaxRegistrationsRequest
     * @return A Java Future containing the result of the ListSupplementalTaxRegistrations operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The exception when the input doesn't pass validation for at least one of the
     *         input parameters.</li>
     *         <li>ResourceNotFoundException The exception thrown when the input doesn't have a resource associated to
     *         it.</li>
     *         <li>InternalServerException The exception thrown when an unexpected error occurs when processing a
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TaxSettingsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TaxSettingsAsyncClient.ListSupplementalTaxRegistrations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/taxsettings-2018-05-10/ListSupplementalTaxRegistrations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListSupplementalTaxRegistrationsResponse> listSupplementalTaxRegistrations(
            ListSupplementalTaxRegistrationsRequest listSupplementalTaxRegistrationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listSupplementalTaxRegistrationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listSupplementalTaxRegistrationsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "TaxSettings");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSupplementalTaxRegistrations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListSupplementalTaxRegistrationsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, ListSupplementalTaxRegistrationsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "CaseCreationLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CaseCreationLimitExceededException")
                            .httpStatusCode(413).exceptionBuilderSupplier(CaseCreationLimitExceededException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "AttachmentUploadException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AttachmentUploadException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AttachmentUploadException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListSupplementalTaxRegistrationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListSupplementalTaxRegistrationsRequest, ListSupplementalTaxRegistrationsResponse>()
                            .withOperationName("ListSupplementalTaxRegistrations").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListSupplementalTaxRegistrationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listSupplementalTaxRegistrationsRequest));
            CompletableFuture<ListSupplementalTaxRegistrationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the tax exemption of accounts listed in a consolidated billing family. The IAM action is
     * <code>tax:GetExemptions</code>.
     * </p>
     *
     * @param listTaxExemptionsRequest
     * @return A Java Future containing the result of the ListTaxExemptions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The exception when the input doesn't pass validation for at least one of the
     *         input parameters.</li>
     *         <li>ResourceNotFoundException The exception thrown when the input doesn't have a resource associated to
     *         it.</li>
     *         <li>InternalServerException The exception thrown when an unexpected error occurs when processing a
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TaxSettingsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TaxSettingsAsyncClient.ListTaxExemptions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/taxsettings-2018-05-10/ListTaxExemptions" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTaxExemptionsResponse> listTaxExemptions(ListTaxExemptionsRequest listTaxExemptionsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTaxExemptionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTaxExemptionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "TaxSettings");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTaxExemptions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTaxExemptionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTaxExemptionsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "CaseCreationLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CaseCreationLimitExceededException")
                            .httpStatusCode(413).exceptionBuilderSupplier(CaseCreationLimitExceededException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "AttachmentUploadException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AttachmentUploadException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AttachmentUploadException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListTaxExemptionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTaxExemptionsRequest, ListTaxExemptionsResponse>()
                            .withOperationName("ListTaxExemptions").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTaxExemptionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTaxExemptionsRequest));
            CompletableFuture<ListTaxExemptionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the tax registration of accounts listed in a consolidated billing family. This can be used to retrieve
     * up to 100 accounts' tax registrations in one call (default 50).
     * </p>
     *
     * @param listTaxRegistrationsRequest
     * @return A Java Future containing the result of the ListTaxRegistrations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The exception when the input doesn't pass validation for at least one of the
     *         input parameters.</li>
     *         <li>ResourceNotFoundException The exception thrown when the input doesn't have a resource associated to
     *         it.</li>
     *         <li>InternalServerException The exception thrown when an unexpected error occurs when processing a
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TaxSettingsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TaxSettingsAsyncClient.ListTaxRegistrations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/taxsettings-2018-05-10/ListTaxRegistrations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTaxRegistrationsResponse> listTaxRegistrations(
            ListTaxRegistrationsRequest listTaxRegistrationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTaxRegistrationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTaxRegistrationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "TaxSettings");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTaxRegistrations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTaxRegistrationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTaxRegistrationsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "CaseCreationLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CaseCreationLimitExceededException")
                            .httpStatusCode(413).exceptionBuilderSupplier(CaseCreationLimitExceededException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "AttachmentUploadException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AttachmentUploadException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AttachmentUploadException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListTaxRegistrationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTaxRegistrationsRequest, ListTaxRegistrationsResponse>()
                            .withOperationName("ListTaxRegistrations").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTaxRegistrationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTaxRegistrationsRequest));
            CompletableFuture<ListTaxRegistrationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Stores supplemental tax registration for a single account.
     * </p>
     *
     * @param putSupplementalTaxRegistrationRequest
     * @return A Java Future containing the result of the PutSupplementalTaxRegistration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The exception when the input doesn't pass validation for at least one of the
     *         input parameters.</li>
     *         <li>ConflictException The exception when the input is creating conflict with the given state.</li>
     *         <li>InternalServerException The exception thrown when an unexpected error occurs when processing a
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TaxSettingsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TaxSettingsAsyncClient.PutSupplementalTaxRegistration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/taxsettings-2018-05-10/PutSupplementalTaxRegistration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutSupplementalTaxRegistrationResponse> putSupplementalTaxRegistration(
            PutSupplementalTaxRegistrationRequest putSupplementalTaxRegistrationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putSupplementalTaxRegistrationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                putSupplementalTaxRegistrationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "TaxSettings");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutSupplementalTaxRegistration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutSupplementalTaxRegistrationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutSupplementalTaxRegistrationResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "CaseCreationLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CaseCreationLimitExceededException")
                            .httpStatusCode(413).exceptionBuilderSupplier(CaseCreationLimitExceededException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "AttachmentUploadException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AttachmentUploadException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AttachmentUploadException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<PutSupplementalTaxRegistrationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutSupplementalTaxRegistrationRequest, PutSupplementalTaxRegistrationResponse>()
                            .withOperationName("PutSupplementalTaxRegistration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PutSupplementalTaxRegistrationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putSupplementalTaxRegistrationRequest));
            CompletableFuture<PutSupplementalTaxRegistrationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds the tax exemption for a single account or all accounts listed in a consolidated billing family. The IAM
     * action is <code>tax:UpdateExemptions</code>.
     * </p>
     *
     * @param putTaxExemptionRequest
     * @return A Java Future containing the result of the PutTaxExemption operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The exception when the input doesn't pass validation for at least one of the
     *         input parameters.</li>
     *         <li>AccessDeniedException The access is denied for the Amazon Web ServicesSupport API.</li>
     *         <li>CaseCreationLimitExceededException You've exceeded the Amazon Web ServicesSupport case creation limit
     *         for your account.</li>
     *         <li>ResourceNotFoundException The exception thrown when the input doesn't have a resource associated to
     *         it.</li>
     *         <li>AttachmentUploadException Failed to upload the tax exemption document to Amazon Web ServicesSupport
     *         case.</li>
     *         <li>InternalServerException The exception thrown when an unexpected error occurs when processing a
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TaxSettingsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TaxSettingsAsyncClient.PutTaxExemption
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/taxsettings-2018-05-10/PutTaxExemption" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<PutTaxExemptionResponse> putTaxExemption(PutTaxExemptionRequest putTaxExemptionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putTaxExemptionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putTaxExemptionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "TaxSettings");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutTaxExemption");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutTaxExemptionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutTaxExemptionResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "CaseCreationLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CaseCreationLimitExceededException")
                            .httpStatusCode(413).exceptionBuilderSupplier(CaseCreationLimitExceededException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "AttachmentUploadException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AttachmentUploadException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AttachmentUploadException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<PutTaxExemptionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutTaxExemptionRequest, PutTaxExemptionResponse>()
                            .withOperationName("PutTaxExemption").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PutTaxExemptionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putTaxExemptionRequest));
            CompletableFuture<PutTaxExemptionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The updated tax inheritance status.
     * </p>
     *
     * @param putTaxInheritanceRequest
     * @return A Java Future containing the result of the PutTaxInheritance operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The exception when the input doesn't pass validation for at least one of the
     *         input parameters.</li>
     *         <li>ConflictException The exception when the input is creating conflict with the given state.</li>
     *         <li>ResourceNotFoundException The exception thrown when the input doesn't have a resource associated to
     *         it.</li>
     *         <li>InternalServerException The exception thrown when an unexpected error occurs when processing a
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TaxSettingsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TaxSettingsAsyncClient.PutTaxInheritance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/taxsettings-2018-05-10/PutTaxInheritance" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<PutTaxInheritanceResponse> putTaxInheritance(PutTaxInheritanceRequest putTaxInheritanceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putTaxInheritanceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putTaxInheritanceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "TaxSettings");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutTaxInheritance");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutTaxInheritanceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutTaxInheritanceResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "CaseCreationLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CaseCreationLimitExceededException")
                            .httpStatusCode(413).exceptionBuilderSupplier(CaseCreationLimitExceededException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "AttachmentUploadException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AttachmentUploadException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AttachmentUploadException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<PutTaxInheritanceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutTaxInheritanceRequest, PutTaxInheritanceResponse>()
                            .withOperationName("PutTaxInheritance").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PutTaxInheritanceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putTaxInheritanceRequest));
            CompletableFuture<PutTaxInheritanceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds or updates tax registration for a single account. You can't set a TRN if there's a pending TRN. You'll need
     * to delete the pending TRN first.
     * </p>
     * <p>
     * To call this API operation for specific countries, see the following country-specific requirements.
     * </p>
     * <p>
     * <b>Bangladesh</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * You must specify the tax registration certificate document in the <code>taxRegistrationDocuments</code> field of
     * the <code>VerificationDetails</code> object.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>Brazil</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * You must complete the tax registration process in the <a
     * href="https://console.aws.amazon.com/billing/home#/paymentpreferences/paymentmethods">Payment preferences</a>
     * page in the Billing and Cost Management console. After your TRN and billing address are verified, you can call
     * this API operation.
     * </p>
     * </li>
     * <li>
     * <p>
     * For Amazon Web Services accounts created through Organizations, you can call this API operation when you don't
     * have a billing address.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>Georgia</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * The valid <code>personType</code> values are <code>Physical Person</code> and <code>Business</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>Indonesia</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>PutTaxRegistration</code>: The use of this operation to submit tax information is subject to the <a
     * href="http://aws.amazon.com/service-terms/">Amazon Web Services service terms</a>. By submitting, you’re
     * providing consent for Amazon Web Services to validate NIK, NPWP, and NITKU data, provided by you with the
     * Directorate General of Taxes of Indonesia in accordance with the Minister of Finance Regulation (PMK) Number
     * 112/PMK.03/2022.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>BatchPutTaxRegistration</code>: The use of this operation to submit tax information is subject to the <a
     * href="http://aws.amazon.com/service-terms/">Amazon Web Services service terms</a>. By submitting, you’re
     * providing consent for Amazon Web Services to validate NIK, NPWP, and NITKU data, provided by you with the
     * Directorate General of Taxes of Indonesia in accordance with the Minister of Finance Regulation (PMK) Number
     * 112/PMK.03/2022, through our third-party partner PT Achilles Advanced Management (OnlinePajak).
     * </p>
     * </li>
     * <li>
     * <p>
     * You must specify the <code>taxRegistrationNumberType</code> in the <code>indonesiaAdditionalInfo</code> field of
     * the <code>additionalTaxInformation</code> object.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you specify <code>decisionNumber</code>, you must specify the <code>ppnExceptionDesignationCode</code> in the
     * <code>indonesiaAdditionalInfo</code> field of the <code>additionalTaxInformation</code> object. If the
     * <code>taxRegistrationNumberType</code> is set to NPWP or NITKU, valid values for
     * <code>ppnExceptionDesignationCode</code> are either <code>01</code>, <code>02</code>, <code>03</code>,
     * <code>07</code>, or <code>08</code>.
     * </p>
     * <p>
     * For other <code>taxRegistrationNumberType</code> values, <code>ppnExceptionDesignationCode</code> must be either
     * <code>01</code>, <code>07</code>, or <code>08</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * If <code>ppnExceptionDesignationCode</code> is <code>07</code>, you must specify the <code>decisionNumber</code>
     * in the <code>indonesiaAdditionalInfo</code> field of the <code>additionalTaxInformation</code> object.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>Kenya</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * You must specify the <code>personType</code> in the <code>kenyaAdditionalInfo</code> field of the
     * <code>additionalTaxInformation</code> object.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the <code>personType</code> is <code>Physical Person</code>, you must specify the tax registration certificate
     * document in the <code>taxRegistrationDocuments</code> field of the <code>VerificationDetails</code> object.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>Malaysia</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * The sector valid values are <code>Business</code> and <code>Individual</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>RegistrationType</code> valid values are <code>NRIC</code> for individual, and TIN and sales and service
     * tax (SST) for Business.
     * </p>
     * </li>
     * <li>
     * <p>
     * For individual, you can specify the <code>taxInformationNumber</code> in <code>MalaysiaAdditionalInfo</code> with
     * NRIC type, and a valid <code>MyKad</code> or NRIC number.
     * </p>
     * </li>
     * <li>
     * <p>
     * For business, you must specify a <code>businessRegistrationNumber</code> in <code>MalaysiaAdditionalInfo</code>
     * with a TIN type and tax identification number.
     * </p>
     * </li>
     * <li>
     * <p>
     * For business resellers, you must specify a <code>businessRegistrationNumber</code> and
     * <code>taxInformationNumber</code> in <code>MalaysiaAdditionalInfo</code> with a sales and service tax (SST) type
     * and a valid SST number.
     * </p>
     * </li>
     * <li>
     * <p>
     * For business resellers with service codes, you must specify <code>businessRegistrationNumber</code>,
     * <code>taxInformationNumber</code>, and distinct <code>serviceTaxCodes</code> in
     * <code>MalaysiaAdditionalInfo</code> with a SST type and valid sales and service tax (SST) number. By using this
     * API operation, Amazon Web Services registers your self-declaration that you’re an authorized business reseller
     * registered with the Royal Malaysia Customs Department (RMCD), and have a valid SST number.
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon Web Services reserves the right to seek additional information and/or take other actions to support your
     * self-declaration as appropriate.
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon Web Services is currently registered under the following service tax codes. You must include at least one
     * of the service tax codes in the service tax code strings to declare yourself as an authorized registered business
     * reseller.
     * </p>
     * <p>
     * Taxable service and service tax codes:
     * </p>
     * <p>
     * Consultancy - 9907061674
     * </p>
     * <p>
     * Training or coaching service - 9907071685
     * </p>
     * <p>
     * IT service - 9907101676
     * </p>
     * <p>
     * Digital services and electronic medium - 9907121690
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>Nepal</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * The sector valid values are <code>Business</code> and <code>Individual</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>Saudi Arabia</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * For <code>address</code>, you must specify <code>addressLine3</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>South Korea</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * You must specify the <code>certifiedEmailId</code> and <code>legalName</code> in the
     * <code>TaxRegistrationEntry</code> object. Use Korean characters for <code>legalName</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * You must specify the <code>businessRepresentativeName</code>, <code>itemOfBusiness</code>, and
     * <code>lineOfBusiness</code> in the <code>southKoreaAdditionalInfo</code> field of the
     * <code>additionalTaxInformation</code> object. Use Korean characters for these fields.
     * </p>
     * </li>
     * <li>
     * <p>
     * You must specify the tax registration certificate document in the <code>taxRegistrationDocuments</code> field of
     * the <code>VerificationDetails</code> object.
     * </p>
     * </li>
     * <li>
     * <p>
     * For the <code>address</code> object, use Korean characters for <code>addressLine1</code>,
     * <code>addressLine2</code> <code>city</code>, <code>postalCode</code>, and <code>stateOrRegion</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>Spain</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * You must specify the <code>registrationType</code> in the <code>spainAdditionalInfo</code> field of the
     * <code>additionalTaxInformation</code> object.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the <code>registrationType</code> is <code>Local</code>, you must specify the tax registration certificate
     * document in the <code>taxRegistrationDocuments</code> field of the <code>VerificationDetails</code> object.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>Turkey</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * You must specify the <code>sector</code> in the <code>taxRegistrationEntry</code> object.
     * </p>
     * </li>
     * <li>
     * <p>
     * If your <code>sector</code> is <code>Business</code>, <code>Individual</code>, or <code>Government</code>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Specify the <code>taxOffice</code>. If your <code>sector</code> is <code>Individual</code>, don't enter this
     * value.
     * </p>
     * </li>
     * <li>
     * <p>
     * (Optional) Specify the <code>kepEmailId</code>. If your <code>sector</code> is <code>Individual</code>, don't
     * enter this value.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Note:</b> In the <b>Tax Settings</b> page of the Billing console, <code>Government</code> appears as <b>Public
     * institutions</b>
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * If your <code>sector</code> is <code>Business</code> and you're subject to KDV tax, you must specify your
     * industry in the <code>industries</code> field.
     * </p>
     * </li>
     * <li>
     * <p>
     * For <code>address</code>, you must specify <code>districtOrCounty</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>Ukraine</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * The sector valid values are <code>Business</code> and <code>Individual</code>.
     * </p>
     * </li>
     * </ul>
     *
     * @param putTaxRegistrationRequest
     * @return A Java Future containing the result of the PutTaxRegistration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The exception when the input doesn't pass validation for at least one of the
     *         input parameters.</li>
     *         <li>ConflictException The exception when the input is creating conflict with the given state.</li>
     *         <li>InternalServerException The exception thrown when an unexpected error occurs when processing a
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TaxSettingsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TaxSettingsAsyncClient.PutTaxRegistration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/taxsettings-2018-05-10/PutTaxRegistration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutTaxRegistrationResponse> putTaxRegistration(PutTaxRegistrationRequest putTaxRegistrationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putTaxRegistrationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putTaxRegistrationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "TaxSettings");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutTaxRegistration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutTaxRegistrationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutTaxRegistrationResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "CaseCreationLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CaseCreationLimitExceededException")
                            .httpStatusCode(413).exceptionBuilderSupplier(CaseCreationLimitExceededException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "AttachmentUploadException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AttachmentUploadException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AttachmentUploadException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<PutTaxRegistrationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutTaxRegistrationRequest, PutTaxRegistrationResponse>()
                            .withOperationName("PutTaxRegistration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PutTaxRegistrationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putTaxRegistrationRequest));
            CompletableFuture<PutTaxRegistrationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final TaxSettingsServiceClientConfiguration serviceClientConfiguration() {
        return new TaxSettingsServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder.clientConfiguration(clientConfiguration).defaultServiceExceptionSupplier(TaxSettingsException::builder)
                .protocol(AwsJsonProtocol.REST_JSON).protocolVersion("1.1");
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        TaxSettingsServiceClientConfigurationBuilder serviceConfigBuilder = new TaxSettingsServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
