/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.textract.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Sets whether or not your output will go to a user created bucket. Used to set the name of the bucket, and the prefix
 * on the output file.
 * </p>
 * <p>
 * <code>OutputConfig</code> is an optional parameter which lets you adjust where your output will be placed. By
 * default, Amazon Textract will store the results internally and can only be accessed by the Get API operations. With
 * OutputConfig enabled, you can set the name of the bucket the output will be sent to and the file prefix of the
 * results where you can download your results. Additionally, you can set the <code>KMSKeyID</code> parameter to a
 * customer master key (CMK) to encrypt your output. Without this parameter set Amazon Textract will encrypt server-side
 * using the AWS managed CMK for Amazon S3.
 * </p>
 * <p>
 * Decryption of Customer Content is necessary for processing of the documents by Amazon Textract. If your account is
 * opted out under an AI services opt out policy then all unencrypted Customer Content is immediately and permanently
 * deleted after the Customer Content has been processed by the service. No copy of of the output is retained by Amazon
 * Textract. For information about how to opt out, see <a
 * href="https://docs.aws.amazon.com/organizations/latest/userguide/orgs_manage_policies_ai-opt-out.html"> Managing AI
 * services opt-out policy. </a>
 * </p>
 * <p>
 * For more information on data privacy, see the <a href="https://aws.amazon.com/compliance/data-privacy-faq/">Data
 * Privacy FAQ</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class OutputConfig implements SdkPojo, Serializable, ToCopyableBuilder<OutputConfig.Builder, OutputConfig> {
    private static final SdkField<String> S3_BUCKET_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("S3Bucket").getter(getter(OutputConfig::s3Bucket)).setter(setter(Builder::s3Bucket))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("S3Bucket").build()).build();

    private static final SdkField<String> S3_PREFIX_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("S3Prefix").getter(getter(OutputConfig::s3Prefix)).setter(setter(Builder::s3Prefix))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("S3Prefix").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(S3_BUCKET_FIELD,
            S3_PREFIX_FIELD));

    private static final long serialVersionUID = 1L;

    private final String s3Bucket;

    private final String s3Prefix;

    private OutputConfig(BuilderImpl builder) {
        this.s3Bucket = builder.s3Bucket;
        this.s3Prefix = builder.s3Prefix;
    }

    /**
     * <p>
     * The name of the bucket your output will go to.
     * </p>
     * 
     * @return The name of the bucket your output will go to.
     */
    public final String s3Bucket() {
        return s3Bucket;
    }

    /**
     * <p>
     * The prefix of the object key that the output will be saved to. When not enabled, the prefix will be
     * “textract_output".
     * </p>
     * 
     * @return The prefix of the object key that the output will be saved to. When not enabled, the prefix will be
     *         “textract_output".
     */
    public final String s3Prefix() {
        return s3Prefix;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(s3Bucket());
        hashCode = 31 * hashCode + Objects.hashCode(s3Prefix());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof OutputConfig)) {
            return false;
        }
        OutputConfig other = (OutputConfig) obj;
        return Objects.equals(s3Bucket(), other.s3Bucket()) && Objects.equals(s3Prefix(), other.s3Prefix());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("OutputConfig").add("S3Bucket", s3Bucket()).add("S3Prefix", s3Prefix()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "S3Bucket":
            return Optional.ofNullable(clazz.cast(s3Bucket()));
        case "S3Prefix":
            return Optional.ofNullable(clazz.cast(s3Prefix()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<OutputConfig, T> g) {
        return obj -> g.apply((OutputConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, OutputConfig> {
        /**
         * <p>
         * The name of the bucket your output will go to.
         * </p>
         * 
         * @param s3Bucket
         *        The name of the bucket your output will go to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3Bucket(String s3Bucket);

        /**
         * <p>
         * The prefix of the object key that the output will be saved to. When not enabled, the prefix will be
         * “textract_output".
         * </p>
         * 
         * @param s3Prefix
         *        The prefix of the object key that the output will be saved to. When not enabled, the prefix will be
         *        “textract_output".
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3Prefix(String s3Prefix);
    }

    static final class BuilderImpl implements Builder {
        private String s3Bucket;

        private String s3Prefix;

        private BuilderImpl() {
        }

        private BuilderImpl(OutputConfig model) {
            s3Bucket(model.s3Bucket);
            s3Prefix(model.s3Prefix);
        }

        public final String getS3Bucket() {
            return s3Bucket;
        }

        public final void setS3Bucket(String s3Bucket) {
            this.s3Bucket = s3Bucket;
        }

        @Override
        @Transient
        public final Builder s3Bucket(String s3Bucket) {
            this.s3Bucket = s3Bucket;
            return this;
        }

        public final String getS3Prefix() {
            return s3Prefix;
        }

        public final void setS3Prefix(String s3Prefix) {
            this.s3Prefix = s3Prefix;
        }

        @Override
        @Transient
        public final Builder s3Prefix(String s3Prefix) {
            this.s3Prefix = s3Prefix;
            return this;
        }

        @Override
        public OutputConfig build() {
            return new OutputConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
