/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.textract;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.textract.model.AccessDeniedException;
import software.amazon.awssdk.services.textract.model.AnalyzeDocumentRequest;
import software.amazon.awssdk.services.textract.model.AnalyzeDocumentResponse;
import software.amazon.awssdk.services.textract.model.AnalyzeExpenseRequest;
import software.amazon.awssdk.services.textract.model.AnalyzeExpenseResponse;
import software.amazon.awssdk.services.textract.model.AnalyzeIdRequest;
import software.amazon.awssdk.services.textract.model.AnalyzeIdResponse;
import software.amazon.awssdk.services.textract.model.BadDocumentException;
import software.amazon.awssdk.services.textract.model.DetectDocumentTextRequest;
import software.amazon.awssdk.services.textract.model.DetectDocumentTextResponse;
import software.amazon.awssdk.services.textract.model.DocumentTooLargeException;
import software.amazon.awssdk.services.textract.model.GetDocumentAnalysisRequest;
import software.amazon.awssdk.services.textract.model.GetDocumentAnalysisResponse;
import software.amazon.awssdk.services.textract.model.GetDocumentTextDetectionRequest;
import software.amazon.awssdk.services.textract.model.GetDocumentTextDetectionResponse;
import software.amazon.awssdk.services.textract.model.GetExpenseAnalysisRequest;
import software.amazon.awssdk.services.textract.model.GetExpenseAnalysisResponse;
import software.amazon.awssdk.services.textract.model.HumanLoopQuotaExceededException;
import software.amazon.awssdk.services.textract.model.IdempotentParameterMismatchException;
import software.amazon.awssdk.services.textract.model.InternalServerErrorException;
import software.amazon.awssdk.services.textract.model.InvalidJobIdException;
import software.amazon.awssdk.services.textract.model.InvalidKmsKeyException;
import software.amazon.awssdk.services.textract.model.InvalidParameterException;
import software.amazon.awssdk.services.textract.model.InvalidS3ObjectException;
import software.amazon.awssdk.services.textract.model.LimitExceededException;
import software.amazon.awssdk.services.textract.model.ProvisionedThroughputExceededException;
import software.amazon.awssdk.services.textract.model.StartDocumentAnalysisRequest;
import software.amazon.awssdk.services.textract.model.StartDocumentAnalysisResponse;
import software.amazon.awssdk.services.textract.model.StartDocumentTextDetectionRequest;
import software.amazon.awssdk.services.textract.model.StartDocumentTextDetectionResponse;
import software.amazon.awssdk.services.textract.model.StartExpenseAnalysisRequest;
import software.amazon.awssdk.services.textract.model.StartExpenseAnalysisResponse;
import software.amazon.awssdk.services.textract.model.TextractException;
import software.amazon.awssdk.services.textract.model.ThrottlingException;
import software.amazon.awssdk.services.textract.model.UnsupportedDocumentException;
import software.amazon.awssdk.services.textract.transform.AnalyzeDocumentRequestMarshaller;
import software.amazon.awssdk.services.textract.transform.AnalyzeExpenseRequestMarshaller;
import software.amazon.awssdk.services.textract.transform.AnalyzeIdRequestMarshaller;
import software.amazon.awssdk.services.textract.transform.DetectDocumentTextRequestMarshaller;
import software.amazon.awssdk.services.textract.transform.GetDocumentAnalysisRequestMarshaller;
import software.amazon.awssdk.services.textract.transform.GetDocumentTextDetectionRequestMarshaller;
import software.amazon.awssdk.services.textract.transform.GetExpenseAnalysisRequestMarshaller;
import software.amazon.awssdk.services.textract.transform.StartDocumentAnalysisRequestMarshaller;
import software.amazon.awssdk.services.textract.transform.StartDocumentTextDetectionRequestMarshaller;
import software.amazon.awssdk.services.textract.transform.StartExpenseAnalysisRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link TextractClient}.
 *
 * @see TextractClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultTextractClient implements TextractClient {
    private static final Logger log = Logger.loggerFor(DefaultTextractClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultTextractClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Analyzes an input document for relationships between detected items.
     * </p>
     * <p>
     * The types of information returned are as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Form data (key-value pairs). The related information is returned in two <a>Block</a> objects, each of type
     * <code>KEY_VALUE_SET</code>: a KEY <code>Block</code> object and a VALUE <code>Block</code> object. For example,
     * <i>Name: Ana Silva Carolina</i> contains a key and value. <i>Name:</i> is the key. <i>Ana Silva Carolina</i> is
     * the value.
     * </p>
     * </li>
     * <li>
     * <p>
     * Table and table cell data. A TABLE <code>Block</code> object contains information about a detected table. A CELL
     * <code>Block</code> object is returned for each cell in a table.
     * </p>
     * </li>
     * <li>
     * <p>
     * Lines and words of text. A LINE <code>Block</code> object contains one or more WORD <code>Block</code> objects.
     * All lines and words that are detected in the document are returned (including text that doesn't have a
     * relationship with the value of <code>FeatureTypes</code>).
     * </p>
     * </li>
     * </ul>
     * <p>
     * Selection elements such as check boxes and option buttons (radio buttons) can be detected in form data and in
     * tables. A SELECTION_ELEMENT <code>Block</code> object contains information about a selection element, including
     * the selection status.
     * </p>
     * <p>
     * You can choose which type of analysis to perform by specifying the <code>FeatureTypes</code> list.
     * </p>
     * <p>
     * The output is returned in a list of <code>Block</code> objects.
     * </p>
     * <p>
     * <code>AnalyzeDocument</code> is a synchronous operation. To analyze documents asynchronously, use
     * <a>StartDocumentAnalysis</a>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/textract/latest/dg/how-it-works-analyzing.html">Document Text Analysis</a>.
     * </p>
     *
     * @param analyzeDocumentRequest
     * @return Result of the AnalyzeDocument operation returned by the service.
     * @throws InvalidParameterException
     *         An input parameter violated a constraint. For example, in synchronous operations, an
     *         <code>InvalidParameterException</code> exception occurs when neither of the <code>S3Object</code> or
     *         <code>Bytes</code> values are supplied in the <code>Document</code> request parameter. Validate your
     *         parameter before calling the API operation again.
     * @throws InvalidS3ObjectException
     *         Amazon Textract is unable to access the S3 object that's specified in the request. for more information,
     *         <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-access-control.html">Configure Access to
     *         Amazon S3</a> For troubleshooting information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/troubleshooting.html">Troubleshooting Amazon S3</a>
     * @throws UnsupportedDocumentException
     *         The format of the input document isn't supported. Documents for synchronous operations can be in PNG or
     *         JPEG format only. Documents for asynchronous operations can be in PDF format.
     * @throws DocumentTooLargeException
     *         The document can't be processed because it's too large. The maximum document size for synchronous
     *         operations 10 MB. The maximum document size for asynchronous operations is 500 MB for PDF files.
     * @throws BadDocumentException
     *         Amazon Textract isn't able to read the document. For more information on the document limits in Amazon
     *         Textract, see <a>limits</a>.
     * @throws AccessDeniedException
     *         You aren't authorized to perform the action. Use the Amazon Resource Name (ARN) of an authorized user or
     *         IAM role to perform the operation.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Textract.
     * @throws InternalServerErrorException
     *         Amazon Textract experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Textract is temporarily unable to process the request. Try your call again.
     * @throws HumanLoopQuotaExceededException
     *         Indicates you have exceeded the maximum number of active human in the loop workflows available
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TextractException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TextractClient.AnalyzeDocument
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/AnalyzeDocument" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public AnalyzeDocumentResponse analyzeDocument(AnalyzeDocumentRequest analyzeDocumentRequest)
            throws InvalidParameterException, InvalidS3ObjectException, UnsupportedDocumentException, DocumentTooLargeException,
            BadDocumentException, AccessDeniedException, ProvisionedThroughputExceededException, InternalServerErrorException,
            ThrottlingException, HumanLoopQuotaExceededException, AwsServiceException, SdkClientException, TextractException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AnalyzeDocumentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                AnalyzeDocumentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, analyzeDocumentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Textract");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AnalyzeDocument");

            return clientHandler.execute(new ClientExecutionParams<AnalyzeDocumentRequest, AnalyzeDocumentResponse>()
                    .withOperationName("AnalyzeDocument").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(analyzeDocumentRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new AnalyzeDocumentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * <code>AnalyzeExpense</code> synchronously analyzes an input document for financially related relationships
     * between text.
     * </p>
     * <p>
     * Information is returned as <code>ExpenseDocuments</code> and seperated as follows.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>LineItemGroups</code>- A data set containing <code>LineItems</code> which store information about the lines
     * of text, such as an item purchased and its price on a receipt.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SummaryFields</code>- Contains all other information a receipt, such as header information or the vendors
     * name.
     * </p>
     * </li>
     * </ul>
     *
     * @param analyzeExpenseRequest
     * @return Result of the AnalyzeExpense operation returned by the service.
     * @throws InvalidParameterException
     *         An input parameter violated a constraint. For example, in synchronous operations, an
     *         <code>InvalidParameterException</code> exception occurs when neither of the <code>S3Object</code> or
     *         <code>Bytes</code> values are supplied in the <code>Document</code> request parameter. Validate your
     *         parameter before calling the API operation again.
     * @throws InvalidS3ObjectException
     *         Amazon Textract is unable to access the S3 object that's specified in the request. for more information,
     *         <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-access-control.html">Configure Access to
     *         Amazon S3</a> For troubleshooting information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/troubleshooting.html">Troubleshooting Amazon S3</a>
     * @throws UnsupportedDocumentException
     *         The format of the input document isn't supported. Documents for synchronous operations can be in PNG or
     *         JPEG format only. Documents for asynchronous operations can be in PDF format.
     * @throws DocumentTooLargeException
     *         The document can't be processed because it's too large. The maximum document size for synchronous
     *         operations 10 MB. The maximum document size for asynchronous operations is 500 MB for PDF files.
     * @throws BadDocumentException
     *         Amazon Textract isn't able to read the document. For more information on the document limits in Amazon
     *         Textract, see <a>limits</a>.
     * @throws AccessDeniedException
     *         You aren't authorized to perform the action. Use the Amazon Resource Name (ARN) of an authorized user or
     *         IAM role to perform the operation.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Textract.
     * @throws InternalServerErrorException
     *         Amazon Textract experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Textract is temporarily unable to process the request. Try your call again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TextractException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TextractClient.AnalyzeExpense
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/AnalyzeExpense" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public AnalyzeExpenseResponse analyzeExpense(AnalyzeExpenseRequest analyzeExpenseRequest) throws InvalidParameterException,
            InvalidS3ObjectException, UnsupportedDocumentException, DocumentTooLargeException, BadDocumentException,
            AccessDeniedException, ProvisionedThroughputExceededException, InternalServerErrorException, ThrottlingException,
            AwsServiceException, SdkClientException, TextractException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AnalyzeExpenseResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                AnalyzeExpenseResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, analyzeExpenseRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Textract");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AnalyzeExpense");

            return clientHandler.execute(new ClientExecutionParams<AnalyzeExpenseRequest, AnalyzeExpenseResponse>()
                    .withOperationName("AnalyzeExpense").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(analyzeExpenseRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new AnalyzeExpenseRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Analyzes identity documents for relevant information. This information is extracted and returned as
     * <code>IdentityDocumentFields</code>, which records both the normalized field and value of the extracted text.
     * </p>
     *
     * @param analyzeIdRequest
     * @return Result of the AnalyzeID operation returned by the service.
     * @throws InvalidParameterException
     *         An input parameter violated a constraint. For example, in synchronous operations, an
     *         <code>InvalidParameterException</code> exception occurs when neither of the <code>S3Object</code> or
     *         <code>Bytes</code> values are supplied in the <code>Document</code> request parameter. Validate your
     *         parameter before calling the API operation again.
     * @throws InvalidS3ObjectException
     *         Amazon Textract is unable to access the S3 object that's specified in the request. for more information,
     *         <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-access-control.html">Configure Access to
     *         Amazon S3</a> For troubleshooting information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/troubleshooting.html">Troubleshooting Amazon S3</a>
     * @throws UnsupportedDocumentException
     *         The format of the input document isn't supported. Documents for synchronous operations can be in PNG or
     *         JPEG format only. Documents for asynchronous operations can be in PDF format.
     * @throws DocumentTooLargeException
     *         The document can't be processed because it's too large. The maximum document size for synchronous
     *         operations 10 MB. The maximum document size for asynchronous operations is 500 MB for PDF files.
     * @throws BadDocumentException
     *         Amazon Textract isn't able to read the document. For more information on the document limits in Amazon
     *         Textract, see <a>limits</a>.
     * @throws AccessDeniedException
     *         You aren't authorized to perform the action. Use the Amazon Resource Name (ARN) of an authorized user or
     *         IAM role to perform the operation.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Textract.
     * @throws InternalServerErrorException
     *         Amazon Textract experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Textract is temporarily unable to process the request. Try your call again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TextractException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TextractClient.AnalyzeID
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/AnalyzeID" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public AnalyzeIdResponse analyzeID(AnalyzeIdRequest analyzeIdRequest) throws InvalidParameterException,
            InvalidS3ObjectException, UnsupportedDocumentException, DocumentTooLargeException, BadDocumentException,
            AccessDeniedException, ProvisionedThroughputExceededException, InternalServerErrorException, ThrottlingException,
            AwsServiceException, SdkClientException, TextractException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AnalyzeIdResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                AnalyzeIdResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, analyzeIdRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Textract");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AnalyzeID");

            return clientHandler.execute(new ClientExecutionParams<AnalyzeIdRequest, AnalyzeIdResponse>()
                    .withOperationName("AnalyzeID").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(analyzeIdRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new AnalyzeIdRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Detects text in the input document. Amazon Textract can detect lines of text and the words that make up a line of
     * text. The input document must be an image in JPEG or PNG format. <code>DetectDocumentText</code> returns the
     * detected text in an array of <a>Block</a> objects.
     * </p>
     * <p>
     * Each document page has as an associated <code>Block</code> of type PAGE. Each PAGE <code>Block</code> object is
     * the parent of LINE <code>Block</code> objects that represent the lines of detected text on a page. A LINE
     * <code>Block</code> object is a parent for each word that makes up the line. Words are represented by
     * <code>Block</code> objects of type WORD.
     * </p>
     * <p>
     * <code>DetectDocumentText</code> is a synchronous operation. To analyze documents asynchronously, use
     * <a>StartDocumentTextDetection</a>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/textract/latest/dg/how-it-works-detecting.html">Document Text Detection</a>.
     * </p>
     *
     * @param detectDocumentTextRequest
     * @return Result of the DetectDocumentText operation returned by the service.
     * @throws InvalidParameterException
     *         An input parameter violated a constraint. For example, in synchronous operations, an
     *         <code>InvalidParameterException</code> exception occurs when neither of the <code>S3Object</code> or
     *         <code>Bytes</code> values are supplied in the <code>Document</code> request parameter. Validate your
     *         parameter before calling the API operation again.
     * @throws InvalidS3ObjectException
     *         Amazon Textract is unable to access the S3 object that's specified in the request. for more information,
     *         <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-access-control.html">Configure Access to
     *         Amazon S3</a> For troubleshooting information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/troubleshooting.html">Troubleshooting Amazon S3</a>
     * @throws UnsupportedDocumentException
     *         The format of the input document isn't supported. Documents for synchronous operations can be in PNG or
     *         JPEG format only. Documents for asynchronous operations can be in PDF format.
     * @throws DocumentTooLargeException
     *         The document can't be processed because it's too large. The maximum document size for synchronous
     *         operations 10 MB. The maximum document size for asynchronous operations is 500 MB for PDF files.
     * @throws BadDocumentException
     *         Amazon Textract isn't able to read the document. For more information on the document limits in Amazon
     *         Textract, see <a>limits</a>.
     * @throws AccessDeniedException
     *         You aren't authorized to perform the action. Use the Amazon Resource Name (ARN) of an authorized user or
     *         IAM role to perform the operation.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Textract.
     * @throws InternalServerErrorException
     *         Amazon Textract experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Textract is temporarily unable to process the request. Try your call again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TextractException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TextractClient.DetectDocumentText
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/DetectDocumentText" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DetectDocumentTextResponse detectDocumentText(DetectDocumentTextRequest detectDocumentTextRequest)
            throws InvalidParameterException, InvalidS3ObjectException, UnsupportedDocumentException, DocumentTooLargeException,
            BadDocumentException, AccessDeniedException, ProvisionedThroughputExceededException, InternalServerErrorException,
            ThrottlingException, AwsServiceException, SdkClientException, TextractException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DetectDocumentTextResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DetectDocumentTextResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, detectDocumentTextRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Textract");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DetectDocumentText");

            return clientHandler.execute(new ClientExecutionParams<DetectDocumentTextRequest, DetectDocumentTextResponse>()
                    .withOperationName("DetectDocumentText").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(detectDocumentTextRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DetectDocumentTextRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the results for an Amazon Textract asynchronous operation that analyzes text in a document.
     * </p>
     * <p>
     * You start asynchronous text analysis by calling <a>StartDocumentAnalysis</a>, which returns a job identifier (
     * <code>JobId</code>). When the text analysis operation finishes, Amazon Textract publishes a completion status to
     * the Amazon Simple Notification Service (Amazon SNS) topic that's registered in the initial call to
     * <code>StartDocumentAnalysis</code>. To get the results of the text-detection operation, first check that the
     * status value published to the Amazon SNS topic is <code>SUCCEEDED</code>. If so, call
     * <code>GetDocumentAnalysis</code>, and pass the job identifier (<code>JobId</code>) from the initial call to
     * <code>StartDocumentAnalysis</code>.
     * </p>
     * <p>
     * <code>GetDocumentAnalysis</code> returns an array of <a>Block</a> objects. The following types of information are
     * returned:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Form data (key-value pairs). The related information is returned in two <a>Block</a> objects, each of type
     * <code>KEY_VALUE_SET</code>: a KEY <code>Block</code> object and a VALUE <code>Block</code> object. For example,
     * <i>Name: Ana Silva Carolina</i> contains a key and value. <i>Name:</i> is the key. <i>Ana Silva Carolina</i> is
     * the value.
     * </p>
     * </li>
     * <li>
     * <p>
     * Table and table cell data. A TABLE <code>Block</code> object contains information about a detected table. A CELL
     * <code>Block</code> object is returned for each cell in a table.
     * </p>
     * </li>
     * <li>
     * <p>
     * Lines and words of text. A LINE <code>Block</code> object contains one or more WORD <code>Block</code> objects.
     * All lines and words that are detected in the document are returned (including text that doesn't have a
     * relationship with the value of the <code>StartDocumentAnalysis</code> <code>FeatureTypes</code> input parameter).
     * </p>
     * </li>
     * </ul>
     * <p>
     * Selection elements such as check boxes and option buttons (radio buttons) can be detected in form data and in
     * tables. A SELECTION_ELEMENT <code>Block</code> object contains information about a selection element, including
     * the selection status.
     * </p>
     * <p>
     * Use the <code>MaxResults</code> parameter to limit the number of blocks that are returned. If there are more
     * results than specified in <code>MaxResults</code>, the value of <code>NextToken</code> in the operation response
     * contains a pagination token for getting the next set of results. To get the next page of results, call
     * <code>GetDocumentAnalysis</code>, and populate the <code>NextToken</code> request parameter with the token value
     * that's returned from the previous call to <code>GetDocumentAnalysis</code>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/textract/latest/dg/how-it-works-analyzing.html">Document Text Analysis</a>.
     * </p>
     *
     * @param getDocumentAnalysisRequest
     * @return Result of the GetDocumentAnalysis operation returned by the service.
     * @throws InvalidParameterException
     *         An input parameter violated a constraint. For example, in synchronous operations, an
     *         <code>InvalidParameterException</code> exception occurs when neither of the <code>S3Object</code> or
     *         <code>Bytes</code> values are supplied in the <code>Document</code> request parameter. Validate your
     *         parameter before calling the API operation again.
     * @throws AccessDeniedException
     *         You aren't authorized to perform the action. Use the Amazon Resource Name (ARN) of an authorized user or
     *         IAM role to perform the operation.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Textract.
     * @throws InvalidJobIdException
     *         An invalid job identifier was passed to <a>GetDocumentAnalysis</a> or to <a>GetDocumentAnalysis</a>.
     * @throws InternalServerErrorException
     *         Amazon Textract experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Textract is temporarily unable to process the request. Try your call again.
     * @throws InvalidS3ObjectException
     *         Amazon Textract is unable to access the S3 object that's specified in the request. for more information,
     *         <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-access-control.html">Configure Access to
     *         Amazon S3</a> For troubleshooting information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/troubleshooting.html">Troubleshooting Amazon S3</a>
     * @throws InvalidKmsKeyException
     *         Indicates you do not have decrypt permissions with the KMS key entered, or the KMS key was entered
     *         incorrectly.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TextractException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TextractClient.GetDocumentAnalysis
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/GetDocumentAnalysis" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetDocumentAnalysisResponse getDocumentAnalysis(GetDocumentAnalysisRequest getDocumentAnalysisRequest)
            throws InvalidParameterException, AccessDeniedException, ProvisionedThroughputExceededException,
            InvalidJobIdException, InternalServerErrorException, ThrottlingException, InvalidS3ObjectException,
            InvalidKmsKeyException, AwsServiceException, SdkClientException, TextractException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetDocumentAnalysisResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetDocumentAnalysisResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDocumentAnalysisRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Textract");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDocumentAnalysis");

            return clientHandler.execute(new ClientExecutionParams<GetDocumentAnalysisRequest, GetDocumentAnalysisResponse>()
                    .withOperationName("GetDocumentAnalysis").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getDocumentAnalysisRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetDocumentAnalysisRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the results for an Amazon Textract asynchronous operation that detects text in a document. Amazon Textract
     * can detect lines of text and the words that make up a line of text.
     * </p>
     * <p>
     * You start asynchronous text detection by calling <a>StartDocumentTextDetection</a>, which returns a job
     * identifier (<code>JobId</code>). When the text detection operation finishes, Amazon Textract publishes a
     * completion status to the Amazon Simple Notification Service (Amazon SNS) topic that's registered in the initial
     * call to <code>StartDocumentTextDetection</code>. To get the results of the text-detection operation, first check
     * that the status value published to the Amazon SNS topic is <code>SUCCEEDED</code>. If so, call
     * <code>GetDocumentTextDetection</code>, and pass the job identifier (<code>JobId</code>) from the initial call to
     * <code>StartDocumentTextDetection</code>.
     * </p>
     * <p>
     * <code>GetDocumentTextDetection</code> returns an array of <a>Block</a> objects.
     * </p>
     * <p>
     * Each document page has as an associated <code>Block</code> of type PAGE. Each PAGE <code>Block</code> object is
     * the parent of LINE <code>Block</code> objects that represent the lines of detected text on a page. A LINE
     * <code>Block</code> object is a parent for each word that makes up the line. Words are represented by
     * <code>Block</code> objects of type WORD.
     * </p>
     * <p>
     * Use the MaxResults parameter to limit the number of blocks that are returned. If there are more results than
     * specified in <code>MaxResults</code>, the value of <code>NextToken</code> in the operation response contains a
     * pagination token for getting the next set of results. To get the next page of results, call
     * <code>GetDocumentTextDetection</code>, and populate the <code>NextToken</code> request parameter with the token
     * value that's returned from the previous call to <code>GetDocumentTextDetection</code>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/textract/latest/dg/how-it-works-detecting.html">Document Text Detection</a>.
     * </p>
     *
     * @param getDocumentTextDetectionRequest
     * @return Result of the GetDocumentTextDetection operation returned by the service.
     * @throws InvalidParameterException
     *         An input parameter violated a constraint. For example, in synchronous operations, an
     *         <code>InvalidParameterException</code> exception occurs when neither of the <code>S3Object</code> or
     *         <code>Bytes</code> values are supplied in the <code>Document</code> request parameter. Validate your
     *         parameter before calling the API operation again.
     * @throws AccessDeniedException
     *         You aren't authorized to perform the action. Use the Amazon Resource Name (ARN) of an authorized user or
     *         IAM role to perform the operation.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Textract.
     * @throws InvalidJobIdException
     *         An invalid job identifier was passed to <a>GetDocumentAnalysis</a> or to <a>GetDocumentAnalysis</a>.
     * @throws InternalServerErrorException
     *         Amazon Textract experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Textract is temporarily unable to process the request. Try your call again.
     * @throws InvalidS3ObjectException
     *         Amazon Textract is unable to access the S3 object that's specified in the request. for more information,
     *         <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-access-control.html">Configure Access to
     *         Amazon S3</a> For troubleshooting information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/troubleshooting.html">Troubleshooting Amazon S3</a>
     * @throws InvalidKmsKeyException
     *         Indicates you do not have decrypt permissions with the KMS key entered, or the KMS key was entered
     *         incorrectly.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TextractException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TextractClient.GetDocumentTextDetection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/GetDocumentTextDetection"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetDocumentTextDetectionResponse getDocumentTextDetection(
            GetDocumentTextDetectionRequest getDocumentTextDetectionRequest) throws InvalidParameterException,
            AccessDeniedException, ProvisionedThroughputExceededException, InvalidJobIdException, InternalServerErrorException,
            ThrottlingException, InvalidS3ObjectException, InvalidKmsKeyException, AwsServiceException, SdkClientException,
            TextractException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetDocumentTextDetectionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetDocumentTextDetectionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDocumentTextDetectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Textract");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDocumentTextDetection");

            return clientHandler
                    .execute(new ClientExecutionParams<GetDocumentTextDetectionRequest, GetDocumentTextDetectionResponse>()
                            .withOperationName("GetDocumentTextDetection").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(getDocumentTextDetectionRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetDocumentTextDetectionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the results for an Amazon Textract asynchronous operation that analyzes invoices and receipts. Amazon
     * Textract finds contact information, items purchased, and vendor name, from input invoices and receipts.
     * </p>
     * <p>
     * You start asynchronous invoice/receipt analysis by calling <a>StartExpenseAnalysis</a>, which returns a job
     * identifier (<code>JobId</code>). Upon completion of the invoice/receipt analysis, Amazon Textract publishes the
     * completion status to the Amazon Simple Notification Service (Amazon SNS) topic. This topic must be registered in
     * the initial call to <code>StartExpenseAnalysis</code>. To get the results of the invoice/receipt analysis
     * operation, first ensure that the status value published to the Amazon SNS topic is <code>SUCCEEDED</code>. If so,
     * call <code>GetExpenseAnalysis</code>, and pass the job identifier (<code>JobId</code>) from the initial call to
     * <code>StartExpenseAnalysis</code>.
     * </p>
     * <p>
     * Use the MaxResults parameter to limit the number of blocks that are returned. If there are more results than
     * specified in <code>MaxResults</code>, the value of <code>NextToken</code> in the operation response contains a
     * pagination token for getting the next set of results. To get the next page of results, call
     * <code>GetExpenseAnalysis</code>, and populate the <code>NextToken</code> request parameter with the token value
     * that's returned from the previous call to <code>GetExpenseAnalysis</code>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/textract/latest/dg/invoices-receipts.html">Analyzing Invoices and Receipts</a>.
     * </p>
     *
     * @param getExpenseAnalysisRequest
     * @return Result of the GetExpenseAnalysis operation returned by the service.
     * @throws InvalidParameterException
     *         An input parameter violated a constraint. For example, in synchronous operations, an
     *         <code>InvalidParameterException</code> exception occurs when neither of the <code>S3Object</code> or
     *         <code>Bytes</code> values are supplied in the <code>Document</code> request parameter. Validate your
     *         parameter before calling the API operation again.
     * @throws AccessDeniedException
     *         You aren't authorized to perform the action. Use the Amazon Resource Name (ARN) of an authorized user or
     *         IAM role to perform the operation.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Textract.
     * @throws InvalidJobIdException
     *         An invalid job identifier was passed to <a>GetDocumentAnalysis</a> or to <a>GetDocumentAnalysis</a>.
     * @throws InternalServerErrorException
     *         Amazon Textract experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Textract is temporarily unable to process the request. Try your call again.
     * @throws InvalidS3ObjectException
     *         Amazon Textract is unable to access the S3 object that's specified in the request. for more information,
     *         <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-access-control.html">Configure Access to
     *         Amazon S3</a> For troubleshooting information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/troubleshooting.html">Troubleshooting Amazon S3</a>
     * @throws InvalidKmsKeyException
     *         Indicates you do not have decrypt permissions with the KMS key entered, or the KMS key was entered
     *         incorrectly.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TextractException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TextractClient.GetExpenseAnalysis
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/GetExpenseAnalysis" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetExpenseAnalysisResponse getExpenseAnalysis(GetExpenseAnalysisRequest getExpenseAnalysisRequest)
            throws InvalidParameterException, AccessDeniedException, ProvisionedThroughputExceededException,
            InvalidJobIdException, InternalServerErrorException, ThrottlingException, InvalidS3ObjectException,
            InvalidKmsKeyException, AwsServiceException, SdkClientException, TextractException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetExpenseAnalysisResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetExpenseAnalysisResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getExpenseAnalysisRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Textract");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetExpenseAnalysis");

            return clientHandler.execute(new ClientExecutionParams<GetExpenseAnalysisRequest, GetExpenseAnalysisResponse>()
                    .withOperationName("GetExpenseAnalysis").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getExpenseAnalysisRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetExpenseAnalysisRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Starts the asynchronous analysis of an input document for relationships between detected items such as key-value
     * pairs, tables, and selection elements.
     * </p>
     * <p>
     * <code>StartDocumentAnalysis</code> can analyze text in documents that are in JPEG, PNG, TIFF, and PDF format. The
     * documents are stored in an Amazon S3 bucket. Use <a>DocumentLocation</a> to specify the bucket name and file name
     * of the document.
     * </p>
     * <p>
     * <code>StartDocumentAnalysis</code> returns a job identifier (<code>JobId</code>) that you use to get the results
     * of the operation. When text analysis is finished, Amazon Textract publishes a completion status to the Amazon
     * Simple Notification Service (Amazon SNS) topic that you specify in <code>NotificationChannel</code>. To get the
     * results of the text analysis operation, first check that the status value published to the Amazon SNS topic is
     * <code>SUCCEEDED</code>. If so, call <a>GetDocumentAnalysis</a>, and pass the job identifier (<code>JobId</code>)
     * from the initial call to <code>StartDocumentAnalysis</code>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/textract/latest/dg/how-it-works-analyzing.html">Document Text Analysis</a>.
     * </p>
     *
     * @param startDocumentAnalysisRequest
     * @return Result of the StartDocumentAnalysis operation returned by the service.
     * @throws InvalidParameterException
     *         An input parameter violated a constraint. For example, in synchronous operations, an
     *         <code>InvalidParameterException</code> exception occurs when neither of the <code>S3Object</code> or
     *         <code>Bytes</code> values are supplied in the <code>Document</code> request parameter. Validate your
     *         parameter before calling the API operation again.
     * @throws InvalidS3ObjectException
     *         Amazon Textract is unable to access the S3 object that's specified in the request. for more information,
     *         <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-access-control.html">Configure Access to
     *         Amazon S3</a> For troubleshooting information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/troubleshooting.html">Troubleshooting Amazon S3</a>
     * @throws InvalidKmsKeyException
     *         Indicates you do not have decrypt permissions with the KMS key entered, or the KMS key was entered
     *         incorrectly.
     * @throws UnsupportedDocumentException
     *         The format of the input document isn't supported. Documents for synchronous operations can be in PNG or
     *         JPEG format only. Documents for asynchronous operations can be in PDF format.
     * @throws DocumentTooLargeException
     *         The document can't be processed because it's too large. The maximum document size for synchronous
     *         operations 10 MB. The maximum document size for asynchronous operations is 500 MB for PDF files.
     * @throws BadDocumentException
     *         Amazon Textract isn't able to read the document. For more information on the document limits in Amazon
     *         Textract, see <a>limits</a>.
     * @throws AccessDeniedException
     *         You aren't authorized to perform the action. Use the Amazon Resource Name (ARN) of an authorized user or
     *         IAM role to perform the operation.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Textract.
     * @throws InternalServerErrorException
     *         Amazon Textract experienced a service issue. Try your call again.
     * @throws IdempotentParameterMismatchException
     *         A <code>ClientRequestToken</code> input parameter was reused with an operation, but at least one of the
     *         other input parameters is different from the previous call to the operation.
     * @throws ThrottlingException
     *         Amazon Textract is temporarily unable to process the request. Try your call again.
     * @throws LimitExceededException
     *         An Amazon Textract service limit was exceeded. For example, if you start too many asynchronous jobs
     *         concurrently, calls to start operations (<code>StartDocumentTextDetection</code>, for example) raise a
     *         LimitExceededException exception (HTTP status code: 400) until the number of concurrently running jobs is
     *         below the Amazon Textract service limit.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TextractException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TextractClient.StartDocumentAnalysis
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/StartDocumentAnalysis"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StartDocumentAnalysisResponse startDocumentAnalysis(StartDocumentAnalysisRequest startDocumentAnalysisRequest)
            throws InvalidParameterException, InvalidS3ObjectException, InvalidKmsKeyException, UnsupportedDocumentException,
            DocumentTooLargeException, BadDocumentException, AccessDeniedException, ProvisionedThroughputExceededException,
            InternalServerErrorException, IdempotentParameterMismatchException, ThrottlingException, LimitExceededException,
            AwsServiceException, SdkClientException, TextractException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartDocumentAnalysisResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StartDocumentAnalysisResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startDocumentAnalysisRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Textract");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartDocumentAnalysis");

            return clientHandler.execute(new ClientExecutionParams<StartDocumentAnalysisRequest, StartDocumentAnalysisResponse>()
                    .withOperationName("StartDocumentAnalysis").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(startDocumentAnalysisRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartDocumentAnalysisRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Starts the asynchronous detection of text in a document. Amazon Textract can detect lines of text and the words
     * that make up a line of text.
     * </p>
     * <p>
     * <code>StartDocumentTextDetection</code> can analyze text in documents that are in JPEG, PNG, TIFF, and PDF
     * format. The documents are stored in an Amazon S3 bucket. Use <a>DocumentLocation</a> to specify the bucket name
     * and file name of the document.
     * </p>
     * <p>
     * <code>StartTextDetection</code> returns a job identifier (<code>JobId</code>) that you use to get the results of
     * the operation. When text detection is finished, Amazon Textract publishes a completion status to the Amazon
     * Simple Notification Service (Amazon SNS) topic that you specify in <code>NotificationChannel</code>. To get the
     * results of the text detection operation, first check that the status value published to the Amazon SNS topic is
     * <code>SUCCEEDED</code>. If so, call <a>GetDocumentTextDetection</a>, and pass the job identifier (
     * <code>JobId</code>) from the initial call to <code>StartDocumentTextDetection</code>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/textract/latest/dg/how-it-works-detecting.html">Document Text Detection</a>.
     * </p>
     *
     * @param startDocumentTextDetectionRequest
     * @return Result of the StartDocumentTextDetection operation returned by the service.
     * @throws InvalidParameterException
     *         An input parameter violated a constraint. For example, in synchronous operations, an
     *         <code>InvalidParameterException</code> exception occurs when neither of the <code>S3Object</code> or
     *         <code>Bytes</code> values are supplied in the <code>Document</code> request parameter. Validate your
     *         parameter before calling the API operation again.
     * @throws InvalidS3ObjectException
     *         Amazon Textract is unable to access the S3 object that's specified in the request. for more information,
     *         <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-access-control.html">Configure Access to
     *         Amazon S3</a> For troubleshooting information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/troubleshooting.html">Troubleshooting Amazon S3</a>
     * @throws InvalidKmsKeyException
     *         Indicates you do not have decrypt permissions with the KMS key entered, or the KMS key was entered
     *         incorrectly.
     * @throws UnsupportedDocumentException
     *         The format of the input document isn't supported. Documents for synchronous operations can be in PNG or
     *         JPEG format only. Documents for asynchronous operations can be in PDF format.
     * @throws DocumentTooLargeException
     *         The document can't be processed because it's too large. The maximum document size for synchronous
     *         operations 10 MB. The maximum document size for asynchronous operations is 500 MB for PDF files.
     * @throws BadDocumentException
     *         Amazon Textract isn't able to read the document. For more information on the document limits in Amazon
     *         Textract, see <a>limits</a>.
     * @throws AccessDeniedException
     *         You aren't authorized to perform the action. Use the Amazon Resource Name (ARN) of an authorized user or
     *         IAM role to perform the operation.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Textract.
     * @throws InternalServerErrorException
     *         Amazon Textract experienced a service issue. Try your call again.
     * @throws IdempotentParameterMismatchException
     *         A <code>ClientRequestToken</code> input parameter was reused with an operation, but at least one of the
     *         other input parameters is different from the previous call to the operation.
     * @throws ThrottlingException
     *         Amazon Textract is temporarily unable to process the request. Try your call again.
     * @throws LimitExceededException
     *         An Amazon Textract service limit was exceeded. For example, if you start too many asynchronous jobs
     *         concurrently, calls to start operations (<code>StartDocumentTextDetection</code>, for example) raise a
     *         LimitExceededException exception (HTTP status code: 400) until the number of concurrently running jobs is
     *         below the Amazon Textract service limit.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TextractException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TextractClient.StartDocumentTextDetection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/StartDocumentTextDetection"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StartDocumentTextDetectionResponse startDocumentTextDetection(
            StartDocumentTextDetectionRequest startDocumentTextDetectionRequest) throws InvalidParameterException,
            InvalidS3ObjectException, InvalidKmsKeyException, UnsupportedDocumentException, DocumentTooLargeException,
            BadDocumentException, AccessDeniedException, ProvisionedThroughputExceededException, InternalServerErrorException,
            IdempotentParameterMismatchException, ThrottlingException, LimitExceededException, AwsServiceException,
            SdkClientException, TextractException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartDocumentTextDetectionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StartDocumentTextDetectionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startDocumentTextDetectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Textract");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartDocumentTextDetection");

            return clientHandler
                    .execute(new ClientExecutionParams<StartDocumentTextDetectionRequest, StartDocumentTextDetectionResponse>()
                            .withOperationName("StartDocumentTextDetection").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(startDocumentTextDetectionRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new StartDocumentTextDetectionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Starts the asynchronous analysis of invoices or receipts for data like contact information, items purchased, and
     * vendor names.
     * </p>
     * <p>
     * <code>StartExpenseAnalysis</code> can analyze text in documents that are in JPEG, PNG, and PDF format. The
     * documents must be stored in an Amazon S3 bucket. Use the <a>DocumentLocation</a> parameter to specify the name of
     * your S3 bucket and the name of the document in that bucket.
     * </p>
     * <p>
     * <code>StartExpenseAnalysis</code> returns a job identifier (<code>JobId</code>) that you will provide to
     * <code>GetExpenseAnalysis</code> to retrieve the results of the operation. When the analysis of the input
     * invoices/receipts is finished, Amazon Textract publishes a completion status to the Amazon Simple Notification
     * Service (Amazon SNS) topic that you provide to the <code>NotificationChannel</code>. To obtain the results of the
     * invoice and receipt analysis operation, ensure that the status value published to the Amazon SNS topic is
     * <code>SUCCEEDED</code>. If so, call <a>GetExpenseAnalysis</a>, and pass the job identifier (<code>JobId</code>)
     * that was returned by your call to <code>StartExpenseAnalysis</code>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/textract/latest/dg/invoice-receipts.html">Analyzing Invoices and Receipts</a>.
     * </p>
     *
     * @param startExpenseAnalysisRequest
     * @return Result of the StartExpenseAnalysis operation returned by the service.
     * @throws InvalidParameterException
     *         An input parameter violated a constraint. For example, in synchronous operations, an
     *         <code>InvalidParameterException</code> exception occurs when neither of the <code>S3Object</code> or
     *         <code>Bytes</code> values are supplied in the <code>Document</code> request parameter. Validate your
     *         parameter before calling the API operation again.
     * @throws InvalidS3ObjectException
     *         Amazon Textract is unable to access the S3 object that's specified in the request. for more information,
     *         <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-access-control.html">Configure Access to
     *         Amazon S3</a> For troubleshooting information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/troubleshooting.html">Troubleshooting Amazon S3</a>
     * @throws InvalidKmsKeyException
     *         Indicates you do not have decrypt permissions with the KMS key entered, or the KMS key was entered
     *         incorrectly.
     * @throws UnsupportedDocumentException
     *         The format of the input document isn't supported. Documents for synchronous operations can be in PNG or
     *         JPEG format only. Documents for asynchronous operations can be in PDF format.
     * @throws DocumentTooLargeException
     *         The document can't be processed because it's too large. The maximum document size for synchronous
     *         operations 10 MB. The maximum document size for asynchronous operations is 500 MB for PDF files.
     * @throws BadDocumentException
     *         Amazon Textract isn't able to read the document. For more information on the document limits in Amazon
     *         Textract, see <a>limits</a>.
     * @throws AccessDeniedException
     *         You aren't authorized to perform the action. Use the Amazon Resource Name (ARN) of an authorized user or
     *         IAM role to perform the operation.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Textract.
     * @throws InternalServerErrorException
     *         Amazon Textract experienced a service issue. Try your call again.
     * @throws IdempotentParameterMismatchException
     *         A <code>ClientRequestToken</code> input parameter was reused with an operation, but at least one of the
     *         other input parameters is different from the previous call to the operation.
     * @throws ThrottlingException
     *         Amazon Textract is temporarily unable to process the request. Try your call again.
     * @throws LimitExceededException
     *         An Amazon Textract service limit was exceeded. For example, if you start too many asynchronous jobs
     *         concurrently, calls to start operations (<code>StartDocumentTextDetection</code>, for example) raise a
     *         LimitExceededException exception (HTTP status code: 400) until the number of concurrently running jobs is
     *         below the Amazon Textract service limit.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TextractException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TextractClient.StartExpenseAnalysis
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/StartExpenseAnalysis" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public StartExpenseAnalysisResponse startExpenseAnalysis(StartExpenseAnalysisRequest startExpenseAnalysisRequest)
            throws InvalidParameterException, InvalidS3ObjectException, InvalidKmsKeyException, UnsupportedDocumentException,
            DocumentTooLargeException, BadDocumentException, AccessDeniedException, ProvisionedThroughputExceededException,
            InternalServerErrorException, IdempotentParameterMismatchException, ThrottlingException, LimitExceededException,
            AwsServiceException, SdkClientException, TextractException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartExpenseAnalysisResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StartExpenseAnalysisResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startExpenseAnalysisRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Textract");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartExpenseAnalysis");

            return clientHandler.execute(new ClientExecutionParams<StartExpenseAnalysisRequest, StartExpenseAnalysisResponse>()
                    .withOperationName("StartExpenseAnalysis").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(startExpenseAnalysisRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartExpenseAnalysisRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(TextractException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadDocumentException")
                                .exceptionBuilderSupplier(BadDocumentException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterException")
                                .exceptionBuilderSupplier(InvalidParameterException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("IdempotentParameterMismatchException")
                                .exceptionBuilderSupplier(IdempotentParameterMismatchException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("HumanLoopQuotaExceededException")
                                .exceptionBuilderSupplier(HumanLoopQuotaExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidKMSKeyException")
                                .exceptionBuilderSupplier(InvalidKmsKeyException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DocumentTooLargeException")
                                .exceptionBuilderSupplier(DocumentTooLargeException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ProvisionedThroughputExceededException")
                                .exceptionBuilderSupplier(ProvisionedThroughputExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnsupportedDocumentException")
                                .exceptionBuilderSupplier(UnsupportedDocumentException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidJobIdException")
                                .exceptionBuilderSupplier(InvalidJobIdException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidS3ObjectException")
                                .exceptionBuilderSupplier(InvalidS3ObjectException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerError")
                                .exceptionBuilderSupplier(InternalServerErrorException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
