/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.textract;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsClient;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.regions.ServiceMetadata;
import software.amazon.awssdk.services.textract.model.AccessDeniedException;
import software.amazon.awssdk.services.textract.model.AnalyzeDocumentRequest;
import software.amazon.awssdk.services.textract.model.AnalyzeDocumentResponse;
import software.amazon.awssdk.services.textract.model.AnalyzeExpenseRequest;
import software.amazon.awssdk.services.textract.model.AnalyzeExpenseResponse;
import software.amazon.awssdk.services.textract.model.AnalyzeIdRequest;
import software.amazon.awssdk.services.textract.model.AnalyzeIdResponse;
import software.amazon.awssdk.services.textract.model.BadDocumentException;
import software.amazon.awssdk.services.textract.model.DetectDocumentTextRequest;
import software.amazon.awssdk.services.textract.model.DetectDocumentTextResponse;
import software.amazon.awssdk.services.textract.model.DocumentTooLargeException;
import software.amazon.awssdk.services.textract.model.GetDocumentAnalysisRequest;
import software.amazon.awssdk.services.textract.model.GetDocumentAnalysisResponse;
import software.amazon.awssdk.services.textract.model.GetDocumentTextDetectionRequest;
import software.amazon.awssdk.services.textract.model.GetDocumentTextDetectionResponse;
import software.amazon.awssdk.services.textract.model.GetExpenseAnalysisRequest;
import software.amazon.awssdk.services.textract.model.GetExpenseAnalysisResponse;
import software.amazon.awssdk.services.textract.model.GetLendingAnalysisRequest;
import software.amazon.awssdk.services.textract.model.GetLendingAnalysisResponse;
import software.amazon.awssdk.services.textract.model.GetLendingAnalysisSummaryRequest;
import software.amazon.awssdk.services.textract.model.GetLendingAnalysisSummaryResponse;
import software.amazon.awssdk.services.textract.model.HumanLoopQuotaExceededException;
import software.amazon.awssdk.services.textract.model.IdempotentParameterMismatchException;
import software.amazon.awssdk.services.textract.model.InternalServerErrorException;
import software.amazon.awssdk.services.textract.model.InvalidJobIdException;
import software.amazon.awssdk.services.textract.model.InvalidKmsKeyException;
import software.amazon.awssdk.services.textract.model.InvalidParameterException;
import software.amazon.awssdk.services.textract.model.InvalidS3ObjectException;
import software.amazon.awssdk.services.textract.model.LimitExceededException;
import software.amazon.awssdk.services.textract.model.ProvisionedThroughputExceededException;
import software.amazon.awssdk.services.textract.model.StartDocumentAnalysisRequest;
import software.amazon.awssdk.services.textract.model.StartDocumentAnalysisResponse;
import software.amazon.awssdk.services.textract.model.StartDocumentTextDetectionRequest;
import software.amazon.awssdk.services.textract.model.StartDocumentTextDetectionResponse;
import software.amazon.awssdk.services.textract.model.StartExpenseAnalysisRequest;
import software.amazon.awssdk.services.textract.model.StartExpenseAnalysisResponse;
import software.amazon.awssdk.services.textract.model.StartLendingAnalysisRequest;
import software.amazon.awssdk.services.textract.model.StartLendingAnalysisResponse;
import software.amazon.awssdk.services.textract.model.TextractException;
import software.amazon.awssdk.services.textract.model.ThrottlingException;
import software.amazon.awssdk.services.textract.model.UnsupportedDocumentException;

/**
 * Service client for accessing Amazon Textract. This can be created using the static {@link #builder()} method.
 *
 * <p>
 * Amazon Textract detects and analyzes text in documents and converts it into machine-readable text. This is the API
 * reference documentation for Amazon Textract.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
public interface TextractClient extends AwsClient {
    String SERVICE_NAME = "textract";

    /**
     * Value for looking up the service's metadata from the
     * {@link software.amazon.awssdk.regions.ServiceMetadataProvider}.
     */
    String SERVICE_METADATA_ID = "textract";

    /**
     * <p>
     * Analyzes an input document for relationships between detected items.
     * </p>
     * <p>
     * The types of information returned are as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Form data (key-value pairs). The related information is returned in two <a>Block</a> objects, each of type
     * <code>KEY_VALUE_SET</code>: a KEY <code>Block</code> object and a VALUE <code>Block</code> object. For example,
     * <i>Name: Ana Silva Carolina</i> contains a key and value. <i>Name:</i> is the key. <i>Ana Silva Carolina</i> is
     * the value.
     * </p>
     * </li>
     * <li>
     * <p>
     * Table and table cell data. A TABLE <code>Block</code> object contains information about a detected table. A CELL
     * <code>Block</code> object is returned for each cell in a table.
     * </p>
     * </li>
     * <li>
     * <p>
     * Lines and words of text. A LINE <code>Block</code> object contains one or more WORD <code>Block</code> objects.
     * All lines and words that are detected in the document are returned (including text that doesn't have a
     * relationship with the value of <code>FeatureTypes</code>).
     * </p>
     * </li>
     * <li>
     * <p>
     * Signatures. A SIGNATURE <code>Block</code> object contains the location information of a signature in a document.
     * If used in conjunction with forms or tables, a signature can be given a Key-Value pairing or be detected in the
     * cell of a table.
     * </p>
     * </li>
     * <li>
     * <p>
     * Query. A QUERY Block object contains the query text, alias and link to the associated Query results block object.
     * </p>
     * </li>
     * <li>
     * <p>
     * Query Result. A QUERY_RESULT Block object contains the answer to the query and an ID that connects it to the
     * query asked. This Block also contains a confidence score.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Selection elements such as check boxes and option buttons (radio buttons) can be detected in form data and in
     * tables. A SELECTION_ELEMENT <code>Block</code> object contains information about a selection element, including
     * the selection status.
     * </p>
     * <p>
     * You can choose which type of analysis to perform by specifying the <code>FeatureTypes</code> list.
     * </p>
     * <p>
     * The output is returned in a list of <code>Block</code> objects.
     * </p>
     * <p>
     * <code>AnalyzeDocument</code> is a synchronous operation. To analyze documents asynchronously, use
     * <a>StartDocumentAnalysis</a>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/textract/latest/dg/how-it-works-analyzing.html">Document Text Analysis</a>.
     * </p>
     *
     * @param analyzeDocumentRequest
     * @return Result of the AnalyzeDocument operation returned by the service.
     * @throws InvalidParameterException
     *         An input parameter violated a constraint. For example, in synchronous operations, an
     *         <code>InvalidParameterException</code> exception occurs when neither of the <code>S3Object</code> or
     *         <code>Bytes</code> values are supplied in the <code>Document</code> request parameter. Validate your
     *         parameter before calling the API operation again.
     * @throws InvalidS3ObjectException
     *         Amazon Textract is unable to access the S3 object that's specified in the request. for more information,
     *         <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-access-control.html">Configure Access to
     *         Amazon S3</a> For troubleshooting information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/troubleshooting.html">Troubleshooting Amazon S3</a>
     * @throws UnsupportedDocumentException
     *         The format of the input document isn't supported. Documents for operations can be in PNG, JPEG, PDF, or
     *         TIFF format.
     * @throws DocumentTooLargeException
     *         The document can't be processed because it's too large. The maximum document size for synchronous
     *         operations 10 MB. The maximum document size for asynchronous operations is 500 MB for PDF files.
     * @throws BadDocumentException
     *         Amazon Textract isn't able to read the document. For more information on the document limits in Amazon
     *         Textract, see <a>limits</a>.
     * @throws AccessDeniedException
     *         You aren't authorized to perform the action. Use the Amazon Resource Name (ARN) of an authorized user or
     *         IAM role to perform the operation.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Textract.
     * @throws InternalServerErrorException
     *         Amazon Textract experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Textract is temporarily unable to process the request. Try your call again.
     * @throws HumanLoopQuotaExceededException
     *         Indicates you have exceeded the maximum number of active human in the loop workflows available
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TextractException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TextractClient.AnalyzeDocument
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/AnalyzeDocument" target="_top">AWS API
     *      Documentation</a>
     */
    default AnalyzeDocumentResponse analyzeDocument(AnalyzeDocumentRequest analyzeDocumentRequest)
            throws InvalidParameterException, InvalidS3ObjectException, UnsupportedDocumentException, DocumentTooLargeException,
            BadDocumentException, AccessDeniedException, ProvisionedThroughputExceededException, InternalServerErrorException,
            ThrottlingException, HumanLoopQuotaExceededException, AwsServiceException, SdkClientException, TextractException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Analyzes an input document for relationships between detected items.
     * </p>
     * <p>
     * The types of information returned are as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Form data (key-value pairs). The related information is returned in two <a>Block</a> objects, each of type
     * <code>KEY_VALUE_SET</code>: a KEY <code>Block</code> object and a VALUE <code>Block</code> object. For example,
     * <i>Name: Ana Silva Carolina</i> contains a key and value. <i>Name:</i> is the key. <i>Ana Silva Carolina</i> is
     * the value.
     * </p>
     * </li>
     * <li>
     * <p>
     * Table and table cell data. A TABLE <code>Block</code> object contains information about a detected table. A CELL
     * <code>Block</code> object is returned for each cell in a table.
     * </p>
     * </li>
     * <li>
     * <p>
     * Lines and words of text. A LINE <code>Block</code> object contains one or more WORD <code>Block</code> objects.
     * All lines and words that are detected in the document are returned (including text that doesn't have a
     * relationship with the value of <code>FeatureTypes</code>).
     * </p>
     * </li>
     * <li>
     * <p>
     * Signatures. A SIGNATURE <code>Block</code> object contains the location information of a signature in a document.
     * If used in conjunction with forms or tables, a signature can be given a Key-Value pairing or be detected in the
     * cell of a table.
     * </p>
     * </li>
     * <li>
     * <p>
     * Query. A QUERY Block object contains the query text, alias and link to the associated Query results block object.
     * </p>
     * </li>
     * <li>
     * <p>
     * Query Result. A QUERY_RESULT Block object contains the answer to the query and an ID that connects it to the
     * query asked. This Block also contains a confidence score.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Selection elements such as check boxes and option buttons (radio buttons) can be detected in form data and in
     * tables. A SELECTION_ELEMENT <code>Block</code> object contains information about a selection element, including
     * the selection status.
     * </p>
     * <p>
     * You can choose which type of analysis to perform by specifying the <code>FeatureTypes</code> list.
     * </p>
     * <p>
     * The output is returned in a list of <code>Block</code> objects.
     * </p>
     * <p>
     * <code>AnalyzeDocument</code> is a synchronous operation. To analyze documents asynchronously, use
     * <a>StartDocumentAnalysis</a>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/textract/latest/dg/how-it-works-analyzing.html">Document Text Analysis</a>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link AnalyzeDocumentRequest.Builder} avoiding the need
     * to create one manually via {@link AnalyzeDocumentRequest#builder()}
     * </p>
     *
     * @param analyzeDocumentRequest
     *        A {@link Consumer} that will call methods on {@link AnalyzeDocumentRequest.Builder} to create a request.
     * @return Result of the AnalyzeDocument operation returned by the service.
     * @throws InvalidParameterException
     *         An input parameter violated a constraint. For example, in synchronous operations, an
     *         <code>InvalidParameterException</code> exception occurs when neither of the <code>S3Object</code> or
     *         <code>Bytes</code> values are supplied in the <code>Document</code> request parameter. Validate your
     *         parameter before calling the API operation again.
     * @throws InvalidS3ObjectException
     *         Amazon Textract is unable to access the S3 object that's specified in the request. for more information,
     *         <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-access-control.html">Configure Access to
     *         Amazon S3</a> For troubleshooting information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/troubleshooting.html">Troubleshooting Amazon S3</a>
     * @throws UnsupportedDocumentException
     *         The format of the input document isn't supported. Documents for operations can be in PNG, JPEG, PDF, or
     *         TIFF format.
     * @throws DocumentTooLargeException
     *         The document can't be processed because it's too large. The maximum document size for synchronous
     *         operations 10 MB. The maximum document size for asynchronous operations is 500 MB for PDF files.
     * @throws BadDocumentException
     *         Amazon Textract isn't able to read the document. For more information on the document limits in Amazon
     *         Textract, see <a>limits</a>.
     * @throws AccessDeniedException
     *         You aren't authorized to perform the action. Use the Amazon Resource Name (ARN) of an authorized user or
     *         IAM role to perform the operation.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Textract.
     * @throws InternalServerErrorException
     *         Amazon Textract experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Textract is temporarily unable to process the request. Try your call again.
     * @throws HumanLoopQuotaExceededException
     *         Indicates you have exceeded the maximum number of active human in the loop workflows available
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TextractException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TextractClient.AnalyzeDocument
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/AnalyzeDocument" target="_top">AWS API
     *      Documentation</a>
     */
    default AnalyzeDocumentResponse analyzeDocument(Consumer<AnalyzeDocumentRequest.Builder> analyzeDocumentRequest)
            throws InvalidParameterException, InvalidS3ObjectException, UnsupportedDocumentException, DocumentTooLargeException,
            BadDocumentException, AccessDeniedException, ProvisionedThroughputExceededException, InternalServerErrorException,
            ThrottlingException, HumanLoopQuotaExceededException, AwsServiceException, SdkClientException, TextractException {
        return analyzeDocument(AnalyzeDocumentRequest.builder().applyMutation(analyzeDocumentRequest).build());
    }

    /**
     * <p>
     * <code>AnalyzeExpense</code> synchronously analyzes an input document for financially related relationships
     * between text.
     * </p>
     * <p>
     * Information is returned as <code>ExpenseDocuments</code> and seperated as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>LineItemGroups</code>- A data set containing <code>LineItems</code> which store information about the lines
     * of text, such as an item purchased and its price on a receipt.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SummaryFields</code>- Contains all other information a receipt, such as header information or the vendors
     * name.
     * </p>
     * </li>
     * </ul>
     *
     * @param analyzeExpenseRequest
     * @return Result of the AnalyzeExpense operation returned by the service.
     * @throws InvalidParameterException
     *         An input parameter violated a constraint. For example, in synchronous operations, an
     *         <code>InvalidParameterException</code> exception occurs when neither of the <code>S3Object</code> or
     *         <code>Bytes</code> values are supplied in the <code>Document</code> request parameter. Validate your
     *         parameter before calling the API operation again.
     * @throws InvalidS3ObjectException
     *         Amazon Textract is unable to access the S3 object that's specified in the request. for more information,
     *         <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-access-control.html">Configure Access to
     *         Amazon S3</a> For troubleshooting information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/troubleshooting.html">Troubleshooting Amazon S3</a>
     * @throws UnsupportedDocumentException
     *         The format of the input document isn't supported. Documents for operations can be in PNG, JPEG, PDF, or
     *         TIFF format.
     * @throws DocumentTooLargeException
     *         The document can't be processed because it's too large. The maximum document size for synchronous
     *         operations 10 MB. The maximum document size for asynchronous operations is 500 MB for PDF files.
     * @throws BadDocumentException
     *         Amazon Textract isn't able to read the document. For more information on the document limits in Amazon
     *         Textract, see <a>limits</a>.
     * @throws AccessDeniedException
     *         You aren't authorized to perform the action. Use the Amazon Resource Name (ARN) of an authorized user or
     *         IAM role to perform the operation.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Textract.
     * @throws InternalServerErrorException
     *         Amazon Textract experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Textract is temporarily unable to process the request. Try your call again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TextractException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TextractClient.AnalyzeExpense
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/AnalyzeExpense" target="_top">AWS API
     *      Documentation</a>
     */
    default AnalyzeExpenseResponse analyzeExpense(AnalyzeExpenseRequest analyzeExpenseRequest) throws InvalidParameterException,
            InvalidS3ObjectException, UnsupportedDocumentException, DocumentTooLargeException, BadDocumentException,
            AccessDeniedException, ProvisionedThroughputExceededException, InternalServerErrorException, ThrottlingException,
            AwsServiceException, SdkClientException, TextractException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * <code>AnalyzeExpense</code> synchronously analyzes an input document for financially related relationships
     * between text.
     * </p>
     * <p>
     * Information is returned as <code>ExpenseDocuments</code> and seperated as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>LineItemGroups</code>- A data set containing <code>LineItems</code> which store information about the lines
     * of text, such as an item purchased and its price on a receipt.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SummaryFields</code>- Contains all other information a receipt, such as header information or the vendors
     * name.
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link AnalyzeExpenseRequest.Builder} avoiding the need to
     * create one manually via {@link AnalyzeExpenseRequest#builder()}
     * </p>
     *
     * @param analyzeExpenseRequest
     *        A {@link Consumer} that will call methods on {@link AnalyzeExpenseRequest.Builder} to create a request.
     * @return Result of the AnalyzeExpense operation returned by the service.
     * @throws InvalidParameterException
     *         An input parameter violated a constraint. For example, in synchronous operations, an
     *         <code>InvalidParameterException</code> exception occurs when neither of the <code>S3Object</code> or
     *         <code>Bytes</code> values are supplied in the <code>Document</code> request parameter. Validate your
     *         parameter before calling the API operation again.
     * @throws InvalidS3ObjectException
     *         Amazon Textract is unable to access the S3 object that's specified in the request. for more information,
     *         <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-access-control.html">Configure Access to
     *         Amazon S3</a> For troubleshooting information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/troubleshooting.html">Troubleshooting Amazon S3</a>
     * @throws UnsupportedDocumentException
     *         The format of the input document isn't supported. Documents for operations can be in PNG, JPEG, PDF, or
     *         TIFF format.
     * @throws DocumentTooLargeException
     *         The document can't be processed because it's too large. The maximum document size for synchronous
     *         operations 10 MB. The maximum document size for asynchronous operations is 500 MB for PDF files.
     * @throws BadDocumentException
     *         Amazon Textract isn't able to read the document. For more information on the document limits in Amazon
     *         Textract, see <a>limits</a>.
     * @throws AccessDeniedException
     *         You aren't authorized to perform the action. Use the Amazon Resource Name (ARN) of an authorized user or
     *         IAM role to perform the operation.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Textract.
     * @throws InternalServerErrorException
     *         Amazon Textract experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Textract is temporarily unable to process the request. Try your call again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TextractException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TextractClient.AnalyzeExpense
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/AnalyzeExpense" target="_top">AWS API
     *      Documentation</a>
     */
    default AnalyzeExpenseResponse analyzeExpense(Consumer<AnalyzeExpenseRequest.Builder> analyzeExpenseRequest)
            throws InvalidParameterException, InvalidS3ObjectException, UnsupportedDocumentException, DocumentTooLargeException,
            BadDocumentException, AccessDeniedException, ProvisionedThroughputExceededException, InternalServerErrorException,
            ThrottlingException, AwsServiceException, SdkClientException, TextractException {
        return analyzeExpense(AnalyzeExpenseRequest.builder().applyMutation(analyzeExpenseRequest).build());
    }

    /**
     * <p>
     * Analyzes identity documents for relevant information. This information is extracted and returned as
     * <code>IdentityDocumentFields</code>, which records both the normalized field and value of the extracted text.
     * Unlike other Amazon Textract operations, <code>AnalyzeID</code> doesn't return any Geometry data.
     * </p>
     *
     * @param analyzeIdRequest
     * @return Result of the AnalyzeID operation returned by the service.
     * @throws InvalidParameterException
     *         An input parameter violated a constraint. For example, in synchronous operations, an
     *         <code>InvalidParameterException</code> exception occurs when neither of the <code>S3Object</code> or
     *         <code>Bytes</code> values are supplied in the <code>Document</code> request parameter. Validate your
     *         parameter before calling the API operation again.
     * @throws InvalidS3ObjectException
     *         Amazon Textract is unable to access the S3 object that's specified in the request. for more information,
     *         <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-access-control.html">Configure Access to
     *         Amazon S3</a> For troubleshooting information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/troubleshooting.html">Troubleshooting Amazon S3</a>
     * @throws UnsupportedDocumentException
     *         The format of the input document isn't supported. Documents for operations can be in PNG, JPEG, PDF, or
     *         TIFF format.
     * @throws DocumentTooLargeException
     *         The document can't be processed because it's too large. The maximum document size for synchronous
     *         operations 10 MB. The maximum document size for asynchronous operations is 500 MB for PDF files.
     * @throws BadDocumentException
     *         Amazon Textract isn't able to read the document. For more information on the document limits in Amazon
     *         Textract, see <a>limits</a>.
     * @throws AccessDeniedException
     *         You aren't authorized to perform the action. Use the Amazon Resource Name (ARN) of an authorized user or
     *         IAM role to perform the operation.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Textract.
     * @throws InternalServerErrorException
     *         Amazon Textract experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Textract is temporarily unable to process the request. Try your call again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TextractException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TextractClient.AnalyzeID
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/AnalyzeID" target="_top">AWS API
     *      Documentation</a>
     */
    default AnalyzeIdResponse analyzeID(AnalyzeIdRequest analyzeIdRequest) throws InvalidParameterException,
            InvalidS3ObjectException, UnsupportedDocumentException, DocumentTooLargeException, BadDocumentException,
            AccessDeniedException, ProvisionedThroughputExceededException, InternalServerErrorException, ThrottlingException,
            AwsServiceException, SdkClientException, TextractException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Analyzes identity documents for relevant information. This information is extracted and returned as
     * <code>IdentityDocumentFields</code>, which records both the normalized field and value of the extracted text.
     * Unlike other Amazon Textract operations, <code>AnalyzeID</code> doesn't return any Geometry data.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link AnalyzeIdRequest.Builder} avoiding the need to
     * create one manually via {@link AnalyzeIdRequest#builder()}
     * </p>
     *
     * @param analyzeIdRequest
     *        A {@link Consumer} that will call methods on {@link AnalyzeIDRequest.Builder} to create a request.
     * @return Result of the AnalyzeID operation returned by the service.
     * @throws InvalidParameterException
     *         An input parameter violated a constraint. For example, in synchronous operations, an
     *         <code>InvalidParameterException</code> exception occurs when neither of the <code>S3Object</code> or
     *         <code>Bytes</code> values are supplied in the <code>Document</code> request parameter. Validate your
     *         parameter before calling the API operation again.
     * @throws InvalidS3ObjectException
     *         Amazon Textract is unable to access the S3 object that's specified in the request. for more information,
     *         <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-access-control.html">Configure Access to
     *         Amazon S3</a> For troubleshooting information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/troubleshooting.html">Troubleshooting Amazon S3</a>
     * @throws UnsupportedDocumentException
     *         The format of the input document isn't supported. Documents for operations can be in PNG, JPEG, PDF, or
     *         TIFF format.
     * @throws DocumentTooLargeException
     *         The document can't be processed because it's too large. The maximum document size for synchronous
     *         operations 10 MB. The maximum document size for asynchronous operations is 500 MB for PDF files.
     * @throws BadDocumentException
     *         Amazon Textract isn't able to read the document. For more information on the document limits in Amazon
     *         Textract, see <a>limits</a>.
     * @throws AccessDeniedException
     *         You aren't authorized to perform the action. Use the Amazon Resource Name (ARN) of an authorized user or
     *         IAM role to perform the operation.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Textract.
     * @throws InternalServerErrorException
     *         Amazon Textract experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Textract is temporarily unable to process the request. Try your call again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TextractException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TextractClient.AnalyzeID
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/AnalyzeID" target="_top">AWS API
     *      Documentation</a>
     */
    default AnalyzeIdResponse analyzeID(Consumer<AnalyzeIdRequest.Builder> analyzeIdRequest) throws InvalidParameterException,
            InvalidS3ObjectException, UnsupportedDocumentException, DocumentTooLargeException, BadDocumentException,
            AccessDeniedException, ProvisionedThroughputExceededException, InternalServerErrorException, ThrottlingException,
            AwsServiceException, SdkClientException, TextractException {
        return analyzeID(AnalyzeIdRequest.builder().applyMutation(analyzeIdRequest).build());
    }

    /**
     * <p>
     * Detects text in the input document. Amazon Textract can detect lines of text and the words that make up a line of
     * text. The input document must be in one of the following image formats: JPEG, PNG, PDF, or TIFF.
     * <code>DetectDocumentText</code> returns the detected text in an array of <a>Block</a> objects.
     * </p>
     * <p>
     * Each document page has as an associated <code>Block</code> of type PAGE. Each PAGE <code>Block</code> object is
     * the parent of LINE <code>Block</code> objects that represent the lines of detected text on a page. A LINE
     * <code>Block</code> object is a parent for each word that makes up the line. Words are represented by
     * <code>Block</code> objects of type WORD.
     * </p>
     * <p>
     * <code>DetectDocumentText</code> is a synchronous operation. To analyze documents asynchronously, use
     * <a>StartDocumentTextDetection</a>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/textract/latest/dg/how-it-works-detecting.html">Document Text Detection</a>.
     * </p>
     *
     * @param detectDocumentTextRequest
     * @return Result of the DetectDocumentText operation returned by the service.
     * @throws InvalidParameterException
     *         An input parameter violated a constraint. For example, in synchronous operations, an
     *         <code>InvalidParameterException</code> exception occurs when neither of the <code>S3Object</code> or
     *         <code>Bytes</code> values are supplied in the <code>Document</code> request parameter. Validate your
     *         parameter before calling the API operation again.
     * @throws InvalidS3ObjectException
     *         Amazon Textract is unable to access the S3 object that's specified in the request. for more information,
     *         <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-access-control.html">Configure Access to
     *         Amazon S3</a> For troubleshooting information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/troubleshooting.html">Troubleshooting Amazon S3</a>
     * @throws UnsupportedDocumentException
     *         The format of the input document isn't supported. Documents for operations can be in PNG, JPEG, PDF, or
     *         TIFF format.
     * @throws DocumentTooLargeException
     *         The document can't be processed because it's too large. The maximum document size for synchronous
     *         operations 10 MB. The maximum document size for asynchronous operations is 500 MB for PDF files.
     * @throws BadDocumentException
     *         Amazon Textract isn't able to read the document. For more information on the document limits in Amazon
     *         Textract, see <a>limits</a>.
     * @throws AccessDeniedException
     *         You aren't authorized to perform the action. Use the Amazon Resource Name (ARN) of an authorized user or
     *         IAM role to perform the operation.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Textract.
     * @throws InternalServerErrorException
     *         Amazon Textract experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Textract is temporarily unable to process the request. Try your call again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TextractException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TextractClient.DetectDocumentText
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/DetectDocumentText" target="_top">AWS
     *      API Documentation</a>
     */
    default DetectDocumentTextResponse detectDocumentText(DetectDocumentTextRequest detectDocumentTextRequest)
            throws InvalidParameterException, InvalidS3ObjectException, UnsupportedDocumentException, DocumentTooLargeException,
            BadDocumentException, AccessDeniedException, ProvisionedThroughputExceededException, InternalServerErrorException,
            ThrottlingException, AwsServiceException, SdkClientException, TextractException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Detects text in the input document. Amazon Textract can detect lines of text and the words that make up a line of
     * text. The input document must be in one of the following image formats: JPEG, PNG, PDF, or TIFF.
     * <code>DetectDocumentText</code> returns the detected text in an array of <a>Block</a> objects.
     * </p>
     * <p>
     * Each document page has as an associated <code>Block</code> of type PAGE. Each PAGE <code>Block</code> object is
     * the parent of LINE <code>Block</code> objects that represent the lines of detected text on a page. A LINE
     * <code>Block</code> object is a parent for each word that makes up the line. Words are represented by
     * <code>Block</code> objects of type WORD.
     * </p>
     * <p>
     * <code>DetectDocumentText</code> is a synchronous operation. To analyze documents asynchronously, use
     * <a>StartDocumentTextDetection</a>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/textract/latest/dg/how-it-works-detecting.html">Document Text Detection</a>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DetectDocumentTextRequest.Builder} avoiding the
     * need to create one manually via {@link DetectDocumentTextRequest#builder()}
     * </p>
     *
     * @param detectDocumentTextRequest
     *        A {@link Consumer} that will call methods on {@link DetectDocumentTextRequest.Builder} to create a
     *        request.
     * @return Result of the DetectDocumentText operation returned by the service.
     * @throws InvalidParameterException
     *         An input parameter violated a constraint. For example, in synchronous operations, an
     *         <code>InvalidParameterException</code> exception occurs when neither of the <code>S3Object</code> or
     *         <code>Bytes</code> values are supplied in the <code>Document</code> request parameter. Validate your
     *         parameter before calling the API operation again.
     * @throws InvalidS3ObjectException
     *         Amazon Textract is unable to access the S3 object that's specified in the request. for more information,
     *         <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-access-control.html">Configure Access to
     *         Amazon S3</a> For troubleshooting information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/troubleshooting.html">Troubleshooting Amazon S3</a>
     * @throws UnsupportedDocumentException
     *         The format of the input document isn't supported. Documents for operations can be in PNG, JPEG, PDF, or
     *         TIFF format.
     * @throws DocumentTooLargeException
     *         The document can't be processed because it's too large. The maximum document size for synchronous
     *         operations 10 MB. The maximum document size for asynchronous operations is 500 MB for PDF files.
     * @throws BadDocumentException
     *         Amazon Textract isn't able to read the document. For more information on the document limits in Amazon
     *         Textract, see <a>limits</a>.
     * @throws AccessDeniedException
     *         You aren't authorized to perform the action. Use the Amazon Resource Name (ARN) of an authorized user or
     *         IAM role to perform the operation.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Textract.
     * @throws InternalServerErrorException
     *         Amazon Textract experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Textract is temporarily unable to process the request. Try your call again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TextractException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TextractClient.DetectDocumentText
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/DetectDocumentText" target="_top">AWS
     *      API Documentation</a>
     */
    default DetectDocumentTextResponse detectDocumentText(Consumer<DetectDocumentTextRequest.Builder> detectDocumentTextRequest)
            throws InvalidParameterException, InvalidS3ObjectException, UnsupportedDocumentException, DocumentTooLargeException,
            BadDocumentException, AccessDeniedException, ProvisionedThroughputExceededException, InternalServerErrorException,
            ThrottlingException, AwsServiceException, SdkClientException, TextractException {
        return detectDocumentText(DetectDocumentTextRequest.builder().applyMutation(detectDocumentTextRequest).build());
    }

    /**
     * <p>
     * Gets the results for an Amazon Textract asynchronous operation that analyzes text in a document.
     * </p>
     * <p>
     * You start asynchronous text analysis by calling <a>StartDocumentAnalysis</a>, which returns a job identifier (
     * <code>JobId</code>). When the text analysis operation finishes, Amazon Textract publishes a completion status to
     * the Amazon Simple Notification Service (Amazon SNS) topic that's registered in the initial call to
     * <code>StartDocumentAnalysis</code>. To get the results of the text-detection operation, first check that the
     * status value published to the Amazon SNS topic is <code>SUCCEEDED</code>. If so, call
     * <code>GetDocumentAnalysis</code>, and pass the job identifier (<code>JobId</code>) from the initial call to
     * <code>StartDocumentAnalysis</code>.
     * </p>
     * <p>
     * <code>GetDocumentAnalysis</code> returns an array of <a>Block</a> objects. The following types of information are
     * returned:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Form data (key-value pairs). The related information is returned in two <a>Block</a> objects, each of type
     * <code>KEY_VALUE_SET</code>: a KEY <code>Block</code> object and a VALUE <code>Block</code> object. For example,
     * <i>Name: Ana Silva Carolina</i> contains a key and value. <i>Name:</i> is the key. <i>Ana Silva Carolina</i> is
     * the value.
     * </p>
     * </li>
     * <li>
     * <p>
     * Table and table cell data. A TABLE <code>Block</code> object contains information about a detected table. A CELL
     * <code>Block</code> object is returned for each cell in a table.
     * </p>
     * </li>
     * <li>
     * <p>
     * Lines and words of text. A LINE <code>Block</code> object contains one or more WORD <code>Block</code> objects.
     * All lines and words that are detected in the document are returned (including text that doesn't have a
     * relationship with the value of the <code>StartDocumentAnalysis</code> <code>FeatureTypes</code> input parameter).
     * </p>
     * </li>
     * <li>
     * <p>
     * Query. A QUERY Block object contains the query text, alias and link to the associated Query results block object.
     * </p>
     * </li>
     * <li>
     * <p>
     * Query Results. A QUERY_RESULT Block object contains the answer to the query and an ID that connects it to the
     * query asked. This Block also contains a confidence score.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * While processing a document with queries, look out for <code>INVALID_REQUEST_PARAMETERS</code> output. This
     * indicates that either the per page query limit has been exceeded or that the operation is trying to query a page
     * in the document which doesn’t exist.
     * </p>
     * </note>
     * <p>
     * Selection elements such as check boxes and option buttons (radio buttons) can be detected in form data and in
     * tables. A SELECTION_ELEMENT <code>Block</code> object contains information about a selection element, including
     * the selection status.
     * </p>
     * <p>
     * Use the <code>MaxResults</code> parameter to limit the number of blocks that are returned. If there are more
     * results than specified in <code>MaxResults</code>, the value of <code>NextToken</code> in the operation response
     * contains a pagination token for getting the next set of results. To get the next page of results, call
     * <code>GetDocumentAnalysis</code>, and populate the <code>NextToken</code> request parameter with the token value
     * that's returned from the previous call to <code>GetDocumentAnalysis</code>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/textract/latest/dg/how-it-works-analyzing.html">Document Text Analysis</a>.
     * </p>
     *
     * @param getDocumentAnalysisRequest
     * @return Result of the GetDocumentAnalysis operation returned by the service.
     * @throws InvalidParameterException
     *         An input parameter violated a constraint. For example, in synchronous operations, an
     *         <code>InvalidParameterException</code> exception occurs when neither of the <code>S3Object</code> or
     *         <code>Bytes</code> values are supplied in the <code>Document</code> request parameter. Validate your
     *         parameter before calling the API operation again.
     * @throws AccessDeniedException
     *         You aren't authorized to perform the action. Use the Amazon Resource Name (ARN) of an authorized user or
     *         IAM role to perform the operation.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Textract.
     * @throws InvalidJobIdException
     *         An invalid job identifier was passed to an asynchronous analysis operation.
     * @throws InternalServerErrorException
     *         Amazon Textract experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Textract is temporarily unable to process the request. Try your call again.
     * @throws InvalidS3ObjectException
     *         Amazon Textract is unable to access the S3 object that's specified in the request. for more information,
     *         <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-access-control.html">Configure Access to
     *         Amazon S3</a> For troubleshooting information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/troubleshooting.html">Troubleshooting Amazon S3</a>
     * @throws InvalidKmsKeyException
     *         Indicates you do not have decrypt permissions with the KMS key entered, or the KMS key was entered
     *         incorrectly.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TextractException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TextractClient.GetDocumentAnalysis
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/GetDocumentAnalysis" target="_top">AWS
     *      API Documentation</a>
     */
    default GetDocumentAnalysisResponse getDocumentAnalysis(GetDocumentAnalysisRequest getDocumentAnalysisRequest)
            throws InvalidParameterException, AccessDeniedException, ProvisionedThroughputExceededException,
            InvalidJobIdException, InternalServerErrorException, ThrottlingException, InvalidS3ObjectException,
            InvalidKmsKeyException, AwsServiceException, SdkClientException, TextractException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Gets the results for an Amazon Textract asynchronous operation that analyzes text in a document.
     * </p>
     * <p>
     * You start asynchronous text analysis by calling <a>StartDocumentAnalysis</a>, which returns a job identifier (
     * <code>JobId</code>). When the text analysis operation finishes, Amazon Textract publishes a completion status to
     * the Amazon Simple Notification Service (Amazon SNS) topic that's registered in the initial call to
     * <code>StartDocumentAnalysis</code>. To get the results of the text-detection operation, first check that the
     * status value published to the Amazon SNS topic is <code>SUCCEEDED</code>. If so, call
     * <code>GetDocumentAnalysis</code>, and pass the job identifier (<code>JobId</code>) from the initial call to
     * <code>StartDocumentAnalysis</code>.
     * </p>
     * <p>
     * <code>GetDocumentAnalysis</code> returns an array of <a>Block</a> objects. The following types of information are
     * returned:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Form data (key-value pairs). The related information is returned in two <a>Block</a> objects, each of type
     * <code>KEY_VALUE_SET</code>: a KEY <code>Block</code> object and a VALUE <code>Block</code> object. For example,
     * <i>Name: Ana Silva Carolina</i> contains a key and value. <i>Name:</i> is the key. <i>Ana Silva Carolina</i> is
     * the value.
     * </p>
     * </li>
     * <li>
     * <p>
     * Table and table cell data. A TABLE <code>Block</code> object contains information about a detected table. A CELL
     * <code>Block</code> object is returned for each cell in a table.
     * </p>
     * </li>
     * <li>
     * <p>
     * Lines and words of text. A LINE <code>Block</code> object contains one or more WORD <code>Block</code> objects.
     * All lines and words that are detected in the document are returned (including text that doesn't have a
     * relationship with the value of the <code>StartDocumentAnalysis</code> <code>FeatureTypes</code> input parameter).
     * </p>
     * </li>
     * <li>
     * <p>
     * Query. A QUERY Block object contains the query text, alias and link to the associated Query results block object.
     * </p>
     * </li>
     * <li>
     * <p>
     * Query Results. A QUERY_RESULT Block object contains the answer to the query and an ID that connects it to the
     * query asked. This Block also contains a confidence score.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * While processing a document with queries, look out for <code>INVALID_REQUEST_PARAMETERS</code> output. This
     * indicates that either the per page query limit has been exceeded or that the operation is trying to query a page
     * in the document which doesn’t exist.
     * </p>
     * </note>
     * <p>
     * Selection elements such as check boxes and option buttons (radio buttons) can be detected in form data and in
     * tables. A SELECTION_ELEMENT <code>Block</code> object contains information about a selection element, including
     * the selection status.
     * </p>
     * <p>
     * Use the <code>MaxResults</code> parameter to limit the number of blocks that are returned. If there are more
     * results than specified in <code>MaxResults</code>, the value of <code>NextToken</code> in the operation response
     * contains a pagination token for getting the next set of results. To get the next page of results, call
     * <code>GetDocumentAnalysis</code>, and populate the <code>NextToken</code> request parameter with the token value
     * that's returned from the previous call to <code>GetDocumentAnalysis</code>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/textract/latest/dg/how-it-works-analyzing.html">Document Text Analysis</a>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetDocumentAnalysisRequest.Builder} avoiding the
     * need to create one manually via {@link GetDocumentAnalysisRequest#builder()}
     * </p>
     *
     * @param getDocumentAnalysisRequest
     *        A {@link Consumer} that will call methods on {@link GetDocumentAnalysisRequest.Builder} to create a
     *        request.
     * @return Result of the GetDocumentAnalysis operation returned by the service.
     * @throws InvalidParameterException
     *         An input parameter violated a constraint. For example, in synchronous operations, an
     *         <code>InvalidParameterException</code> exception occurs when neither of the <code>S3Object</code> or
     *         <code>Bytes</code> values are supplied in the <code>Document</code> request parameter. Validate your
     *         parameter before calling the API operation again.
     * @throws AccessDeniedException
     *         You aren't authorized to perform the action. Use the Amazon Resource Name (ARN) of an authorized user or
     *         IAM role to perform the operation.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Textract.
     * @throws InvalidJobIdException
     *         An invalid job identifier was passed to an asynchronous analysis operation.
     * @throws InternalServerErrorException
     *         Amazon Textract experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Textract is temporarily unable to process the request. Try your call again.
     * @throws InvalidS3ObjectException
     *         Amazon Textract is unable to access the S3 object that's specified in the request. for more information,
     *         <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-access-control.html">Configure Access to
     *         Amazon S3</a> For troubleshooting information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/troubleshooting.html">Troubleshooting Amazon S3</a>
     * @throws InvalidKmsKeyException
     *         Indicates you do not have decrypt permissions with the KMS key entered, or the KMS key was entered
     *         incorrectly.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TextractException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TextractClient.GetDocumentAnalysis
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/GetDocumentAnalysis" target="_top">AWS
     *      API Documentation</a>
     */
    default GetDocumentAnalysisResponse getDocumentAnalysis(
            Consumer<GetDocumentAnalysisRequest.Builder> getDocumentAnalysisRequest) throws InvalidParameterException,
            AccessDeniedException, ProvisionedThroughputExceededException, InvalidJobIdException, InternalServerErrorException,
            ThrottlingException, InvalidS3ObjectException, InvalidKmsKeyException, AwsServiceException, SdkClientException,
            TextractException {
        return getDocumentAnalysis(GetDocumentAnalysisRequest.builder().applyMutation(getDocumentAnalysisRequest).build());
    }

    /**
     * <p>
     * Gets the results for an Amazon Textract asynchronous operation that detects text in a document. Amazon Textract
     * can detect lines of text and the words that make up a line of text.
     * </p>
     * <p>
     * You start asynchronous text detection by calling <a>StartDocumentTextDetection</a>, which returns a job
     * identifier (<code>JobId</code>). When the text detection operation finishes, Amazon Textract publishes a
     * completion status to the Amazon Simple Notification Service (Amazon SNS) topic that's registered in the initial
     * call to <code>StartDocumentTextDetection</code>. To get the results of the text-detection operation, first check
     * that the status value published to the Amazon SNS topic is <code>SUCCEEDED</code>. If so, call
     * <code>GetDocumentTextDetection</code>, and pass the job identifier (<code>JobId</code>) from the initial call to
     * <code>StartDocumentTextDetection</code>.
     * </p>
     * <p>
     * <code>GetDocumentTextDetection</code> returns an array of <a>Block</a> objects.
     * </p>
     * <p>
     * Each document page has as an associated <code>Block</code> of type PAGE. Each PAGE <code>Block</code> object is
     * the parent of LINE <code>Block</code> objects that represent the lines of detected text on a page. A LINE
     * <code>Block</code> object is a parent for each word that makes up the line. Words are represented by
     * <code>Block</code> objects of type WORD.
     * </p>
     * <p>
     * Use the MaxResults parameter to limit the number of blocks that are returned. If there are more results than
     * specified in <code>MaxResults</code>, the value of <code>NextToken</code> in the operation response contains a
     * pagination token for getting the next set of results. To get the next page of results, call
     * <code>GetDocumentTextDetection</code>, and populate the <code>NextToken</code> request parameter with the token
     * value that's returned from the previous call to <code>GetDocumentTextDetection</code>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/textract/latest/dg/how-it-works-detecting.html">Document Text Detection</a>.
     * </p>
     *
     * @param getDocumentTextDetectionRequest
     * @return Result of the GetDocumentTextDetection operation returned by the service.
     * @throws InvalidParameterException
     *         An input parameter violated a constraint. For example, in synchronous operations, an
     *         <code>InvalidParameterException</code> exception occurs when neither of the <code>S3Object</code> or
     *         <code>Bytes</code> values are supplied in the <code>Document</code> request parameter. Validate your
     *         parameter before calling the API operation again.
     * @throws AccessDeniedException
     *         You aren't authorized to perform the action. Use the Amazon Resource Name (ARN) of an authorized user or
     *         IAM role to perform the operation.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Textract.
     * @throws InvalidJobIdException
     *         An invalid job identifier was passed to an asynchronous analysis operation.
     * @throws InternalServerErrorException
     *         Amazon Textract experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Textract is temporarily unable to process the request. Try your call again.
     * @throws InvalidS3ObjectException
     *         Amazon Textract is unable to access the S3 object that's specified in the request. for more information,
     *         <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-access-control.html">Configure Access to
     *         Amazon S3</a> For troubleshooting information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/troubleshooting.html">Troubleshooting Amazon S3</a>
     * @throws InvalidKmsKeyException
     *         Indicates you do not have decrypt permissions with the KMS key entered, or the KMS key was entered
     *         incorrectly.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TextractException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TextractClient.GetDocumentTextDetection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/GetDocumentTextDetection"
     *      target="_top">AWS API Documentation</a>
     */
    default GetDocumentTextDetectionResponse getDocumentTextDetection(
            GetDocumentTextDetectionRequest getDocumentTextDetectionRequest) throws InvalidParameterException,
            AccessDeniedException, ProvisionedThroughputExceededException, InvalidJobIdException, InternalServerErrorException,
            ThrottlingException, InvalidS3ObjectException, InvalidKmsKeyException, AwsServiceException, SdkClientException,
            TextractException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Gets the results for an Amazon Textract asynchronous operation that detects text in a document. Amazon Textract
     * can detect lines of text and the words that make up a line of text.
     * </p>
     * <p>
     * You start asynchronous text detection by calling <a>StartDocumentTextDetection</a>, which returns a job
     * identifier (<code>JobId</code>). When the text detection operation finishes, Amazon Textract publishes a
     * completion status to the Amazon Simple Notification Service (Amazon SNS) topic that's registered in the initial
     * call to <code>StartDocumentTextDetection</code>. To get the results of the text-detection operation, first check
     * that the status value published to the Amazon SNS topic is <code>SUCCEEDED</code>. If so, call
     * <code>GetDocumentTextDetection</code>, and pass the job identifier (<code>JobId</code>) from the initial call to
     * <code>StartDocumentTextDetection</code>.
     * </p>
     * <p>
     * <code>GetDocumentTextDetection</code> returns an array of <a>Block</a> objects.
     * </p>
     * <p>
     * Each document page has as an associated <code>Block</code> of type PAGE. Each PAGE <code>Block</code> object is
     * the parent of LINE <code>Block</code> objects that represent the lines of detected text on a page. A LINE
     * <code>Block</code> object is a parent for each word that makes up the line. Words are represented by
     * <code>Block</code> objects of type WORD.
     * </p>
     * <p>
     * Use the MaxResults parameter to limit the number of blocks that are returned. If there are more results than
     * specified in <code>MaxResults</code>, the value of <code>NextToken</code> in the operation response contains a
     * pagination token for getting the next set of results. To get the next page of results, call
     * <code>GetDocumentTextDetection</code>, and populate the <code>NextToken</code> request parameter with the token
     * value that's returned from the previous call to <code>GetDocumentTextDetection</code>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/textract/latest/dg/how-it-works-detecting.html">Document Text Detection</a>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetDocumentTextDetectionRequest.Builder} avoiding
     * the need to create one manually via {@link GetDocumentTextDetectionRequest#builder()}
     * </p>
     *
     * @param getDocumentTextDetectionRequest
     *        A {@link Consumer} that will call methods on {@link GetDocumentTextDetectionRequest.Builder} to create a
     *        request.
     * @return Result of the GetDocumentTextDetection operation returned by the service.
     * @throws InvalidParameterException
     *         An input parameter violated a constraint. For example, in synchronous operations, an
     *         <code>InvalidParameterException</code> exception occurs when neither of the <code>S3Object</code> or
     *         <code>Bytes</code> values are supplied in the <code>Document</code> request parameter. Validate your
     *         parameter before calling the API operation again.
     * @throws AccessDeniedException
     *         You aren't authorized to perform the action. Use the Amazon Resource Name (ARN) of an authorized user or
     *         IAM role to perform the operation.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Textract.
     * @throws InvalidJobIdException
     *         An invalid job identifier was passed to an asynchronous analysis operation.
     * @throws InternalServerErrorException
     *         Amazon Textract experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Textract is temporarily unable to process the request. Try your call again.
     * @throws InvalidS3ObjectException
     *         Amazon Textract is unable to access the S3 object that's specified in the request. for more information,
     *         <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-access-control.html">Configure Access to
     *         Amazon S3</a> For troubleshooting information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/troubleshooting.html">Troubleshooting Amazon S3</a>
     * @throws InvalidKmsKeyException
     *         Indicates you do not have decrypt permissions with the KMS key entered, or the KMS key was entered
     *         incorrectly.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TextractException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TextractClient.GetDocumentTextDetection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/GetDocumentTextDetection"
     *      target="_top">AWS API Documentation</a>
     */
    default GetDocumentTextDetectionResponse getDocumentTextDetection(
            Consumer<GetDocumentTextDetectionRequest.Builder> getDocumentTextDetectionRequest) throws InvalidParameterException,
            AccessDeniedException, ProvisionedThroughputExceededException, InvalidJobIdException, InternalServerErrorException,
            ThrottlingException, InvalidS3ObjectException, InvalidKmsKeyException, AwsServiceException, SdkClientException,
            TextractException {
        return getDocumentTextDetection(GetDocumentTextDetectionRequest.builder().applyMutation(getDocumentTextDetectionRequest)
                .build());
    }

    /**
     * <p>
     * Gets the results for an Amazon Textract asynchronous operation that analyzes invoices and receipts. Amazon
     * Textract finds contact information, items purchased, and vendor name, from input invoices and receipts.
     * </p>
     * <p>
     * You start asynchronous invoice/receipt analysis by calling <a>StartExpenseAnalysis</a>, which returns a job
     * identifier (<code>JobId</code>). Upon completion of the invoice/receipt analysis, Amazon Textract publishes the
     * completion status to the Amazon Simple Notification Service (Amazon SNS) topic. This topic must be registered in
     * the initial call to <code>StartExpenseAnalysis</code>. To get the results of the invoice/receipt analysis
     * operation, first ensure that the status value published to the Amazon SNS topic is <code>SUCCEEDED</code>. If so,
     * call <code>GetExpenseAnalysis</code>, and pass the job identifier (<code>JobId</code>) from the initial call to
     * <code>StartExpenseAnalysis</code>.
     * </p>
     * <p>
     * Use the MaxResults parameter to limit the number of blocks that are returned. If there are more results than
     * specified in <code>MaxResults</code>, the value of <code>NextToken</code> in the operation response contains a
     * pagination token for getting the next set of results. To get the next page of results, call
     * <code>GetExpenseAnalysis</code>, and populate the <code>NextToken</code> request parameter with the token value
     * that's returned from the previous call to <code>GetExpenseAnalysis</code>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/textract/latest/dg/invoices-receipts.html">Analyzing Invoices and Receipts</a>.
     * </p>
     *
     * @param getExpenseAnalysisRequest
     * @return Result of the GetExpenseAnalysis operation returned by the service.
     * @throws InvalidParameterException
     *         An input parameter violated a constraint. For example, in synchronous operations, an
     *         <code>InvalidParameterException</code> exception occurs when neither of the <code>S3Object</code> or
     *         <code>Bytes</code> values are supplied in the <code>Document</code> request parameter. Validate your
     *         parameter before calling the API operation again.
     * @throws AccessDeniedException
     *         You aren't authorized to perform the action. Use the Amazon Resource Name (ARN) of an authorized user or
     *         IAM role to perform the operation.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Textract.
     * @throws InvalidJobIdException
     *         An invalid job identifier was passed to an asynchronous analysis operation.
     * @throws InternalServerErrorException
     *         Amazon Textract experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Textract is temporarily unable to process the request. Try your call again.
     * @throws InvalidS3ObjectException
     *         Amazon Textract is unable to access the S3 object that's specified in the request. for more information,
     *         <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-access-control.html">Configure Access to
     *         Amazon S3</a> For troubleshooting information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/troubleshooting.html">Troubleshooting Amazon S3</a>
     * @throws InvalidKmsKeyException
     *         Indicates you do not have decrypt permissions with the KMS key entered, or the KMS key was entered
     *         incorrectly.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TextractException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TextractClient.GetExpenseAnalysis
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/GetExpenseAnalysis" target="_top">AWS
     *      API Documentation</a>
     */
    default GetExpenseAnalysisResponse getExpenseAnalysis(GetExpenseAnalysisRequest getExpenseAnalysisRequest)
            throws InvalidParameterException, AccessDeniedException, ProvisionedThroughputExceededException,
            InvalidJobIdException, InternalServerErrorException, ThrottlingException, InvalidS3ObjectException,
            InvalidKmsKeyException, AwsServiceException, SdkClientException, TextractException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Gets the results for an Amazon Textract asynchronous operation that analyzes invoices and receipts. Amazon
     * Textract finds contact information, items purchased, and vendor name, from input invoices and receipts.
     * </p>
     * <p>
     * You start asynchronous invoice/receipt analysis by calling <a>StartExpenseAnalysis</a>, which returns a job
     * identifier (<code>JobId</code>). Upon completion of the invoice/receipt analysis, Amazon Textract publishes the
     * completion status to the Amazon Simple Notification Service (Amazon SNS) topic. This topic must be registered in
     * the initial call to <code>StartExpenseAnalysis</code>. To get the results of the invoice/receipt analysis
     * operation, first ensure that the status value published to the Amazon SNS topic is <code>SUCCEEDED</code>. If so,
     * call <code>GetExpenseAnalysis</code>, and pass the job identifier (<code>JobId</code>) from the initial call to
     * <code>StartExpenseAnalysis</code>.
     * </p>
     * <p>
     * Use the MaxResults parameter to limit the number of blocks that are returned. If there are more results than
     * specified in <code>MaxResults</code>, the value of <code>NextToken</code> in the operation response contains a
     * pagination token for getting the next set of results. To get the next page of results, call
     * <code>GetExpenseAnalysis</code>, and populate the <code>NextToken</code> request parameter with the token value
     * that's returned from the previous call to <code>GetExpenseAnalysis</code>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/textract/latest/dg/invoices-receipts.html">Analyzing Invoices and Receipts</a>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetExpenseAnalysisRequest.Builder} avoiding the
     * need to create one manually via {@link GetExpenseAnalysisRequest#builder()}
     * </p>
     *
     * @param getExpenseAnalysisRequest
     *        A {@link Consumer} that will call methods on {@link GetExpenseAnalysisRequest.Builder} to create a
     *        request.
     * @return Result of the GetExpenseAnalysis operation returned by the service.
     * @throws InvalidParameterException
     *         An input parameter violated a constraint. For example, in synchronous operations, an
     *         <code>InvalidParameterException</code> exception occurs when neither of the <code>S3Object</code> or
     *         <code>Bytes</code> values are supplied in the <code>Document</code> request parameter. Validate your
     *         parameter before calling the API operation again.
     * @throws AccessDeniedException
     *         You aren't authorized to perform the action. Use the Amazon Resource Name (ARN) of an authorized user or
     *         IAM role to perform the operation.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Textract.
     * @throws InvalidJobIdException
     *         An invalid job identifier was passed to an asynchronous analysis operation.
     * @throws InternalServerErrorException
     *         Amazon Textract experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Textract is temporarily unable to process the request. Try your call again.
     * @throws InvalidS3ObjectException
     *         Amazon Textract is unable to access the S3 object that's specified in the request. for more information,
     *         <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-access-control.html">Configure Access to
     *         Amazon S3</a> For troubleshooting information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/troubleshooting.html">Troubleshooting Amazon S3</a>
     * @throws InvalidKmsKeyException
     *         Indicates you do not have decrypt permissions with the KMS key entered, or the KMS key was entered
     *         incorrectly.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TextractException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TextractClient.GetExpenseAnalysis
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/GetExpenseAnalysis" target="_top">AWS
     *      API Documentation</a>
     */
    default GetExpenseAnalysisResponse getExpenseAnalysis(Consumer<GetExpenseAnalysisRequest.Builder> getExpenseAnalysisRequest)
            throws InvalidParameterException, AccessDeniedException, ProvisionedThroughputExceededException,
            InvalidJobIdException, InternalServerErrorException, ThrottlingException, InvalidS3ObjectException,
            InvalidKmsKeyException, AwsServiceException, SdkClientException, TextractException {
        return getExpenseAnalysis(GetExpenseAnalysisRequest.builder().applyMutation(getExpenseAnalysisRequest).build());
    }

    /**
     * <p>
     * Gets the results for an Amazon Textract asynchronous operation that analyzes text in a lending document.
     * </p>
     * <p>
     * You start asynchronous text analysis by calling <code>StartLendingAnalysis</code>, which returns a job identifier
     * (<code>JobId</code>). When the text analysis operation finishes, Amazon Textract publishes a completion status to
     * the Amazon Simple Notification Service (Amazon SNS) topic that's registered in the initial call to
     * <code>StartLendingAnalysis</code>.
     * </p>
     * <p>
     * To get the results of the text analysis operation, first check that the status value published to the Amazon SNS
     * topic is SUCCEEDED. If so, call GetLendingAnalysis, and pass the job identifier (<code>JobId</code>) from the
     * initial call to <code>StartLendingAnalysis</code>.
     * </p>
     *
     * @param getLendingAnalysisRequest
     * @return Result of the GetLendingAnalysis operation returned by the service.
     * @throws InvalidParameterException
     *         An input parameter violated a constraint. For example, in synchronous operations, an
     *         <code>InvalidParameterException</code> exception occurs when neither of the <code>S3Object</code> or
     *         <code>Bytes</code> values are supplied in the <code>Document</code> request parameter. Validate your
     *         parameter before calling the API operation again.
     * @throws AccessDeniedException
     *         You aren't authorized to perform the action. Use the Amazon Resource Name (ARN) of an authorized user or
     *         IAM role to perform the operation.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Textract.
     * @throws InvalidJobIdException
     *         An invalid job identifier was passed to an asynchronous analysis operation.
     * @throws InternalServerErrorException
     *         Amazon Textract experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Textract is temporarily unable to process the request. Try your call again.
     * @throws InvalidS3ObjectException
     *         Amazon Textract is unable to access the S3 object that's specified in the request. for more information,
     *         <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-access-control.html">Configure Access to
     *         Amazon S3</a> For troubleshooting information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/troubleshooting.html">Troubleshooting Amazon S3</a>
     * @throws InvalidKmsKeyException
     *         Indicates you do not have decrypt permissions with the KMS key entered, or the KMS key was entered
     *         incorrectly.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TextractException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TextractClient.GetLendingAnalysis
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/GetLendingAnalysis" target="_top">AWS
     *      API Documentation</a>
     */
    default GetLendingAnalysisResponse getLendingAnalysis(GetLendingAnalysisRequest getLendingAnalysisRequest)
            throws InvalidParameterException, AccessDeniedException, ProvisionedThroughputExceededException,
            InvalidJobIdException, InternalServerErrorException, ThrottlingException, InvalidS3ObjectException,
            InvalidKmsKeyException, AwsServiceException, SdkClientException, TextractException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Gets the results for an Amazon Textract asynchronous operation that analyzes text in a lending document.
     * </p>
     * <p>
     * You start asynchronous text analysis by calling <code>StartLendingAnalysis</code>, which returns a job identifier
     * (<code>JobId</code>). When the text analysis operation finishes, Amazon Textract publishes a completion status to
     * the Amazon Simple Notification Service (Amazon SNS) topic that's registered in the initial call to
     * <code>StartLendingAnalysis</code>.
     * </p>
     * <p>
     * To get the results of the text analysis operation, first check that the status value published to the Amazon SNS
     * topic is SUCCEEDED. If so, call GetLendingAnalysis, and pass the job identifier (<code>JobId</code>) from the
     * initial call to <code>StartLendingAnalysis</code>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetLendingAnalysisRequest.Builder} avoiding the
     * need to create one manually via {@link GetLendingAnalysisRequest#builder()}
     * </p>
     *
     * @param getLendingAnalysisRequest
     *        A {@link Consumer} that will call methods on {@link GetLendingAnalysisRequest.Builder} to create a
     *        request.
     * @return Result of the GetLendingAnalysis operation returned by the service.
     * @throws InvalidParameterException
     *         An input parameter violated a constraint. For example, in synchronous operations, an
     *         <code>InvalidParameterException</code> exception occurs when neither of the <code>S3Object</code> or
     *         <code>Bytes</code> values are supplied in the <code>Document</code> request parameter. Validate your
     *         parameter before calling the API operation again.
     * @throws AccessDeniedException
     *         You aren't authorized to perform the action. Use the Amazon Resource Name (ARN) of an authorized user or
     *         IAM role to perform the operation.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Textract.
     * @throws InvalidJobIdException
     *         An invalid job identifier was passed to an asynchronous analysis operation.
     * @throws InternalServerErrorException
     *         Amazon Textract experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Textract is temporarily unable to process the request. Try your call again.
     * @throws InvalidS3ObjectException
     *         Amazon Textract is unable to access the S3 object that's specified in the request. for more information,
     *         <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-access-control.html">Configure Access to
     *         Amazon S3</a> For troubleshooting information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/troubleshooting.html">Troubleshooting Amazon S3</a>
     * @throws InvalidKmsKeyException
     *         Indicates you do not have decrypt permissions with the KMS key entered, or the KMS key was entered
     *         incorrectly.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TextractException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TextractClient.GetLendingAnalysis
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/GetLendingAnalysis" target="_top">AWS
     *      API Documentation</a>
     */
    default GetLendingAnalysisResponse getLendingAnalysis(Consumer<GetLendingAnalysisRequest.Builder> getLendingAnalysisRequest)
            throws InvalidParameterException, AccessDeniedException, ProvisionedThroughputExceededException,
            InvalidJobIdException, InternalServerErrorException, ThrottlingException, InvalidS3ObjectException,
            InvalidKmsKeyException, AwsServiceException, SdkClientException, TextractException {
        return getLendingAnalysis(GetLendingAnalysisRequest.builder().applyMutation(getLendingAnalysisRequest).build());
    }

    /**
     * <p>
     * Gets summarized results for the <code>StartLendingAnalysis</code> operation, which analyzes text in a lending
     * document. The returned summary consists of information about documents grouped together by a common document
     * type. Information like detected signatures, page numbers, and split documents is returned with respect to the
     * type of grouped document.
     * </p>
     * <p>
     * You start asynchronous text analysis by calling <code>StartLendingAnalysis</code>, which returns a job identifier
     * (<code>JobId</code>). When the text analysis operation finishes, Amazon Textract publishes a completion status to
     * the Amazon Simple Notification Service (Amazon SNS) topic that's registered in the initial call to
     * <code>StartLendingAnalysis</code>.
     * </p>
     * <p>
     * To get the results of the text analysis operation, first check that the status value published to the Amazon SNS
     * topic is SUCCEEDED. If so, call <code>GetLendingAnalysisSummary</code>, and pass the job identifier (
     * <code>JobId</code>) from the initial call to <code>StartLendingAnalysis</code>.
     * </p>
     *
     * @param getLendingAnalysisSummaryRequest
     * @return Result of the GetLendingAnalysisSummary operation returned by the service.
     * @throws InvalidParameterException
     *         An input parameter violated a constraint. For example, in synchronous operations, an
     *         <code>InvalidParameterException</code> exception occurs when neither of the <code>S3Object</code> or
     *         <code>Bytes</code> values are supplied in the <code>Document</code> request parameter. Validate your
     *         parameter before calling the API operation again.
     * @throws AccessDeniedException
     *         You aren't authorized to perform the action. Use the Amazon Resource Name (ARN) of an authorized user or
     *         IAM role to perform the operation.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Textract.
     * @throws InvalidJobIdException
     *         An invalid job identifier was passed to an asynchronous analysis operation.
     * @throws InternalServerErrorException
     *         Amazon Textract experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Textract is temporarily unable to process the request. Try your call again.
     * @throws InvalidS3ObjectException
     *         Amazon Textract is unable to access the S3 object that's specified in the request. for more information,
     *         <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-access-control.html">Configure Access to
     *         Amazon S3</a> For troubleshooting information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/troubleshooting.html">Troubleshooting Amazon S3</a>
     * @throws InvalidKmsKeyException
     *         Indicates you do not have decrypt permissions with the KMS key entered, or the KMS key was entered
     *         incorrectly.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TextractException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TextractClient.GetLendingAnalysisSummary
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/GetLendingAnalysisSummary"
     *      target="_top">AWS API Documentation</a>
     */
    default GetLendingAnalysisSummaryResponse getLendingAnalysisSummary(
            GetLendingAnalysisSummaryRequest getLendingAnalysisSummaryRequest) throws InvalidParameterException,
            AccessDeniedException, ProvisionedThroughputExceededException, InvalidJobIdException, InternalServerErrorException,
            ThrottlingException, InvalidS3ObjectException, InvalidKmsKeyException, AwsServiceException, SdkClientException,
            TextractException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Gets summarized results for the <code>StartLendingAnalysis</code> operation, which analyzes text in a lending
     * document. The returned summary consists of information about documents grouped together by a common document
     * type. Information like detected signatures, page numbers, and split documents is returned with respect to the
     * type of grouped document.
     * </p>
     * <p>
     * You start asynchronous text analysis by calling <code>StartLendingAnalysis</code>, which returns a job identifier
     * (<code>JobId</code>). When the text analysis operation finishes, Amazon Textract publishes a completion status to
     * the Amazon Simple Notification Service (Amazon SNS) topic that's registered in the initial call to
     * <code>StartLendingAnalysis</code>.
     * </p>
     * <p>
     * To get the results of the text analysis operation, first check that the status value published to the Amazon SNS
     * topic is SUCCEEDED. If so, call <code>GetLendingAnalysisSummary</code>, and pass the job identifier (
     * <code>JobId</code>) from the initial call to <code>StartLendingAnalysis</code>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetLendingAnalysisSummaryRequest.Builder} avoiding
     * the need to create one manually via {@link GetLendingAnalysisSummaryRequest#builder()}
     * </p>
     *
     * @param getLendingAnalysisSummaryRequest
     *        A {@link Consumer} that will call methods on {@link GetLendingAnalysisSummaryRequest.Builder} to create a
     *        request.
     * @return Result of the GetLendingAnalysisSummary operation returned by the service.
     * @throws InvalidParameterException
     *         An input parameter violated a constraint. For example, in synchronous operations, an
     *         <code>InvalidParameterException</code> exception occurs when neither of the <code>S3Object</code> or
     *         <code>Bytes</code> values are supplied in the <code>Document</code> request parameter. Validate your
     *         parameter before calling the API operation again.
     * @throws AccessDeniedException
     *         You aren't authorized to perform the action. Use the Amazon Resource Name (ARN) of an authorized user or
     *         IAM role to perform the operation.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Textract.
     * @throws InvalidJobIdException
     *         An invalid job identifier was passed to an asynchronous analysis operation.
     * @throws InternalServerErrorException
     *         Amazon Textract experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Textract is temporarily unable to process the request. Try your call again.
     * @throws InvalidS3ObjectException
     *         Amazon Textract is unable to access the S3 object that's specified in the request. for more information,
     *         <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-access-control.html">Configure Access to
     *         Amazon S3</a> For troubleshooting information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/troubleshooting.html">Troubleshooting Amazon S3</a>
     * @throws InvalidKmsKeyException
     *         Indicates you do not have decrypt permissions with the KMS key entered, or the KMS key was entered
     *         incorrectly.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TextractException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TextractClient.GetLendingAnalysisSummary
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/GetLendingAnalysisSummary"
     *      target="_top">AWS API Documentation</a>
     */
    default GetLendingAnalysisSummaryResponse getLendingAnalysisSummary(
            Consumer<GetLendingAnalysisSummaryRequest.Builder> getLendingAnalysisSummaryRequest)
            throws InvalidParameterException, AccessDeniedException, ProvisionedThroughputExceededException,
            InvalidJobIdException, InternalServerErrorException, ThrottlingException, InvalidS3ObjectException,
            InvalidKmsKeyException, AwsServiceException, SdkClientException, TextractException {
        return getLendingAnalysisSummary(GetLendingAnalysisSummaryRequest.builder()
                .applyMutation(getLendingAnalysisSummaryRequest).build());
    }

    /**
     * <p>
     * Starts the asynchronous analysis of an input document for relationships between detected items such as key-value
     * pairs, tables, and selection elements.
     * </p>
     * <p>
     * <code>StartDocumentAnalysis</code> can analyze text in documents that are in JPEG, PNG, TIFF, and PDF format. The
     * documents are stored in an Amazon S3 bucket. Use <a>DocumentLocation</a> to specify the bucket name and file name
     * of the document.
     * </p>
     * <p>
     * <code>StartDocumentAnalysis</code> returns a job identifier (<code>JobId</code>) that you use to get the results
     * of the operation. When text analysis is finished, Amazon Textract publishes a completion status to the Amazon
     * Simple Notification Service (Amazon SNS) topic that you specify in <code>NotificationChannel</code>. To get the
     * results of the text analysis operation, first check that the status value published to the Amazon SNS topic is
     * <code>SUCCEEDED</code>. If so, call <a>GetDocumentAnalysis</a>, and pass the job identifier (<code>JobId</code>)
     * from the initial call to <code>StartDocumentAnalysis</code>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/textract/latest/dg/how-it-works-analyzing.html">Document Text Analysis</a>.
     * </p>
     *
     * @param startDocumentAnalysisRequest
     * @return Result of the StartDocumentAnalysis operation returned by the service.
     * @throws InvalidParameterException
     *         An input parameter violated a constraint. For example, in synchronous operations, an
     *         <code>InvalidParameterException</code> exception occurs when neither of the <code>S3Object</code> or
     *         <code>Bytes</code> values are supplied in the <code>Document</code> request parameter. Validate your
     *         parameter before calling the API operation again.
     * @throws InvalidS3ObjectException
     *         Amazon Textract is unable to access the S3 object that's specified in the request. for more information,
     *         <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-access-control.html">Configure Access to
     *         Amazon S3</a> For troubleshooting information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/troubleshooting.html">Troubleshooting Amazon S3</a>
     * @throws InvalidKmsKeyException
     *         Indicates you do not have decrypt permissions with the KMS key entered, or the KMS key was entered
     *         incorrectly.
     * @throws UnsupportedDocumentException
     *         The format of the input document isn't supported. Documents for operations can be in PNG, JPEG, PDF, or
     *         TIFF format.
     * @throws DocumentTooLargeException
     *         The document can't be processed because it's too large. The maximum document size for synchronous
     *         operations 10 MB. The maximum document size for asynchronous operations is 500 MB for PDF files.
     * @throws BadDocumentException
     *         Amazon Textract isn't able to read the document. For more information on the document limits in Amazon
     *         Textract, see <a>limits</a>.
     * @throws AccessDeniedException
     *         You aren't authorized to perform the action. Use the Amazon Resource Name (ARN) of an authorized user or
     *         IAM role to perform the operation.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Textract.
     * @throws InternalServerErrorException
     *         Amazon Textract experienced a service issue. Try your call again.
     * @throws IdempotentParameterMismatchException
     *         A <code>ClientRequestToken</code> input parameter was reused with an operation, but at least one of the
     *         other input parameters is different from the previous call to the operation.
     * @throws ThrottlingException
     *         Amazon Textract is temporarily unable to process the request. Try your call again.
     * @throws LimitExceededException
     *         An Amazon Textract service limit was exceeded. For example, if you start too many asynchronous jobs
     *         concurrently, calls to start operations (<code>StartDocumentTextDetection</code>, for example) raise a
     *         LimitExceededException exception (HTTP status code: 400) until the number of concurrently running jobs is
     *         below the Amazon Textract service limit.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TextractException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TextractClient.StartDocumentAnalysis
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/StartDocumentAnalysis"
     *      target="_top">AWS API Documentation</a>
     */
    default StartDocumentAnalysisResponse startDocumentAnalysis(StartDocumentAnalysisRequest startDocumentAnalysisRequest)
            throws InvalidParameterException, InvalidS3ObjectException, InvalidKmsKeyException, UnsupportedDocumentException,
            DocumentTooLargeException, BadDocumentException, AccessDeniedException, ProvisionedThroughputExceededException,
            InternalServerErrorException, IdempotentParameterMismatchException, ThrottlingException, LimitExceededException,
            AwsServiceException, SdkClientException, TextractException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Starts the asynchronous analysis of an input document for relationships between detected items such as key-value
     * pairs, tables, and selection elements.
     * </p>
     * <p>
     * <code>StartDocumentAnalysis</code> can analyze text in documents that are in JPEG, PNG, TIFF, and PDF format. The
     * documents are stored in an Amazon S3 bucket. Use <a>DocumentLocation</a> to specify the bucket name and file name
     * of the document.
     * </p>
     * <p>
     * <code>StartDocumentAnalysis</code> returns a job identifier (<code>JobId</code>) that you use to get the results
     * of the operation. When text analysis is finished, Amazon Textract publishes a completion status to the Amazon
     * Simple Notification Service (Amazon SNS) topic that you specify in <code>NotificationChannel</code>. To get the
     * results of the text analysis operation, first check that the status value published to the Amazon SNS topic is
     * <code>SUCCEEDED</code>. If so, call <a>GetDocumentAnalysis</a>, and pass the job identifier (<code>JobId</code>)
     * from the initial call to <code>StartDocumentAnalysis</code>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/textract/latest/dg/how-it-works-analyzing.html">Document Text Analysis</a>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link StartDocumentAnalysisRequest.Builder} avoiding the
     * need to create one manually via {@link StartDocumentAnalysisRequest#builder()}
     * </p>
     *
     * @param startDocumentAnalysisRequest
     *        A {@link Consumer} that will call methods on {@link StartDocumentAnalysisRequest.Builder} to create a
     *        request.
     * @return Result of the StartDocumentAnalysis operation returned by the service.
     * @throws InvalidParameterException
     *         An input parameter violated a constraint. For example, in synchronous operations, an
     *         <code>InvalidParameterException</code> exception occurs when neither of the <code>S3Object</code> or
     *         <code>Bytes</code> values are supplied in the <code>Document</code> request parameter. Validate your
     *         parameter before calling the API operation again.
     * @throws InvalidS3ObjectException
     *         Amazon Textract is unable to access the S3 object that's specified in the request. for more information,
     *         <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-access-control.html">Configure Access to
     *         Amazon S3</a> For troubleshooting information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/troubleshooting.html">Troubleshooting Amazon S3</a>
     * @throws InvalidKmsKeyException
     *         Indicates you do not have decrypt permissions with the KMS key entered, or the KMS key was entered
     *         incorrectly.
     * @throws UnsupportedDocumentException
     *         The format of the input document isn't supported. Documents for operations can be in PNG, JPEG, PDF, or
     *         TIFF format.
     * @throws DocumentTooLargeException
     *         The document can't be processed because it's too large. The maximum document size for synchronous
     *         operations 10 MB. The maximum document size for asynchronous operations is 500 MB for PDF files.
     * @throws BadDocumentException
     *         Amazon Textract isn't able to read the document. For more information on the document limits in Amazon
     *         Textract, see <a>limits</a>.
     * @throws AccessDeniedException
     *         You aren't authorized to perform the action. Use the Amazon Resource Name (ARN) of an authorized user or
     *         IAM role to perform the operation.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Textract.
     * @throws InternalServerErrorException
     *         Amazon Textract experienced a service issue. Try your call again.
     * @throws IdempotentParameterMismatchException
     *         A <code>ClientRequestToken</code> input parameter was reused with an operation, but at least one of the
     *         other input parameters is different from the previous call to the operation.
     * @throws ThrottlingException
     *         Amazon Textract is temporarily unable to process the request. Try your call again.
     * @throws LimitExceededException
     *         An Amazon Textract service limit was exceeded. For example, if you start too many asynchronous jobs
     *         concurrently, calls to start operations (<code>StartDocumentTextDetection</code>, for example) raise a
     *         LimitExceededException exception (HTTP status code: 400) until the number of concurrently running jobs is
     *         below the Amazon Textract service limit.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TextractException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TextractClient.StartDocumentAnalysis
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/StartDocumentAnalysis"
     *      target="_top">AWS API Documentation</a>
     */
    default StartDocumentAnalysisResponse startDocumentAnalysis(
            Consumer<StartDocumentAnalysisRequest.Builder> startDocumentAnalysisRequest) throws InvalidParameterException,
            InvalidS3ObjectException, InvalidKmsKeyException, UnsupportedDocumentException, DocumentTooLargeException,
            BadDocumentException, AccessDeniedException, ProvisionedThroughputExceededException, InternalServerErrorException,
            IdempotentParameterMismatchException, ThrottlingException, LimitExceededException, AwsServiceException,
            SdkClientException, TextractException {
        return startDocumentAnalysis(StartDocumentAnalysisRequest.builder().applyMutation(startDocumentAnalysisRequest).build());
    }

    /**
     * <p>
     * Starts the asynchronous detection of text in a document. Amazon Textract can detect lines of text and the words
     * that make up a line of text.
     * </p>
     * <p>
     * <code>StartDocumentTextDetection</code> can analyze text in documents that are in JPEG, PNG, TIFF, and PDF
     * format. The documents are stored in an Amazon S3 bucket. Use <a>DocumentLocation</a> to specify the bucket name
     * and file name of the document.
     * </p>
     * <p>
     * <code>StartTextDetection</code> returns a job identifier (<code>JobId</code>) that you use to get the results of
     * the operation. When text detection is finished, Amazon Textract publishes a completion status to the Amazon
     * Simple Notification Service (Amazon SNS) topic that you specify in <code>NotificationChannel</code>. To get the
     * results of the text detection operation, first check that the status value published to the Amazon SNS topic is
     * <code>SUCCEEDED</code>. If so, call <a>GetDocumentTextDetection</a>, and pass the job identifier (
     * <code>JobId</code>) from the initial call to <code>StartDocumentTextDetection</code>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/textract/latest/dg/how-it-works-detecting.html">Document Text Detection</a>.
     * </p>
     *
     * @param startDocumentTextDetectionRequest
     * @return Result of the StartDocumentTextDetection operation returned by the service.
     * @throws InvalidParameterException
     *         An input parameter violated a constraint. For example, in synchronous operations, an
     *         <code>InvalidParameterException</code> exception occurs when neither of the <code>S3Object</code> or
     *         <code>Bytes</code> values are supplied in the <code>Document</code> request parameter. Validate your
     *         parameter before calling the API operation again.
     * @throws InvalidS3ObjectException
     *         Amazon Textract is unable to access the S3 object that's specified in the request. for more information,
     *         <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-access-control.html">Configure Access to
     *         Amazon S3</a> For troubleshooting information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/troubleshooting.html">Troubleshooting Amazon S3</a>
     * @throws InvalidKmsKeyException
     *         Indicates you do not have decrypt permissions with the KMS key entered, or the KMS key was entered
     *         incorrectly.
     * @throws UnsupportedDocumentException
     *         The format of the input document isn't supported. Documents for operations can be in PNG, JPEG, PDF, or
     *         TIFF format.
     * @throws DocumentTooLargeException
     *         The document can't be processed because it's too large. The maximum document size for synchronous
     *         operations 10 MB. The maximum document size for asynchronous operations is 500 MB for PDF files.
     * @throws BadDocumentException
     *         Amazon Textract isn't able to read the document. For more information on the document limits in Amazon
     *         Textract, see <a>limits</a>.
     * @throws AccessDeniedException
     *         You aren't authorized to perform the action. Use the Amazon Resource Name (ARN) of an authorized user or
     *         IAM role to perform the operation.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Textract.
     * @throws InternalServerErrorException
     *         Amazon Textract experienced a service issue. Try your call again.
     * @throws IdempotentParameterMismatchException
     *         A <code>ClientRequestToken</code> input parameter was reused with an operation, but at least one of the
     *         other input parameters is different from the previous call to the operation.
     * @throws ThrottlingException
     *         Amazon Textract is temporarily unable to process the request. Try your call again.
     * @throws LimitExceededException
     *         An Amazon Textract service limit was exceeded. For example, if you start too many asynchronous jobs
     *         concurrently, calls to start operations (<code>StartDocumentTextDetection</code>, for example) raise a
     *         LimitExceededException exception (HTTP status code: 400) until the number of concurrently running jobs is
     *         below the Amazon Textract service limit.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TextractException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TextractClient.StartDocumentTextDetection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/StartDocumentTextDetection"
     *      target="_top">AWS API Documentation</a>
     */
    default StartDocumentTextDetectionResponse startDocumentTextDetection(
            StartDocumentTextDetectionRequest startDocumentTextDetectionRequest) throws InvalidParameterException,
            InvalidS3ObjectException, InvalidKmsKeyException, UnsupportedDocumentException, DocumentTooLargeException,
            BadDocumentException, AccessDeniedException, ProvisionedThroughputExceededException, InternalServerErrorException,
            IdempotentParameterMismatchException, ThrottlingException, LimitExceededException, AwsServiceException,
            SdkClientException, TextractException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Starts the asynchronous detection of text in a document. Amazon Textract can detect lines of text and the words
     * that make up a line of text.
     * </p>
     * <p>
     * <code>StartDocumentTextDetection</code> can analyze text in documents that are in JPEG, PNG, TIFF, and PDF
     * format. The documents are stored in an Amazon S3 bucket. Use <a>DocumentLocation</a> to specify the bucket name
     * and file name of the document.
     * </p>
     * <p>
     * <code>StartTextDetection</code> returns a job identifier (<code>JobId</code>) that you use to get the results of
     * the operation. When text detection is finished, Amazon Textract publishes a completion status to the Amazon
     * Simple Notification Service (Amazon SNS) topic that you specify in <code>NotificationChannel</code>. To get the
     * results of the text detection operation, first check that the status value published to the Amazon SNS topic is
     * <code>SUCCEEDED</code>. If so, call <a>GetDocumentTextDetection</a>, and pass the job identifier (
     * <code>JobId</code>) from the initial call to <code>StartDocumentTextDetection</code>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/textract/latest/dg/how-it-works-detecting.html">Document Text Detection</a>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link StartDocumentTextDetectionRequest.Builder} avoiding
     * the need to create one manually via {@link StartDocumentTextDetectionRequest#builder()}
     * </p>
     *
     * @param startDocumentTextDetectionRequest
     *        A {@link Consumer} that will call methods on {@link StartDocumentTextDetectionRequest.Builder} to create a
     *        request.
     * @return Result of the StartDocumentTextDetection operation returned by the service.
     * @throws InvalidParameterException
     *         An input parameter violated a constraint. For example, in synchronous operations, an
     *         <code>InvalidParameterException</code> exception occurs when neither of the <code>S3Object</code> or
     *         <code>Bytes</code> values are supplied in the <code>Document</code> request parameter. Validate your
     *         parameter before calling the API operation again.
     * @throws InvalidS3ObjectException
     *         Amazon Textract is unable to access the S3 object that's specified in the request. for more information,
     *         <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-access-control.html">Configure Access to
     *         Amazon S3</a> For troubleshooting information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/troubleshooting.html">Troubleshooting Amazon S3</a>
     * @throws InvalidKmsKeyException
     *         Indicates you do not have decrypt permissions with the KMS key entered, or the KMS key was entered
     *         incorrectly.
     * @throws UnsupportedDocumentException
     *         The format of the input document isn't supported. Documents for operations can be in PNG, JPEG, PDF, or
     *         TIFF format.
     * @throws DocumentTooLargeException
     *         The document can't be processed because it's too large. The maximum document size for synchronous
     *         operations 10 MB. The maximum document size for asynchronous operations is 500 MB for PDF files.
     * @throws BadDocumentException
     *         Amazon Textract isn't able to read the document. For more information on the document limits in Amazon
     *         Textract, see <a>limits</a>.
     * @throws AccessDeniedException
     *         You aren't authorized to perform the action. Use the Amazon Resource Name (ARN) of an authorized user or
     *         IAM role to perform the operation.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Textract.
     * @throws InternalServerErrorException
     *         Amazon Textract experienced a service issue. Try your call again.
     * @throws IdempotentParameterMismatchException
     *         A <code>ClientRequestToken</code> input parameter was reused with an operation, but at least one of the
     *         other input parameters is different from the previous call to the operation.
     * @throws ThrottlingException
     *         Amazon Textract is temporarily unable to process the request. Try your call again.
     * @throws LimitExceededException
     *         An Amazon Textract service limit was exceeded. For example, if you start too many asynchronous jobs
     *         concurrently, calls to start operations (<code>StartDocumentTextDetection</code>, for example) raise a
     *         LimitExceededException exception (HTTP status code: 400) until the number of concurrently running jobs is
     *         below the Amazon Textract service limit.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TextractException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TextractClient.StartDocumentTextDetection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/StartDocumentTextDetection"
     *      target="_top">AWS API Documentation</a>
     */
    default StartDocumentTextDetectionResponse startDocumentTextDetection(
            Consumer<StartDocumentTextDetectionRequest.Builder> startDocumentTextDetectionRequest)
            throws InvalidParameterException, InvalidS3ObjectException, InvalidKmsKeyException, UnsupportedDocumentException,
            DocumentTooLargeException, BadDocumentException, AccessDeniedException, ProvisionedThroughputExceededException,
            InternalServerErrorException, IdempotentParameterMismatchException, ThrottlingException, LimitExceededException,
            AwsServiceException, SdkClientException, TextractException {
        return startDocumentTextDetection(StartDocumentTextDetectionRequest.builder()
                .applyMutation(startDocumentTextDetectionRequest).build());
    }

    /**
     * <p>
     * Starts the asynchronous analysis of invoices or receipts for data like contact information, items purchased, and
     * vendor names.
     * </p>
     * <p>
     * <code>StartExpenseAnalysis</code> can analyze text in documents that are in JPEG, PNG, and PDF format. The
     * documents must be stored in an Amazon S3 bucket. Use the <a>DocumentLocation</a> parameter to specify the name of
     * your S3 bucket and the name of the document in that bucket.
     * </p>
     * <p>
     * <code>StartExpenseAnalysis</code> returns a job identifier (<code>JobId</code>) that you will provide to
     * <code>GetExpenseAnalysis</code> to retrieve the results of the operation. When the analysis of the input
     * invoices/receipts is finished, Amazon Textract publishes a completion status to the Amazon Simple Notification
     * Service (Amazon SNS) topic that you provide to the <code>NotificationChannel</code>. To obtain the results of the
     * invoice and receipt analysis operation, ensure that the status value published to the Amazon SNS topic is
     * <code>SUCCEEDED</code>. If so, call <a>GetExpenseAnalysis</a>, and pass the job identifier (<code>JobId</code>)
     * that was returned by your call to <code>StartExpenseAnalysis</code>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/textract/latest/dg/invoice-receipts.html">Analyzing Invoices and Receipts</a>.
     * </p>
     *
     * @param startExpenseAnalysisRequest
     * @return Result of the StartExpenseAnalysis operation returned by the service.
     * @throws InvalidParameterException
     *         An input parameter violated a constraint. For example, in synchronous operations, an
     *         <code>InvalidParameterException</code> exception occurs when neither of the <code>S3Object</code> or
     *         <code>Bytes</code> values are supplied in the <code>Document</code> request parameter. Validate your
     *         parameter before calling the API operation again.
     * @throws InvalidS3ObjectException
     *         Amazon Textract is unable to access the S3 object that's specified in the request. for more information,
     *         <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-access-control.html">Configure Access to
     *         Amazon S3</a> For troubleshooting information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/troubleshooting.html">Troubleshooting Amazon S3</a>
     * @throws InvalidKmsKeyException
     *         Indicates you do not have decrypt permissions with the KMS key entered, or the KMS key was entered
     *         incorrectly.
     * @throws UnsupportedDocumentException
     *         The format of the input document isn't supported. Documents for operations can be in PNG, JPEG, PDF, or
     *         TIFF format.
     * @throws DocumentTooLargeException
     *         The document can't be processed because it's too large. The maximum document size for synchronous
     *         operations 10 MB. The maximum document size for asynchronous operations is 500 MB for PDF files.
     * @throws BadDocumentException
     *         Amazon Textract isn't able to read the document. For more information on the document limits in Amazon
     *         Textract, see <a>limits</a>.
     * @throws AccessDeniedException
     *         You aren't authorized to perform the action. Use the Amazon Resource Name (ARN) of an authorized user or
     *         IAM role to perform the operation.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Textract.
     * @throws InternalServerErrorException
     *         Amazon Textract experienced a service issue. Try your call again.
     * @throws IdempotentParameterMismatchException
     *         A <code>ClientRequestToken</code> input parameter was reused with an operation, but at least one of the
     *         other input parameters is different from the previous call to the operation.
     * @throws ThrottlingException
     *         Amazon Textract is temporarily unable to process the request. Try your call again.
     * @throws LimitExceededException
     *         An Amazon Textract service limit was exceeded. For example, if you start too many asynchronous jobs
     *         concurrently, calls to start operations (<code>StartDocumentTextDetection</code>, for example) raise a
     *         LimitExceededException exception (HTTP status code: 400) until the number of concurrently running jobs is
     *         below the Amazon Textract service limit.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TextractException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TextractClient.StartExpenseAnalysis
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/StartExpenseAnalysis" target="_top">AWS
     *      API Documentation</a>
     */
    default StartExpenseAnalysisResponse startExpenseAnalysis(StartExpenseAnalysisRequest startExpenseAnalysisRequest)
            throws InvalidParameterException, InvalidS3ObjectException, InvalidKmsKeyException, UnsupportedDocumentException,
            DocumentTooLargeException, BadDocumentException, AccessDeniedException, ProvisionedThroughputExceededException,
            InternalServerErrorException, IdempotentParameterMismatchException, ThrottlingException, LimitExceededException,
            AwsServiceException, SdkClientException, TextractException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Starts the asynchronous analysis of invoices or receipts for data like contact information, items purchased, and
     * vendor names.
     * </p>
     * <p>
     * <code>StartExpenseAnalysis</code> can analyze text in documents that are in JPEG, PNG, and PDF format. The
     * documents must be stored in an Amazon S3 bucket. Use the <a>DocumentLocation</a> parameter to specify the name of
     * your S3 bucket and the name of the document in that bucket.
     * </p>
     * <p>
     * <code>StartExpenseAnalysis</code> returns a job identifier (<code>JobId</code>) that you will provide to
     * <code>GetExpenseAnalysis</code> to retrieve the results of the operation. When the analysis of the input
     * invoices/receipts is finished, Amazon Textract publishes a completion status to the Amazon Simple Notification
     * Service (Amazon SNS) topic that you provide to the <code>NotificationChannel</code>. To obtain the results of the
     * invoice and receipt analysis operation, ensure that the status value published to the Amazon SNS topic is
     * <code>SUCCEEDED</code>. If so, call <a>GetExpenseAnalysis</a>, and pass the job identifier (<code>JobId</code>)
     * that was returned by your call to <code>StartExpenseAnalysis</code>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/textract/latest/dg/invoice-receipts.html">Analyzing Invoices and Receipts</a>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link StartExpenseAnalysisRequest.Builder} avoiding the
     * need to create one manually via {@link StartExpenseAnalysisRequest#builder()}
     * </p>
     *
     * @param startExpenseAnalysisRequest
     *        A {@link Consumer} that will call methods on {@link StartExpenseAnalysisRequest.Builder} to create a
     *        request.
     * @return Result of the StartExpenseAnalysis operation returned by the service.
     * @throws InvalidParameterException
     *         An input parameter violated a constraint. For example, in synchronous operations, an
     *         <code>InvalidParameterException</code> exception occurs when neither of the <code>S3Object</code> or
     *         <code>Bytes</code> values are supplied in the <code>Document</code> request parameter. Validate your
     *         parameter before calling the API operation again.
     * @throws InvalidS3ObjectException
     *         Amazon Textract is unable to access the S3 object that's specified in the request. for more information,
     *         <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-access-control.html">Configure Access to
     *         Amazon S3</a> For troubleshooting information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/troubleshooting.html">Troubleshooting Amazon S3</a>
     * @throws InvalidKmsKeyException
     *         Indicates you do not have decrypt permissions with the KMS key entered, or the KMS key was entered
     *         incorrectly.
     * @throws UnsupportedDocumentException
     *         The format of the input document isn't supported. Documents for operations can be in PNG, JPEG, PDF, or
     *         TIFF format.
     * @throws DocumentTooLargeException
     *         The document can't be processed because it's too large. The maximum document size for synchronous
     *         operations 10 MB. The maximum document size for asynchronous operations is 500 MB for PDF files.
     * @throws BadDocumentException
     *         Amazon Textract isn't able to read the document. For more information on the document limits in Amazon
     *         Textract, see <a>limits</a>.
     * @throws AccessDeniedException
     *         You aren't authorized to perform the action. Use the Amazon Resource Name (ARN) of an authorized user or
     *         IAM role to perform the operation.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Textract.
     * @throws InternalServerErrorException
     *         Amazon Textract experienced a service issue. Try your call again.
     * @throws IdempotentParameterMismatchException
     *         A <code>ClientRequestToken</code> input parameter was reused with an operation, but at least one of the
     *         other input parameters is different from the previous call to the operation.
     * @throws ThrottlingException
     *         Amazon Textract is temporarily unable to process the request. Try your call again.
     * @throws LimitExceededException
     *         An Amazon Textract service limit was exceeded. For example, if you start too many asynchronous jobs
     *         concurrently, calls to start operations (<code>StartDocumentTextDetection</code>, for example) raise a
     *         LimitExceededException exception (HTTP status code: 400) until the number of concurrently running jobs is
     *         below the Amazon Textract service limit.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TextractException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TextractClient.StartExpenseAnalysis
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/StartExpenseAnalysis" target="_top">AWS
     *      API Documentation</a>
     */
    default StartExpenseAnalysisResponse startExpenseAnalysis(
            Consumer<StartExpenseAnalysisRequest.Builder> startExpenseAnalysisRequest) throws InvalidParameterException,
            InvalidS3ObjectException, InvalidKmsKeyException, UnsupportedDocumentException, DocumentTooLargeException,
            BadDocumentException, AccessDeniedException, ProvisionedThroughputExceededException, InternalServerErrorException,
            IdempotentParameterMismatchException, ThrottlingException, LimitExceededException, AwsServiceException,
            SdkClientException, TextractException {
        return startExpenseAnalysis(StartExpenseAnalysisRequest.builder().applyMutation(startExpenseAnalysisRequest).build());
    }

    /**
     * <p>
     * Starts the classification and analysis of an input document. <code>StartLendingAnalysis</code> initiates the
     * classification and analysis of a packet of lending documents. <code>StartLendingAnalysis</code> operates on a
     * document file located in an Amazon S3 bucket.
     * </p>
     * <p>
     * <code>StartLendingAnalysis</code> can analyze text in documents that are in one of the following formats: JPEG,
     * PNG, TIFF, PDF. Use <code>DocumentLocation</code> to specify the bucket name and the file name of the document.
     * </p>
     * <p>
     * <code>StartLendingAnalysis</code> returns a job identifier (<code>JobId</code>) that you use to get the results
     * of the operation. When the text analysis is finished, Amazon Textract publishes a completion status to the Amazon
     * Simple Notification Service (Amazon SNS) topic that you specify in <code>NotificationChannel</code>. To get the
     * results of the text analysis operation, first check that the status value published to the Amazon SNS topic is
     * SUCCEEDED. If the status is SUCCEEDED you can call either <code>GetLendingAnalysis</code> or
     * <code>GetLendingAnalysisSummary</code> and provide the <code>JobId</code> to obtain the results of the analysis.
     * </p>
     * <p>
     * If using <code>OutputConfig</code> to specify an Amazon S3 bucket, the output will be contained within the
     * specified prefix in a directory labeled with the job-id. In the directory there are 3 sub-directories:
     * </p>
     * <ul>
     * <li>
     * <p>
     * detailedResponse (contains the GetLendingAnalysis response)
     * </p>
     * </li>
     * <li>
     * <p>
     * summaryResponse (for the GetLendingAnalysisSummary response)
     * </p>
     * </li>
     * <li>
     * <p>
     * splitDocuments (documents split across logical boundaries)
     * </p>
     * </li>
     * </ul>
     *
     * @param startLendingAnalysisRequest
     * @return Result of the StartLendingAnalysis operation returned by the service.
     * @throws InvalidParameterException
     *         An input parameter violated a constraint. For example, in synchronous operations, an
     *         <code>InvalidParameterException</code> exception occurs when neither of the <code>S3Object</code> or
     *         <code>Bytes</code> values are supplied in the <code>Document</code> request parameter. Validate your
     *         parameter before calling the API operation again.
     * @throws InvalidS3ObjectException
     *         Amazon Textract is unable to access the S3 object that's specified in the request. for more information,
     *         <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-access-control.html">Configure Access to
     *         Amazon S3</a> For troubleshooting information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/troubleshooting.html">Troubleshooting Amazon S3</a>
     * @throws InvalidKmsKeyException
     *         Indicates you do not have decrypt permissions with the KMS key entered, or the KMS key was entered
     *         incorrectly.
     * @throws UnsupportedDocumentException
     *         The format of the input document isn't supported. Documents for operations can be in PNG, JPEG, PDF, or
     *         TIFF format.
     * @throws DocumentTooLargeException
     *         The document can't be processed because it's too large. The maximum document size for synchronous
     *         operations 10 MB. The maximum document size for asynchronous operations is 500 MB for PDF files.
     * @throws BadDocumentException
     *         Amazon Textract isn't able to read the document. For more information on the document limits in Amazon
     *         Textract, see <a>limits</a>.
     * @throws AccessDeniedException
     *         You aren't authorized to perform the action. Use the Amazon Resource Name (ARN) of an authorized user or
     *         IAM role to perform the operation.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Textract.
     * @throws InternalServerErrorException
     *         Amazon Textract experienced a service issue. Try your call again.
     * @throws IdempotentParameterMismatchException
     *         A <code>ClientRequestToken</code> input parameter was reused with an operation, but at least one of the
     *         other input parameters is different from the previous call to the operation.
     * @throws ThrottlingException
     *         Amazon Textract is temporarily unable to process the request. Try your call again.
     * @throws LimitExceededException
     *         An Amazon Textract service limit was exceeded. For example, if you start too many asynchronous jobs
     *         concurrently, calls to start operations (<code>StartDocumentTextDetection</code>, for example) raise a
     *         LimitExceededException exception (HTTP status code: 400) until the number of concurrently running jobs is
     *         below the Amazon Textract service limit.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TextractException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TextractClient.StartLendingAnalysis
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/StartLendingAnalysis" target="_top">AWS
     *      API Documentation</a>
     */
    default StartLendingAnalysisResponse startLendingAnalysis(StartLendingAnalysisRequest startLendingAnalysisRequest)
            throws InvalidParameterException, InvalidS3ObjectException, InvalidKmsKeyException, UnsupportedDocumentException,
            DocumentTooLargeException, BadDocumentException, AccessDeniedException, ProvisionedThroughputExceededException,
            InternalServerErrorException, IdempotentParameterMismatchException, ThrottlingException, LimitExceededException,
            AwsServiceException, SdkClientException, TextractException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Starts the classification and analysis of an input document. <code>StartLendingAnalysis</code> initiates the
     * classification and analysis of a packet of lending documents. <code>StartLendingAnalysis</code> operates on a
     * document file located in an Amazon S3 bucket.
     * </p>
     * <p>
     * <code>StartLendingAnalysis</code> can analyze text in documents that are in one of the following formats: JPEG,
     * PNG, TIFF, PDF. Use <code>DocumentLocation</code> to specify the bucket name and the file name of the document.
     * </p>
     * <p>
     * <code>StartLendingAnalysis</code> returns a job identifier (<code>JobId</code>) that you use to get the results
     * of the operation. When the text analysis is finished, Amazon Textract publishes a completion status to the Amazon
     * Simple Notification Service (Amazon SNS) topic that you specify in <code>NotificationChannel</code>. To get the
     * results of the text analysis operation, first check that the status value published to the Amazon SNS topic is
     * SUCCEEDED. If the status is SUCCEEDED you can call either <code>GetLendingAnalysis</code> or
     * <code>GetLendingAnalysisSummary</code> and provide the <code>JobId</code> to obtain the results of the analysis.
     * </p>
     * <p>
     * If using <code>OutputConfig</code> to specify an Amazon S3 bucket, the output will be contained within the
     * specified prefix in a directory labeled with the job-id. In the directory there are 3 sub-directories:
     * </p>
     * <ul>
     * <li>
     * <p>
     * detailedResponse (contains the GetLendingAnalysis response)
     * </p>
     * </li>
     * <li>
     * <p>
     * summaryResponse (for the GetLendingAnalysisSummary response)
     * </p>
     * </li>
     * <li>
     * <p>
     * splitDocuments (documents split across logical boundaries)
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link StartLendingAnalysisRequest.Builder} avoiding the
     * need to create one manually via {@link StartLendingAnalysisRequest#builder()}
     * </p>
     *
     * @param startLendingAnalysisRequest
     *        A {@link Consumer} that will call methods on {@link StartLendingAnalysisRequest.Builder} to create a
     *        request.
     * @return Result of the StartLendingAnalysis operation returned by the service.
     * @throws InvalidParameterException
     *         An input parameter violated a constraint. For example, in synchronous operations, an
     *         <code>InvalidParameterException</code> exception occurs when neither of the <code>S3Object</code> or
     *         <code>Bytes</code> values are supplied in the <code>Document</code> request parameter. Validate your
     *         parameter before calling the API operation again.
     * @throws InvalidS3ObjectException
     *         Amazon Textract is unable to access the S3 object that's specified in the request. for more information,
     *         <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-access-control.html">Configure Access to
     *         Amazon S3</a> For troubleshooting information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/troubleshooting.html">Troubleshooting Amazon S3</a>
     * @throws InvalidKmsKeyException
     *         Indicates you do not have decrypt permissions with the KMS key entered, or the KMS key was entered
     *         incorrectly.
     * @throws UnsupportedDocumentException
     *         The format of the input document isn't supported. Documents for operations can be in PNG, JPEG, PDF, or
     *         TIFF format.
     * @throws DocumentTooLargeException
     *         The document can't be processed because it's too large. The maximum document size for synchronous
     *         operations 10 MB. The maximum document size for asynchronous operations is 500 MB for PDF files.
     * @throws BadDocumentException
     *         Amazon Textract isn't able to read the document. For more information on the document limits in Amazon
     *         Textract, see <a>limits</a>.
     * @throws AccessDeniedException
     *         You aren't authorized to perform the action. Use the Amazon Resource Name (ARN) of an authorized user or
     *         IAM role to perform the operation.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Textract.
     * @throws InternalServerErrorException
     *         Amazon Textract experienced a service issue. Try your call again.
     * @throws IdempotentParameterMismatchException
     *         A <code>ClientRequestToken</code> input parameter was reused with an operation, but at least one of the
     *         other input parameters is different from the previous call to the operation.
     * @throws ThrottlingException
     *         Amazon Textract is temporarily unable to process the request. Try your call again.
     * @throws LimitExceededException
     *         An Amazon Textract service limit was exceeded. For example, if you start too many asynchronous jobs
     *         concurrently, calls to start operations (<code>StartDocumentTextDetection</code>, for example) raise a
     *         LimitExceededException exception (HTTP status code: 400) until the number of concurrently running jobs is
     *         below the Amazon Textract service limit.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TextractException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TextractClient.StartLendingAnalysis
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/StartLendingAnalysis" target="_top">AWS
     *      API Documentation</a>
     */
    default StartLendingAnalysisResponse startLendingAnalysis(
            Consumer<StartLendingAnalysisRequest.Builder> startLendingAnalysisRequest) throws InvalidParameterException,
            InvalidS3ObjectException, InvalidKmsKeyException, UnsupportedDocumentException, DocumentTooLargeException,
            BadDocumentException, AccessDeniedException, ProvisionedThroughputExceededException, InternalServerErrorException,
            IdempotentParameterMismatchException, ThrottlingException, LimitExceededException, AwsServiceException,
            SdkClientException, TextractException {
        return startLendingAnalysis(StartLendingAnalysisRequest.builder().applyMutation(startLendingAnalysisRequest).build());
    }

    /**
     * Create a {@link TextractClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static TextractClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link TextractClient}.
     */
    static TextractClientBuilder builder() {
        return new DefaultTextractClientBuilder();
    }

    static ServiceMetadata serviceMetadata() {
        return ServiceMetadata.of(SERVICE_METADATA_ID);
    }

    @Override
    TextractServiceClientConfiguration serviceClientConfiguration();
}
