/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.textract.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information on the adapter, including the adapter ID, Name, Creation time, and feature types.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AdapterOverview implements SdkPojo, Serializable, ToCopyableBuilder<AdapterOverview.Builder, AdapterOverview> {
    private static final SdkField<String> ADAPTER_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AdapterId").getter(getter(AdapterOverview::adapterId)).setter(setter(Builder::adapterId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AdapterId").build()).build();

    private static final SdkField<String> ADAPTER_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AdapterName").getter(getter(AdapterOverview::adapterName)).setter(setter(Builder::adapterName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AdapterName").build()).build();

    private static final SdkField<Instant> CREATION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreationTime").getter(getter(AdapterOverview::creationTime)).setter(setter(Builder::creationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationTime").build()).build();

    private static final SdkField<List<String>> FEATURE_TYPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("FeatureTypes")
            .getter(getter(AdapterOverview::featureTypesAsStrings))
            .setter(setter(Builder::featureTypesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FeatureTypes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ADAPTER_ID_FIELD,
            ADAPTER_NAME_FIELD, CREATION_TIME_FIELD, FEATURE_TYPES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String adapterId;

    private final String adapterName;

    private final Instant creationTime;

    private final List<String> featureTypes;

    private AdapterOverview(BuilderImpl builder) {
        this.adapterId = builder.adapterId;
        this.adapterName = builder.adapterName;
        this.creationTime = builder.creationTime;
        this.featureTypes = builder.featureTypes;
    }

    /**
     * <p>
     * A unique identifier for the adapter resource.
     * </p>
     * 
     * @return A unique identifier for the adapter resource.
     */
    public final String adapterId() {
        return adapterId;
    }

    /**
     * <p>
     * A string naming the adapter resource.
     * </p>
     * 
     * @return A string naming the adapter resource.
     */
    public final String adapterName() {
        return adapterName;
    }

    /**
     * <p>
     * The date and time that the adapter was created.
     * </p>
     * 
     * @return The date and time that the adapter was created.
     */
    public final Instant creationTime() {
        return creationTime;
    }

    /**
     * <p>
     * The feature types that the adapter is operating on.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasFeatureTypes} method.
     * </p>
     * 
     * @return The feature types that the adapter is operating on.
     */
    public final List<FeatureType> featureTypes() {
        return FeatureTypesCopier.copyStringToEnum(featureTypes);
    }

    /**
     * For responses, this returns true if the service returned a value for the FeatureTypes property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasFeatureTypes() {
        return featureTypes != null && !(featureTypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The feature types that the adapter is operating on.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasFeatureTypes} method.
     * </p>
     * 
     * @return The feature types that the adapter is operating on.
     */
    public final List<String> featureTypesAsStrings() {
        return featureTypes;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(adapterId());
        hashCode = 31 * hashCode + Objects.hashCode(adapterName());
        hashCode = 31 * hashCode + Objects.hashCode(creationTime());
        hashCode = 31 * hashCode + Objects.hashCode(hasFeatureTypes() ? featureTypesAsStrings() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AdapterOverview)) {
            return false;
        }
        AdapterOverview other = (AdapterOverview) obj;
        return Objects.equals(adapterId(), other.adapterId()) && Objects.equals(adapterName(), other.adapterName())
                && Objects.equals(creationTime(), other.creationTime()) && hasFeatureTypes() == other.hasFeatureTypes()
                && Objects.equals(featureTypesAsStrings(), other.featureTypesAsStrings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AdapterOverview").add("AdapterId", adapterId()).add("AdapterName", adapterName())
                .add("CreationTime", creationTime()).add("FeatureTypes", hasFeatureTypes() ? featureTypesAsStrings() : null)
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AdapterId":
            return Optional.ofNullable(clazz.cast(adapterId()));
        case "AdapterName":
            return Optional.ofNullable(clazz.cast(adapterName()));
        case "CreationTime":
            return Optional.ofNullable(clazz.cast(creationTime()));
        case "FeatureTypes":
            return Optional.ofNullable(clazz.cast(featureTypesAsStrings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("AdapterId", ADAPTER_ID_FIELD);
        map.put("AdapterName", ADAPTER_NAME_FIELD);
        map.put("CreationTime", CREATION_TIME_FIELD);
        map.put("FeatureTypes", FEATURE_TYPES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<AdapterOverview, T> g) {
        return obj -> g.apply((AdapterOverview) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AdapterOverview> {
        /**
         * <p>
         * A unique identifier for the adapter resource.
         * </p>
         * 
         * @param adapterId
         *        A unique identifier for the adapter resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder adapterId(String adapterId);

        /**
         * <p>
         * A string naming the adapter resource.
         * </p>
         * 
         * @param adapterName
         *        A string naming the adapter resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder adapterName(String adapterName);

        /**
         * <p>
         * The date and time that the adapter was created.
         * </p>
         * 
         * @param creationTime
         *        The date and time that the adapter was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTime(Instant creationTime);

        /**
         * <p>
         * The feature types that the adapter is operating on.
         * </p>
         * 
         * @param featureTypes
         *        The feature types that the adapter is operating on.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder featureTypesWithStrings(Collection<String> featureTypes);

        /**
         * <p>
         * The feature types that the adapter is operating on.
         * </p>
         * 
         * @param featureTypes
         *        The feature types that the adapter is operating on.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder featureTypesWithStrings(String... featureTypes);

        /**
         * <p>
         * The feature types that the adapter is operating on.
         * </p>
         * 
         * @param featureTypes
         *        The feature types that the adapter is operating on.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder featureTypes(Collection<FeatureType> featureTypes);

        /**
         * <p>
         * The feature types that the adapter is operating on.
         * </p>
         * 
         * @param featureTypes
         *        The feature types that the adapter is operating on.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder featureTypes(FeatureType... featureTypes);
    }

    static final class BuilderImpl implements Builder {
        private String adapterId;

        private String adapterName;

        private Instant creationTime;

        private List<String> featureTypes = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(AdapterOverview model) {
            adapterId(model.adapterId);
            adapterName(model.adapterName);
            creationTime(model.creationTime);
            featureTypesWithStrings(model.featureTypes);
        }

        public final String getAdapterId() {
            return adapterId;
        }

        public final void setAdapterId(String adapterId) {
            this.adapterId = adapterId;
        }

        @Override
        public final Builder adapterId(String adapterId) {
            this.adapterId = adapterId;
            return this;
        }

        public final String getAdapterName() {
            return adapterName;
        }

        public final void setAdapterName(String adapterName) {
            this.adapterName = adapterName;
        }

        @Override
        public final Builder adapterName(String adapterName) {
            this.adapterName = adapterName;
            return this;
        }

        public final Instant getCreationTime() {
            return creationTime;
        }

        public final void setCreationTime(Instant creationTime) {
            this.creationTime = creationTime;
        }

        @Override
        public final Builder creationTime(Instant creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        public final Collection<String> getFeatureTypes() {
            if (featureTypes instanceof SdkAutoConstructList) {
                return null;
            }
            return featureTypes;
        }

        public final void setFeatureTypes(Collection<String> featureTypes) {
            this.featureTypes = FeatureTypesCopier.copy(featureTypes);
        }

        @Override
        public final Builder featureTypesWithStrings(Collection<String> featureTypes) {
            this.featureTypes = FeatureTypesCopier.copy(featureTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder featureTypesWithStrings(String... featureTypes) {
            featureTypesWithStrings(Arrays.asList(featureTypes));
            return this;
        }

        @Override
        public final Builder featureTypes(Collection<FeatureType> featureTypes) {
            this.featureTypes = FeatureTypesCopier.copyEnumToString(featureTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder featureTypes(FeatureType... featureTypes) {
            featureTypes(Arrays.asList(featureTypes));
            return this;
        }

        @Override
        public AdapterOverview build() {
            return new AdapterOverview(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
