/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.textract;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.textract.internal.ServiceVersionInfo;
import software.amazon.awssdk.services.textract.internal.TextractServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.textract.model.AccessDeniedException;
import software.amazon.awssdk.services.textract.model.AnalyzeDocumentRequest;
import software.amazon.awssdk.services.textract.model.AnalyzeDocumentResponse;
import software.amazon.awssdk.services.textract.model.AnalyzeExpenseRequest;
import software.amazon.awssdk.services.textract.model.AnalyzeExpenseResponse;
import software.amazon.awssdk.services.textract.model.AnalyzeIdRequest;
import software.amazon.awssdk.services.textract.model.AnalyzeIdResponse;
import software.amazon.awssdk.services.textract.model.BadDocumentException;
import software.amazon.awssdk.services.textract.model.ConflictException;
import software.amazon.awssdk.services.textract.model.CreateAdapterRequest;
import software.amazon.awssdk.services.textract.model.CreateAdapterResponse;
import software.amazon.awssdk.services.textract.model.CreateAdapterVersionRequest;
import software.amazon.awssdk.services.textract.model.CreateAdapterVersionResponse;
import software.amazon.awssdk.services.textract.model.DeleteAdapterRequest;
import software.amazon.awssdk.services.textract.model.DeleteAdapterResponse;
import software.amazon.awssdk.services.textract.model.DeleteAdapterVersionRequest;
import software.amazon.awssdk.services.textract.model.DeleteAdapterVersionResponse;
import software.amazon.awssdk.services.textract.model.DetectDocumentTextRequest;
import software.amazon.awssdk.services.textract.model.DetectDocumentTextResponse;
import software.amazon.awssdk.services.textract.model.DocumentTooLargeException;
import software.amazon.awssdk.services.textract.model.GetAdapterRequest;
import software.amazon.awssdk.services.textract.model.GetAdapterResponse;
import software.amazon.awssdk.services.textract.model.GetAdapterVersionRequest;
import software.amazon.awssdk.services.textract.model.GetAdapterVersionResponse;
import software.amazon.awssdk.services.textract.model.GetDocumentAnalysisRequest;
import software.amazon.awssdk.services.textract.model.GetDocumentAnalysisResponse;
import software.amazon.awssdk.services.textract.model.GetDocumentTextDetectionRequest;
import software.amazon.awssdk.services.textract.model.GetDocumentTextDetectionResponse;
import software.amazon.awssdk.services.textract.model.GetExpenseAnalysisRequest;
import software.amazon.awssdk.services.textract.model.GetExpenseAnalysisResponse;
import software.amazon.awssdk.services.textract.model.GetLendingAnalysisRequest;
import software.amazon.awssdk.services.textract.model.GetLendingAnalysisResponse;
import software.amazon.awssdk.services.textract.model.GetLendingAnalysisSummaryRequest;
import software.amazon.awssdk.services.textract.model.GetLendingAnalysisSummaryResponse;
import software.amazon.awssdk.services.textract.model.HumanLoopQuotaExceededException;
import software.amazon.awssdk.services.textract.model.IdempotentParameterMismatchException;
import software.amazon.awssdk.services.textract.model.InternalServerErrorException;
import software.amazon.awssdk.services.textract.model.InvalidJobIdException;
import software.amazon.awssdk.services.textract.model.InvalidKmsKeyException;
import software.amazon.awssdk.services.textract.model.InvalidParameterException;
import software.amazon.awssdk.services.textract.model.InvalidS3ObjectException;
import software.amazon.awssdk.services.textract.model.LimitExceededException;
import software.amazon.awssdk.services.textract.model.ListAdapterVersionsRequest;
import software.amazon.awssdk.services.textract.model.ListAdapterVersionsResponse;
import software.amazon.awssdk.services.textract.model.ListAdaptersRequest;
import software.amazon.awssdk.services.textract.model.ListAdaptersResponse;
import software.amazon.awssdk.services.textract.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.textract.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.textract.model.ProvisionedThroughputExceededException;
import software.amazon.awssdk.services.textract.model.ResourceNotFoundException;
import software.amazon.awssdk.services.textract.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.textract.model.StartDocumentAnalysisRequest;
import software.amazon.awssdk.services.textract.model.StartDocumentAnalysisResponse;
import software.amazon.awssdk.services.textract.model.StartDocumentTextDetectionRequest;
import software.amazon.awssdk.services.textract.model.StartDocumentTextDetectionResponse;
import software.amazon.awssdk.services.textract.model.StartExpenseAnalysisRequest;
import software.amazon.awssdk.services.textract.model.StartExpenseAnalysisResponse;
import software.amazon.awssdk.services.textract.model.StartLendingAnalysisRequest;
import software.amazon.awssdk.services.textract.model.StartLendingAnalysisResponse;
import software.amazon.awssdk.services.textract.model.TagResourceRequest;
import software.amazon.awssdk.services.textract.model.TagResourceResponse;
import software.amazon.awssdk.services.textract.model.TextractException;
import software.amazon.awssdk.services.textract.model.ThrottlingException;
import software.amazon.awssdk.services.textract.model.UnsupportedDocumentException;
import software.amazon.awssdk.services.textract.model.UntagResourceRequest;
import software.amazon.awssdk.services.textract.model.UntagResourceResponse;
import software.amazon.awssdk.services.textract.model.UpdateAdapterRequest;
import software.amazon.awssdk.services.textract.model.UpdateAdapterResponse;
import software.amazon.awssdk.services.textract.model.ValidationException;
import software.amazon.awssdk.services.textract.transform.AnalyzeDocumentRequestMarshaller;
import software.amazon.awssdk.services.textract.transform.AnalyzeExpenseRequestMarshaller;
import software.amazon.awssdk.services.textract.transform.AnalyzeIdRequestMarshaller;
import software.amazon.awssdk.services.textract.transform.CreateAdapterRequestMarshaller;
import software.amazon.awssdk.services.textract.transform.CreateAdapterVersionRequestMarshaller;
import software.amazon.awssdk.services.textract.transform.DeleteAdapterRequestMarshaller;
import software.amazon.awssdk.services.textract.transform.DeleteAdapterVersionRequestMarshaller;
import software.amazon.awssdk.services.textract.transform.DetectDocumentTextRequestMarshaller;
import software.amazon.awssdk.services.textract.transform.GetAdapterRequestMarshaller;
import software.amazon.awssdk.services.textract.transform.GetAdapterVersionRequestMarshaller;
import software.amazon.awssdk.services.textract.transform.GetDocumentAnalysisRequestMarshaller;
import software.amazon.awssdk.services.textract.transform.GetDocumentTextDetectionRequestMarshaller;
import software.amazon.awssdk.services.textract.transform.GetExpenseAnalysisRequestMarshaller;
import software.amazon.awssdk.services.textract.transform.GetLendingAnalysisRequestMarshaller;
import software.amazon.awssdk.services.textract.transform.GetLendingAnalysisSummaryRequestMarshaller;
import software.amazon.awssdk.services.textract.transform.ListAdapterVersionsRequestMarshaller;
import software.amazon.awssdk.services.textract.transform.ListAdaptersRequestMarshaller;
import software.amazon.awssdk.services.textract.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.textract.transform.StartDocumentAnalysisRequestMarshaller;
import software.amazon.awssdk.services.textract.transform.StartDocumentTextDetectionRequestMarshaller;
import software.amazon.awssdk.services.textract.transform.StartExpenseAnalysisRequestMarshaller;
import software.amazon.awssdk.services.textract.transform.StartLendingAnalysisRequestMarshaller;
import software.amazon.awssdk.services.textract.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.textract.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.textract.transform.UpdateAdapterRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link TextractClient}.
 *
 * @see TextractClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultTextractClient implements TextractClient {
    private static final Logger log = Logger.loggerFor(DefaultTextractClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultTextractClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this)
                .option(SdkClientOption.API_METADATA, "Textract" + "#" + ServiceVersionInfo.VERSION).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Analyzes an input document for relationships between detected items.
     * </p>
     * <p>
     * The types of information returned are as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Form data (key-value pairs). The related information is returned in two <a>Block</a> objects, each of type
     * <code>KEY_VALUE_SET</code>: a KEY <code>Block</code> object and a VALUE <code>Block</code> object. For example,
     * <i>Name: Ana Silva Carolina</i> contains a key and value. <i>Name:</i> is the key. <i>Ana Silva Carolina</i> is
     * the value.
     * </p>
     * </li>
     * <li>
     * <p>
     * Table and table cell data. A TABLE <code>Block</code> object contains information about a detected table. A CELL
     * <code>Block</code> object is returned for each cell in a table.
     * </p>
     * </li>
     * <li>
     * <p>
     * Lines and words of text. A LINE <code>Block</code> object contains one or more WORD <code>Block</code> objects.
     * All lines and words that are detected in the document are returned (including text that doesn't have a
     * relationship with the value of <code>FeatureTypes</code>).
     * </p>
     * </li>
     * <li>
     * <p>
     * Signatures. A SIGNATURE <code>Block</code> object contains the location information of a signature in a document.
     * If used in conjunction with forms or tables, a signature can be given a Key-Value pairing or be detected in the
     * cell of a table.
     * </p>
     * </li>
     * <li>
     * <p>
     * Query. A QUERY Block object contains the query text, alias and link to the associated Query results block object.
     * </p>
     * </li>
     * <li>
     * <p>
     * Query Result. A QUERY_RESULT Block object contains the answer to the query and an ID that connects it to the
     * query asked. This Block also contains a confidence score.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Selection elements such as check boxes and option buttons (radio buttons) can be detected in form data and in
     * tables. A SELECTION_ELEMENT <code>Block</code> object contains information about a selection element, including
     * the selection status.
     * </p>
     * <p>
     * You can choose which type of analysis to perform by specifying the <code>FeatureTypes</code> list.
     * </p>
     * <p>
     * The output is returned in a list of <code>Block</code> objects.
     * </p>
     * <p>
     * <code>AnalyzeDocument</code> is a synchronous operation. To analyze documents asynchronously, use
     * <a>StartDocumentAnalysis</a>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/textract/latest/dg/how-it-works-analyzing.html">Document Text Analysis</a>.
     * </p>
     *
     * @param analyzeDocumentRequest
     * @return Result of the AnalyzeDocument operation returned by the service.
     * @throws InvalidParameterException
     *         An input parameter violated a constraint. For example, in synchronous operations, an
     *         <code>InvalidParameterException</code> exception occurs when neither of the <code>S3Object</code> or
     *         <code>Bytes</code> values are supplied in the <code>Document</code> request parameter. Validate your
     *         parameter before calling the API operation again.
     * @throws InvalidS3ObjectException
     *         Amazon Textract is unable to access the S3 object that's specified in the request. for more information,
     *         <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-access-control.html">Configure Access to
     *         Amazon S3</a> For troubleshooting information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/troubleshooting.html">Troubleshooting Amazon S3</a>
     * @throws UnsupportedDocumentException
     *         The format of the input document isn't supported. Documents for operations can be in PNG, JPEG, PDF, or
     *         TIFF format.
     * @throws DocumentTooLargeException
     *         The document can't be processed because it's too large. The maximum document size for synchronous
     *         operations 10 MB. The maximum document size for asynchronous operations is 500 MB for PDF files.
     * @throws BadDocumentException
     *         Amazon Textract isn't able to read the document. For more information on the document limits in Amazon
     *         Textract, see <a>limits</a>.
     * @throws AccessDeniedException
     *         You aren't authorized to perform the action. Use the Amazon Resource Name (ARN) of an authorized user or
     *         IAM role to perform the operation.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Textract.
     * @throws InternalServerErrorException
     *         Amazon Textract experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Textract is temporarily unable to process the request. Try your call again.
     * @throws HumanLoopQuotaExceededException
     *         Indicates you have exceeded the maximum number of active human in the loop workflows available
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TextractException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TextractClient.AnalyzeDocument
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/AnalyzeDocument" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public AnalyzeDocumentResponse analyzeDocument(AnalyzeDocumentRequest analyzeDocumentRequest)
            throws InvalidParameterException, InvalidS3ObjectException, UnsupportedDocumentException, DocumentTooLargeException,
            BadDocumentException, AccessDeniedException, ProvisionedThroughputExceededException, InternalServerErrorException,
            ThrottlingException, HumanLoopQuotaExceededException, AwsServiceException, SdkClientException, TextractException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AnalyzeDocumentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                AnalyzeDocumentResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BadDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadDocumentException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "HumanLoopQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("HumanLoopQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(HumanLoopQuotaExceededException::builder).build());
            case "InvalidKMSKeyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidKMSKeyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidKmsKeyException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "DocumentTooLargeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DocumentTooLargeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DocumentTooLargeException::builder).build());
            case "UnsupportedDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedDocumentException::builder).build());
            case "InvalidS3ObjectException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidS3ObjectException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidS3ObjectException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "IdempotentParameterMismatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("IdempotentParameterMismatchException")
                        .httpStatusCode(400).exceptionBuilderSupplier(IdempotentParameterMismatchException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ProvisionedThroughputExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ProvisionedThroughputExceededException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ProvisionedThroughputExceededException::builder).build());
            case "InvalidJobIdException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidJobIdException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidJobIdException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(analyzeDocumentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, analyzeDocumentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Textract");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AnalyzeDocument");

            return clientHandler.execute(new ClientExecutionParams<AnalyzeDocumentRequest, AnalyzeDocumentResponse>()
                    .withOperationName("AnalyzeDocument").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(analyzeDocumentRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new AnalyzeDocumentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * <code>AnalyzeExpense</code> synchronously analyzes an input document for financially related relationships
     * between text.
     * </p>
     * <p>
     * Information is returned as <code>ExpenseDocuments</code> and seperated as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>LineItemGroups</code>- A data set containing <code>LineItems</code> which store information about the lines
     * of text, such as an item purchased and its price on a receipt.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SummaryFields</code>- Contains all other information a receipt, such as header information or the vendors
     * name.
     * </p>
     * </li>
     * </ul>
     *
     * @param analyzeExpenseRequest
     * @return Result of the AnalyzeExpense operation returned by the service.
     * @throws InvalidParameterException
     *         An input parameter violated a constraint. For example, in synchronous operations, an
     *         <code>InvalidParameterException</code> exception occurs when neither of the <code>S3Object</code> or
     *         <code>Bytes</code> values are supplied in the <code>Document</code> request parameter. Validate your
     *         parameter before calling the API operation again.
     * @throws InvalidS3ObjectException
     *         Amazon Textract is unable to access the S3 object that's specified in the request. for more information,
     *         <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-access-control.html">Configure Access to
     *         Amazon S3</a> For troubleshooting information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/troubleshooting.html">Troubleshooting Amazon S3</a>
     * @throws UnsupportedDocumentException
     *         The format of the input document isn't supported. Documents for operations can be in PNG, JPEG, PDF, or
     *         TIFF format.
     * @throws DocumentTooLargeException
     *         The document can't be processed because it's too large. The maximum document size for synchronous
     *         operations 10 MB. The maximum document size for asynchronous operations is 500 MB for PDF files.
     * @throws BadDocumentException
     *         Amazon Textract isn't able to read the document. For more information on the document limits in Amazon
     *         Textract, see <a>limits</a>.
     * @throws AccessDeniedException
     *         You aren't authorized to perform the action. Use the Amazon Resource Name (ARN) of an authorized user or
     *         IAM role to perform the operation.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Textract.
     * @throws InternalServerErrorException
     *         Amazon Textract experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Textract is temporarily unable to process the request. Try your call again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TextractException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TextractClient.AnalyzeExpense
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/AnalyzeExpense" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public AnalyzeExpenseResponse analyzeExpense(AnalyzeExpenseRequest analyzeExpenseRequest) throws InvalidParameterException,
            InvalidS3ObjectException, UnsupportedDocumentException, DocumentTooLargeException, BadDocumentException,
            AccessDeniedException, ProvisionedThroughputExceededException, InternalServerErrorException, ThrottlingException,
            AwsServiceException, SdkClientException, TextractException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AnalyzeExpenseResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                AnalyzeExpenseResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BadDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadDocumentException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "HumanLoopQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("HumanLoopQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(HumanLoopQuotaExceededException::builder).build());
            case "InvalidKMSKeyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidKMSKeyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidKmsKeyException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "DocumentTooLargeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DocumentTooLargeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DocumentTooLargeException::builder).build());
            case "UnsupportedDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedDocumentException::builder).build());
            case "InvalidS3ObjectException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidS3ObjectException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidS3ObjectException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "IdempotentParameterMismatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("IdempotentParameterMismatchException")
                        .httpStatusCode(400).exceptionBuilderSupplier(IdempotentParameterMismatchException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ProvisionedThroughputExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ProvisionedThroughputExceededException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ProvisionedThroughputExceededException::builder).build());
            case "InvalidJobIdException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidJobIdException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidJobIdException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(analyzeExpenseRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, analyzeExpenseRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Textract");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AnalyzeExpense");

            return clientHandler.execute(new ClientExecutionParams<AnalyzeExpenseRequest, AnalyzeExpenseResponse>()
                    .withOperationName("AnalyzeExpense").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(analyzeExpenseRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new AnalyzeExpenseRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Analyzes identity documents for relevant information. This information is extracted and returned as
     * <code>IdentityDocumentFields</code>, which records both the normalized field and value of the extracted text.
     * Unlike other Amazon Textract operations, <code>AnalyzeID</code> doesn't return any Geometry data.
     * </p>
     *
     * @param analyzeIdRequest
     * @return Result of the AnalyzeID operation returned by the service.
     * @throws InvalidParameterException
     *         An input parameter violated a constraint. For example, in synchronous operations, an
     *         <code>InvalidParameterException</code> exception occurs when neither of the <code>S3Object</code> or
     *         <code>Bytes</code> values are supplied in the <code>Document</code> request parameter. Validate your
     *         parameter before calling the API operation again.
     * @throws InvalidS3ObjectException
     *         Amazon Textract is unable to access the S3 object that's specified in the request. for more information,
     *         <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-access-control.html">Configure Access to
     *         Amazon S3</a> For troubleshooting information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/troubleshooting.html">Troubleshooting Amazon S3</a>
     * @throws UnsupportedDocumentException
     *         The format of the input document isn't supported. Documents for operations can be in PNG, JPEG, PDF, or
     *         TIFF format.
     * @throws DocumentTooLargeException
     *         The document can't be processed because it's too large. The maximum document size for synchronous
     *         operations 10 MB. The maximum document size for asynchronous operations is 500 MB for PDF files.
     * @throws BadDocumentException
     *         Amazon Textract isn't able to read the document. For more information on the document limits in Amazon
     *         Textract, see <a>limits</a>.
     * @throws AccessDeniedException
     *         You aren't authorized to perform the action. Use the Amazon Resource Name (ARN) of an authorized user or
     *         IAM role to perform the operation.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Textract.
     * @throws InternalServerErrorException
     *         Amazon Textract experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Textract is temporarily unable to process the request. Try your call again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TextractException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TextractClient.AnalyzeID
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/AnalyzeID" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public AnalyzeIdResponse analyzeID(AnalyzeIdRequest analyzeIdRequest) throws InvalidParameterException,
            InvalidS3ObjectException, UnsupportedDocumentException, DocumentTooLargeException, BadDocumentException,
            AccessDeniedException, ProvisionedThroughputExceededException, InternalServerErrorException, ThrottlingException,
            AwsServiceException, SdkClientException, TextractException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AnalyzeIdResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                AnalyzeIdResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BadDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadDocumentException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "HumanLoopQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("HumanLoopQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(HumanLoopQuotaExceededException::builder).build());
            case "InvalidKMSKeyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidKMSKeyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidKmsKeyException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "DocumentTooLargeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DocumentTooLargeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DocumentTooLargeException::builder).build());
            case "UnsupportedDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedDocumentException::builder).build());
            case "InvalidS3ObjectException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidS3ObjectException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidS3ObjectException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "IdempotentParameterMismatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("IdempotentParameterMismatchException")
                        .httpStatusCode(400).exceptionBuilderSupplier(IdempotentParameterMismatchException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ProvisionedThroughputExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ProvisionedThroughputExceededException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ProvisionedThroughputExceededException::builder).build());
            case "InvalidJobIdException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidJobIdException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidJobIdException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(analyzeIdRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, analyzeIdRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Textract");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AnalyzeID");

            return clientHandler.execute(new ClientExecutionParams<AnalyzeIdRequest, AnalyzeIdResponse>()
                    .withOperationName("AnalyzeID").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(analyzeIdRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new AnalyzeIdRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates an adapter, which can be fine-tuned for enhanced performance on user provided documents. Takes an
     * AdapterName and FeatureType. Currently the only supported feature type is <code>QUERIES</code>. You can also
     * provide a Description, Tags, and a ClientRequestToken. You can choose whether or not the adapter should be
     * AutoUpdated with the AutoUpdate argument. By default, AutoUpdate is set to DISABLED.
     * </p>
     *
     * @param createAdapterRequest
     * @return Result of the CreateAdapter operation returned by the service.
     * @throws InvalidParameterException
     *         An input parameter violated a constraint. For example, in synchronous operations, an
     *         <code>InvalidParameterException</code> exception occurs when neither of the <code>S3Object</code> or
     *         <code>Bytes</code> values are supplied in the <code>Document</code> request parameter. Validate your
     *         parameter before calling the API operation again.
     * @throws AccessDeniedException
     *         You aren't authorized to perform the action. Use the Amazon Resource Name (ARN) of an authorized user or
     *         IAM role to perform the operation.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Textract.
     * @throws InternalServerErrorException
     *         Amazon Textract experienced a service issue. Try your call again.
     * @throws IdempotentParameterMismatchException
     *         A <code>ClientRequestToken</code> input parameter was reused with an operation, but at least one of the
     *         other input parameters is different from the previous call to the operation.
     * @throws ThrottlingException
     *         Amazon Textract is temporarily unable to process the request. Try your call again.
     * @throws LimitExceededException
     *         An Amazon Textract service limit was exceeded. For example, if you start too many asynchronous jobs
     *         concurrently, calls to start operations (<code>StartDocumentTextDetection</code>, for example) raise a
     *         LimitExceededException exception (HTTP status code: 400) until the number of concurrently running jobs is
     *         below the Amazon Textract service limit.
     * @throws ValidationException
     *         Indicates that a request was not valid. Check request for proper formatting.
     * @throws ServiceQuotaExceededException
     *         Returned when a request cannot be completed as it would exceed a maximum service quota.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TextractException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TextractClient.CreateAdapter
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/CreateAdapter" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateAdapterResponse createAdapter(CreateAdapterRequest createAdapterRequest) throws InvalidParameterException,
            AccessDeniedException, ConflictException, ProvisionedThroughputExceededException, InternalServerErrorException,
            IdempotentParameterMismatchException, ThrottlingException, LimitExceededException, ValidationException,
            ServiceQuotaExceededException, AwsServiceException, SdkClientException, TextractException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateAdapterResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateAdapterResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BadDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadDocumentException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "HumanLoopQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("HumanLoopQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(HumanLoopQuotaExceededException::builder).build());
            case "InvalidKMSKeyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidKMSKeyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidKmsKeyException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "DocumentTooLargeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DocumentTooLargeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DocumentTooLargeException::builder).build());
            case "UnsupportedDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedDocumentException::builder).build());
            case "InvalidS3ObjectException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidS3ObjectException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidS3ObjectException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "IdempotentParameterMismatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("IdempotentParameterMismatchException")
                        .httpStatusCode(400).exceptionBuilderSupplier(IdempotentParameterMismatchException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ProvisionedThroughputExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ProvisionedThroughputExceededException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ProvisionedThroughputExceededException::builder).build());
            case "InvalidJobIdException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidJobIdException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidJobIdException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createAdapterRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createAdapterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Textract");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateAdapter");

            return clientHandler.execute(new ClientExecutionParams<CreateAdapterRequest, CreateAdapterResponse>()
                    .withOperationName("CreateAdapter").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createAdapterRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateAdapterRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new version of an adapter. Operates on a provided AdapterId and a specified dataset provided via the
     * DatasetConfig argument. Requires that you specify an Amazon S3 bucket with the OutputConfig argument. You can
     * provide an optional KMSKeyId, an optional ClientRequestToken, and optional tags.
     * </p>
     *
     * @param createAdapterVersionRequest
     * @return Result of the CreateAdapterVersion operation returned by the service.
     * @throws InvalidParameterException
     *         An input parameter violated a constraint. For example, in synchronous operations, an
     *         <code>InvalidParameterException</code> exception occurs when neither of the <code>S3Object</code> or
     *         <code>Bytes</code> values are supplied in the <code>Document</code> request parameter. Validate your
     *         parameter before calling the API operation again.
     * @throws InvalidS3ObjectException
     *         Amazon Textract is unable to access the S3 object that's specified in the request. for more information,
     *         <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-access-control.html">Configure Access to
     *         Amazon S3</a> For troubleshooting information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/troubleshooting.html">Troubleshooting Amazon S3</a>
     * @throws InvalidKmsKeyException
     *         Indicates you do not have decrypt permissions with the KMS key entered, or the KMS key was entered
     *         incorrectly.
     * @throws AccessDeniedException
     *         You aren't authorized to perform the action. Use the Amazon Resource Name (ARN) of an authorized user or
     *         IAM role to perform the operation.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Textract.
     * @throws InternalServerErrorException
     *         Amazon Textract experienced a service issue. Try your call again.
     * @throws IdempotentParameterMismatchException
     *         A <code>ClientRequestToken</code> input parameter was reused with an operation, but at least one of the
     *         other input parameters is different from the previous call to the operation.
     * @throws ThrottlingException
     *         Amazon Textract is temporarily unable to process the request. Try your call again.
     * @throws LimitExceededException
     *         An Amazon Textract service limit was exceeded. For example, if you start too many asynchronous jobs
     *         concurrently, calls to start operations (<code>StartDocumentTextDetection</code>, for example) raise a
     *         LimitExceededException exception (HTTP status code: 400) until the number of concurrently running jobs is
     *         below the Amazon Textract service limit.
     * @throws ValidationException
     *         Indicates that a request was not valid. Check request for proper formatting.
     * @throws ServiceQuotaExceededException
     *         Returned when a request cannot be completed as it would exceed a maximum service quota.
     * @throws ResourceNotFoundException
     *         Returned when an operation tried to access a nonexistent resource.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TextractException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TextractClient.CreateAdapterVersion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/CreateAdapterVersion" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateAdapterVersionResponse createAdapterVersion(CreateAdapterVersionRequest createAdapterVersionRequest)
            throws InvalidParameterException, InvalidS3ObjectException, InvalidKmsKeyException, AccessDeniedException,
            ProvisionedThroughputExceededException, InternalServerErrorException, IdempotentParameterMismatchException,
            ThrottlingException, LimitExceededException, ValidationException, ServiceQuotaExceededException,
            ResourceNotFoundException, ConflictException, AwsServiceException, SdkClientException, TextractException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateAdapterVersionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateAdapterVersionResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BadDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadDocumentException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "HumanLoopQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("HumanLoopQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(HumanLoopQuotaExceededException::builder).build());
            case "InvalidKMSKeyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidKMSKeyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidKmsKeyException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "DocumentTooLargeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DocumentTooLargeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DocumentTooLargeException::builder).build());
            case "UnsupportedDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedDocumentException::builder).build());
            case "InvalidS3ObjectException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidS3ObjectException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidS3ObjectException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "IdempotentParameterMismatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("IdempotentParameterMismatchException")
                        .httpStatusCode(400).exceptionBuilderSupplier(IdempotentParameterMismatchException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ProvisionedThroughputExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ProvisionedThroughputExceededException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ProvisionedThroughputExceededException::builder).build());
            case "InvalidJobIdException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidJobIdException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidJobIdException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createAdapterVersionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createAdapterVersionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Textract");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateAdapterVersion");

            return clientHandler.execute(new ClientExecutionParams<CreateAdapterVersionRequest, CreateAdapterVersionResponse>()
                    .withOperationName("CreateAdapterVersion").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createAdapterVersionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateAdapterVersionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an Amazon Textract adapter. Takes an AdapterId and deletes the adapter specified by the ID.
     * </p>
     *
     * @param deleteAdapterRequest
     * @return Result of the DeleteAdapter operation returned by the service.
     * @throws InvalidParameterException
     *         An input parameter violated a constraint. For example, in synchronous operations, an
     *         <code>InvalidParameterException</code> exception occurs when neither of the <code>S3Object</code> or
     *         <code>Bytes</code> values are supplied in the <code>Document</code> request parameter. Validate your
     *         parameter before calling the API operation again.
     * @throws AccessDeniedException
     *         You aren't authorized to perform the action. Use the Amazon Resource Name (ARN) of an authorized user or
     *         IAM role to perform the operation.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Textract.
     * @throws InternalServerErrorException
     *         Amazon Textract experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Textract is temporarily unable to process the request. Try your call again.
     * @throws ValidationException
     *         Indicates that a request was not valid. Check request for proper formatting.
     * @throws ResourceNotFoundException
     *         Returned when an operation tried to access a nonexistent resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TextractException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TextractClient.DeleteAdapter
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/DeleteAdapter" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteAdapterResponse deleteAdapter(DeleteAdapterRequest deleteAdapterRequest) throws InvalidParameterException,
            AccessDeniedException, ConflictException, ProvisionedThroughputExceededException, InternalServerErrorException,
            ThrottlingException, ValidationException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            TextractException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteAdapterResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteAdapterResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BadDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadDocumentException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "HumanLoopQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("HumanLoopQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(HumanLoopQuotaExceededException::builder).build());
            case "InvalidKMSKeyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidKMSKeyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidKmsKeyException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "DocumentTooLargeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DocumentTooLargeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DocumentTooLargeException::builder).build());
            case "UnsupportedDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedDocumentException::builder).build());
            case "InvalidS3ObjectException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidS3ObjectException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidS3ObjectException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "IdempotentParameterMismatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("IdempotentParameterMismatchException")
                        .httpStatusCode(400).exceptionBuilderSupplier(IdempotentParameterMismatchException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ProvisionedThroughputExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ProvisionedThroughputExceededException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ProvisionedThroughputExceededException::builder).build());
            case "InvalidJobIdException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidJobIdException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidJobIdException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteAdapterRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAdapterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Textract");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAdapter");

            return clientHandler.execute(new ClientExecutionParams<DeleteAdapterRequest, DeleteAdapterResponse>()
                    .withOperationName("DeleteAdapter").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteAdapterRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteAdapterRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an Amazon Textract adapter version. Requires that you specify both an AdapterId and a AdapterVersion.
     * Deletes the adapter version specified by the AdapterId and the AdapterVersion.
     * </p>
     *
     * @param deleteAdapterVersionRequest
     * @return Result of the DeleteAdapterVersion operation returned by the service.
     * @throws InvalidParameterException
     *         An input parameter violated a constraint. For example, in synchronous operations, an
     *         <code>InvalidParameterException</code> exception occurs when neither of the <code>S3Object</code> or
     *         <code>Bytes</code> values are supplied in the <code>Document</code> request parameter. Validate your
     *         parameter before calling the API operation again.
     * @throws AccessDeniedException
     *         You aren't authorized to perform the action. Use the Amazon Resource Name (ARN) of an authorized user or
     *         IAM role to perform the operation.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Textract.
     * @throws InternalServerErrorException
     *         Amazon Textract experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Textract is temporarily unable to process the request. Try your call again.
     * @throws ValidationException
     *         Indicates that a request was not valid. Check request for proper formatting.
     * @throws ResourceNotFoundException
     *         Returned when an operation tried to access a nonexistent resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TextractException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TextractClient.DeleteAdapterVersion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/DeleteAdapterVersion" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteAdapterVersionResponse deleteAdapterVersion(DeleteAdapterVersionRequest deleteAdapterVersionRequest)
            throws InvalidParameterException, AccessDeniedException, ConflictException, ProvisionedThroughputExceededException,
            InternalServerErrorException, ThrottlingException, ValidationException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, TextractException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteAdapterVersionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteAdapterVersionResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BadDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadDocumentException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "HumanLoopQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("HumanLoopQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(HumanLoopQuotaExceededException::builder).build());
            case "InvalidKMSKeyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidKMSKeyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidKmsKeyException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "DocumentTooLargeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DocumentTooLargeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DocumentTooLargeException::builder).build());
            case "UnsupportedDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedDocumentException::builder).build());
            case "InvalidS3ObjectException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidS3ObjectException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidS3ObjectException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "IdempotentParameterMismatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("IdempotentParameterMismatchException")
                        .httpStatusCode(400).exceptionBuilderSupplier(IdempotentParameterMismatchException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ProvisionedThroughputExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ProvisionedThroughputExceededException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ProvisionedThroughputExceededException::builder).build());
            case "InvalidJobIdException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidJobIdException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidJobIdException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteAdapterVersionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAdapterVersionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Textract");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAdapterVersion");

            return clientHandler.execute(new ClientExecutionParams<DeleteAdapterVersionRequest, DeleteAdapterVersionResponse>()
                    .withOperationName("DeleteAdapterVersion").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteAdapterVersionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteAdapterVersionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Detects text in the input document. Amazon Textract can detect lines of text and the words that make up a line of
     * text. The input document must be in one of the following image formats: JPEG, PNG, PDF, or TIFF.
     * <code>DetectDocumentText</code> returns the detected text in an array of <a>Block</a> objects.
     * </p>
     * <p>
     * Each document page has as an associated <code>Block</code> of type PAGE. Each PAGE <code>Block</code> object is
     * the parent of LINE <code>Block</code> objects that represent the lines of detected text on a page. A LINE
     * <code>Block</code> object is a parent for each word that makes up the line. Words are represented by
     * <code>Block</code> objects of type WORD.
     * </p>
     * <p>
     * <code>DetectDocumentText</code> is a synchronous operation. To analyze documents asynchronously, use
     * <a>StartDocumentTextDetection</a>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/textract/latest/dg/how-it-works-detecting.html">Document Text Detection</a>.
     * </p>
     *
     * @param detectDocumentTextRequest
     * @return Result of the DetectDocumentText operation returned by the service.
     * @throws InvalidParameterException
     *         An input parameter violated a constraint. For example, in synchronous operations, an
     *         <code>InvalidParameterException</code> exception occurs when neither of the <code>S3Object</code> or
     *         <code>Bytes</code> values are supplied in the <code>Document</code> request parameter. Validate your
     *         parameter before calling the API operation again.
     * @throws InvalidS3ObjectException
     *         Amazon Textract is unable to access the S3 object that's specified in the request. for more information,
     *         <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-access-control.html">Configure Access to
     *         Amazon S3</a> For troubleshooting information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/troubleshooting.html">Troubleshooting Amazon S3</a>
     * @throws UnsupportedDocumentException
     *         The format of the input document isn't supported. Documents for operations can be in PNG, JPEG, PDF, or
     *         TIFF format.
     * @throws DocumentTooLargeException
     *         The document can't be processed because it's too large. The maximum document size for synchronous
     *         operations 10 MB. The maximum document size for asynchronous operations is 500 MB for PDF files.
     * @throws BadDocumentException
     *         Amazon Textract isn't able to read the document. For more information on the document limits in Amazon
     *         Textract, see <a>limits</a>.
     * @throws AccessDeniedException
     *         You aren't authorized to perform the action. Use the Amazon Resource Name (ARN) of an authorized user or
     *         IAM role to perform the operation.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Textract.
     * @throws InternalServerErrorException
     *         Amazon Textract experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Textract is temporarily unable to process the request. Try your call again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TextractException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TextractClient.DetectDocumentText
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/DetectDocumentText" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DetectDocumentTextResponse detectDocumentText(DetectDocumentTextRequest detectDocumentTextRequest)
            throws InvalidParameterException, InvalidS3ObjectException, UnsupportedDocumentException, DocumentTooLargeException,
            BadDocumentException, AccessDeniedException, ProvisionedThroughputExceededException, InternalServerErrorException,
            ThrottlingException, AwsServiceException, SdkClientException, TextractException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DetectDocumentTextResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DetectDocumentTextResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BadDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadDocumentException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "HumanLoopQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("HumanLoopQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(HumanLoopQuotaExceededException::builder).build());
            case "InvalidKMSKeyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidKMSKeyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidKmsKeyException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "DocumentTooLargeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DocumentTooLargeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DocumentTooLargeException::builder).build());
            case "UnsupportedDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedDocumentException::builder).build());
            case "InvalidS3ObjectException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidS3ObjectException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidS3ObjectException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "IdempotentParameterMismatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("IdempotentParameterMismatchException")
                        .httpStatusCode(400).exceptionBuilderSupplier(IdempotentParameterMismatchException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ProvisionedThroughputExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ProvisionedThroughputExceededException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ProvisionedThroughputExceededException::builder).build());
            case "InvalidJobIdException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidJobIdException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidJobIdException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(detectDocumentTextRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, detectDocumentTextRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Textract");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DetectDocumentText");

            return clientHandler.execute(new ClientExecutionParams<DetectDocumentTextRequest, DetectDocumentTextResponse>()
                    .withOperationName("DetectDocumentText").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(detectDocumentTextRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DetectDocumentTextRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets configuration information for an adapter specified by an AdapterId, returning information on AdapterName,
     * Description, CreationTime, AutoUpdate status, and FeatureTypes.
     * </p>
     *
     * @param getAdapterRequest
     * @return Result of the GetAdapter operation returned by the service.
     * @throws InvalidParameterException
     *         An input parameter violated a constraint. For example, in synchronous operations, an
     *         <code>InvalidParameterException</code> exception occurs when neither of the <code>S3Object</code> or
     *         <code>Bytes</code> values are supplied in the <code>Document</code> request parameter. Validate your
     *         parameter before calling the API operation again.
     * @throws AccessDeniedException
     *         You aren't authorized to perform the action. Use the Amazon Resource Name (ARN) of an authorized user or
     *         IAM role to perform the operation.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Textract.
     * @throws InternalServerErrorException
     *         Amazon Textract experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Textract is temporarily unable to process the request. Try your call again.
     * @throws ValidationException
     *         Indicates that a request was not valid. Check request for proper formatting.
     * @throws ResourceNotFoundException
     *         Returned when an operation tried to access a nonexistent resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TextractException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TextractClient.GetAdapter
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/GetAdapter" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetAdapterResponse getAdapter(GetAdapterRequest getAdapterRequest) throws InvalidParameterException,
            AccessDeniedException, ProvisionedThroughputExceededException, InternalServerErrorException, ThrottlingException,
            ValidationException, ResourceNotFoundException, AwsServiceException, SdkClientException, TextractException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetAdapterResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetAdapterResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BadDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadDocumentException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "HumanLoopQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("HumanLoopQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(HumanLoopQuotaExceededException::builder).build());
            case "InvalidKMSKeyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidKMSKeyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidKmsKeyException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "DocumentTooLargeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DocumentTooLargeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DocumentTooLargeException::builder).build());
            case "UnsupportedDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedDocumentException::builder).build());
            case "InvalidS3ObjectException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidS3ObjectException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidS3ObjectException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "IdempotentParameterMismatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("IdempotentParameterMismatchException")
                        .httpStatusCode(400).exceptionBuilderSupplier(IdempotentParameterMismatchException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ProvisionedThroughputExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ProvisionedThroughputExceededException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ProvisionedThroughputExceededException::builder).build());
            case "InvalidJobIdException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidJobIdException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidJobIdException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getAdapterRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAdapterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Textract");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAdapter");

            return clientHandler.execute(new ClientExecutionParams<GetAdapterRequest, GetAdapterResponse>()
                    .withOperationName("GetAdapter").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getAdapterRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetAdapterRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets configuration information for the specified adapter version, including: AdapterId, AdapterVersion,
     * FeatureTypes, Status, StatusMessage, DatasetConfig, KMSKeyId, OutputConfig, Tags and EvaluationMetrics.
     * </p>
     *
     * @param getAdapterVersionRequest
     * @return Result of the GetAdapterVersion operation returned by the service.
     * @throws InvalidParameterException
     *         An input parameter violated a constraint. For example, in synchronous operations, an
     *         <code>InvalidParameterException</code> exception occurs when neither of the <code>S3Object</code> or
     *         <code>Bytes</code> values are supplied in the <code>Document</code> request parameter. Validate your
     *         parameter before calling the API operation again.
     * @throws AccessDeniedException
     *         You aren't authorized to perform the action. Use the Amazon Resource Name (ARN) of an authorized user or
     *         IAM role to perform the operation.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Textract.
     * @throws InternalServerErrorException
     *         Amazon Textract experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Textract is temporarily unable to process the request. Try your call again.
     * @throws ValidationException
     *         Indicates that a request was not valid. Check request for proper formatting.
     * @throws ResourceNotFoundException
     *         Returned when an operation tried to access a nonexistent resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TextractException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TextractClient.GetAdapterVersion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/GetAdapterVersion" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetAdapterVersionResponse getAdapterVersion(GetAdapterVersionRequest getAdapterVersionRequest)
            throws InvalidParameterException, AccessDeniedException, ProvisionedThroughputExceededException,
            InternalServerErrorException, ThrottlingException, ValidationException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, TextractException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetAdapterVersionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetAdapterVersionResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BadDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadDocumentException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "HumanLoopQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("HumanLoopQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(HumanLoopQuotaExceededException::builder).build());
            case "InvalidKMSKeyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidKMSKeyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidKmsKeyException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "DocumentTooLargeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DocumentTooLargeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DocumentTooLargeException::builder).build());
            case "UnsupportedDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedDocumentException::builder).build());
            case "InvalidS3ObjectException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidS3ObjectException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidS3ObjectException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "IdempotentParameterMismatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("IdempotentParameterMismatchException")
                        .httpStatusCode(400).exceptionBuilderSupplier(IdempotentParameterMismatchException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ProvisionedThroughputExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ProvisionedThroughputExceededException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ProvisionedThroughputExceededException::builder).build());
            case "InvalidJobIdException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidJobIdException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidJobIdException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getAdapterVersionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAdapterVersionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Textract");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAdapterVersion");

            return clientHandler.execute(new ClientExecutionParams<GetAdapterVersionRequest, GetAdapterVersionResponse>()
                    .withOperationName("GetAdapterVersion").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getAdapterVersionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetAdapterVersionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the results for an Amazon Textract asynchronous operation that analyzes text in a document.
     * </p>
     * <p>
     * You start asynchronous text analysis by calling <a>StartDocumentAnalysis</a>, which returns a job identifier (
     * <code>JobId</code>). When the text analysis operation finishes, Amazon Textract publishes a completion status to
     * the Amazon Simple Notification Service (Amazon SNS) topic that's registered in the initial call to
     * <code>StartDocumentAnalysis</code>. To get the results of the text-detection operation, first check that the
     * status value published to the Amazon SNS topic is <code>SUCCEEDED</code>. If so, call
     * <code>GetDocumentAnalysis</code>, and pass the job identifier (<code>JobId</code>) from the initial call to
     * <code>StartDocumentAnalysis</code>.
     * </p>
     * <p>
     * <code>GetDocumentAnalysis</code> returns an array of <a>Block</a> objects. The following types of information are
     * returned:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Form data (key-value pairs). The related information is returned in two <a>Block</a> objects, each of type
     * <code>KEY_VALUE_SET</code>: a KEY <code>Block</code> object and a VALUE <code>Block</code> object. For example,
     * <i>Name: Ana Silva Carolina</i> contains a key and value. <i>Name:</i> is the key. <i>Ana Silva Carolina</i> is
     * the value.
     * </p>
     * </li>
     * <li>
     * <p>
     * Table and table cell data. A TABLE <code>Block</code> object contains information about a detected table. A CELL
     * <code>Block</code> object is returned for each cell in a table.
     * </p>
     * </li>
     * <li>
     * <p>
     * Lines and words of text. A LINE <code>Block</code> object contains one or more WORD <code>Block</code> objects.
     * All lines and words that are detected in the document are returned (including text that doesn't have a
     * relationship with the value of the <code>StartDocumentAnalysis</code> <code>FeatureTypes</code> input parameter).
     * </p>
     * </li>
     * <li>
     * <p>
     * Query. A QUERY Block object contains the query text, alias and link to the associated Query results block object.
     * </p>
     * </li>
     * <li>
     * <p>
     * Query Results. A QUERY_RESULT Block object contains the answer to the query and an ID that connects it to the
     * query asked. This Block also contains a confidence score.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * While processing a document with queries, look out for <code>INVALID_REQUEST_PARAMETERS</code> output. This
     * indicates that either the per page query limit has been exceeded or that the operation is trying to query a page
     * in the document which doesn’t exist.
     * </p>
     * </note>
     * <p>
     * Selection elements such as check boxes and option buttons (radio buttons) can be detected in form data and in
     * tables. A SELECTION_ELEMENT <code>Block</code> object contains information about a selection element, including
     * the selection status.
     * </p>
     * <p>
     * Use the <code>MaxResults</code> parameter to limit the number of blocks that are returned. If there are more
     * results than specified in <code>MaxResults</code>, the value of <code>NextToken</code> in the operation response
     * contains a pagination token for getting the next set of results. To get the next page of results, call
     * <code>GetDocumentAnalysis</code>, and populate the <code>NextToken</code> request parameter with the token value
     * that's returned from the previous call to <code>GetDocumentAnalysis</code>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/textract/latest/dg/how-it-works-analyzing.html">Document Text Analysis</a>.
     * </p>
     *
     * @param getDocumentAnalysisRequest
     * @return Result of the GetDocumentAnalysis operation returned by the service.
     * @throws InvalidParameterException
     *         An input parameter violated a constraint. For example, in synchronous operations, an
     *         <code>InvalidParameterException</code> exception occurs when neither of the <code>S3Object</code> or
     *         <code>Bytes</code> values are supplied in the <code>Document</code> request parameter. Validate your
     *         parameter before calling the API operation again.
     * @throws AccessDeniedException
     *         You aren't authorized to perform the action. Use the Amazon Resource Name (ARN) of an authorized user or
     *         IAM role to perform the operation.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Textract.
     * @throws InvalidJobIdException
     *         An invalid job identifier was passed to an asynchronous analysis operation.
     * @throws InternalServerErrorException
     *         Amazon Textract experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Textract is temporarily unable to process the request. Try your call again.
     * @throws InvalidS3ObjectException
     *         Amazon Textract is unable to access the S3 object that's specified in the request. for more information,
     *         <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-access-control.html">Configure Access to
     *         Amazon S3</a> For troubleshooting information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/troubleshooting.html">Troubleshooting Amazon S3</a>
     * @throws InvalidKmsKeyException
     *         Indicates you do not have decrypt permissions with the KMS key entered, or the KMS key was entered
     *         incorrectly.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TextractException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TextractClient.GetDocumentAnalysis
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/GetDocumentAnalysis" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetDocumentAnalysisResponse getDocumentAnalysis(GetDocumentAnalysisRequest getDocumentAnalysisRequest)
            throws InvalidParameterException, AccessDeniedException, ProvisionedThroughputExceededException,
            InvalidJobIdException, InternalServerErrorException, ThrottlingException, InvalidS3ObjectException,
            InvalidKmsKeyException, AwsServiceException, SdkClientException, TextractException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetDocumentAnalysisResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetDocumentAnalysisResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BadDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadDocumentException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "HumanLoopQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("HumanLoopQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(HumanLoopQuotaExceededException::builder).build());
            case "InvalidKMSKeyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidKMSKeyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidKmsKeyException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "DocumentTooLargeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DocumentTooLargeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DocumentTooLargeException::builder).build());
            case "UnsupportedDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedDocumentException::builder).build());
            case "InvalidS3ObjectException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidS3ObjectException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidS3ObjectException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "IdempotentParameterMismatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("IdempotentParameterMismatchException")
                        .httpStatusCode(400).exceptionBuilderSupplier(IdempotentParameterMismatchException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ProvisionedThroughputExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ProvisionedThroughputExceededException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ProvisionedThroughputExceededException::builder).build());
            case "InvalidJobIdException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidJobIdException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidJobIdException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getDocumentAnalysisRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDocumentAnalysisRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Textract");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDocumentAnalysis");

            return clientHandler.execute(new ClientExecutionParams<GetDocumentAnalysisRequest, GetDocumentAnalysisResponse>()
                    .withOperationName("GetDocumentAnalysis").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getDocumentAnalysisRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetDocumentAnalysisRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the results for an Amazon Textract asynchronous operation that detects text in a document. Amazon Textract
     * can detect lines of text and the words that make up a line of text.
     * </p>
     * <p>
     * You start asynchronous text detection by calling <a>StartDocumentTextDetection</a>, which returns a job
     * identifier (<code>JobId</code>). When the text detection operation finishes, Amazon Textract publishes a
     * completion status to the Amazon Simple Notification Service (Amazon SNS) topic that's registered in the initial
     * call to <code>StartDocumentTextDetection</code>. To get the results of the text-detection operation, first check
     * that the status value published to the Amazon SNS topic is <code>SUCCEEDED</code>. If so, call
     * <code>GetDocumentTextDetection</code>, and pass the job identifier (<code>JobId</code>) from the initial call to
     * <code>StartDocumentTextDetection</code>.
     * </p>
     * <p>
     * <code>GetDocumentTextDetection</code> returns an array of <a>Block</a> objects.
     * </p>
     * <p>
     * Each document page has as an associated <code>Block</code> of type PAGE. Each PAGE <code>Block</code> object is
     * the parent of LINE <code>Block</code> objects that represent the lines of detected text on a page. A LINE
     * <code>Block</code> object is a parent for each word that makes up the line. Words are represented by
     * <code>Block</code> objects of type WORD.
     * </p>
     * <p>
     * Use the MaxResults parameter to limit the number of blocks that are returned. If there are more results than
     * specified in <code>MaxResults</code>, the value of <code>NextToken</code> in the operation response contains a
     * pagination token for getting the next set of results. To get the next page of results, call
     * <code>GetDocumentTextDetection</code>, and populate the <code>NextToken</code> request parameter with the token
     * value that's returned from the previous call to <code>GetDocumentTextDetection</code>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/textract/latest/dg/how-it-works-detecting.html">Document Text Detection</a>.
     * </p>
     *
     * @param getDocumentTextDetectionRequest
     * @return Result of the GetDocumentTextDetection operation returned by the service.
     * @throws InvalidParameterException
     *         An input parameter violated a constraint. For example, in synchronous operations, an
     *         <code>InvalidParameterException</code> exception occurs when neither of the <code>S3Object</code> or
     *         <code>Bytes</code> values are supplied in the <code>Document</code> request parameter. Validate your
     *         parameter before calling the API operation again.
     * @throws AccessDeniedException
     *         You aren't authorized to perform the action. Use the Amazon Resource Name (ARN) of an authorized user or
     *         IAM role to perform the operation.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Textract.
     * @throws InvalidJobIdException
     *         An invalid job identifier was passed to an asynchronous analysis operation.
     * @throws InternalServerErrorException
     *         Amazon Textract experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Textract is temporarily unable to process the request. Try your call again.
     * @throws InvalidS3ObjectException
     *         Amazon Textract is unable to access the S3 object that's specified in the request. for more information,
     *         <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-access-control.html">Configure Access to
     *         Amazon S3</a> For troubleshooting information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/troubleshooting.html">Troubleshooting Amazon S3</a>
     * @throws InvalidKmsKeyException
     *         Indicates you do not have decrypt permissions with the KMS key entered, or the KMS key was entered
     *         incorrectly.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TextractException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TextractClient.GetDocumentTextDetection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/GetDocumentTextDetection"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetDocumentTextDetectionResponse getDocumentTextDetection(
            GetDocumentTextDetectionRequest getDocumentTextDetectionRequest) throws InvalidParameterException,
            AccessDeniedException, ProvisionedThroughputExceededException, InvalidJobIdException, InternalServerErrorException,
            ThrottlingException, InvalidS3ObjectException, InvalidKmsKeyException, AwsServiceException, SdkClientException,
            TextractException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetDocumentTextDetectionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetDocumentTextDetectionResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BadDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadDocumentException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "HumanLoopQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("HumanLoopQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(HumanLoopQuotaExceededException::builder).build());
            case "InvalidKMSKeyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidKMSKeyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidKmsKeyException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "DocumentTooLargeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DocumentTooLargeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DocumentTooLargeException::builder).build());
            case "UnsupportedDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedDocumentException::builder).build());
            case "InvalidS3ObjectException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidS3ObjectException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidS3ObjectException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "IdempotentParameterMismatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("IdempotentParameterMismatchException")
                        .httpStatusCode(400).exceptionBuilderSupplier(IdempotentParameterMismatchException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ProvisionedThroughputExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ProvisionedThroughputExceededException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ProvisionedThroughputExceededException::builder).build());
            case "InvalidJobIdException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidJobIdException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidJobIdException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getDocumentTextDetectionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDocumentTextDetectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Textract");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDocumentTextDetection");

            return clientHandler
                    .execute(new ClientExecutionParams<GetDocumentTextDetectionRequest, GetDocumentTextDetectionResponse>()
                            .withOperationName("GetDocumentTextDetection").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(getDocumentTextDetectionRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetDocumentTextDetectionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the results for an Amazon Textract asynchronous operation that analyzes invoices and receipts. Amazon
     * Textract finds contact information, items purchased, and vendor name, from input invoices and receipts.
     * </p>
     * <p>
     * You start asynchronous invoice/receipt analysis by calling <a>StartExpenseAnalysis</a>, which returns a job
     * identifier (<code>JobId</code>). Upon completion of the invoice/receipt analysis, Amazon Textract publishes the
     * completion status to the Amazon Simple Notification Service (Amazon SNS) topic. This topic must be registered in
     * the initial call to <code>StartExpenseAnalysis</code>. To get the results of the invoice/receipt analysis
     * operation, first ensure that the status value published to the Amazon SNS topic is <code>SUCCEEDED</code>. If so,
     * call <code>GetExpenseAnalysis</code>, and pass the job identifier (<code>JobId</code>) from the initial call to
     * <code>StartExpenseAnalysis</code>.
     * </p>
     * <p>
     * Use the MaxResults parameter to limit the number of blocks that are returned. If there are more results than
     * specified in <code>MaxResults</code>, the value of <code>NextToken</code> in the operation response contains a
     * pagination token for getting the next set of results. To get the next page of results, call
     * <code>GetExpenseAnalysis</code>, and populate the <code>NextToken</code> request parameter with the token value
     * that's returned from the previous call to <code>GetExpenseAnalysis</code>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/textract/latest/dg/invoices-receipts.html">Analyzing Invoices and Receipts</a>.
     * </p>
     *
     * @param getExpenseAnalysisRequest
     * @return Result of the GetExpenseAnalysis operation returned by the service.
     * @throws InvalidParameterException
     *         An input parameter violated a constraint. For example, in synchronous operations, an
     *         <code>InvalidParameterException</code> exception occurs when neither of the <code>S3Object</code> or
     *         <code>Bytes</code> values are supplied in the <code>Document</code> request parameter. Validate your
     *         parameter before calling the API operation again.
     * @throws AccessDeniedException
     *         You aren't authorized to perform the action. Use the Amazon Resource Name (ARN) of an authorized user or
     *         IAM role to perform the operation.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Textract.
     * @throws InvalidJobIdException
     *         An invalid job identifier was passed to an asynchronous analysis operation.
     * @throws InternalServerErrorException
     *         Amazon Textract experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Textract is temporarily unable to process the request. Try your call again.
     * @throws InvalidS3ObjectException
     *         Amazon Textract is unable to access the S3 object that's specified in the request. for more information,
     *         <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-access-control.html">Configure Access to
     *         Amazon S3</a> For troubleshooting information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/troubleshooting.html">Troubleshooting Amazon S3</a>
     * @throws InvalidKmsKeyException
     *         Indicates you do not have decrypt permissions with the KMS key entered, or the KMS key was entered
     *         incorrectly.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TextractException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TextractClient.GetExpenseAnalysis
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/GetExpenseAnalysis" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetExpenseAnalysisResponse getExpenseAnalysis(GetExpenseAnalysisRequest getExpenseAnalysisRequest)
            throws InvalidParameterException, AccessDeniedException, ProvisionedThroughputExceededException,
            InvalidJobIdException, InternalServerErrorException, ThrottlingException, InvalidS3ObjectException,
            InvalidKmsKeyException, AwsServiceException, SdkClientException, TextractException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetExpenseAnalysisResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetExpenseAnalysisResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BadDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadDocumentException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "HumanLoopQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("HumanLoopQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(HumanLoopQuotaExceededException::builder).build());
            case "InvalidKMSKeyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidKMSKeyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidKmsKeyException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "DocumentTooLargeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DocumentTooLargeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DocumentTooLargeException::builder).build());
            case "UnsupportedDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedDocumentException::builder).build());
            case "InvalidS3ObjectException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidS3ObjectException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidS3ObjectException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "IdempotentParameterMismatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("IdempotentParameterMismatchException")
                        .httpStatusCode(400).exceptionBuilderSupplier(IdempotentParameterMismatchException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ProvisionedThroughputExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ProvisionedThroughputExceededException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ProvisionedThroughputExceededException::builder).build());
            case "InvalidJobIdException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidJobIdException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidJobIdException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getExpenseAnalysisRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getExpenseAnalysisRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Textract");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetExpenseAnalysis");

            return clientHandler.execute(new ClientExecutionParams<GetExpenseAnalysisRequest, GetExpenseAnalysisResponse>()
                    .withOperationName("GetExpenseAnalysis").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getExpenseAnalysisRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetExpenseAnalysisRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the results for an Amazon Textract asynchronous operation that analyzes text in a lending document.
     * </p>
     * <p>
     * You start asynchronous text analysis by calling <code>StartLendingAnalysis</code>, which returns a job identifier
     * (<code>JobId</code>). When the text analysis operation finishes, Amazon Textract publishes a completion status to
     * the Amazon Simple Notification Service (Amazon SNS) topic that's registered in the initial call to
     * <code>StartLendingAnalysis</code>.
     * </p>
     * <p>
     * To get the results of the text analysis operation, first check that the status value published to the Amazon SNS
     * topic is SUCCEEDED. If so, call GetLendingAnalysis, and pass the job identifier (<code>JobId</code>) from the
     * initial call to <code>StartLendingAnalysis</code>.
     * </p>
     *
     * @param getLendingAnalysisRequest
     * @return Result of the GetLendingAnalysis operation returned by the service.
     * @throws InvalidParameterException
     *         An input parameter violated a constraint. For example, in synchronous operations, an
     *         <code>InvalidParameterException</code> exception occurs when neither of the <code>S3Object</code> or
     *         <code>Bytes</code> values are supplied in the <code>Document</code> request parameter. Validate your
     *         parameter before calling the API operation again.
     * @throws AccessDeniedException
     *         You aren't authorized to perform the action. Use the Amazon Resource Name (ARN) of an authorized user or
     *         IAM role to perform the operation.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Textract.
     * @throws InvalidJobIdException
     *         An invalid job identifier was passed to an asynchronous analysis operation.
     * @throws InternalServerErrorException
     *         Amazon Textract experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Textract is temporarily unable to process the request. Try your call again.
     * @throws InvalidS3ObjectException
     *         Amazon Textract is unable to access the S3 object that's specified in the request. for more information,
     *         <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-access-control.html">Configure Access to
     *         Amazon S3</a> For troubleshooting information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/troubleshooting.html">Troubleshooting Amazon S3</a>
     * @throws InvalidKmsKeyException
     *         Indicates you do not have decrypt permissions with the KMS key entered, or the KMS key was entered
     *         incorrectly.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TextractException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TextractClient.GetLendingAnalysis
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/GetLendingAnalysis" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetLendingAnalysisResponse getLendingAnalysis(GetLendingAnalysisRequest getLendingAnalysisRequest)
            throws InvalidParameterException, AccessDeniedException, ProvisionedThroughputExceededException,
            InvalidJobIdException, InternalServerErrorException, ThrottlingException, InvalidS3ObjectException,
            InvalidKmsKeyException, AwsServiceException, SdkClientException, TextractException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetLendingAnalysisResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetLendingAnalysisResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BadDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadDocumentException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "HumanLoopQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("HumanLoopQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(HumanLoopQuotaExceededException::builder).build());
            case "InvalidKMSKeyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidKMSKeyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidKmsKeyException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "DocumentTooLargeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DocumentTooLargeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DocumentTooLargeException::builder).build());
            case "UnsupportedDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedDocumentException::builder).build());
            case "InvalidS3ObjectException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidS3ObjectException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidS3ObjectException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "IdempotentParameterMismatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("IdempotentParameterMismatchException")
                        .httpStatusCode(400).exceptionBuilderSupplier(IdempotentParameterMismatchException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ProvisionedThroughputExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ProvisionedThroughputExceededException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ProvisionedThroughputExceededException::builder).build());
            case "InvalidJobIdException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidJobIdException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidJobIdException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getLendingAnalysisRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getLendingAnalysisRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Textract");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetLendingAnalysis");

            return clientHandler.execute(new ClientExecutionParams<GetLendingAnalysisRequest, GetLendingAnalysisResponse>()
                    .withOperationName("GetLendingAnalysis").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getLendingAnalysisRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetLendingAnalysisRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets summarized results for the <code>StartLendingAnalysis</code> operation, which analyzes text in a lending
     * document. The returned summary consists of information about documents grouped together by a common document
     * type. Information like detected signatures, page numbers, and split documents is returned with respect to the
     * type of grouped document.
     * </p>
     * <p>
     * You start asynchronous text analysis by calling <code>StartLendingAnalysis</code>, which returns a job identifier
     * (<code>JobId</code>). When the text analysis operation finishes, Amazon Textract publishes a completion status to
     * the Amazon Simple Notification Service (Amazon SNS) topic that's registered in the initial call to
     * <code>StartLendingAnalysis</code>.
     * </p>
     * <p>
     * To get the results of the text analysis operation, first check that the status value published to the Amazon SNS
     * topic is SUCCEEDED. If so, call <code>GetLendingAnalysisSummary</code>, and pass the job identifier (
     * <code>JobId</code>) from the initial call to <code>StartLendingAnalysis</code>.
     * </p>
     *
     * @param getLendingAnalysisSummaryRequest
     * @return Result of the GetLendingAnalysisSummary operation returned by the service.
     * @throws InvalidParameterException
     *         An input parameter violated a constraint. For example, in synchronous operations, an
     *         <code>InvalidParameterException</code> exception occurs when neither of the <code>S3Object</code> or
     *         <code>Bytes</code> values are supplied in the <code>Document</code> request parameter. Validate your
     *         parameter before calling the API operation again.
     * @throws AccessDeniedException
     *         You aren't authorized to perform the action. Use the Amazon Resource Name (ARN) of an authorized user or
     *         IAM role to perform the operation.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Textract.
     * @throws InvalidJobIdException
     *         An invalid job identifier was passed to an asynchronous analysis operation.
     * @throws InternalServerErrorException
     *         Amazon Textract experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Textract is temporarily unable to process the request. Try your call again.
     * @throws InvalidS3ObjectException
     *         Amazon Textract is unable to access the S3 object that's specified in the request. for more information,
     *         <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-access-control.html">Configure Access to
     *         Amazon S3</a> For troubleshooting information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/troubleshooting.html">Troubleshooting Amazon S3</a>
     * @throws InvalidKmsKeyException
     *         Indicates you do not have decrypt permissions with the KMS key entered, or the KMS key was entered
     *         incorrectly.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TextractException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TextractClient.GetLendingAnalysisSummary
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/GetLendingAnalysisSummary"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetLendingAnalysisSummaryResponse getLendingAnalysisSummary(
            GetLendingAnalysisSummaryRequest getLendingAnalysisSummaryRequest) throws InvalidParameterException,
            AccessDeniedException, ProvisionedThroughputExceededException, InvalidJobIdException, InternalServerErrorException,
            ThrottlingException, InvalidS3ObjectException, InvalidKmsKeyException, AwsServiceException, SdkClientException,
            TextractException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetLendingAnalysisSummaryResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetLendingAnalysisSummaryResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BadDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadDocumentException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "HumanLoopQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("HumanLoopQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(HumanLoopQuotaExceededException::builder).build());
            case "InvalidKMSKeyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidKMSKeyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidKmsKeyException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "DocumentTooLargeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DocumentTooLargeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DocumentTooLargeException::builder).build());
            case "UnsupportedDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedDocumentException::builder).build());
            case "InvalidS3ObjectException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidS3ObjectException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidS3ObjectException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "IdempotentParameterMismatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("IdempotentParameterMismatchException")
                        .httpStatusCode(400).exceptionBuilderSupplier(IdempotentParameterMismatchException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ProvisionedThroughputExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ProvisionedThroughputExceededException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ProvisionedThroughputExceededException::builder).build());
            case "InvalidJobIdException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidJobIdException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidJobIdException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getLendingAnalysisSummaryRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getLendingAnalysisSummaryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Textract");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetLendingAnalysisSummary");

            return clientHandler
                    .execute(new ClientExecutionParams<GetLendingAnalysisSummaryRequest, GetLendingAnalysisSummaryResponse>()
                            .withOperationName("GetLendingAnalysisSummary").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(getLendingAnalysisSummaryRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetLendingAnalysisSummaryRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * List all version of an adapter that meet the specified filtration criteria.
     * </p>
     *
     * @param listAdapterVersionsRequest
     * @return Result of the ListAdapterVersions operation returned by the service.
     * @throws InvalidParameterException
     *         An input parameter violated a constraint. For example, in synchronous operations, an
     *         <code>InvalidParameterException</code> exception occurs when neither of the <code>S3Object</code> or
     *         <code>Bytes</code> values are supplied in the <code>Document</code> request parameter. Validate your
     *         parameter before calling the API operation again.
     * @throws AccessDeniedException
     *         You aren't authorized to perform the action. Use the Amazon Resource Name (ARN) of an authorized user or
     *         IAM role to perform the operation.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Textract.
     * @throws InternalServerErrorException
     *         Amazon Textract experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Textract is temporarily unable to process the request. Try your call again.
     * @throws ValidationException
     *         Indicates that a request was not valid. Check request for proper formatting.
     * @throws ResourceNotFoundException
     *         Returned when an operation tried to access a nonexistent resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TextractException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TextractClient.ListAdapterVersions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/ListAdapterVersions" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListAdapterVersionsResponse listAdapterVersions(ListAdapterVersionsRequest listAdapterVersionsRequest)
            throws InvalidParameterException, AccessDeniedException, ProvisionedThroughputExceededException,
            InternalServerErrorException, ThrottlingException, ValidationException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, TextractException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListAdapterVersionsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListAdapterVersionsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BadDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadDocumentException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "HumanLoopQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("HumanLoopQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(HumanLoopQuotaExceededException::builder).build());
            case "InvalidKMSKeyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidKMSKeyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidKmsKeyException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "DocumentTooLargeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DocumentTooLargeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DocumentTooLargeException::builder).build());
            case "UnsupportedDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedDocumentException::builder).build());
            case "InvalidS3ObjectException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidS3ObjectException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidS3ObjectException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "IdempotentParameterMismatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("IdempotentParameterMismatchException")
                        .httpStatusCode(400).exceptionBuilderSupplier(IdempotentParameterMismatchException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ProvisionedThroughputExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ProvisionedThroughputExceededException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ProvisionedThroughputExceededException::builder).build());
            case "InvalidJobIdException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidJobIdException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidJobIdException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listAdapterVersionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAdapterVersionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Textract");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAdapterVersions");

            return clientHandler.execute(new ClientExecutionParams<ListAdapterVersionsRequest, ListAdapterVersionsResponse>()
                    .withOperationName("ListAdapterVersions").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listAdapterVersionsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListAdapterVersionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all adapters that match the specified filtration criteria.
     * </p>
     *
     * @param listAdaptersRequest
     * @return Result of the ListAdapters operation returned by the service.
     * @throws InvalidParameterException
     *         An input parameter violated a constraint. For example, in synchronous operations, an
     *         <code>InvalidParameterException</code> exception occurs when neither of the <code>S3Object</code> or
     *         <code>Bytes</code> values are supplied in the <code>Document</code> request parameter. Validate your
     *         parameter before calling the API operation again.
     * @throws AccessDeniedException
     *         You aren't authorized to perform the action. Use the Amazon Resource Name (ARN) of an authorized user or
     *         IAM role to perform the operation.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Textract.
     * @throws InternalServerErrorException
     *         Amazon Textract experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Textract is temporarily unable to process the request. Try your call again.
     * @throws ValidationException
     *         Indicates that a request was not valid. Check request for proper formatting.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TextractException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TextractClient.ListAdapters
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/ListAdapters" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListAdaptersResponse listAdapters(ListAdaptersRequest listAdaptersRequest) throws InvalidParameterException,
            AccessDeniedException, ProvisionedThroughputExceededException, InternalServerErrorException, ThrottlingException,
            ValidationException, AwsServiceException, SdkClientException, TextractException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListAdaptersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListAdaptersResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BadDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadDocumentException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "HumanLoopQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("HumanLoopQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(HumanLoopQuotaExceededException::builder).build());
            case "InvalidKMSKeyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidKMSKeyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidKmsKeyException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "DocumentTooLargeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DocumentTooLargeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DocumentTooLargeException::builder).build());
            case "UnsupportedDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedDocumentException::builder).build());
            case "InvalidS3ObjectException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidS3ObjectException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidS3ObjectException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "IdempotentParameterMismatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("IdempotentParameterMismatchException")
                        .httpStatusCode(400).exceptionBuilderSupplier(IdempotentParameterMismatchException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ProvisionedThroughputExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ProvisionedThroughputExceededException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ProvisionedThroughputExceededException::builder).build());
            case "InvalidJobIdException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidJobIdException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidJobIdException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listAdaptersRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAdaptersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Textract");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAdapters");

            return clientHandler.execute(new ClientExecutionParams<ListAdaptersRequest, ListAdaptersResponse>()
                    .withOperationName("ListAdapters").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listAdaptersRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListAdaptersRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all tags for an Amazon Textract resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         Returned when an operation tried to access a nonexistent resource.
     * @throws InvalidParameterException
     *         An input parameter violated a constraint. For example, in synchronous operations, an
     *         <code>InvalidParameterException</code> exception occurs when neither of the <code>S3Object</code> or
     *         <code>Bytes</code> values are supplied in the <code>Document</code> request parameter. Validate your
     *         parameter before calling the API operation again.
     * @throws AccessDeniedException
     *         You aren't authorized to perform the action. Use the Amazon Resource Name (ARN) of an authorized user or
     *         IAM role to perform the operation.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Textract.
     * @throws InternalServerErrorException
     *         Amazon Textract experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Textract is temporarily unable to process the request. Try your call again.
     * @throws ValidationException
     *         Indicates that a request was not valid. Check request for proper formatting.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TextractException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TextractClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ResourceNotFoundException, InvalidParameterException, AccessDeniedException,
            ProvisionedThroughputExceededException, InternalServerErrorException, ThrottlingException, ValidationException,
            AwsServiceException, SdkClientException, TextractException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BadDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadDocumentException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "HumanLoopQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("HumanLoopQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(HumanLoopQuotaExceededException::builder).build());
            case "InvalidKMSKeyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidKMSKeyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidKmsKeyException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "DocumentTooLargeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DocumentTooLargeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DocumentTooLargeException::builder).build());
            case "UnsupportedDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedDocumentException::builder).build());
            case "InvalidS3ObjectException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidS3ObjectException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidS3ObjectException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "IdempotentParameterMismatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("IdempotentParameterMismatchException")
                        .httpStatusCode(400).exceptionBuilderSupplier(IdempotentParameterMismatchException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ProvisionedThroughputExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ProvisionedThroughputExceededException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ProvisionedThroughputExceededException::builder).build());
            case "InvalidJobIdException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidJobIdException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidJobIdException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Textract");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Starts the asynchronous analysis of an input document for relationships between detected items such as key-value
     * pairs, tables, and selection elements.
     * </p>
     * <p>
     * <code>StartDocumentAnalysis</code> can analyze text in documents that are in JPEG, PNG, TIFF, and PDF format. The
     * documents are stored in an Amazon S3 bucket. Use <a>DocumentLocation</a> to specify the bucket name and file name
     * of the document.
     * </p>
     * <p>
     * <code>StartDocumentAnalysis</code> returns a job identifier (<code>JobId</code>) that you use to get the results
     * of the operation. When text analysis is finished, Amazon Textract publishes a completion status to the Amazon
     * Simple Notification Service (Amazon SNS) topic that you specify in <code>NotificationChannel</code>. To get the
     * results of the text analysis operation, first check that the status value published to the Amazon SNS topic is
     * <code>SUCCEEDED</code>. If so, call <a>GetDocumentAnalysis</a>, and pass the job identifier (<code>JobId</code>)
     * from the initial call to <code>StartDocumentAnalysis</code>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/textract/latest/dg/how-it-works-analyzing.html">Document Text Analysis</a>.
     * </p>
     *
     * @param startDocumentAnalysisRequest
     * @return Result of the StartDocumentAnalysis operation returned by the service.
     * @throws InvalidParameterException
     *         An input parameter violated a constraint. For example, in synchronous operations, an
     *         <code>InvalidParameterException</code> exception occurs when neither of the <code>S3Object</code> or
     *         <code>Bytes</code> values are supplied in the <code>Document</code> request parameter. Validate your
     *         parameter before calling the API operation again.
     * @throws InvalidS3ObjectException
     *         Amazon Textract is unable to access the S3 object that's specified in the request. for more information,
     *         <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-access-control.html">Configure Access to
     *         Amazon S3</a> For troubleshooting information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/troubleshooting.html">Troubleshooting Amazon S3</a>
     * @throws InvalidKmsKeyException
     *         Indicates you do not have decrypt permissions with the KMS key entered, or the KMS key was entered
     *         incorrectly.
     * @throws UnsupportedDocumentException
     *         The format of the input document isn't supported. Documents for operations can be in PNG, JPEG, PDF, or
     *         TIFF format.
     * @throws DocumentTooLargeException
     *         The document can't be processed because it's too large. The maximum document size for synchronous
     *         operations 10 MB. The maximum document size for asynchronous operations is 500 MB for PDF files.
     * @throws BadDocumentException
     *         Amazon Textract isn't able to read the document. For more information on the document limits in Amazon
     *         Textract, see <a>limits</a>.
     * @throws AccessDeniedException
     *         You aren't authorized to perform the action. Use the Amazon Resource Name (ARN) of an authorized user or
     *         IAM role to perform the operation.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Textract.
     * @throws InternalServerErrorException
     *         Amazon Textract experienced a service issue. Try your call again.
     * @throws IdempotentParameterMismatchException
     *         A <code>ClientRequestToken</code> input parameter was reused with an operation, but at least one of the
     *         other input parameters is different from the previous call to the operation.
     * @throws ThrottlingException
     *         Amazon Textract is temporarily unable to process the request. Try your call again.
     * @throws LimitExceededException
     *         An Amazon Textract service limit was exceeded. For example, if you start too many asynchronous jobs
     *         concurrently, calls to start operations (<code>StartDocumentTextDetection</code>, for example) raise a
     *         LimitExceededException exception (HTTP status code: 400) until the number of concurrently running jobs is
     *         below the Amazon Textract service limit.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TextractException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TextractClient.StartDocumentAnalysis
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/StartDocumentAnalysis"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StartDocumentAnalysisResponse startDocumentAnalysis(StartDocumentAnalysisRequest startDocumentAnalysisRequest)
            throws InvalidParameterException, InvalidS3ObjectException, InvalidKmsKeyException, UnsupportedDocumentException,
            DocumentTooLargeException, BadDocumentException, AccessDeniedException, ProvisionedThroughputExceededException,
            InternalServerErrorException, IdempotentParameterMismatchException, ThrottlingException, LimitExceededException,
            AwsServiceException, SdkClientException, TextractException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartDocumentAnalysisResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StartDocumentAnalysisResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BadDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadDocumentException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "HumanLoopQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("HumanLoopQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(HumanLoopQuotaExceededException::builder).build());
            case "InvalidKMSKeyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidKMSKeyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidKmsKeyException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "DocumentTooLargeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DocumentTooLargeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DocumentTooLargeException::builder).build());
            case "UnsupportedDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedDocumentException::builder).build());
            case "InvalidS3ObjectException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidS3ObjectException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidS3ObjectException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "IdempotentParameterMismatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("IdempotentParameterMismatchException")
                        .httpStatusCode(400).exceptionBuilderSupplier(IdempotentParameterMismatchException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ProvisionedThroughputExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ProvisionedThroughputExceededException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ProvisionedThroughputExceededException::builder).build());
            case "InvalidJobIdException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidJobIdException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidJobIdException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startDocumentAnalysisRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startDocumentAnalysisRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Textract");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartDocumentAnalysis");

            return clientHandler.execute(new ClientExecutionParams<StartDocumentAnalysisRequest, StartDocumentAnalysisResponse>()
                    .withOperationName("StartDocumentAnalysis").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(startDocumentAnalysisRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartDocumentAnalysisRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Starts the asynchronous detection of text in a document. Amazon Textract can detect lines of text and the words
     * that make up a line of text.
     * </p>
     * <p>
     * <code>StartDocumentTextDetection</code> can analyze text in documents that are in JPEG, PNG, TIFF, and PDF
     * format. The documents are stored in an Amazon S3 bucket. Use <a>DocumentLocation</a> to specify the bucket name
     * and file name of the document.
     * </p>
     * <p>
     * <code>StartDocumentTextDetection</code> returns a job identifier (<code>JobId</code>) that you use to get the
     * results of the operation. When text detection is finished, Amazon Textract publishes a completion status to the
     * Amazon Simple Notification Service (Amazon SNS) topic that you specify in <code>NotificationChannel</code>. To
     * get the results of the text detection operation, first check that the status value published to the Amazon SNS
     * topic is <code>SUCCEEDED</code>. If so, call <a>GetDocumentTextDetection</a>, and pass the job identifier (
     * <code>JobId</code>) from the initial call to <code>StartDocumentTextDetection</code>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/textract/latest/dg/how-it-works-detecting.html">Document Text Detection</a>.
     * </p>
     *
     * @param startDocumentTextDetectionRequest
     * @return Result of the StartDocumentTextDetection operation returned by the service.
     * @throws InvalidParameterException
     *         An input parameter violated a constraint. For example, in synchronous operations, an
     *         <code>InvalidParameterException</code> exception occurs when neither of the <code>S3Object</code> or
     *         <code>Bytes</code> values are supplied in the <code>Document</code> request parameter. Validate your
     *         parameter before calling the API operation again.
     * @throws InvalidS3ObjectException
     *         Amazon Textract is unable to access the S3 object that's specified in the request. for more information,
     *         <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-access-control.html">Configure Access to
     *         Amazon S3</a> For troubleshooting information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/troubleshooting.html">Troubleshooting Amazon S3</a>
     * @throws InvalidKmsKeyException
     *         Indicates you do not have decrypt permissions with the KMS key entered, or the KMS key was entered
     *         incorrectly.
     * @throws UnsupportedDocumentException
     *         The format of the input document isn't supported. Documents for operations can be in PNG, JPEG, PDF, or
     *         TIFF format.
     * @throws DocumentTooLargeException
     *         The document can't be processed because it's too large. The maximum document size for synchronous
     *         operations 10 MB. The maximum document size for asynchronous operations is 500 MB for PDF files.
     * @throws BadDocumentException
     *         Amazon Textract isn't able to read the document. For more information on the document limits in Amazon
     *         Textract, see <a>limits</a>.
     * @throws AccessDeniedException
     *         You aren't authorized to perform the action. Use the Amazon Resource Name (ARN) of an authorized user or
     *         IAM role to perform the operation.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Textract.
     * @throws InternalServerErrorException
     *         Amazon Textract experienced a service issue. Try your call again.
     * @throws IdempotentParameterMismatchException
     *         A <code>ClientRequestToken</code> input parameter was reused with an operation, but at least one of the
     *         other input parameters is different from the previous call to the operation.
     * @throws ThrottlingException
     *         Amazon Textract is temporarily unable to process the request. Try your call again.
     * @throws LimitExceededException
     *         An Amazon Textract service limit was exceeded. For example, if you start too many asynchronous jobs
     *         concurrently, calls to start operations (<code>StartDocumentTextDetection</code>, for example) raise a
     *         LimitExceededException exception (HTTP status code: 400) until the number of concurrently running jobs is
     *         below the Amazon Textract service limit.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TextractException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TextractClient.StartDocumentTextDetection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/StartDocumentTextDetection"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StartDocumentTextDetectionResponse startDocumentTextDetection(
            StartDocumentTextDetectionRequest startDocumentTextDetectionRequest) throws InvalidParameterException,
            InvalidS3ObjectException, InvalidKmsKeyException, UnsupportedDocumentException, DocumentTooLargeException,
            BadDocumentException, AccessDeniedException, ProvisionedThroughputExceededException, InternalServerErrorException,
            IdempotentParameterMismatchException, ThrottlingException, LimitExceededException, AwsServiceException,
            SdkClientException, TextractException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartDocumentTextDetectionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StartDocumentTextDetectionResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BadDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadDocumentException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "HumanLoopQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("HumanLoopQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(HumanLoopQuotaExceededException::builder).build());
            case "InvalidKMSKeyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidKMSKeyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidKmsKeyException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "DocumentTooLargeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DocumentTooLargeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DocumentTooLargeException::builder).build());
            case "UnsupportedDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedDocumentException::builder).build());
            case "InvalidS3ObjectException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidS3ObjectException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidS3ObjectException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "IdempotentParameterMismatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("IdempotentParameterMismatchException")
                        .httpStatusCode(400).exceptionBuilderSupplier(IdempotentParameterMismatchException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ProvisionedThroughputExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ProvisionedThroughputExceededException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ProvisionedThroughputExceededException::builder).build());
            case "InvalidJobIdException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidJobIdException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidJobIdException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startDocumentTextDetectionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startDocumentTextDetectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Textract");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartDocumentTextDetection");

            return clientHandler
                    .execute(new ClientExecutionParams<StartDocumentTextDetectionRequest, StartDocumentTextDetectionResponse>()
                            .withOperationName("StartDocumentTextDetection").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(startDocumentTextDetectionRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new StartDocumentTextDetectionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Starts the asynchronous analysis of invoices or receipts for data like contact information, items purchased, and
     * vendor names.
     * </p>
     * <p>
     * <code>StartExpenseAnalysis</code> can analyze text in documents that are in JPEG, PNG, and PDF format. The
     * documents must be stored in an Amazon S3 bucket. Use the <a>DocumentLocation</a> parameter to specify the name of
     * your S3 bucket and the name of the document in that bucket.
     * </p>
     * <p>
     * <code>StartExpenseAnalysis</code> returns a job identifier (<code>JobId</code>) that you will provide to
     * <code>GetExpenseAnalysis</code> to retrieve the results of the operation. When the analysis of the input
     * invoices/receipts is finished, Amazon Textract publishes a completion status to the Amazon Simple Notification
     * Service (Amazon SNS) topic that you provide to the <code>NotificationChannel</code>. To obtain the results of the
     * invoice and receipt analysis operation, ensure that the status value published to the Amazon SNS topic is
     * <code>SUCCEEDED</code>. If so, call <a>GetExpenseAnalysis</a>, and pass the job identifier (<code>JobId</code>)
     * that was returned by your call to <code>StartExpenseAnalysis</code>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/textract/latest/dg/invoice-receipts.html">Analyzing Invoices and Receipts</a>.
     * </p>
     *
     * @param startExpenseAnalysisRequest
     * @return Result of the StartExpenseAnalysis operation returned by the service.
     * @throws InvalidParameterException
     *         An input parameter violated a constraint. For example, in synchronous operations, an
     *         <code>InvalidParameterException</code> exception occurs when neither of the <code>S3Object</code> or
     *         <code>Bytes</code> values are supplied in the <code>Document</code> request parameter. Validate your
     *         parameter before calling the API operation again.
     * @throws InvalidS3ObjectException
     *         Amazon Textract is unable to access the S3 object that's specified in the request. for more information,
     *         <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-access-control.html">Configure Access to
     *         Amazon S3</a> For troubleshooting information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/troubleshooting.html">Troubleshooting Amazon S3</a>
     * @throws InvalidKmsKeyException
     *         Indicates you do not have decrypt permissions with the KMS key entered, or the KMS key was entered
     *         incorrectly.
     * @throws UnsupportedDocumentException
     *         The format of the input document isn't supported. Documents for operations can be in PNG, JPEG, PDF, or
     *         TIFF format.
     * @throws DocumentTooLargeException
     *         The document can't be processed because it's too large. The maximum document size for synchronous
     *         operations 10 MB. The maximum document size for asynchronous operations is 500 MB for PDF files.
     * @throws BadDocumentException
     *         Amazon Textract isn't able to read the document. For more information on the document limits in Amazon
     *         Textract, see <a>limits</a>.
     * @throws AccessDeniedException
     *         You aren't authorized to perform the action. Use the Amazon Resource Name (ARN) of an authorized user or
     *         IAM role to perform the operation.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Textract.
     * @throws InternalServerErrorException
     *         Amazon Textract experienced a service issue. Try your call again.
     * @throws IdempotentParameterMismatchException
     *         A <code>ClientRequestToken</code> input parameter was reused with an operation, but at least one of the
     *         other input parameters is different from the previous call to the operation.
     * @throws ThrottlingException
     *         Amazon Textract is temporarily unable to process the request. Try your call again.
     * @throws LimitExceededException
     *         An Amazon Textract service limit was exceeded. For example, if you start too many asynchronous jobs
     *         concurrently, calls to start operations (<code>StartDocumentTextDetection</code>, for example) raise a
     *         LimitExceededException exception (HTTP status code: 400) until the number of concurrently running jobs is
     *         below the Amazon Textract service limit.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TextractException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TextractClient.StartExpenseAnalysis
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/StartExpenseAnalysis" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public StartExpenseAnalysisResponse startExpenseAnalysis(StartExpenseAnalysisRequest startExpenseAnalysisRequest)
            throws InvalidParameterException, InvalidS3ObjectException, InvalidKmsKeyException, UnsupportedDocumentException,
            DocumentTooLargeException, BadDocumentException, AccessDeniedException, ProvisionedThroughputExceededException,
            InternalServerErrorException, IdempotentParameterMismatchException, ThrottlingException, LimitExceededException,
            AwsServiceException, SdkClientException, TextractException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartExpenseAnalysisResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StartExpenseAnalysisResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BadDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadDocumentException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "HumanLoopQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("HumanLoopQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(HumanLoopQuotaExceededException::builder).build());
            case "InvalidKMSKeyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidKMSKeyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidKmsKeyException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "DocumentTooLargeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DocumentTooLargeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DocumentTooLargeException::builder).build());
            case "UnsupportedDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedDocumentException::builder).build());
            case "InvalidS3ObjectException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidS3ObjectException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidS3ObjectException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "IdempotentParameterMismatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("IdempotentParameterMismatchException")
                        .httpStatusCode(400).exceptionBuilderSupplier(IdempotentParameterMismatchException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ProvisionedThroughputExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ProvisionedThroughputExceededException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ProvisionedThroughputExceededException::builder).build());
            case "InvalidJobIdException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidJobIdException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidJobIdException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startExpenseAnalysisRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startExpenseAnalysisRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Textract");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartExpenseAnalysis");

            return clientHandler.execute(new ClientExecutionParams<StartExpenseAnalysisRequest, StartExpenseAnalysisResponse>()
                    .withOperationName("StartExpenseAnalysis").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(startExpenseAnalysisRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartExpenseAnalysisRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Starts the classification and analysis of an input document. <code>StartLendingAnalysis</code> initiates the
     * classification and analysis of a packet of lending documents. <code>StartLendingAnalysis</code> operates on a
     * document file located in an Amazon S3 bucket.
     * </p>
     * <p>
     * <code>StartLendingAnalysis</code> can analyze text in documents that are in one of the following formats: JPEG,
     * PNG, TIFF, PDF. Use <code>DocumentLocation</code> to specify the bucket name and the file name of the document.
     * </p>
     * <p>
     * <code>StartLendingAnalysis</code> returns a job identifier (<code>JobId</code>) that you use to get the results
     * of the operation. When the text analysis is finished, Amazon Textract publishes a completion status to the Amazon
     * Simple Notification Service (Amazon SNS) topic that you specify in <code>NotificationChannel</code>. To get the
     * results of the text analysis operation, first check that the status value published to the Amazon SNS topic is
     * SUCCEEDED. If the status is SUCCEEDED you can call either <code>GetLendingAnalysis</code> or
     * <code>GetLendingAnalysisSummary</code> and provide the <code>JobId</code> to obtain the results of the analysis.
     * </p>
     * <p>
     * If using <code>OutputConfig</code> to specify an Amazon S3 bucket, the output will be contained within the
     * specified prefix in a directory labeled with the job-id. In the directory there are 3 sub-directories:
     * </p>
     * <ul>
     * <li>
     * <p>
     * detailedResponse (contains the GetLendingAnalysis response)
     * </p>
     * </li>
     * <li>
     * <p>
     * summaryResponse (for the GetLendingAnalysisSummary response)
     * </p>
     * </li>
     * <li>
     * <p>
     * splitDocuments (documents split across logical boundaries)
     * </p>
     * </li>
     * </ul>
     *
     * @param startLendingAnalysisRequest
     * @return Result of the StartLendingAnalysis operation returned by the service.
     * @throws InvalidParameterException
     *         An input parameter violated a constraint. For example, in synchronous operations, an
     *         <code>InvalidParameterException</code> exception occurs when neither of the <code>S3Object</code> or
     *         <code>Bytes</code> values are supplied in the <code>Document</code> request parameter. Validate your
     *         parameter before calling the API operation again.
     * @throws InvalidS3ObjectException
     *         Amazon Textract is unable to access the S3 object that's specified in the request. for more information,
     *         <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-access-control.html">Configure Access to
     *         Amazon S3</a> For troubleshooting information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/troubleshooting.html">Troubleshooting Amazon S3</a>
     * @throws InvalidKmsKeyException
     *         Indicates you do not have decrypt permissions with the KMS key entered, or the KMS key was entered
     *         incorrectly.
     * @throws UnsupportedDocumentException
     *         The format of the input document isn't supported. Documents for operations can be in PNG, JPEG, PDF, or
     *         TIFF format.
     * @throws DocumentTooLargeException
     *         The document can't be processed because it's too large. The maximum document size for synchronous
     *         operations 10 MB. The maximum document size for asynchronous operations is 500 MB for PDF files.
     * @throws BadDocumentException
     *         Amazon Textract isn't able to read the document. For more information on the document limits in Amazon
     *         Textract, see <a>limits</a>.
     * @throws AccessDeniedException
     *         You aren't authorized to perform the action. Use the Amazon Resource Name (ARN) of an authorized user or
     *         IAM role to perform the operation.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Textract.
     * @throws InternalServerErrorException
     *         Amazon Textract experienced a service issue. Try your call again.
     * @throws IdempotentParameterMismatchException
     *         A <code>ClientRequestToken</code> input parameter was reused with an operation, but at least one of the
     *         other input parameters is different from the previous call to the operation.
     * @throws ThrottlingException
     *         Amazon Textract is temporarily unable to process the request. Try your call again.
     * @throws LimitExceededException
     *         An Amazon Textract service limit was exceeded. For example, if you start too many asynchronous jobs
     *         concurrently, calls to start operations (<code>StartDocumentTextDetection</code>, for example) raise a
     *         LimitExceededException exception (HTTP status code: 400) until the number of concurrently running jobs is
     *         below the Amazon Textract service limit.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TextractException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TextractClient.StartLendingAnalysis
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/StartLendingAnalysis" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public StartLendingAnalysisResponse startLendingAnalysis(StartLendingAnalysisRequest startLendingAnalysisRequest)
            throws InvalidParameterException, InvalidS3ObjectException, InvalidKmsKeyException, UnsupportedDocumentException,
            DocumentTooLargeException, BadDocumentException, AccessDeniedException, ProvisionedThroughputExceededException,
            InternalServerErrorException, IdempotentParameterMismatchException, ThrottlingException, LimitExceededException,
            AwsServiceException, SdkClientException, TextractException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartLendingAnalysisResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StartLendingAnalysisResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BadDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadDocumentException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "HumanLoopQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("HumanLoopQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(HumanLoopQuotaExceededException::builder).build());
            case "InvalidKMSKeyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidKMSKeyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidKmsKeyException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "DocumentTooLargeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DocumentTooLargeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DocumentTooLargeException::builder).build());
            case "UnsupportedDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedDocumentException::builder).build());
            case "InvalidS3ObjectException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidS3ObjectException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidS3ObjectException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "IdempotentParameterMismatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("IdempotentParameterMismatchException")
                        .httpStatusCode(400).exceptionBuilderSupplier(IdempotentParameterMismatchException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ProvisionedThroughputExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ProvisionedThroughputExceededException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ProvisionedThroughputExceededException::builder).build());
            case "InvalidJobIdException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidJobIdException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidJobIdException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startLendingAnalysisRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startLendingAnalysisRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Textract");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartLendingAnalysis");

            return clientHandler.execute(new ClientExecutionParams<StartLendingAnalysisRequest, StartLendingAnalysisResponse>()
                    .withOperationName("StartLendingAnalysis").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(startLendingAnalysisRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartLendingAnalysisRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds one or more tags to the specified resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         Returned when an operation tried to access a nonexistent resource.
     * @throws InvalidParameterException
     *         An input parameter violated a constraint. For example, in synchronous operations, an
     *         <code>InvalidParameterException</code> exception occurs when neither of the <code>S3Object</code> or
     *         <code>Bytes</code> values are supplied in the <code>Document</code> request parameter. Validate your
     *         parameter before calling the API operation again.
     * @throws ServiceQuotaExceededException
     *         Returned when a request cannot be completed as it would exceed a maximum service quota.
     * @throws AccessDeniedException
     *         You aren't authorized to perform the action. Use the Amazon Resource Name (ARN) of an authorized user or
     *         IAM role to perform the operation.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Textract.
     * @throws InternalServerErrorException
     *         Amazon Textract experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Textract is temporarily unable to process the request. Try your call again.
     * @throws ValidationException
     *         Indicates that a request was not valid. Check request for proper formatting.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TextractException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TextractClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ResourceNotFoundException,
            InvalidParameterException, ServiceQuotaExceededException, AccessDeniedException,
            ProvisionedThroughputExceededException, InternalServerErrorException, ThrottlingException, ValidationException,
            AwsServiceException, SdkClientException, TextractException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BadDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadDocumentException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "HumanLoopQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("HumanLoopQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(HumanLoopQuotaExceededException::builder).build());
            case "InvalidKMSKeyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidKMSKeyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidKmsKeyException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "DocumentTooLargeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DocumentTooLargeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DocumentTooLargeException::builder).build());
            case "UnsupportedDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedDocumentException::builder).build());
            case "InvalidS3ObjectException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidS3ObjectException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidS3ObjectException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "IdempotentParameterMismatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("IdempotentParameterMismatchException")
                        .httpStatusCode(400).exceptionBuilderSupplier(IdempotentParameterMismatchException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ProvisionedThroughputExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ProvisionedThroughputExceededException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ProvisionedThroughputExceededException::builder).build());
            case "InvalidJobIdException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidJobIdException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidJobIdException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Textract");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes any tags with the specified keys from the specified resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         Returned when an operation tried to access a nonexistent resource.
     * @throws InvalidParameterException
     *         An input parameter violated a constraint. For example, in synchronous operations, an
     *         <code>InvalidParameterException</code> exception occurs when neither of the <code>S3Object</code> or
     *         <code>Bytes</code> values are supplied in the <code>Document</code> request parameter. Validate your
     *         parameter before calling the API operation again.
     * @throws AccessDeniedException
     *         You aren't authorized to perform the action. Use the Amazon Resource Name (ARN) of an authorized user or
     *         IAM role to perform the operation.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Textract.
     * @throws InternalServerErrorException
     *         Amazon Textract experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Textract is temporarily unable to process the request. Try your call again.
     * @throws ValidationException
     *         Indicates that a request was not valid. Check request for proper formatting.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TextractException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TextractClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ResourceNotFoundException,
            InvalidParameterException, AccessDeniedException, ProvisionedThroughputExceededException,
            InternalServerErrorException, ThrottlingException, ValidationException, AwsServiceException, SdkClientException,
            TextractException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BadDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadDocumentException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "HumanLoopQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("HumanLoopQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(HumanLoopQuotaExceededException::builder).build());
            case "InvalidKMSKeyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidKMSKeyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidKmsKeyException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "DocumentTooLargeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DocumentTooLargeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DocumentTooLargeException::builder).build());
            case "UnsupportedDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedDocumentException::builder).build());
            case "InvalidS3ObjectException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidS3ObjectException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidS3ObjectException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "IdempotentParameterMismatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("IdempotentParameterMismatchException")
                        .httpStatusCode(400).exceptionBuilderSupplier(IdempotentParameterMismatchException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ProvisionedThroughputExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ProvisionedThroughputExceededException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ProvisionedThroughputExceededException::builder).build());
            case "InvalidJobIdException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidJobIdException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidJobIdException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Textract");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Update the configuration for an adapter. FeatureTypes configurations cannot be updated. At least one new
     * parameter must be specified as an argument.
     * </p>
     *
     * @param updateAdapterRequest
     * @return Result of the UpdateAdapter operation returned by the service.
     * @throws InvalidParameterException
     *         An input parameter violated a constraint. For example, in synchronous operations, an
     *         <code>InvalidParameterException</code> exception occurs when neither of the <code>S3Object</code> or
     *         <code>Bytes</code> values are supplied in the <code>Document</code> request parameter. Validate your
     *         parameter before calling the API operation again.
     * @throws AccessDeniedException
     *         You aren't authorized to perform the action. Use the Amazon Resource Name (ARN) of an authorized user or
     *         IAM role to perform the operation.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws ProvisionedThroughputExceededException
     *         The number of requests exceeded your throughput limit. If you want to increase this limit, contact Amazon
     *         Textract.
     * @throws InternalServerErrorException
     *         Amazon Textract experienced a service issue. Try your call again.
     * @throws ThrottlingException
     *         Amazon Textract is temporarily unable to process the request. Try your call again.
     * @throws ValidationException
     *         Indicates that a request was not valid. Check request for proper formatting.
     * @throws ResourceNotFoundException
     *         Returned when an operation tried to access a nonexistent resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TextractException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TextractClient.UpdateAdapter
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/UpdateAdapter" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateAdapterResponse updateAdapter(UpdateAdapterRequest updateAdapterRequest) throws InvalidParameterException,
            AccessDeniedException, ConflictException, ProvisionedThroughputExceededException, InternalServerErrorException,
            ThrottlingException, ValidationException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            TextractException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateAdapterResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateAdapterResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BadDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadDocumentException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "HumanLoopQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("HumanLoopQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(HumanLoopQuotaExceededException::builder).build());
            case "InvalidKMSKeyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidKMSKeyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidKmsKeyException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "DocumentTooLargeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DocumentTooLargeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DocumentTooLargeException::builder).build());
            case "UnsupportedDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedDocumentException::builder).build());
            case "InvalidS3ObjectException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidS3ObjectException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidS3ObjectException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "IdempotentParameterMismatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("IdempotentParameterMismatchException")
                        .httpStatusCode(400).exceptionBuilderSupplier(IdempotentParameterMismatchException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ProvisionedThroughputExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ProvisionedThroughputExceededException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ProvisionedThroughputExceededException::builder).build());
            case "InvalidJobIdException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidJobIdException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidJobIdException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateAdapterRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateAdapterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Textract");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateAdapter");

            return clientHandler.execute(new ClientExecutionParams<UpdateAdapterRequest, UpdateAdapterResponse>()
                    .withOperationName("UpdateAdapter").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateAdapterRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateAdapterRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        TextractServiceClientConfigurationBuilder serviceConfigBuilder = new TextractServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder.clientConfiguration(clientConfiguration).defaultServiceExceptionSupplier(TextractException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON).protocolVersion("1.1");
    }

    @Override
    public final TextractServiceClientConfiguration serviceClientConfiguration() {
        return new TextractServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
