/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.timestreaminfluxdb;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.timestreaminfluxdb.internal.TimestreamInfluxDbServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.timestreaminfluxdb.model.AccessDeniedException;
import software.amazon.awssdk.services.timestreaminfluxdb.model.ConflictException;
import software.amazon.awssdk.services.timestreaminfluxdb.model.CreateDbInstanceRequest;
import software.amazon.awssdk.services.timestreaminfluxdb.model.CreateDbInstanceResponse;
import software.amazon.awssdk.services.timestreaminfluxdb.model.CreateDbParameterGroupRequest;
import software.amazon.awssdk.services.timestreaminfluxdb.model.CreateDbParameterGroupResponse;
import software.amazon.awssdk.services.timestreaminfluxdb.model.DeleteDbInstanceRequest;
import software.amazon.awssdk.services.timestreaminfluxdb.model.DeleteDbInstanceResponse;
import software.amazon.awssdk.services.timestreaminfluxdb.model.GetDbInstanceRequest;
import software.amazon.awssdk.services.timestreaminfluxdb.model.GetDbInstanceResponse;
import software.amazon.awssdk.services.timestreaminfluxdb.model.GetDbParameterGroupRequest;
import software.amazon.awssdk.services.timestreaminfluxdb.model.GetDbParameterGroupResponse;
import software.amazon.awssdk.services.timestreaminfluxdb.model.InternalServerException;
import software.amazon.awssdk.services.timestreaminfluxdb.model.ListDbInstancesRequest;
import software.amazon.awssdk.services.timestreaminfluxdb.model.ListDbInstancesResponse;
import software.amazon.awssdk.services.timestreaminfluxdb.model.ListDbParameterGroupsRequest;
import software.amazon.awssdk.services.timestreaminfluxdb.model.ListDbParameterGroupsResponse;
import software.amazon.awssdk.services.timestreaminfluxdb.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.timestreaminfluxdb.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.timestreaminfluxdb.model.ResourceNotFoundException;
import software.amazon.awssdk.services.timestreaminfluxdb.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.timestreaminfluxdb.model.TagResourceRequest;
import software.amazon.awssdk.services.timestreaminfluxdb.model.TagResourceResponse;
import software.amazon.awssdk.services.timestreaminfluxdb.model.ThrottlingException;
import software.amazon.awssdk.services.timestreaminfluxdb.model.TimestreamInfluxDbException;
import software.amazon.awssdk.services.timestreaminfluxdb.model.UntagResourceRequest;
import software.amazon.awssdk.services.timestreaminfluxdb.model.UntagResourceResponse;
import software.amazon.awssdk.services.timestreaminfluxdb.model.UpdateDbInstanceRequest;
import software.amazon.awssdk.services.timestreaminfluxdb.model.UpdateDbInstanceResponse;
import software.amazon.awssdk.services.timestreaminfluxdb.model.ValidationException;
import software.amazon.awssdk.services.timestreaminfluxdb.transform.CreateDbInstanceRequestMarshaller;
import software.amazon.awssdk.services.timestreaminfluxdb.transform.CreateDbParameterGroupRequestMarshaller;
import software.amazon.awssdk.services.timestreaminfluxdb.transform.DeleteDbInstanceRequestMarshaller;
import software.amazon.awssdk.services.timestreaminfluxdb.transform.GetDbInstanceRequestMarshaller;
import software.amazon.awssdk.services.timestreaminfluxdb.transform.GetDbParameterGroupRequestMarshaller;
import software.amazon.awssdk.services.timestreaminfluxdb.transform.ListDbInstancesRequestMarshaller;
import software.amazon.awssdk.services.timestreaminfluxdb.transform.ListDbParameterGroupsRequestMarshaller;
import software.amazon.awssdk.services.timestreaminfluxdb.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.timestreaminfluxdb.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.timestreaminfluxdb.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.timestreaminfluxdb.transform.UpdateDbInstanceRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link TimestreamInfluxDbClient}.
 *
 * @see TimestreamInfluxDbClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultTimestreamInfluxDbClient implements TimestreamInfluxDbClient {
    private static final Logger log = Logger.loggerFor(DefaultTimestreamInfluxDbClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultTimestreamInfluxDbClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a new Timestream for InfluxDB DB instance.
     * </p>
     *
     * @param createDbInstanceRequest
     * @return Result of the CreateDbInstance operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         The request exceeds the service quota.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by Timestream for InfluxDB.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ConflictException
     *         The request conflicts with an existing resource in Timestream for InfluxDB.
     * @throws ResourceNotFoundException
     *         The requested resource was not found or does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TimestreamInfluxDbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TimestreamInfluxDbClient.CreateDbInstance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-influxdb-2023-01-27/CreateDbInstance"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateDbInstanceResponse createDbInstance(CreateDbInstanceRequest createDbInstanceRequest)
            throws ServiceQuotaExceededException, ValidationException, AccessDeniedException, InternalServerException,
            ConflictException, ResourceNotFoundException, ThrottlingException, AwsServiceException, SdkClientException,
            TimestreamInfluxDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateDbInstanceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateDbInstanceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createDbInstanceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDbInstanceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream InfluxDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDbInstance");

            return clientHandler.execute(new ClientExecutionParams<CreateDbInstanceRequest, CreateDbInstanceResponse>()
                    .withOperationName("CreateDbInstance").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createDbInstanceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateDbInstanceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new Timestream for InfluxDB DB parameter group to associate with DB instances.
     * </p>
     *
     * @param createDbParameterGroupRequest
     * @return Result of the CreateDbParameterGroup operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         The request exceeds the service quota.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by Timestream for InfluxDB.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ConflictException
     *         The request conflicts with an existing resource in Timestream for InfluxDB.
     * @throws ResourceNotFoundException
     *         The requested resource was not found or does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TimestreamInfluxDbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TimestreamInfluxDbClient.CreateDbParameterGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-influxdb-2023-01-27/CreateDbParameterGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateDbParameterGroupResponse createDbParameterGroup(CreateDbParameterGroupRequest createDbParameterGroupRequest)
            throws ServiceQuotaExceededException, ValidationException, AccessDeniedException, InternalServerException,
            ConflictException, ResourceNotFoundException, ThrottlingException, AwsServiceException, SdkClientException,
            TimestreamInfluxDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateDbParameterGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateDbParameterGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createDbParameterGroupRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDbParameterGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream InfluxDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDbParameterGroup");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateDbParameterGroupRequest, CreateDbParameterGroupResponse>()
                            .withOperationName("CreateDbParameterGroup").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(createDbParameterGroupRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateDbParameterGroupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a Timestream for InfluxDB DB instance.
     * </p>
     *
     * @param deleteDbInstanceRequest
     * @return Result of the DeleteDbInstance operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by Timestream for InfluxDB.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ConflictException
     *         The request conflicts with an existing resource in Timestream for InfluxDB.
     * @throws ResourceNotFoundException
     *         The requested resource was not found or does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TimestreamInfluxDbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TimestreamInfluxDbClient.DeleteDbInstance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-influxdb-2023-01-27/DeleteDbInstance"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteDbInstanceResponse deleteDbInstance(DeleteDbInstanceRequest deleteDbInstanceRequest) throws ValidationException,
            AccessDeniedException, InternalServerException, ConflictException, ResourceNotFoundException, ThrottlingException,
            AwsServiceException, SdkClientException, TimestreamInfluxDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteDbInstanceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteDbInstanceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteDbInstanceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDbInstanceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream InfluxDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDbInstance");

            return clientHandler.execute(new ClientExecutionParams<DeleteDbInstanceRequest, DeleteDbInstanceResponse>()
                    .withOperationName("DeleteDbInstance").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteDbInstanceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteDbInstanceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a Timestream for InfluxDB DB instance.
     * </p>
     *
     * @param getDbInstanceRequest
     * @return Result of the GetDbInstance operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by Timestream for InfluxDB.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ResourceNotFoundException
     *         The requested resource was not found or does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TimestreamInfluxDbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TimestreamInfluxDbClient.GetDbInstance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-influxdb-2023-01-27/GetDbInstance"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetDbInstanceResponse getDbInstance(GetDbInstanceRequest getDbInstanceRequest) throws ValidationException,
            AccessDeniedException, InternalServerException, ResourceNotFoundException, ThrottlingException, AwsServiceException,
            SdkClientException, TimestreamInfluxDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetDbInstanceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetDbInstanceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getDbInstanceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDbInstanceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream InfluxDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDbInstance");

            return clientHandler.execute(new ClientExecutionParams<GetDbInstanceRequest, GetDbInstanceResponse>()
                    .withOperationName("GetDbInstance").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getDbInstanceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetDbInstanceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a Timestream for InfluxDB DB parameter group.
     * </p>
     *
     * @param getDbParameterGroupRequest
     * @return Result of the GetDbParameterGroup operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by Timestream for InfluxDB.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ResourceNotFoundException
     *         The requested resource was not found or does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TimestreamInfluxDbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TimestreamInfluxDbClient.GetDbParameterGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-influxdb-2023-01-27/GetDbParameterGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetDbParameterGroupResponse getDbParameterGroup(GetDbParameterGroupRequest getDbParameterGroupRequest)
            throws ValidationException, AccessDeniedException, InternalServerException, ResourceNotFoundException,
            ThrottlingException, AwsServiceException, SdkClientException, TimestreamInfluxDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetDbParameterGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetDbParameterGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getDbParameterGroupRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDbParameterGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream InfluxDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDbParameterGroup");

            return clientHandler.execute(new ClientExecutionParams<GetDbParameterGroupRequest, GetDbParameterGroupResponse>()
                    .withOperationName("GetDbParameterGroup").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getDbParameterGroupRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetDbParameterGroupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of Timestream for InfluxDB DB instances.
     * </p>
     *
     * @param listDbInstancesRequest
     * @return Result of the ListDbInstances operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by Timestream for InfluxDB.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ResourceNotFoundException
     *         The requested resource was not found or does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TimestreamInfluxDbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TimestreamInfluxDbClient.ListDbInstances
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-influxdb-2023-01-27/ListDbInstances"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListDbInstancesResponse listDbInstances(ListDbInstancesRequest listDbInstancesRequest) throws ValidationException,
            AccessDeniedException, InternalServerException, ResourceNotFoundException, ThrottlingException, AwsServiceException,
            SdkClientException, TimestreamInfluxDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListDbInstancesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListDbInstancesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listDbInstancesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDbInstancesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream InfluxDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDbInstances");

            return clientHandler.execute(new ClientExecutionParams<ListDbInstancesRequest, ListDbInstancesResponse>()
                    .withOperationName("ListDbInstances").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listDbInstancesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListDbInstancesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of Timestream for InfluxDB DB parameter groups.
     * </p>
     *
     * @param listDbParameterGroupsRequest
     * @return Result of the ListDbParameterGroups operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by Timestream for InfluxDB.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ResourceNotFoundException
     *         The requested resource was not found or does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TimestreamInfluxDbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TimestreamInfluxDbClient.ListDbParameterGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-influxdb-2023-01-27/ListDbParameterGroups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListDbParameterGroupsResponse listDbParameterGroups(ListDbParameterGroupsRequest listDbParameterGroupsRequest)
            throws ValidationException, AccessDeniedException, InternalServerException, ResourceNotFoundException,
            ThrottlingException, AwsServiceException, SdkClientException, TimestreamInfluxDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListDbParameterGroupsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListDbParameterGroupsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listDbParameterGroupsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDbParameterGroupsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream InfluxDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDbParameterGroups");

            return clientHandler.execute(new ClientExecutionParams<ListDbParameterGroupsRequest, ListDbParameterGroupsResponse>()
                    .withOperationName("ListDbParameterGroups").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listDbParameterGroupsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListDbParameterGroupsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * A list of tags applied to the resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource was not found or does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TimestreamInfluxDbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TimestreamInfluxDbClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-influxdb-2023-01-27/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ResourceNotFoundException, AwsServiceException, SdkClientException, TimestreamInfluxDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream InfluxDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Tags are composed of a Key/Value pairs. You can use tags to categorize and track your Timestream for InfluxDB
     * resources.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource was not found or does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TimestreamInfluxDbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TimestreamInfluxDbClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-influxdb-2023-01-27/TagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ResourceNotFoundException,
            AwsServiceException, SdkClientException, TimestreamInfluxDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream InfluxDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes the tag from the specified resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource was not found or does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TimestreamInfluxDbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TimestreamInfluxDbClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-influxdb-2023-01-27/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ResourceNotFoundException,
            AwsServiceException, SdkClientException, TimestreamInfluxDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream InfluxDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates a Timestream for InfluxDB DB instance.
     * </p>
     *
     * @param updateDbInstanceRequest
     * @return Result of the UpdateDbInstance operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by Timestream for InfluxDB.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ConflictException
     *         The request conflicts with an existing resource in Timestream for InfluxDB.
     * @throws ResourceNotFoundException
     *         The requested resource was not found or does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TimestreamInfluxDbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TimestreamInfluxDbClient.UpdateDbInstance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-influxdb-2023-01-27/UpdateDbInstance"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateDbInstanceResponse updateDbInstance(UpdateDbInstanceRequest updateDbInstanceRequest) throws ValidationException,
            AccessDeniedException, InternalServerException, ConflictException, ResourceNotFoundException, ThrottlingException,
            AwsServiceException, SdkClientException, TimestreamInfluxDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateDbInstanceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateDbInstanceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateDbInstanceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateDbInstanceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream InfluxDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDbInstance");

            return clientHandler.execute(new ClientExecutionParams<UpdateDbInstanceRequest, UpdateDbInstanceResponse>()
                    .withOperationName("UpdateDbInstance").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateDbInstanceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateDbInstanceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        TimestreamInfluxDbServiceClientConfigurationBuilder serviceConfigBuilder = new TimestreamInfluxDbServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(TimestreamInfluxDbException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.0")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(400).build());
    }

    @Override
    public final TimestreamInfluxDbServiceClientConfiguration serviceClientConfiguration() {
        return new TimestreamInfluxDbServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
