/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.timestreamquery;

import java.net.URI;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.config.AwsClientOption;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.endpointdiscovery.EndpointDiscoveryRefreshCache;
import software.amazon.awssdk.core.endpointdiscovery.EndpointDiscoveryRequest;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.timestreamquery.model.AccessDeniedException;
import software.amazon.awssdk.services.timestreamquery.model.CancelQueryRequest;
import software.amazon.awssdk.services.timestreamquery.model.CancelQueryResponse;
import software.amazon.awssdk.services.timestreamquery.model.ConflictException;
import software.amazon.awssdk.services.timestreamquery.model.CreateScheduledQueryRequest;
import software.amazon.awssdk.services.timestreamquery.model.CreateScheduledQueryResponse;
import software.amazon.awssdk.services.timestreamquery.model.DeleteScheduledQueryRequest;
import software.amazon.awssdk.services.timestreamquery.model.DeleteScheduledQueryResponse;
import software.amazon.awssdk.services.timestreamquery.model.DescribeEndpointsRequest;
import software.amazon.awssdk.services.timestreamquery.model.DescribeEndpointsResponse;
import software.amazon.awssdk.services.timestreamquery.model.DescribeScheduledQueryRequest;
import software.amazon.awssdk.services.timestreamquery.model.DescribeScheduledQueryResponse;
import software.amazon.awssdk.services.timestreamquery.model.ExecuteScheduledQueryRequest;
import software.amazon.awssdk.services.timestreamquery.model.ExecuteScheduledQueryResponse;
import software.amazon.awssdk.services.timestreamquery.model.InternalServerException;
import software.amazon.awssdk.services.timestreamquery.model.InvalidEndpointException;
import software.amazon.awssdk.services.timestreamquery.model.ListScheduledQueriesRequest;
import software.amazon.awssdk.services.timestreamquery.model.ListScheduledQueriesResponse;
import software.amazon.awssdk.services.timestreamquery.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.timestreamquery.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.timestreamquery.model.PrepareQueryRequest;
import software.amazon.awssdk.services.timestreamquery.model.PrepareQueryResponse;
import software.amazon.awssdk.services.timestreamquery.model.QueryExecutionException;
import software.amazon.awssdk.services.timestreamquery.model.QueryRequest;
import software.amazon.awssdk.services.timestreamquery.model.QueryResponse;
import software.amazon.awssdk.services.timestreamquery.model.ResourceNotFoundException;
import software.amazon.awssdk.services.timestreamquery.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.timestreamquery.model.TagResourceRequest;
import software.amazon.awssdk.services.timestreamquery.model.TagResourceResponse;
import software.amazon.awssdk.services.timestreamquery.model.ThrottlingException;
import software.amazon.awssdk.services.timestreamquery.model.TimestreamQueryException;
import software.amazon.awssdk.services.timestreamquery.model.TimestreamQueryRequest;
import software.amazon.awssdk.services.timestreamquery.model.UntagResourceRequest;
import software.amazon.awssdk.services.timestreamquery.model.UntagResourceResponse;
import software.amazon.awssdk.services.timestreamquery.model.UpdateScheduledQueryRequest;
import software.amazon.awssdk.services.timestreamquery.model.UpdateScheduledQueryResponse;
import software.amazon.awssdk.services.timestreamquery.model.ValidationException;
import software.amazon.awssdk.services.timestreamquery.paginators.ListScheduledQueriesPublisher;
import software.amazon.awssdk.services.timestreamquery.paginators.ListTagsForResourcePublisher;
import software.amazon.awssdk.services.timestreamquery.paginators.QueryPublisher;
import software.amazon.awssdk.services.timestreamquery.transform.CancelQueryRequestMarshaller;
import software.amazon.awssdk.services.timestreamquery.transform.CreateScheduledQueryRequestMarshaller;
import software.amazon.awssdk.services.timestreamquery.transform.DeleteScheduledQueryRequestMarshaller;
import software.amazon.awssdk.services.timestreamquery.transform.DescribeEndpointsRequestMarshaller;
import software.amazon.awssdk.services.timestreamquery.transform.DescribeScheduledQueryRequestMarshaller;
import software.amazon.awssdk.services.timestreamquery.transform.ExecuteScheduledQueryRequestMarshaller;
import software.amazon.awssdk.services.timestreamquery.transform.ListScheduledQueriesRequestMarshaller;
import software.amazon.awssdk.services.timestreamquery.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.timestreamquery.transform.PrepareQueryRequestMarshaller;
import software.amazon.awssdk.services.timestreamquery.transform.QueryRequestMarshaller;
import software.amazon.awssdk.services.timestreamquery.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.timestreamquery.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.timestreamquery.transform.UpdateScheduledQueryRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link TimestreamQueryAsyncClient}.
 *
 * @see TimestreamQueryAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultTimestreamQueryAsyncClient implements TimestreamQueryAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultTimestreamQueryAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private EndpointDiscoveryRefreshCache endpointDiscoveryCache;

    protected DefaultTimestreamQueryAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
        if (clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED)) {
            this.endpointDiscoveryCache = EndpointDiscoveryRefreshCache.create(TimestreamQueryAsyncEndpointDiscoveryCacheLoader
                    .create(this));
            if (clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE) {
                log.warn("Endpoint discovery is enabled for this client, and an endpoint override was also specified. This will disable endpoint discovery for methods that require it, instead using the specified endpoint override. This may or may not be what you intended.");
            }
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Cancels a query that has been issued. Cancellation is provided only if the query has not completed running before
     * the cancellation request was issued. Because cancellation is an idempotent operation, subsequent cancellation
     * requests will return a <code>CancellationMessage</code>, indicating that the query has already been canceled. See
     * <a href="https://docs.aws.amazon.com/Timestream/latest/developerguide/code-samples.cancel-query.html">code
     * sample</a> for details.
     * </p>
     *
     * @param cancelQueryRequest
     * @return A Java Future containing the result of the CancelQuery operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You are not authorized to perform this action.</li>
     *         <li>InternalServerException Timestream was unable to fully process this request because of an internal
     *         server error.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException Invalid or malformed request.</li>
     *         <li>InvalidEndpointException The requested endpoint was not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TimestreamQueryException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample TimestreamQueryAsyncClient.CancelQuery
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-query-2018-11-01/CancelQuery" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CancelQueryResponse> cancelQuery(CancelQueryRequest cancelQueryRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelQueryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Query");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelQuery");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CancelQueryResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CancelQueryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
            boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
            if (endpointOverridden) {
                endpointDiscoveryEnabled = false;
            } else if (!endpointDiscoveryEnabled) {
                throw new IllegalStateException(
                        "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
            }
            URI cachedEndpoint = null;
            if (endpointDiscoveryEnabled) {
                String key = cancelQueryRequest.overrideConfiguration()
                        .flatMap(AwsRequestOverrideConfiguration::credentialsProvider)
                        .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER)).resolveCredentials()
                        .accessKeyId();
                EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                        .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT))
                        .overrideConfiguration(cancelQueryRequest.overrideConfiguration().orElse(null)).build();
                cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
            }

            CompletableFuture<CancelQueryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CancelQueryRequest, CancelQueryResponse>()
                            .withOperationName("CancelQuery").withMarshaller(new CancelQueryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).discoveredEndpoint(cachedEndpoint)
                            .withInput(cancelQueryRequest));
            CompletableFuture<CancelQueryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Create a scheduled query that will be run on your behalf at the configured schedule. Timestream assumes the
     * execution role provided as part of the <code>ScheduledQueryExecutionRoleArn</code> parameter to run the query.
     * You can use the <code>NotificationConfiguration</code> parameter to configure notification for your scheduled
     * query operations.
     * </p>
     *
     * @param createScheduledQueryRequest
     * @return A Java Future containing the result of the CreateScheduledQuery operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You are not authorized to perform this action.</li>
     *         <li>ConflictException Unable to poll results for a cancelled query.</li>
     *         <li>InternalServerException Timestream was unable to fully process this request because of an internal
     *         server error.</li>
     *         <li>ServiceQuotaExceededException You have exceeded the service quota.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException Invalid or malformed request.</li>
     *         <li>InvalidEndpointException The requested endpoint was not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TimestreamQueryException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample TimestreamQueryAsyncClient.CreateScheduledQuery
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-query-2018-11-01/CreateScheduledQuery"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateScheduledQueryResponse> createScheduledQuery(
            CreateScheduledQueryRequest createScheduledQueryRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createScheduledQueryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Query");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateScheduledQuery");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateScheduledQueryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateScheduledQueryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
            boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
            if (endpointOverridden) {
                endpointDiscoveryEnabled = false;
            } else if (!endpointDiscoveryEnabled) {
                throw new IllegalStateException(
                        "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
            }
            URI cachedEndpoint = null;
            if (endpointDiscoveryEnabled) {
                String key = createScheduledQueryRequest.overrideConfiguration()
                        .flatMap(AwsRequestOverrideConfiguration::credentialsProvider)
                        .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER)).resolveCredentials()
                        .accessKeyId();
                EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                        .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT))
                        .overrideConfiguration(createScheduledQueryRequest.overrideConfiguration().orElse(null)).build();
                cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
            }

            CompletableFuture<CreateScheduledQueryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateScheduledQueryRequest, CreateScheduledQueryResponse>()
                            .withOperationName("CreateScheduledQuery")
                            .withMarshaller(new CreateScheduledQueryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).discoveredEndpoint(cachedEndpoint)
                            .withInput(createScheduledQueryRequest));
            CompletableFuture<CreateScheduledQueryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a given scheduled query. This is an irreversible operation.
     * </p>
     *
     * @param deleteScheduledQueryRequest
     * @return A Java Future containing the result of the DeleteScheduledQuery operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You are not authorized to perform this action.</li>
     *         <li>InternalServerException Timestream was unable to fully process this request because of an internal
     *         server error.</li>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException Invalid or malformed request.</li>
     *         <li>InvalidEndpointException The requested endpoint was not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TimestreamQueryException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample TimestreamQueryAsyncClient.DeleteScheduledQuery
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-query-2018-11-01/DeleteScheduledQuery"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteScheduledQueryResponse> deleteScheduledQuery(
            DeleteScheduledQueryRequest deleteScheduledQueryRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteScheduledQueryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Query");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteScheduledQuery");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteScheduledQueryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteScheduledQueryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
            boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
            if (endpointOverridden) {
                endpointDiscoveryEnabled = false;
            } else if (!endpointDiscoveryEnabled) {
                throw new IllegalStateException(
                        "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
            }
            URI cachedEndpoint = null;
            if (endpointDiscoveryEnabled) {
                String key = deleteScheduledQueryRequest.overrideConfiguration()
                        .flatMap(AwsRequestOverrideConfiguration::credentialsProvider)
                        .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER)).resolveCredentials()
                        .accessKeyId();
                EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                        .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT))
                        .overrideConfiguration(deleteScheduledQueryRequest.overrideConfiguration().orElse(null)).build();
                cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
            }

            CompletableFuture<DeleteScheduledQueryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteScheduledQueryRequest, DeleteScheduledQueryResponse>()
                            .withOperationName("DeleteScheduledQuery")
                            .withMarshaller(new DeleteScheduledQueryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).discoveredEndpoint(cachedEndpoint)
                            .withInput(deleteScheduledQueryRequest));
            CompletableFuture<DeleteScheduledQueryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * DescribeEndpoints returns a list of available endpoints to make Timestream API calls against. This API is
     * available through both Write and Query.
     * </p>
     * <p>
     * Because the Timestream SDKs are designed to transparently work with the service’s architecture, including the
     * management and mapping of the service endpoints, <i>it is not recommended that you use this API unless</i>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * You are using <a href="https://docs.aws.amazon.com/Timestream/latest/developerguide/VPCEndpoints">VPC endpoints
     * (Amazon Web Services PrivateLink) with Timestream </a>
     * </p>
     * </li>
     * <li>
     * <p>
     * Your application uses a programming language that does not yet have SDK support
     * </p>
     * </li>
     * <li>
     * <p>
     * You require better control over the client-side implementation
     * </p>
     * </li>
     * </ul>
     * <p>
     * For detailed information on how and when to use and implement DescribeEndpoints, see <a
     * href="https://docs.aws.amazon.com/Timestream/latest/developerguide/Using.API.html#Using-API.endpoint-discovery"
     * >The Endpoint Discovery Pattern</a>.
     * </p>
     *
     * @param describeEndpointsRequest
     * @return A Java Future containing the result of the DescribeEndpoints operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Timestream was unable to fully process this request because of an internal
     *         server error.</li>
     *         <li>ValidationException Invalid or malformed request.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TimestreamQueryException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample TimestreamQueryAsyncClient.DescribeEndpoints
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-query-2018-11-01/DescribeEndpoints"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeEndpointsResponse> describeEndpoints(DescribeEndpointsRequest describeEndpointsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeEndpointsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Query");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeEndpoints");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeEndpointsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeEndpointsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeEndpointsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeEndpointsRequest, DescribeEndpointsResponse>()
                            .withOperationName("DescribeEndpoints")
                            .withMarshaller(new DescribeEndpointsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeEndpointsRequest));
            CompletableFuture<DescribeEndpointsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides detailed information about a scheduled query.
     * </p>
     *
     * @param describeScheduledQueryRequest
     * @return A Java Future containing the result of the DescribeScheduledQuery operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You are not authorized to perform this action.</li>
     *         <li>InternalServerException Timestream was unable to fully process this request because of an internal
     *         server error.</li>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException Invalid or malformed request.</li>
     *         <li>InvalidEndpointException The requested endpoint was not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TimestreamQueryException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample TimestreamQueryAsyncClient.DescribeScheduledQuery
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-query-2018-11-01/DescribeScheduledQuery"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeScheduledQueryResponse> describeScheduledQuery(
            DescribeScheduledQueryRequest describeScheduledQueryRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeScheduledQueryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Query");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeScheduledQuery");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeScheduledQueryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeScheduledQueryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
            boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
            if (endpointOverridden) {
                endpointDiscoveryEnabled = false;
            } else if (!endpointDiscoveryEnabled) {
                throw new IllegalStateException(
                        "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
            }
            URI cachedEndpoint = null;
            if (endpointDiscoveryEnabled) {
                String key = describeScheduledQueryRequest.overrideConfiguration()
                        .flatMap(AwsRequestOverrideConfiguration::credentialsProvider)
                        .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER)).resolveCredentials()
                        .accessKeyId();
                EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                        .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT))
                        .overrideConfiguration(describeScheduledQueryRequest.overrideConfiguration().orElse(null)).build();
                cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
            }

            CompletableFuture<DescribeScheduledQueryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeScheduledQueryRequest, DescribeScheduledQueryResponse>()
                            .withOperationName("DescribeScheduledQuery")
                            .withMarshaller(new DescribeScheduledQueryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).discoveredEndpoint(cachedEndpoint)
                            .withInput(describeScheduledQueryRequest));
            CompletableFuture<DescribeScheduledQueryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * You can use this API to run a scheduled query manually.
     * </p>
     *
     * @param executeScheduledQueryRequest
     * @return A Java Future containing the result of the ExecuteScheduledQuery operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You are not authorized to perform this action.</li>
     *         <li>InternalServerException Timestream was unable to fully process this request because of an internal
     *         server error.</li>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException Invalid or malformed request.</li>
     *         <li>InvalidEndpointException The requested endpoint was not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TimestreamQueryException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample TimestreamQueryAsyncClient.ExecuteScheduledQuery
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-query-2018-11-01/ExecuteScheduledQuery"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ExecuteScheduledQueryResponse> executeScheduledQuery(
            ExecuteScheduledQueryRequest executeScheduledQueryRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, executeScheduledQueryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Query");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ExecuteScheduledQuery");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ExecuteScheduledQueryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ExecuteScheduledQueryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
            boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
            if (endpointOverridden) {
                endpointDiscoveryEnabled = false;
            } else if (!endpointDiscoveryEnabled) {
                throw new IllegalStateException(
                        "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
            }
            URI cachedEndpoint = null;
            if (endpointDiscoveryEnabled) {
                String key = executeScheduledQueryRequest.overrideConfiguration()
                        .flatMap(AwsRequestOverrideConfiguration::credentialsProvider)
                        .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER)).resolveCredentials()
                        .accessKeyId();
                EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                        .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT))
                        .overrideConfiguration(executeScheduledQueryRequest.overrideConfiguration().orElse(null)).build();
                cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
            }

            CompletableFuture<ExecuteScheduledQueryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ExecuteScheduledQueryRequest, ExecuteScheduledQueryResponse>()
                            .withOperationName("ExecuteScheduledQuery")
                            .withMarshaller(new ExecuteScheduledQueryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).discoveredEndpoint(cachedEndpoint)
                            .withInput(executeScheduledQueryRequest));
            CompletableFuture<ExecuteScheduledQueryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets a list of all scheduled queries in the caller's Amazon account and Region. <code>ListScheduledQueries</code>
     * is eventually consistent.
     * </p>
     *
     * @param listScheduledQueriesRequest
     * @return A Java Future containing the result of the ListScheduledQueries operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You are not authorized to perform this action.</li>
     *         <li>InternalServerException Timestream was unable to fully process this request because of an internal
     *         server error.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException Invalid or malformed request.</li>
     *         <li>InvalidEndpointException The requested endpoint was not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TimestreamQueryException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample TimestreamQueryAsyncClient.ListScheduledQueries
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-query-2018-11-01/ListScheduledQueries"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListScheduledQueriesResponse> listScheduledQueries(
            ListScheduledQueriesRequest listScheduledQueriesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listScheduledQueriesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Query");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListScheduledQueries");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListScheduledQueriesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListScheduledQueriesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
            boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
            if (endpointOverridden) {
                endpointDiscoveryEnabled = false;
            } else if (!endpointDiscoveryEnabled) {
                throw new IllegalStateException(
                        "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
            }
            URI cachedEndpoint = null;
            if (endpointDiscoveryEnabled) {
                String key = listScheduledQueriesRequest.overrideConfiguration()
                        .flatMap(AwsRequestOverrideConfiguration::credentialsProvider)
                        .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER)).resolveCredentials()
                        .accessKeyId();
                EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                        .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT))
                        .overrideConfiguration(listScheduledQueriesRequest.overrideConfiguration().orElse(null)).build();
                cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
            }

            CompletableFuture<ListScheduledQueriesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListScheduledQueriesRequest, ListScheduledQueriesResponse>()
                            .withOperationName("ListScheduledQueries")
                            .withMarshaller(new ListScheduledQueriesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).discoveredEndpoint(cachedEndpoint)
                            .withInput(listScheduledQueriesRequest));
            CompletableFuture<ListScheduledQueriesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets a list of all scheduled queries in the caller's Amazon account and Region. <code>ListScheduledQueries</code>
     * is eventually consistent.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listScheduledQueries(software.amazon.awssdk.services.timestreamquery.model.ListScheduledQueriesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.timestreamquery.paginators.ListScheduledQueriesPublisher publisher = client.listScheduledQueriesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.timestreamquery.paginators.ListScheduledQueriesPublisher publisher = client.listScheduledQueriesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.timestreamquery.model.ListScheduledQueriesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.timestreamquery.model.ListScheduledQueriesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listScheduledQueries(software.amazon.awssdk.services.timestreamquery.model.ListScheduledQueriesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listScheduledQueriesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You are not authorized to perform this action.</li>
     *         <li>InternalServerException Timestream was unable to fully process this request because of an internal
     *         server error.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException Invalid or malformed request.</li>
     *         <li>InvalidEndpointException The requested endpoint was not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TimestreamQueryException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample TimestreamQueryAsyncClient.ListScheduledQueries
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-query-2018-11-01/ListScheduledQueries"
     *      target="_top">AWS API Documentation</a>
     */
    public ListScheduledQueriesPublisher listScheduledQueriesPaginator(ListScheduledQueriesRequest listScheduledQueriesRequest) {
        return new ListScheduledQueriesPublisher(this, applyPaginatorUserAgent(listScheduledQueriesRequest));
    }

    /**
     * <p>
     * List all tags on a Timestream query resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException Invalid or malformed request.</li>
     *         <li>InvalidEndpointException The requested endpoint was not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TimestreamQueryException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample TimestreamQueryAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-query-2018-11-01/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Query");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
            boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
            if (endpointOverridden) {
                endpointDiscoveryEnabled = false;
            } else if (!endpointDiscoveryEnabled) {
                throw new IllegalStateException(
                        "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
            }
            URI cachedEndpoint = null;
            if (endpointDiscoveryEnabled) {
                String key = listTagsForResourceRequest.overrideConfiguration()
                        .flatMap(AwsRequestOverrideConfiguration::credentialsProvider)
                        .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER)).resolveCredentials()
                        .accessKeyId();
                EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                        .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT))
                        .overrideConfiguration(listTagsForResourceRequest.overrideConfiguration().orElse(null)).build();
                cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
            }

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).discoveredEndpoint(cachedEndpoint)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List all tags on a Timestream query resource.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listTagsForResource(software.amazon.awssdk.services.timestreamquery.model.ListTagsForResourceRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.timestreamquery.paginators.ListTagsForResourcePublisher publisher = client.listTagsForResourcePaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.timestreamquery.paginators.ListTagsForResourcePublisher publisher = client.listTagsForResourcePaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.timestreamquery.model.ListTagsForResourceResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.timestreamquery.model.ListTagsForResourceResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listTagsForResource(software.amazon.awssdk.services.timestreamquery.model.ListTagsForResourceRequest)}
     * operation.</b>
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException Invalid or malformed request.</li>
     *         <li>InvalidEndpointException The requested endpoint was not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TimestreamQueryException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample TimestreamQueryAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-query-2018-11-01/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    public ListTagsForResourcePublisher listTagsForResourcePaginator(ListTagsForResourceRequest listTagsForResourceRequest) {
        return new ListTagsForResourcePublisher(this, applyPaginatorUserAgent(listTagsForResourceRequest));
    }

    /**
     * <p>
     * A synchronous operation that allows you to submit a query with parameters to be stored by Timestream for later
     * running. Timestream only supports using this operation with the <code>PrepareQueryRequest&#36ValidateOnly</code>
     * set to <code>true</code>.
     * </p>
     *
     * @param prepareQueryRequest
     * @return A Java Future containing the result of the PrepareQuery operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You are not authorized to perform this action.</li>
     *         <li>InternalServerException Timestream was unable to fully process this request because of an internal
     *         server error.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException Invalid or malformed request.</li>
     *         <li>InvalidEndpointException The requested endpoint was not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TimestreamQueryException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample TimestreamQueryAsyncClient.PrepareQuery
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-query-2018-11-01/PrepareQuery" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<PrepareQueryResponse> prepareQuery(PrepareQueryRequest prepareQueryRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, prepareQueryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Query");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PrepareQuery");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PrepareQueryResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    PrepareQueryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
            boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
            if (endpointOverridden) {
                endpointDiscoveryEnabled = false;
            } else if (!endpointDiscoveryEnabled) {
                throw new IllegalStateException(
                        "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
            }
            URI cachedEndpoint = null;
            if (endpointDiscoveryEnabled) {
                String key = prepareQueryRequest.overrideConfiguration()
                        .flatMap(AwsRequestOverrideConfiguration::credentialsProvider)
                        .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER)).resolveCredentials()
                        .accessKeyId();
                EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                        .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT))
                        .overrideConfiguration(prepareQueryRequest.overrideConfiguration().orElse(null)).build();
                cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
            }

            CompletableFuture<PrepareQueryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PrepareQueryRequest, PrepareQueryResponse>()
                            .withOperationName("PrepareQuery").withMarshaller(new PrepareQueryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).discoveredEndpoint(cachedEndpoint)
                            .withInput(prepareQueryRequest));
            CompletableFuture<PrepareQueryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * <code>Query</code> is a synchronous operation that enables you to run a query against your Amazon Timestream
     * data. <code>Query</code> will time out after 60 seconds. You must update the default timeout in the SDK to
     * support a timeout of 60 seconds. See the <a
     * href="https://docs.aws.amazon.com/Timestream/latest/developerguide/code-samples.run-query.html">code sample</a>
     * for details.
     * </p>
     * <p>
     * Your query request will fail in the following cases:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If you submit a <code>Query</code> request with the same client token outside of the 5-minute idempotency window.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you submit a <code>Query</code> request with the same client token, but change other parameters, within the
     * 5-minute idempotency window.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the size of the row (including the query metadata) exceeds 1 MB, then the query will fail with the following
     * error message:
     * </p>
     * <p>
     * <code>Query aborted as max page response size has been exceeded by the output result row</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * If the IAM principal of the query initiator and the result reader are not the same and/or the query initiator and
     * the result reader do not have the same query string in the query requests, the query will fail with an
     * <code>Invalid pagination token</code> error.
     * </p>
     * </li>
     * </ul>
     *
     * @param queryRequest
     * @return A Java Future containing the result of the Query operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You are not authorized to perform this action.</li>
     *         <li>ConflictException Unable to poll results for a cancelled query.</li>
     *         <li>InternalServerException Timestream was unable to fully process this request because of an internal
     *         server error.</li>
     *         <li>QueryExecutionException Timestream was unable to run the query successfully.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException Invalid or malformed request.</li>
     *         <li>InvalidEndpointException The requested endpoint was not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TimestreamQueryException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample TimestreamQueryAsyncClient.Query
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-query-2018-11-01/Query" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<QueryResponse> query(QueryRequest queryRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, queryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Query");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "Query");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<QueryResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    QueryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
            boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
            if (endpointOverridden) {
                endpointDiscoveryEnabled = false;
            } else if (!endpointDiscoveryEnabled) {
                throw new IllegalStateException(
                        "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
            }
            URI cachedEndpoint = null;
            if (endpointDiscoveryEnabled) {
                String key = queryRequest.overrideConfiguration().flatMap(AwsRequestOverrideConfiguration::credentialsProvider)
                        .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER)).resolveCredentials()
                        .accessKeyId();
                EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                        .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT))
                        .overrideConfiguration(queryRequest.overrideConfiguration().orElse(null)).build();
                cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
            }

            CompletableFuture<QueryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<QueryRequest, QueryResponse>().withOperationName("Query")
                            .withMarshaller(new QueryRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .discoveredEndpoint(cachedEndpoint).withInput(queryRequest));
            CompletableFuture<QueryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * <code>Query</code> is a synchronous operation that enables you to run a query against your Amazon Timestream
     * data. <code>Query</code> will time out after 60 seconds. You must update the default timeout in the SDK to
     * support a timeout of 60 seconds. See the <a
     * href="https://docs.aws.amazon.com/Timestream/latest/developerguide/code-samples.run-query.html">code sample</a>
     * for details.
     * </p>
     * <p>
     * Your query request will fail in the following cases:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If you submit a <code>Query</code> request with the same client token outside of the 5-minute idempotency window.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you submit a <code>Query</code> request with the same client token, but change other parameters, within the
     * 5-minute idempotency window.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the size of the row (including the query metadata) exceeds 1 MB, then the query will fail with the following
     * error message:
     * </p>
     * <p>
     * <code>Query aborted as max page response size has been exceeded by the output result row</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * If the IAM principal of the query initiator and the result reader are not the same and/or the query initiator and
     * the result reader do not have the same query string in the query requests, the query will fail with an
     * <code>Invalid pagination token</code> error.
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a variant of {@link #query(software.amazon.awssdk.services.timestreamquery.model.QueryRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.timestreamquery.paginators.QueryPublisher publisher = client.queryPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.timestreamquery.paginators.QueryPublisher publisher = client.queryPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.timestreamquery.model.QueryResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.timestreamquery.model.QueryResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxRows won't limit the number of results you get with the paginator.
     * It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #query(software.amazon.awssdk.services.timestreamquery.model.QueryRequest)} operation.</b>
     * </p>
     *
     * @param queryRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You are not authorized to perform this action.</li>
     *         <li>ConflictException Unable to poll results for a cancelled query.</li>
     *         <li>InternalServerException Timestream was unable to fully process this request because of an internal
     *         server error.</li>
     *         <li>QueryExecutionException Timestream was unable to run the query successfully.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException Invalid or malformed request.</li>
     *         <li>InvalidEndpointException The requested endpoint was not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TimestreamQueryException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample TimestreamQueryAsyncClient.Query
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-query-2018-11-01/Query" target="_top">AWS API
     *      Documentation</a>
     */
    public QueryPublisher queryPaginator(QueryRequest queryRequest) {
        return new QueryPublisher(this, applyPaginatorUserAgent(queryRequest));
    }

    /**
     * <p>
     * Associate a set of tags with a Timestream resource. You can then activate these user-defined tags so that they
     * appear on the Billing and Cost Management console for cost allocation tracking.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>ServiceQuotaExceededException You have exceeded the service quota.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException Invalid or malformed request.</li>
     *         <li>InvalidEndpointException The requested endpoint was not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TimestreamQueryException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample TimestreamQueryAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-query-2018-11-01/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Query");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
            boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
            if (endpointOverridden) {
                endpointDiscoveryEnabled = false;
            } else if (!endpointDiscoveryEnabled) {
                throw new IllegalStateException(
                        "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
            }
            URI cachedEndpoint = null;
            if (endpointDiscoveryEnabled) {
                String key = tagResourceRequest.overrideConfiguration()
                        .flatMap(AwsRequestOverrideConfiguration::credentialsProvider)
                        .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER)).resolveCredentials()
                        .accessKeyId();
                EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                        .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT))
                        .overrideConfiguration(tagResourceRequest.overrideConfiguration().orElse(null)).build();
                cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
            }

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).discoveredEndpoint(cachedEndpoint)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes the association of tags from a Timestream query resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException Invalid or malformed request.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>InvalidEndpointException The requested endpoint was not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TimestreamQueryException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample TimestreamQueryAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-query-2018-11-01/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Query");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
            boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
            if (endpointOverridden) {
                endpointDiscoveryEnabled = false;
            } else if (!endpointDiscoveryEnabled) {
                throw new IllegalStateException(
                        "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
            }
            URI cachedEndpoint = null;
            if (endpointDiscoveryEnabled) {
                String key = untagResourceRequest.overrideConfiguration()
                        .flatMap(AwsRequestOverrideConfiguration::credentialsProvider)
                        .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER)).resolveCredentials()
                        .accessKeyId();
                EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                        .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT))
                        .overrideConfiguration(untagResourceRequest.overrideConfiguration().orElse(null)).build();
                cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
            }

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).discoveredEndpoint(cachedEndpoint)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Update a scheduled query.
     * </p>
     *
     * @param updateScheduledQueryRequest
     * @return A Java Future containing the result of the UpdateScheduledQuery operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You are not authorized to perform this action.</li>
     *         <li>InternalServerException Timestream was unable to fully process this request because of an internal
     *         server error.</li>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException Invalid or malformed request.</li>
     *         <li>InvalidEndpointException The requested endpoint was not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TimestreamQueryException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample TimestreamQueryAsyncClient.UpdateScheduledQuery
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-query-2018-11-01/UpdateScheduledQuery"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateScheduledQueryResponse> updateScheduledQuery(
            UpdateScheduledQueryRequest updateScheduledQueryRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateScheduledQueryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Query");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateScheduledQuery");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateScheduledQueryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateScheduledQueryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
            boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
            if (endpointOverridden) {
                endpointDiscoveryEnabled = false;
            } else if (!endpointDiscoveryEnabled) {
                throw new IllegalStateException(
                        "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
            }
            URI cachedEndpoint = null;
            if (endpointDiscoveryEnabled) {
                String key = updateScheduledQueryRequest.overrideConfiguration()
                        .flatMap(AwsRequestOverrideConfiguration::credentialsProvider)
                        .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER)).resolveCredentials()
                        .accessKeyId();
                EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                        .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT))
                        .overrideConfiguration(updateScheduledQueryRequest.overrideConfiguration().orElse(null)).build();
                cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
            }

            CompletableFuture<UpdateScheduledQueryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateScheduledQueryRequest, UpdateScheduledQueryResponse>()
                            .withOperationName("UpdateScheduledQuery")
                            .withMarshaller(new UpdateScheduledQueryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).discoveredEndpoint(cachedEndpoint)
                            .withInput(updateScheduledQueryRequest));
            CompletableFuture<UpdateScheduledQueryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(TimestreamQueryException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.0")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidEndpointException")
                                .exceptionBuilderSupplier(InvalidEndpointException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("QueryExecutionException")
                                .exceptionBuilderSupplier(QueryExecutionException::builder).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends TimestreamQueryRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
