/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.timestreamquery;

import java.net.URI;
import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.config.AwsClientOption;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.endpointdiscovery.EndpointDiscoveryRefreshCache;
import software.amazon.awssdk.core.endpointdiscovery.EndpointDiscoveryRequest;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.timestreamquery.model.AccessDeniedException;
import software.amazon.awssdk.services.timestreamquery.model.CancelQueryRequest;
import software.amazon.awssdk.services.timestreamquery.model.CancelQueryResponse;
import software.amazon.awssdk.services.timestreamquery.model.ConflictException;
import software.amazon.awssdk.services.timestreamquery.model.CreateScheduledQueryRequest;
import software.amazon.awssdk.services.timestreamquery.model.CreateScheduledQueryResponse;
import software.amazon.awssdk.services.timestreamquery.model.DeleteScheduledQueryRequest;
import software.amazon.awssdk.services.timestreamquery.model.DeleteScheduledQueryResponse;
import software.amazon.awssdk.services.timestreamquery.model.DescribeEndpointsRequest;
import software.amazon.awssdk.services.timestreamquery.model.DescribeEndpointsResponse;
import software.amazon.awssdk.services.timestreamquery.model.DescribeScheduledQueryRequest;
import software.amazon.awssdk.services.timestreamquery.model.DescribeScheduledQueryResponse;
import software.amazon.awssdk.services.timestreamquery.model.ExecuteScheduledQueryRequest;
import software.amazon.awssdk.services.timestreamquery.model.ExecuteScheduledQueryResponse;
import software.amazon.awssdk.services.timestreamquery.model.InternalServerException;
import software.amazon.awssdk.services.timestreamquery.model.InvalidEndpointException;
import software.amazon.awssdk.services.timestreamquery.model.ListScheduledQueriesRequest;
import software.amazon.awssdk.services.timestreamquery.model.ListScheduledQueriesResponse;
import software.amazon.awssdk.services.timestreamquery.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.timestreamquery.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.timestreamquery.model.PrepareQueryRequest;
import software.amazon.awssdk.services.timestreamquery.model.PrepareQueryResponse;
import software.amazon.awssdk.services.timestreamquery.model.QueryExecutionException;
import software.amazon.awssdk.services.timestreamquery.model.QueryRequest;
import software.amazon.awssdk.services.timestreamquery.model.QueryResponse;
import software.amazon.awssdk.services.timestreamquery.model.ResourceNotFoundException;
import software.amazon.awssdk.services.timestreamquery.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.timestreamquery.model.TagResourceRequest;
import software.amazon.awssdk.services.timestreamquery.model.TagResourceResponse;
import software.amazon.awssdk.services.timestreamquery.model.ThrottlingException;
import software.amazon.awssdk.services.timestreamquery.model.TimestreamQueryException;
import software.amazon.awssdk.services.timestreamquery.model.TimestreamQueryRequest;
import software.amazon.awssdk.services.timestreamquery.model.UntagResourceRequest;
import software.amazon.awssdk.services.timestreamquery.model.UntagResourceResponse;
import software.amazon.awssdk.services.timestreamquery.model.UpdateScheduledQueryRequest;
import software.amazon.awssdk.services.timestreamquery.model.UpdateScheduledQueryResponse;
import software.amazon.awssdk.services.timestreamquery.model.ValidationException;
import software.amazon.awssdk.services.timestreamquery.paginators.ListScheduledQueriesIterable;
import software.amazon.awssdk.services.timestreamquery.paginators.ListTagsForResourceIterable;
import software.amazon.awssdk.services.timestreamquery.paginators.QueryIterable;
import software.amazon.awssdk.services.timestreamquery.transform.CancelQueryRequestMarshaller;
import software.amazon.awssdk.services.timestreamquery.transform.CreateScheduledQueryRequestMarshaller;
import software.amazon.awssdk.services.timestreamquery.transform.DeleteScheduledQueryRequestMarshaller;
import software.amazon.awssdk.services.timestreamquery.transform.DescribeEndpointsRequestMarshaller;
import software.amazon.awssdk.services.timestreamquery.transform.DescribeScheduledQueryRequestMarshaller;
import software.amazon.awssdk.services.timestreamquery.transform.ExecuteScheduledQueryRequestMarshaller;
import software.amazon.awssdk.services.timestreamquery.transform.ListScheduledQueriesRequestMarshaller;
import software.amazon.awssdk.services.timestreamquery.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.timestreamquery.transform.PrepareQueryRequestMarshaller;
import software.amazon.awssdk.services.timestreamquery.transform.QueryRequestMarshaller;
import software.amazon.awssdk.services.timestreamquery.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.timestreamquery.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.timestreamquery.transform.UpdateScheduledQueryRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link TimestreamQueryClient}.
 *
 * @see TimestreamQueryClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultTimestreamQueryClient implements TimestreamQueryClient {
    private static final Logger log = Logger.loggerFor(DefaultTimestreamQueryClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private EndpointDiscoveryRefreshCache endpointDiscoveryCache;

    protected DefaultTimestreamQueryClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
        if (clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED)) {
            this.endpointDiscoveryCache = EndpointDiscoveryRefreshCache.create(TimestreamQueryEndpointDiscoveryCacheLoader
                    .create(this));
            if (clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE) {
                log.warn(() -> "Endpoint discovery is enabled for this client, and an endpoint override was also specified. This will disable endpoint discovery for methods that require it, instead using the specified endpoint override. This may or may not be what you intended.");
            }
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Cancels a query that has been issued. Cancellation is provided only if the query has not completed running before
     * the cancellation request was issued. Because cancellation is an idempotent operation, subsequent cancellation
     * requests will return a <code>CancellationMessage</code>, indicating that the query has already been canceled. See
     * <a href="https://docs.aws.amazon.com/Timestream/latest/developerguide/code-samples.cancel-query.html">code
     * sample</a> for details.
     * </p>
     *
     * @param cancelQueryRequest
     * @return Result of the CancelQuery operation returned by the service.
     * @throws AccessDeniedException
     *         You are not authorized to perform this action.
     * @throws InternalServerException
     *         Timestream was unable to fully process this request because of an internal server error.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         Invalid or malformed request.
     * @throws InvalidEndpointException
     *         The requested endpoint was not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TimestreamQueryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TimestreamQueryClient.CancelQuery
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-query-2018-11-01/CancelQuery" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CancelQueryResponse cancelQuery(CancelQueryRequest cancelQueryRequest) throws AccessDeniedException,
            InternalServerException, ThrottlingException, ValidationException, InvalidEndpointException, AwsServiceException,
            SdkClientException, TimestreamQueryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CancelQueryResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CancelQueryResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
        if (endpointOverridden) {
            endpointDiscoveryEnabled = false;
        } else if (!endpointDiscoveryEnabled) {
            throw new IllegalStateException(
                    "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
        }
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            String key = cancelQueryRequest.overrideConfiguration().flatMap(AwsRequestOverrideConfiguration::credentialsProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER)).resolveCredentials()
                    .accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT))
                    .overrideConfiguration(cancelQueryRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelQueryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Query");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelQuery");

            return clientHandler.execute(new ClientExecutionParams<CancelQueryRequest, CancelQueryResponse>()
                    .withOperationName("CancelQuery").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).discoveredEndpoint(cachedEndpoint)
                    .withInput(cancelQueryRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CancelQueryRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Create a scheduled query that will be run on your behalf at the configured schedule. Timestream assumes the
     * execution role provided as part of the <code>ScheduledQueryExecutionRoleArn</code> parameter to run the query.
     * You can use the <code>NotificationConfiguration</code> parameter to configure notification for your scheduled
     * query operations.
     * </p>
     *
     * @param createScheduledQueryRequest
     * @return Result of the CreateScheduledQuery operation returned by the service.
     * @throws AccessDeniedException
     *         You are not authorized to perform this action.
     * @throws ConflictException
     *         Unable to poll results for a cancelled query.
     * @throws InternalServerException
     *         Timestream was unable to fully process this request because of an internal server error.
     * @throws ServiceQuotaExceededException
     *         You have exceeded the service quota.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         Invalid or malformed request.
     * @throws InvalidEndpointException
     *         The requested endpoint was not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TimestreamQueryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TimestreamQueryClient.CreateScheduledQuery
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-query-2018-11-01/CreateScheduledQuery"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateScheduledQueryResponse createScheduledQuery(CreateScheduledQueryRequest createScheduledQueryRequest)
            throws AccessDeniedException, ConflictException, InternalServerException, ServiceQuotaExceededException,
            ThrottlingException, ValidationException, InvalidEndpointException, AwsServiceException, SdkClientException,
            TimestreamQueryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateScheduledQueryResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateScheduledQueryResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
        if (endpointOverridden) {
            endpointDiscoveryEnabled = false;
        } else if (!endpointDiscoveryEnabled) {
            throw new IllegalStateException(
                    "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
        }
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            String key = createScheduledQueryRequest.overrideConfiguration()
                    .flatMap(AwsRequestOverrideConfiguration::credentialsProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER)).resolveCredentials()
                    .accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT))
                    .overrideConfiguration(createScheduledQueryRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createScheduledQueryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Query");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateScheduledQuery");

            return clientHandler.execute(new ClientExecutionParams<CreateScheduledQueryRequest, CreateScheduledQueryResponse>()
                    .withOperationName("CreateScheduledQuery").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).discoveredEndpoint(cachedEndpoint)
                    .withInput(createScheduledQueryRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateScheduledQueryRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a given scheduled query. This is an irreversible operation.
     * </p>
     *
     * @param deleteScheduledQueryRequest
     * @return Result of the DeleteScheduledQuery operation returned by the service.
     * @throws AccessDeniedException
     *         You are not authorized to perform this action.
     * @throws InternalServerException
     *         Timestream was unable to fully process this request because of an internal server error.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         Invalid or malformed request.
     * @throws InvalidEndpointException
     *         The requested endpoint was not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TimestreamQueryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TimestreamQueryClient.DeleteScheduledQuery
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-query-2018-11-01/DeleteScheduledQuery"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteScheduledQueryResponse deleteScheduledQuery(DeleteScheduledQueryRequest deleteScheduledQueryRequest)
            throws AccessDeniedException, InternalServerException, ResourceNotFoundException, ThrottlingException,
            ValidationException, InvalidEndpointException, AwsServiceException, SdkClientException, TimestreamQueryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteScheduledQueryResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteScheduledQueryResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
        if (endpointOverridden) {
            endpointDiscoveryEnabled = false;
        } else if (!endpointDiscoveryEnabled) {
            throw new IllegalStateException(
                    "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
        }
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            String key = deleteScheduledQueryRequest.overrideConfiguration()
                    .flatMap(AwsRequestOverrideConfiguration::credentialsProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER)).resolveCredentials()
                    .accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT))
                    .overrideConfiguration(deleteScheduledQueryRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteScheduledQueryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Query");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteScheduledQuery");

            return clientHandler.execute(new ClientExecutionParams<DeleteScheduledQueryRequest, DeleteScheduledQueryResponse>()
                    .withOperationName("DeleteScheduledQuery").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).discoveredEndpoint(cachedEndpoint)
                    .withInput(deleteScheduledQueryRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteScheduledQueryRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * DescribeEndpoints returns a list of available endpoints to make Timestream API calls against. This API is
     * available through both Write and Query.
     * </p>
     * <p>
     * Because the Timestream SDKs are designed to transparently work with the service’s architecture, including the
     * management and mapping of the service endpoints, <i>it is not recommended that you use this API unless</i>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * You are using <a href="https://docs.aws.amazon.com/Timestream/latest/developerguide/VPCEndpoints">VPC endpoints
     * (Amazon Web Services PrivateLink) with Timestream </a>
     * </p>
     * </li>
     * <li>
     * <p>
     * Your application uses a programming language that does not yet have SDK support
     * </p>
     * </li>
     * <li>
     * <p>
     * You require better control over the client-side implementation
     * </p>
     * </li>
     * </ul>
     * <p>
     * For detailed information on how and when to use and implement DescribeEndpoints, see <a
     * href="https://docs.aws.amazon.com/Timestream/latest/developerguide/Using.API.html#Using-API.endpoint-discovery"
     * >The Endpoint Discovery Pattern</a>.
     * </p>
     *
     * @param describeEndpointsRequest
     * @return Result of the DescribeEndpoints operation returned by the service.
     * @throws InternalServerException
     *         Timestream was unable to fully process this request because of an internal server error.
     * @throws ValidationException
     *         Invalid or malformed request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TimestreamQueryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TimestreamQueryClient.DescribeEndpoints
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-query-2018-11-01/DescribeEndpoints"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeEndpointsResponse describeEndpoints(DescribeEndpointsRequest describeEndpointsRequest)
            throws InternalServerException, ValidationException, ThrottlingException, AwsServiceException, SdkClientException,
            TimestreamQueryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeEndpointsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeEndpointsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeEndpointsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Query");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeEndpoints");

            return clientHandler.execute(new ClientExecutionParams<DescribeEndpointsRequest, DescribeEndpointsResponse>()
                    .withOperationName("DescribeEndpoints").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeEndpointsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeEndpointsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides detailed information about a scheduled query.
     * </p>
     *
     * @param describeScheduledQueryRequest
     * @return Result of the DescribeScheduledQuery operation returned by the service.
     * @throws AccessDeniedException
     *         You are not authorized to perform this action.
     * @throws InternalServerException
     *         Timestream was unable to fully process this request because of an internal server error.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         Invalid or malformed request.
     * @throws InvalidEndpointException
     *         The requested endpoint was not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TimestreamQueryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TimestreamQueryClient.DescribeScheduledQuery
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-query-2018-11-01/DescribeScheduledQuery"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeScheduledQueryResponse describeScheduledQuery(DescribeScheduledQueryRequest describeScheduledQueryRequest)
            throws AccessDeniedException, InternalServerException, ResourceNotFoundException, ThrottlingException,
            ValidationException, InvalidEndpointException, AwsServiceException, SdkClientException, TimestreamQueryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeScheduledQueryResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeScheduledQueryResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
        if (endpointOverridden) {
            endpointDiscoveryEnabled = false;
        } else if (!endpointDiscoveryEnabled) {
            throw new IllegalStateException(
                    "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
        }
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            String key = describeScheduledQueryRequest.overrideConfiguration()
                    .flatMap(AwsRequestOverrideConfiguration::credentialsProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER)).resolveCredentials()
                    .accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT))
                    .overrideConfiguration(describeScheduledQueryRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeScheduledQueryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Query");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeScheduledQuery");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeScheduledQueryRequest, DescribeScheduledQueryResponse>()
                            .withOperationName("DescribeScheduledQuery").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).discoveredEndpoint(cachedEndpoint)
                            .withInput(describeScheduledQueryRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeScheduledQueryRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * You can use this API to run a scheduled query manually.
     * </p>
     *
     * @param executeScheduledQueryRequest
     * @return Result of the ExecuteScheduledQuery operation returned by the service.
     * @throws AccessDeniedException
     *         You are not authorized to perform this action.
     * @throws InternalServerException
     *         Timestream was unable to fully process this request because of an internal server error.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         Invalid or malformed request.
     * @throws InvalidEndpointException
     *         The requested endpoint was not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TimestreamQueryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TimestreamQueryClient.ExecuteScheduledQuery
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-query-2018-11-01/ExecuteScheduledQuery"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ExecuteScheduledQueryResponse executeScheduledQuery(ExecuteScheduledQueryRequest executeScheduledQueryRequest)
            throws AccessDeniedException, InternalServerException, ResourceNotFoundException, ThrottlingException,
            ValidationException, InvalidEndpointException, AwsServiceException, SdkClientException, TimestreamQueryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ExecuteScheduledQueryResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ExecuteScheduledQueryResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
        if (endpointOverridden) {
            endpointDiscoveryEnabled = false;
        } else if (!endpointDiscoveryEnabled) {
            throw new IllegalStateException(
                    "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
        }
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            String key = executeScheduledQueryRequest.overrideConfiguration()
                    .flatMap(AwsRequestOverrideConfiguration::credentialsProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER)).resolveCredentials()
                    .accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT))
                    .overrideConfiguration(executeScheduledQueryRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, executeScheduledQueryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Query");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ExecuteScheduledQuery");

            return clientHandler.execute(new ClientExecutionParams<ExecuteScheduledQueryRequest, ExecuteScheduledQueryResponse>()
                    .withOperationName("ExecuteScheduledQuery").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).discoveredEndpoint(cachedEndpoint)
                    .withInput(executeScheduledQueryRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ExecuteScheduledQueryRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets a list of all scheduled queries in the caller's Amazon account and Region. <code>ListScheduledQueries</code>
     * is eventually consistent.
     * </p>
     *
     * @param listScheduledQueriesRequest
     * @return Result of the ListScheduledQueries operation returned by the service.
     * @throws AccessDeniedException
     *         You are not authorized to perform this action.
     * @throws InternalServerException
     *         Timestream was unable to fully process this request because of an internal server error.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         Invalid or malformed request.
     * @throws InvalidEndpointException
     *         The requested endpoint was not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TimestreamQueryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TimestreamQueryClient.ListScheduledQueries
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-query-2018-11-01/ListScheduledQueries"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListScheduledQueriesResponse listScheduledQueries(ListScheduledQueriesRequest listScheduledQueriesRequest)
            throws AccessDeniedException, InternalServerException, ThrottlingException, ValidationException,
            InvalidEndpointException, AwsServiceException, SdkClientException, TimestreamQueryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListScheduledQueriesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListScheduledQueriesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
        if (endpointOverridden) {
            endpointDiscoveryEnabled = false;
        } else if (!endpointDiscoveryEnabled) {
            throw new IllegalStateException(
                    "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
        }
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            String key = listScheduledQueriesRequest.overrideConfiguration()
                    .flatMap(AwsRequestOverrideConfiguration::credentialsProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER)).resolveCredentials()
                    .accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT))
                    .overrideConfiguration(listScheduledQueriesRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listScheduledQueriesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Query");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListScheduledQueries");

            return clientHandler.execute(new ClientExecutionParams<ListScheduledQueriesRequest, ListScheduledQueriesResponse>()
                    .withOperationName("ListScheduledQueries").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).discoveredEndpoint(cachedEndpoint)
                    .withInput(listScheduledQueriesRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListScheduledQueriesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets a list of all scheduled queries in the caller's Amazon account and Region. <code>ListScheduledQueries</code>
     * is eventually consistent.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listScheduledQueries(software.amazon.awssdk.services.timestreamquery.model.ListScheduledQueriesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.timestreamquery.paginators.ListScheduledQueriesIterable responses = client.listScheduledQueriesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.timestreamquery.paginators.ListScheduledQueriesIterable responses = client
     *             .listScheduledQueriesPaginator(request);
     *     for (software.amazon.awssdk.services.timestreamquery.model.ListScheduledQueriesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.timestreamquery.paginators.ListScheduledQueriesIterable responses = client.listScheduledQueriesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listScheduledQueries(software.amazon.awssdk.services.timestreamquery.model.ListScheduledQueriesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listScheduledQueriesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws AccessDeniedException
     *         You are not authorized to perform this action.
     * @throws InternalServerException
     *         Timestream was unable to fully process this request because of an internal server error.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         Invalid or malformed request.
     * @throws InvalidEndpointException
     *         The requested endpoint was not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TimestreamQueryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TimestreamQueryClient.ListScheduledQueries
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-query-2018-11-01/ListScheduledQueries"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListScheduledQueriesIterable listScheduledQueriesPaginator(ListScheduledQueriesRequest listScheduledQueriesRequest)
            throws AccessDeniedException, InternalServerException, ThrottlingException, ValidationException,
            InvalidEndpointException, AwsServiceException, SdkClientException, TimestreamQueryException {
        return new ListScheduledQueriesIterable(this, applyPaginatorUserAgent(listScheduledQueriesRequest));
    }

    /**
     * <p>
     * List all tags on a Timestream query resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         Invalid or malformed request.
     * @throws InvalidEndpointException
     *         The requested endpoint was not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TimestreamQueryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TimestreamQueryClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-query-2018-11-01/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ResourceNotFoundException, ThrottlingException, ValidationException, InvalidEndpointException,
            AwsServiceException, SdkClientException, TimestreamQueryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
        if (endpointOverridden) {
            endpointDiscoveryEnabled = false;
        } else if (!endpointDiscoveryEnabled) {
            throw new IllegalStateException(
                    "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
        }
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            String key = listTagsForResourceRequest.overrideConfiguration()
                    .flatMap(AwsRequestOverrideConfiguration::credentialsProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER)).resolveCredentials()
                    .accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT))
                    .overrideConfiguration(listTagsForResourceRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Query");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).discoveredEndpoint(cachedEndpoint)
                    .withInput(listTagsForResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * List all tags on a Timestream query resource.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listTagsForResource(software.amazon.awssdk.services.timestreamquery.model.ListTagsForResourceRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.timestreamquery.paginators.ListTagsForResourceIterable responses = client.listTagsForResourcePaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.timestreamquery.paginators.ListTagsForResourceIterable responses = client
     *             .listTagsForResourcePaginator(request);
     *     for (software.amazon.awssdk.services.timestreamquery.model.ListTagsForResourceResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.timestreamquery.paginators.ListTagsForResourceIterable responses = client.listTagsForResourcePaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listTagsForResource(software.amazon.awssdk.services.timestreamquery.model.ListTagsForResourceRequest)}
     * operation.</b>
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         Invalid or malformed request.
     * @throws InvalidEndpointException
     *         The requested endpoint was not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TimestreamQueryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TimestreamQueryClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-query-2018-11-01/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceIterable listTagsForResourcePaginator(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ResourceNotFoundException, ThrottlingException, ValidationException, InvalidEndpointException,
            AwsServiceException, SdkClientException, TimestreamQueryException {
        return new ListTagsForResourceIterable(this, applyPaginatorUserAgent(listTagsForResourceRequest));
    }

    /**
     * <p>
     * A synchronous operation that allows you to submit a query with parameters to be stored by Timestream for later
     * running. Timestream only supports using this operation with the <code>PrepareQueryRequest&#36ValidateOnly</code>
     * set to <code>true</code>.
     * </p>
     *
     * @param prepareQueryRequest
     * @return Result of the PrepareQuery operation returned by the service.
     * @throws AccessDeniedException
     *         You are not authorized to perform this action.
     * @throws InternalServerException
     *         Timestream was unable to fully process this request because of an internal server error.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         Invalid or malformed request.
     * @throws InvalidEndpointException
     *         The requested endpoint was not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TimestreamQueryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TimestreamQueryClient.PrepareQuery
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-query-2018-11-01/PrepareQuery" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public PrepareQueryResponse prepareQuery(PrepareQueryRequest prepareQueryRequest) throws AccessDeniedException,
            InternalServerException, ThrottlingException, ValidationException, InvalidEndpointException, AwsServiceException,
            SdkClientException, TimestreamQueryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PrepareQueryResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PrepareQueryResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
        if (endpointOverridden) {
            endpointDiscoveryEnabled = false;
        } else if (!endpointDiscoveryEnabled) {
            throw new IllegalStateException(
                    "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
        }
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            String key = prepareQueryRequest.overrideConfiguration()
                    .flatMap(AwsRequestOverrideConfiguration::credentialsProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER)).resolveCredentials()
                    .accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT))
                    .overrideConfiguration(prepareQueryRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, prepareQueryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Query");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PrepareQuery");

            return clientHandler.execute(new ClientExecutionParams<PrepareQueryRequest, PrepareQueryResponse>()
                    .withOperationName("PrepareQuery").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).discoveredEndpoint(cachedEndpoint)
                    .withInput(prepareQueryRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PrepareQueryRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * <code>Query</code> is a synchronous operation that enables you to run a query against your Amazon Timestream
     * data. <code>Query</code> will time out after 60 seconds. You must update the default timeout in the SDK to
     * support a timeout of 60 seconds. See the <a
     * href="https://docs.aws.amazon.com/Timestream/latest/developerguide/code-samples.run-query.html">code sample</a>
     * for details.
     * </p>
     * <p>
     * Your query request will fail in the following cases:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If you submit a <code>Query</code> request with the same client token outside of the 5-minute idempotency window.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you submit a <code>Query</code> request with the same client token, but change other parameters, within the
     * 5-minute idempotency window.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the size of the row (including the query metadata) exceeds 1 MB, then the query will fail with the following
     * error message:
     * </p>
     * <p>
     * <code>Query aborted as max page response size has been exceeded by the output result row</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * If the IAM principal of the query initiator and the result reader are not the same and/or the query initiator and
     * the result reader do not have the same query string in the query requests, the query will fail with an
     * <code>Invalid pagination token</code> error.
     * </p>
     * </li>
     * </ul>
     *
     * @param queryRequest
     * @return Result of the Query operation returned by the service.
     * @throws AccessDeniedException
     *         You are not authorized to perform this action.
     * @throws ConflictException
     *         Unable to poll results for a cancelled query.
     * @throws InternalServerException
     *         Timestream was unable to fully process this request because of an internal server error.
     * @throws QueryExecutionException
     *         Timestream was unable to run the query successfully.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         Invalid or malformed request.
     * @throws InvalidEndpointException
     *         The requested endpoint was not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TimestreamQueryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TimestreamQueryClient.Query
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-query-2018-11-01/Query" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public QueryResponse query(QueryRequest queryRequest) throws AccessDeniedException, ConflictException,
            InternalServerException, QueryExecutionException, ThrottlingException, ValidationException, InvalidEndpointException,
            AwsServiceException, SdkClientException, TimestreamQueryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<QueryResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                QueryResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
        if (endpointOverridden) {
            endpointDiscoveryEnabled = false;
        } else if (!endpointDiscoveryEnabled) {
            throw new IllegalStateException(
                    "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
        }
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            String key = queryRequest.overrideConfiguration().flatMap(AwsRequestOverrideConfiguration::credentialsProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER)).resolveCredentials()
                    .accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT))
                    .overrideConfiguration(queryRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, queryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Query");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "Query");

            return clientHandler.execute(new ClientExecutionParams<QueryRequest, QueryResponse>().withOperationName("Query")
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .discoveredEndpoint(cachedEndpoint).withInput(queryRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new QueryRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * <code>Query</code> is a synchronous operation that enables you to run a query against your Amazon Timestream
     * data. <code>Query</code> will time out after 60 seconds. You must update the default timeout in the SDK to
     * support a timeout of 60 seconds. See the <a
     * href="https://docs.aws.amazon.com/Timestream/latest/developerguide/code-samples.run-query.html">code sample</a>
     * for details.
     * </p>
     * <p>
     * Your query request will fail in the following cases:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If you submit a <code>Query</code> request with the same client token outside of the 5-minute idempotency window.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you submit a <code>Query</code> request with the same client token, but change other parameters, within the
     * 5-minute idempotency window.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the size of the row (including the query metadata) exceeds 1 MB, then the query will fail with the following
     * error message:
     * </p>
     * <p>
     * <code>Query aborted as max page response size has been exceeded by the output result row</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * If the IAM principal of the query initiator and the result reader are not the same and/or the query initiator and
     * the result reader do not have the same query string in the query requests, the query will fail with an
     * <code>Invalid pagination token</code> error.
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a variant of {@link #query(software.amazon.awssdk.services.timestreamquery.model.QueryRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.timestreamquery.paginators.QueryIterable responses = client.queryPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.timestreamquery.paginators.QueryIterable responses = client.queryPaginator(request);
     *     for (software.amazon.awssdk.services.timestreamquery.model.QueryResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.timestreamquery.paginators.QueryIterable responses = client.queryPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxRows won't limit the number of results you get with the paginator.
     * It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #query(software.amazon.awssdk.services.timestreamquery.model.QueryRequest)} operation.</b>
     * </p>
     *
     * @param queryRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws AccessDeniedException
     *         You are not authorized to perform this action.
     * @throws ConflictException
     *         Unable to poll results for a cancelled query.
     * @throws InternalServerException
     *         Timestream was unable to fully process this request because of an internal server error.
     * @throws QueryExecutionException
     *         Timestream was unable to run the query successfully.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         Invalid or malformed request.
     * @throws InvalidEndpointException
     *         The requested endpoint was not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TimestreamQueryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TimestreamQueryClient.Query
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-query-2018-11-01/Query" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public QueryIterable queryPaginator(QueryRequest queryRequest) throws AccessDeniedException, ConflictException,
            InternalServerException, QueryExecutionException, ThrottlingException, ValidationException, InvalidEndpointException,
            AwsServiceException, SdkClientException, TimestreamQueryException {
        return new QueryIterable(this, applyPaginatorUserAgent(queryRequest));
    }

    /**
     * <p>
     * Associate a set of tags with a Timestream resource. You can then activate these user-defined tags so that they
     * appear on the Billing and Cost Management console for cost allocation tracking.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws ServiceQuotaExceededException
     *         You have exceeded the service quota.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         Invalid or malformed request.
     * @throws InvalidEndpointException
     *         The requested endpoint was not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TimestreamQueryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TimestreamQueryClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-query-2018-11-01/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ResourceNotFoundException,
            ServiceQuotaExceededException, ThrottlingException, ValidationException, InvalidEndpointException,
            AwsServiceException, SdkClientException, TimestreamQueryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
        if (endpointOverridden) {
            endpointDiscoveryEnabled = false;
        } else if (!endpointDiscoveryEnabled) {
            throw new IllegalStateException(
                    "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
        }
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            String key = tagResourceRequest.overrideConfiguration().flatMap(AwsRequestOverrideConfiguration::credentialsProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER)).resolveCredentials()
                    .accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT))
                    .overrideConfiguration(tagResourceRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Query");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).discoveredEndpoint(cachedEndpoint)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes the association of tags from a Timestream query resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ValidationException
     *         Invalid or malformed request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws InvalidEndpointException
     *         The requested endpoint was not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TimestreamQueryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TimestreamQueryClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-query-2018-11-01/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ValidationException,
            ThrottlingException, ResourceNotFoundException, InvalidEndpointException, AwsServiceException, SdkClientException,
            TimestreamQueryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
        if (endpointOverridden) {
            endpointDiscoveryEnabled = false;
        } else if (!endpointDiscoveryEnabled) {
            throw new IllegalStateException(
                    "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
        }
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            String key = untagResourceRequest.overrideConfiguration()
                    .flatMap(AwsRequestOverrideConfiguration::credentialsProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER)).resolveCredentials()
                    .accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT))
                    .overrideConfiguration(untagResourceRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Query");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).discoveredEndpoint(cachedEndpoint)
                    .withInput(untagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Update a scheduled query.
     * </p>
     *
     * @param updateScheduledQueryRequest
     * @return Result of the UpdateScheduledQuery operation returned by the service.
     * @throws AccessDeniedException
     *         You are not authorized to perform this action.
     * @throws InternalServerException
     *         Timestream was unable to fully process this request because of an internal server error.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         Invalid or malformed request.
     * @throws InvalidEndpointException
     *         The requested endpoint was not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TimestreamQueryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TimestreamQueryClient.UpdateScheduledQuery
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-query-2018-11-01/UpdateScheduledQuery"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateScheduledQueryResponse updateScheduledQuery(UpdateScheduledQueryRequest updateScheduledQueryRequest)
            throws AccessDeniedException, InternalServerException, ResourceNotFoundException, ThrottlingException,
            ValidationException, InvalidEndpointException, AwsServiceException, SdkClientException, TimestreamQueryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateScheduledQueryResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateScheduledQueryResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
        if (endpointOverridden) {
            endpointDiscoveryEnabled = false;
        } else if (!endpointDiscoveryEnabled) {
            throw new IllegalStateException(
                    "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
        }
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            String key = updateScheduledQueryRequest.overrideConfiguration()
                    .flatMap(AwsRequestOverrideConfiguration::credentialsProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER)).resolveCredentials()
                    .accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT))
                    .overrideConfiguration(updateScheduledQueryRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateScheduledQueryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Query");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateScheduledQuery");

            return clientHandler.execute(new ClientExecutionParams<UpdateScheduledQueryRequest, UpdateScheduledQueryResponse>()
                    .withOperationName("UpdateScheduledQuery").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).discoveredEndpoint(cachedEndpoint)
                    .withInput(updateScheduledQueryRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateScheduledQueryRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(TimestreamQueryException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.0")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidEndpointException")
                                .exceptionBuilderSupplier(InvalidEndpointException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("QueryExecutionException")
                                .exceptionBuilderSupplier(QueryExecutionException::builder).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends TimestreamQueryRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
