/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.timestreamquery.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Details on S3 location for error reports that result from running a query.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class S3Configuration implements SdkPojo, Serializable, ToCopyableBuilder<S3Configuration.Builder, S3Configuration> {
    private static final SdkField<String> BUCKET_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("BucketName").getter(getter(S3Configuration::bucketName)).setter(setter(Builder::bucketName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BucketName").build()).build();

    private static final SdkField<String> OBJECT_KEY_PREFIX_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ObjectKeyPrefix").getter(getter(S3Configuration::objectKeyPrefix))
            .setter(setter(Builder::objectKeyPrefix))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ObjectKeyPrefix").build()).build();

    private static final SdkField<String> ENCRYPTION_OPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EncryptionOption").getter(getter(S3Configuration::encryptionOptionAsString))
            .setter(setter(Builder::encryptionOption))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EncryptionOption").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(BUCKET_NAME_FIELD,
            OBJECT_KEY_PREFIX_FIELD, ENCRYPTION_OPTION_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String bucketName;

    private final String objectKeyPrefix;

    private final String encryptionOption;

    private S3Configuration(BuilderImpl builder) {
        this.bucketName = builder.bucketName;
        this.objectKeyPrefix = builder.objectKeyPrefix;
        this.encryptionOption = builder.encryptionOption;
    }

    /**
     * <p>
     * Name of the S3 bucket under which error reports will be created.
     * </p>
     * 
     * @return Name of the S3 bucket under which error reports will be created.
     */
    public final String bucketName() {
        return bucketName;
    }

    /**
     * <p>
     * Prefix for the error report key. Timestream by default adds the following prefix to the error report path.
     * </p>
     * 
     * @return Prefix for the error report key. Timestream by default adds the following prefix to the error report
     *         path.
     */
    public final String objectKeyPrefix() {
        return objectKeyPrefix;
    }

    /**
     * <p>
     * Encryption at rest options for the error reports. If no encryption option is specified, Timestream will choose
     * SSE_S3 as default.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #encryptionOption}
     * will return {@link S3EncryptionOption#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #encryptionOptionAsString}.
     * </p>
     * 
     * @return Encryption at rest options for the error reports. If no encryption option is specified, Timestream will
     *         choose SSE_S3 as default.
     * @see S3EncryptionOption
     */
    public final S3EncryptionOption encryptionOption() {
        return S3EncryptionOption.fromValue(encryptionOption);
    }

    /**
     * <p>
     * Encryption at rest options for the error reports. If no encryption option is specified, Timestream will choose
     * SSE_S3 as default.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #encryptionOption}
     * will return {@link S3EncryptionOption#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #encryptionOptionAsString}.
     * </p>
     * 
     * @return Encryption at rest options for the error reports. If no encryption option is specified, Timestream will
     *         choose SSE_S3 as default.
     * @see S3EncryptionOption
     */
    public final String encryptionOptionAsString() {
        return encryptionOption;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(bucketName());
        hashCode = 31 * hashCode + Objects.hashCode(objectKeyPrefix());
        hashCode = 31 * hashCode + Objects.hashCode(encryptionOptionAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof S3Configuration)) {
            return false;
        }
        S3Configuration other = (S3Configuration) obj;
        return Objects.equals(bucketName(), other.bucketName()) && Objects.equals(objectKeyPrefix(), other.objectKeyPrefix())
                && Objects.equals(encryptionOptionAsString(), other.encryptionOptionAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("S3Configuration").add("BucketName", bucketName()).add("ObjectKeyPrefix", objectKeyPrefix())
                .add("EncryptionOption", encryptionOptionAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "BucketName":
            return Optional.ofNullable(clazz.cast(bucketName()));
        case "ObjectKeyPrefix":
            return Optional.ofNullable(clazz.cast(objectKeyPrefix()));
        case "EncryptionOption":
            return Optional.ofNullable(clazz.cast(encryptionOptionAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("BucketName", BUCKET_NAME_FIELD);
        map.put("ObjectKeyPrefix", OBJECT_KEY_PREFIX_FIELD);
        map.put("EncryptionOption", ENCRYPTION_OPTION_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<S3Configuration, T> g) {
        return obj -> g.apply((S3Configuration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, S3Configuration> {
        /**
         * <p>
         * Name of the S3 bucket under which error reports will be created.
         * </p>
         * 
         * @param bucketName
         *        Name of the S3 bucket under which error reports will be created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bucketName(String bucketName);

        /**
         * <p>
         * Prefix for the error report key. Timestream by default adds the following prefix to the error report path.
         * </p>
         * 
         * @param objectKeyPrefix
         *        Prefix for the error report key. Timestream by default adds the following prefix to the error report
         *        path.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder objectKeyPrefix(String objectKeyPrefix);

        /**
         * <p>
         * Encryption at rest options for the error reports. If no encryption option is specified, Timestream will
         * choose SSE_S3 as default.
         * </p>
         * 
         * @param encryptionOption
         *        Encryption at rest options for the error reports. If no encryption option is specified, Timestream
         *        will choose SSE_S3 as default.
         * @see S3EncryptionOption
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see S3EncryptionOption
         */
        Builder encryptionOption(String encryptionOption);

        /**
         * <p>
         * Encryption at rest options for the error reports. If no encryption option is specified, Timestream will
         * choose SSE_S3 as default.
         * </p>
         * 
         * @param encryptionOption
         *        Encryption at rest options for the error reports. If no encryption option is specified, Timestream
         *        will choose SSE_S3 as default.
         * @see S3EncryptionOption
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see S3EncryptionOption
         */
        Builder encryptionOption(S3EncryptionOption encryptionOption);
    }

    static final class BuilderImpl implements Builder {
        private String bucketName;

        private String objectKeyPrefix;

        private String encryptionOption;

        private BuilderImpl() {
        }

        private BuilderImpl(S3Configuration model) {
            bucketName(model.bucketName);
            objectKeyPrefix(model.objectKeyPrefix);
            encryptionOption(model.encryptionOption);
        }

        public final String getBucketName() {
            return bucketName;
        }

        public final void setBucketName(String bucketName) {
            this.bucketName = bucketName;
        }

        @Override
        public final Builder bucketName(String bucketName) {
            this.bucketName = bucketName;
            return this;
        }

        public final String getObjectKeyPrefix() {
            return objectKeyPrefix;
        }

        public final void setObjectKeyPrefix(String objectKeyPrefix) {
            this.objectKeyPrefix = objectKeyPrefix;
        }

        @Override
        public final Builder objectKeyPrefix(String objectKeyPrefix) {
            this.objectKeyPrefix = objectKeyPrefix;
            return this;
        }

        public final String getEncryptionOption() {
            return encryptionOption;
        }

        public final void setEncryptionOption(String encryptionOption) {
            this.encryptionOption = encryptionOption;
        }

        @Override
        public final Builder encryptionOption(String encryptionOption) {
            this.encryptionOption = encryptionOption;
            return this;
        }

        @Override
        public final Builder encryptionOption(S3EncryptionOption encryptionOption) {
            this.encryptionOption(encryptionOption == null ? null : encryptionOption.toString());
            return this;
        }

        @Override
        public S3Configuration build() {
            return new S3Configuration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
