/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.timestreamwrite.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Record represents a time series data point being written into Timestream. Each record contains an array of
 * dimensions. Dimensions represent the meta data attributes of a time series data point such as the instance name or
 * availability zone of an EC2 instance. A record also contains the measure name which is the name of the measure being
 * collected for example the CPU utilization of an EC2 instance. A record also contains the measure value and the value
 * type which is the data type of the measure value. In addition, the record contains the timestamp when the measure was
 * collected that the timestamp unit which represents the granularity of the timestamp.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Record implements SdkPojo, Serializable, ToCopyableBuilder<Record.Builder, Record> {
    private static final SdkField<List<Dimension>> DIMENSIONS_FIELD = SdkField
            .<List<Dimension>> builder(MarshallingType.LIST)
            .memberName("Dimensions")
            .getter(getter(Record::dimensions))
            .setter(setter(Builder::dimensions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Dimensions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Dimension> builder(MarshallingType.SDK_POJO)
                                            .constructor(Dimension::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> MEASURE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MeasureName").getter(getter(Record::measureName)).setter(setter(Builder::measureName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MeasureName").build()).build();

    private static final SdkField<String> MEASURE_VALUE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MeasureValue").getter(getter(Record::measureValue)).setter(setter(Builder::measureValue))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MeasureValue").build()).build();

    private static final SdkField<String> MEASURE_VALUE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MeasureValueType").getter(getter(Record::measureValueTypeAsString))
            .setter(setter(Builder::measureValueType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MeasureValueType").build()).build();

    private static final SdkField<String> TIME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Time")
            .getter(getter(Record::time)).setter(setter(Builder::time))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Time").build()).build();

    private static final SdkField<String> TIME_UNIT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TimeUnit").getter(getter(Record::timeUnitAsString)).setter(setter(Builder::timeUnit))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TimeUnit").build()).build();

    private static final SdkField<Long> VERSION_FIELD = SdkField.<Long> builder(MarshallingType.LONG).memberName("Version")
            .getter(getter(Record::version)).setter(setter(Builder::version))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Version").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DIMENSIONS_FIELD,
            MEASURE_NAME_FIELD, MEASURE_VALUE_FIELD, MEASURE_VALUE_TYPE_FIELD, TIME_FIELD, TIME_UNIT_FIELD, VERSION_FIELD));

    private static final long serialVersionUID = 1L;

    private final List<Dimension> dimensions;

    private final String measureName;

    private final String measureValue;

    private final String measureValueType;

    private final String time;

    private final String timeUnit;

    private final Long version;

    private Record(BuilderImpl builder) {
        this.dimensions = builder.dimensions;
        this.measureName = builder.measureName;
        this.measureValue = builder.measureValue;
        this.measureValueType = builder.measureValueType;
        this.time = builder.time;
        this.timeUnit = builder.timeUnit;
        this.version = builder.version;
    }

    /**
     * Returns true if the Dimensions property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasDimensions() {
        return dimensions != null && !(dimensions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Contains the list of dimensions for time series data points.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasDimensions()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Contains the list of dimensions for time series data points.
     */
    public final List<Dimension> dimensions() {
        return dimensions;
    }

    /**
     * <p>
     * Measure represents the data attribute of the time series. For example, the CPU utilization of an EC2 instance or
     * the RPM of a wind turbine are measures.
     * </p>
     * 
     * @return Measure represents the data attribute of the time series. For example, the CPU utilization of an EC2
     *         instance or the RPM of a wind turbine are measures.
     */
    public final String measureName() {
        return measureName;
    }

    /**
     * <p>
     * Contains the measure value for the time series data point.
     * </p>
     * 
     * @return Contains the measure value for the time series data point.
     */
    public final String measureValue() {
        return measureValue;
    }

    /**
     * <p>
     * Contains the data type of the measure value for the time series data point.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #measureValueType}
     * will return {@link MeasureValueType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #measureValueTypeAsString}.
     * </p>
     * 
     * @return Contains the data type of the measure value for the time series data point.
     * @see MeasureValueType
     */
    public final MeasureValueType measureValueType() {
        return MeasureValueType.fromValue(measureValueType);
    }

    /**
     * <p>
     * Contains the data type of the measure value for the time series data point.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #measureValueType}
     * will return {@link MeasureValueType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #measureValueTypeAsString}.
     * </p>
     * 
     * @return Contains the data type of the measure value for the time series data point.
     * @see MeasureValueType
     */
    public final String measureValueTypeAsString() {
        return measureValueType;
    }

    /**
     * <p>
     * Contains the time at which the measure value for the data point was collected. The time value plus the unit
     * provides the time elapsed since the epoch. For example, if the time value is <code>12345</code> and the unit is
     * <code>ms</code>, then <code>12345 ms</code> have elapsed since the epoch.
     * </p>
     * 
     * @return Contains the time at which the measure value for the data point was collected. The time value plus the
     *         unit provides the time elapsed since the epoch. For example, if the time value is <code>12345</code> and
     *         the unit is <code>ms</code>, then <code>12345 ms</code> have elapsed since the epoch.
     */
    public final String time() {
        return time;
    }

    /**
     * <p>
     * The granularity of the timestamp unit. It indicates if the time value is in seconds, milliseconds, nanoseconds or
     * other supported values.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #timeUnit} will
     * return {@link TimeUnit#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #timeUnitAsString}.
     * </p>
     * 
     * @return The granularity of the timestamp unit. It indicates if the time value is in seconds, milliseconds,
     *         nanoseconds or other supported values.
     * @see TimeUnit
     */
    public final TimeUnit timeUnit() {
        return TimeUnit.fromValue(timeUnit);
    }

    /**
     * <p>
     * The granularity of the timestamp unit. It indicates if the time value is in seconds, milliseconds, nanoseconds or
     * other supported values.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #timeUnit} will
     * return {@link TimeUnit#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #timeUnitAsString}.
     * </p>
     * 
     * @return The granularity of the timestamp unit. It indicates if the time value is in seconds, milliseconds,
     *         nanoseconds or other supported values.
     * @see TimeUnit
     */
    public final String timeUnitAsString() {
        return timeUnit;
    }

    /**
     * <p>
     * 64-bit attribute used for record updates. Write requests for duplicate data with a higher version number will
     * update the existing measure value and version. In cases where the measure value is the same, <code>Version</code>
     * will still be updated . Default value is to 1.
     * </p>
     * 
     * @return 64-bit attribute used for record updates. Write requests for duplicate data with a higher version number
     *         will update the existing measure value and version. In cases where the measure value is the same,
     *         <code>Version</code> will still be updated . Default value is to 1.
     */
    public final Long version() {
        return version;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasDimensions() ? dimensions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(measureName());
        hashCode = 31 * hashCode + Objects.hashCode(measureValue());
        hashCode = 31 * hashCode + Objects.hashCode(measureValueTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(time());
        hashCode = 31 * hashCode + Objects.hashCode(timeUnitAsString());
        hashCode = 31 * hashCode + Objects.hashCode(version());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Record)) {
            return false;
        }
        Record other = (Record) obj;
        return hasDimensions() == other.hasDimensions() && Objects.equals(dimensions(), other.dimensions())
                && Objects.equals(measureName(), other.measureName()) && Objects.equals(measureValue(), other.measureValue())
                && Objects.equals(measureValueTypeAsString(), other.measureValueTypeAsString())
                && Objects.equals(time(), other.time()) && Objects.equals(timeUnitAsString(), other.timeUnitAsString())
                && Objects.equals(version(), other.version());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Record").add("Dimensions", hasDimensions() ? dimensions() : null)
                .add("MeasureName", measureName()).add("MeasureValue", measureValue())
                .add("MeasureValueType", measureValueTypeAsString()).add("Time", time()).add("TimeUnit", timeUnitAsString())
                .add("Version", version()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Dimensions":
            return Optional.ofNullable(clazz.cast(dimensions()));
        case "MeasureName":
            return Optional.ofNullable(clazz.cast(measureName()));
        case "MeasureValue":
            return Optional.ofNullable(clazz.cast(measureValue()));
        case "MeasureValueType":
            return Optional.ofNullable(clazz.cast(measureValueTypeAsString()));
        case "Time":
            return Optional.ofNullable(clazz.cast(time()));
        case "TimeUnit":
            return Optional.ofNullable(clazz.cast(timeUnitAsString()));
        case "Version":
            return Optional.ofNullable(clazz.cast(version()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Record, T> g) {
        return obj -> g.apply((Record) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Record> {
        /**
         * <p>
         * Contains the list of dimensions for time series data points.
         * </p>
         * 
         * @param dimensions
         *        Contains the list of dimensions for time series data points.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dimensions(Collection<Dimension> dimensions);

        /**
         * <p>
         * Contains the list of dimensions for time series data points.
         * </p>
         * 
         * @param dimensions
         *        Contains the list of dimensions for time series data points.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dimensions(Dimension... dimensions);

        /**
         * <p>
         * Contains the list of dimensions for time series data points.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Dimension>.Builder} avoiding the need to
         * create one manually via {@link List<Dimension>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Dimension>.Builder#build()} is called immediately and its
         * result is passed to {@link #dimensions(List<Dimension>)}.
         * 
         * @param dimensions
         *        a consumer that will call methods on {@link List<Dimension>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #dimensions(List<Dimension>)
         */
        Builder dimensions(Consumer<Dimension.Builder>... dimensions);

        /**
         * <p>
         * Measure represents the data attribute of the time series. For example, the CPU utilization of an EC2 instance
         * or the RPM of a wind turbine are measures.
         * </p>
         * 
         * @param measureName
         *        Measure represents the data attribute of the time series. For example, the CPU utilization of an EC2
         *        instance or the RPM of a wind turbine are measures.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder measureName(String measureName);

        /**
         * <p>
         * Contains the measure value for the time series data point.
         * </p>
         * 
         * @param measureValue
         *        Contains the measure value for the time series data point.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder measureValue(String measureValue);

        /**
         * <p>
         * Contains the data type of the measure value for the time series data point.
         * </p>
         * 
         * @param measureValueType
         *        Contains the data type of the measure value for the time series data point.
         * @see MeasureValueType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MeasureValueType
         */
        Builder measureValueType(String measureValueType);

        /**
         * <p>
         * Contains the data type of the measure value for the time series data point.
         * </p>
         * 
         * @param measureValueType
         *        Contains the data type of the measure value for the time series data point.
         * @see MeasureValueType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MeasureValueType
         */
        Builder measureValueType(MeasureValueType measureValueType);

        /**
         * <p>
         * Contains the time at which the measure value for the data point was collected. The time value plus the unit
         * provides the time elapsed since the epoch. For example, if the time value is <code>12345</code> and the unit
         * is <code>ms</code>, then <code>12345 ms</code> have elapsed since the epoch.
         * </p>
         * 
         * @param time
         *        Contains the time at which the measure value for the data point was collected. The time value plus the
         *        unit provides the time elapsed since the epoch. For example, if the time value is <code>12345</code>
         *        and the unit is <code>ms</code>, then <code>12345 ms</code> have elapsed since the epoch.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder time(String time);

        /**
         * <p>
         * The granularity of the timestamp unit. It indicates if the time value is in seconds, milliseconds,
         * nanoseconds or other supported values.
         * </p>
         * 
         * @param timeUnit
         *        The granularity of the timestamp unit. It indicates if the time value is in seconds, milliseconds,
         *        nanoseconds or other supported values.
         * @see TimeUnit
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TimeUnit
         */
        Builder timeUnit(String timeUnit);

        /**
         * <p>
         * The granularity of the timestamp unit. It indicates if the time value is in seconds, milliseconds,
         * nanoseconds or other supported values.
         * </p>
         * 
         * @param timeUnit
         *        The granularity of the timestamp unit. It indicates if the time value is in seconds, milliseconds,
         *        nanoseconds or other supported values.
         * @see TimeUnit
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TimeUnit
         */
        Builder timeUnit(TimeUnit timeUnit);

        /**
         * <p>
         * 64-bit attribute used for record updates. Write requests for duplicate data with a higher version number will
         * update the existing measure value and version. In cases where the measure value is the same,
         * <code>Version</code> will still be updated . Default value is to 1.
         * </p>
         * 
         * @param version
         *        64-bit attribute used for record updates. Write requests for duplicate data with a higher version
         *        number will update the existing measure value and version. In cases where the measure value is the
         *        same, <code>Version</code> will still be updated . Default value is to 1.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder version(Long version);
    }

    static final class BuilderImpl implements Builder {
        private List<Dimension> dimensions = DefaultSdkAutoConstructList.getInstance();

        private String measureName;

        private String measureValue;

        private String measureValueType;

        private String time;

        private String timeUnit;

        private Long version;

        private BuilderImpl() {
        }

        private BuilderImpl(Record model) {
            dimensions(model.dimensions);
            measureName(model.measureName);
            measureValue(model.measureValue);
            measureValueType(model.measureValueType);
            time(model.time);
            timeUnit(model.timeUnit);
            version(model.version);
        }

        public final List<Dimension.Builder> getDimensions() {
            List<Dimension.Builder> result = DimensionsCopier.copyToBuilder(this.dimensions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        @Override
        public final Builder dimensions(Collection<Dimension> dimensions) {
            this.dimensions = DimensionsCopier.copy(dimensions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dimensions(Dimension... dimensions) {
            dimensions(Arrays.asList(dimensions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dimensions(Consumer<Dimension.Builder>... dimensions) {
            dimensions(Stream.of(dimensions).map(c -> Dimension.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setDimensions(Collection<Dimension.BuilderImpl> dimensions) {
            this.dimensions = DimensionsCopier.copyFromBuilder(dimensions);
        }

        public final String getMeasureName() {
            return measureName;
        }

        @Override
        public final Builder measureName(String measureName) {
            this.measureName = measureName;
            return this;
        }

        public final void setMeasureName(String measureName) {
            this.measureName = measureName;
        }

        public final String getMeasureValue() {
            return measureValue;
        }

        @Override
        public final Builder measureValue(String measureValue) {
            this.measureValue = measureValue;
            return this;
        }

        public final void setMeasureValue(String measureValue) {
            this.measureValue = measureValue;
        }

        public final String getMeasureValueType() {
            return measureValueType;
        }

        @Override
        public final Builder measureValueType(String measureValueType) {
            this.measureValueType = measureValueType;
            return this;
        }

        @Override
        public final Builder measureValueType(MeasureValueType measureValueType) {
            this.measureValueType(measureValueType == null ? null : measureValueType.toString());
            return this;
        }

        public final void setMeasureValueType(String measureValueType) {
            this.measureValueType = measureValueType;
        }

        public final String getTime() {
            return time;
        }

        @Override
        public final Builder time(String time) {
            this.time = time;
            return this;
        }

        public final void setTime(String time) {
            this.time = time;
        }

        public final String getTimeUnit() {
            return timeUnit;
        }

        @Override
        public final Builder timeUnit(String timeUnit) {
            this.timeUnit = timeUnit;
            return this;
        }

        @Override
        public final Builder timeUnit(TimeUnit timeUnit) {
            this.timeUnit(timeUnit == null ? null : timeUnit.toString());
            return this;
        }

        public final void setTimeUnit(String timeUnit) {
            this.timeUnit = timeUnit;
        }

        public final Long getVersion() {
            return version;
        }

        @Override
        public final Builder version(Long version) {
            this.version = version;
            return this;
        }

        public final void setVersion(Long version) {
            this.version = version;
        }

        @Override
        public Record build() {
            return new Record(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
