/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.timestreamwrite.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents the data attribute of the time series. For example, the CPU utilization of an EC2 instance or the RPM of a
 * wind turbine are measures. MeasureValue has both name and value.
 * </p>
 * <p>
 * MeasureValue is only allowed for type <code>MULTI</code>. Using <code>MULTI</code> type, you can pass multiple data
 * attributes associated with the same time series in a single record
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class MeasureValue implements SdkPojo, Serializable, ToCopyableBuilder<MeasureValue.Builder, MeasureValue> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(MeasureValue::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> VALUE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Value")
            .getter(getter(MeasureValue::value)).setter(setter(Builder::value))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Value").build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Type")
            .getter(getter(MeasureValue::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Type").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, VALUE_FIELD,
            TYPE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String value;

    private final String type;

    private MeasureValue(BuilderImpl builder) {
        this.name = builder.name;
        this.value = builder.value;
        this.type = builder.type;
    }

    /**
     * <p>
     * The name of the MeasureValue.
     * </p>
     * <p>
     * For constraints on MeasureValue names, see <a
     * href="https://docs.aws.amazon.com/timestream/latest/developerguide/ts-limits.html#limits.naming"> Naming
     * Constraints</a> in the Amazon Timestream Developer Guide.
     * </p>
     * 
     * @return The name of the MeasureValue. </p>
     *         <p>
     *         For constraints on MeasureValue names, see <a
     *         href="https://docs.aws.amazon.com/timestream/latest/developerguide/ts-limits.html#limits.naming"> Naming
     *         Constraints</a> in the Amazon Timestream Developer Guide.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The value for the MeasureValue. For information, see <a
     * href="https://docs.aws.amazon.com/timestream/latest/developerguide/writes.html#writes.data-types">Data types</a>.
     * </p>
     * 
     * @return The value for the MeasureValue. For information, see <a
     *         href="https://docs.aws.amazon.com/timestream/latest/developerguide/writes.html#writes.data-types">Data
     *         types</a>.
     */
    public final String value() {
        return value;
    }

    /**
     * <p>
     * Contains the data type of the MeasureValue for the time-series data point.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link MeasureValueType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return Contains the data type of the MeasureValue for the time-series data point.
     * @see MeasureValueType
     */
    public final MeasureValueType type() {
        return MeasureValueType.fromValue(type);
    }

    /**
     * <p>
     * Contains the data type of the MeasureValue for the time-series data point.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link MeasureValueType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return Contains the data type of the MeasureValue for the time-series data point.
     * @see MeasureValueType
     */
    public final String typeAsString() {
        return type;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(value());
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof MeasureValue)) {
            return false;
        }
        MeasureValue other = (MeasureValue) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(value(), other.value())
                && Objects.equals(typeAsString(), other.typeAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("MeasureValue").add("Name", name()).add("Value", value()).add("Type", typeAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Value":
            return Optional.ofNullable(clazz.cast(value()));
        case "Type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Name", NAME_FIELD);
        map.put("Value", VALUE_FIELD);
        map.put("Type", TYPE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<MeasureValue, T> g) {
        return obj -> g.apply((MeasureValue) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, MeasureValue> {
        /**
         * <p>
         * The name of the MeasureValue.
         * </p>
         * <p>
         * For constraints on MeasureValue names, see <a
         * href="https://docs.aws.amazon.com/timestream/latest/developerguide/ts-limits.html#limits.naming"> Naming
         * Constraints</a> in the Amazon Timestream Developer Guide.
         * </p>
         * 
         * @param name
         *        The name of the MeasureValue. </p>
         *        <p>
         *        For constraints on MeasureValue names, see <a
         *        href="https://docs.aws.amazon.com/timestream/latest/developerguide/ts-limits.html#limits.naming">
         *        Naming Constraints</a> in the Amazon Timestream Developer Guide.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The value for the MeasureValue. For information, see <a
         * href="https://docs.aws.amazon.com/timestream/latest/developerguide/writes.html#writes.data-types">Data
         * types</a>.
         * </p>
         * 
         * @param value
         *        The value for the MeasureValue. For information, see <a
         *        href="https://docs.aws.amazon.com/timestream/latest/developerguide/writes.html#writes.data-types">Data
         *        types</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder value(String value);

        /**
         * <p>
         * Contains the data type of the MeasureValue for the time-series data point.
         * </p>
         * 
         * @param type
         *        Contains the data type of the MeasureValue for the time-series data point.
         * @see MeasureValueType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MeasureValueType
         */
        Builder type(String type);

        /**
         * <p>
         * Contains the data type of the MeasureValue for the time-series data point.
         * </p>
         * 
         * @param type
         *        Contains the data type of the MeasureValue for the time-series data point.
         * @see MeasureValueType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MeasureValueType
         */
        Builder type(MeasureValueType type);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String value;

        private String type;

        private BuilderImpl() {
        }

        private BuilderImpl(MeasureValue model) {
            name(model.name);
            value(model.value);
            type(model.type);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getValue() {
            return value;
        }

        public final void setValue(String value) {
            this.value = value;
        }

        @Override
        public final Builder value(String value) {
            this.value = value;
            return this;
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(MeasureValueType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        @Override
        public MeasureValue build() {
            return new MeasureValue(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
